﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto.h>
#include <nn/util/util_FormatString.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonFileSystemTests.h"

namespace {

// 指定したファイルを作成します。
nn::Result EnsureFileRecursively(
               nn::fssystem::save::IFileSystem* pFileSystem,
               const nn::fssystem::save::Path& path,
               int64_t size
           ) NN_NOEXCEPT
{
    nn::fssystem::save::Path::Char buf[nn::fs::EntryNameLengthMax + 1];
    size_t indexBuf = 0;
    size_t indexPath = 0;
    size_t lengthFileName = 0;
    const nn::fssystem::save::Path::Char* pPath = path.GetString();
    size_t lengthPath = path.GetLength();
    for( ; ; )
    {
        // '/'があればディレクトリを作成します。
        if( pPath[indexPath] == static_cast<nn::fssystem::save::Path::Char>('/') &&
            lengthFileName > 0 )
        {
            buf[indexBuf] = static_cast<nn::fssystem::save::Path::Char>('\0');
            nn::fssystem::save::Path bufPath;
            NN_RESULT_DO(bufPath.Initialize(buf));
            nn::Result result = pFileSystem->CreateDirectory(bufPath);
            if( result.IsFailure() )
            {
                if( !nn::fs::ResultAlreadyExists::Includes(result) )
                {
                    return result;
                }
            }
            lengthFileName = 0;
        }

        // パスの最後をファイルとして作成します。
        if( pPath[indexPath] == static_cast<nn::fssystem::save::Path::Char>('\0') )
        {
            if( lengthFileName > 0 )
            {
                buf[indexBuf] = static_cast<nn::fssystem::save::Path::Char>('\0');
                nn::fssystem::save::Path bufPath;
                NN_RESULT_DO(bufPath.Initialize(buf));
                nn::Result result = pFileSystem->CreateFile(bufPath, size);
                if( result.IsFailure() )
                {
                    NN_SDK_ASSERT(nn::fs::ResultAlreadyExists::Includes(result));
                    return result;
                }
                break;
            }
            else
            {
                // パスの最後が'/'で終わっています。
                return nn::fs::ResultNotFound();
            }
        }
        if( indexPath == lengthPath )
        {
            break;
        }
        buf[indexBuf] = pPath[indexPath];

        indexBuf++;
        indexPath++;
        lengthFileName++;
    }

    NN_RESULT_SUCCESS;
}

// 指定したディレクトリを作成します。
nn::Result EnsureDirectoryRecursively(
               nn::fssystem::save::IFileSystem* pFileSystem,
               const nn::fssystem::save::Path& path
           ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pFileSystem);

    nn::fssystem::save::Path::Char buf[nn::fs::EntryNameLengthMax + 1];
    size_t indexBuf = 0;
    size_t indexPath = 0;
    size_t lengthDirectoryName = 0;
    const nn::fssystem::save::Path::Char* pPath = path.GetString();
    size_t lengthPath = path.GetLength();
    for( ; ; )
    {
        if( (pPath[indexPath] == static_cast<nn::fssystem::save::Path::Char>('/') ||
             pPath[indexPath] == static_cast<nn::fssystem::save::Path::Char>('\0')) &&
            lengthDirectoryName > 0 )
        {
            buf[indexBuf] = static_cast<nn::fssystem::save::Path::Char>('\0');
            nn::fssystem::save::Path bufPath;
            NN_RESULT_DO(bufPath.Initialize(buf));
            nn::Result result = pFileSystem->CreateDirectory(bufPath);
            if( result.IsFailure() )
            {
                if( !nn::fs::ResultAlreadyExists::Includes(result) )
                {
                    return result;
                }
            }
            lengthDirectoryName = 0;
        }

        if( indexPath == lengthPath )
        {
            break;
        }
        buf[indexBuf] = pPath[indexPath];

        indexBuf++;
        indexPath++;
        lengthDirectoryName++;
    }

    NN_RESULT_SUCCESS;
}

}

// ファイル、ディレクトリが作成可能なアーカイブに対して
// ディレクトリ作成、削除テストを行います。
void FileSystemTest::TestDirectory(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT
{
    static const int32_t CountDirectory = 10;

    bool isExists;
    {
        const int32_t numDirectoryEntry = 3;
        nn::fs::DirectoryEntry directoryEntry[3];
        nn::fssystem::save::IDirectory* pDirectory;
        nn::fssystem::save::Path pathRoot;
        NNT_ASSERT_RESULT_SUCCESS(pathRoot.Initialize("/"));

        // 存在しないディレクトリを指定します。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNotFound,
                pFileSystem->OpenDirectory(&pDirectory, path)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir1/"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNotFound,
                pFileSystem->OpenDirectory(&pDirectory, path)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir1/dir2"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNotFound,
                pFileSystem->OpenDirectory(&pDirectory, path)
            );
        }

        // ルートディレクトリを開く。まだ何もない
        int32_t resultCount;
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDirectory, pathRoot));
            NNT_ASSERT_RESULT_SUCCESS(
                pDirectory->Read(&resultCount, directoryEntry, numDirectoryEntry));
            ASSERT_EQ(0, resultCount);
            NNT_ASSERT_RESULT_SUCCESS(
                pDirectory->Read(&resultCount, directoryEntry, numDirectoryEntry));
            ASSERT_EQ(0, resultCount);
            pFileSystem->CloseDirectory(pDirectory);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, pathRoot));
            ASSERT_EQ(true, isExists);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathRoot));
            ASSERT_EQ(false, isExists);
        }

        // ディレクトリの再帰生成はサポートされません。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir1/dir2"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNotFound,
                pFileSystem->CreateDirectory(path)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir1/dir2/"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNotFound,
                pFileSystem->CreateDirectory(path)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir1/dir2/dir3"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNotFound,
                pFileSystem->CreateDirectory(path)
            );
        }

        // 先頭にスラッシュなしの指定は失敗します。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("dir1"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidPath,
                pFileSystem->CreateDirectory(path)
            );
        }

        // ルートディレクトリはすでに存在しています。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->CreateDirectory(path)
            );
        }

        // ルートディレクトリに親ディレクトリは存在しません。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/../"));
            nn::Result result = pFileSystem->CreateDirectory(path);
            ASSERT_TRUE(
                nn::fs::ResultInvalidPath::Includes(result) ||
                nn::fs::ResultUnexpected::Includes(result)
            );
        }

        // 区切り文字が連続していますが、同一視します
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("//"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->CreateDirectory(path)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("///"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->CreateDirectory(path)
            );
        }

        // 自身をさす場合も同様に処理されます。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/./"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->CreateDirectory(path)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/././"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->CreateDirectory(path)
            );
        }

        // ディレクトリ名が長すぎます。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidPath,
                path.Initialize(
                    "/00000000001111111111222222222233333333334444444444"
                    "55555555556666666666777777777788888888889999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                    "00000000001111111111222222222233333333334444444444"
                    "555555555566666666667777777777888888888899999999999"
                )
            );
        }

        // ディレクトリを1つ作ります。
        nn::fssystem::save::Path pathDirectory;
        NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDirectory, pathRoot));
            NNT_ASSERT_RESULT_SUCCESS(
                pDirectory->Read(&resultCount, directoryEntry, numDirectoryEntry));
            ASSERT_EQ(1, resultCount);    // 一つ出来た
            ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, directoryEntry[0].directoryEntryType);
            ASSERT_EQ(0, std::memcmp("directory", directoryEntry[0].name, 8));
            NNT_ASSERT_RESULT_SUCCESS(
                pDirectory->Read(&resultCount, directoryEntry, numDirectoryEntry));
            ASSERT_EQ(0, resultCount);
            pFileSystem->CloseDirectory(pDirectory);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, pathDirectory));
            ASSERT_EQ(true, isExists);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathDirectory));
            ASSERT_EQ(false, isExists);
        }

        // サイズが負のファイルは作れません。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/file3"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidSize,
                pFileSystem->CreateFile(path, -1)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/file3"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidSize,
                pFileSystem->CreateFile(path, -2)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/directory/file3"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidSize,
                pFileSystem->CreateFile(path, -1)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/directory/file3"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidSize,
                pFileSystem->CreateFile(path, -2)
            );
        }

        // ファイルを一つ作ります。
        nn::fssystem::save::Path pathFile;
        NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/filefilefile"));
        {
            nn::fssystem::save::IFile* pFile = nullptr;
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, 0));
            NNT_ASSERT_RESULT_SUCCESS(
                pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDirectory, pathRoot));
            NNT_ASSERT_RESULT_SUCCESS(
                pDirectory->Read(&resultCount, directoryEntry, numDirectoryEntry));
            ASSERT_EQ(2, resultCount);    // 2つ出来た
            ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, directoryEntry[0].directoryEntryType);
            ASSERT_EQ(0, std::memcmp("directory", directoryEntry[0].name, 8));
            ASSERT_EQ(nn::fs::DirectoryEntryType_File, directoryEntry[1].directoryEntryType);
            ASSERT_EQ(0, std::memcmp("filefilefile", directoryEntry[1].name, 12));
            NNT_ASSERT_RESULT_SUCCESS(
                pDirectory->Read(&resultCount, directoryEntry, numDirectoryEntry));
            ASSERT_EQ(0, resultCount);
            pFileSystem->CloseDirectory(pDirectory);
            pFileSystem->CloseFile(pFile);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathFile));
            ASSERT_EQ(true, isExists);
        }

        // ファイル以下にファイルは作れません。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/filefilefile/file"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidPathForOperation,
                pFileSystem->CreateFile(path, 1)
            );
        }

        // ディレクトリとファイルを消去します。
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidPathForOperation,
                pFileSystem->DeleteDirectory(pathFile)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathDirectory));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNotFound, pFileSystem->DeleteFile(pathFile));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNotFound, pFileSystem->DeleteDirectory(pathDirectory));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, pathDirectory));
            ASSERT_EQ(false, isExists);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathFile));
            ASSERT_EQ(false, isExists);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, pathRoot));
            ASSERT_EQ(true, isExists);
        }
    }

    // ディレクトリのテストを行います。
    {
        int32_t resultCount;
        nn::fs::DirectoryEntry directoryEntry[3];
        int32_t numDirectoryEntry = sizeof(directoryEntry) / sizeof(directoryEntry[0]);
        nn::fssystem::save::IDirectory* pDirectory;
        nn::fssystem::save::Path pathRoot;
        NNT_ASSERT_RESULT_SUCCESS(pathRoot.Initialize("/"));

        // ディレクトリを作成します。
        for( int32_t i = 0; i < CountDirectory; i++ )
        {
            char name[32];
            std::sprintf(name, "/dir%u", i);
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize(name));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));
        }

        // ディレクトリの列挙テストを行います。
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDirectory, pathRoot));
            for( int32_t i = 0; i + numDirectoryEntry <= CountDirectory; i += numDirectoryEntry )
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pDirectory->Read(&resultCount, directoryEntry, numDirectoryEntry));
                ASSERT_EQ(numDirectoryEntry, resultCount);
            }
            NNT_ASSERT_RESULT_SUCCESS(
                pDirectory->Read(&resultCount, directoryEntry, numDirectoryEntry));
            ASSERT_EQ(CountDirectory % numDirectoryEntry, resultCount);
            NNT_ASSERT_RESULT_SUCCESS(
                pDirectory->Read(&resultCount, directoryEntry, numDirectoryEntry));
            ASSERT_EQ(0, resultCount);

            pFileSystem->CloseDirectory(pDirectory);
        }

        for( int32_t i = 0; i < CountDirectory; ++i )
        {
            char name[32];
            std::sprintf(name, "/dir%u", i);
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize(name));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathDirectory));
        }
    }
} // NOLINT(impl/function_size)

// ファイル、ディレクトリが作成可能なアーカイブに対して
// ファイルの作成、削除テストを行います。
void FileSystemTest::TestFile(
                         nn::fssystem::save::IFileSystem* pFileSystem,
                         bool isEnableResizeFile,
                         bool isAutoExtendSizeOnWrite
                     ) NN_NOEXCEPT
{
    int64_t size;
    bool isExists;

    // ファイルの処理
    {
        nn::fssystem::save::Path file1;
        NNT_ASSERT_RESULT_SUCCESS(file1.Initialize("/file1"));

        // 存在しないファイルを開く
        nn::fssystem::save::IFile* pFile = nullptr;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNotFound,
            pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Read)
        );
        ASSERT_EQ(nullptr, pFile);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNotFound,
            pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write)
        );
        ASSERT_EQ(nullptr, pFile);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(false, isExists);

        // 誤った指定(モード指定テスト)
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOperationForOpenMode,
            pFileSystem->OpenFile(&pFile, file1, 0)
        );
        ASSERT_EQ(nullptr, pFile);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(false, isExists);

        // 正しい指定(モード指定テスト)
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(file1, 0));
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Read)
        );
        pFileSystem->CloseFile(pFile);
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write)
        );

        // ファイルを開いている状態で存在確認
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(true, isExists);

        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAlreadyExists,
            pFileSystem->CreateFile(file1, 10)
        );

        // サイズの取得と設定
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        const int64_t sizeOld = size;
        if( isEnableResizeFile )
        {
            static const int64_t Giga = 1024 * 1024 * 1024;

            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(0, size);
            nn::Result result = pFile->SetSize(1024 * Giga);
            ASSERT_TRUE(
                nn::fs::ResultOutOfResource::Includes(result)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(1));
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(1, size);
            NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeOld / 2));
            NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeOld));
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(sizeOld, size);

            // サイズに負の値を指定することはできません。
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidArgument,
                pFile->SetSize(-1)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidArgument,
                pFile->SetSize(-2)
            );
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultUnsupportedOperation,
                pFile->SetSize(0)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultUnsupportedOperation,
                pFile->SetSize(1)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultUnsupportedOperation,
                pFile->SetSize(sizeOld)
            );
        }

        // ファイルを閉じる
        pFileSystem->CloseFile(pFile);

        // サイズ設定を再度する
        if( isEnableResizeFile )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(sizeOld, size);
            NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(0));
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(0, size);
            pFileSystem->CloseFile(pFile);
        }
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(true, isExists);

        // リード状態ではサイズを設定できません
        if( isEnableResizeFile )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Read)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(0, size);
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOperationForOpenMode,
                pFile->SetSize(16)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(0, size);
            pFileSystem->CloseFile(pFile);
        }

        nnt::fs::util::Vector<char> bufWrite(128);
        nnt::fs::util::Vector<char> bufRead(128);
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(
                &pFile,
                file1,
                nn::fs::OpenMode_Write | nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));

        // 0 バイトのファイルに 0 バイトの書き込みます。
        ASSERT_EQ(0, size);
        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, &bufWrite[0], 0));

        // 範囲外への書き込み
        if( isAutoExtendSizeOnWrite )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(1, &bufWrite[0], 0)
            );
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidArgument,
                pFile->WriteBytes(1, &bufWrite[0], 0)
            );
        }

        // 0 バイトのファイルから 0 バイトの読み込みます。
        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &bufRead[0], 0));

        // 範囲外からの読み込み
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            pFile->ReadBytes(1, &bufRead[0], 0)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &bufRead[0], 1));

        // ファイルサイズを30バイトに設定しました。
        if( isEnableResizeFile )
        {
            NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(30));
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(30, size);

            // 範囲内のアクセス
            NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(28, &bufWrite[0], 2));
            NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(28, &bufRead[0], 2));
            NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(1, &bufWrite[0], 29));
            NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(1, &bufRead[0], 29));

            // 範囲を少々超えるアクセス -> 自動拡張
            NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(28, &bufWrite[0], 4));
            NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(28, &bufRead[0], 4));
            NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, &bufWrite[0], 31));
            NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &bufRead[0], 31));

            // 完全に範囲外のアクセス
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                pFile->ReadBytes(33, &bufRead[0], 1)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                pFile->ReadBytes(34, &bufRead[0], 1)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(33, &bufWrite[0], 1)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(34, &bufWrite[0], 1)
            );
            // 拡張されたのでアクセス可能に
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->ReadBytes(33, &bufRead[0], 1)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->ReadBytes(34, &bufRead[0], 1)
            );
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultUnsupportedOperation,
                pFile->SetSize(30)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(0, size);

            // 範囲を超えるアクセス
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultUnsupportedOperation,
                pFile->WriteBytes(0, &bufWrite[0], 31)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &bufRead[0], 31));

            // 完全に範囲外のアクセス
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                pFile->WriteBytes(33, &bufWrite[0], 1)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                pFile->WriteBytes(34, &bufWrite[0], 1)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                pFile->ReadBytes(33, &bufRead[0], 1)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                pFile->ReadBytes(34, &bufRead[0], 1)
            );
        }

        pFileSystem->CloseFile(pFile);

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(true, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(file1));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(false, isExists);
    }
} // NOLINT(impl/function_size)

// ファイル、ディレクトリが作成可能なアーカイブに対して
// ファイル、ディレクトリの作成と削除テストを行います。
void FileSystemTest::TestCreation(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT
{
    static const uint32_t CountDirectory = 100;
    static const uint32_t CountFile = 100;

    // ルートディレクトリは削除できません。
    nn::fssystem::save::Path pathRoot;
    NNT_ASSERT_RESULT_SUCCESS(pathRoot.Initialize("/"));
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathForOperation,
        pFileSystem->DeleteDirectory(pathRoot)
    );

    nnt::fs::util::Vector<nn::fssystem::save::IFile*> pFiles(CountFile);

    uint32_t countFilesPerDirectory = ((CountFile + (CountDirectory - 1)) / CountDirectory);
    if( countFilesPerDirectory == 0 )
    {
        countFilesPerDirectory = CountFile;
    }
    for( uint32_t i = 0; i < CountFile; i++ )
    {
        const size_t directoryId = i / countFilesPerDirectory;
        const size_t fileId = i;

        // ディレクトリ作成
        char buf[32];
        nn::util::SNPrintf(buf, sizeof(buf), "/%X", directoryId);
        nn::fssystem::save::Path pathDirectory;
        NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize(buf));
        if( (i % countFilesPerDirectory) == 0 )
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDbmAlreadyExists,
                pFileSystem->CreateDirectory(pathDirectory)
            );
        }

        // ".." などを用いてもルートディレクトリは削除できません。

        // ファイルの作成
        nn::util::SNPrintf(buf, sizeof(buf), "/%X/%X", directoryId, fileId);
        nn::fssystem::save::Path pathFile;
        NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize(buf));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, 0));
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFiles[i], pathFile, nn::fs::OpenMode_Write)
        );
    }

    // 一斉に閉じます
    for( uint32_t i = 0; i < CountFile; ++i )
    {
        pFileSystem->CloseFile(pFiles[i]);
    }

    for( uint32_t i = 0; i < CountFile; ++i )
    {
        const size_t directoryId = i / countFilesPerDirectory;
        const size_t fileId = i;

        // ファイルを消します。
        char buf[32];
        nn::util::SNPrintf(buf, 32, "/%X/%X", directoryId, fileId);

        nn::fssystem::save::Path pathBuf;
        NNT_ASSERT_RESULT_SUCCESS(pathBuf.Initialize(buf));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathBuf));

        // ディレクトリを削除します。
        nn::util::SNPrintf(buf, 32, "/%X", directoryId);
        NNT_ASSERT_RESULT_SUCCESS(pathBuf.Initialize(buf));
        if( ((i % countFilesPerDirectory) == countFilesPerDirectory - 1) || (i == CountFile - 1) )
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathBuf));
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultPermissionDenied,
                pFileSystem->DeleteDirectory(pathBuf)
            );
        }
    }
}

// アーカイブにおけるファイル、ディレクトリのアクセス制御をテストします。
void FileSystemTest::TestOpenClose(
                         nn::fssystem::save::IFileSystem* pFileSystem,
                         bool isEnableExclusiveAccess
                     ) NN_NOEXCEPT
{
    nn::fssystem::save::Path path1;
    NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/file1"));

    nn::fssystem::save::Path path2;
    NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/file2"));

    nn::fssystem::save::Path path3;
    NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/dir1/dir1"));

    nn::fssystem::save::Path path4;
    NNT_ASSERT_RESULT_SUCCESS(path4.Initialize("/dir2/dir2/file"));

    bool isExists;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path1));
    ASSERT_EQ(false, isExists);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path3));
    ASSERT_EQ(false, isExists);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path4));
    ASSERT_EQ(false, isExists);

    // "/file1", "/file2" を同時に書き込みモードで開きます。
    nn::fssystem::save::IFile* pFile1;
    nn::fssystem::save::IFile* pFile2;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path1, 0));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile1, path1, nn::fs::OpenMode_Write));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path2, 0));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile2, path2, nn::fs::OpenMode_Write));
    pFileSystem->CloseFile(pFile2);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path2));
    pFileSystem->CloseFile(pFile1);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path1));
    ASSERT_EQ(true, isExists);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path2));
    ASSERT_EQ(false, isExists);

    static const int Mode[3] =
    {
        nn::fs::OpenMode_Write | nn::fs::OpenMode_Read,
        nn::fs::OpenMode_Write,
        nn::fs::OpenMode_Read
    };

    for( size_t i = 0; i < sizeof(Mode) / sizeof(Mode[0]); ++i )
    {
        nn::fssystem::save::Path path = path1;

        // /file1をいろいろなモードでオープンします
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile1, path, Mode[i]));

        // オープン済みのファイルに対し、書き込みモードでのオープンは失敗する
        if( isEnableExclusiveAccess )
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultPermissionDenied,
                pFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Write)
            );

            // オープン済みのファイルに対し、削除、リネームは失敗する
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultPermissionDenied,
                pFileSystem->DeleteFile(path)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultPermissionDenied,
                pFileSystem->RenameFile(path, path2)
            );

            if( (Mode[i] & nn::fs::OpenMode_Write) != 0 )
            {
                // 書き込み属性で既に開かれているファイルをさらに開こうとしたら失敗
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultPermissionDenied,
                    pFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Read)
                );
            }
            else
            {
                // 読み込み属性(書き込みなし)で開かれているファイルをさらに開こうとしたら成功
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Read));
                pFileSystem->CloseFile(pFile2);
            }
        }

        // Closeは成功する
        pFileSystem->CloseFile(pFile1);

        // 同名ファイルへのリネームは常に成功します。
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(path, path));

        // リネームしなおしてファイルが存在することを確認します。
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(path, path2));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path1));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(path2, path));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path1));
        ASSERT_EQ(true, isExists);
    }

    // リード多重、リード+ライト
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile1, path1, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile2, path1, nn::fs::OpenMode_Read));
    pFileSystem->CloseFile(pFile2);
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultPermissionDenied,
        pFileSystem->OpenFile(&pFile2, path1, nn::fs::OpenMode_Write)
    );
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPermissionDenied, pFileSystem->DeleteFile(path1));
    pFileSystem->CloseFile(pFile1);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path1));
} // NOLINT(impl/function_size)

#if 0 // マルチスレッドに対応したら有効化
struct ThreadTestArg
{
    nn::fssystem::save::IFileSystem* pFileSystem;
    uint32_t nId;
    nn::fssystem::save::Random random;
    bool isEnableResizeFile;
    char reserved[3];
};

// 複数のスレッドから同時にファイルシステムにアクセスします。
static void OnTest4Thread1(const void* ptr)
{
    ThreadTestArg* pArg = reinterpret_cast<ThreadTestArg*>(ptr);
    nn::fssystem::save::IFileSystem* pFileSystem = pArg->pFileSystem;
    uint32_t threadId = pArg->nId;
    bool isEnableResizeFile = pArg->isEnableResizeFile;
    nn::fssystem::save::Random& random = pArg->random;
    static const size_t TestBufSize = 128;
    char workBuf1[TestBufSize];
    char workBuf2[TestBufSize];
    char buf[32];
    char buf2[32];
    char buf3[32];
    nn::Result result;

    // テストバッファを乱数で埋めます。
    for( int32_t j = 0; j < TestBufSize; j++ )
    {
        workBuf1[j] = random.Get32(256) & 0xFF;
    }
    for( int32_t i = 0; i < 10; i++ )
    {
        nn::fssystem::save::IFile* pFile;
        nn::fssystem::save::IDirectory* pDirectory;

        switch( random.Get32(6) )
        {
        case 0:
            {
                std::memset(buf, 0, sizeof(buf));
                std::sprintf(buf, "/file%u", threadId);
                nn::fssystem::save::Path pathCurr(static_cast<const char *>(buf));

                result = pFileSystem->CreateFile(pathCurr, 0);
                if( result.IsFailure() )
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmNotFound, result);
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        pFileSystem->OpenFile(&pFile, pathCurr, nn::fs::OpenMode_Write));

                    int64_t size = random.Get32(TestBufSize);
                    if( isEnableResizeFile )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(size));
                        NNT_ASSERT_RESULT_SUCCESS(
                            pFile->WriteBytes(0, workBuf1, static_cast<size_t>(size)));
                    }
                    pFileSystem->CloseFile(pFile);

                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(
                        &pFile, pathCurr, nn::fs::OpenMode_Read));
                    if( isEnableResizeFile )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            pFile->ReadBytes(0, workBuf2, static_cast<size_t>(size)));
                        ASSERT_EQ(0, std::memcmp(workBuf1, workBuf2, static_cast<size_t>(size)));
                    }
                    pFileSystem->CloseFile(pFile);

                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathCurr));
                }
            }
            break;
        case 1:
            {
                std::memset(buf, 0, sizeof(buf));
                std::sprintf(buf, "/dir%u-1", threadId);
                std::memset(buf2, 0, sizeof(buf));
                std::sprintf(buf2, "/dir%u-2", threadId);
                nn::fssystem::save::Path pathCurr1(static_cast<const char *>(buf));
                nn::fssystem::save::Path pathCurr2(static_cast<const char *>(buf2));
                nn::fssystem::save::Path pathRoot("/");

                result = pFileSystem->CreateDirectory(pathCurr1);
                if( result.IsFailure() )
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                }
                else
                {
                    result = pFileSystem->CreateDirectory(pathCurr2);
                    if( result.IsFailure() )
                    {
                        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCurr1));
                    }
                    else
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            pFileSystem->OpenDirectory(&pDirectory, pathRoot));
                        int32_t nDoneCount;
                        nn::fs::DirectoryEntry de[2];
                        NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&nDoneCount, de, 2));
                        ASSERT_EQ(2, nDoneCount);
                        pFileSystem->CloseDirectory(pDirectory);
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCurr1));
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCurr2));
                    }
                }
            }
            break;
        case 2:
            {
                std::memset(buf, 0, sizeof(buf));
                std::sprintf(buf, "/dir%u-1", threadId);
                std::memset(buf2, 0, sizeof(buf));
                std::sprintf(buf2, "/dir%u-2", threadId);
                nn::fssystem::save::Path pathCurr1(static_cast<const char *>(buf));
                nn::fssystem::save::Path pathCurr2(static_cast<const char *>(buf2));
                result = pFileSystem->CreateDirectory(pathCurr1);
                if( result.IsFailure() )
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameDirectory(pathCurr1, pathCurr2));
                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCurr2));
                }
            }
            break;
        case 3:
            {
                std::memset(buf, 0, sizeof(buf));
                std::sprintf(buf, "/file%u-1", threadId);
                std::memset(buf2, 0, sizeof(buf));
                std::sprintf(buf2, "/file%u-2", threadId);
                nn::fssystem::save::Path pathCurr1(static_cast<const char *>(buf));
                nn::fssystem::save::Path pathCurr2(static_cast<const char *>(buf2));
                result = pFileSystem->CreateFile(pathCurr1, 0);
                if( result.IsFailure() )
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        pFileSystem->OpenFile(&pFile, pathCurr1, nn::fs::OpenMode_Write));
                    if( isEnableResizeFile )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(TestBufSize));
                        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, workBuf1, TestBufSize));
                    }
                    pFileSystem->CloseFile(pFile);

                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(pathCurr1, pathCurr2));

                    NNT_ASSERT_RESULT_SUCCESS(
                        pFileSystem->OpenFile(&pFile, pathCurr2, nn::fs::OpenMode_Read));
                    if( isEnableResizeFile )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, workBuf2, TestBufSize));
                        ASSERT_EQ(0, std::memcmp(workBuf1, workBuf2, TestBufSize));
                    }
                    pFileSystem->CloseFile(pFile);

                    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathCurr2));
                }
            }
            break;
        case 4:
            {
                std::memset(buf, 0, sizeof(buf));
                std::sprintf(buf, "/dir%u", threadId);
                nn::fssystem::save::Path pathDirectory(static_cast<const char *>(buf));
                result = pFileSystem->CreateDirectory(pathDirectory);
                if( result.IsFailure() )
                {
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                }
                else
                {
                    std::memset(buf2, 0, sizeof(buf2));
                    std::sprintf(buf2, "/dir%u/file%u", threadId, threadId);
                    nn::fssystem::save::Path pathCurr(static_cast<const char *>(buf2));

                    std::memset(buf3, 0, sizeof(buf3));
                    std::sprintf(buf3, "/file%u", threadId);
                    nn::fssystem::save::Path pathRename(static_cast<const char *>(buf3));

                    result = pFileSystem->CreateFile(pathCurr, 0);
                    if( result.IsFailure() )
                    {
                        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDbmKeyFull, result);
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathDirectory));
                    }
                    else
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            pFileSystem->OpenFile(&pFile, pathCurr, nn::fs::OpenMode_Write));
                        int64_t size = random.Get32(TestBufSize);
                        if( isEnableResizeFile )
                        {
                            NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(size));
                            NNT_ASSERT_RESULT_SUCCESS(
                                pFile->WriteBytes(0, workBuf1, static_cast<sizet_t>(size)));
                        }
                        pFileSystem->CloseFile(pFile);

                        NNT_ASSERT_RESULT_SUCCESS(
                            pFileSystem->OpenFile(&pFile, pathCurr, nn::fs::OpenMode_Read));
                        if( isEnableResizeFile )
                        {
                            NNT_ASSERT_RESULT_SUCCESS(
                                pFile->ReadBytes(0, workBuf2, static_cast<sizet_t>(size)));
                            ASSERT_EQ(
                                0, std::memcmp(workBuf1, workBuf2, static_cast<sizet_t>(size)));
                        }
                        pFileSystem->CloseFile(pFile);

                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(pathCurr, pathRename));
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathDirectory));
                        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathRename));
                    }
                }
            }
            break;
        case 5:
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    delete pArg;
} // NOLINT(impl/function_size)

// 複数のスレッドからアーカイブに同時アクセステストを行います。
void FileSystemTest::TestThread(
    nn::fssystem::save::IFileSystem* pFileSystem,
    bool isEnableResizeFile
    )
{
    nn::Result result;

    nn::os::ThreadType threads[4];

    // 複数のスレッドから同時にファイルシステムにアクセスします。
    for( uint32_t i = 0; i < sizeof(threads) / sizeof(threads[0]); i++ )
    {
        ThreadTestArg* pArg = new ThreadTestArg ();
        pArg->pFileSystem = pFileSystem;
        pArg->nId = i;
        pArg->rnd.SetSeed(123456789 * (i + 1));
        pArg->isEnableResizeFile = isEnableResizeFile;
        result = nn::os::CreateThread(
            &threads[i],
            OnTest4Thread1,
            pArg,
            16 * 1024,
            NN_OS_DEFAULT_THREAD_PRIORITY - i
        );
        NNT_ASSERT_RESULT_SUCCESS(result);
        nn::os::StartThread(&threads[i]);
    }
    for( uint32_t i = 0; i < sizeof(threads) / sizeof(threads[0]); i++ )
    {
        nn::os::WaitThread(&threads[i]);
        nn::os::DestroyThread(&threads[i]);
    }
}
#endif

// リネーム機能のテストを行ないます。
void FileSystemTest::TestRename(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT
{
    bool isExists;

    {
        nn::fssystem::save::Path path3;
        NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/file1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path3, 1));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path3));
        ASSERT_EQ(true, isExists);
        nn::fssystem::save::Path path5;
        NNT_ASSERT_RESULT_SUCCESS(path5.Initialize("/File1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path5));
        ASSERT_EQ(false, isExists);

        // 自身へのリネーム
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(path3, path3));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path3));
            ASSERT_EQ(true, isExists);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(path3, path3));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path3));
            ASSERT_EQ(true, isExists);
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/./file1"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->RenameFile(path, path3)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path3));
            ASSERT_EQ(true, isExists);
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/./file1"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->RenameFile(path3, path)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path3));
            ASSERT_EQ(true, isExists);
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/File1"));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(path3, path));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path3));
            ASSERT_EQ(false, isExists);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path));
            ASSERT_EQ(true, isExists);

            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(path, path3));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path3));
            ASSERT_EQ(true, isExists);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path));
            ASSERT_EQ(false, isExists);
        }

        // "." を用いた名称にはリネーム出来ません。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/."));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidPathFormat,
                pFileSystem->RenameFile(path3, path)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/././."));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidPathFormat,
                pFileSystem->RenameFile(path3, path)
            );
        }

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path3));
    }

    {
#if 0 // TODO: 現在は上書きリネームができないので無効にしておく
        nn::fssystem::save::Path path1;
        NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/dir1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path1));
        nn::fssystem::save::Path path2;
        NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/dir1/dir2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path2));
        nn::fssystem::save::Path path3;
        NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/dir1/dir2/file1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path3, 1));

        nn::fssystem::save::Path path4;
        NNT_ASSERT_RESULT_SUCCESS(path4.Initialize("/dir3"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path4));
        nn::fssystem::save::Path    ;
        NNT_ASSERT_RESULT_SUCCESS(path5.Initialize("/dir3/file2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path5, 1));

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameDirectory(path1, path4));

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path1));
        ASSERT_EQ(false, isExists);
        nn::fssystem::save::Path path6;
        NNT_ASSERT_RESULT_SUCCESS(path6.Initialize("/dir3/dir2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path6));
        ASSERT_EQ(true, isExists);
        nn::fssystem::save::Path path7;
        NNT_ASSERT_RESULT_SUCCESS(path7.Initialize("/dir3/dir2/file1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path7));
        ASSERT_EQ(true, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path5));
        ASSERT_EQ(false, isExists);
#else
        nn::fssystem::save::Path path4;
        NNT_ASSERT_RESULT_SUCCESS(path4.Initialize("/dir3"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path4));
        nn::fssystem::save::Path path2;
        NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/dir3/dir2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path2));
        nn::fssystem::save::Path path3;
        NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/dir3/dir2/file1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path3, 1));
        nn::fssystem::save::Path path5;
        NNT_ASSERT_RESULT_SUCCESS(path5.Initialize("/dir3/file2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path5, 1));

        nn::fssystem::save::Path path6;
        NNT_ASSERT_RESULT_SUCCESS(path6.Initialize("/dir3/dir2"));
        nn::fssystem::save::Path path7;
        NNT_ASSERT_RESULT_SUCCESS(path7.Initialize("/dir3/dir2/file1"));
#endif

        // 自身へのリネームは常に成功します。
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(path7, path7));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path7));
            ASSERT_EQ(true, isExists);
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir3/../dir3/./dir2/file1"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->RenameFile(path7, path)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path7));
            ASSERT_EQ(true, isExists);
        }
        {
            nn::fssystem::save::Path pathA;
            NNT_ASSERT_RESULT_SUCCESS(pathA.Initialize("/dir3/../dir3/dir2/.///file1"));
            nn::fssystem::save::Path pathB;
            NNT_ASSERT_RESULT_SUCCESS(pathB.Initialize("/dir3/../dir3/./dir2/file1"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->RenameFile(pathA, pathB)
            );
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path7));
            ASSERT_EQ(true, isExists);
        }

        // ルートディレクトリにリネームは出来ません。
        {
            nn::fssystem::save::Path pathRoot;
            NNT_ASSERT_RESULT_SUCCESS(pathRoot.Initialize("/"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                pFileSystem->RenameDirectory(path4, pathRoot)
            );
        }

        // ".", ".." を用いたルートディレクトリへのリネームも出来ません。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir3/.."));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                pFileSystem->RenameDirectory(path4, path)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir3/../dir3/dir2/../../."));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                pFileSystem->RenameDirectory(path4, path)
            );
        }

        // ".", ".." を用いた名称にはリネーム出来ません。
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir3/dir2/."));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidPathFormat,
                pFileSystem->RenameFile(path7, path)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir3/dir2/././.."));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidPathFormat,
                pFileSystem->RenameFile(path7, path)
            );
        }

        // 親ディレクトリにリネームは出来ません。
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                pFileSystem->RenameDirectory(path6, path4)
            );
        }

        // 子ディレクトリにもリネーム出来ません。
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                pFileSystem->RenameDirectory(path4, path6)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir3/dir1"));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                pFileSystem->RenameDirectory(path4, path)
            );
        }
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir3/dir1/."));
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultNotFound,
                pFileSystem->RenameDirectory(path4, path)
            );
        }

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectoryRecursively(path4));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path4));
        ASSERT_EQ(false, isExists);
    }

    {
#if 0 // TODO: 現在は上書きリネームができないので無効にしておく
        nn::fssystem::save::Path path1;
        NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/dir1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path1));

        nn::fssystem::save::Path path2;
        NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/dir1/dir2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path2));

        nn::fssystem::save::Path path3;
        NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/dir1/dir3"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path3));

        nn::fssystem::save::Path path4;
        NNT_ASSERT_RESULT_SUCCESS(path4.Initialize("/dir1/file1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path4, 1));

        nn::fssystem::save::Path path5;
        NNT_ASSERT_RESULT_SUCCESS(path5.Initialize("/dir1/file2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path5, 1));

        nn::fssystem::save::Path path6;
        NNT_ASSERT_RESULT_SUCCESS(path6.Initialize("/dir1/dir2/file3"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path6, 1));

        nn::fssystem::save::Path path7;
        NNT_ASSERT_RESULT_SUCCESS(path7.Initialize("/dir4"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path7));

        nn::fssystem::save::Path path8;
        NNT_ASSERT_RESULT_SUCCESS(path8.Initialize("/dir4/dir5"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path8));

        nn::fssystem::save::Path path9;
        NNT_ASSERT_RESULT_SUCCESS(path9.Initialize("/dir4/file5"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path9, 1));

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameDirectory(path1, path7));

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path1));
        ASSERT_EQ(false, isExists);
#else
        nn::fssystem::save::Path path1;
        NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/dir1"));
        nn::fssystem::save::Path path7;
        NNT_ASSERT_RESULT_SUCCESS(path7.Initialize("/dir4"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path7));

        nn::fssystem::save::Path path2;
        NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/dir4/dir2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path2));

        nn::fssystem::save::Path path3;
        NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/dir4/dir3"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path3));

        nn::fssystem::save::Path path4;
        NNT_ASSERT_RESULT_SUCCESS(path4.Initialize("/dir4/file1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path4, 1));

        nn::fssystem::save::Path path5;
        NNT_ASSERT_RESULT_SUCCESS(path5.Initialize("/dir4/file2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path5, 1));

        nn::fssystem::save::Path path6;
        NNT_ASSERT_RESULT_SUCCESS(path6.Initialize("/dir4/dir2/file3"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path6, 1));

        nn::fssystem::save::Path path8;
        NNT_ASSERT_RESULT_SUCCESS(path8.Initialize("/dir4/dir5"));
        nn::fssystem::save::Path path9;
        NNT_ASSERT_RESULT_SUCCESS(path9.Initialize("/dir4/file5"));
#endif
        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir4/dir2"));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path));
            ASSERT_EQ(true, isExists);
        }

        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir4/dir3"));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path));
            ASSERT_EQ(true, isExists);
        }

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path8));
        ASSERT_EQ(false, isExists);

        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir4/file1"));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path));
            ASSERT_EQ(true, isExists);
        }

        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir4/file2"));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path));
            ASSERT_EQ(true, isExists);
        }

        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir4/dir2/file3"));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path));
            ASSERT_EQ(true, isExists);
        }

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path9));
        ASSERT_EQ(false, isExists);

#if 0 // TODO: 上書きリネームが動作したら有効化
        // リネームしたディレクトリは削除できないことを確認します。
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUndeletable,
            pFileSystem->DeleteDirectory(path1)
        );
#endif

        // リネームで空いたパスに作成できることを確認します。
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path1));

        {
            nn::fssystem::save::Path path;
            NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir1/file8"));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path, 1));
        }

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectoryRecursively(path7));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path7));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectoryRecursively(path1));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path1));
        ASSERT_EQ(false, isExists);
    }

#if 0 // TODO: 現在は上書きリネームができないので無効にしておく
    // ファイルを開いた状態での上書きリネームテスト
    {
        nn::fssystem::save::Path path1;
        NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/dir1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path1));

        nn::fssystem::save::Path pathDir1File1;
        NNT_ASSERT_RESULT_SUCCESS(pathDir1File1.Initialize("/dir1/file1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathDir1File1, 1));

        nn::fssystem::save::Path path2;
        NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/dir2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path2));

        // リネーム元のファイルを開いている場合は上書きリネームできます。
        nn::fssystem::save::IFile* pFile = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFile, pathDir1File1, nn::fs::OpenMode_Write)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameDirectory(path1, path2));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path1));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path2));
        ASSERT_EQ(true, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathDir1File1));
        ASSERT_EQ(false, isExists);
        nn::fssystem::save::Path pathDir2File1;
        NNT_ASSERT_RESULT_SUCCESS(pathDir2File1.Initialize("/dir2/file1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathDir2File1));
        ASSERT_EQ(true, isExists);
        pFileSystem->CloseFile(pFile);

        // リネーム先のファイルを開いている場合は上書きリネームできません。
        nn::fssystem::save::Path path4;
        NNT_ASSERT_RESULT_SUCCESS(path4.Initialize("/dir3"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path4));

        nn::fssystem::save::Path pathDir3File2;
        NNT_ASSERT_RESULT_SUCCESS(pathDir3File2.Initialize("/dir3/file2"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathDir3File2, 1));
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFile, pathDir3File2, nn::fs::OpenMode_Write)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultPermissionDenied,
            pFileSystem->RenameDirectory(path2, path4)
        );

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path2));
        ASSERT_EQ(true, isExists);

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path4));
        ASSERT_EQ(true, isExists);

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathDir3File2));
        ASSERT_EQ(true, isExists);

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathDir2File1));
        ASSERT_EQ(true, isExists);

        nn::fssystem::save::Path path7;
        NNT_ASSERT_RESULT_SUCCESS(path7.Initialize("/dir3/file1"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path7));
        ASSERT_EQ(false, isExists);

        pFileSystem->CloseFile(pFile);

        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            pFileSystem->DeleteDirectoryRecursively(path1)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectoryRecursively(path2));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectoryRecursively(path4));
    }
#endif
} // NOLINT(impl/function_size)

// Ensure系機能のテストを行ないます。
void FileSystemTest::TestEnsure(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT
{
    bool isExists;

    nn::fssystem::save::Path path1;
    NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/dir1"));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path1));
    if( isExists )
    {
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectoryRecursively(path1));
    }
    else
    {
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNotFound,
            pFileSystem->DeleteDirectoryRecursively(path1)
        );
    }

    nn::fssystem::save::Path path2;
    NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/dir1/dir2/dir3"));
    NNT_ASSERT_RESULT_SUCCESS(EnsureDirectoryRecursively(pFileSystem, path2));

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path1));
    ASSERT_EQ(true, isExists);

    nn::fssystem::save::Path path3;
    NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/dir2"));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path3));
    ASSERT_EQ(false, isExists);

    nn::fssystem::save::Path path4;
    NNT_ASSERT_RESULT_SUCCESS(path4.Initialize("/dir1/dir2"));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path4));
    ASSERT_EQ(true, isExists);

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path2));
    ASSERT_EQ(true, isExists);

    nn::fssystem::save::Path path5;
    NNT_ASSERT_RESULT_SUCCESS(path5.Initialize("/dir1/dir2/dir4"));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path5));
    ASSERT_EQ(false, isExists);

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir1/dir2/dir3/"));
        NNT_ASSERT_RESULT_SUCCESS(EnsureDirectoryRecursively(pFileSystem, path));
    }
    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir1/dir2/dir3/../dir3/"));
        NNT_ASSERT_RESULT_SUCCESS(EnsureDirectoryRecursively(pFileSystem, path));
    }
    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir1///dir2/dir3/../dir3///"));
        NNT_ASSERT_RESULT_SUCCESS(EnsureDirectoryRecursively(pFileSystem, path));
    }
    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir1////dir2/./dir3/../dir4"));
        NNT_ASSERT_RESULT_SUCCESS(EnsureDirectoryRecursively(pFileSystem, path));
    }

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path5));
    ASSERT_EQ(true, isExists);

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/file1"));
        NNT_ASSERT_RESULT_SUCCESS(EnsureFileRecursively(pFileSystem, path, 1));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path));
        ASSERT_EQ(true, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path));
        ASSERT_EQ(false, isExists);
    }

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir2/dir4/file3"));
        NNT_ASSERT_RESULT_SUCCESS(EnsureFileRecursively(pFileSystem, path, 1));
    }

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path1));
    ASSERT_EQ(true, isExists);

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path4));
    ASSERT_EQ(true, isExists);

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir4"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path));
        ASSERT_EQ(false, isExists);
    }

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir2/dir4"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path));
        ASSERT_EQ(true, isExists);
    }

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir2/dir4/dir3"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path));
        ASSERT_EQ(false, isExists);
    }

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir6/../file3"));
        NNT_ASSERT_RESULT_SUCCESS(EnsureFileRecursively(pFileSystem, path, 1));
    }

    nn::fssystem::save::Path path6;
    NNT_ASSERT_RESULT_SUCCESS(path6.Initialize("/dir6"));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path6));
    ASSERT_EQ(true, isExists);

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path1));
    ASSERT_EQ(true, isExists);

    nn::fssystem::save::Path path7;
    NNT_ASSERT_RESULT_SUCCESS(path7.Initialize("/file3"));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path7));
    ASSERT_EQ(true, isExists);

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir6/file3"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path));
        ASSERT_EQ(false, isExists);
    }

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir6/./file4"));
        NNT_ASSERT_RESULT_SUCCESS(EnsureFileRecursively(pFileSystem, path, 1));
    }
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path1));
    ASSERT_EQ(true, isExists);


    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path6));
    ASSERT_EQ(true, isExists);

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path7));
    ASSERT_EQ(true, isExists);

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/file4"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path));
        ASSERT_EQ(false, isExists);
    }

    {
        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/dir6/file4"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path));
        ASSERT_EQ(true, isExists);
    }
} // NOLINT(impl/function_size)

namespace {

    static const int64_t LargeOffsetList[] = {
        0,
        static_cast<int64_t>(4) * 1024 * 1024 * 1024,
        static_cast<int64_t>(8) * 1024 * 1024 * 1024,
        static_cast<int64_t>(16) * 1024 * 1024 * 1024,
        static_cast<int64_t>(32) * 1024 * 1024 * 1024,
        static_cast<int64_t>(64) * 1024 * 1024 * 1024,
    };

    static const size_t LargeOffsetListLength = sizeof(LargeOffsetList) / sizeof(int64_t);

    static const int64_t LargeOffsetMax = LargeOffsetList[LargeOffsetListLength - 1];

}

// 4 GB を超えるオフセットでファイルにアクセスする（AllowAppend 無効時）
void TestLargeOffsetAccess(nn::fssystem::save::IFile* pFile, size_t bufferSize) NN_NOEXCEPT
{
    int64_t fileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSize));
    ASSERT_GE(fileSize, LargeOffsetMax + static_cast<int64_t>(bufferSize));

    // バッファ初期化
    std::unique_ptr<char> readBuffer(new char[bufferSize]);
    std::unique_ptr<char> writeBufferList[LargeOffsetListLength];
    for( auto& writeBuffer : writeBufferList )
    {
        writeBuffer.reset(new char[bufferSize]);
        nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), bufferSize);
    }

    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(LargeOffsetList[i], writeBufferList[i].get(), bufferSize));
    }
    NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());

    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(LargeOffsetList[i], readBuffer.get(), bufferSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), bufferSize);
    }

    // 範囲外への読み書きが失敗する
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
        pFile->WriteBytes(fileSize, writeBufferList[0].get(), bufferSize));
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        pFile->ReadBytes(fileSize + 1, readBuffer.get(), bufferSize));
}

// AllowAppend の動作テスト
void TestAllowAppendAccess(nn::fssystem::save::IFile* pFile, size_t bufferSize) NN_NOEXCEPT
{
    auto readBuffer = nnt::fs::util::AllocateBuffer(bufferSize);
    auto writeBuffer = nnt::fs::util::AllocateBuffer(bufferSize);

    int64_t fileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSize));

    // 範囲外への読み書き
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(fileSize, writeBuffer.get(), bufferSize));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(fileSize, readBuffer.get(), bufferSize));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), bufferSize);

    // 拡張されたサイズをテスト
    int64_t size = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
    EXPECT_GE(fileSize + static_cast<int64_t>(bufferSize), size);
}

// 4 GB を超えるオフセットでファイルにアクセスする共通テスト
void TestLargeOffsetAccess(nn::fssystem::save::IFileSystem* pFileSystem, const char* filePath) NN_NOEXCEPT
{
    static const size_t AccessSize = 1024;
    static const int64_t FileSize = LargeOffsetMax + AccessSize;

    nn::fssystem::save::Path saveFilePath;
    NNT_ASSERT_RESULT_SUCCESS(saveFilePath.Initialize(filePath));

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(saveFilePath, FileSize));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(saveFilePath));
    };

    {
        nn::fssystem::save::IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(
            &pFile,
            saveFilePath,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));
        NN_UTIL_SCOPE_EXIT
        {
            pFileSystem->CloseFile(pFile);
        };

        TestLargeOffsetAccess(pFile, AccessSize);
    }
    {
        // AllowAppend ありのテスト
        nn::fssystem::save::IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(
            &pFile,
            saveFilePath,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)));
        NN_UTIL_SCOPE_EXIT
        {
            pFileSystem->CloseFile(pFile);
        };

        TestAllowAppendAccess(pFile, AccessSize);
    }
}

