﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <numeric>
#include <nn/os/os_Thread.h>
#include <nn/util/util_BinTypes.h>
#include <nn/util/util_FormatString.h>
#include <nn/fssystem/save/fs_SaveDataFileSystemCore.h>

// SaveDataFileSystemCore::QueryTotalSize を呼び出すためにインクルードします。
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.impl.h>
#include <nn/fssystem/dbm/fs_FileObjectTemplate.impl.h>

#include <nn/fssystem/dbm/fs_FileSystemTemplate.impl.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"
#include "testFs_util_CommonFileSystemTests.h"

namespace {
    nn::fssystem::IBufferManager* GetBufferManager() NN_NOEXCEPT
    {
        static nn::fssystem::IBufferManager* s_pBufferManager = nullptr;

        if( s_pBufferManager == nullptr )
        {
            static const auto MaxCacheCount = 1024;
            static const auto SizeBlock = 16 * 1024;
            static const auto BufferManagerHeapSize = 6 * 1024 * 1024;
            static NN_ALIGNAS(4096) char s_BufferManagerHeap[BufferManagerHeapSize];
            static nn::fssystem::FileSystemBufferManager s_BufferManagerInstance;

            s_BufferManagerInstance.Initialize(
                MaxCacheCount,
                reinterpret_cast<uintptr_t>(s_BufferManagerHeap),
                sizeof(s_BufferManagerHeap),
                SizeBlock);
            s_pBufferManager = &s_BufferManagerInstance;
        }

        return s_pBufferManager;
    }
}

// テストクラスです。
template<typename TStorage>
class FsSaveDataFileSystemCoreTestTemplate : public ::testing::Test
{
public:
    typedef nn::fssystem::dbm::FileObjectTemplate<
        nn::fssystem::dbm::DirectoryName,
        nn::fssystem::dbm::FileName
    > FileObject;

public:
    // セーブデータのファイルシステム
    class SaveDataFileSystemSetup
    {
    public:
        // 初期化します。
        void Initialize(int64_t sizeMeta, int64_t sizeData, size_t sizeBlock) NN_NOEXCEPT
        {
            static const int64_t sizeControlArea =
                sizeof(nn::fssystem::save::SaveDataFileSystemCore::FileSystemHeader);

            m_StorageControlArea.Initialize(sizeControlArea);
            m_StorageMeta.Initialize(sizeMeta);
            m_StorageData.Initialize(sizeData);

            nn::Result result = m_FileSystem.Format(
                nn::fs::SubStorage(&m_StorageControlArea, 0, sizeControlArea),
                nn::fs::SubStorage(&m_StorageMeta, 0, sizeMeta),
                nn::fs::SubStorage(&m_StorageData, 0, sizeData),
                static_cast<uint32_t>(sizeBlock),
                GetBufferManager()
            );
            NN_ASSERT(result.IsSuccess());

            result = m_FileSystem.Initialize(
                nn::fs::SubStorage(&m_StorageControlArea, 0, sizeControlArea),
                nn::fs::SubStorage(&m_StorageMeta, 0, sizeMeta),
                nn::fs::SubStorage(&m_StorageData, 0, sizeData),
                GetBufferManager()
            );
            NN_ASSERT(result.IsSuccess());

            m_SizeTotal = sizeControlArea + sizeMeta + sizeData;
        }

        // 終了処理をします。
        void Finalize() NN_NOEXCEPT
        {
            m_FileSystem.Finalize();
        }

        // ストレージを取得します。
        TStorage& GetStorageControlArea() NN_NOEXCEPT
        {
            return m_StorageControlArea;
        }

        // ストレージを取得します。
        TStorage& GetStorageMeta() NN_NOEXCEPT
        {
            return m_StorageMeta;
        }

        // ストレージを取得します。
        TStorage& GetStorageData() NN_NOEXCEPT
        {
            return m_StorageData;
        }

        // ファイルシステムを取得します。
        nn::fssystem::save::SaveDataFileSystemCore& GetFileSystem() NN_NOEXCEPT
        {
            return m_FileSystem;
        }

    private:
        TStorage m_StorageControlArea;
        TStorage m_StorageMeta;
        TStorage m_StorageData;
        nn::fssystem::save::SaveDataFileSystemCore m_FileSystem;
        int64_t m_SizeTotal;
    };

};

class FsSaveDataFileSystemCoreTest : public FsSaveDataFileSystemCoreTestTemplate<nnt::fs::util::SafeMemoryStorage>
{
public:
    static nn::Result OpenFile(
                          FileObject* outFile,
                          nn::fssystem::save::SaveDataFileSystemCore* pFileSystem,
                          const char* path
                      ) NN_NOEXCEPT
    {
        return pFileSystem->m_FileSystem.OpenFile(outFile, path);
    }

    // 空きブロック数を取得します。
    static nn::Result CalcFreeListLength(
                          nn::fssystem::save::SaveDataFileSystemCore* pFileSystem,
                          uint32_t* outCount
                      ) NN_NOEXCEPT
    {
        return pFileSystem->CalcFreeListLength(outCount);
    }

    // ディレクトリのテストをします。
    void TestDirectory(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT;

    // ファイルのテストをします。
    void TestFile(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT;

    // アロケーションテーブルのテストをします。
    void TestAllocation(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT;

    // メタデータ領域と実データ領域が別ファイルの
    // セーブデータアーカイブに対するテストをします。
    void TestSeparateSaveData(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT
    {
        TestDirectory(pFileSystem);
        TestFile(pFileSystem);
        TestAllocation(pFileSystem);
    }
};

typedef FsSaveDataFileSystemCoreTestTemplate<nnt::fs::util::VirtualMemoryStorage> FsSaveDataFileSystemCoreLargeTest;

// セーブデータのマウントテスト
TEST_F(FsSaveDataFileSystemCoreTest, Mount)
{
    static const size_t TestLoop = 3;

    // メモリ不足でフォーマットできないはず
    {
        nnt::fs::util::SafeMemoryStorage storageMem;
        storageMem.Initialize(512 * 3);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultOutOfResource,
            nn::fssystem::save::SaveDataFileSystemCore::Format(
                nn::fs::SubStorage(&storageMem, 0, 512),
                nn::fs::SubStorage(&storageMem, 512, 512),
                nn::fs::SubStorage(&storageMem, 1024, 512),
                512,
                GetBufferManager()
            )
        );
    }

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( uint32_t sizeBlock = 128; sizeBlock <= 512; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( int32_t i = 0; i < TestLoop; ++i )
        {
            const int64_t sizeControlArea = nn::util::align_up(
                sizeof(nn::fssystem::save::SaveDataFileSystemCore::FileSystemHeader),
                sizeBlock
            );
            const int64_t sizeMeta =
                nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(sizeBlock, 2);
            const int64_t sizeData =
                nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(sizeBlock, 2);

            // 計算したサイズ未満であればフォーマット時にメモリが足りません。
            const int64_t sizeMeta2 = std::uniform_int_distribution<int64_t>(0, sizeMeta - 1)(mt);
            const int64_t sizeData2 = std::uniform_int_distribution<int64_t>(0, sizeData - 1)(mt);
            const size_t offsetMount = std::uniform_int_distribution<size_t>(0, 511)(mt);

            nnt::fs::util::SafeMemoryStorage storageBase;
            storageBase.Initialize(sizeControlArea + sizeMeta2 + sizeData2 + offsetMount);

            const int64_t offsetMeta = sizeControlArea;
            const int64_t offsetData = offsetMeta + sizeMeta2;

            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultOutOfResource,
                nn::fssystem::save::SaveDataFileSystemCore::Format(
                    nn::fs::SubStorage(&storageBase, 0, sizeControlArea),
                    nn::fs::SubStorage(&storageBase, offsetMeta, sizeMeta2),
                    nn::fs::SubStorage(&storageBase, offsetData, sizeData2),
                    sizeBlock,
                    GetBufferManager()
                )
            );
        }

        for( int32_t i = 0; i < TestLoop; ++i )
        {
            const size_t countAdditionalBlocks = std::uniform_int_distribution<>(100, 289)(mt);
            const size_t offsetMount = std::uniform_int_distribution<>(0, 511)(mt);
            const uint32_t countFile = std::uniform_int_distribution<>(1, 100)(mt);

            // 計算したサイズ以上であればフォーマットに成功します。
            // メタ情報より多い部分は実データとして割り当てされます。
            const int64_t sizeControlArea = nn::util::align_up(
                sizeof(nn::fssystem::save::SaveDataFileSystemCore::FileSystemHeader),
                sizeBlock
            );
            const int64_t sizeMeta = nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(
                sizeBlock,
                countAdditionalBlocks
            );
            const int64_t sizeData = nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(
                sizeBlock,
                countAdditionalBlocks
            );

            nnt::fs::util::SafeMemoryStorage storageBase;
            storageBase.Initialize(sizeControlArea + sizeMeta + sizeData + offsetMount);

            nn::fssystem::save::SaveDataFileSystemCore::Format(
                nn::fs::SubStorage(&storageBase, 0, sizeControlArea),
                nn::fs::SubStorage(&storageBase, sizeControlArea, sizeMeta),
                nn::fs::SubStorage(&storageBase, sizeControlArea + sizeMeta, sizeData),
                sizeBlock,
                GetBufferManager()
            );

            nn::fssystem::save::SaveDataFileSystemCore fileSystem;
            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    nn::fs::SubStorage(&storageBase, 0, sizeControlArea),
                    nn::fs::SubStorage(&storageBase, sizeControlArea, sizeMeta),
                    nn::fs::SubStorage(&storageBase, sizeControlArea + sizeMeta, sizeData),
                    GetBufferManager()
                )
            );

            // メタデータサイズ丁度でフォーマットした場合、GetFreeBlockCountは 0 を返します。
            uint32_t countsFreeBlocks;
            NNT_ASSERT_RESULT_SUCCESS(CalcFreeListLength(&fileSystem, &countsFreeBlocks));
            if( countAdditionalBlocks == 0 )
            {
                ASSERT_EQ(0, countsFreeBlocks);
            }

            bool isExists;

            // 最大個数まで作成します。
            for( size_t j = 0; j < countFile; ++j )
            {
                nn::fssystem::save::IFile* pFile;
                char buf[16];
                std::memset(buf, 0, sizeof(buf));
                nn::util::SNPrintf(buf, sizeof(buf), "/%08u", j);
                nn::fssystem::save::Path path;
                NNT_ASSERT_RESULT_SUCCESS(path.Initialize(buf));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(path, 0));
                NNT_ASSERT_RESULT_SUCCESS(
                    fileSystem.OpenFile(&pFile, path, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.HasFile(&isExists, path));
                ASSERT_EQ(true, isExists);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.HasDirectory(&isExists, path));
                ASSERT_EQ(false, isExists);
                fileSystem.CloseFile(pFile);
            }

            // 全て削除します。
            for( size_t j = 0; j < countFile; ++j )
            {
                char buf[16];
                std::memset(buf, 0, sizeof(buf));
                nn::util::SNPrintf(buf, sizeof(buf), "/%08u", j);
                nn::fssystem::save::Path path;
                NNT_ASSERT_RESULT_SUCCESS(path.Initialize(buf));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFileImpl(path));
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.HasFileImpl(&isExists, path));
                ASSERT_EQ(false, isExists);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.HasDirectoryImpl(&isExists, path));
                ASSERT_EQ(false, isExists);
            }

            // 空き容量が減っていることを確認します。
            uint32_t countsFreeBlocksLast;
            NNT_ASSERT_RESULT_SUCCESS(CalcFreeListLength(&fileSystem, &countsFreeBlocksLast));
            ASSERT_TRUE(countsFreeBlocksLast <= countsFreeBlocks);

            fileSystem.Finalize();
        }
    }
    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

// セーブデータのディレクトリテスト
TEST_F(FsSaveDataFileSystemCoreTest, Directory)
{
    static const int32_t CountDirectory = 10;
    static const int32_t CountEntry = 3;
    static const size_t SizeBlock = 512;
    static const size_t CountBlock = (40 * 1024) / SizeBlock;

    const int64_t sizeMeta =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock);
    const int64_t sizeData =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);

    SaveDataFileSystemSetup setup;
    setup.Initialize(sizeMeta, sizeData, SizeBlock);

    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

    {
        nn::fs::DirectoryEntry directoryEntry[CountEntry];
        nn::fssystem::save::IDirectory* pDir;
        nn::fssystem::save::Path pathRoot;
        NNT_ASSERT_RESULT_SUCCESS(pathRoot.Initialize("/"));

        bool isExists;
        int32_t resultCount;

        // ルートディレクトリを開く。まだ何もない
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDir, pathRoot));
        NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&resultCount, directoryEntry, CountEntry));
        ASSERT_EQ(0, resultCount);
        NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&resultCount, directoryEntry, CountEntry));
        ASSERT_EQ(0, resultCount);
        pFileSystem->CloseDirectory(pDir);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, pathRoot));
        ASSERT_EQ(true, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathRoot));
        ASSERT_EQ(false, isExists);

        // ディレクトリを1つ作ります。
        nn::fssystem::save::Path pathDirectory;
        NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/director"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDir, pathRoot));
        NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&resultCount, directoryEntry, CountEntry));
        ASSERT_EQ(1, resultCount);    // 一つ出来た
        ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, directoryEntry[0].directoryEntryType);
        ASSERT_EQ(0, std::memcmp("director", directoryEntry[0].name, 8));
        NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&resultCount, directoryEntry, CountEntry));
        ASSERT_EQ(0, resultCount);
        pFileSystem->CloseDirectory(pDir);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, pathDirectory));
        ASSERT_EQ(true, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathDirectory));
        ASSERT_EQ(false, isExists);

        // ファイルを一つ作ります。
        nn::fssystem::save::Path pathFile;
        NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/filefilefile"));
        nn::fssystem::save::IFile* pFile = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, 0));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDir, pathRoot));
        NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&resultCount, directoryEntry, CountEntry));
        ASSERT_EQ(2, resultCount);    // 2つ出来た
        ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, directoryEntry[0].directoryEntryType);
        ASSERT_EQ(0, std::memcmp("director", directoryEntry[0].name, 8));
        ASSERT_EQ(0, std::memcmp("filefilefile", directoryEntry[1].name, 12));
        NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&resultCount, directoryEntry, CountEntry));
        ASSERT_EQ(0, resultCount);
        pFileSystem->CloseDirectory(pDir);
        pFileSystem->CloseFile(pFile);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathFile));
        ASSERT_EQ(true, isExists);

        // ディレクトリとファイルを消去します。
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidPathForOperation,
            pFileSystem->DeleteFile(pathDirectory)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidPathForOperation,
            pFileSystem->DeleteDirectory(pathFile)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathDirectory));
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNotFound,
            pFileSystem->DeleteFile(pathFile)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNotFound,
            pFileSystem->DeleteDirectory(pathDirectory)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, pathDirectory));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, pathFile));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, pathRoot));
        ASSERT_EQ(true, isExists);
    }

    // ディレクトリのテストを行います。
    {
        int32_t resultCount;
        nn::fs::DirectoryEntry directoryEntry[CountEntry];
        nn::fssystem::save::IDirectory* pDir;
        nn::fssystem::save::Path pathRoot;
        NNT_ASSERT_RESULT_SUCCESS(pathRoot.Initialize("/"));

        // ディレクトリを作成します。
        for( int i = 0; i < 100; ++i )
        {
            char name[32];
            std::sprintf(name, "/dir%d", i);
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize(name));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));
        }

        // ディレクトリの列挙テストを行います。
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDir, pathRoot));
            for( int i = 1; i < 99; i += 3 )
            {
                NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&resultCount, directoryEntry, CountEntry));
                ASSERT_EQ(3, resultCount);
            }
            NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&resultCount, directoryEntry, CountEntry));
            ASSERT_EQ(CountDirectory % 3, resultCount);
            NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&resultCount, directoryEntry, CountEntry));
            ASSERT_EQ(0, resultCount);

            pFileSystem->CloseDirectory(pDir);
        }

        for( int i = 0; i < CountDirectory; ++i )
        {
            char name[32];
            std::sprintf(name, "/dir%d", i);
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize(name));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathDirectory));
        }
    }

    // 先頭と末尾の破壊チェック用
    setup.Finalize();
} // NOLINT(impl/function_size)

// セーブデータのファイルテスト
TEST_F(FsSaveDataFileSystemCoreTest, File)
{
    static const size_t SizeBlock = 512;
    static const size_t CountBlock = (40 * 1024) / SizeBlock;

    const int64_t sizeMeta =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock);
    const int64_t sizeData =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);

    SaveDataFileSystemSetup setup;
    setup.Initialize(sizeMeta, sizeData, SizeBlock);

    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

    // ファイルの処理
    {
        int64_t size;
        bool isExists;

        nn::fssystem::save::Path file1;
        NNT_ASSERT_RESULT_SUCCESS(file1.Initialize("/file1"));

        // 存在しないファイルを開く
        nn::fssystem::save::IFile* pFile = nullptr;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNotFound,
            pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Read)
        );
        ASSERT_EQ(nullptr, pFile);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNotFound,
            pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write)
        );
        ASSERT_EQ(nullptr, pFile);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(false, isExists);

        // 誤った指定(モード指定テスト)
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOperationForOpenMode,
            pFileSystem->OpenFile(&pFile, file1, 0)
        );
        ASSERT_EQ(nullptr, pFile);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(false, isExists);

        // 正しい指定(モード指定テスト)
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(file1, 0));
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write)
        );
        ASSERT_NE(nullptr, pFile);
        pFileSystem->CloseFile(pFile);
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Read)
        );
        ASSERT_NE(nullptr, pFile);
        pFileSystem->CloseFile(pFile);
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write)
        );
        ASSERT_NE(nullptr, pFile);

        // 開いているファイルの生存確認
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(true, isExists);

        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAlreadyExists,
            pFileSystem->CreateFile(file1, 10)
        );

        // サイズの取得と設定
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        int64_t sizeOld = size;
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(0, size);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultOutOfResource,
            pFile->SetSize(1024 * 1024 * 1024)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(1));
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(1, size);
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeOld / 2));
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeOld));
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(sizeOld, size);
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(0));
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(0, size);
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(16 * 1024));

        // ファイルを閉じる
        pFileSystem->CloseFile(pFile);

        // サイズ設定を再度する
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write));
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(16 * 1024, size);
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(0));
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(0, size);
        pFileSystem->CloseFile(pFile);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(true, isExists);

        // リード状態ではサイズを設定できません
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Read));
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(0, size);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOperationForOpenMode,
            pFile->SetSize(16)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(0, size);
        pFileSystem->CloseFile(pFile);

        nnt::fs::util::Vector<char> bufWrite(128);
        nnt::fs::util::Vector<char> bufRead(128);
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(
                &pFile,
                file1,
                nn::fs::OpenMode_Write | nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));

        // 0 バイトのファイルに 0 バイトの書き込みます。
        ASSERT_EQ(0, size);
        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, &bufWrite[0], 0));

        // 範囲外への書き込み
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(1, &bufWrite[0], 0)
        );

        // 0 バイトのファイルから 0 バイトの読み込みます。
        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &bufRead[0], 0));

        // 範囲外の読み込み
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            pFile->ReadBytes(1, &bufRead[0], 0)
        );

        // ファイルサイズを30バイトに設定しました。
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(30));
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(30, size);

        // 範囲内のアクセス
        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(28, &bufWrite[0], 2));
        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(28, &bufRead[0], 2));
        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(1, &bufWrite[0], 29));
        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(1, &bufRead[0], 29));

        // 範囲を少々超えるアクセス -> 自動拡張
        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(28, &bufWrite[0], 4));
        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(28, &bufRead[0], 4));
        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, &bufWrite[0], 31));
        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &bufRead[0], 31));

        // 完全に範囲外のアクセス
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            pFile->ReadBytes(33, &bufRead[0], 1)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            pFile->ReadBytes(34, &bufRead[0], 1)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(33, &bufWrite[0], 1)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(34, &bufWrite[0], 1)
        );
        // Write によって拡張されたのでアクセス可能に
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(33, &bufRead[0], 1)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->ReadBytes(34, &bufRead[0], 1)
        );

        pFileSystem->CloseFile(pFile);

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, file1));
        ASSERT_EQ(true, isExists);
    }

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // CreateFile のテストを行います。
    for( int i = 0; i < 100; ++i )
    {
        nn::fssystem::save::IFile* pFile3;
        char buf[16];
        std::memset(buf, 0, sizeof(buf));
        nn::util::SNPrintf(
            buf, sizeof(buf), "/%08u", std::uniform_int_distribution<>(0, 99999)(mt));
        nn::fssystem::save::Path path3;
        NNT_ASSERT_RESULT_SUCCESS(path3.Initialize(buf));
        int64_t size3 = std::uniform_int_distribution<>(0, 23)(mt);
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(path3, size3));

        // ファイルサイズを確認します。
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&pFile3, path3, nn::fs::OpenMode_Read));
        int64_t size;
        NNT_ASSERT_RESULT_SUCCESS(pFile3->GetSize(&size));
        ASSERT_EQ(size3, size);
        bool isExists;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.HasFile(&isExists, path3));
        ASSERT_EQ(true, isExists);
        fileSystem.CloseFile(pFile3);

        // 存在しているファイルは作成できないことを確認します。
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAlreadyExists,
            fileSystem.CreateFile(path3, 100)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAlreadyExists,
            fileSystem.CreateFile(path3, 100)
        );

        // ファイルを削除します。
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile(path3));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.HasFile(&isExists, path3));
        ASSERT_EQ(false, isExists);
    }

    // 先頭と末尾の破壊チェック用
    setup.Finalize();
} // NOLINT(impl/function_size)

// アロケーションテーブルのテスト
TEST_F(FsSaveDataFileSystemCoreTest, AllocationTable)
{
    // 初期化周りの確認
    static const size_t SizeBlock = 512;
    static const size_t CountBlock = (40 * 1024) / SizeBlock;
    static const uint32_t CountFile = 64;

    const int64_t sizeMeta =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock);
    const int64_t sizeData =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);

    SaveDataFileSystemSetup setup;
    setup.Initialize(sizeMeta, sizeData, SizeBlock);

    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

    // 複数のファイルでアローケーションテーブルを取りあう
    {
        nn::fssystem::save::IFile* pFile[CountFile];

        for( uint32_t m = 0; m < CountFile; ++m )
        {
            char buf[32];
            std::memset(buf, 0, sizeof(buf));
            std::sprintf(buf, "/file%u", m);
            nn::fssystem::save::Path pathCur;
            NNT_ASSERT_RESULT_SUCCESS(pathCur.Initialize(buf));
            nn::Result result = pFileSystem->CreateFile(pathCur, 0);
            if( result.IsFailure() )
            {
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAlreadyExists, result);
            }
            NNT_ASSERT_RESULT_SUCCESS(
                pFileSystem->OpenFile(&pFile[m], pathCur, nn::fs::OpenMode_Write));
        }

        bool isContinue = true;
        size_t sizeAllocate = 0;
        while( isContinue )
        {
            for( uint32_t m = 0; m < CountFile; ++m )
            {
                nn::Result result = pFile[m]->SetSize(sizeAllocate);
                if( result.IsFailure() )
                {
                    isContinue = false;
                    break;
                }
            }
            sizeAllocate += SizeBlock;
        }

        // 合計サイズ
        sizeAllocate = 0;
        for( uint32_t m = 0; m < CountFile; ++m )
        {
            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(pFile[m]->GetSize(&size));
            sizeAllocate += static_cast<size_t>(size);
        }

        // 各ファイルを特定のデータで埋めます。
        nnt::fs::util::Vector<char> fillBuf(32 * 1024);
        nnt::fs::util::Vector<char> baseBuf(32 * 1024);
        for( uint32_t m = 0; m < CountFile; ++m )
        {
            char* p = &fillBuf[0];
            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(pFile[m]->GetSize(&size));
            for( uint32_t n = 0; n < size; ++n )
            {
                p[n] = (m + n) & 0xFF;
            }
            NNT_ASSERT_RESULT_SUCCESS(pFile[m]->WriteBytes(0, p, static_cast<size_t>(size)));
        }

        // ファイルを全て閉じます。
        for( uint32_t m = 0; m < CountFile; ++m )
        {
            pFileSystem->CloseFile(pFile[m]);
        }

        // ファイルを読み込み専用で開きなおします。
        for( uint32_t m = 0; m < CountFile; ++m )
        {
            char buf[32];
            std::memset(buf, 0, sizeof(buf));
            std::sprintf(buf, "/file%u", m);
            nn::fssystem::save::Path pathCur;
            NNT_ASSERT_RESULT_SUCCESS(pathCur.Initialize(buf));
            NNT_ASSERT_RESULT_SUCCESS(
                pFileSystem->OpenFile(&pFile[m], pathCur, nn::fs::OpenMode_Read));
        }

        for( uint32_t m = 0; m < CountFile; ++m )
        {
            char* pBase = &baseBuf[0];
            char* pFill = &fillBuf[0];
            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(pFile[m]->GetSize(&size));

            // 書き込まれているデータを生成します
            for( uint32_t n = 0; n < size; ++n )
            {
                pFill[n] = (m + n) & 0xFF;
            }

            // ファイルを後ろから読み取ります。
            for( uint32_t n = 0; n < size; ++n )
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile[m]->ReadBytes(size - n - 1, &pBase[size - n - 1], 1));
            }
            ASSERT_EQ(0, std::memcmp(pBase, pFill, static_cast<size_t>(size)));
            // ファイルを前から読み取ります。
            for( uint32_t n = 0; n < size; ++n )
            {
                NNT_ASSERT_RESULT_SUCCESS(pFile[m]->ReadBytes(n, &pBase[n], 1));
            }
            ASSERT_EQ(0, std::memcmp(pBase, pFill, static_cast<size_t>(size)));
            // ファイルを一括で読み込みます。
            NNT_ASSERT_RESULT_SUCCESS(pFile[m]->ReadBytes(0, pBase, static_cast<size_t>(size)));
            ASSERT_EQ(0, std::memcmp(pBase, pFill, static_cast<size_t>(size)));
        }

        // ファイルを全て閉じます。
        for( uint32_t m = 0; m < CountFile; ++m )
        {
            pFileSystem->CloseFile(pFile[m]);

            char buf[32];
            std::memset(buf, 0, sizeof(buf));
            std::sprintf(buf, "/file%u", m);
            nn::fssystem::save::Path pathCur;
            NNT_ASSERT_RESULT_SUCCESS(pathCur.Initialize(buf));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathCur));
        }
    }

    // 先頭と末尾の破壊チェック用
    setup.Finalize();
} // NOLINT(impl/function_size)

// セーブデータの操作
TEST_F(FsSaveDataFileSystemCoreTest, Operation)
{
    static const size_t SizeBlock = 1024;
    static const size_t CountBlock = (256 * 1024) / SizeBlock;

    const int64_t sizeMeta =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock);
    const int64_t sizeData =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);

    SaveDataFileSystemSetup setup;
    setup.Initialize(sizeMeta, sizeData, SizeBlock);

    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

    nn::fssystem::save::IFile *pFile1;
    nn::fssystem::save::IFile *pFile2;
    nn::fssystem::save::IDirectory* pDirectory;
    nn::fssystem::save::Path path1;
    NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/file1"));
    nn::fssystem::save::Path path2;
    NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/file2"));
    nn::fssystem::save::Path path3;
    NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/dir1"));
    nn::fssystem::save::Path path4;
    NNT_ASSERT_RESULT_SUCCESS(path4.Initialize("/dir2"));
    nn::fssystem::save::Path path5;
    NNT_ASSERT_RESULT_SUCCESS(path5.Initialize("/dir1/dir1"));
    nn::fssystem::save::Path path6;
    NNT_ASSERT_RESULT_SUCCESS(path6.Initialize("/dir1/dir1/dir1"));
    nn::fssystem::save::Path path7;
    NNT_ASSERT_RESULT_SUCCESS(path7.Initialize("/dir1/dir2"));
    nn::fssystem::save::Path path8;
    NNT_ASSERT_RESULT_SUCCESS(path8.Initialize("/dir2/dir2"));
    nn::fssystem::save::Path path9;
    NNT_ASSERT_RESULT_SUCCESS(path9.Initialize("/dir1/dir1/file"));
    nn::fssystem::save::Path path10;
    NNT_ASSERT_RESULT_SUCCESS(path10.Initialize("/dir2/dir2/file"));

    bool isExists;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path1));
    ASSERT_EQ(false, isExists);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path5));
    ASSERT_EQ(false, isExists);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path10));
    ASSERT_EQ(false, isExists);

    // "/file1", "/file2" を同時に書き込みモードで開きます。
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path1, 0));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile1, path1, nn::fs::OpenMode_Write));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path2, 0));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile2, path2, nn::fs::OpenMode_Write));
    pFileSystem->CloseFile(pFile2);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path2));
    pFileSystem->CloseFile(pFile1);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path1));
    ASSERT_EQ(true, isExists);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path2));
    ASSERT_EQ(false, isExists);

    static const int Mode[3] =
    {
        nn::fs::OpenMode_Write | nn::fs::OpenMode_Read,
        nn::fs::OpenMode_Write,
        nn::fs::OpenMode_Read
    };

    for( uint32_t i = 0; i < sizeof(Mode) / sizeof(Mode[0]); ++i )
    {
        nn::fssystem::save::Path path = path1;

        // /file1をいろいろなモードでオープンします
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile1, path, Mode[i]));

        // オープン済みのファイルに対し、書き込みモードでのオープンは失敗する
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultPermissionDenied,
            pFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Write)
        );

        // オープン済みのファイルに対し、削除、リネームは失敗する
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultPermissionDenied,
            pFileSystem->DeleteFile(path)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultPermissionDenied,
            pFileSystem->RenameFile(path, path2)
        );

        if( (Mode[i] & nn::fs::OpenMode_Write) != 0 )
        {
            // 書き込み属性で既に開かれているファイルをさらに開こうとしたら失敗
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultPermissionDenied,
                pFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Read)
            );
        }
        else
        {
            // 読み込み属性(書き込みなし)で開かれているファイルをさらに開こうとしたら成功
            NNT_ASSERT_RESULT_SUCCESS(
                pFileSystem->OpenFile(&pFile2, path, nn::fs::OpenMode_Read));

            pFileSystem->CloseFile(pFile2);
        }

        // Closeは成功する
        pFileSystem->CloseFile(pFile1);
    }

    {
        // 開いているファイル、フォルダに対してリネームなど
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path3));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path3));
        ASSERT_EQ(true, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path5));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path5));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path3));
        ASSERT_EQ(true, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path5));
        ASSERT_EQ(true, isExists);

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path9, 0));
        for( uint32_t i = 0; i < sizeof(Mode) / sizeof(Mode[0]); ++i )
        {
            // /dir1/dir1/file をいろいろなモードでオープンします。
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile1, path9, Mode[i]));

            // /dir1 をリネームしようとしますが失敗します。
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultPermissionDenied,
                pFileSystem->RenameDirectory(path5, path7)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultPermissionDenied,
                pFileSystem->RenameDirectory(path3, path4)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDirectoryNotFound,
                pFileSystem->RenameDirectory(path4, path3)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDirectoryNotFound,
                pFileSystem->RenameDirectory(path7, path5)
            );

            // /dir1 はすでに存在しているので作成できません。
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->CreateDirectory(path3)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->CreateDirectory(path5)
            );

            pFileSystem->CloseFile(pFile1);
        }

        // /dir1/dir1/file のClose後は、ディレクトリ、ファイルのリネームや削除が成功する
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameDirectory(path5, path7));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path5));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path7));
        ASSERT_EQ(true, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameDirectory(path3, path4));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path3));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path4));
        ASSERT_EQ(true, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path10));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasFile(&isExists, path10));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(path8));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(path4));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path8));
        ASSERT_EQ(false, isExists);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->HasDirectory(&isExists, path4));
        ASSERT_EQ(false, isExists);
    }

    // ディレクトリの列挙
    {
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path3));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path5));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDirectory, path5));

        // 列挙中、中間のディレクトリはリネームできません。
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultPermissionDenied,
            pFileSystem->RenameDirectory(path5, path7)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultPermissionDenied,
            pFileSystem->RenameDirectory(path3, path4)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultDirectoryNotFound,
            pFileSystem->RenameDirectory(path4, path3)
        );

        // 列挙中、ディレクトリの新規作成は可能。
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path6));
        // すでに存在しているディレクトリは作成できません。
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAlreadyExists,
            pFileSystem->CreateDirectory(path3)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAlreadyExists,
            pFileSystem->CreateDirectory(path5)
        );
        // 削除も成功します。
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(path6));
        // 列挙中、ファイルの新規作成は可能です。
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path9, 0));
            NNT_ASSERT_RESULT_SUCCESS(
                pFileSystem->OpenFile(&pFile1, path9, nn::fs::OpenMode_Write));
            pFileSystem->CloseFile(pFile1);
            NNT_ASSERT_RESULT_SUCCESS(
                pFileSystem->OpenFile(&pFile1, path9, nn::fs::OpenMode_Read));
            pFileSystem->CloseFile(pFile1);
            pFileSystem->CloseDirectory(pDirectory);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path9));
        }

        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path9, 0));
            NNT_ASSERT_RESULT_SUCCESS(
                pFileSystem->OpenFile(&pFile1, path9, nn::fs::OpenMode_Write));
            pFileSystem->CloseFile(pFile1);
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path2, 0));
            NNT_ASSERT_RESULT_SUCCESS(
                pFileSystem->OpenFile(&pFile1, path2, nn::fs::OpenMode_Write));
            pFileSystem->CloseFile(pFile1);
        }

        // 列挙中、リネームや削除が出来ない
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDirectory, path5));

#if 0
        // ディレクトリの上書きリネームは可能。
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(path2, path9));

        // ディレクトリの上書きリネームは可能。
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(path9, path2));

        pFileSystem->CloseDirectory(pDirectory);

        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path2));
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNotFound,
            pFileSystem->DeleteFile(path9)
        );
#else
        // ディレクトリの上書きリネームは禁止。
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAlreadyExists,
            pFileSystem->RenameFile(path2, path9)
        );

        // ディレクトリの上書きリネームは禁止。
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAlreadyExists,
            pFileSystem->RenameFile(path9, path2)
        );
        pFileSystem->CloseDirectory(pDirectory);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path2));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path9));
#endif
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(path5));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(path3));
    }

    // リード多重、リード+ライト
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile1, path1, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile2, path1, nn::fs::OpenMode_Read));
    pFileSystem->CloseFile(pFile2);
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultPermissionDenied,
        pFileSystem->OpenFile(&pFile2, path1, nn::fs::OpenMode_Write)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultPermissionDenied,
        pFileSystem->DeleteFile(path1)
    );
    pFileSystem->CloseFile(pFile1);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path1));

    setup.Finalize();
} // NOLINT(impl/function_size)

namespace {

struct ThreadTestArg
{
    nn::fssystem::save::IFileSystem* pFileSystem;
    uint32_t threadId;
    std::mt19937::result_type seed;
};

}

// 複数のスレッドから同時にファイルシステムにアクセスします。
static void ThreadFunction(void* ptr) NN_NOEXCEPT
{
    static const size_t TestBufSize = 128;

    const ThreadTestArg* pArg = reinterpret_cast<ThreadTestArg*>(ptr);
    nn::fssystem::save::IFileSystem* pFileSystem = pArg->pFileSystem;
    uint32_t threadId = pArg->threadId;
    std::mt19937 mt(pArg->seed);
    char workBuf1[TestBufSize];
    char workBuf2[TestBufSize];
    char buf[32];
    char buf2[32];
    char buf3[32];

    // テストバッファを乱数で埋めます。
    for( int32_t j = 0; j < TestBufSize; ++j )
    {
        workBuf1[j] = static_cast<char>(std::uniform_int_distribution<>(0, 0xFF)(mt));
    }
    for( int32_t i = 0; i < 100; ++i )
    {
        if( (threadId == 0) && ((i % 10) == 0) )
        {
            NN_SDK_LOG(".");
        }

        nn::fssystem::save::IFile* pFile;
        nn::fssystem::save::IDirectory* pDir;

        switch( std::uniform_int_distribution<>(0, 5)(mt) )
        {
        case 0:
            {
                std::memset(buf, 0, sizeof(buf));
                std::sprintf(buf, "/file%u", threadId);
                nn::fssystem::save::Path pathCur;
                NNT_ASSERT_RESULT_SUCCESS(pathCur.Initialize(buf));

                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathCur, 0));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathCur, nn::fs::OpenMode_Write));
                size_t size = std::uniform_int_distribution<size_t>(0, TestBufSize - 1)(mt);
                NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(size));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, workBuf1, size));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathCur, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->ReadBytes(0, workBuf2, size));
                ASSERT_EQ(0, std::memcmp(workBuf1, workBuf2, size));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathCur));
            }
            break;
        case 1:
            {
                std::memset(buf, 0, sizeof(buf));
                std::sprintf(buf, "/dir%u-1", threadId);
                std::memset(buf2, 0, sizeof(buf));
                std::sprintf(buf2, "/dir%u-2", threadId);
                nn::fssystem::save::Path pathCur1;
                NNT_ASSERT_RESULT_SUCCESS(pathCur1.Initialize(buf));
                nn::fssystem::save::Path pathCur2;
                NNT_ASSERT_RESULT_SUCCESS(pathCur2.Initialize(buf2));
                nn::fssystem::save::Path pathRoot;
                NNT_ASSERT_RESULT_SUCCESS(pathRoot.Initialize("/"));
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathCur1));
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathCur2));
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDir, pathRoot));
                int32_t doneCount;
                nn::fs::DirectoryEntry entry[2];
                NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&doneCount, entry, 2));
                ASSERT_EQ(2, doneCount);
                pFileSystem->CloseDirectory(pDir);
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCur1));
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCur2));
            }
            break;
        case 2:
            {
                std::memset(buf, 0, sizeof(buf));
                std::sprintf(buf, "/dir%u-1", threadId);
                std::memset(buf2, 0, sizeof(buf));
                std::sprintf(buf2, "/dir%u-2", threadId);
                nn::fssystem::save::Path pathCur1;
                NNT_ASSERT_RESULT_SUCCESS(pathCur1.Initialize(buf));
                nn::fssystem::save::Path pathCur2;
                NNT_ASSERT_RESULT_SUCCESS(pathCur2.Initialize(buf2));
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathCur1));
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameDirectory(pathCur1, pathCur2));
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathCur2));
            }
            break;
        case 3:
            {
                std::memset(buf, 0, sizeof(buf));
                std::sprintf(buf, "/file%u-1", threadId);
                std::memset(buf2, 0, sizeof(buf));
                std::sprintf(buf2, "/file%u-2", threadId);
                nn::fssystem::save::Path pathCur1;
                NNT_ASSERT_RESULT_SUCCESS(pathCur1.Initialize(buf));
                nn::fssystem::save::Path pathCur2;
                NNT_ASSERT_RESULT_SUCCESS(pathCur2.Initialize(buf2));
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathCur1, 0));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathCur1, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(TestBufSize));
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, workBuf1, TestBufSize));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(pathCur1, pathCur2));

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathCur2, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, workBuf2, TestBufSize));
                ASSERT_EQ(0, std::memcmp(workBuf1, workBuf2, TestBufSize));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathCur2));
            }
            break;
        case 4:
            {
                std::memset(buf, 0, sizeof(buf));
                std::sprintf(buf, "/dir%u", threadId);
                nn::fssystem::save::Path pathDirectory;
                NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize(buf));
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));

                std::memset(buf2, 0, sizeof(buf2));
                std::sprintf(buf2, "/dir%u/file%u", threadId, threadId);
                nn::fssystem::save::Path pathCur;
                NNT_ASSERT_RESULT_SUCCESS(pathCur.Initialize(buf2));

                std::memset(buf3, 0, sizeof(buf3));
                std::sprintf(buf3, "/file%u", threadId);
                nn::fssystem::save::Path pathRename;
                NNT_ASSERT_RESULT_SUCCESS(pathRename.Initialize(buf3));

                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathCur, 0));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathCur, nn::fs::OpenMode_Write));
                size_t size = std::uniform_int_distribution<size_t>(0, TestBufSize - 1)(mt);
                NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(size));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->WriteBytes(0, workBuf1, size));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, pathCur, nn::fs::OpenMode_Read));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFile->ReadBytes(0, workBuf2, size));
                ASSERT_EQ(0, std::memcmp(workBuf1, workBuf2, size));
                pFileSystem->CloseFile(pFile);

                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->RenameFile(pathCur, pathRename));

                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathDirectory));
                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathRename));
            }
            break;
        case 5:
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

// 複数のスレッドから同時にファイルシステムにアクセスするテスト
TEST_F(FsSaveDataFileSystemCoreTest, MultiThread)
{
    static const size_t StackSize = 32 * 1024;
    static const int ThreadCount = 8;

    NN_OS_ALIGNAS_THREAD_STACK static char s_ThreadStack[StackSize * ThreadCount] = {};

    ThreadTestArg args[ThreadCount];
    nn::os::ThreadType threads[ThreadCount];

    for( int sizeBlock = 128; sizeBlock <= 256; sizeBlock *= 8 )
    {
        const int countBlock = (256 * 1024) / sizeBlock;

        const int64_t sizeMeta =
            nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(sizeBlock, countBlock);
        const int64_t sizeData =
            nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(sizeBlock, countBlock);

        SaveDataFileSystemSetup setup;
        setup.Initialize(sizeMeta, sizeData, sizeBlock);

        nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
        nn::fssystem::save::IFileSystem* pFileSystem =
            static_cast<nn::fssystem::save::IFileSystem*>(&fileSystem);

        // 複数のスレッドから同時にファイルシステムにアクセスします。
        for( int i = 0; i < ThreadCount; ++i )
        {
            args[i].pFileSystem = pFileSystem;
            args[i].threadId = i;
            args[i].seed = nnt::fs::util::GetRandomSeed();

            NNT_ASSERT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &threads[i],
                    ThreadFunction,
                    &args[i],
                    s_ThreadStack + StackSize * i,
                    StackSize,
                    nn::os::DefaultThreadPriority - i
                )
            );
            nn::os::StartThread(&threads[i]);
        }

        // スレッドの後始末
        for( auto& thread : threads )
        {
            nn::os::WaitThread(&thread);
            nn::os::DestroyThread(&thread);
        }

        NN_SDK_LOG("\n");

        setup.Finalize();
    }
}

// ファイルを複数開くテスト
TEST_F(FsSaveDataFileSystemCoreTest, ParallelOpen)
{
    static const size_t SizeBlock = 4096;
    static const size_t CountBlock = (1 * 1024 * 1024) / SizeBlock;
    static const size_t CountFile = 400;

    const int64_t sizeMeta =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock);
    const int64_t sizeData =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);

    SaveDataFileSystemSetup setup;
    setup.Initialize(sizeMeta, sizeData, SizeBlock);

    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

    nnt::fs::util::Vector<nn::fssystem::save::IFile*> file(CountFile);

    for( size_t i = 0; i < CountFile; ++i )
    {
        const size_t directoryId = i / 120;
        const size_t fileId = i;// - directoryId * 120;

        // ディレクトリ作成
        char buf[32];
        nn::util::SNPrintf(buf, 32, "/%X", directoryId);
        nn::fssystem::save::Path pathDirectory;
        NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize(buf));
        if( (i % 120) == 0 )
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                pFileSystem->CreateDirectory(pathDirectory)
            );
        }

        // ファイルの作成
        nn::util::SNPrintf(buf, 32, "/%X/%X", directoryId, fileId);
        nn::fssystem::save::Path pathFile;
        NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize(buf));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, 0));
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&file[i], pathFile, nn::fs::OpenMode_Write)
        );
    }

    // 一斉に閉じます
    for( size_t i = 0; i < CountFile; ++i )
    {
        pFileSystem->CloseFile(file[i]);
    }

    for( size_t i = 0; i < CountFile; ++i )
    {
        const size_t directoryId = i / 120;
        const size_t fileId = i;// - directoryId * 120;

        // ファイルを消します
        char buf[32];
        nn::util::SNPrintf(buf, 32, "/%X/%X", directoryId, fileId);
        nn::fssystem::save::Path pathFile;
        NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize(buf));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));

        // ディレクトリを削除します
        nn::util::SNPrintf(buf, 32, "/%X", directoryId);
        nn::fssystem::save::Path pathDirectory;
        NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize(buf));
        if( ((i % 120) == 119) || (i == CountFile - 1) )
        {
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(pathDirectory));
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDirectoryNotEmpty,
                pFileSystem->DeleteDirectory(pathDirectory)
            );
        }
    }
    setup.Finalize();
}

// セーブデータの拡張テスト
TEST_F(FsSaveDataFileSystemCoreTest, Expand)
{
    static const size_t SizeBlock = 16 * 1024;
    static const size_t SizeFile = 2 * 1024 * 1024;
    static const size_t CountBlock = SizeFile / SizeBlock;

    const int64_t sizeMeta1 =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock + 2);
    const int64_t sizeData1 =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock + 2);

    SaveDataFileSystemSetup setup;
    setup.Initialize(sizeMeta1, sizeData1, SizeBlock);

    nnt::fs::util::Vector<char> bufWrite(SizeFile, 'A');
    nnt::fs::util::Vector<char> bufRead(SizeFile);

    nn::fssystem::save::SaveDataFileSystemCore& fileSystem1 = setup.GetFileSystem();

    // 最大サイズのファイルを作成
    nn::fssystem::save::Path path1;
    {
        nn::fssystem::save::IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/file1"));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem1.CreateFile(path1, SizeFile));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem1.OpenFile(&pFile, path1, nn::fs::OpenMode_Write));
        NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, bufWrite.data(), SizeFile));
        fileSystem1.CloseFile(pFile);
        bool isExists;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem1.HasFile(&isExists, path1));
        ASSERT_EQ(true, isExists);
    }

    // 追加のファイルは作成できない
    {
        nn::fssystem::save::Path path2;
        NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/file2"));
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAllocationTableFull,
            fileSystem1.CreateFile(path2, 1)
        );
    }

    const int64_t sizeControlArea =
        sizeof(nn::fssystem::save::SaveDataFileSystemCore::FileSystemHeader);
    const int64_t sizeMeta2 =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock * 2 + 2);
    const int64_t sizeData2 =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock * 2 + 2);

    // 拡張用のストレージを作成
    nnt::fs::util::SafeMemoryStorage storageControlArea;
    storageControlArea.Initialize(sizeControlArea);
    nnt::fs::util::SafeMemoryStorage storageMeta;
    storageMeta.Initialize(sizeMeta2);
    nnt::fs::util::SafeMemoryStorage storageData;
    storageData.Initialize(sizeData2);

    // ヘッダの正当性チェック
    {
        std::memset(storageControlArea.GetBuffer(), 0, static_cast<size_t>(sizeControlArea));

        // マジックコードチェック
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultIncorrectSaveDataFileSystemMagicCode,
            nn::fssystem::save::SaveDataFileSystemCore::ExpandControlArea(
                nn::fs::SubStorage(&storageControlArea, 0, sizeControlArea),
                sizeData2
            )
        );

        *reinterpret_cast<int32_t*>(storageControlArea.GetBuffer()) =
            NN_UTIL_CREATE_SIGNATURE_4('S','A','V','E');

        // バージョン番号チェック
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultUnsupportedVersion,
            nn::fssystem::save::SaveDataFileSystemCore::ExpandControlArea(
                nn::fs::SubStorage(&storageControlArea, 0, sizeControlArea),
                sizeData2
            )
        );
    }

    // 拡張前のデータをコピー
    NNT_ASSERT_RESULT_SUCCESS(
        storageControlArea.Write(
            0,
            setup.GetStorageControlArea().GetBuffer(),
            static_cast<size_t>(sizeControlArea)
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(
        storageMeta.Write(0, setup.GetStorageMeta().GetBuffer(), static_cast<size_t>(sizeMeta1)));
    NNT_ASSERT_RESULT_SUCCESS(
        storageData.Write(0, setup.GetStorageData().GetBuffer(), static_cast<size_t>(sizeData1)));

    // Expand の境界値分析
    {
        NNT_ASSERT_RESULT_FAILURE( // sizeDataNew が下限
            nn::fs::ResultInvalidSize,
            nn::fssystem::save::SaveDataFileSystemCore::ExpandControlArea(
                nn::fs::SubStorage(&storageControlArea, 0, sizeControlArea),
                0
            )
        );
        NNT_ASSERT_RESULT_FAILURE( // sizeDataNew が上限より上
            nn::fs::ResultInvalidSize,
            nn::fssystem::save::SaveDataFileSystemCore::ExpandControlArea(
                nn::fs::SubStorage(&storageControlArea, 0, sizeControlArea),
                (CountBlock * 0x100000000) * SizeBlock
            )
        );
        NNT_ASSERT_RESULT_FAILURE( // sizeDataNew < sizeDataOld の境界
            nn::fs::ResultInvalidSize,
            nn::fssystem::save::SaveDataFileSystemCore::ExpandControlArea(
                nn::fs::SubStorage(&storageControlArea, 0, sizeControlArea),
                sizeData1 - 1
            )
        );
        NNT_ASSERT_RESULT_SUCCESS( // sizeDataNew == sizeDataOld
            nn::fssystem::save::SaveDataFileSystemCore::ExpandControlArea(
                nn::fs::SubStorage(&storageControlArea, 0, sizeControlArea),
                sizeData1
            )
        );
        NNT_ASSERT_RESULT_SUCCESS( // sizeDataNew == sizeDataOld
            nn::fssystem::save::SaveDataFileSystemCore::ExpandMeta(
                nn::fs::SubStorage(&storageMeta, 0, sizeMeta2),
                SizeBlock,
                sizeData1,
                sizeData1
            )
        );
    }

    // セーブデータの拡張
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::SaveDataFileSystemCore::ExpandControlArea(
            nn::fs::SubStorage(&storageControlArea, 0, sizeControlArea),
            sizeData2
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::SaveDataFileSystemCore::ExpandMeta(
            nn::fs::SubStorage(&storageMeta, 0, sizeMeta2),
            SizeBlock,
            sizeData1,
            sizeData2
        )
    );

    // 再初期化
    nn::fssystem::save::SaveDataFileSystemCore fileSystem2;
    NNT_ASSERT_RESULT_SUCCESS(
        fileSystem2.Initialize(
            nn::fs::SubStorage(&storageControlArea, 0, sizeControlArea),
            nn::fs::SubStorage(&storageMeta, 0, sizeMeta2),
            nn::fs::SubStorage(&storageData, 0, sizeData2),
            GetBufferManager()
        )
    );

    // 拡張・再初期化でセーブデータが変わっていないことを確認
    {
        nn::fssystem::save::IFile* pFile;
        NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/file1"));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem2.OpenFile(&pFile, path1, nn::fs::OpenMode_Read));
        NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, bufRead.data(), SizeFile));
        fileSystem2.CloseFile(pFile);
        ASSERT_EQ(0, std::memcmp(bufRead.data(), bufWrite.data(), SizeFile));
    }

    // ファイルの拡縮
    {
        FsSaveDataFileSystemCoreTest::FileObject file;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file, &fileSystem2, path1.GetString()));

        NNT_ASSERT_RESULT_SUCCESS(file.Resize(SizeFile / 2));
        NNT_ASSERT_RESULT_SUCCESS(file.Resize(SizeFile * 2));
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAllocationTableFull,
            file.Resize(SizeFile * 2 + 1)
        );
    }

    fileSystem2.Finalize();
    setup.Finalize();
} // NOLINT(impl/function_size)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(FsSaveDataFileSystemCoreDeathTest, Expand)
{
    static const size_t SizeBlock = 16 * 1024;
    static const size_t SizeFile = 2 * 1024 * 1024;
    static const size_t CountBlock = SizeFile / SizeBlock;

    const int64_t sizeControlArea =
        sizeof(nn::fssystem::save::SaveDataFileSystemCore::FileSystemHeader);
    const int64_t sizeData1 =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);
    const int64_t sizeData2 =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock * 2);
    const int64_t sizeMeta2 =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock * 2);

    // 拡張用のストレージを作成
    nnt::fs::util::SafeMemoryStorage storageControlArea;
    storageControlArea.Initialize(sizeControlArea);
    nnt::fs::util::SafeMemoryStorage storageMeta;
    storageMeta.Initialize(sizeMeta2);
    nnt::fs::util::SafeMemoryStorage storageData;
    storageData.Initialize(sizeData2);

    // Expand の境界値分析
    EXPECT_DEATH_IF_SUPPORTED( // sizeDataNew が下限より下
        nn::fssystem::save::SaveDataFileSystemCore::ExpandControlArea(
            nn::fs::SubStorage(&storageControlArea, 0, sizeControlArea),
            -1
        ),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED( // sizeBlock が下限より下
        nn::fssystem::save::SaveDataFileSystemCore::ExpandMeta(
            nn::fs::SubStorage(&storageMeta, 0, sizeMeta2),
            0,
            sizeData1,
            sizeData2
        ),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED( // sizeDataOld が下限より下
        nn::fssystem::save::SaveDataFileSystemCore::ExpandMeta(
            nn::fs::SubStorage(&storageMeta, 0, sizeMeta2),
            SizeBlock,
            -1,
            sizeData2
        ),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED( // sizeDataNew < sizeDataOld の境界
        nn::fssystem::save::SaveDataFileSystemCore::ExpandMeta(
            nn::fs::SubStorage(&storageMeta, 0, sizeMeta2),
            SizeBlock,
            sizeData1,
            sizeData1 - 1
        ),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED( // sizeDataOld が上限より上
        nn::fssystem::save::SaveDataFileSystemCore::ExpandMeta(
            nn::fs::SubStorage(&storageMeta, 0, sizeMeta2),
            1,
            0x100000000,
            0x100000001
        ),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED( // sizeDataNew が上限より上
        nn::fssystem::save::SaveDataFileSystemCore::ExpandMeta(
            nn::fs::SubStorage(&storageMeta, 0, sizeMeta2),
            1,
            sizeData1,
            0x100000000
        ),
        ""
    );
}
#endif

// ディレクトリのテスト
void FsSaveDataFileSystemCoreTest::TestDirectory(
        nn::fssystem::save::IFileSystem* pFileSystem
    ) NN_NOEXCEPT
{
    static const int32_t CountEntry = 3;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    const uint32_t countDirectory = std::uniform_int_distribution<>(0, 99)(mt);

    {
        // ルートディレクトリを開く。まだ何もない
        nn::fs::DirectoryEntry entry[CountEntry];
        nn::fssystem::save::IDirectory* pDirectory;
        nn::fssystem::save::Path pathRoot;
        NNT_ASSERT_RESULT_SUCCESS(pathRoot.Initialize("/"));

        int32_t resultCount;
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDirectory, pathRoot));
        NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&resultCount, entry, CountEntry));
        ASSERT_EQ(0, resultCount);
        NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&resultCount, entry, CountEntry));
        ASSERT_EQ(0, resultCount);
        pFileSystem->CloseDirectory(pDirectory);

        // ディレクトリを1つ作ります。
        nn::fssystem::save::Path path2;
        NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/director"));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path2));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDirectory, pathRoot));
        NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&resultCount, entry, CountEntry));
        ASSERT_EQ(1, resultCount);    // 一つ出来た
        ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, entry[0].directoryEntryType);
        ASSERT_EQ(0, std::memcmp("director", entry[0].name, 8));
        NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&resultCount, entry, CountEntry));
        ASSERT_EQ(0, resultCount);
        pFileSystem->CloseDirectory(pDirectory);

        // ファイルを一つ作ります。
        nn::fssystem::save::Path path3;
        NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/filefilefile"));
        nn::fssystem::save::IFile* pFile = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(path3, 0));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, path3, nn::fs::OpenMode_Write));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDirectory, pathRoot));
        NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&resultCount, entry, CountEntry));
        ASSERT_EQ(2, resultCount);    // 2つ出来た
        ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, entry[0].directoryEntryType);
        ASSERT_EQ(0, std::memcmp("director", entry[0].name, 8));
        ASSERT_EQ(0, std::memcmp("filefilefile", entry[1].name, 12));
        NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&resultCount, entry, CountEntry));
        ASSERT_EQ(0, resultCount);
        pFileSystem->CloseDirectory(pDirectory);
        pFileSystem->CloseFile(pFile);

        // ディレクトリとファイルを消去します。
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidPathForOperation,
            pFileSystem->DeleteFile(path2)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidPathForOperation,
            pFileSystem->DeleteDirectory(path3)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(path3));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteDirectory(path2));
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNotFound,
            pFileSystem->DeleteFile(path3)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNotFound,
            pFileSystem->DeleteDirectory(path2)
        );
    }

    // ディレクトリのテストを行います。
    {
        nn::fs::DirectoryEntry entry[CountEntry];
        nn::fssystem::save::IDirectory* pDirectory;
        nn::fssystem::save::Path pathRoot;
        NNT_ASSERT_RESULT_SUCCESS(pathRoot.Initialize("/"));

        // ディレクトリを作成します。
        for( uint32_t i = 0; i < countDirectory; ++i )
        {
            char name[32];
            std::sprintf(name, "/dir%u", i);
            nn::fssystem::save::Path path3;
            NNT_ASSERT_RESULT_SUCCESS(path3.Initialize(name));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(path3));
        }

        // ディレクトリの列挙テストを行います。
        {
            int32_t resultCount;
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDirectory, pathRoot));
            for( uint32_t i = 0; i + 3 <= countDirectory; i += 3 )
            {
                NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&resultCount, entry, CountEntry));
                ASSERT_EQ(3, resultCount);
            }
            if( countDirectory % 3 != 0 )
            {
                NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&resultCount, entry, CountEntry));
                ASSERT_EQ(countDirectory % 3, static_cast<uint32_t>(resultCount));
            }
            NNT_ASSERT_RESULT_SUCCESS(pDirectory->Read(&resultCount, entry, CountEntry));
            ASSERT_EQ(0, resultCount);

            pFileSystem->CloseDirectory(pDirectory);
        }
    }
}

// ファイルのテスト
void FsSaveDataFileSystemCoreTest::TestFile(
        nn::fssystem::save::IFileSystem* pFileSystem
    ) NN_NOEXCEPT
{
    nn::fssystem::save::Path file1;
    NNT_ASSERT_RESULT_SUCCESS(file1.Initialize("/file1"));

    // 存在しないファイルを開く
    nn::fssystem::save::IFile* pFile = nullptr;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNotFound,
        pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Read)
    );
    ASSERT_EQ(nullptr, pFile);
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNotFound,
        pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write)
    );
    ASSERT_EQ(nullptr, pFile);

    // 誤った指定(モード指定テスト)
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOperationForOpenMode,
        pFileSystem->OpenFile(&pFile, file1, 0)
    );
    ASSERT_EQ(nullptr, pFile);

    // 正しい指定(モード指定テスト)
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(file1, 0));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write));
    pFileSystem->CloseFile(pFile);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Read));
    ASSERT_NE(nullptr, pFile);
    pFileSystem->CloseFile(pFile);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write));
    ASSERT_NE(nullptr, pFile);
    pFileSystem->CloseFile(pFile);

    int64_t size;
    {
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Write));
        ASSERT_NE(nullptr, pFile);
        NN_UTIL_SCOPE_EXIT
        {
            pFileSystem->CloseFile(pFile);
        };

        // サイズの取得と設定
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        const int64_t sizeOld = size;
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(0, size);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAllocationTableFull,
            pFile->SetSize(64 * 1024 * 1024)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(0, size);
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(1));
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(1, size);
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeOld));
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAllocationTableFull,
            pFile->SetSize(64 * 1024 * 1024)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(sizeOld, size);
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeOld));
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        ASSERT_EQ(sizeOld, size);
    }

    // リード状態ではサイズを設定できません
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, file1, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
    ASSERT_EQ(0, size);
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOperationForOpenMode,
        pFile->SetSize(16)
    );
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
    ASSERT_EQ(0, size);
    pFileSystem->CloseFile(pFile);

    nnt::fs::util::Vector<char> bufWrite(128);
    nnt::fs::util::Vector<char> bufRead(128);
    NNT_ASSERT_RESULT_SUCCESS(
        pFileSystem->OpenFile(
            &pFile,
            file1,
            nn::fs::OpenMode_Write | nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));

    // 0 バイトのファイルに 0 バイトの書き込みます。
    ASSERT_EQ(0, size);
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, &bufWrite[0], 0));

    // 範囲外の書き込み
    NNT_ASSERT_RESULT_SUCCESS(
        pFile->WriteBytes(1, &bufWrite[0], 0)
    );

    // 0 バイトのファイルから 0 バイトの読み込みます。
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &bufRead[0], 0));

    // 範囲外の読み込み
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        pFile->ReadBytes(1, &bufRead[0], 0)
    );

    // ファイルサイズを30バイトに設定しました
    NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(30));
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
    ASSERT_EQ(30, size);

    // 範囲内のアクセス
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(28, &bufWrite[0], 2));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(28, &bufRead[0], 2));
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(1, &bufWrite[0], 29));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(1, &bufRead[0], 29));

    // 範囲を少々超えるアクセス -> 自動拡張
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(28, &bufWrite[0], 4));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(28, &bufRead[0], 4));
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, &bufWrite[0], 31));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, &bufRead[0], 31));

    // 完全に範囲外のアクセス
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        pFile->ReadBytes(33, &bufRead[0], 1)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        pFile->ReadBytes(34, &bufRead[0], 1)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        pFile->WriteBytes(33, &bufWrite[0], 1)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        pFile->WriteBytes(34, &bufWrite[0], 1)
    );
    // 拡張されたのでアクセス可能に
    NNT_ASSERT_RESULT_SUCCESS(
        pFile->ReadBytes(33, &bufRead[0], 1)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        pFile->ReadBytes(34, &bufRead[0], 1)
    );

    pFileSystem->CloseFile(pFile);
} // NOLINT(impl/function_size)

// アロケーションテーブルのテスト
void FsSaveDataFileSystemCoreTest::TestAllocation(
        nn::fssystem::save::IFileSystem* pFileSystem
    ) NN_NOEXCEPT
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    const uint32_t countFile = std::uniform_int_distribution<>(0, 99)(mt);

    // 複数のファイルでアローケーションテーブルを取りあう
    nnt::fs::util::Vector<nn::fssystem::save::IFile *> pFiles(countFile);

    for( uint32_t m = 0; m < countFile; m++ )
    {
        char buf[32];
        std::memset(buf, 0, sizeof(buf));
        std::sprintf(buf, "/file%u", m);
        nn::fssystem::save::Path pathCur;
        NNT_ASSERT_RESULT_SUCCESS(pathCur.Initialize(buf));
        nn::Result result = pFileSystem->CreateFile(pathCur, 0);
        if( result.IsFailure() )
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAlreadyExists, result);
        }
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFiles[m], pathCur, nn::fs::OpenMode_Write));
    }

    for( uint32_t m = 0; m < countFile; ++m )
    {
        nn::Result result = pFiles[m]->SetSize(std::uniform_int_distribution<>(0, 1023)(mt));
        if( result.IsFailure() )
        {
            break;
        }
    }

    // 合計サイズ
    int64_t sizeAlloc = 0;
    for( uint32_t m = 0; m < countFile; m++ )
    {
        int64_t size;
        NNT_ASSERT_RESULT_SUCCESS(pFiles[m]->GetSize(&size));
        sizeAlloc += size;
    }

    nnt::fs::util::Vector<char> fillBuf;
    nnt::fs::util::Vector<char> baseBuf;

    // 各ファイルを特定のデータで埋めます。
    for( uint32_t m = 0; m < countFile; ++m )
    {
        int64_t size;
        NNT_ASSERT_RESULT_SUCCESS(pFiles[m]->GetSize(&size));
        if( size > 0 )
        {
            fillBuf.resize(static_cast<size_t>(size));
            char* p = &fillBuf[0];
            for( uint32_t n = 0; n < size; n++ )
            {
                p[n] = (m + n) & 0xFF;
            }
            NNT_ASSERT_RESULT_SUCCESS(pFiles[m]->WriteBytes(0, p, static_cast<size_t>(size)));
        }
    }

    // ファイルを全て閉じます。
    for( uint32_t m = 0; m < countFile; ++m )
    {
        if( pFiles[m] != nullptr )
        {
            pFileSystem->CloseFile(pFiles[m]);
            pFiles[m] = nullptr;
        }
    }

    // ファイルを読み込み専用で開きなおします。
    for( uint32_t m = 0; m < countFile; ++m )
    {
        char buf[32];
        std::memset(buf, 0, sizeof(buf));
        std::sprintf(buf, "/file%u", m);
        nn::fssystem::save::Path pathCur;
        NNT_ASSERT_RESULT_SUCCESS(pathCur.Initialize(buf));
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFiles[m], pathCur, nn::fs::OpenMode_Read));
    }

    for( uint32_t m = 0; m < countFile; ++m )
    {
        int64_t size;
        NNT_ASSERT_RESULT_SUCCESS(pFiles[m]->GetSize(&size));

        if( size > 0 )
        {
            fillBuf.resize(static_cast<size_t>(size));
            baseBuf.resize(static_cast<size_t>(size));
            char* pBase = &baseBuf[0];
            char* pFill = &fillBuf[0];

            // 書き込まれているデータを生成します
            for( uint32_t n = 0; n < size; n++ )
            {
                pFill[n] = (m + n) & 0xFF;
            }

            // ファイルを後ろから読み取ります。
            for( uint32_t n = 0; n < size; n++ )
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pFiles[m]->ReadBytes(size - n - 1, &pBase[size - n - 1], 1));
            }
            ASSERT_EQ(0, std::memcmp(pBase, pFill, static_cast<size_t>(size)));
            // ファイルを前から読み取ります。
            for( uint32_t n = 0; n < size; n++ )
            {
                NNT_ASSERT_RESULT_SUCCESS(pFiles[m]->ReadBytes(n, &pBase[n], 1));
            }
            ASSERT_EQ(0, std::memcmp(pBase, pFill, static_cast<size_t>(size)));
            // ファイルを一括で読み込みます。
            NNT_ASSERT_RESULT_SUCCESS(pFiles[m]->ReadBytes(0, pBase, static_cast<size_t>(size)));
            ASSERT_EQ(0, std::memcmp(pBase, pFill, static_cast<size_t>(size)));
        }
    }

    // ファイルを全て閉じます。
    for( uint32_t m = 0; m < countFile; m++ )
    {
        if( pFiles[m] != nullptr )
        {
            pFileSystem->CloseFile(pFiles[m]);
            pFiles[m] = nullptr;
        }
    }
} // NOLINT(impl/function_size)

// メタデータ領域と実データ領域が別ファイルの
// セーブデータアーカイブに対するテスト
TEST_F(FsSaveDataFileSystemCoreTest, SeparatedSaveData)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( uint32_t loop = 0; loop < 10; ++loop )
    {
        auto sizeBlock = GenerateRandomBlockSize(128, MaxBlockSize);

        // ディレクトリ 99 個、サイズ 1023 のファイル 99 個が最大で作られるので、
        // それに対応できるブロック数を求める
        auto sizeDirectoryEntry = 96;
        auto sizeFileEntry = 96;
        auto countMaxDirectoryBlock = nn::util::DivideUp((99 + 3) * sizeDirectoryEntry, sizeBlock);
        auto countMaxFileBlock = nn::util::DivideUp((99 + 2) * sizeFileEntry, sizeBlock) + 99 * nn::util::DivideUp(1023, sizeBlock);

        auto minCountDataBlock = countMaxFileBlock + countMaxDirectoryBlock;
        auto countDataBlock = std::uniform_int_distribution<>(minCountDataBlock, 1000)(mt);

        const auto sizeControlArea = nn::util::align_up(
            sizeof(nn::fssystem::save::SaveDataFileSystemCore::FileSystemHeader),
            sizeBlock
        );
        const auto sizeDataMin = nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(
            sizeBlock,
            countDataBlock
        );
        const auto sizeData =
            nn::util::align_down(std::uniform_int_distribution<>(0, 32 * 1024 - 1)(mt), sizeBlock)
                + sizeDataMin;
        const auto sizeMeta = nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(
            sizeBlock,
            static_cast<size_t>(sizeData / sizeBlock)
        );

        nnt::fs::util::SafeMemoryStorage storageControlArea;
        storageControlArea.Initialize(sizeControlArea);
        nn::fs::SubStorage subStorageControlArea(&storageControlArea, 0, sizeControlArea);

        nnt::fs::util::SafeMemoryStorage storageMeta;
        storageMeta.Initialize(sizeMeta);
        nn::fs::SubStorage subStorageMeta(&storageMeta, 0, sizeMeta);

        nnt::fs::util::SafeMemoryStorage storageData;
        storageData.Initialize(sizeData);
        nn::fs::SubStorage subStorageData(&storageData, 0, sizeData);

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::SaveDataFileSystemCore::Format(
                subStorageControlArea,
                subStorageMeta,
                subStorageData,
                static_cast<uint32_t>(sizeBlock),
                GetBufferManager()
            )
        );

        nn::fssystem::save::SaveDataFileSystemCore fileSystem;
        NNT_ASSERT_RESULT_SUCCESS(
            fileSystem.Initialize(
                subStorageControlArea,
                subStorageMeta,
                subStorageData,
                GetBufferManager()
            )
        );

        TestSeparateSaveData(&fileSystem);

        // 先頭と末尾の破壊チェック用
        fileSystem.Finalize();
    }
}

// ディレクトリのリードをテストします。
TEST_F(FsSaveDataFileSystemCoreTest, ReadDirectory)
{
    static const size_t CountFile = 200;
    static const size_t TotalSize = 1024 * 1024;
    static const size_t SizeBlock = 4096;
    static const size_t CountBlock = TotalSize / SizeBlock;

    const int64_t sizeMeta =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock);
    const int64_t sizeData =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);

    SaveDataFileSystemSetup setup;
    setup.Initialize(sizeMeta, sizeData, SizeBlock);

    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

    nn::fssystem::save::Path pathDirectory;
    NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<nn::fssystem::save::IFile *> pFiles(CountFile);
    nnt::fs::util::Vector<uint32_t> sizes(CountFile);
    for( uint32_t i = 0; i < CountFile; ++i )
    {
        char buf[32];
        std::memset(buf, 0, sizeof(buf));
        std::sprintf(buf, "/directory/file%08u", i);
        nn::fssystem::save::Path pathCur;
        NNT_ASSERT_RESULT_SUCCESS(pathCur.Initialize(buf));
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathCur, 0));
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(&pFiles[i], pathCur, nn::fs::OpenMode_Write));
        sizes[i] = std::uniform_int_distribution<>(0, 511)(mt);
        NNT_ASSERT_RESULT_SUCCESS(pFiles[i]->SetSize(sizes[i]));
        pFileSystem->CloseFile(pFiles[i]);
    }

    nn::fssystem::save::IDirectory* pDir;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenDirectory(&pDir, pathDirectory));
    for( size_t i = CountFile - 1; i != 0; --i )
    {
        nn::fs::DirectoryEntry entry;
        int32_t resultCount;
        NNT_ASSERT_RESULT_SUCCESS(pDir->Read(&resultCount, &entry, 1));
        ASSERT_EQ(1, resultCount);
        ASSERT_EQ(nn::fs::DirectoryEntryType_File, entry.directoryEntryType);
        ASSERT_EQ(sizes[i], entry.fileSize);
    }
    pFileSystem->CloseDirectory(pDir);

    setup.Finalize();
}

// 不正なサイズのテスト
TEST_F(FsSaveDataFileSystemCoreTest, InvalidFileSize)
{
    static const size_t TotalSize = 1024 * 1024;
    static const size_t SizeBlock = 4096;
    static const size_t CountBlock = TotalSize / SizeBlock;

    const int64_t sizeMeta =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock);
    const int64_t sizeData =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);

    SaveDataFileSystemSetup setup;
    setup.Initialize(sizeMeta, sizeData, SizeBlock);

    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

    char buf[32];
    std::memset(buf, 0, sizeof(buf));
    std::sprintf(buf, "/file");
    nn::fssystem::save::Path pathCur;
    NNT_ASSERT_RESULT_SUCCESS(pathCur.Initialize(buf));
    nn::fssystem::save::IFile* pFile;

    const int64_t FileSize = 5 * 1024;

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathCur, FileSize));

    std::unique_ptr<char[]> writeBuffer(new char[FileSize + 2]);
    std::iota(writeBuffer.get(), writeBuffer.get() + FileSize + 2, '\x0');
    std::unique_ptr<char[]> readBuffer(new char[FileSize]);

    // AllowAppend を指定せずに実行
    {
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathCur, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            pFileSystem->CloseFile(pFile);
        };

        // ファイルにデータを書き込んでおく
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, writeBuffer.get(), FileSize)
        );

        // SetSize にわざと失敗する
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidArgument,
                pFile->SetSize(0x8000000000000000)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultOutOfResource,
                pFile->SetSize(0x7FFFFFFFFFFFFFFF)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultOutOfResource,
                pFile->SetSize(TotalSize)
            );
        }

        // ファイルの中身を確認
        {
            // サイズが変わっていないか確認
            int64_t size = 0;
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(FileSize, size);

            // ファイルの中身が書き換わっていないか確認
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->ReadBytes(0, readBuffer.get(), FileSize)
            );
            ASSERT_EQ(0, memcmp(readBuffer.get(), writeBuffer.get(), FileSize));

            // 読み書きを行うことで、実際にファイルサイズが変わっていないかを確認
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                pFile->WriteBytes(FileSize + 1, writeBuffer.get(), 1)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
                pFile->WriteBytes(0, writeBuffer.get() + 1, FileSize + 1)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(0, writeBuffer.get() + 2, FileSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->ReadBytes(0, readBuffer.get(), FileSize)
            );
            ASSERT_EQ(0, memcmp(readBuffer.get(), writeBuffer.get() + 2, FileSize));
        }
    }

    {
        // AllowAppend にして同じ事を行う
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathCur, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT
        {
            pFileSystem->CloseFile(pFile);
        };

        // ファイルにデータを書き込んでおく
        writeBuffer.reset(new char[FileSize + 2]);
        std::iota(writeBuffer.get(), writeBuffer.get() + FileSize + 2, '\x0');
        NNT_ASSERT_RESULT_SUCCESS(
            pFile->WriteBytes(0, writeBuffer.get(), FileSize)
        );

        // SetSize にわざと失敗する
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidArgument,
                pFile->SetSize(0x8000000000000000)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultOutOfResource,
                pFile->SetSize(0x7FFFFFFFFFFFFFFF)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultOutOfResource,
                pFile->SetSize(TotalSize)
            );
        }

        // ファイルの中身を確認
        {
            // サイズが変わっていないか確認
            int64_t size = 0;
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
            ASSERT_EQ(FileSize, size);

            // ファイルの中身が書き換わっていないか確認
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->ReadBytes(0, readBuffer.get(), FileSize)
            );
            ASSERT_EQ(0, memcmp(readBuffer.get(), writeBuffer.get(), FileSize));
        }

        // 読み書きを行って確認するために AllowAppend をはずして開き直す
        pFileSystem->CloseFile(pFile);
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathCur, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                pFile->WriteBytes(FileSize + 1, writeBuffer.get(), 1)
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
                pFile->WriteBytes(0, writeBuffer.get() + 1, FileSize + 1)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->WriteBytes(0, writeBuffer.get() + 2, FileSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->ReadBytes(0, readBuffer.get(), FileSize)
            );
            ASSERT_EQ(0, memcmp(readBuffer.get(), writeBuffer.get() + 2, FileSize));
        }
    }

    setup.Finalize();
} // NOLINT(impl/function_size)

class SaveDataArchiveTestSetup : public TestFileSystemSetup
{
public:
    // コンストラクタです。
    SaveDataArchiveTestSetup() NN_NOEXCEPT
    {
    }

    // アーカイブを初期化します。
    virtual void Initialize(
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        const size_t countBlock = 1024;
        const int64_t sizeControlArea = nn::util::align_up(
            sizeof(nn::fssystem::save::SaveDataFileSystemCore::FileSystemHeader), sizeBlock);
        const int64_t sizeMeta =
            nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(sizeBlock, countBlock);
        const int64_t sizeData =
            nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(sizeBlock, countBlock);

        m_pFileControlArea.reset(new nnt::fs::util::SafeMemoryStorage());
        m_pFileControlArea->Initialize(sizeControlArea + offset * 2);
        nn::fs::SubStorage subStorageControlArea(
            m_pFileControlArea.get(), offset, sizeControlArea);

        m_pFileMeta.reset(new nnt::fs::util::SafeMemoryStorage());
        m_pFileMeta->Initialize(sizeMeta + offset * 2);
        nn::fs::SubStorage subStorageMeta(m_pFileMeta.get(), offset, sizeMeta);

        m_pFileData.reset(new nnt::fs::util::SafeMemoryStorage());
        m_pFileData->Initialize(sizeData + offset * 2);
        nn::fs::SubStorage subStorageData(m_pFileData.get(), offset, sizeData);

        m_pFileSystemhive.reset(new nn::fssystem::save::SaveDataFileSystemCore());
        NNT_ASSERT_RESULT_SUCCESS(
            m_pFileSystemhive->Format(
                subStorageControlArea,
                subStorageMeta,
                subStorageData,
                static_cast<uint32_t>(sizeBlock),
                GetBufferManager()
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            m_pFileSystemhive->Initialize(
                subStorageControlArea,
                subStorageMeta,
                subStorageData,
                GetBufferManager()
            )
        );

        SetFileSystem(m_pFileSystemhive.get());
    }

    // アーカイブを破棄します。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pFileSystemhive->Finalize();
    }

    // アーカイブへのアクセスはスレッドセーフかどうか
    virtual bool IsEnableThreadSafeAccess() NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // アーカイブへの排他制御は有効かどうか
    virtual bool IsEnableExclusiveAccess() NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResizeFile() NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    //!< ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

private:
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pFileControlArea;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pFileMeta;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pFileData;
    std::unique_ptr<nn::fssystem::save::SaveDataFileSystemCore> m_pFileSystemhive;
};

// アーカイブ共通テストを行います。
TEST_F(FsSaveDataFileSystemCoreTest, ArchiveCommonTest)
{
    FileSystemTest::RunTest<SaveDataArchiveTestSetup>();
}

// ファイルシステム上のファイルに対してテストを実行します
class SaveDataFileTestSetup : public TestStorageSetup
{
public:
    static const int64_t MinFileSize = 1;

public:
    // コンストラクタです。
    SaveDataFileTestSetup() NN_NOEXCEPT
        : m_pStorage(nullptr)
    {
    }

    // ファイルオブジェクトをフォーマットします。
    virtual void Format(
                     int64_t sizeFile,
                     int64_t offset,
                     size_t sizeBlock0
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        // sizeBlock はエントリサイズ(96 バイト)以上である必要があります。
        size_t sizeBlock = std::max(static_cast<size_t>(128), sizeBlock0);

        // TODO: #pragma では値が化けるので volatile にする
        volatile auto randomValue = m_Random.Get<>(20);
        const size_t countDataBlock =
            static_cast<size_t>((randomValue + sizeFile) / sizeBlock + 10);
        const int64_t sizeControlArea = nn::util::align_up(
            sizeof(nn::fssystem::save::SaveDataFileSystemCore::FileSystemHeader), sizeBlock);
        const int64_t sizeMeta =
            nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(sizeBlock, countDataBlock);
        const int64_t sizeData =
            nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(sizeBlock, countDataBlock);

        m_pFileControlArea.reset(new nnt::fs::util::SafeMemoryStorage());
        m_pFileControlArea->Initialize(sizeControlArea + offset * 2);
        m_SubStorageControlArea =
            nn::fs::SubStorage(m_pFileControlArea.get(), 0, sizeControlArea + offset * 2);

        m_pFileMeta.reset(new nnt::fs::util::SafeMemoryStorage());
        m_pFileMeta->Initialize(sizeMeta + offset * 2);
        m_SubStorageMeta = nn::fs::SubStorage(m_pFileMeta.get(), 0, sizeMeta + offset * 2);

        m_pFileData.reset(new nnt::fs::util::SafeMemoryStorage());
        m_pFileData->Initialize(sizeData + offset * 2);
        m_SubStorageData = nn::fs::SubStorage(m_pFileData.get(), 0, sizeData + offset * 2);

        m_pFileSystemhive.reset(new nn::fssystem::save::SaveDataFileSystemCore());
        NNT_ASSERT_RESULT_SUCCESS(
            m_pFileSystemhive->Format(
                m_SubStorageControlArea,
                m_SubStorageMeta,
                m_SubStorageData,
                static_cast<uint32_t>(sizeBlock),
                GetBufferManager()
            )
        );
    }

    // ファイルオブジェクトを初期化します。
    virtual void Initialize(
                     int64_t sizeFile,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(offset);
        NN_UNUSED(sizeBlock);

        NNT_ASSERT_RESULT_SUCCESS(
            m_pFileSystemhive->Initialize(
                m_SubStorageControlArea,
                m_SubStorageMeta,
                m_SubStorageData,
                GetBufferManager()
            )
        );

        nn::fssystem::save::IFileSystem* pFileSystem = m_pFileSystemhive.get();

        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/file"));
        nn::Result result = pFileSystem->CreateFile(path, 0);
        if( result.IsFailure() )
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAlreadyExists, result);
        }
        NNT_ASSERT_RESULT_SUCCESS(
            pFileSystem->OpenFile(
                &m_pFile,
                path,
                nn::fs::OpenMode_Write | nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(m_pFile->SetSize(sizeFile));

        m_pStorage.reset(new nn::fssystem::save::FileStorage());
        m_pStorage->Initialize(m_pFile);
        SetStorage(m_pStorage.get());
    }

    // ファイルオブジェクトをアンマウントします。
    virtual void Unmount() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fssystem::save::IFileSystem* pFileSystem = m_pFileSystemhive.get();
        pFileSystem->CloseFile(m_pFile);
    }

    // ファイルオブジェクトを破棄します。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fssystem::save::IFileSystem* pFileSystem = m_pFileSystemhive.get();

        nn::fssystem::save::Path path;
        NNT_ASSERT_RESULT_SUCCESS(path.Initialize("/file"));
        pFileSystem->DeleteFile(path);

        pFileSystem->CloseFile(m_pFile);

        m_pFileSystemhive->Finalize();
    }

    // 読み込み専用かどうか
    virtual bool IsReadOnly() const NN_NOEXCEPT NN_OVERRIDE
    {
        // TODO: オープンモードから求める
        return false;
    }

    // ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    // サイズ 0 のファイルを作成可能かどうか
    virtual bool IsAcceptsSizeZero() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

private:
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pFileControlArea;
    nn::fs::SubStorage m_SubStorageControlArea;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pFileMeta;
    nn::fs::SubStorage m_SubStorageMeta;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pFileData;
    nn::fs::SubStorage m_SubStorageData;
    std::unique_ptr<nn::fssystem::save::SaveDataFileSystemCore> m_pFileSystemhive;
    nn::fssystem::save::IFile* m_pFile;
    std::unique_ptr<nn::fssystem::save::FileStorage> m_pStorage;
};

// 共通テストを行います。
TEST_F(FsSaveDataFileSystemCoreTest, StorageCommonTest)
{
    StorageTest::RunTest<SaveDataFileTestSetup>();
}

// 0 バイト書き込みやクローズ時のファイルアクセスなどをテストします。
TEST_F(FsSaveDataFileSystemCoreTest, Flush)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    static const size_t TestLoop = 20;

    for( size_t sizeBlock = 128; sizeBlock <= 512; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        const uint32_t countFile = std::uniform_int_distribution<>(1, 50)(mt);

        for( int32_t i = 0; i < TestLoop; ++i )
        {
            nnt::fs::util::AccessCountedMemoryStorage storageMem;
            nn::fssystem::save::SaveDataFileSystemCore fileSystem;

            const size_t countDataBlock =
                std::uniform_int_distribution<>(0, 19)(mt) +
                std::uniform_int_distribution<>(0, 1023)(mt) / sizeBlock +
                100;
            const int64_t offsetMount = std::uniform_int_distribution<>(0, 511)(mt);

            const int64_t sizeControlArea = nn::util::align_up(
                sizeof(nn::fssystem::save::SaveDataFileSystemCore::FileSystemHeader),
                sizeBlock
            );
            const int64_t sizeMeta = nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(
                sizeBlock,
                countDataBlock
            );
            const int64_t sizeData = nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(
                sizeBlock,
                countDataBlock
            );

            storageMem.Initialize(sizeControlArea + sizeMeta + sizeData + offsetMount * 2);

            int64_t offset = offsetMount;
            nn::fs::SubStorage subStorageControlArea(&storageMem, offset, sizeControlArea);
            offset += sizeControlArea;
            nn::fs::SubStorage subStorageMeta(&storageMem, offset, sizeMeta);
            offset += sizeMeta;
            nn::fs::SubStorage subStorageData(&storageMem, offset, sizeData);

            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::SaveDataFileSystemCore::Format(
                    subStorageControlArea,
                    subStorageMeta,
                    subStorageData,
                    static_cast<uint32_t>(sizeBlock),
                    GetBufferManager()
                )
            );

            NNT_ASSERT_RESULT_SUCCESS(
                fileSystem.Initialize(
                    subStorageControlArea,
                    subStorageMeta,
                    subStorageData,
                    GetBufferManager()
                )
            );

            // アクセス回数リセット
            storageMem.ResetAccessCounter();

            // バッファ作成
            nnt::fs::util::Vector<char> data(sizeBlock * 2);
            char* ptr = &data[0];
            for( size_t t = 0; t < data.size(); ++t )
            {
                ptr[i] = i & 0xFF;
            }

            nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

            // ディレクトリを新規作成します。
            nn::fssystem::save::Path pathDirectory;
            NNT_ASSERT_RESULT_SUCCESS(pathDirectory.Initialize("/directory"));
            NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateDirectory(pathDirectory));

            // 適当な位置に書き込み
            for( size_t t = 0; t < countFile; ++t )
            {
                char buf[32];
                nn::util::SNPrintf(buf, sizeof(buf), "/directory/file%08u", t);
                nn::fssystem::save::Path filePath;
                NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                nn::fssystem::save::IFile* pFile = nullptr;
                int64_t sizeFile =
                    std::uniform_int_distribution<int64_t>(0, sizeBlock - 1)(mt) + 8;
                int64_t writeOffset =
                    std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                size_t size =
                    std::uniform_int_distribution<size_t>(
                        1, static_cast<size_t>(sizeFile - writeOffset))(mt);

                // アクセス回数リセット
                storageMem.ResetAccessCounter();

                NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(filePath, 0));
                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, filePath, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(sizeFile));
                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(writeOffset, buf, size));

                ASSERT_EQ(0, storageMem.GetFlushTimes());

                pFileSystem->CloseFile(pFile);

            }

            // アクセス回数リセット
            storageMem.ResetAccessCounter();

            // 0 バイト書き込み
            for( size_t t = 0; t < countFile; ++t )
            {
                char buf[32];
                nn::util::SNPrintf(buf, sizeof(buf), "/directory/file%08u", t);
                nn::fssystem::save::Path filePath;
                NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                nn::fssystem::save::IFile* pFile = nullptr;
                int64_t sizeFile;

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, filePath, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&sizeFile));

                int64_t writeOffset = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);

                // アクセス回数リセット
                storageMem.ResetAccessCounter();

                NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(writeOffset, buf, 0));

                // 0 バイト書き込みではファイルアクセスは発生しない
                ASSERT_EQ(0, storageMem.GetFlushTimes());
                ASSERT_EQ(0, storageMem.GetReadTimes());
                ASSERT_EQ(0, storageMem.GetWriteTimes());

                pFileSystem->CloseFile(pFile);

                // クローズでファイルアクセスは発生しない
                ASSERT_EQ(0, storageMem.GetFlushTimes());
                ASSERT_EQ(0, storageMem.GetReadTimes());
                ASSERT_EQ(0, storageMem.GetWriteTimes());
            }

            // アクセス回数リセット
            storageMem.ResetAccessCounter();

            // 範囲外に書き込み
            for( size_t t = 0; t < countFile; t++ )
            {
                char buf[32];
                nn::util::SNPrintf(buf, sizeof(buf), "/directory/file%08u", t);
                nn::fssystem::save::Path filePath;
                NNT_ASSERT_RESULT_SUCCESS(filePath.Initialize(buf));

                nn::fssystem::save::IFile* pFile = nullptr;
                int64_t sizeFile;

                NNT_ASSERT_RESULT_SUCCESS(
                    pFileSystem->OpenFile(&pFile, filePath, nn::fs::OpenMode_Write));
                NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&sizeFile));

                const int64_t writeOffset =
                    std::uniform_int_distribution<int64_t>(sizeFile + 1, sizeFile * 2)(mt);
                const size_t size =
                    std::uniform_int_distribution<size_t>(1, static_cast<size_t>(sizeFile))(mt);

                // アクセス回数リセット
                storageMem.ResetAccessCounter();

                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultInvalidOffset,
                    pFile->WriteBytes(writeOffset, buf, size)
                );

                // 範囲外エラー発生時に書き込みは行われない
                ASSERT_EQ(0, storageMem.GetFlushTimes());

                // アクセス回数リセット
                storageMem.ResetAccessCounter();

                pFileSystem->CloseFile(pFile);

                // クローズでファイルアクセスは発生しない
                ASSERT_EQ(0, storageMem.GetReadTimes());
                ASSERT_EQ(0, storageMem.GetWriteTimes());
            }

            // アクセス回数リセット
            storageMem.ResetAccessCounter();

            // アーカイブをアンマウント
            fileSystem.Finalize();

            // アンマウントでファイルアクセスは発生しない
            ASSERT_EQ(0, storageMem.GetReadTimes());
            ASSERT_EQ(0, storageMem.GetWriteTimes());
        }
    }
    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

//!< 新規作成したファイルがゼロ埋めされていることを検証します。
TEST_F(FsSaveDataFileSystemCoreTest, FillZero)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    const size_t totalSize = 1024 * 1024;
    const int32_t sizeBlock = 4096;
    const size_t countBlock = totalSize / sizeBlock;

    const int64_t sizeMeta =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(sizeBlock, countBlock);
    const int64_t sizeData =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(sizeBlock, countBlock);

    SaveDataFileSystemSetup setup;
    setup.Initialize(sizeMeta, sizeData, sizeBlock);

    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

    char buf[1024];
    nn::fssystem::save::Path pathFile;
    NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/file"));
    nn::fssystem::save::IFile* pFile = nullptr;

    // 先頭をランダムなデータで埋めます。
    size_t sizeFile = 200;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
    for( size_t i = 0; i < sizeFile; i++ )
    {
        buf[i] = static_cast<char>(std::uniform_int_distribution<>(0, 0xFF)(mt));
    }
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, buf, sizeFile));
    pFileSystem->CloseFile(pFile);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));

    // ファイルがゼロ埋めされていることを確認します。
    sizeFile = 600;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, buf, sizeFile));
    for( size_t i = 0; i < sizeFile; i++ )
    {
        ASSERT_EQ(0, buf[i]);
    }
    pFileSystem->CloseFile(pFile);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));

    // 再度先頭をランダムなデータで埋めます。
    sizeFile = 200;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write));
    for( size_t i = 0; i < sizeFile; ++i )
    {
        buf[i] = static_cast<char>(std::uniform_int_distribution<>(0, 0xFF)(mt));
    }
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, buf, sizeFile));
    pFileSystem->CloseFile(pFile);
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(pathFile));

    // 新規作成したファイルがゼロ埋めされていることを確認します。
    sizeFile = 128;
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, sizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, buf, sizeFile));
    for( size_t i = 0; i < sizeFile; ++i )
    {
        ASSERT_EQ(0, buf[i]);
    }
    pFileSystem->CloseFile(pFile);

    setup.Finalize();
}

//! AllowAppend を指定せずファイル外まで読み書きを行った場合を検証します。
TEST_F(FsSaveDataFileSystemCoreTest, NotAllowAppend)
{
    static const size_t SizeTotal = 1024 * 1024;
    static const int32_t SizeBlock = 4096;
    static const size_t CountBlock = SizeTotal / SizeBlock;
    static const int64_t SizeFile = 512;
    int64_t sizeMeta = nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock);
    int64_t sizeData = nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);
    SaveDataFileSystemSetup setup;
    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    setup.Initialize(sizeMeta, sizeData, SizeBlock);

    nn::fssystem::save::IFileSystem* pFileSystem = static_cast<nn::fssystem::save::IFileSystem*>(&fileSystem);

    nn::fssystem::save::IFile* pFile = nullptr;
    std::unique_ptr<char[]> buffer(new char[SizeFile * 2]);
    nn::fssystem::save::Path pathFile;
    NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/file"));

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, SizeFile));
    NNT_ASSERT_RESULT_SUCCESS(
        pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)
    );
    // ストレージサイズを超えない範囲を読み込み
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, buffer.get(), SizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(SizeFile - 1, buffer.get(), 1));

    // ストレージサイズを超える範囲でも読み込みエラーにならない
    NNT_ASSERT_RESULT_SUCCESS(
        pFile->ReadBytes(0, buffer.get(), SizeFile + 1)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        pFile->ReadBytes(SizeFile - 1, buffer.get(), 2)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        pFile->ReadBytes(0, buffer.get(), SizeFile * 2)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        pFile->ReadBytes(SizeFile, buffer.get(), SizeFile)
    );

    // ストレージサイズを超えない範囲に書き込み
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, buffer.get(), SizeFile));
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(SizeFile - 1, buffer.get(), 1));

    // ストレージサイズを超える範囲ならエラー
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
        pFile->WriteBytes(0, buffer.get(), SizeFile + 1)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
        pFile->WriteBytes(SizeFile - 1, buffer.get(), 2)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
        pFile->WriteBytes(0, buffer.get(), SizeFile * 2)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
        pFile->WriteBytes(SizeFile, buffer.get(), SizeFile)
    );
    pFileSystem->CloseFile(pFile);

    setup.Finalize();
}

//! AllowAppend を指定して容量オーバーまで書き込む処理を検証します
TEST_F(FsSaveDataFileSystemCoreTest, OverAllowAppend)
{
    static const size_t SizeTotal = 1024 * 1024;
    static const int32_t SizeBlock = 4096;
    static const size_t CountBlock = SizeTotal / SizeBlock;
    static const int64_t SizeFile = SizeBlock * 2;
    int64_t sizeMeta = nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock);
    int64_t sizeData = nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);
    SaveDataFileSystemSetup setup;
    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    setup.Initialize(sizeMeta, sizeData, SizeBlock);

    nn::fssystem::save::IFileSystem* pFileSystem = &fileSystem;

    nn::fssystem::save::IFile* pFile = nullptr;

    std::unique_ptr<char[]> buffer(new char[SizeTotal]);
    std::iota(buffer.get(), buffer.get() + SizeTotal, '\x0');

    nn::fssystem::save::Path pathFile;
    NNT_ASSERT_RESULT_SUCCESS(pathFile.Initialize("/file"));
    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(pathFile, SizeFile));
    NNT_ASSERT_RESULT_SUCCESS(
        pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend )
    );
    NN_UTIL_SCOPE_EXIT
    {
        pFileSystem->CloseFile(pFile);
        setup.Finalize();
    };

    // ストレージサイズを超えない範囲に書き込み
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, buffer.get(), SizeFile));

    // 限界以上まで拡張するように書き込み
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultOutOfResource,
        pFile->WriteBytes(0, buffer.get(), SizeTotal)
    );

    // サイズが正しいか確認する
    int64_t size = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
    ASSERT_EQ(SizeFile, size);

    // データが書き換わってないか確認する
    std::unique_ptr<char[]> readBuffer(new char[SizeFile]);
    NNT_ASSERT_RESULT_SUCCESS(pFile->ReadBytes(0, readBuffer.get(), SizeFile));
    ASSERT_EQ(0, memcmp(readBuffer.get(), buffer.get(), SizeFile));

    // ファイルを開き直すことで、ファイルサイズが実質的に変わってないことを確認する
    pFileSystem->CloseFile(pFile);

    NNT_ASSERT_RESULT_SUCCESS(
        pFileSystem->OpenFile(&pFile, pathFile, nn::fs::OpenMode_Write )
    );
    NNT_ASSERT_RESULT_SUCCESS(pFile->WriteBytes(0, buffer.get(), SizeFile));
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
        pFile->WriteBytes(0, buffer.get(), SizeFile + 1)
    );
}

// 4 GB を超えるオフセットのテスト
TEST_F(FsSaveDataFileSystemCoreLargeTest, File)
{
    static const size_t SizeBlock = 16 * 1024;
    static const size_t CountBlock = 8 * 1024 * 1024;

    const int64_t sizeMeta =
        nn::fssystem::save::SaveDataFileSystemCore::QueryMetaSize(SizeBlock, CountBlock);
    const int64_t sizeData =
        nn::fssystem::save::SaveDataFileSystemCore::QueryDataSize(SizeBlock, CountBlock);

    SaveDataFileSystemSetup setup;
    setup.Initialize(sizeMeta, sizeData, SizeBlock);

    nn::fssystem::save::SaveDataFileSystemCore& fileSystem = setup.GetFileSystem();
    TestLargeOffsetAccess(&fileSystem, "/file");

    // 先頭と末尾の破壊チェック用
    setup.Finalize();
}

