﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <algorithm>
#include <memory>
#include <random>
#include <nn/fs.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/save/fs_RemapStorage.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Compiler.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

class RemapStorageTest : public ::testing::Test
{
protected:
    static const int MapCountMax = 1000;

    static const int DataStorageCount = 2;
    static const size_t DataStorageSize =
        nn::fssystem::save::RemapStorage::SizeBlockLarge * MapCountMax;
    static const size_t EntryStorageSize = 16 * 1024;

protected:
    RemapStorageTest() NN_NOEXCEPT
    {
        // メタデータ用ストレージを用意する
        const size_t metaSize
            = static_cast<size_t>(nn::fssystem::save::RemapStorage::QueryMetaSize());
        m_MetaBuffer.resize(metaSize);
        nn::fs::IStorage* pMetaStorage
            = new nn::fs::MemoryStorage(m_MetaBuffer.data(), metaSize);
        m_MetaStorage.reset(pMetaStorage);

        // エントリ用データを用意する
        m_EntryBuffer.resize(EntryStorageSize);
        nn::fs::IStorage* pEntryStorage
            = new nn::fs::MemoryStorage(m_EntryBuffer.data(), EntryStorageSize);
        m_EntryStorage.reset(pEntryStorage);

        // データ用ストレージを用意する
        for( int dataStorageIndex = 0; dataStorageIndex < DataStorageCount; ++dataStorageIndex )
        {
            m_DataBuffers[dataStorageIndex].resize(DataStorageSize);
            nn::fs::IStorage* pDataStorage = new nn::fs::MemoryStorage(
                m_DataBuffers[dataStorageIndex].data(),
                DataStorageSize
            );
            m_DataStorages[dataStorageIndex].reset(pDataStorage);
        }

        // ストレージを用意する
        nn::fssystem::save::RemapStorage::Format(
            nn::fs::SubStorage(m_MetaStorage.get(), 0, m_MetaBuffer.size()),
            MapCountMax
        );
        nn::fssystem::save::RemapStorage* pRemapStorage = new nn::fssystem::save::RemapStorage;
        m_RemapStorage.reset(pRemapStorage);
        m_RemapStorage->Initialize(
            nn::fs::SubStorage(m_MetaStorage.get(), 0, m_MetaBuffer.size()),
            nn::fs::SubStorage(m_EntryStorage.get(), 0, m_EntryBuffer.size())
        );
        for( int dataStorageIndex = 0; dataStorageIndex < DataStorageCount; ++dataStorageIndex )
        {
            m_RemapStorage->RegisterStorage(
                nn::fs::SubStorage(
                    m_DataStorages[dataStorageIndex].get(),
                    0,
                    DataStorageSize
                ),
                dataStorageIndex
            );
        }
    }

    nn::fssystem::save::RemapStorage& GetRemapStorage() NN_NOEXCEPT
    {
        return *m_RemapStorage.get();
    }

    void Reload() NN_NOEXCEPT
    {
        m_RemapStorage->Flush();

        nn::fssystem::save::RemapStorage* pRemapStorage = new nn::fssystem::save::RemapStorage;
        m_RemapStorage.reset(pRemapStorage);
        m_RemapStorage->Initialize(
            nn::fs::SubStorage(m_MetaStorage.get(), 0, m_MetaBuffer.size()),
            nn::fs::SubStorage(m_EntryStorage.get(), 0, m_EntryBuffer.size())
        );

        for( int dataStorageIndex = 0; dataStorageIndex < DataStorageCount; ++dataStorageIndex )
        {
            m_RemapStorage->RegisterStorage(
                nn::fs::SubStorage(
                    m_DataStorages[dataStorageIndex].get(),
                    0,
                    DataStorageSize
                ),
                dataStorageIndex
            );
        }
    }

private:
    nnt::fs::util::Vector<char> m_MetaBuffer;
    nnt::fs::util::Vector<char> m_EntryBuffer;
    nnt::fs::util::Vector<char> m_DataBuffers[DataStorageCount];
    std::unique_ptr<nn::fs::IStorage> m_MetaStorage;
    std::unique_ptr<nn::fs::IStorage> m_EntryStorage;
    std::unique_ptr<nn::fs::IStorage> m_DataStorages[DataStorageCount];
    std::unique_ptr<nn::fssystem::save::RemapStorage> m_RemapStorage;
};

NN_DEFINE_STATIC_CONSTANT(const int RemapStorageTest::MapCountMax);
NN_DEFINE_STATIC_CONSTANT(const int RemapStorageTest::DataStorageCount);
NN_DEFINE_STATIC_CONSTANT(const size_t RemapStorageTest::DataStorageSize);
NN_DEFINE_STATIC_CONSTANT(const size_t RemapStorageTest::EntryStorageSize);

// 基本的な挙動を確認するテストです
TEST_F(RemapStorageTest, Basic)
{
    const int64_t mapSize = nn::fssystem::save::RemapStorage::SizeBlockSmall;
    const int64_t resizedMapSize = nn::fssystem::save::RemapStorage::SizeBlockSmall * 2;
    const int64_t alignment = nn::fssystem::save::RemapStorage::AlignmentSmall;
    const int mapCount = 16;

    int64_t virtualOffsets[mapCount];

    nnt::fs::util::Vector<char> readBuffer(mapSize);
    nnt::fs::util::Vector<char> writeBuffer(mapSize);

    // マッピングを登録する
    for( auto& offset : virtualOffsets )
    {
        nn::Result result = GetRemapStorage().RegisterMap(
                     &offset,
                     mapSize,
                     alignment,
                     (&offset - virtualOffsets) % DataStorageCount
                 );
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // 読み書きする
    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

    for( auto offset : virtualOffsets )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(
                offset,
                writeBuffer.data(),
                writeBuffer.size()
            )
        );

        std::fill(readBuffer.begin(), readBuffer.end(), '\x0');
        NNT_EXPECT_RESULT_SUCCESS(
            GetRemapStorage().Read(
                offset,
                readBuffer.data(),
                readBuffer.size()
            )
        );
        EXPECT_EQ(readBuffer, writeBuffer);

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            GetRemapStorage().Read(
                offset + readBuffer.size(),
                readBuffer.data(),
                1
            )
        );
    }

    // リサイズする
    for( auto offset : virtualOffsets )
    {
        nn::Result result = GetRemapStorage().ExpandMap(offset, resizedMapSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    readBuffer.resize(resizedMapSize);
    writeBuffer.resize(resizedMapSize);

    // 読み書きする
    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

    for( auto offset : virtualOffsets )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(
                offset,
                writeBuffer.data(),
                writeBuffer.size()
            )
        );

        std::fill(readBuffer.begin(), readBuffer.end(), '\x0');
        NNT_EXPECT_RESULT_SUCCESS(
            GetRemapStorage().Read(
                offset,
                readBuffer.data(),
                readBuffer.size()
            )
        );
        EXPECT_EQ(readBuffer, writeBuffer);
    }

    // リロードする
    Reload();

    // 読み書きする
    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

    for( auto offset : virtualOffsets )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(
                offset,
                writeBuffer.data(),
                writeBuffer.size()
            )
        );

        std::fill(readBuffer.begin(), readBuffer.end(), '\x0');
        NNT_EXPECT_RESULT_SUCCESS(
            GetRemapStorage().Read(
                offset,
                readBuffer.data(),
                readBuffer.size()
            )
        );
        EXPECT_EQ(readBuffer, writeBuffer);
    }
}

// 境界値テストです
TEST_F(RemapStorageTest, Boundary)
{
    // RegisterMap に境界値付近の値を渡したときの挙動を確認する
    {
        int64_t dummyVirtualOffset;

        // サイズ
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                -1,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                0,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                1,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall - 1,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );
        NNT_EXPECT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall + 1,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );
        NNT_EXPECT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                GetRemapStorage().GetMapSizeMax(),
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                GetRemapStorage().GetMapSizeMax()
                    + nn::fssystem::save::RemapStorage::SizeBlockSmall,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                -1,
                nn::fssystem::save::RemapStorage::AlignmentLarge,
                0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                0,
                nn::fssystem::save::RemapStorage::AlignmentLarge,
                0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                1,
                nn::fssystem::save::RemapStorage::AlignmentLarge,
                0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockLarge - 1,
                nn::fssystem::save::RemapStorage::AlignmentLarge,
                0
            )
        );
        NNT_EXPECT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockLarge,
                nn::fssystem::save::RemapStorage::AlignmentLarge,
                0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockLarge + 1,
                nn::fssystem::save::RemapStorage::AlignmentLarge,
                0
            )
        );
        NNT_EXPECT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                GetRemapStorage().GetMapSizeMax(),
                nn::fssystem::save::RemapStorage::AlignmentLarge,
                0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().RegisterMap(
                &dummyVirtualOffset,
                GetRemapStorage().GetMapSizeMax()
                    + nn::fssystem::save::RemapStorage::SizeBlockLarge,
                nn::fssystem::save::RemapStorage::AlignmentLarge,
                0
            )
        );
    }

    // Read に境界値付近の値を渡したときの挙動を確認する
    {
        int64_t virtualOffset;
        char buffer[1025];

        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                &virtualOffset,
                1024,
                nn::fssystem::save::RemapStorage::SizeBlockSmall,
                1
            )
        );

        // サイズとオフセット
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Read(virtualOffset, buffer, 1024)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Read(virtualOffset, buffer, 1025)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Read(virtualOffset + 1, buffer, 1023)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Read(virtualOffset + 1, buffer, 1024)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Read(virtualOffset + 1023, buffer, 1)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Read(virtualOffset + 1023, buffer, 2)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            GetRemapStorage().Read(virtualOffset + 1024, buffer, 1)
        );
    }

    // Write に境界値付近の値を渡したときの挙動を確認する
    {
        int64_t virtualOffset;
        char buffer[1024];

        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                &virtualOffset,
                1024,
                nn::fssystem::save::RemapStorage::SizeBlockSmall,
                1
            )
        );

        // サイズとオフセット
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(virtualOffset, buffer, 1024)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(virtualOffset, buffer, 1025)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(virtualOffset + 1, buffer, 1023)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(virtualOffset + 1, buffer, 1024)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(virtualOffset + 1023, buffer, 1)
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(virtualOffset + 1023, buffer, 2)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            GetRemapStorage().Write(virtualOffset + 1024, buffer, 1)
        );
    }

    // ExpandMap に境界値付近の値を渡したときの挙動を確認する
    {
        int64_t virtualOffset;

        // サイズ
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                &virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * 2,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                1
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            GetRemapStorage().ExpandMap(
                virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * 3 - 1
            )
        );
        NNT_EXPECT_RESULT_SUCCESS(
            GetRemapStorage().ExpandMap(
                virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * 3
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            GetRemapStorage().ExpandMap(
                virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * 3 + 1
            )
        );
        NNT_EXPECT_RESULT_SUCCESS(
            GetRemapStorage().ExpandMap(
                virtualOffset,
                GetRemapStorage().GetMapSizeMax()
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().ExpandMap(
                virtualOffset,
                GetRemapStorage().GetMapSizeMax()
                    + nn::fssystem::save::RemapStorage::SizeBlockSmall
            )
        );

        // オフセット
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                &virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * 2,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                1
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            GetRemapStorage().ExpandMap(
                virtualOffset - 1,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * 3
            )
        );
        NNT_EXPECT_RESULT_SUCCESS(
            GetRemapStorage().ExpandMap(
                virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * 4
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            GetRemapStorage().ExpandMap(
                virtualOffset + 1,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * 5
            )
        );
    }
} // NOLINT(impl/function_size)

// 不正な入力を与えるテストです
TEST_F(RemapStorageTest, Error)
{
    int64_t virtualOffset;
    nn::Result result = GetRemapStorage().RegisterMap(
                 &virtualOffset,
                 nn::fssystem::save::RemapStorage::SizeBlockSmall,
                 nn::fssystem::save::RemapStorage::AlignmentSmall,
                 0
             );
    NNT_ASSERT_RESULT_SUCCESS(result);

    // ExpandMap が期待通りのエラーを出力することを確認する
    {
        // 無効なサイズ
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().ExpandMap(virtualOffset, -1)
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().ExpandMap(virtualOffset, 0)
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().ExpandMap(virtualOffset, 1)
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().ExpandMap(
                virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall - 1
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().ExpandMap(
                virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall
            )
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            GetRemapStorage().ExpandMap(
                virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall + 1
            )
        );

        // 無効なオフセット
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            GetRemapStorage().ExpandMap(
                0x7f00000000000000,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * 3
            )
        );
    }

    // RegisterMap が期待通りのエラーを出力することを確認する
    {
        int64_t resizeVirtualOffset;

        // 無効なサイズ
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().RegisterMap(
                &virtualOffset,
                -1,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().RegisterMap(
                &virtualOffset,
                0,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().RegisterMap(
                &virtualOffset,
                0x1000000000000000,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );

        // アライメント違反
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            GetRemapStorage().RegisterMap(
                &virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall,
                nn::fssystem::save::RemapStorage::AlignmentLarge,
                0
            )
        );

        // 登録数溢れ
        result = nn::ResultSuccess();

        for( int mapIndex = 1; result.IsSuccess(); ++mapIndex )
        {
            result = GetRemapStorage().RegisterMap(
                         &resizeVirtualOffset,
                         nn::fssystem::save::RemapStorage::SizeBlockSmall * 2,
                         nn::fssystem::save::RemapStorage::AlignmentSmall,
                         mapIndex % DataStorageCount
                     );
        }

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultMapFull, result);
    }

    // Read が期待通りのエラーを出力することを確認する
    {
        char buffer[nn::fssystem::save::RemapStorage::SizeBlockSmall * 3] = {};

        // 無効なオフセット
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            GetRemapStorage().Read(
                0x7f00000000000000,
                buffer,
                nn::fssystem::save::RemapStorage::SizeBlockSmall
            )
        );
    }

    // Write が期待通りのエラーを出力することを確認する
    {
        char buffer[nn::fssystem::save::RemapStorage::SizeBlockSmall * 3] = {};

        // 無効なオフセット
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            GetRemapStorage().Write(
                0x7f00000000000000,
                buffer,
                nn::fssystem::save::RemapStorage::SizeBlockSmall
            )
        );
    }
} // NOLINT(impl/function_size)

// 正しく再マウントできることを確認するテストです
TEST_F(RemapStorageTest, Remount)
{
    const int countMax = 64;
    int64_t virtualOffsets[countMax] = {};

    const size_t blockSize = 1024;
    nnt::fs::util::Vector<char> readBuffer(blockSize);
    nnt::fs::util::Vector<char> writeBuffer(blockSize * 2);

    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

    for( int count = 0; count < countMax; ++count )
    {
        Reload();

        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                virtualOffsets + count,
                blockSize,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(
                virtualOffsets[count],
                writeBuffer.data() + count,
                blockSize
            )
        );

        for( int index = 0; index <= count; ++index )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                GetRemapStorage().Read(
                    virtualOffsets[index],
                    readBuffer.data(),
                    blockSize
                )
            );
            EXPECT_TRUE(std::equal(
                readBuffer.begin(),
                readBuffer.end(),
                writeBuffer.begin() + index
            ));
        }
    }
}

// 登録及び拡張回数の上下限の取得テストです
TEST_F(RemapStorageTest, UpdateCountBounds)
{
    // 現実装で期待される結果を返すかどうか
    for( size_t entrySize = 1024; entrySize <= 1024 * 1024 * 1024; entrySize *= 2 )
    {
        const int64_t lower
            = nn::fssystem::save::RemapStorage::GetMapUpdateCountLowerBound(entrySize);
        const int64_t upper
            = nn::fssystem::save::RemapStorage::GetMapUpdateCountUpperBound(entrySize);
        EXPECT_GE(upper, lower + (lower - 1) / 2);
        EXPECT_LE(upper, lower + (lower - 1) / 2 + 1);
    }

    // ストレージを用意する
    const size_t metaSize
        = static_cast<size_t>(nn::fssystem::save::RemapStorage::QueryMetaSize());
    nnt::fs::util::Vector<char> metaBuffer(metaSize);
    nn::fs::MemoryStorage metaStorage(metaBuffer.data(), metaSize);

    const size_t dataSize = 1024;
    nnt::fs::util::Vector<char> dataBuffer(dataSize);
    nn::fs::MemoryStorage dataStorage(dataBuffer.data(), dataSize);

    size_t entrySize = 2 * 1024;
    int mapCountMax = 128;

    for( int count = 0; count < 8; ++count )
    {
        nnt::fs::util::Vector<char> entryBuffer(entrySize);
        nn::fs::MemoryStorage entryStorage(entryBuffer.data(), entrySize);

        // 現実装では同じアライメントしか指定しなければ上限に等しい回数確保できるはず
        {
            nn::fssystem::save::RemapStorage remapStorage;

            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::RemapStorage::Format(
                    nn::fs::SubStorage(&metaStorage, 0, metaSize),
                    mapCountMax
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                remapStorage.Initialize(
                    nn::fs::SubStorage(&metaStorage, 0, metaSize),
                    nn::fs::SubStorage(&entryStorage, 0, entrySize)
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                remapStorage.RegisterStorage(
                    nn::fs::SubStorage(&dataStorage, 0, dataSize),
                    0
                )
            );

            int64_t virtualOffset = 0;

            NNT_ASSERT_RESULT_SUCCESS(
                remapStorage.RegisterMap(
                    &virtualOffset,
                    nn::fssystem::save::RemapStorage::SizeBlockLarge / 2,
                    nn::fssystem::save::RemapStorage::AlignmentSmall,
                    0
                )
            );

            const int64_t upper
                = nn::fssystem::save::RemapStorage::GetMapUpdateCountUpperBound(entrySize);

            for( int64_t registerCount = 1; registerCount < upper; ++registerCount )
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    remapStorage.RegisterMap(
                        &virtualOffset,
                        nn::fssystem::save::RemapStorage::SizeBlockLarge,
                        nn::fssystem::save::RemapStorage::AlignmentSmall,
                        0
                    )
                );
            }

            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultMapFull,
                remapStorage.RegisterMap(
                    &virtualOffset,
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentSmall,
                    0
                )
            );
        }

        // 現実装では大きいほうのブロックサイズ以上でアライメントを交互に指定すれば下限に等しい回数しか確保できないはず
        {
            nn::fssystem::save::RemapStorage remapStorage;

            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::RemapStorage::Format(
                    nn::fs::SubStorage(&metaStorage, 0, metaSize),
                    mapCountMax
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                remapStorage.Initialize(
                    nn::fs::SubStorage(&metaStorage, 0, metaSize),
                    nn::fs::SubStorage(&entryStorage, 0, entrySize)
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                remapStorage.RegisterStorage(
                    nn::fs::SubStorage(&dataStorage, 0, dataSize),
                    0
                )
            );

            int64_t virtualOffset = 0;

            NNT_ASSERT_RESULT_SUCCESS(
                remapStorage.RegisterMap(
                    &virtualOffset,
                    nn::fssystem::save::RemapStorage::SizeBlockLarge / 2,
                    nn::fssystem::save::RemapStorage::AlignmentSmall,
                    0
                )
            );

            const int64_t lower
                = nn::fssystem::save::RemapStorage::GetMapUpdateCountLowerBound(entrySize);

            for( int64_t registerCount = 1; registerCount < lower; ++registerCount )
            {
                nn::Result result = remapStorage.RegisterMap(
                    &virtualOffset,
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    ((registerCount % 2) == 0)
                        ? nn::fssystem::save::RemapStorage::AlignmentSmall
                        : nn::fssystem::save::RemapStorage::AlignmentLarge,
                    0
                );
                NNT_EXPECT_RESULT_SUCCESS(result) << registerCount;
            }

            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultMapFull,
                remapStorage.RegisterMap(
                    &virtualOffset,
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    ((lower % 2) == 0)
                        ? nn::fssystem::save::RemapStorage::AlignmentSmall
                        : nn::fssystem::save::RemapStorage::AlignmentLarge,
                    0
                )
            );
        }

        entrySize *= 2;
        mapCountMax *= 2;
    }
} // NOLINT(impl/function_size)

// 登録できるだけマッピングを登録してみるテストです
TEST_F(RemapStorageTest, RegisterMapLimit)
{
    const size_t metaSize
        = static_cast<size_t>(nn::fssystem::save::RemapStorage::QueryMetaSize());
    nnt::fs::util::Vector<char> metaBuffer(metaSize);
    nn::fs::MemoryStorage metaStorage(metaBuffer.data(), metaSize);

    const size_t dataSize = 1024;
    nnt::fs::util::Vector<char> dataBuffer(dataSize);
    nn::fs::MemoryStorage dataStorage(dataBuffer.data(), dataSize);

    // エントリの限界まで
    for( size_t entrySize = 4 * 1024; entrySize <= 64 * 1024; entrySize *= 2 )
    {
        const auto mapCountMax
            = nn::fssystem::save::RemapStorage::GetMapUpdateCountLowerBound(entrySize);
        nnt::fs::util::Vector<char> entryBuffer(entrySize);
        nn::fs::MemoryStorage entryStorage(entryBuffer.data(), entrySize);

        nn::fssystem::save::RemapStorage remapStorage;

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::RemapStorage::Format(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                static_cast<int>(mapCountMax)
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.Initialize(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                nn::fs::SubStorage(&entryStorage, 0, entrySize)
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.RegisterStorage(
                nn::fs::SubStorage(&dataStorage, 0, dataSize),
                0
            )
        );

        int64_t virtualOffset = 0;

        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.RegisterMap(
                &virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockLarge / 2,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );

        for( int mapCount = 1; ; ++mapCount )
        {
            const int64_t alignment = mapCount % 2 == 0
                ? nn::fssystem::save::RemapStorage::AlignmentSmall
                : nn::fssystem::save::RemapStorage::AlignmentLarge;
            nn::Result result = remapStorage.RegisterMap(
                         &virtualOffset,
                         nn::fssystem::save::RemapStorage::SizeBlockLarge,
                         alignment,
                         0
                     );

            if( result.IsFailure() )
            {
                EXPECT_GE(
                    mapCount,
                    nn::fssystem::save::RemapStorage::GetMapUpdateCountLowerBound(entrySize)
                );
                EXPECT_LE(
                    mapCount,
                    nn::fssystem::save::RemapStorage::GetMapUpdateCountUpperBound(entrySize)
                );
                NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultMapFull, result);
                break;
            }
        }
    }

    // 仮想アドレス空間の限界まで
    {
        const size_t entrySize = 64 * 1024;
        const int mapCountMax = 256;
        nnt::fs::util::Vector<char> entryBuffer(entrySize);
        nn::fs::MemoryStorage entryStorage(entryBuffer.data(), entrySize);

        nn::fssystem::save::RemapStorage remapStorage;

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::RemapStorage::Format(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                mapCountMax
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.Initialize(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                nn::fs::SubStorage(&entryStorage, 0, entrySize)
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.RegisterStorage(
                nn::fs::SubStorage(&dataStorage, 0, dataSize),
                0
            )
        );

        int64_t virtualOffset = 0;

        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.RegisterMap(
                &virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );

        // 拡張はエントリに余裕がある限りできる
        for( int expandCount = 0; expandCount < mapCountMax * 2; ++expandCount )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                remapStorage.ExpandMap(
                    virtualOffset,
                    nn::fssystem::save::RemapStorage::SizeBlockSmall * (expandCount + 2)
                )
            );
        }

        // 新規登録回数は Format 時に指定したパラメータで決まる
        for( int registerCount = 1; registerCount < mapCountMax; ++registerCount )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                remapStorage.RegisterMap(
                    &virtualOffset,
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    0
                )
            );
        }

        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultMapFull,
            remapStorage.RegisterMap(
                &virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockLarge,
                nn::fssystem::save::RemapStorage::AlignmentLarge,
                0
            )
        );
    }
} // NOLINT(impl/function_size)

NNT_DISABLE_OPTIMIZATION
// あるマッピングに書き込んでも他のマッピングを上書きしないことを確認するテストです
TEST_F(RemapStorageTest, NotOverwriteOtherMap)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    const int mapCountMax = 200;
    int64_t virtualOffsets[mapCountMax] = {};
    size_t sizes[mapCountMax] = {};
    size_t maxSize = 0;

    for( int mapIndex = 0; mapIndex < mapCountMax; ++mapIndex )
    {
        const bool isLarge = std::uniform_int_distribution<>(0, 1)(mt) == 0;
        const int64_t blockSize = isLarge
            ? nn::fssystem::save::RemapStorage::SizeBlockLarge
            : nn::fssystem::save::RemapStorage::SizeBlockSmall;
        const int64_t alignment = isLarge
            ? nn::fssystem::save::RemapStorage::AlignmentLarge
            : nn::fssystem::save::RemapStorage::AlignmentSmall;
        const int storageType = std::uniform_int_distribution<>(0, DataStorageCount - 1)(mt);

        sizes[mapIndex]
            = static_cast<size_t>(std::uniform_int_distribution<>(1, 5)(mt) * blockSize);
        maxSize = std::max(maxSize, sizes[mapIndex]);
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                virtualOffsets + mapIndex,
                sizes[mapIndex],
                alignment,
                storageType
            )
        );
    }

    nnt::fs::util::Vector<char> readBuffer(maxSize);
    nnt::fs::util::Vector<char> writeBuffer(maxSize);
    nnt::fs::util::Vector<char> overwriteBuffer(maxSize, 0);

    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');
    for( int mapIndex = 0; mapIndex < mapCountMax; ++mapIndex )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(
                virtualOffsets[mapIndex],
                writeBuffer.data(),
                sizes[mapIndex]
            )
        );
    }

    for( int overwriteIndex = 0; overwriteIndex < mapCountMax; ++overwriteIndex )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(
                virtualOffsets[overwriteIndex],
                overwriteBuffer.data(),
                sizes[overwriteIndex]
            )
        );

        for( int mapIndex = 0; mapIndex < mapCountMax; ++mapIndex )
        {
            if( mapIndex != overwriteIndex )
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    GetRemapStorage().Read(
                        virtualOffsets[mapIndex],
                        readBuffer.data(),
                        sizes[mapIndex]
                    )
                );
                EXPECT_TRUE(std::equal(
                    readBuffer.begin(),
                    readBuffer.begin() + sizes[mapIndex],
                    writeBuffer.begin()
                ));
            }
        }

        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().Write(
                virtualOffsets[overwriteIndex],
                writeBuffer.data(),
                sizes[overwriteIndex]
            )
        );
    }
}
NNT_RESTORE_OPTIMIZATION

// 32 ビットでは扱えないサイズのマッピングを正しく扱えることを確認するテストです
TEST_F(RemapStorageTest, HugeMap)
{
    int64_t virtualOffset = 0;
    const int64_t size = 0x100000000 + nn::fssystem::save::RemapStorage::SizeBlockSmall;

    NNT_ASSERT_RESULT_SUCCESS(
        GetRemapStorage().RegisterMap(
            &virtualOffset,
            size,
            nn::fssystem::save::RemapStorage::AlignmentSmall,
            0
        )
    );

    nnt::fs::util::Vector<char> readBuffer(nn::fssystem::save::RemapStorage::SizeBlockSmall * 4);
    nnt::fs::util::Vector<char> writeBuffer(nn::fssystem::save::RemapStorage::SizeBlockSmall * 4);

    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');
    NNT_ASSERT_RESULT_SUCCESS(
        GetRemapStorage().Write(
            virtualOffset,
            writeBuffer.data(),
            writeBuffer.size()
        )
    );
    std::fill(readBuffer.begin(), readBuffer.end(), '\x0');
    NNT_ASSERT_RESULT_SUCCESS(
        GetRemapStorage().Read(
            virtualOffset,
            readBuffer.data(),
            readBuffer.size()
        )
    );
    EXPECT_EQ(writeBuffer, readBuffer);

    NNT_ASSERT_RESULT_SUCCESS(
        GetRemapStorage().ExpandMap(
            virtualOffset,
            size + nn::fssystem::save::RemapStorage::SizeBlockSmall
        )
    );

    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x1');
    NNT_ASSERT_RESULT_SUCCESS(
        GetRemapStorage().Write(
            virtualOffset,
            writeBuffer.data(),
            writeBuffer.size()
        )
    );
    std::fill(readBuffer.begin(), readBuffer.end(), '\x0');
    NNT_ASSERT_RESULT_SUCCESS(
        GetRemapStorage().Read(
            virtualOffset,
            readBuffer.data(),
            readBuffer.size()
        )
    );
    EXPECT_EQ(writeBuffer, readBuffer);
}

// 正しくランダムアクセスができることを確認するテストです
TEST_F(RemapStorageTest, RandomAccess)
{
    int64_t virtualOffset = 0;
    const size_t blockSize = nn::fssystem::save::RemapStorage::SizeBlockSmall;
    const int64_t blockCountMax = 100;

    NNT_ASSERT_RESULT_SUCCESS(
        GetRemapStorage().RegisterMap(
            &virtualOffset,
            nn::fssystem::save::RemapStorage::SizeBlockSmall,
            nn::fssystem::save::RemapStorage::AlignmentSmall,
            0
        )
    );
    for( int blockCount = 2; blockCount <= blockCountMax; ++blockCount )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().ExpandMap(
                virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * blockCount
            )
        );
    }

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    nnt::fs::util::Vector<char> readBuffer(blockSize * blockCountMax);
    nnt::fs::util::Vector<char> writeBuffer(blockSize * blockCountMax);

    std::generate(writeBuffer.begin(), writeBuffer.end(), [&mt]
    {
        return static_cast<char>(std::uniform_int_distribution<>(0, 127)(mt));
    });
    nn::Result result
        = GetRemapStorage().Write(virtualOffset, writeBuffer.data(), blockSize * blockCountMax);
    NNT_ASSERT_RESULT_SUCCESS(result);

    {
        const size_t offsetBegin = blockSize * (blockCountMax - 5);
        const size_t offsetEnd = blockSize * blockCountMax;
        for( size_t offset = offsetBegin; offset < offsetEnd; offset -= blockSize * 5 )
        {
            result = GetRemapStorage().Read(
                         virtualOffset + offset,
                         readBuffer.data(),
                         blockSize * 5
                     );
            NNT_ASSERT_RESULT_SUCCESS(result) << "failed to read " << offset;
            EXPECT_TRUE(
                std::equal(
                    writeBuffer.begin() + offset,
                    writeBuffer.begin() + offset + blockSize * 5,
                    readBuffer.begin()
                )
            );
        }
    }

    {
        const size_t offsetBegin = 0;
        const size_t offsetEnd = blockSize * blockCountMax;
        for( size_t offset = offsetBegin; offset < offsetEnd; offset += blockSize * 5 )
        {
            result = GetRemapStorage().Read(
                         virtualOffset + offset,
                         readBuffer.data(),
                         blockSize * 5
                     );
            NNT_EXPECT_RESULT_SUCCESS(result) << "failed to read " << offset;
            EXPECT_TRUE(
                std::equal(
                    writeBuffer.begin() + offset,
                    writeBuffer.begin() + offset + blockSize * 5,
                    readBuffer.begin()
                )
            );
        }
    }

    {
        const size_t offsetBegin = blockSize * (blockCountMax - 2);
        const size_t offsetEnd = blockSize * blockCountMax;
        for( size_t offset = offsetBegin; offset < offsetEnd; offset -= blockSize * 4 )
        {
            result = GetRemapStorage().Read(
                         virtualOffset + offset,
                         readBuffer.data(),
                         blockSize * 2
                     );
            NNT_EXPECT_RESULT_SUCCESS(result) << "failed to read " << offset;
            EXPECT_TRUE(
                std::equal(
                    writeBuffer.begin() + offset,
                    writeBuffer.begin() + offset + blockSize * 2,
                    readBuffer.begin()
                )
            );
        }
    }

    {
        const size_t offsetBegin = 0;
        const size_t offsetEnd = blockSize * blockCountMax;
        for( size_t offset = offsetBegin; offset < offsetEnd; offset += blockSize * 4 )
        {
            result = GetRemapStorage().Read(
                         virtualOffset + offset,
                         readBuffer.data(),
                         blockSize * 2
                     );
            NNT_EXPECT_RESULT_SUCCESS(result) << "failed to read " << offset;
            EXPECT_TRUE(
                std::equal(
                    writeBuffer.begin() + offset,
                    writeBuffer.begin() + offset + blockSize * 2,
                    readBuffer.begin()
                )
            );
        }
    }
} // NOLINT(impl/function_size)

// ストレージを登録しない状態での挙動を確認するテストです
TEST_F(RemapStorageTest, UnregisteredStorages)
{
    const int mapCount = 256;

    const size_t metaSize = static_cast<size_t>(nn::fssystem::save::RemapStorage::QueryMetaSize());
    nnt::fs::util::Vector<char> metaBuffer(metaSize);
    nn::fs::MemoryStorage metaStorage(metaBuffer.data(), metaSize);

    const size_t entrySize = 4 * 1024;
    nnt::fs::util::Vector<char> entryBuffer(entrySize);
    nn::fs::MemoryStorage entryStorage(entryBuffer.data(), entrySize);

    const size_t dataSize = 1024;
    nnt::fs::util::Vector<char> dataBuffer(dataSize);
    nn::fs::MemoryStorage dataStorage(dataBuffer.data(), dataSize);

    // フォーマットとストレージの登録をしてフラッシュ
    {
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::RemapStorage::Format(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                mapCount
            )
        );

        nn::fssystem::save::RemapStorage remapStorage;

        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.Initialize(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                nn::fs::SubStorage(&entryStorage, 0, entrySize)
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.RegisterStorage(
                nn::fs::SubStorage(&dataStorage, 0, dataSize),
                0
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(remapStorage.Flush());
    }

    // ストレージを登録し直さずに処理
    {
        nn::fssystem::save::RemapStorage remapStorage;
        int64_t virtualOffset = 0;
        nnt::fs::util::Vector<char> buffer(nn::fssystem::save::RemapStorage::SizeBlockSmall);

        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.Initialize(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                nn::fs::SubStorage(&entryStorage, 0, entrySize)
            )
        );

        // マッピングの登録やリサイズとフラッシュはできる
        NNT_ASSERT_RESULT_SUCCESS(remapStorage.Flush());

        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.RegisterMap(
                &virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );

        NNT_EXPECT_RESULT_SUCCESS(
            remapStorage.ExpandMap(
                virtualOffset,
                nn::fssystem::save::RemapStorage::SizeBlockSmall * 2
            )
        );

        // 読み書きはできない
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultMapStorageNotFound,
            remapStorage.Read(
                virtualOffset,
                buffer.data(),
                buffer.size()
            )
        );

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultMapStorageNotFound,
            remapStorage.Write(
                virtualOffset,
                buffer.data(),
                buffer.size()
            )
        );
    }
}

// リサイズおよび読み書きを規則的に繰り返すテストです
TEST_F(RemapStorageTest, Repeat)
{
    int64_t virtualOffsets[DataStorageCount] = {};
    nnt::fs::util::Vector<char> readBuffer(DataStorageSize);
    nnt::fs::util::Vector<char> writeBuffer(DataStorageSize);
    const size_t unitSize = nn::fssystem::save::RemapStorage::SizeBlockSmall;
    const size_t maxSize = unitSize * 200;
    const int64_t alignment = nn::fssystem::save::RemapStorage::AlignmentSmall;

    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

    for( int storageIndex = 0; storageIndex < DataStorageCount; ++storageIndex )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            GetRemapStorage().RegisterMap(
                virtualOffsets + storageIndex,
                unitSize,
                alignment,
                storageIndex
            )
        );
    }

    for( size_t size = unitSize * 2; size <= maxSize; size += unitSize )
    {
        nn::Result result;
        const int64_t offset = virtualOffsets[(size / unitSize) % DataStorageCount];
        result = GetRemapStorage().ExpandMap(offset, size);
        NNT_ASSERT_RESULT_SUCCESS(result)
            << "failed to resize to " << size << " bytes";
        result = GetRemapStorage().Write(offset, writeBuffer.data(), size);
        NNT_EXPECT_RESULT_SUCCESS(result)
            << "failed to write " << size << " bytes";
        std::fill(readBuffer.begin(), readBuffer.begin() + size, '\x0');
        result = GetRemapStorage().Read(offset, readBuffer.data(), size);
        NNT_EXPECT_RESULT_SUCCESS(result)
            << "failed to read " << size << " bytes";
        EXPECT_TRUE(
            std::equal(
                readBuffer.begin(),
                readBuffer.begin() + size,
                writeBuffer.begin()
            )
        );
    }
}

NNT_DISABLE_OPTIMIZATION
// マッピング登録、リサイズおよび読み書きを無作為に繰り返すテストです
TEST_F(RemapStorageTest, Random)
{
    struct Map
    {
        int64_t virtualOffset;
        int64_t size;
        int64_t alignment;
        int storageType;
    };

    nnt::fs::util::Vector<Map> maps;
    nnt::fs::util::Vector<char> buffer(DataStorageSize);
    int64_t sizes[DataStorageCount] = {};
    int updateCount = 0;
    const int64_t updateCountLowerBound
        = nn::fssystem::save::RemapStorage::GetMapUpdateCountLowerBound(EntryStorageSize);

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int count = 0; count < 16384; ++count )
    {
        switch( std::uniform_int_distribution<int>(0, 3)(mt) )
        {
        case 0:
            {
                // 登録します
                int64_t size = 0;
                int64_t alignment = 0;

                if(std::uniform_int_distribution<int>(0, 1)(mt) == 0)
                {
                    size = (std::uniform_int_distribution<int>(-4, 12)(mt)
                        * nn::fssystem::save::RemapStorage::SizeBlockSmall)
                        + std::uniform_int_distribution<int>(-5, 5)(mt) % 4;
                    alignment = nn::fssystem::save::RemapStorage::AlignmentSmall;
                }
                else
                {
                    size = (std::uniform_int_distribution<int>(-4, 12)(mt)
                        * nn::fssystem::save::RemapStorage::SizeBlockLarge)
                        + std::uniform_int_distribution<int>(-5, 5)(mt) % 4;
                    alignment = nn::fssystem::save::RemapStorage::AlignmentLarge;
                }

                Map map = {
                    0,
                    size,
                    alignment,
                    std::uniform_int_distribution<int>(0, DataStorageCount - 1)(mt)
                };

                if( sizes[map.storageType] + map.size < DataStorageSize )
                {
                    nn::Result result = GetRemapStorage().RegisterMap(
                                 &map.virtualOffset,
                                 map.size,
                                 map.alignment,
                                 map.storageType
                             );
                    if( map.size <= 0 )
                    {
                        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, result)
                            << "succeeded to register size=" << map.size
                            << ", type=" << map.storageType;
                    }
                    else if( map.size % nn::fssystem::save::RemapStorage::AlignmentSmall != 0 )
                    {
                        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidAlignment, result)
                            << "succeeded to register size=" << map.size
                            << ", type=" << map.storageType;
                    }
                    else
                    {
                        if( updateCount <= updateCountLowerBound )
                        {
                            NNT_ASSERT_RESULT_SUCCESS(result)
                                << "failed to resigter size=" << map.size
                                << ", type=" << map.storageType;
                        }
                        if( result.IsSuccess() )
                        {
                            ++updateCount;
                            maps.push_back(map);
                            sizes[map.storageType] += map.size;
                        }
                    }
                }
            }
            break;

        case 1:
            {
                // リサイズします
                if (!maps.empty())
                {
                    Map& map = maps[std::uniform_int_distribution<size_t>(0, maps.size() - 1)(mt)];
                    int64_t expandSize
                        = std::uniform_int_distribution<int>(-2, 6)(mt) * map.alignment
                        + std::uniform_int_distribution<int>(-5, 5)(mt) % 4;
                    if( sizes[map.storageType] + expandSize < DataStorageSize )
                    {
                        nn::Result result = GetRemapStorage().ExpandMap(
                                     map.virtualOffset,
                                     map.size + expandSize
                                 );
                        if( expandSize < 0 )
                        {
                            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, result);
                        }
                        else if( expandSize % map.alignment != 0 )
                        {
                            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidAlignment, result);
                        }
                        else
                        {
                            if( updateCount <= updateCountLowerBound )
                            {
                                NNT_ASSERT_RESULT_SUCCESS(result)
                                    << "failed to resize offset=" << map.virtualOffset
                                    << ", size=" << map.size
                                    << "+" << expandSize;
                            }
                            if( result.IsSuccess() )
                            {
                                ++updateCount;
                                map.size += expandSize;
                                sizes[map.storageType] += expandSize;
                            }
                        }
                    }
                }
            }
            break;

        case 2:
            {
                // 読み込みます
                if (!maps.empty())
                {
                    const Map& map
                        = maps[std::uniform_int_distribution<size_t>(0, maps.size() - 1)(mt)];
                    nn::Result result = GetRemapStorage().Read(
                                 map.virtualOffset,
                                 buffer.data(),
                                 static_cast<size_t>(map.size)
                             );
                    NNT_EXPECT_RESULT_SUCCESS(result)
                        << "failed to read offset=" << map.virtualOffset
                        << ", size=" << map.size;
                }
            }
            break;

        case 3:
            {
                // 書き込みます
                if (!maps.empty())
                {
                    Map& map
                        = maps[std::uniform_int_distribution<size_t>(0, maps.size() - 1)(mt)];
                    nn::Result result = GetRemapStorage().Write(
                                 map.virtualOffset,
                                 buffer.data(),
                                 static_cast<size_t>(map.size)
                             );
                    NNT_EXPECT_RESULT_SUCCESS(result)
                        << "failed to write offset=" << map.virtualOffset
                        << ", size=" << map.size;
                }
            }
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }
} // NOLINT(impl/function_size)
NNT_RESTORE_OPTIMIZATION

// QueryRange を確認するテストです
TEST_F(RemapStorageTest, QueryRange)
{
    const int64_t mapSize = nn::fssystem::save::RemapStorage::SizeBlockSmall;
    const int64_t alignment = nn::fssystem::save::RemapStorage::AlignmentSmall;
    const int mapCount = 16;

    int64_t virtualOffsets[mapCount];

    // マッピングを登録する
    for( auto& offset : virtualOffsets )
    {
        NNT_ASSERT_RESULT_SUCCESS(GetRemapStorage().RegisterMap(
            &offset,
            mapSize,
            alignment,
            (&offset - virtualOffsets) % DataStorageCount
        ));
    }

    {
        nn::fs::QueryRangeInfo info;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNullptrArgument,
            GetRemapStorage().OperateRange(
                nullptr,
                sizeof(info),
                nn::fs::OperationId::QueryRange,
                0,
                mapSize,
                nullptr,
                0));

        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            GetRemapStorage().OperateRange(
                &info,
                0,
                nn::fs::OperationId::QueryRange,
                0,
                mapSize,
                nullptr,
                0));
    }

    for( auto offset : virtualOffsets )
    {
        nn::fs::QueryRangeInfo info;
        NNT_ASSERT_RESULT_SUCCESS(GetRemapStorage().OperateRange(
            &info,
            sizeof(info),
            nn::fs::OperationId::QueryRange,
            offset,
            mapSize,
            nullptr,
            0));
        EXPECT_EQ(0, info.aesCtrKeyTypeFlag);
        EXPECT_EQ(0, info.speedEmulationTypeFlag);
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
// 事前条件違反で正しく落ちることを確認するテストです
TEST(RemapStorageDeathTest, Precondition)
{
    const size_t entrySize = 16 * 1024;
    const size_t dataSize = 1024;
    const auto mapCount
        = nn::fssystem::save::RemapStorage::GetMapUpdateCountLowerBound(entrySize);

    const size_t metaSize
        = static_cast<size_t>(nn::fssystem::save::RemapStorage::QueryMetaSize());
    nnt::fs::util::Vector<char> metaBuffer(metaSize);
    nn::fs::MemoryStorage metaStorage(metaBuffer.data(), metaSize);

    nnt::fs::util::Vector<char> entryBuffer(entrySize);
    nn::fs::MemoryStorage entryStorage(entryBuffer.data(), entrySize);

    nnt::fs::util::Vector<char> dataBuffer(dataSize);
    nn::fs::MemoryStorage dataStorage(dataBuffer.data(), dataSize);

    nn::fssystem::save::RemapStorage remapStorage;

    const size_t bufferSize = 512;

    // QueryEntryTableSize
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fssystem::save::RemapStorage::QueryEntryTableSize(0),
            ""
        );
        nn::fssystem::save::RemapStorage::QueryEntryTableSize(1);
    }

    // GetUpdateCountLowerBound
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fssystem::save::RemapStorage::GetMapUpdateCountLowerBound(0),
            ""
        );
        nn::fssystem::save::RemapStorage::GetMapUpdateCountLowerBound(1);
    }

    // GetUpdateCountUpperBound
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fssystem::save::RemapStorage::GetMapUpdateCountUpperBound(0),
            ""
        );
        nn::fssystem::save::RemapStorage::GetMapUpdateCountUpperBound(1);
    }

    // Format
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fssystem::save::RemapStorage::Format(
                nn::fs::SubStorage(&metaStorage, 0, metaSize - 1),
                static_cast<int>(mapCount)
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fssystem::save::RemapStorage::Format(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                0
            ),
            ""
        );
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::RemapStorage::Format(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                static_cast<int>(mapCount)
            )
        );
    }

    // Initialize
    {
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.Initialize(
                nn::fs::SubStorage(&metaStorage, 0, metaSize - 1),
                nn::fs::SubStorage(&entryStorage, 0, entrySize)
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.Initialize(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                nn::fs::SubStorage(&entryStorage, 0, 0)
            ),
            ""
        );
        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.Initialize(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                nn::fs::SubStorage(&entryStorage, 0, entrySize)
            )
        );
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.Initialize(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                nn::fs::SubStorage(&entryStorage, 0, entrySize)
            ),
            ""
        );
    }

    // RegisterStorage
    {
        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.RegisterStorage(
                nn::fs::SubStorage(&dataStorage, 0, dataSize),
                0
            )
        );
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.RegisterStorage(
                nn::fs::SubStorage(&dataStorage, 0, dataSize),
                0
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.RegisterStorage(
                nn::fs::SubStorage(&dataStorage, 0, dataSize),
                -1
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.RegisterStorage(
                nn::fs::SubStorage(&dataStorage, 0, dataSize),
                4
            ),
            ""
        );
    }

    // RegisterMap
    {
        int64_t offset;
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.RegisterMap(
                nullptr,
                bufferSize,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.RegisterMap(
                &offset,
                bufferSize,
                nn::fssystem::save::RemapStorage::AlignmentSmall - 1,
                0
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.RegisterMap(
                &offset,
                bufferSize,
                nn::fssystem::save::RemapStorage::AlignmentSmall + 1,
                0
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.RegisterMap(
                &offset,
                bufferSize,
                nn::fssystem::save::RemapStorage::AlignmentLarge - 1,
                0
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            remapStorage.RegisterMap(
                &offset,
                bufferSize,
                nn::fssystem::save::RemapStorage::AlignmentLarge + 1,
                0
            ),
            ""
        );
        NNT_ASSERT_RESULT_SUCCESS(
            remapStorage.RegisterMap(
                &offset,
                bufferSize,
                nn::fssystem::save::RemapStorage::AlignmentSmall,
                0
            )
        );
    }
} // NOLINT(impl/function_size)
#endif

class RemapStoragePrivateTest : public RemapStorageTest
{
protected:
    // アロケーションが意図通り行われることを確認するテストです
    void AllocateTest() NN_NOEXCEPT
    {
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        const int entryCountMax = 50;
        const int resizeCountMax = 3;
        int64_t virtualOffsets[entryCountMax] = {};
        int64_t blockCounts[entryCountMax] = {};

        for( int mapCount = 0; mapCount < entryCountMax; ++mapCount )
        {
            const int64_t blockSize = mapCount % 2 == 0
                ? nn::fssystem::save::RemapStorage::SizeBlockSmall
                : nn::fssystem::save::RemapStorage::SizeBlockLarge;
            const int64_t alignment = mapCount % 2 == 0
                ? nn::fssystem::save::RemapStorage::AlignmentSmall
                : nn::fssystem::save::RemapStorage::AlignmentLarge;
            const int storageType = mapCount % DataStorageCount;

            blockCounts[mapCount] = std::uniform_int_distribution<int64_t>(1, 100)(mt);
            nn::Result result = GetRemapStorage().RegisterMap(
                         virtualOffsets + mapCount,
                         blockSize * blockCounts[mapCount],
                         alignment,
                         storageType
                     );
            NNT_ASSERT_RESULT_SUCCESS(result);
            CheckMapEntries(GetRemapStorage());
        }

        for( int resizeCount = 0; resizeCount < resizeCountMax; ++resizeCount )
        {
            for( int mapCount = 0; mapCount < entryCountMax; ++mapCount )
            {
                const int64_t blockSize = mapCount % 2 == 0
                    ? nn::fssystem::save::RemapStorage::SizeBlockSmall
                    : nn::fssystem::save::RemapStorage::SizeBlockLarge;

                blockCounts[mapCount] += std::uniform_int_distribution<int64_t>(1, 50)(mt);
                nn::Result result = GetRemapStorage().ExpandMap(
                             virtualOffsets[mapCount],
                             blockSize * blockCounts[mapCount]
                         );
                NNT_ASSERT_RESULT_SUCCESS(result);
                CheckMapEntries(GetRemapStorage());
            }
        }

        EXPECT_EQ(entryCountMax, GetRemapStorage().m_ControlArea.countMaps);
    }

    // 登録の内部処理が正しく実行できることを確認するテストです
    void RegisterCoreTest() NN_NOEXCEPT
    {
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        const int64_t smallVirtualOffset = GetRemapStorage().MakeVirtualOffset(1, 0);
        const int64_t largeVirtualOffset = GetRemapStorage().MakeVirtualOffset(2, 0);
        const int64_t countLowerBound
            = nn::fssystem::save::RemapStorage::GetMapUpdateCountLowerBound(EntryStorageSize);
        int64_t currentSmallSize = 0;
        int64_t currentLargeSize = 0;

        for( int count = 0; count < MapCountMax; ++count )
        {
            const bool isLarge = count % 4 == 0;
            int64_t& currentSize = isLarge ? currentLargeSize : currentSmallSize;
            const int64_t offset = (isLarge ? largeVirtualOffset : smallVirtualOffset);
            const int64_t alignment = isLarge
                ? nn::fssystem::save::RemapStorage::AlignmentLarge
                : nn::fssystem::save::RemapStorage::AlignmentSmall;
            const int64_t blockSize = isLarge
                ? nn::fssystem::save::RemapStorage::SizeBlockLarge
                : nn::fssystem::save::RemapStorage::SizeBlockSmall;
            const int64_t size = std::uniform_int_distribution<int64_t>(1, 10)(mt) * blockSize;

            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidAlignment,
                GetRemapStorage().RegisterMapCore(
                    offset - 1,
                    size + currentSize,
                    currentSize,
                    alignment,
                    0
                )
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidAlignment,
                GetRemapStorage().RegisterMapCore(
                    offset + 1,
                    size + currentSize,
                    currentSize,
                    alignment,
                    0
                )
            );
            if( 0 < (offset & (GetRemapStorage().GetMapSizeMax() - 1)) )
            {
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultMapAddressAlreadyRegistered,
                    GetRemapStorage().RegisterMapCore(
                        offset - blockSize,
                        size + currentSize,
                        currentSize,
                        alignment,
                        0
                    )
                );
            }

            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidAlignment,
                GetRemapStorage().RegisterMapCore(
                    offset,
                    size + currentSize - 1,
                    currentSize,
                    alignment,
                    0
                )
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidAlignment,
                GetRemapStorage().RegisterMapCore(
                    offset,
                    size + currentSize + 1,
                    currentSize,
                    alignment,
                    0
                )
            );

            nn::Result result = GetRemapStorage().RegisterMapCore(
                         offset,
                         size + currentSize,
                         currentSize,
                         alignment,
                         0
                     );
            if( count <= countLowerBound )
            {
                NNT_EXPECT_RESULT_SUCCESS(result);
            }

            currentSize += size;
        }
    }

    // エントリの読み書きが正しくできることを確認するテストです
    void EntryTest() NN_NOEXCEPT
    {
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        for( int mapCount = 0; mapCount < GetRemapStorage().GetMapEntriesCountMax(); ++mapCount )
        {
            int64_t virtualOffset = 0;
            const int64_t blockCount = std::uniform_int_distribution<int64_t>(1, 100)(mt);
            const int64_t blockSize = mapCount % 2 == 0
                ? nn::fssystem::save::RemapStorage::SizeBlockSmall
                : nn::fssystem::save::RemapStorage::SizeBlockLarge;
            const int64_t alignment = mapCount % 2 == 0
                ? nn::fssystem::save::RemapStorage::AlignmentSmall
                : nn::fssystem::save::RemapStorage::AlignmentLarge;
            const int storageType = mapCount % DataStorageCount;

            NNT_ASSERT_RESULT_SUCCESS(
                GetRemapStorage().RegisterMap(
                    &virtualOffset,
                    blockSize * blockCount,
                    alignment,
                    storageType
                )
            );
        }

        {
            nn::fssystem::save::RemapStorage::MapEntry entry;

            int entryCount = GetRemapStorage().m_ControlArea.countMapEntries;
            for( int entryIndex = 0; entryIndex < entryCount; ++entryIndex )
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    GetRemapStorage().ReadMapEntry(
                        &entry,
                        entryIndex
                    )
                );
            }

            int64_t entryCountMax = GetRemapStorage().GetMapEntriesCountMax();
            for( int entryIndex = 0; entryIndex < entryCountMax; ++entryIndex )
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    GetRemapStorage().WriteMapEntry(
                        &entry,
                        entryIndex
                    )
                );
            }
        }
    }

    // イテレータの挙動を確認するテストです
    void IterateTest() NN_NOEXCEPT
    {
        nn::Result result;

        static const auto MapCount = 16;
        static const auto Size = nn::fssystem::save::RemapStorage::SizeBlockLarge * MapCount;
        int64_t virtualOffset = 0;

        // 正しくイテレーションできることを確認する
        {
            NNT_ASSERT_RESULT_SUCCESS(
                GetRemapStorage().RegisterMap(
                    &virtualOffset,
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    1
                )
            );

            for( int count = 1; count < MapCount; ++count )
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    GetRemapStorage().ExpandMap(
                        virtualOffset,
                        nn::fssystem::save::RemapStorage::SizeBlockLarge * (count + 1)
                    )
                );
            }

            for ( int64_t offset = virtualOffset; offset < virtualOffset + Size; offset += 123 )
            {
                nn::fssystem::save::RemapStorage::MapIterator iterator(&GetRemapStorage(), offset);
                nn::fssystem::save::RemapStorage::MapEntry previousEntry = {};
                int64_t currentSize = 0;

                while( iterator.MoveNext(Size - currentSize) )
                {
                    NNT_EXPECT_RESULT_SUCCESS(iterator.GetLastResult());
                    EXPECT_LE(offset, iterator.GetMapEntry().offsetVirtual);
                    EXPECT_GE(
                        virtualOffset + Size,
                        iterator.GetMapEntry().offsetVirtual + iterator.GetMapEntry().size
                    );
                    EXPECT_EQ(
                        nn::fssystem::save::RemapStorage::AlignmentLarge,
                        iterator.GetMapEntry().alignment
                    );
                    EXPECT_EQ(1, iterator.GetMapEntry().storageType);
                    EXPECT_NE(nullptr, iterator.GetStorage());
                    if( 0 < currentSize )
                    {
                        EXPECT_EQ(
                            previousEntry.offsetVirtual + previousEntry.size,
                            iterator.GetMapEntry().offsetVirtual
                        );
                        EXPECT_EQ(
                            nn::fssystem::save::RemapStorage::SizeBlockLarge,
                            iterator.GetMapEntry().size
                        );
                    }
                    else
                    {
                        const int64_t currentBlockSize =
                            nn::fssystem::save::RemapStorage::SizeBlockLarge
                            - (offset % nn::fssystem::save::RemapStorage::SizeBlockLarge);
                        EXPECT_EQ(currentBlockSize, iterator.GetMapEntry().size);
                    }
                    currentSize += iterator.GetMapEntry().size;
                    previousEntry = iterator.GetMapEntry();
                }

                EXPECT_EQ(Size - offset, currentSize);
                NNT_EXPECT_RESULT_SUCCESS(iterator.GetLastResult());
            }

            nn::fssystem::save::RemapStorage::MapIterator iterator(
                                                              &GetRemapStorage(),
                                                              virtualOffset + Size
                                                          );
            EXPECT_FALSE(iterator.MoveNext(GetRemapStorage().GetMapSizeMax()));
            NNT_EXPECT_RESULT_SUCCESS(iterator.GetLastResult());
        }

        // 限界までメモリを確保しても正しくイテレーションできることを確認する
        {
            int64_t virtualOffsets[3] = {};

            for( int offsetIndex = 0; offsetIndex < 3; ++offsetIndex )
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    GetRemapStorage().RegisterMap(
                        virtualOffsets + offsetIndex,
                        GetRemapStorage().GetMapSizeMax(),
                        nn::fssystem::save::RemapStorage::AlignmentLarge,
                        0
                    )
                );
            }

            for( int offsetIndex = 0; offsetIndex < 3; ++offsetIndex )
            {
                nn::fssystem::save::RemapStorage::MapIterator iterator(
                                                                  &GetRemapStorage(),
                                                                  virtualOffsets[offsetIndex]
                                                              );
                int64_t mapSize = 0;

                while( iterator.MoveNext(GetRemapStorage().GetMapSizeMax()))
                {
                    mapSize += iterator.GetMapEntry().size;
                }

                NNT_EXPECT_RESULT_SUCCESS(iterator.GetLastResult());
                EXPECT_EQ(GetRemapStorage().GetMapSizeMax(), mapSize);
            }
        }
    } // NOLINT(impl/function_size)

NNT_DISABLE_OPTIMIZATION
    // 正しくキャッシュされていることを確認するテストです
    void CacheTest() NN_NOEXCEPT
    {
        nn::Result result;

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        const int mapCountMax = 10;
        const size_t blockSize = 1024;
        nnt::fs::util::Vector<char> buffer(blockSize);
        int64_t virtualOffsets[mapCountMax] = {};

        // 最初は空のはず
        EXPECT_TRUE(IsEmpty(GetRemapStorage().m_MapEntryCache));

        // 登録
        for( int mapCount = 0; mapCount < mapCountMax; ++mapCount )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                GetRemapStorage().RegisterMap(
                    virtualOffsets + mapCount,
                    nn::fssystem::save::RemapStorage::SizeBlockSmall,
                    nn::fssystem::save::RemapStorage::AlignmentSmall,
                    mapCount % DataStorageCount
                )
            );
        }

        // 登録後も空のはず
        EXPECT_TRUE(IsEmpty(GetRemapStorage().m_MapEntryCache));

        for( int expandCount = 0; expandCount < 30; ++expandCount )
        {
            // 拡張
            for( int mapCount = 0; mapCount < mapCountMax; ++mapCount )
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    GetRemapStorage().ExpandMap(
                        virtualOffsets[mapCount],
                        nn::fssystem::save::RemapStorage::SizeBlockSmall * (expandCount + 2)
                    )
                );
            }

            // 拡張後は空になるはず
            EXPECT_TRUE(IsEmpty(GetRemapStorage().m_MapEntryCache));

            for( int accessCount = 0; accessCount < 50; ++accessCount )
            {
                // 読み込み後にキャッシュが壊れていないかどうか
                {
                    volatile int readIndex // TODO: #pragma だけでは変数が化けるので volatile にする
                        = std::uniform_int_distribution<int>(0, mapCountMax - 1)(mt);
                    const int64_t lengthMax
                        = nn::fssystem::save::RemapStorage::SizeBlockSmall * (expandCount + 2);
                    const int64_t readOffset
                        = std::uniform_int_distribution<int64_t>(0, lengthMax - 1)(mt);
                    const int64_t virtualReadOffset = virtualOffsets[readIndex] + readOffset;
                    const size_t remainLength = static_cast<size_t>(lengthMax - readOffset);
                    const size_t readLengthMax = std::min(remainLength, blockSize);
                    const size_t readLength
                        = std::uniform_int_distribution<size_t>(1, readLengthMax)(mt);

                    nn::fssystem::save::RemapStorage::MapEntry cachedMapEntry = {};
                    nn::fssystem::save::RemapStorage::MapEntry actualMapEntry = {};
                    int cachedIndexMapEntry = 0;
                    auto hit = false;

                    for( auto count = 0; count < 2 && !hit; ++count )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            GetRemapStorage().Read(
                                virtualReadOffset,
                                buffer.data(),
                                readLength
                            )
                        );

                        nn::fssystem::save::RemapStorage::MapEntryCache& cache
                            = GetRemapStorage().m_MapEntryCache;

                        hit = cache.GetMapEntry(
                                  &cachedMapEntry,
                                  &cachedIndexMapEntry,
                                  virtualReadOffset
                              );
                    }

                    ASSERT_TRUE(hit);
                    const auto mask = ~(nn::fssystem::save::RemapStorage::SizeBlockSmall - 1);
                    EXPECT_EQ(
                        virtualReadOffset & mask,
                        cachedMapEntry.offsetVirtual
                    );

                    NNT_ASSERT_RESULT_SUCCESS(
                        GetRemapStorage().ReadMapEntry(
                            &actualMapEntry,
                            cachedIndexMapEntry
                        )
                    );

                    EXPECT_EQ(actualMapEntry.offsetVirtual, cachedMapEntry.offsetVirtual);
                    EXPECT_EQ(actualMapEntry.offsetPhysical, cachedMapEntry.offsetPhysical);
                    EXPECT_EQ(actualMapEntry.size, cachedMapEntry.size);
                }

                // 書き込み後にキャッシュが壊れていないかどうか
                {
                    volatile int writeIndex // TODO: #pragma だけでは変数が化けるので volatile にする
                        = std::uniform_int_distribution<int>(0, mapCountMax - 1)(mt);
                    const int64_t lengthMax
                        = nn::fssystem::save::RemapStorage::SizeBlockSmall * (expandCount + 2);
                    const int64_t writeOffset
                        = std::uniform_int_distribution<int64_t>(0, lengthMax - 1)(mt);
                    const int64_t virtualWriteOffset = virtualOffsets[writeIndex] + writeOffset;
                    const size_t remainLength = static_cast<size_t>(lengthMax - writeOffset);
                    const size_t writeLengthMax = std::min(remainLength, blockSize);
                    const size_t writeLength
                        = std::uniform_int_distribution<size_t>(1, writeLengthMax)(mt);

                    nn::fssystem::save::RemapStorage::MapEntry cachedMapEntry = {};
                    nn::fssystem::save::RemapStorage::MapEntry actualMapEntry = {};
                    int cachedIndexMapEntry = 0;
                    auto hit = false;

                    for( auto count = 0; count < 2 && !hit; ++count )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(
                            GetRemapStorage().Read(
                                virtualWriteOffset,
                                buffer.data(),
                                writeLength
                            )
                        );

                        nn::fssystem::save::RemapStorage::MapEntryCache& cache
                            = GetRemapStorage().m_MapEntryCache;

                        hit = cache.GetMapEntry(
                                  &cachedMapEntry,
                                  &cachedIndexMapEntry,
                                  virtualWriteOffset
                              );
                    }

                    ASSERT_TRUE(hit);
                    const auto mask = ~(nn::fssystem::save::RemapStorage::SizeBlockSmall - 1);
                    EXPECT_EQ(
                        virtualWriteOffset & mask,
                        cachedMapEntry.offsetVirtual
                    );

                    NNT_ASSERT_RESULT_SUCCESS(
                        GetRemapStorage().ReadMapEntry(
                            &actualMapEntry,
                            cachedIndexMapEntry
                        )
                    );

                    EXPECT_EQ(actualMapEntry.offsetVirtual, cachedMapEntry.offsetVirtual);
                    EXPECT_EQ(actualMapEntry.offsetPhysical, cachedMapEntry.offsetPhysical);
                    EXPECT_EQ(actualMapEntry.size, cachedMapEntry.size);
                }
            }
        }
    } // NOLINT(impl/function_size)
NNT_RESTORE_OPTIMIZATION

    // テーブルを壊したときにエラーが出ることをテストします
    void CorruptTableTest() NN_NOEXCEPT
    {
        nn::Result result;

        // ストレージを準備します
        static const auto EntryCount = 16;
        static const auto EntrySize
            = EntryCount * sizeof(nn::fssystem::save::RemapStorage::MapEntry);
        static const size_t DataSize = 4 * 1024 * 1024;

        const size_t metaSize
            = static_cast<size_t>(nn::fssystem::save::RemapStorage::QueryMetaSize());
        nnt::fs::util::Vector<char> metaBuffer(metaSize);
        nn::fs::MemoryStorage metaStorage(metaBuffer.data(), metaSize);

        nnt::fs::util::Vector<nn::fssystem::save::RemapStorage::MapEntry> entryBuffer(EntryCount);
        nn::fs::MemoryStorage entryStorage(entryBuffer.data(), EntrySize);

        nnt::fs::util::Vector<char> dataBuffer(DataSize);
        nn::fs::MemoryStorage dataStorage(dataBuffer.data(), DataSize);

        nn::fssystem::save::RemapStorage remapStorage;

        // フォーマットしてマッピングを登録します
        result = nn::fssystem::save::RemapStorage::Format(
                     nn::fs::SubStorage(&metaStorage, 0, metaSize),
                     4
                 );
        NNT_ASSERT_RESULT_SUCCESS(result);

        result = remapStorage.Initialize(
                     nn::fs::SubStorage(&metaStorage, 0, metaSize),
                     nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                 );
        NNT_ASSERT_RESULT_SUCCESS(result);

        static const auto ExpandCount = 12;
        int64_t virtualOffset = 0;

        result = remapStorage.RegisterMap(
                     &virtualOffset,
                     nn::fssystem::save::RemapStorage::SizeBlockSmall,
                     nn::fssystem::save::RemapStorage::AlignmentSmall,
                     0
                 );
        NNT_ASSERT_RESULT_SUCCESS(result);
        for( auto count = 0; count < ExpandCount; ++count )
        {
            result = remapStorage.ExpandMap(
                         virtualOffset,
                         nn::fssystem::save::RemapStorage::SizeBlockSmall * (count + 2) * 2
                     );
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        result = remapStorage.Flush();
        NNT_ASSERT_RESULT_SUCCESS(result);

        remapStorage.Finalize();

        // エントリテーブルを改竄するとエラーが出ることを確認します
        static const auto BufferSize = nn::fssystem::save::RemapStorage::SizeBlockLarge;
        nnt::fs::util::Vector<char> buffer(BufferSize);

        // 不正なストレージ種別
        {
            const auto temporary = entryBuffer;
            entryBuffer[0].storageType = -1;

            result = remapStorage.Initialize(
                         nn::fs::SubStorage(&metaStorage, 0, metaSize),
                         nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                     );
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapStorageType, result);

            entryBuffer = temporary;
        }
        {
            const auto temporary = entryBuffer;
            entryBuffer[1].storageType = nn::fssystem::save::RemapStorage::CountMaxStorages;

            result = remapStorage.Initialize(
                         nn::fs::SubStorage(&metaStorage, 0, metaSize),
                         nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                     );
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapStorageType, result);

            entryBuffer = temporary;
        }

        // サイズが不正
        {
            const auto temporary = entryBuffer;
            entryBuffer[2].size = -1;

            result = remapStorage.Initialize(
                         nn::fs::SubStorage(&metaStorage, 0, metaSize),
                         nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                     );
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapOffset, result);

            entryBuffer = temporary;
        }

        // 論理オフセットの昇順に並んでいない
        {
            const auto temporary = entryBuffer;
            std::swap(entryBuffer[3], entryBuffer[4]);

            result = remapStorage.Initialize(
                         nn::fs::SubStorage(&metaStorage, 0, metaSize),
                         nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                     );
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapOffset, result);

            entryBuffer = temporary;
        }

        // 不正なアライメント
        {
            const auto temporary = entryBuffer;
            entryBuffer[5].alignment = 42;

            result = remapStorage.Initialize(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                );
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapAlignment, result);

            entryBuffer = temporary;
        }

        // アライメント制約に従っていない
        {
            const auto temporary = entryBuffer;
            ++entryBuffer[6].offsetPhysical;

            result = remapStorage.Initialize(
                         nn::fs::SubStorage(&metaStorage, 0, metaSize),
                         nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                     );
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapAlignment, result);

            entryBuffer = temporary;
        }
        {
            const auto temporary = entryBuffer;
            --entryBuffer[7].offsetVirtual;

            result = remapStorage.Initialize(
                         nn::fs::SubStorage(&metaStorage, 0, metaSize),
                         nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                     );
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapOffset, result);

            entryBuffer = temporary;
        }
        {
            const auto temporary = entryBuffer;
            entryBuffer[8].size /= 4;

            result = remapStorage.Initialize(
                         nn::fs::SubStorage(&metaStorage, 0, metaSize),
                         nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                     );
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapAlignment, result);

            entryBuffer = temporary;
        }

        // 論理オフセットが重複
        {
            const auto temporary = entryBuffer;
            entryBuffer[9].offsetVirtual -= nn::fssystem::save::RemapStorage::SizeBlockSmall;
            entryBuffer[9].size += nn::fssystem::save::RemapStorage::SizeBlockSmall;

            result = remapStorage.Initialize(
                nn::fs::SubStorage(&metaStorage, 0, metaSize),
                nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                );
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapOffset, result);

            entryBuffer = temporary;
        }

        // サイズが大きすぎる
        {
            const auto temporary = entryBuffer;
            entryBuffer[ExpandCount].size *= 1024 * 1024; // 最後のエントリを壊さないと Initialize で失敗する

            result = remapStorage.Initialize(
                         nn::fs::SubStorage(&metaStorage, 0, metaSize),
                         nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                     );
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = remapStorage.RegisterStorage(
                         nn::fs::SubStorage(&dataStorage, 0, DataSize),
                         0
                     );
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, result);
            remapStorage.Finalize();

            entryBuffer = temporary;
        }

        // 論理オフセットが不連続
        {
            const auto temporary = entryBuffer;
            entryBuffer[1].offsetVirtual += nn::fssystem::save::RemapStorage::SizeBlockSmall;
            entryBuffer[1].size -= nn::fssystem::save::RemapStorage::SizeBlockSmall;

            result = remapStorage.Initialize(
                         nn::fs::SubStorage(&metaStorage, 0, metaSize),
                         nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                     );
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = remapStorage.RegisterStorage(
                         nn::fs::SubStorage(&dataStorage, 0, DataSize),
                         0
                     );
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = remapStorage.Read(0, buffer.data(), buffer.size());
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapOffset, result);
            remapStorage.Finalize();

            entryBuffer = temporary;
        }

        // ストレージタイプの不一致
        {
            const auto temporary = entryBuffer;
            entryBuffer[2].storageType = 1;

            result = remapStorage.Initialize(
                         nn::fs::SubStorage(&metaStorage, 0, metaSize),
                         nn::fs::SubStorage(&entryStorage, 0, EntrySize)
                     );
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = remapStorage.RegisterStorage(
                         nn::fs::SubStorage(&dataStorage, 0, DataSize),
                         0
                     );
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = remapStorage.Write(0, buffer.data(), buffer.size());
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidMapStorageType, result);
            remapStorage.Finalize();

            entryBuffer = temporary;
        }
    } // NOLINT(impl/function_size)

    // 事前条件違反で正しく落ちることを確認するテストです
    void PreconditionTest() NN_NOEXCEPT
    {
        nn::Result result;

        const int mapCount = 10;

        for( int count = 0; count < mapCount; ++count )
        {
            int64_t virtualOffset = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                GetRemapStorage().RegisterMap(
                    &virtualOffset,
                    nn::fssystem::save::RemapStorage::SizeBlockSmall,
                    nn::fssystem::save::RemapStorage::AlignmentSmall,
                    0
                )
            );
        }

        // ReadMapEntry
        {
            nn::fssystem::save::RemapStorage::MapEntry entry = {};
            EXPECT_DEATH_IF_SUPPORTED(GetRemapStorage().ReadMapEntry(nullptr, 0), "");
            EXPECT_DEATH_IF_SUPPORTED(GetRemapStorage().ReadMapEntry(&entry, -1), "");
            EXPECT_DEATH_IF_SUPPORTED(GetRemapStorage().ReadMapEntry(&entry, mapCount), "");
            NNT_EXPECT_RESULT_SUCCESS(GetRemapStorage().ReadMapEntry(&entry, 0));
            NNT_EXPECT_RESULT_SUCCESS(GetRemapStorage().ReadMapEntry(&entry, mapCount - 1));
        }

        // WriteMapEntry
        {
            nn::fssystem::save::RemapStorage::MapEntry entry = {};
            EXPECT_DEATH_IF_SUPPORTED(GetRemapStorage().WriteMapEntry(nullptr, 0), "");
            EXPECT_DEATH_IF_SUPPORTED(GetRemapStorage().WriteMapEntry(&entry, -1), "");
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().WriteMapEntry(
                    &entry,
                    static_cast<int>(GetRemapStorage().GetMapEntriesCountMax())
                ),
                ""
            );
            NNT_EXPECT_RESULT_SUCCESS(
                GetRemapStorage().WriteMapEntry(
                    &entry,
                    0
                )
            );
            NNT_EXPECT_RESULT_SUCCESS(
                GetRemapStorage().WriteMapEntry(
                    &entry,
                    static_cast<int>(GetRemapStorage().GetMapEntriesCountMax() - 1)
                )
            );
        }

        // MakeVirtualOffset
        {
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().MakeVirtualOffset(
                    1,
                    GetRemapStorage().GetMapSizeMax()
                ),
                ""
            );
            GetRemapStorage().MakeVirtualOffset(1, GetRemapStorage().GetMapSizeMax() - 1);
        }

        // RegisterMapCore
        {
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().RegisterMapCore(
                    GetRemapStorage().MakeVirtualOffset(mapCount, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    0
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().RegisterMapCore(
                    GetRemapStorage().MakeVirtualOffset(mapCount, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    0,
                    nn::fssystem::save::RemapStorage::AlignmentLarge - 1,
                    0
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().RegisterMapCore(
                    GetRemapStorage().MakeVirtualOffset(mapCount, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    0,
                    nn::fssystem::save::RemapStorage::AlignmentLarge + 1,
                    0
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().RegisterMapCore(
                    GetRemapStorage().MakeVirtualOffset(mapCount, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    0,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    -1
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().RegisterMapCore(
                    GetRemapStorage().MakeVirtualOffset(mapCount, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    0,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    nn::fssystem::save::RemapStorage::CountMaxStorages
                ),
                ""
            );
            NNT_EXPECT_RESULT_SUCCESS(
                GetRemapStorage().RegisterMapCore(
                    GetRemapStorage().MakeVirtualOffset(mapCount, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    0,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    0
                )
            );
        }

        // MakeMapEntry
        {
            auto controlArea = GetRemapStorage().m_ControlArea;
            auto storageEntry = GetRemapStorage().m_StorageEntries[0];
            nn::fssystem::save::RemapStorage::MapEntry mapEntry = {};

            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().MakeMapEntry(
                    nullptr,
                    &storageEntry,
                    &controlArea,
                    GetRemapStorage().MakeVirtualOffset(mapCount + 1, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    0
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().MakeMapEntry(
                    &mapEntry,
                    nullptr,
                    &controlArea,
                    GetRemapStorage().MakeVirtualOffset(mapCount + 1, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    0
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().MakeMapEntry(
                    &mapEntry,
                    &storageEntry,
                    nullptr,
                    GetRemapStorage().MakeVirtualOffset(mapCount + 1, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    0
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().MakeMapEntry(
                    &mapEntry,
                    &storageEntry,
                    &controlArea,
                    GetRemapStorage().MakeVirtualOffset(mapCount + 1, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentLarge - 1,
                    0
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().MakeMapEntry(
                    &mapEntry,
                    &storageEntry,
                    &controlArea,
                    GetRemapStorage().MakeVirtualOffset(mapCount + 1, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentLarge + 1,
                    0
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().MakeMapEntry(
                    &mapEntry,
                    &storageEntry,
                    &controlArea,
                    GetRemapStorage().MakeVirtualOffset(mapCount + 1, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    -1
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().MakeMapEntry(
                    &mapEntry,
                    &storageEntry,
                    &controlArea,
                    GetRemapStorage().MakeVirtualOffset(mapCount + 1, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    nn::fssystem::save::RemapStorage::CountMaxStorages
                ),
                ""
            );
            NNT_EXPECT_RESULT_SUCCESS(
                GetRemapStorage().MakeMapEntry(
                    &mapEntry,
                    &storageEntry,
                    &controlArea,
                    GetRemapStorage().MakeVirtualOffset(mapCount + 1, 0),
                    nn::fssystem::save::RemapStorage::SizeBlockLarge,
                    nn::fssystem::save::RemapStorage::AlignmentLarge,
                    0
                )
            );
        }

        // AddMapEntries
        {
            nn::fssystem::save::RemapStorage::MapEntry mapEntries[2] = {};

            EXPECT_DEATH_IF_SUPPORTED(GetRemapStorage().AddMapEntries(nullptr, 2), "");
            EXPECT_DEATH_IF_SUPPORTED(GetRemapStorage().AddMapEntries(mapEntries, 0), "");

            mapEntries[0].offsetVirtual = GetRemapStorage().MakeVirtualOffset(
                                              mapCount,
                                              nn::fssystem::save::RemapStorage::SizeBlockLarge
                                          );
            mapEntries[1].offsetVirtual = GetRemapStorage().MakeVirtualOffset(mapCount, 0);
            EXPECT_DEATH_IF_SUPPORTED(GetRemapStorage().AddMapEntries(mapEntries, 2), "");

            mapEntries[0].offsetVirtual = GetRemapStorage().MakeVirtualOffset(mapCount, 0);
            mapEntries[0].size = nn::fssystem::save::RemapStorage::SizeBlockLarge;
            mapEntries[1].offsetVirtual = GetRemapStorage().MakeVirtualOffset(
                                              mapCount,
                                              nn::fssystem::save::RemapStorage::SizeBlockLarge
                                          );
            mapEntries[1].size = nn::fssystem::save::RemapStorage::SizeBlockLarge;
            NNT_EXPECT_RESULT_SUCCESS(GetRemapStorage().AddMapEntries(mapEntries, 2));
        }

        // キャッシュ
        {
            nn::fssystem::save::RemapStorage::MapEntry mapEntry;
            int indexMapEntry;

            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().m_MapEntryCache.GetMapEntry(
                    nullptr,
                    &indexMapEntry,
                    0
                ),
                ""
            );
            EXPECT_DEATH_IF_SUPPORTED(
                GetRemapStorage().m_MapEntryCache.GetMapEntry(
                    &mapEntry,
                    nullptr,
                    0
                ),
                ""
            );
            GetRemapStorage().m_MapEntryCache.GetMapEntry(&mapEntry, &indexMapEntry, 0);
        }
    } // NOLINT(impl/function_size)

private:
    // マッピングエントリが正常か検証する
    void CheckMapEntries(nn::fssystem::save::RemapStorage& remapStorage) NN_NOEXCEPT
    {
        nn::Result result;

        // エントリ数は溢れていないはず
        EXPECT_LE(
            remapStorage.m_ControlArea.countMapEntries,
            remapStorage.GetMapEntriesCountMax()
        );

        nnt::fs::util::Vector<nn::fssystem::save::RemapStorage::MapEntry> entries;

        // 論理オフセットの昇順に並んでいるはず
        // アライメント制約を満たすはず
        // ストレージ種別は登録済みであるはず
        const int entryCount = remapStorage.m_ControlArea.countMapEntries;
        for( int entryIndex = 0; entryIndex < entryCount; ++entryIndex )
        {
            nn::fssystem::save::RemapStorage::MapEntry entry;

            result = remapStorage.ReadMapEntry(&entry, entryIndex);
            NNT_ASSERT_RESULT_SUCCESS(result);

            EXPECT_TRUE(
                entry.alignment == nn::fssystem::save::RemapStorage::AlignmentSmall
                || entry.alignment == nn::fssystem::save::RemapStorage::AlignmentLarge
            );
            EXPECT_TRUE(nn::util::is_aligned(
                entry.offsetPhysical,
                static_cast<size_t>(entry.alignment))
            );
            EXPECT_TRUE(entry.size % entry.alignment == 0);
            EXPECT_GE(entry.storageType, 0);
            EXPECT_LT(entry.storageType, DataStorageCount);

            if( 0 < entryIndex )
            {
                EXPECT_LE(entries.back().offsetVirtual + entries.back().size, entry.offsetVirtual);
            }

            entries.push_back(entry);
        }

        // 同じストレージ種別の物理オフセットは多くとも一箇所の隙間を除き連続しているはず
        std::sort(
            entries.begin(),
            entries.end(),
            [](
                const nn::fssystem::save::RemapStorage::MapEntry& lhs,
                const nn::fssystem::save::RemapStorage::MapEntry& rhs
            )
            {
                if( lhs.storageType == rhs.storageType )
                {
                    return lhs.offsetPhysical < rhs.offsetPhysical;
                }
                else
                {
                    return lhs.storageType < rhs.storageType;
                }
            }
        );

        int64_t gap = 0;

        for( size_t entryIndex = 1; entryIndex < entries.size(); ++entryIndex )
        {
            if( entries[entryIndex - 1].storageType == entries[entryIndex].storageType )
            {
                EXPECT_LE(
                    entries[entryIndex - 1].offsetPhysical + entries[entryIndex - 1].size,
                    entries[entryIndex].offsetPhysical
                ) << entries[entryIndex - 1].offsetPhysical << " + "
                    << entries[entryIndex - 1].size << " ["
                    << entries[entryIndex - 1].storageType << "] -> "
                    << entries[entryIndex].offsetPhysical << " + "
                    << entries[entryIndex].size << " ["
                    << entries[entryIndex].storageType << ']';
                if( 0 < gap )
                {
                    EXPECT_EQ(
                        entries[entryIndex - 1].offsetPhysical + entries[entryIndex - 1].size,
                        entries[entryIndex].offsetPhysical
                    );
                }
                else
                {
                    gap = entries[entryIndex].offsetPhysical
                        - (entries[entryIndex - 1].offsetPhysical + entries[entryIndex - 1].size);
                }
            }
            else
            {
                gap = 0;
            }
        }
    }

    static int GetSize(const nn::fssystem::save::RemapStorage::MapEntryCache& cache) NN_NOEXCEPT
    {
        int count = 0;
        for( const auto& node : cache.m_Nodes )
        {
            if( node.indexMapEntry
                != nn::fssystem::save::RemapStorage::MapEntryCache::InvalidMapEntryIndex )
            {
                ++count;
            }
        }
        return count;
    }

    static bool IsEmpty(const nn::fssystem::save::RemapStorage::MapEntryCache& cache) NN_NOEXCEPT
    {
        return GetSize(cache) == 0;
    }
};

typedef RemapStoragePrivateTest RemapStoragePrivateDeathTest;

TEST_F(RemapStoragePrivateTest, Allocate)
{
    AllocateTest();
}

TEST_F(RemapStoragePrivateTest, RegisterCore)
{
    RegisterCoreTest();
}

TEST_F(RemapStoragePrivateTest, Entry)
{
    EntryTest();
}

TEST_F(RemapStoragePrivateTest, Iterate)
{
    IterateTest();
}

TEST_F(RemapStoragePrivateTest, Cache)
{
    CacheTest();
}

TEST_F(RemapStoragePrivateTest, CorruptTable)
{
    CorruptTableTest();
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST_F(RemapStoragePrivateDeathTest, Precondition)
{
    PreconditionTest();
}
#endif

// 4 GB を超えるマッピングのテスト
TEST(RemapStorageLargeTest, ReadWrite)
{
    static const int MapCountMax = 4 * 1024 * 1024;
    static const int DataStorageCount = 2;
    static const int64_t DataStorageSize = nn::fssystem::save::RemapStorage::SizeBlockLarge * MapCountMax;
    static const int64_t EntryStorageSize = 16 * 1024 * 1024;

    static const int64_t MapSize = nn::fssystem::save::RemapStorage::SizeBlockSmall;
    static const int64_t Alignment = nn::fssystem::save::RemapStorage::AlignmentSmall;
    static const int MapCount = 16;

    // メタデータ用ストレージを用意する
    const int64_t metaSize = nn::fssystem::save::RemapStorage::QueryMetaSize();
    nnt::fs::util::SafeMemoryStorage metaStorage(metaSize);

    // エントリ用データを用意する
    nnt::fs::util::VirtualMemoryStorage entryStorage(EntryStorageSize);

    // データ用ストレージを用意する
    nnt::fs::util::VirtualMemoryStorage dataStorages[DataStorageCount];
    for( int dataStorageIndex = 0; dataStorageIndex < DataStorageCount; ++dataStorageIndex )
    {
        dataStorages[dataStorageIndex].Initialize(DataStorageSize);
    }

    // ストレージを用意する
    NNT_ASSERT_RESULT_SUCCESS(nn::fssystem::save::RemapStorage::Format(
        nn::fs::SubStorage(&metaStorage, 0, metaSize),
        MapCountMax));

    nn::fssystem::save::RemapStorage remapStorage;
    NNT_ASSERT_RESULT_SUCCESS(remapStorage.Initialize(
        nn::fs::SubStorage(&metaStorage, 0, metaSize),
        nn::fs::SubStorage(&entryStorage, 0, EntryStorageSize)));

    for( int dataStorageIndex = 0; dataStorageIndex < DataStorageCount; ++dataStorageIndex )
    {
        NNT_ASSERT_RESULT_SUCCESS(remapStorage.RegisterStorage(
            nn::fs::SubStorage(&dataStorages[dataStorageIndex], 0, DataStorageSize),
            dataStorageIndex));
    }

    // マッピングを登録する
    int64_t virtualOffsets[MapCount];
    for( auto& offset : virtualOffsets )
    {
        NNT_ASSERT_RESULT_SUCCESS(remapStorage.RegisterMap(
            &offset,
            MapSize,
            Alignment,
            (&offset - virtualOffsets) % DataStorageCount));
    }

    // バッファ初期化
    std::unique_ptr<char> readBuffer(new char[MapSize]);
    std::unique_ptr<char> writeBufferList[MapCount];
    for( auto& writeBuffer : writeBufferList )
    {
        writeBuffer.reset(new char[MapSize]);
        nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), MapSize);
    }

    for( int i = 0; i < MapCount; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(remapStorage.Write(virtualOffsets[i], writeBufferList[i].get(), MapSize));
    }

    for( int i = 0; i < MapCount; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(remapStorage.Read(virtualOffsets[i], readBuffer.get(), MapSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), MapSize);
    }
}
