﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataPrivate.h>

#include "ExportHelper.h"

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

typedef nn::fs::SaveDataTransferManagerVersion2 SaveDataTransferManagerForCloudBackUp;
typedef nn::fs::InitialDataVersion2 InitialDataForCloudBackUp;

namespace {

struct ImportParam
{
    bool isDuplicate;
    bool isDiff;
};

const ImportParam ImportParamArray[] =
{
    // TODO: 複製インポートのテストの有効化
    // { true, true },
    { false, true },
    { false, false },
};

// Import のバリエーションのための fixture
class DivisionImporter : public ::testing::TestWithParam<ImportParam>
{
};


void CalculateSaveDataHash(Hash *outValue, const SaveDataInfo& info)
{
    const size_t WorkBufferSize = 4 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    crypto::Sha256Generator sha;
    sha.Initialize();

    // save
    {
        nnt::fs::util::Hash hash;
        NN_ABORT_UNLESS_RESULT_SUCCESS(MountSaveData("_hash", info.applicationId, info.saveDataUserId));
        CalculateDirectoryTreeHash(&hash, "_hash:/", workBuffer.get(), WorkBufferSize);
        Unmount("_hash");

        sha.Update(&hash, sizeof(hash));
    }

    // internal storage
    {
        nnt::fs::util::Hash hash;
        NN_ABORT_UNLESS_RESULT_SUCCESS(MountSaveDataInternalStorage("_hash", info.saveDataSpaceId, info.saveDataId));

        CalculateFileHash(&hash, "_hash:/AllocationTableControlArea", workBuffer.get(), WorkBufferSize);
        sha.Update(&hash, sizeof(hash));
        CalculateFileHash(&hash, "_hash:/AllocationTableMeta", workBuffer.get(), WorkBufferSize);
        sha.Update(&hash, sizeof(hash));
        CalculateFileHash(&hash, "_hash:/AllocationTableDataWithZeroFree", workBuffer.get(), WorkBufferSize);
        sha.Update(&hash, sizeof(hash));

        Unmount("_hash");
    }

    // thumbnail
    {
        const size_t headerSize = 32;
        const size_t  bodySize = nn::fs::detail::ThumbnailFileSize - 32;
        char thumbnailHeader[headerSize];
        char thumbnailBody[bodySize];
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::ReadSaveDataThumbnailFile(info.applicationId.value, info.saveDataUserId, thumbnailHeader, headerSize, thumbnailBody, bodySize)
        );

        sha.Update(thumbnailHeader, headerSize);
        sha.Update(thumbnailBody, bodySize);
    }

    sha.GetHash(outValue->value, sizeof(outValue->value));
}

void OpenImporter(
    std::unique_ptr<ISaveDataDivisionImporter>* outImporter,
    SaveDataTransferManagerForCloudBackUp& manager, ExportHelper& exportHelper, nn::fs::UserId uid, ImportParam param)
{
    const SaveDataInfo& info = exportHelper.GetSrcInfo();

    InitialDataForCloudBackUp initialData;
    exportHelper.ReadInitialData(&initialData);

    if (param.isDuplicate && param.isDiff)
    {
        NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataDuplicateDiffImporter(outImporter, initialData, info.saveDataSpaceId, info.saveDataId));
    }
    else if (!param.isDuplicate && !param.isDiff)
    {
        NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataFullImporter(outImporter, initialData, nn::account::Uid{ uid._data[0], uid._data[1] }, info.saveDataSpaceId));
    }
    else if (!param.isDuplicate && param.isDiff)
    {
        NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataDiffImporter(outImporter, initialData, info.saveDataSpaceId, info.saveDataId));
    }
}

void PushChunkData(ISaveDataChunkImporter* chunkImporter, ExportHelper& exportHelper, SaveDataChunkId id)
{
    const size_t BufferSize = 1024 * 1024;
    auto buffer = AllocateBuffer(BufferSize);

    int64_t offset = 0;
    while (true)
    {
        size_t readSize;
        NNT_EXPECT_RESULT_SUCCESS(exportHelper.ReadChunkData(&readSize, id, offset, buffer.get(), BufferSize));

        if (readSize > 0)
        {
            NNT_EXPECT_RESULT_SUCCESS(chunkImporter->Push(buffer.get(), readSize));
        }
        else
        {
            break;
        }

        offset += readSize;
    }
}

// インポートをキャンセルする
void CancelImport(ISaveDataDivisionImporter* importer, ImportParam param)
{
    // キャンセル
    if (!param.isDuplicate && param.isDiff)
    {
        // 上書きインポートの場合はキャンセル未実装
        NNT_ASSERT_RESULT_FAILURE(ResultNotImplemented, importer->CancelImport());
        return;
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(importer->CancelImport());
    }
}

// 各ステートで importer をキャンセル
TEST_P(DivisionImporter, CancelTiming)
{
    auto param = GetParam();

    enum CancelTiming {
        CANCEL_TIMING_NOT_INITIALIZED = 0,
        CANCEL_TIMING_INITIALIZING,
        CANCEL_TIMING_INITIALIZED,
        CANCEL_TIMING_PUSHING,
        CANCEL_TIMING_PUSHED,
        CANCEL_TIMING_FINALIZED,
        CANCEL_TIMING_NONE,
        NUM_OF_CANCEL_TIMING
    };

    // 新規インポートの場合にインポート先として利用するユーザ ID
    UserId altUserId = TestUserId1;

    for (int t = 0; t < NUM_OF_CANCEL_TIMING; t++)
    {
        NN_SDK_LOG("-----------------\n", t);
        NN_SDK_LOG("IsDiff: %d\n", param.isDiff);
        NN_SDK_LOG("IsDuplicate: %d\n", param.isDuplicate);
        NN_SDK_LOG("CancelTiming: %d\n", t);
        NN_SDK_LOG("-----------------\n", t);

        // cleanup
        DeleteAllTestSaveData();

        // どのような形であれ、作業中のセーブデータは最終的に削除されているはず
        NN_UTIL_SCOPE_EXIT
        {
            // Secondary にセーブデータが存在しないことを確認
            {
                Vector<SaveDataInfo> saveDataInfoArray;
                FindSaveData(&saveDataInfoArray, SaveDataSpaceId::User,
                    [&](SaveDataInfo info) {
                        NN_UNUSED(info);
                        return (info.rank == SaveDataRank::Secondary);
                    }
                );
                EXPECT_EQ(0, saveDataInfoArray.size());
            }

            // 新規インポートの場合
            if(t != CANCEL_TIMING_FINALIZED && t != CANCEL_TIMING_NONE)
            {
                Vector<SaveDataInfo> saveDataInfoArray;
                FindSaveData(&saveDataInfoArray, SaveDataSpaceId::User,
                    [&](SaveDataInfo info) {
                        NN_UNUSED(info);
                        return (info.saveDataUserId == altUserId);
                    }
                );
                EXPECT_EQ(0, saveDataInfoArray.size());
            }
        };

        // セーブデータ作成
        SaveDataInfo infoSrc;
        NNT_ASSERT_RESULT_SUCCESS(CreateAndFindSaveData(&infoSrc));
        EXPECT_NE(infoSrc.saveDataUserId, altUserId);

        // ファイル作成
        {
            NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", infoSrc.applicationId, infoSrc.saveDataUserId));
            NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/testFile1", 256 * 1024, 0));
            CommitSaveData("save");
            Unmount("save");
        }

        Hash hashSrc;
        CalculateSaveDataHash(&hashSrc, infoSrc);

        SaveDataTransferManagerForCloudBackUp manager;

        // エクスポート
        ExportHelper exportHelper(manager, infoSrc);
        exportHelper.DoExport(32);

        // セーブデータ更新
        {
            NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", infoSrc.applicationId, infoSrc.saveDataUserId));
            NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/testFile2", 256 * 1024, 5));
            CommitSaveData("save");
            Unmount("save");
        }

        // SetKeySeedPackage
        {
            SaveDataTransferManagerForCloudBackUp::KeySeedPackage ksp;
            nnt::fs::util::FillBufferWith8BitCount(ksp.data, ksp.Size, 0);
            manager.SetKeySeedPackage(ksp);
        }

        // インポータ作成
        std::unique_ptr<ISaveDataDivisionImporter> importer;
        OpenImporter(&importer, manager, exportHelper, altUserId, param);

        // InitializeImport 前にキャンセル
        if (t == CANCEL_TIMING_NOT_INITIALIZED)
        {
            // キャンセル不可
            NNT_EXPECT_RESULT_FAILURE(ResultPreconditionViolation, importer->CancelImport());
            continue;
        }

        // InitializeImport
        int64_t restSize;
        int64_t processSize = 1024 * 1024;
        NNT_ASSERT_RESULT_SUCCESS(importer->InitializeImport(&restSize, processSize));

        if (restSize > 0)
        {
            // InitializeImport 中にキャンセル
            if (t == CANCEL_TIMING_INITIALIZING)
            {
                CancelImport(importer.get(), param);
                continue;
            }

            while (restSize > 0)
            {
                NNT_ASSERT_RESULT_SUCCESS(importer->InitializeImport(&restSize, processSize));
            }
        }
        else
        {
            // 複製インポートの場合は複数回 InitializeImport が呼ばれることを意図していて、ここには来ないはず
            EXPECT_FALSE(param.isDuplicate);

            if (t == CANCEL_TIMING_INITIALIZING)
            {
                importer->CancelImport();
                continue;
            }
        }

        // InitializeImport 直後にキャンセル
        if (t == CANCEL_TIMING_INITIALIZED)
        {
            CancelImport(importer.get(), param);
            continue;
        }

        // Push
        std::unique_ptr<ISaveDataChunkIterator> chunkIter;
        NNT_ASSERT_RESULT_SUCCESS(importer->OpenSaveDataDiffChunkIterator(&chunkIter));

        int importedChunkCount = 0;
        bool isCanceled = false;
        for (; !chunkIter->IsEnd(); chunkIter->Next())
        {
            std::unique_ptr<ISaveDataChunkImporter> chunkImporter;
            importer->OpenSaveDataChunkImporter(&chunkImporter, chunkIter->GetId());
            PushChunkData(chunkImporter.get(), exportHelper, chunkIter->GetId());

            importedChunkCount++;

            // Push 中にキャンセル
            if (t == CANCEL_TIMING_PUSHING)
            {
                CancelImport(importer.get(), param);
                isCanceled = true;
                break;
            }
        }
        if (t == CANCEL_TIMING_PUSHING)
        {
            if (isCanceled)
            {
                continue;
            }
            else
            {
                // キャンセルされているはずなので、ここに来たら何か変
                ADD_FAILURE();
            }
        }

        // Push 完了後にキャンセル
        if (t == CANCEL_TIMING_PUSHED)
        {
            CancelImport(importer.get(), param);
            continue;
        }

        // Finalize
        NNT_ASSERT_RESULT_SUCCESS(importer->FinalizeImport());

        // FinalizeImport 後にキャンセル
        if (t == CANCEL_TIMING_FINALIZED)
        {
            // キャンセル不可
            NNT_EXPECT_RESULT_FAILURE(ResultPreconditionViolation, importer->CancelImport());
            continue;
        }

        // ここに来たらインポートが完了しているはずなので結果を確認
        {
            nn::util::optional<SaveDataInfo> infoTmp;
            if (param.isDiff)
            {
                FindSaveData(&infoTmp, infoSrc.saveDataSpaceId, [&](SaveDataInfo i) {
                    return (i.saveDataUserId == infoSrc.saveDataUserId) && (i.applicationId == infoSrc.applicationId);
                });
            }
            else
            {
                FindSaveData(&infoTmp, infoSrc.saveDataSpaceId, [&](SaveDataInfo i) {
                    return (i.saveDataUserId == altUserId) && (i.applicationId == infoSrc.applicationId);
                });
            }
            EXPECT_TRUE(infoTmp != util::nullopt);
            Hash hashTmp;
            CalculateSaveDataHash(&hashTmp, infoTmp.value());
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashSrc, &hashTmp, sizeof(hashTmp));
        }
    }

    DeleteAllTestSaveData();
} // NOLINT(impl/function_size)

INSTANTIATE_TEST_CASE_P(
    ImportMode,
    DivisionImporter,
    ::testing::ValuesIn(ImportParamArray)
);

}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nn::fs::SetEnabledAutoAbort(false);

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}

