﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/fs_AesCtrStorage.h>
#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_SparseStorage.h>
#include <nn/fssystem/fs_BucketTreeBuilder.h>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

typedef nn::fssystem::BucketTree BucketTree;
typedef nn::fssystem::SparseStorage SparseStorage;

class SparseStorageCreator
{
public:
    SparseStorageCreator() NN_NOEXCEPT
        : m_Buffer()
        , m_TableHeader()
        , m_TableStorage(&m_TableHeader, sizeof(m_TableHeader))
        , m_pBaseStorage()
        , m_NodeStorage()
        , m_EntryStorage()
    {
        m_TableHeader.Format(0);
        m_SparseStorage.Initialize(1024);
    }

    template< int Length >
    SparseStorageCreator(const SparseStorage::EntryData(&entries)[Length], int64_t endOffset) NN_NOEXCEPT
        : m_Buffer()
        , m_TableHeader()
        , m_TableStorage(&m_TableHeader, sizeof(m_TableHeader))
        , m_pBaseStorage()
        , m_NodeStorage()
        , m_EntryStorage()
    {
        Build(entries, Length, endOffset);
    }

    SparseStorage* operator->() NN_NOEXCEPT
    {
        return &m_SparseStorage;
    }

    void Build(const SparseStorage::EntryData* entryArray, int entryCount, int64_t endOffset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(entryArray);
        NN_SDK_REQUIRES_GREATER(entryCount, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(endOffset, 0);

        const auto nodeSize = SparseStorage::QueryNodeStorageSize(entryCount);
        const auto entrySize = SparseStorage::QueryEntryStorageSize(entryCount);

        m_Buffer.reset(new char[size_t(nodeSize + entrySize)]);
        std::memset(m_Buffer.get(), 0, size_t(nodeSize + entrySize));
        NN_ABORT_UNLESS_NOT_NULL(m_Buffer);

        m_pBaseStorage.reset(new nn::fs::MemoryStorage(m_Buffer.get(), nodeSize + entrySize));
        NN_ABORT_UNLESS_NOT_NULL(m_pBaseStorage);

        m_NodeStorage = nn::fs::SubStorage(m_pBaseStorage.get(), 0, nodeSize);
        m_EntryStorage = nn::fs::SubStorage(m_pBaseStorage.get(), nodeSize, entrySize);

        nn::fssystem::BucketTreeBuilder builder;
        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&m_TableStorage, 0, sizeof(m_TableHeader)),
            m_NodeStorage,
            m_EntryStorage,
            SparseStorage::NodeSize,
            sizeof(SparseStorage::Entry),
            entryCount
        ));

        for( int i = 0; i < entryCount; ++i )
        {
            SparseStorage::Entry entry;
            entry.SetVirtualOffset(entryArray[i].virtualOffset);
            entry.SetPhysicalOffset(entryArray[i].physicalOffset);
            entry.storageIndex = entryArray[i].storageIndex;

            NNT_ASSERT_RESULT_SUCCESS(builder.Write(entry));
        }
        NNT_ASSERT_RESULT_SUCCESS(builder.Commit(endOffset));

        NN_ABORT_UNLESS_EQUAL(m_TableHeader.entryCount, entryCount);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_SparseStorage.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(), m_NodeStorage, m_EntryStorage, m_TableHeader.entryCount));
    }

    void SetDataStorage(nn::fs::SubStorage dataStorage) NN_NOEXCEPT
    {
        m_SparseStorage.SetDataStorage(dataStorage);
    }

    int GetEntryCount() const NN_NOEXCEPT
    {
        return m_TableHeader.entryCount;
    }

    nn::fs::SubStorage& GetNodeStorage() NN_NOEXCEPT
    {
        return m_NodeStorage;
    }

    nn::fs::SubStorage& GetEntryStorage() NN_NOEXCEPT
    {
        return m_EntryStorage;
    }

    SparseStorage& Get() NN_NOEXCEPT
    {
        return m_SparseStorage;
    }

public:
    std::unique_ptr<char[]> m_Buffer;
    BucketTree::Header m_TableHeader;
    nn::fs::MemoryStorage m_TableStorage;
    std::unique_ptr<nn::fs::MemoryStorage> m_pBaseStorage;
    nn::fs::SubStorage m_NodeStorage;
    nn::fs::SubStorage m_EntryStorage;
    SparseStorage m_SparseStorage;
};

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 * @brief   事前検証のテストをします。
 */
TEST(SparseStorageDeathTest, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(SparseStorage::QueryNodeStorageSize(-1), "");
    EXPECT_DEATH_IF_SUPPORTED(SparseStorage::QueryEntryStorageSize(-1), "");

    char buffer[1];
    nn::fs::MemoryStorage rawStorage(buffer, sizeof(buffer));
    nn::fs::SubStorage storage(&rawStorage, 0, sizeof(buffer));
    auto pAllocator = nnt::fs::util::GetTestLibraryAllocator();

    // Initialize()
    EXPECT_DEATH_IF_SUPPORTED(SparseStorage().Initialize(nullptr, storage, storage, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(SparseStorage().Initialize(-1), "");
    {
        SparseStorage test;
        test.Initialize(1024);
        EXPECT_DEATH_IF_SUPPORTED(test.Initialize(pAllocator, storage, storage, 1), "");
    }

    // Read()
    EXPECT_DEATH_IF_SUPPORTED(SparseStorage().Read(0, buffer, sizeof(buffer)), "");
    {
        SparseStorage test;
        test.Initialize(1024);
        EXPECT_DEATH_IF_SUPPORTED(test.Read(-1, buffer, sizeof(buffer)), "");
    }

    // OperateRange()
    EXPECT_DEATH_IF_SUPPORTED(SparseStorage().OperateRange(
        nullptr, 0, nn::fs::OperationId::Invalidate, 0, 0, nullptr, 0), "");
    {
        SparseStorage test;
        test.Initialize(1024);
        EXPECT_DEATH_IF_SUPPORTED(test.OperateRange(
            nullptr, 0, nn::fs::OperationId::Invalidate, -1, 0, nullptr, 0), "");
    }
    {
        SparseStorage test;
        test.Initialize(1024);
        EXPECT_DEATH_IF_SUPPORTED(test.OperateRange(
            nullptr, 0, nn::fs::OperationId::Invalidate, 0, -1, nullptr, 0), "");
    }
}
#endif

/**
 * @brief   想定したリザルトを返すかをテストします。
 */
TEST(SparseStorageTest, ReturnResult)
{
    const char Zero[256] = {};
    char buffer[256];
    {
        SparseStorage test;
        test.Initialize(1024);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultNullptrArgument, test.Read(0, nullptr, sizeof(buffer)));
    }
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, SparseStorage().Write(0, nullptr, 0));
    {
        auto pAllocator = nnt::fs::util::GetTestLibraryAllocator();
        nn::fs::MemoryStorage baseStorage(buffer, sizeof(buffer));
        nn::fs::SubStorage storage(&baseStorage, 0, sizeof(buffer));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument, SparseStorage().Initialize(pAllocator, storage, storage, -1));
    }

    // エントリ数 == 0 の時
    {
        SparseStorageCreator storage;

        NNT_EXPECT_RESULT_SUCCESS(storage->Read(0, nullptr, 0));
        NNT_EXPECT_RESULT_SUCCESS(storage->Read(0, buffer, sizeof(buffer)));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer, Zero, sizeof(Zero));

        // サポート対象外のチェック
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultUnsupportedOperation,
            storage->OperateRange(nullptr, 0, nn::fs::OperationId::DestroySignature, 0, 4, nullptr, 0)
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultUnsupportedOperation,
            storage->OperateRange(nullptr, 0, nn::fs::OperationId::FillZero, 0, 4, nullptr, 0)
        );

        // QueryRange の引数チェック
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultNullptrArgument,
            storage->OperateRange(nullptr, 0, nn::fs::OperationId::QueryRange, 0, 4, nullptr, 0)
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            storage->OperateRange(buffer, 0, nn::fs::OperationId::QueryRange, 0, 4, nullptr, 0)
        );
        NNT_EXPECT_RESULT_SUCCESS(
            storage->OperateRange(buffer, sizeof(nn::fs::QueryRangeInfo), nn::fs::OperationId::QueryRange, 0, 4, nullptr, 0)
        );

        NNT_EXPECT_RESULT_SUCCESS(
            storage->OperateRange(nullptr, 0, nn::fs::OperationId::Invalidate, 0, 0, nullptr, 0));
        NNT_EXPECT_RESULT_SUCCESS(
            storage->OperateRange(nullptr, 0, nn::fs::OperationId::Invalidate, 0, 1024, nullptr, 0));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            storage->OperateRange(nullptr, 0, nn::fs::OperationId::Invalidate, 0, 1025, nullptr, 0)
        );
    }

    // エントリ数 > 0 の時
    {
        const SparseStorage::EntryData EntryData[] =
        {
            { 1024 * 0,        0, 1 },
            { 1024 * 1, 1024 * 0, 0 },
            { 1024 * 2,        0, 1 },
            { 1024 * 3, 1024 * 1, 0 },
            { 1024 * 4,        0, 1 },
            { 1024 * 5, 1024 * 2, 0 },
            { 1024 * 6,        0, 1 },
        };
        const int VirtualDataSize = 1024 * 7;
        const int PhysicalDataSize = 1024 * 3;

        std::unique_ptr<char[]> dataBuffer(new char[PhysicalDataSize]); // dataBuffer の値は何でもいいので設定しない
        nn::fs::MemoryStorage dataStorage(dataBuffer.get(), PhysicalDataSize);

        SparseStorageCreator storage(EntryData, VirtualDataSize);
        storage.SetDataStorage(nn::fs::SubStorage(&dataStorage, 0, PhysicalDataSize));

        NNT_EXPECT_RESULT_SUCCESS(storage->Read(0, nullptr, 0));
        NNT_EXPECT_RESULT_SUCCESS(storage->Read(1024 * 2, buffer, sizeof(buffer))); // 切り詰めた領域へのアクセス
        NNT_EXPECT_RESULT_SUCCESS(storage->Read(1024 * 4 - 128, buffer, sizeof(buffer))); // 切り詰めた領域をまたぐアクセス
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, storage->Read(VirtualDataSize, buffer, sizeof(buffer))); // 上限より外のアクセス

        NNT_EXPECT_RESULT_SUCCESS(storage->OperateRange(nullptr, 0, nn::fs::OperationId::Invalidate, 0, 0, nullptr, 0));
        NNT_EXPECT_RESULT_FAILURE( // 上限より外のアクセス
            nn::fs::ResultOutOfRange,
            storage->OperateRange(nullptr, 0, nn::fs::OperationId::Invalidate, VirtualDataSize, 4, nullptr, 0)
        );
        // 切り詰めた領域へのアクセス
        NNT_EXPECT_RESULT_SUCCESS(storage->OperateRange(nullptr, 0, nn::fs::OperationId::Invalidate, 1024 * 4, 4, nullptr, 0));
        // 切り詰めた領域をまたぐアクセス
        NNT_EXPECT_RESULT_SUCCESS(storage->OperateRange(nullptr, 0, nn::fs::OperationId::Invalidate, 1024 * 6 - 128, 256, nullptr, 0));
    }
}

/**
 * @brief   読み込みをテストします。
 */
TEST(SparseStorageTest, Read)
{
    const SparseStorage::EntryData EntryData[] =
    {
        { 1024 * 0,        0, 1 },
        { 1024 * 1, 1024 * 0, 0 },
        { 1024 * 2,        0, 1 },
        { 1024 * 3, 1024 * 1, 0 },
        { 1024 * 4,        0, 1 },
        { 1024 * 5, 1024 * 2, 0 },
        { 1024 * 6,        0, 1 },
    };
    const int VirtualDataSize = 1024 * 7;
    const int PhysicalDataSize = 1024 * 3;

    std::unique_ptr<char[]> dataBuffer(new char[PhysicalDataSize]);
    nnt::fs::util::FillBufferWith8BitCount(dataBuffer.get(), PhysicalDataSize, 0);

    nn::fs::MemoryStorage dataStorage(dataBuffer.get(), PhysicalDataSize);

    SparseStorageCreator storage(EntryData, VirtualDataSize);
    storage.SetDataStorage(nn::fs::SubStorage(&dataStorage, 0, PhysicalDataSize));

    char buffer1[256];
    char buffer2[256];

    NNT_EXPECT_RESULT_SUCCESS(storage->Read(EntryData[1].virtualOffset, buffer1, sizeof(buffer1)));
    nnt::fs::util::FillBufferWith8BitCount(buffer2, sizeof(buffer2), 0);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer1, buffer2, sizeof(buffer1));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer1, dataBuffer.get() + EntryData[1].physicalOffset, sizeof(buffer1));

    NNT_EXPECT_RESULT_SUCCESS(storage->Read(EntryData[2].virtualOffset, buffer1, sizeof(buffer1)));
    std::memset(buffer2, 0, sizeof(buffer2));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer1, buffer2, sizeof(buffer1));

    NNT_EXPECT_RESULT_SUCCESS(storage->Read(EntryData[3].virtualOffset + 16, buffer1, sizeof(buffer1)));
    nnt::fs::util::FillBufferWith8BitCount(buffer2, sizeof(buffer2), 16);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer1, buffer2, sizeof(buffer1));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer1, dataBuffer.get() + EntryData[3].physicalOffset + 16, sizeof(buffer1));

    // 切り詰めた領域をまたぐアクセス
    const char Zero[128] = {};
    NNT_EXPECT_RESULT_SUCCESS(storage->Read(EntryData[3].virtualOffset - sizeof(buffer1) / 2, buffer1, sizeof(buffer1)));
    nnt::fs::util::FillBufferWith8BitCount(buffer2, sizeof(buffer2), 0);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer1, Zero, sizeof(Zero));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer1 + sizeof(buffer1) / 2, buffer2, sizeof(buffer1) / 2);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer1 + sizeof(buffer1) / 2, dataBuffer.get() + EntryData[3].physicalOffset, sizeof(buffer1) / 2);
}

namespace
{

const SparseStorage::EntryData CryptoEntryData[] =
{
    { 1024 * 0,        0, 1 },
    { 1024 * 1, 1024 * 0, 0 },
    { 1024 * 2,        0, 1 },
    { 1024 * 3, 1024 * 1, 0 },
    { 1024 * 4,        0, 1 },
    { 1024 * 5, 1024 * 2, 0 },
    { 1024 * 6,        0, 1 },
};
const int CryptoVirtualDataSize = 1024 * 7;
const int CryptoPhysicalDataSize = 1024 * 3;

template< typename TStorage, typename TCreator >
void CryptoStorageTest(TCreator create) NN_NOEXCEPT
{
    std::unique_ptr<char[]> virtualDataBuffer(new char[CryptoVirtualDataSize]);
    nnt::fs::util::FillBufferWithRandomValue(virtualDataBuffer.get(), CryptoVirtualDataSize);

    const char Iv[TStorage::IvSize] = {};
    char key[TStorage::KeySize];
    nnt::fs::util::FillBufferWithRandomValue(key, sizeof(key));

    // raw データを暗号化
    std::unique_ptr<char[]> encryptedDataBuffer(new char[CryptoVirtualDataSize]);
    {
        nn::fs::MemoryStorage baseStorage(encryptedDataBuffer.get(), CryptoVirtualDataSize);
        auto pEncryptStorage = create(&baseStorage, key, Iv);
        NNT_ASSERT_RESULT_SUCCESS(pEncryptStorage->Write(0, virtualDataBuffer.get(), CryptoVirtualDataSize));
    }

    // エントリに沿ってデータをコピー
    std::unique_ptr<char[]> physicalDataBuffer(new char[CryptoPhysicalDataSize]);
    for( int i = 0; i < 7; ++i )
    {
        if( CryptoEntryData[i].storageIndex == 0 )
        {
            std::memcpy(
                physicalDataBuffer.get() + CryptoEntryData[i].physicalOffset,
                encryptedDataBuffer.get() + CryptoEntryData[i].virtualOffset,
                1024
            );
        }
    }

    nn::fs::MemoryStorage dataStorage(physicalDataBuffer.get(), CryptoPhysicalDataSize);

    SparseStorageCreator storage(CryptoEntryData, CryptoVirtualDataSize);
    storage.SetDataStorage(nn::fs::SubStorage(&dataStorage, 0, CryptoPhysicalDataSize));

    auto pDecryptStorage = create(&storage.Get(), key, Iv);
    char buffer[256];

    NNT_EXPECT_RESULT_SUCCESS(pDecryptStorage->Read(CryptoEntryData[1].virtualOffset, buffer, sizeof(buffer)));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer, virtualDataBuffer.get() + CryptoEntryData[1].virtualOffset, sizeof(buffer));

    NNT_EXPECT_RESULT_SUCCESS(pDecryptStorage->Read(CryptoEntryData[3].virtualOffset + 16, buffer, sizeof(buffer)));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer, virtualDataBuffer.get() + CryptoEntryData[3].virtualOffset + 16, sizeof(buffer));
}

}

/**
 * @brief   AesCtrStorage に挟んで読み込みをテストします。
 */
TEST(SparseStorageTest, AesCtrSparseStorage)
{
    typedef nn::fssystem::AesCtrStorage CryptoStorage;

    CryptoStorageTest<CryptoStorage>(
        [](nn::fs::IStorage* pStorage, const char* pKey, const char* pIv) NN_NOEXCEPT -> std::unique_ptr<CryptoStorage>
        {
            // 事前検証とバッファサイズ渡しは行いません
            return std::unique_ptr<CryptoStorage>(
                new CryptoStorage(pStorage, pKey, CryptoStorage::KeySize, pIv, CryptoStorage::IvSize));
        }
    );
}

/**
 * @brief   AesXtsStorage に挟んで読み込みをテストします。
 */
TEST(SparseStorageTest, AesXtsSparseStorage)
{
    typedef nn::fssystem::AesXtsStorage CryptoStorage;

    CryptoStorageTest<CryptoStorage>(
        [](nn::fs::IStorage* pStorage, const char* pKey, const char* pIv) NN_NOEXCEPT -> std::unique_ptr<CryptoStorage>
        {
            // 事前検証とバッファサイズ渡しは行いません
            char key2[CryptoStorage::KeySize];
            nnt::fs::util::FillBufferWith32BitCount(key2, sizeof(key2), pKey[0]);

            return std::unique_ptr<CryptoStorage>(
                new CryptoStorage(pStorage, pKey, key2, CryptoStorage::KeySize, pIv, CryptoStorage::IvSize, 512));
        }
    );
}
