﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <fstream>
#include <random>
#include <map>
#include <numeric>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/detail/fs_Log.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/utilTool/fs_BinaryMatch.h>
#include <nn/fssystem/utilTool/fs_BinaryRegionFile.h>
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageUtils.h>
#include <nn/util/util_TinyMt.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>


#define __file__ (strrchr(__FILE__, '\\') + 1)
#define NN_DETAIL_FS_RESULT_LOG_DO(r) \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_DETAIL_FS_ERROR("Failure (Module:%d, Description:%d) in %s() - %s:%d\n", _nn_result_do_temporary.GetModule(), _nn_result_do_temporary.GetDescription(), __FUNCTION__, __file__, __LINE__); \
            NN_RESULT_THROW(_nn_result_do_temporary); \
        } \
    }


namespace {

static const int MinimumRegionSize = 16 * 128;

// テスト用ファイルストレージ
class TestFileStorage : public nn::fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(TestFileStorage);

public:
    explicit TestFileStorage(size_t blockSize) NN_NOEXCEPT
        : m_FileHandle()
        , m_pStorage()
        , m_StorageSize(0)
        , m_BlockSize(blockSize)
    {
    }

    virtual ~TestFileStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        if( m_pStorage != nullptr )
        {
            m_pStorage.reset();
            nn::fs::CloseFile(m_FileHandle);
        }
    }

    nn::Result Initialize(const char* path) NN_NOEXCEPT
    {
        NN_DETAIL_FS_RESULT_LOG_DO(nn::fs::OpenFile(&m_FileHandle, path, nn::fs::OpenMode_Read));
        bool isFailure = true;
        NN_UTIL_SCOPE_EXIT
        {
            if (isFailure)
            {
                NN_DETAIL_FS_ERROR("Forced CloseFile (line:%d - scope exit)\n", __LINE__);
                nn::fs::CloseFile(m_FileHandle);
            }
        };

        m_pStorage.reset(new nn::fs::FileHandleStorage(m_FileHandle));
        NN_ABORT_UNLESS(m_pStorage != nullptr);

        NN_DETAIL_FS_RESULT_LOG_DO(m_pStorage->GetSize(&m_StorageSize));

        isFailure = false;
        NN_RESULT_SUCCESS;
    }

    nn::Result Initialize(const char* path, int mode) NN_NOEXCEPT
    {
        NN_DETAIL_FS_RESULT_LOG_DO(nn::fs::OpenFile(&m_FileHandle, path, mode));
        bool isFailure = true;
        NN_UTIL_SCOPE_EXIT
        {
            if (isFailure)
            {
                NN_DETAIL_FS_ERROR("Forced CloseFile (line:%d - scope exit)\n", __LINE__);
                nn::fs::CloseFile(m_FileHandle);
            }
        };

        m_pStorage.reset(new nn::fs::FileHandleStorage(m_FileHandle));
        NN_ABORT_UNLESS(m_pStorage != nullptr);

        NN_DETAIL_FS_RESULT_LOG_DO(m_pStorage->GetSize(&m_StorageSize));

        isFailure = false;
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result Read(
        int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        // 末尾の余った部分を 0 埋めする
        if( m_StorageSize < offset + static_cast<int64_t>(size) )
        {
            const auto readSize = static_cast<size_t>(m_StorageSize - offset);
            NN_RESULT_THROW_UNLESS(readSize < size, nn::fs::ResultInvalidSize());
            NN_RESULT_THROW_UNLESS(size - readSize < m_BlockSize, nn::fs::ResultInvalidSize());

            const auto result = m_pStorage->Read(offset, buffer, readSize);
            if( result.IsSuccess() )
            {
                auto const ptr = reinterpret_cast<char*>(buffer) + readSize;
                std::memset(ptr, 0, size - readSize);
            }
            return result;
        }
        else
        {
            return m_pStorage->Read(offset, buffer, size);
        }
    }

    virtual nn::Result Write(
        int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pStorage->Write(offset, buffer, size);
    }

    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pStorage->Flush();
    }

    virtual nn::Result GetSize(int64_t* pOutSize) NN_NOEXCEPT NN_OVERRIDE
    {
        const auto result = m_pStorage->GetSize(pOutSize);
        if( result.IsSuccess() )
        {
            *pOutSize = nn::util::align_up(*pOutSize, m_BlockSize);
        }
        return result;
    }

private:
    nn::fs::FileHandle m_FileHandle;
    std::unique_ptr<nn::fs::FileHandleStorage> m_pStorage;
    int64_t m_StorageSize;
    const size_t m_BlockSize;
};

// 読み込み計測機能付きファイルストレージ
class MonitorFileStorage : public TestFileStorage
{
    NN_DISALLOW_COPY(MonitorFileStorage);

public:
    explicit MonitorFileStorage(size_t blockSize) NN_NOEXCEPT
        : TestFileStorage(blockSize)
        , m_ElapsedTime(0)
    {
    }

    virtual ~MonitorFileStorage() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        const auto start = std::chrono::system_clock::now();
        const auto result = TestFileStorage::Read(offset, buffer, size);

        m_ElapsedTime +=
            std::chrono::duration_cast<std::chrono::microseconds>(
                std::chrono::system_clock::now() - start).count();

        return result;
    }

    double GetElapsedTime() const NN_NOEXCEPT
    {
        return m_ElapsedTime / (1000.0 * 1000.0);
    }

private:
    double m_ElapsedTime;
};

}

namespace nn { namespace fssystem { namespace utilTool {

/**
 * @brief   IndirectStorageBuilder のテストクラスです。
 */
class IndirectStorageBuilderTest : public ::testing::Test, public nnt::fs::util::PrepareWorkDirFixture
{
public:
    class BuilderWrapper;

    typedef IndirectStorageBuilder::ExcludeRange ExcludeRange;
    typedef IndirectStorageBuilder::MatchEntry MatchEntry;
    typedef IndirectStorageBuilder::MatchResult ExcludeRangePair;

public:
    static const size_t BlockSize = 16;

public:
    static Result RelocateIndirectTableEntries(
                      utilTool::RelocatedIndirectStorageBuilder* pBuilder,
                      const utilTool::RelocationTable& relocationTable
                  ) NN_NOEXCEPT
    {
        return pBuilder->BuildImpl(nnt::fs::util::GetTestLibraryAllocator(), relocationTable);
    }

public:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        CreateWorkRootPath();
        m_TemporaryPath = GetWorkRootPath();
        m_TemporaryPath.append("\\");
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::UnmountHostRoot();
        DeleteWorkRootPath();
    }

    const nnt::fs::util::String& GetTemporaryPath() const NN_NOEXCEPT
    {
        return m_TemporaryPath;
    }

    void ExcludeStorageAccessImpl(
             int loopCount,
             std::function<bool(ExcludeRangePair*, BuilderWrapper*)> getter
         ) NN_NOEXCEPT;

private:
    nnt::fs::util::String m_TemporaryPath;
};

class IndirectStorageBuilderTest::BuilderWrapper
{
public:
    struct Match
    {
        int64_t newOffset;
        int64_t oldOffset;
        size_t size;
    };

public:
    static const char* const OldFileName;
    static const char* const NewFileName;

public:
    // コンストラクタです。
    BuilderWrapper(
        int64_t binarySize,
        size_t fileSizeMin,
        size_t fileSizeMax,
        size_t blockSize,
        size_t regionSize,
        int64_t matchSize,
        int64_t windowSize
    ) NN_NOEXCEPT
        : m_BinarySize(binarySize)
        , m_FileSizeMin(fileSizeMin)
        , m_FileSizeMax(fileSizeMax)
        , m_BlockSize(blockSize)
        , m_RegionSize(regionSize)
        , m_MatchSize(matchSize)
        , m_WindowSize(windowSize)
        , m_FileCountVariation(10)
        , m_NewFilePreference(30)
        , m_OldFilePath()
        , m_NewFilePath()
        , m_pOldStorage(nullptr)
        , m_pNewStorage(nullptr)
        , m_OldFileStorage(blockSize)
        , m_NewFileStorage(blockSize)
        , m_Matches()
        , m_PatchSize(0)
        , m_Impl()
    {
    }

    // 初期化をします。
    Result Initialize(const nnt::fs::util::String& root) NN_NOEXCEPT
    {
        m_OldFilePath = root;
        m_OldFilePath.append(OldFileName);

        m_NewFilePath = root;
        m_NewFilePath.append(NewFileName);

        NN_DETAIL_FS_RESULT_LOG_DO(
            CreateFiles(
                m_OldFilePath.c_str(),
                m_NewFilePath.c_str(),
                [](char* pData, size_t dataSize, int seed) NN_NOEXCEPT
                {
                    std::mt19937 mt(seed);
                    for( size_t index = 0; index < dataSize; ++index )
                    {
                        pData[index] = static_cast<char>(mt());
                    }
                }
            )
        );

        NN_DETAIL_FS_RESULT_LOG_DO(m_OldFileStorage.Initialize(m_OldFilePath.c_str()));
        NN_DETAIL_FS_RESULT_LOG_DO(m_NewFileStorage.Initialize(m_NewFilePath.c_str()));

        NN_DETAIL_FS_RESULT_LOG_DO(m_NewFileStorage.GetSize(&m_BinarySize));

        return Initialize(&m_OldFileStorage, &m_NewFileStorage);
    }

    // 初期化をします。
    Result Initialize(const char* oldFilePath, const char* newFilePath) NN_NOEXCEPT
    {
        NN_DETAIL_FS_RESULT_LOG_DO(m_OldFileStorage.Initialize(oldFilePath));
        NN_DETAIL_FS_RESULT_LOG_DO(m_NewFileStorage.Initialize(newFilePath));

        NN_DETAIL_FS_RESULT_LOG_DO(m_NewFileStorage.GetSize(&m_BinarySize));

        return Initialize(&m_OldFileStorage, &m_NewFileStorage);
    }

    // 初期化をします。
    Result Initialize(fs::IStorage* pOldStorage, fs::IStorage* pNewStorage) NN_NOEXCEPT
    {
        m_pOldStorage = pOldStorage;
        m_pNewStorage = pNewStorage;

        return m_Impl.Initialize(pOldStorage, pNewStorage, m_BlockSize);
    }

    // テーブルを作成します。
    Result Build() NN_NOEXCEPT
    {
        const auto start = std::chrono::system_clock::now();

        NN_DETAIL_FS_RESULT_LOG_DO(m_Impl.Build(m_BlockSize, m_BlockSize, m_RegionSize, m_MatchSize, m_WindowSize));

        const auto seconds =
            std::chrono::duration_cast<std::chrono::microseconds>(
                std::chrono::system_clock::now() - start).count() / (1000.0 * 1000.0);

        const auto ioSeconds =
            m_OldFileStorage.GetElapsedTime() + m_NewFileStorage.GetElapsedTime();

        NN_LOG("----------------------------------------\n");
        NN_LOG("0x%llX bytes (%lldKB) binary\n", m_BinarySize, m_BinarySize / 1024);
        if( 0 < m_FileSizeMin )
        {
            if( m_FileSizeMin == m_FileSizeMax )
            {
                NN_LOG("0x%X bytes file\n", m_FileSizeMin);
            }
            else
            {
                NN_LOG("0x%X-0x%X bytes file\n", m_FileSizeMin, m_FileSizeMax);
            }
        }
        NN_LOG("%d bytes block\n", m_BlockSize);
        NN_LOG("elapsed %lf seconds (io %lf seconds)\n", seconds, ioSeconds);
        NN_LOG("%d entries (%d files)\n", m_Impl.m_MatchCount, m_Matches.size());
        NN_LOG("output data %llX bytes (%.4lf%%)\n",
               m_Impl.QueryDataStorageSize(),
               m_Impl.QueryDataStorageSize() * 100.0 / m_BinarySize);

        NN_RESULT_SUCCESS;
    }

    // 以前に書き出したテーブルを注入します。
    Result Build(IndirectStorageBuilder& builder, fs::IStorage* pStorage) NN_NOEXCEPT
    {
        const int64_t headerOffset = 0;
        const int64_t headerSize = builder.QueryTableHeaderStorageSize();
        const int64_t nodeOffset = headerOffset + headerSize;
        const int64_t nodeSize = builder.QueryTableNodeStorageSize();
        const int64_t entryOffset = nodeOffset + nodeSize;
        const int64_t entrySize = builder.QueryTableEntryStorageSize();

        return m_Impl.Build(
                   nnt::fs::util::GetTestLibraryAllocator(),
                   fs::SubStorage(pStorage, headerOffset, headerSize),
                   fs::SubStorage(pStorage, nodeOffset, nodeSize),
                   fs::SubStorage(pStorage, entryOffset, entrySize)
               );
    }

    // テーブルを作成します。
    Result BuildOnly() NN_NOEXCEPT
    {
        return m_Impl.Build(m_BlockSize, m_BlockSize, m_RegionSize, m_MatchSize, m_WindowSize);
    }

    // テーブルを書き出します。
    Result WriteTable(fs::IStorage* pStorage) NN_NOEXCEPT
    {
        const int64_t headerOffset = 0;
        const int64_t headerSize = m_Impl.QueryTableHeaderStorageSize();
        const int64_t nodeOffset = headerOffset + headerSize;
        const int64_t nodeSize = m_Impl.QueryTableNodeStorageSize();
        const int64_t entryOffset = nodeOffset + nodeSize;
        const int64_t entrySize = m_Impl.QueryTableEntryStorageSize();

        return m_Impl.WriteTable(
                   nnt::fs::util::GetTestLibraryAllocator(),
                   fs::SubStorage(pStorage, headerOffset, headerSize),
                   fs::SubStorage(pStorage, nodeOffset, nodeSize),
                   fs::SubStorage(pStorage, entryOffset, entrySize)
               );
    }

    // 必要なデータを書き出します。
    Result WriteData(fs::IStorage* pStorage) NN_NOEXCEPT
    {
        static const size_t BufferSize = 4 * 1024;

        std::unique_ptr<char[]> buffer(new char[BufferSize]);
        NN_ABORT_UNLESS_NOT_NULL(buffer);

        return m_Impl.WriteData(pStorage, buffer.get(), BufferSize);
    }

    // 必要なデータを書き出します。
    Result WriteData(fs::IStorage* pStorage, std::mt19937* pMt) NN_NOEXCEPT
    {
        static const size_t BufferSize = 2 * 1024;

        std::unique_ptr<char[]> buffer(new char[BufferSize]);
        NN_ABORT_UNLESS_NOT_NULL(buffer);

        int64_t storageSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pStorage->GetSize(&storageSize));

        int64_t offset = 0;
        while( offset < storageSize )
        {
            size_t size = std::uniform_int_distribution<size_t>(64, BufferSize)(*pMt);
            if( storageSize < offset + static_cast<int64_t>(size) )
            {
                size = static_cast<size_t>(storageSize - offset);
            }

            NN_RESULT_DO(m_Impl.ReadData(offset, buffer.get(), size));
            NN_RESULT_DO(pStorage->Write(offset, buffer.get(), size));

            offset += size;
        }
        NN_RESULT_SUCCESS;
    }

    // IndirectStorage を作成します。
    Result MakeStorage(IndirectStorage* pOutStorage, nn::fs::IStorage* pTableStorage) NN_NOEXCEPT
    {
        const int64_t nodeOffset = BucketTree::QueryHeaderStorageSize();
        const int64_t nodeSize = m_Impl.QueryTableNodeStorageSize();
        const int64_t entryOffset = nodeOffset + nodeSize;
        const int64_t entrySize = m_Impl.QueryTableEntryStorageSize();

        return pOutStorage->Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            fs::SubStorage(pTableStorage, nodeOffset, nodeSize),
            fs::SubStorage(pTableStorage, entryOffset, entrySize),
            m_Impl.m_MatchCount
        );
    }

    // 生成したデータをチェックします。
    Result Verify(void* buffer1, void* buffer2, size_t bufferSize) NN_NOEXCEPT
    {
        return Verify(buffer1, buffer2, bufferSize, nullptr);
    }

    // 生成したデータをチェックします。
    Result Verify(void* buffer1, void* buffer2, size_t bufferSize, std::mt19937* pMt) NN_NOEXCEPT
    {
        const int64_t tableStorageSize = QueryTableStorageSize();
        nnt::fs::util::SafeMemoryStorage tableStorage(tableStorageSize);
        NN_RESULT_DO(WriteTable(&tableStorage));

        const int64_t dataStorageSize = QueryDataStorageSize();
        nnt::fs::util::SafeMemoryStorage dataStorage(dataStorageSize);
        if( pMt != nullptr )
        {
            NN_RESULT_DO(WriteData(&dataStorage, pMt));
        }
        else
        {
            NN_RESULT_DO(WriteData(&dataStorage));
        }

        nn::fssystem::IndirectStorage storage;
        NN_RESULT_DO(MakeStorage(&storage, &tableStorage));

        fs::IStorage& oldStorage = GetOldStorage();
        int64_t oldStorageSize = 0;
        NN_RESULT_DO(oldStorage.GetSize(&oldStorageSize));
        storage.SetStorage(0, &oldStorage, 0, oldStorageSize);
        storage.SetStorage(1, &dataStorage, 0, dataStorageSize);

        fs::IStorage& newStorage = GetNewStorage();
        int64_t size = 0;
        NN_RESULT_DO(newStorage.GetSize(&size));

        int64_t offset = 0;
        while( offset < size )
        {
            const auto readSize = static_cast<size_t>(
                std::min(size - offset, static_cast<int64_t>(bufferSize)));

            NN_RESULT_DO(storage.Read(offset, buffer1, readSize));
            NN_RESULT_DO(newStorage.Read(offset, buffer2, readSize));

            if( std::memcmp(buffer1, buffer2, readSize) != 0 )
            {
                NN_LOG("offset %lld\n", offset);
                nnt::fs::util::DumpBufferDiff(buffer1, buffer2, readSize);
                return fs::ResultUnexpected();
            }

            offset += readSize;
        }
        NN_RESULT_SUCCESS;
    }

    // テーブルサイズを取得します。
    int64_t QueryTableStorageSize() const NN_NOEXCEPT
    {
        return BucketTree::QueryHeaderStorageSize() + m_Impl.QueryTableStorageSize();
    }

    // データサイズを取得します。
    int64_t QueryDataStorageSize() const NN_NOEXCEPT
    {
        return m_Impl.QueryDataStorageSize();
    }

    // 古いデータ用のストレージを取得します。
    fs::IStorage& GetOldStorage() NN_NOEXCEPT
    {
        return *m_pOldStorage;
    }

    // 新しいデータ用のストレージを取得します。
    fs::IStorage& GetNewStorage() NN_NOEXCEPT
    {
        return *m_pNewStorage;
    }

    // 古いデータ用のモニタストレージを取得します。
    MonitorFileStorage& GetOldMonitorStorage() NN_NOEXCEPT
    {
        return m_OldFileStorage;
    }

    // 新しいデータ用のモニタストレージを取得します。
    MonitorFileStorage& GetNewMonitorStorage() NN_NOEXCEPT
    {
        return m_NewFileStorage;
    }

    // ストレージの比較情報を取得します。
    const nnt::fs::util::Vector<Match>& GetMatches() const NN_NOEXCEPT
    {
        return m_Matches;
    }

    // ストレージの比較結果を取得します。
    const IndirectStorageBuilder::MatchEntry* GetEntries() const NN_NOEXCEPT
    {
        return m_Impl.m_Match.get();
    }

    // ストレージの比較結果を取得します。
    int GetEntryCount() const NN_NOEXCEPT
    {
        return m_Impl.m_MatchCount;
    }

    // IndirectStorageBuilder を取得します。
    IndirectStorageBuilder& GetImpl() NN_NOEXCEPT
    {
        return m_Impl;
    }

    // IndirectStorageBuilder を取得します。
    const IndirectStorageBuilder& GetImpl() const NN_NOEXCEPT
    {
        return m_Impl;
    }

    const nnt::fs::util::String& GetOldFilePath() const NN_NOEXCEPT
    {
        return m_OldFilePath;
    }

    const nnt::fs::util::String& GetNewFilePath() const NN_NOEXCEPT
    {
        return m_NewFilePath;
    }

    int GetMatchCount() const NN_NOEXCEPT
    {
        return m_Impl.m_MatchCount;
    }

    void SetBinaryRegion(const BinaryRegionArray regions) NN_NOEXCEPT
    {
        m_Impl.SetBinaryRegion(regions);
    }

    const BinaryRegionArray GetBinaryRegion() const NN_NOEXCEPT
    {
        return m_Impl.GetBinaryRegion();
    }

private:
    struct File
    {
        int64_t offset;
        size_t size;
        int seed;
    };

    class SeedGenerator
    {
    public:
        explicit SeedGenerator(std::mt19937* pRandom) NN_NOEXCEPT
            : m_Random(*pRandom)
            , m_Seeds()
        {
        }

        int Generate() NN_NOEXCEPT
        {
            for( ; ; )
            {
                int seed = m_Random();
                auto it = std::find(m_Seeds.begin(), m_Seeds.end(), seed);
                if( it == m_Seeds.end() )
                {
                    m_Seeds.push_back(seed);
                    return seed;
                }
            }
        }

    private:
        std::mt19937& m_Random;
        nnt::fs::util::Vector<int> m_Seeds;
    };

private:
    template< typename Functor >
    Result CreateFiles(
             const char* oldFileName,
             const char* newFileName,
             Functor generate
         ) NN_NOEXCEPT
    {
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        SeedGenerator seeds(&mt);
        nnt::fs::util::Vector<char> buffer(m_FileSizeMax);
        nnt::fs::util::Vector<File> files;
        files.reserve(static_cast<size_t>(m_BinarySize / m_FileSizeMin));

        // 古いデータ用のファイルを作成
        {
            fs::DeleteFile(oldFileName);

            NN_DETAIL_FS_RESULT_LOG_DO(fs::CreateFile(oldFileName, 0));

            fs::FileHandle handle;
            NN_DETAIL_FS_RESULT_LOG_DO(
                fs::OpenFile(
                    &handle,
                    oldFileName,
                    fs::OpenMode_Write | fs::OpenMode_AllowAppend
                )
            );
            bool isFailure = true;
            NN_UTIL_SCOPE_EXIT
            {
                if (isFailure)
                {
                    NN_DETAIL_FS_ERROR("Forced CloseFile (line:%d - scope exit)\n", __LINE__);
                    nn::fs::CloseFile(handle);
                }
            };

            int64_t binarySize = 0;
            while( binarySize < m_BinarySize )
            {
                File file =
                {
                    binarySize,
                    util::align_down(
                        std::uniform_int_distribution<size_t>(
                            m_FileSizeMin, m_FileSizeMax)(mt),
                        m_BlockSize
                    ),
                    seeds.Generate()
                };

                generate(buffer.data(), file.size, file.seed);

                NN_DETAIL_FS_RESULT_LOG_DO(
                    fs::WriteFile(
                        handle,
                        binarySize,
                        buffer.data(),
                        file.size,
                        fs::WriteOption()
                    )
                );

                files.push_back(file);
                binarySize += file.size;
            }

            NN_DETAIL_FS_RESULT_LOG_DO(fs::FlushFile(handle));

            fs::CloseFile(handle);
            isFailure = false;
        }

        // 新しいデータ用のファイルを作成
        {
            const auto fileCountMax =
                std::uniform_int_distribution<size_t>(
                    files.size() * (100 - m_FileCountVariation) / 100,
                    files.size() * (100 + m_FileCountVariation) / 100
                )(mt);

            m_Matches.reserve(fileCountMax);

            fs::DeleteFile(newFileName);

            NN_DETAIL_FS_RESULT_LOG_DO(fs::CreateFile(newFileName, 0));

            fs::FileHandle handle;
            NN_DETAIL_FS_RESULT_LOG_DO(
                fs::OpenFile(
                    &handle,
                    newFileName,
                    fs::OpenMode_Write | fs::OpenMode_AllowAppend
                )
            );
            bool isFailure = true;
            NN_UTIL_SCOPE_EXIT
            {
                if (isFailure)
                {
                    NN_DETAIL_FS_ERROR("Forced CloseFile (line:%d - scope exit)\n", __LINE__);
                    nn::fs::CloseFile(handle);
                }
            };

            size_t fileIndex = 0;
            size_t binarySize = 0;

            while( m_Matches.size() < fileCountMax )
            {
                const auto fileSelect =
                    std::uniform_int_distribution<size_t>(
                        0, files.size() * (100 + m_NewFilePreference) / 100
                    )(mt);

                Match match;
                int seed;

                if( fileSelect < files.size() && fileIndex < files.size() )
                {
                    match.oldOffset = files[fileIndex].offset;
                    match.newOffset = binarySize;
                    match.size = files[fileIndex].size;

                    seed = files[fileIndex].seed;
                }
                else
                {
                    match.oldOffset = std::numeric_limits<int64_t>::min();
                    match.newOffset = binarySize;
                    if( fileIndex < files.size() &&
                        std::uniform_int_distribution<>(0, 8)(mt) != 0 )
                    {
                        match.size = files[fileIndex].size;
                    }
                    else
                    {
                        match.size =
                            util::align_down(
                                std::uniform_int_distribution<size_t>(
                                    m_FileSizeMin, m_FileSizeMax)(mt),
                                m_BlockSize
                            );
                    }

                    seed = seeds.Generate();

                    m_PatchSize += match.size;
                }

                generate(buffer.data(), match.size, seed);

                NN_DETAIL_FS_RESULT_LOG_DO(
                    fs::WriteFile(
                        handle,
                        binarySize,
                        buffer.data(),
                        match.size,
                        fs::WriteOption()
                    )
                );

                m_Matches.push_back(match);
                binarySize += match.size;

                ++fileIndex;
            }

            NN_DETAIL_FS_RESULT_LOG_DO(fs::FlushFile(handle));

            fs::CloseFile(handle);
            isFailure = false;
        }

        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

private:
    int64_t m_BinarySize;
    size_t m_FileSizeMin;
    size_t m_FileSizeMax;
    size_t m_BlockSize;
    size_t m_RegionSize;
    int64_t m_MatchSize;
    int64_t m_WindowSize;
    int m_FileCountVariation;
    int m_NewFilePreference;
    nnt::fs::util::String m_OldFilePath;
    nnt::fs::util::String m_NewFilePath;
    fs::IStorage* m_pOldStorage;
    fs::IStorage* m_pNewStorage;
    MonitorFileStorage m_OldFileStorage;
    MonitorFileStorage m_NewFileStorage;
    nnt::fs::util::Vector<Match> m_Matches;
    int64_t m_PatchSize;
    IndirectStorageBuilder m_Impl;
};

NN_DEFINE_STATIC_CONSTANT(const size_t IndirectStorageBuilderTest::BlockSize);
const char* const IndirectStorageBuilderTest::BuilderWrapper::OldFileName = "old.bin";
const char* const IndirectStorageBuilderTest::BuilderWrapper::NewFileName = "new.bin";

}}}

namespace {

typedef nn::fssystem::utilTool::IndirectStorageBuilder IndirectStorageBuilder;
typedef nn::fssystem::utilTool::IndirectStorageBuilderTest IndirectStorageBuilderTest;
typedef nn::fssystem::utilTool::IndirectStorageBuilderTest IndirectStorageBuilderDeathTest;

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 * @brief   事前検証の範囲外のテストをします。
 */
TEST_F(IndirectStorageBuilderDeathTest, Precondition)
{
    nnt::fs::util::SafeMemoryStorage storage1(128);
    nnt::fs::util::SafeMemoryStorage storage2(128);

    // Initialize() のチェック
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Initialize(nullptr, &storage2, 16), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Initialize(&storage1, nullptr, 16), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Initialize(&storage1, &storage1, 16), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Initialize(&storage1, &storage1, 4), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Initialize(&storage1, &storage1, 15), "");

    // SetExcludeRangeXXX() のチェック
    {
        ExcludeRange range;
        EXPECT_DEATH_IF_SUPPORTED(
            IndirectStorageBuilder().SetExcludeRangeForOldStorage(nullptr, 1),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            IndirectStorageBuilder().SetExcludeRangeForOldStorage(&range, 0),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            IndirectStorageBuilder().SetExcludeRangeForNewStorage(nullptr, 1),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            IndirectStorageBuilder().SetExcludeRangeForNewStorage(&range, 0),
            ""
        );
    }

    // 未初期化
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Build(8, 16, 32), "");

    // 多重初期化のチェック
    {
        IndirectStorageBuilder builder;
        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(&storage1, &storage2, 16));
        EXPECT_DEATH_IF_SUPPORTED(builder.Initialize(&storage1, &storage2, 16), "");
    }

    // Build() のチェック
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Build(7, 16, 32), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Build(12, 16, 32), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Build(16, 8, 32), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Build(8, 17, 32), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Build(8, 32, 16), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Build(0, 16, 32, 64, std::numeric_limits<int64_t>::max()), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Build(3, 16, 32, 64, std::numeric_limits<int64_t>::max()), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Build(32, 16, 32, 64, std::numeric_limits<int64_t>::max()), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().Build(16, 16, 32, 64, 64), "");

    nn::fs::SubStorage storage;
    char buffer[32];

    // WriteTable() のチェック
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder()
            .WriteTable(nullptr, storage, storage, storage),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder()
            .WriteTable(nnt::fs::util::GetTestLibraryAllocator(), storage, storage, storage),
        ""
    );

    // WriteData() のチェック
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().WriteData(nullptr, buffer, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().WriteData(&storage, nullptr, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().WriteData(&storage, buffer, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().WriteData(&storage, buffer, 1), "");

    // ReadData() のチェック
    const int64_t zero = 0;
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().ReadData(-1, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().ReadData(zero, nullptr, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().ReadData(zero, buffer, 1), "");

    // Query 系のチェック
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().QueryTableNodeStorageSize(), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().QueryTableEntryStorageSize(), "");
    EXPECT_DEATH_IF_SUPPORTED(
        IndirectStorageBuilder().QueryDataStorageSize(), "");
}
#endif

/**
 * @brief   IndirectStorage の読み込みをテストします。
 */
TEST_F(IndirectStorageBuilderTest, StorageAccess)
{
    static const size_t BufferSize = 4 * 1024;
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t WindowSize = 4 * Giga;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 10; ++i )
    {
        const int64_t binarySize = (256 * 1024) << std::uniform_int_distribution<>(0, 4)(mt);
        const size_t fileSizeMin = MinimumRegionSize << std::uniform_int_distribution<>(0, 3)(mt);
        const size_t fileSizeMax = fileSizeMin << std::uniform_int_distribution<>(0, 3)(mt);
        const int64_t matchSize = fileSizeMin << std::uniform_int_distribution<>(0, 2)(mt);
        const int blockCount = static_cast<int>(fileSizeMin / BlockSize);
        NN_UNUSED(blockCount);

        BuilderWrapper builder(
            binarySize, fileSizeMin, fileSizeMax, BlockSize, fileSizeMin, matchSize, WindowSize);

        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(GetTemporaryPath()));
        NNT_ASSERT_RESULT_SUCCESS(builder.Build());
        NNT_ASSERT_RESULT_SUCCESS(builder.Verify(buffer1.get(), buffer2.get(), BufferSize));
    }

    // バイナリサイズ < リージョンサイズでテスト
    for( int i = 0; i < 5; ++i )
    {
        BuilderWrapper builder(
            4096 - 512, 128, 512, BlockSize, 4096, 8192, WindowSize);

        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(GetTemporaryPath()));
        NNT_ASSERT_RESULT_SUCCESS(builder.Build());
        NNT_ASSERT_RESULT_SUCCESS(builder.Verify(buffer1.get(), buffer2.get(), BufferSize));
    }
}

/**
 * @brief   IndirectStorage の読み込み（ストレージ全体）をテストします。
 */
TEST_F(IndirectStorageBuilderTest, StorageAccessAtOnce)
{
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t WindowSize = 4 * Giga;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 5; ++i )
    {
        const int64_t binarySize = (256 * 1024) << std::uniform_int_distribution<>(0, 4)(mt);
        const size_t fileSizeMin = MinimumRegionSize << std::uniform_int_distribution<>(0, 3)(mt);
        const size_t fileSizeMax = fileSizeMin << std::uniform_int_distribution<>(0, 3)(mt);
        const int64_t matchSize = fileSizeMin << std::uniform_int_distribution<>(0, 2)(mt);
        const int blockCount = static_cast<int>(fileSizeMin / BlockSize);
        NN_UNUSED(blockCount);

        BuilderWrapper builder(
            binarySize, fileSizeMin, fileSizeMax, BlockSize, fileSizeMin, matchSize, WindowSize);

        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(GetTemporaryPath()));
        NNT_ASSERT_RESULT_SUCCESS(builder.Build());

        int64_t storageSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(builder.GetNewStorage().GetSize(&storageSize));
        const auto bufferSize = static_cast<size_t>(storageSize);

        std::unique_ptr<char[]> buffer1(new char[bufferSize]);
        std::unique_ptr<char[]> buffer2(new char[bufferSize]);

        NNT_ASSERT_RESULT_SUCCESS(builder.Verify(buffer1.get(), buffer2.get(), bufferSize));
    }
}

/**
 * @brief   テーブル生成とテーブル注入とで同じ内部状態になることをテストします。
 */
TEST_F(IndirectStorageBuilderTest, BuildByGenerated)
{
    static const size_t BufferSize = 4 * 1024;
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t WindowSize = 4 * Giga;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 10; ++i )
    {
        const int64_t binarySize = (256 * 1024) << std::uniform_int_distribution<>(0, 4)(mt);
        const size_t fileSizeMin = MinimumRegionSize << std::uniform_int_distribution<>(0, 3)(mt);
        const size_t fileSizeMax = fileSizeMin << std::uniform_int_distribution<>(0, 3)(mt);
        const int blockCount = static_cast<int>(fileSizeMin / BlockSize);
        NN_UNUSED(blockCount);

        BuilderWrapper builder1(
            binarySize, fileSizeMin, fileSizeMax, BlockSize, fileSizeMin, fileSizeMin, WindowSize);

        NNT_ASSERT_RESULT_SUCCESS(builder1.Initialize(GetTemporaryPath()));
        NNT_ASSERT_RESULT_SUCCESS(builder1.Build());
        NNT_ASSERT_RESULT_SUCCESS(builder1.Verify(buffer1.get(), buffer2.get(), BufferSize));

        const int64_t tableStorageSize = builder1.QueryTableStorageSize();
        nnt::fs::util::SafeMemoryStorage tableStorage(tableStorageSize);
        NNT_ASSERT_RESULT_SUCCESS(builder1.WriteTable(&tableStorage));

        BuilderWrapper builder2(
            binarySize, fileSizeMin, fileSizeMax, BlockSize, fileSizeMin, fileSizeMin, WindowSize);

        NNT_ASSERT_RESULT_SUCCESS(builder2.Initialize(builder1.GetOldFilePath().c_str(), builder1.GetNewFilePath().c_str()));
        NNT_ASSERT_RESULT_SUCCESS(builder2.Build(builder1.GetImpl(), &tableStorage));
        NNT_ASSERT_RESULT_SUCCESS(builder2.Verify(buffer1.get(), buffer2.get(), BufferSize));

        auto matches1Count = builder1.GetEntryCount();
        auto matches2Count = builder2.GetEntryCount();
        ASSERT_EQ(matches1Count, matches2Count);
        auto matches1 = builder1.GetEntries();
        auto matches2 = builder2.GetEntries();
        for( int j = 0; j < matches1Count; ++j )
        {
            ASSERT_EQ(matches1[j].virtualOffset, matches2[j].virtualOffset);
            ASSERT_EQ(matches1[j].physicalOffset, matches2[j].physicalOffset);
            ASSERT_EQ(matches1[j].storageIndex, matches2[j].storageIndex);
        }
    }
}

/**
 * @brief   データの部分書き出しをテストします。
 */
TEST_F(IndirectStorageBuilderTest, WriteBuffer)
{
    static const size_t BufferSize = 4 * 1024;
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t WindowSize = 4 * Giga;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 10; ++i )
    {
        const int64_t binarySize = (256 * 1024) << std::uniform_int_distribution<>(0, 4)(mt);
        const size_t fileSizeMin = MinimumRegionSize << std::uniform_int_distribution<>(0, 3)(mt);
        const size_t fileSizeMax = fileSizeMin << std::uniform_int_distribution<>(0, 3)(mt);
        const int blockCount = static_cast<int>(fileSizeMin / BlockSize);
        NN_UNUSED(blockCount);

        BuilderWrapper builder(
            binarySize, fileSizeMin, fileSizeMax, BlockSize, fileSizeMin, 1, WindowSize);

        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(GetTemporaryPath()));
        NNT_ASSERT_RESULT_SUCCESS(builder.Build());
        NNT_ASSERT_RESULT_SUCCESS(builder.Verify(buffer1.get(), buffer2.get(), BufferSize, &mt));
    }

    // QueryDataStorageSize() を超える ReadData() 呼び出しのチェック
    {
        BuilderWrapper builder(2048, 128, 512, BlockSize, 4096, 1, WindowSize);
        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(GetTemporaryPath()));
        NNT_ASSERT_RESULT_SUCCESS(builder.Build());

        const auto dataSize = static_cast<int>(builder.GetImpl().QueryDataStorageSize());
        if( dataSize <= BufferSize )
        {
            std::memset(buffer2.get(), 0, BufferSize);

            // 一部 QueryDataStorageSize() と被る領域のテスト
            {
                std::memset(buffer1.get(), 0xFF, dataSize);

                const auto dataOffset = dataSize / 2;
                NNT_ASSERT_RESULT_SUCCESS(
                    builder.GetImpl().ReadData(dataOffset, buffer1.get(), dataSize));

                NNT_FS_UTIL_ASSERT_MEMCMPEQ(
                    buffer1.get() + dataOffset, buffer2.get(), dataSize / 2);
            }

            // 全て QueryDataStorageSize() と被らない領域のテスト
            {
                std::memset(buffer1.get(), 0xFF, dataSize);

                NNT_ASSERT_RESULT_SUCCESS(
                    builder.GetImpl().ReadData(dataSize, buffer1.get(), dataSize));

                NNT_FS_UTIL_ASSERT_MEMCMPEQ(
                    buffer1.get(), buffer2.get(), dataSize / 2);
            }
        }
    }
}

/**
 * @brief   重複するデータが適切に処理されるかテストします。
 */
TEST_F(IndirectStorageBuilderTest, DuplicateData)
{
    static const size_t BufferSize = 4 * 1024;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 20; ++i )
    {
        const auto regionSize = MinimumRegionSize << std::uniform_int_distribution<>(0, 3)(mt);
        const auto regionCount = std::uniform_int_distribution<>(16, 64)(mt);
        const auto storageSize = regionSize * regionCount;

        nnt::fs::util::Vector<char> oldBuffer(storageSize, 0);
        {
            nn::util::BytePtr ptr(oldBuffer.data());
            for( int j = 0; j < regionCount; ++j )
            {
                *ptr.Get<int>() = std::uniform_int_distribution<>(0, 16)(mt);
                ptr.Advance(regionSize);
            }
        }

        nnt::fs::util::Vector<char> newBuffer(storageSize, 0);
        {
            nn::util::BytePtr ptr(newBuffer.data());
            for( int j = 0; j < regionCount; ++j )
            {
                *ptr.Get<int>() = std::uniform_int_distribution<>(8, 24)(mt);
                ptr.Advance(regionSize);
            }
        }

        nn::fs::MemoryStorage oldStorage(oldBuffer.data(), storageSize);
        nn::fs::MemoryStorage newStorage(newBuffer.data(), storageSize);

        BuilderWrapper builder(
            0, 0, 0, BlockSize, regionSize, 1, std::numeric_limits<int64_t>::max());

        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(&oldStorage, &newStorage));
        NNT_ASSERT_RESULT_SUCCESS(builder.BuildOnly());
        NNT_ASSERT_RESULT_SUCCESS(builder.Verify(buffer1.get(), buffer2.get(), BufferSize));
    }
}

/**
 * @brief   リージョンサイズに満たない部分が適切に処理されるかテストします。
 */
TEST_F(IndirectStorageBuilderTest, FragmentData)
{
    static const size_t BufferSize = 4 * 1024;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 20; ++i )
    {
        const auto regionSize = MinimumRegionSize << std::uniform_int_distribution<>(0, 3)(mt);
        const auto regionCount = std::uniform_int_distribution<>(8, 64)(mt);

        const auto newStorageSize = regionSize * regionCount;

        nnt::fs::util::Vector<char> newBuffer(newStorageSize, 0);
        {
            nn::util::BytePtr ptr(newBuffer.data());
            for( int j = 0; j < regionCount - 1; ++j )
            {
                *ptr.Get<int>() = std::uniform_int_distribution<>(1, 16)(mt);
                ptr.Advance(regionSize);
            }
        }

        const auto oldStorageSize = regionSize * (regionCount - 1)
                                    + std::uniform_int_distribution<>(1, regionSize / BlockSize - 1)(mt) * BlockSize;

        nnt::fs::util::Vector<char> oldBuffer(oldStorageSize, 0);
        std::memcpy(oldBuffer.data(), newBuffer.data(), oldStorageSize);
        {
            const auto offset = regionSize * (regionCount - 2);
            std::memset(newBuffer.data() + offset, 0xFFFFFFFF, regionSize);
        }

        nn::fs::MemoryStorage newStorage(newBuffer.data(), newStorageSize);
        nn::fs::MemoryStorage oldStorage(oldBuffer.data(), oldStorageSize);

        BuilderWrapper builder(
            0, 0, 0, BlockSize, regionSize, 1, std::numeric_limits<int64_t>::max());

        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(&oldStorage, &newStorage));
        NNT_ASSERT_RESULT_SUCCESS(builder.BuildOnly());
        EXPECT_EQ(builder.GetEntryCount(), 2);
        NNT_EXPECT_RESULT_SUCCESS(builder.Verify(buffer1.get(), buffer2.get(), BufferSize));
    }
}

namespace nn { namespace fssystem {

void IndirectStorageBuilderTest::ExcludeStorageAccessImpl(
        int loopCount,
        std::function<bool(ExcludeRangePair*, BuilderWrapper*)> getter
    ) NN_NOEXCEPT
{
    static const size_t BufferSize = 4 * 1024;
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t WindowSize = 4 * Giga;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < loopCount; ++i )
    {
        const int64_t binarySize = (256 * 1024) << std::uniform_int_distribution<>(0, 4)(mt);
        const size_t fileSizeMin = MinimumRegionSize << std::uniform_int_distribution<>(0, 3)(mt);
        const size_t fileSizeMax = fileSizeMin << std::uniform_int_distribution<>(0, 3)(mt);
        const int blockCount = static_cast<int>(fileSizeMin / BlockSize);
        NN_UNUSED(blockCount);

        BuilderWrapper builder(
            binarySize, fileSizeMin, fileSizeMax, BlockSize, fileSizeMin, 1, WindowSize);

        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(GetTemporaryPath()));
        {
            int64_t oldStorageSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(builder.GetOldStorage().GetSize(&oldStorageSize));
            int64_t newStorageSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(builder.GetNewStorage().GetSize(&newStorageSize));

            ExcludeRangePair exclude;
            if( !getter(&exclude, &builder) )
            {
                continue;
            }

            // exclude を比較しないように範囲設定
            const ExcludeRange oldExcludeRange[1] = { { exclude.oldOffset, exclude.size } };
            builder.GetImpl().SetExcludeRangeForOldStorage(oldExcludeRange);

            const ExcludeRange newExcludeRange[1] = { { exclude.newOffset, exclude.size } };
            builder.GetImpl().SetExcludeRangeForNewStorage(newExcludeRange);

            NNT_ASSERT_RESULT_SUCCESS(builder.Build());

            const auto entryArray = builder.GetEntries();
            const auto entryCount = builder.GetEntryCount();

            // newStorage の比較しない領域を使用していないことを確認
            if( exclude.newOffset < newStorageSize )
            {
                if( newStorageSize < exclude.newOffset + exclude.size )
                {
                    exclude.size = newStorageSize - exclude.newOffset;
                }

                // 分割したデータが含まれるデータを取得
                const auto pEntry = std::lower_bound(
                    entryArray,
                    entryArray + entryCount,
                    exclude.newOffset + exclude.size,
                    [](const MatchEntry& entry, int64_t offset) NN_NOEXCEPT
                    {
                        return entry.virtualOffset < offset;
                    }
                ) - 1;
                ASSERT_GE(pEntry, entryArray);

                // 比較結果が newStorage のみを参照しているかチェック
                ASSERT_EQ(pEntry->storageIndex, 1);
                ASSERT_LE(pEntry->virtualOffset, exclude.newOffset);
            }

            // oldStorage の比較しない領域を使用していないことを確認
            if( exclude.oldOffset < oldStorageSize )
            {
                if( oldStorageSize < exclude.oldOffset + exclude.size )
                {
                    exclude.size = oldStorageSize - exclude.oldOffset;
                }

                for( int j = 0; j < entryCount - 1; ++j )
                {
                    const auto& entry = entryArray[j];
                    if( entry.storageIndex == 0 )
                    {
                        auto size = entryArray[j + 1].virtualOffset - entry.virtualOffset;
                        ASSERT_GE(size, 0);
                        ASSERT_TRUE(
                            exclude.oldOffset + exclude.size <= entry.physicalOffset ||
                            entry.physicalOffset + size <= exclude.oldOffset);
                    }
                }
                {
                    const auto& entry = entryArray[entryCount - 1];
                    if( entry.storageIndex == 0 )
                    {
                        auto size = newStorageSize - entry.virtualOffset;
                        ASSERT_GE(size, 0);
                        ASSERT_TRUE(
                            exclude.oldOffset + exclude.size <= entry.physicalOffset ||
                            entry.physicalOffset + size <= exclude.oldOffset);
                    }
                }
            }
        }
        NNT_ASSERT_RESULT_SUCCESS(builder.Verify(buffer1.get(), buffer2.get(), BufferSize));
    }
}

}}

/**
 * @brief   範囲指定した IndirectStorage の読み込みをテストします。
 */
TEST_F(IndirectStorageBuilderTest, ExcludeStorageAccess)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    ExcludeStorageAccessImpl(
        20,
        [&](ExcludeRangePair* pOutRange, BuilderWrapper* pBuilder) NN_NOEXCEPT
        {
            const auto& matches = pBuilder->GetMatches();
            for( size_t i = 0, count = matches.size(); i < count; ++i )
            {
                auto index = std::uniform_int_distribution<size_t>(0, count - 1)(mt);

                const auto& exclude = matches[index];
                if( exclude.oldOffset != std::numeric_limits<int64_t>::min() )
                {
                    pOutRange->newOffset = exclude.newOffset;
                    pOutRange->oldOffset = exclude.oldOffset;
                    pOutRange->size = static_cast<int64_t>(exclude.size);

                    return true;
                }
            }
            return false;
        }
    );
}

/**
 * @brief   ストレージの範囲外を範囲指定した IndirectStorage の読み込みをテストします。
 */
TEST_F(IndirectStorageBuilderTest, ExcludeRangeOverStorageAccess)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    ExcludeStorageAccessImpl(
        10,
        [&](ExcludeRangePair* pOutRange, BuilderWrapper* pBuilder) NN_NOEXCEPT
        {
            int64_t newStorageSize = 0;
            if( pBuilder->GetNewStorage().GetSize(&newStorageSize).IsFailure() )
            {
                return false;
            }

            // ストレージからはみ出す領域を指定
            if( std::uniform_int_distribution<>(0, 1)(mt) == 0 )
            {
                const auto& matches = pBuilder->GetMatches();
                for( int i = static_cast<int>(matches.size()) - 1; 0 <= i; --i )
                {
                    const auto& exclude = matches[i];
                    if( exclude.oldOffset != std::numeric_limits<int64_t>::min() )
                    {
                        pOutRange->newOffset = exclude.newOffset;
                        pOutRange->oldOffset = exclude.oldOffset;
                        pOutRange->size = newStorageSize - exclude.newOffset + 16;

                        break;
                    }
                }
            }
            // ストレージと被らない領域を指定
            else
            {
                int64_t oldStorageSize = 0;
                if( pBuilder->GetOldStorage().GetSize(&oldStorageSize).IsFailure() )
                {
                    return false;
                }

                pOutRange->newOffset = newStorageSize + 16;
                pOutRange->oldOffset = oldStorageSize + 16;
                pOutRange->size = 16;
            }

            return true;
        }
    );
}

/**
 * @brief   BinaryRegionFile をテストします。
 */
TEST_F(IndirectStorageBuilderTest, BinaryRegionFile)
{
    typedef nn::fssystem::utilTool::BinaryRegionArray RegionArray;
    typedef RegionArray::Region Region;
    typedef nn::fssystem::utilTool::BinaryRegionFile RegionFile;
    typedef RegionFile::Header RegionFileHeader;

    static const size_t BufferSize = 4 * 1024;
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t WindowSize = 4 * Giga;
    static const size_t HeaderSize = nn::fssystem::NcaFsHeader::Size;

    const auto rootPath = GetTemporaryPath();

    std::unique_ptr<char[]> header(new char[HeaderSize]);
    std::memset(header.get(), 0, HeaderSize);

    nnt::fs::util::Vector<char> oldBuffer;
    nnt::fs::util::Vector<char> newBuffer;
    nnt::fs::util::Vector<char> fileBuffer;
    nnt::fs::util::Vector<Region> backupRegion;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 5; ++i )
    {
        const int64_t binarySize = (256 * 1024) << std::uniform_int_distribution<>(0, 4)(mt);
        const size_t fileSizeMin = MinimumRegionSize << std::uniform_int_distribution<>(0, 3)(mt);
        const size_t fileSizeMax = fileSizeMin << std::uniform_int_distribution<>(0, 3)(mt);

        // リージョンファイルの出力
        {
            BuilderWrapper builder(
                binarySize, fileSizeMin, fileSizeMax, BlockSize, fileSizeMin, fileSizeMin, WindowSize);

            NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(GetTemporaryPath()));
            NNT_ASSERT_RESULT_SUCCESS(builder.Build());

            // 旧データのバックアップ
            {
                auto& storage = builder.GetOldStorage();
                int64_t storageSize = 0;
                NNT_ASSERT_RESULT_SUCCESS(storage.GetSize(&storageSize));

                oldBuffer.resize(static_cast<size_t>(storageSize));
                NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, oldBuffer.data(), oldBuffer.size()));
            }
            // 新データのバックアップ
            {
                auto& storage = builder.GetNewStorage();
                int64_t storageSize = 0;
                NNT_ASSERT_RESULT_SUCCESS(storage.GetSize(&storageSize));

                newBuffer.resize(static_cast<size_t>(storageSize));
                NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, newBuffer.data(), newBuffer.size()));
            }

            const auto regions = builder.GetBinaryRegion();

            // リージョンファイルを書き出し
            {
                RegionFile regionFile;
                regionFile.SetHeader(header.get(), HeaderSize);
                regionFile.SetData(0, oldBuffer.data(), fileSizeMin);
                regionFile.SetRegion(regions);

                char fileName[128];
                regionFile.GenerateFileName(fileName, sizeof(fileName));

                nnt::fs::util::String path(rootPath);
                path += '\\';
                path += fileName;

                std::ofstream file(path.c_str(), std::ios::binary);

                const void* const headerBuffer = &regionFile.GetHeader();
                file.write(reinterpret_cast<const char*>(headerBuffer), sizeof(RegionFileHeader));

                const void* const regionBuffer = regions.data();
                file.write(reinterpret_cast<const char*>(regionBuffer), regions.GetBytes());
            }

            fileBuffer.resize(sizeof(RegionFileHeader) + regions.GetBytes());
            backupRegion.assign(regions.begin(), regions.end());
        }

        // リージョンファイルの入力
        {
            RegionFile regionFile;
            regionFile.SetHeader(header.get(), HeaderSize);
            regionFile.SetData(0, oldBuffer.data(), fileSizeMin);

            // リージョンファイルの読み込み
            {
                char fileName[128];
                regionFile.GenerateFileName(fileName, sizeof(fileName));

                nnt::fs::util::String path(rootPath);
                path += '\\';
                path += fileName;

                std::ifstream file(path.c_str(), std::ios::binary);

                file.read(fileBuffer.data(), fileBuffer.size());
            }

            ASSERT_TRUE(regionFile.CheckRegion(fileBuffer.data(), fileBuffer.size()));

            // リージョンハッシュのチェック
            {
                const auto regions = regionFile.GetRegion();

                ASSERT_EQ(backupRegion.size(), regions.size());
                ASSERT_TRUE(std::equal(
                    regions.begin(), regions.end(), backupRegion.data(),
                    [](const Region& lhs, const Region& rhs) NN_NOEXCEPT
                    {
                        return std::memcmp(&lhs, &rhs, sizeof(Region)) == 0;
                    }
                ));
            }

            nn::fs::MemoryStorage oldStorage(oldBuffer.data(), oldBuffer.size());
            nn::fs::MemoryStorage newStorage(newBuffer.data(), newBuffer.size());

            BuilderWrapper builder(
                binarySize, fileSizeMin, fileSizeMax, BlockSize, fileSizeMin, fileSizeMin, WindowSize);

            NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(&oldStorage, &newStorage));

            // ビルダーにリージョンハッシュを設定
            builder.SetBinaryRegion(regionFile.GetRegion());

            NNT_ASSERT_RESULT_SUCCESS(builder.BuildOnly());
            NNT_ASSERT_RESULT_SUCCESS(builder.Verify(buffer1.get(), buffer2.get(), BufferSize));
        }

        oldBuffer.clear();
        newBuffer.clear();
        fileBuffer.clear();
        backupRegion.clear();
    }
} // NOLINT(impl/function_size)

TEST_F(IndirectStorageBuilderTest, RelocateIndirectTableEntries)
{
    // ストレージ比較関数
    auto TestStorage = [](nn::fs::IStorage* pLhsStorage, nn::fs::IStorage* pRhsStorage) NN_NOEXCEPT
        {
            int64_t size = 0;
            NNT_ASSERT_RESULT_SUCCESS(pRhsStorage->GetSize(&size));

            static const size_t BufferSize = 4 * 1024;
            std::unique_ptr<char[]> buffer1(new char[BufferSize]);
            std::unique_ptr<char[]> buffer2(new char[BufferSize]);

            int64_t offset = 0;
            while( offset < size )
            {
                const auto readSize = static_cast<size_t>(
                    std::min(size - offset, static_cast<int64_t>(BufferSize)));

                NNT_ASSERT_RESULT_SUCCESS(pLhsStorage->Read(offset, buffer1.get(), readSize));
                NNT_ASSERT_RESULT_SUCCESS(pRhsStorage->Read(offset, buffer2.get(), readSize));

                if( std::memcmp(buffer1.get(), buffer2.get(), readSize) != 0 )
                {
                    NN_LOG("offset %lld\n", offset);
                    NNT_FS_UTIL_ASSERT_MEMCMPEQ(buffer1.get(), buffer2.get(), readSize);
                }

                offset += readSize;
            }
        };

    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t WindowSize = 4 * Giga;

    const int64_t binarySize = (128 * 1024) << 2;
    const size_t fileSizeMin = MinimumRegionSize;
    const size_t fileSizeMax = fileSizeMin << 1;
    const int blockCount = static_cast<int>(fileSizeMin / BlockSize);
    NN_UNUSED(blockCount);

    BuilderWrapper builder(
        binarySize, fileSizeMin, fileSizeMax, BlockSize, fileSizeMin, 1, WindowSize);

    // 新旧のファイル を 作成し、差分を検出しておく
    NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(GetTemporaryPath()));
    NNT_ASSERT_RESULT_SUCCESS(builder.Build());

    const int64_t srcTableStorageSize = builder.QueryTableStorageSize();
    const int64_t srcDataStorageSize = builder.QueryDataStorageSize();

    // srcTableStorage と srcDataStorage を作成
    nnt::fs::util::SafeMemoryStorage srcTableStorage(srcTableStorageSize);
    NNT_ASSERT_RESULT_SUCCESS(builder.WriteTable(&srcTableStorage));
    nnt::fs::util::SafeMemoryStorage srcDataStorage(srcDataStorageSize);
    NNT_ASSERT_RESULT_SUCCESS(builder.WriteData(&srcDataStorage));

    // Src 側の IndirectStorage を作成
    {
        // tableStorage より IndirectStorage を構成する
        nn::fssystem::IndirectStorage storage;
        NNT_ASSERT_RESULT_SUCCESS(builder.MakeStorage(&storage, &srcTableStorage));

        // 旧ファイルとパッチデータを IndirectStorage にストレージとして追加
        MonitorFileStorage& oldStorage = builder.GetOldMonitorStorage();
        int64_t oldStorageSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(oldStorage.GetSize(&oldStorageSize));
        storage.SetStorage(0, &oldStorage, 0, oldStorageSize);
        storage.SetStorage(1, &srcDataStorage, 0, srcDataStorageSize);

        // 新ファイルとIndirectStorageの内容を比較
        MonitorFileStorage& newStorage = builder.GetNewMonitorStorage();
        TestStorage(&newStorage, &storage);
    }

    // RelocationTable を適用する
    {
        MonitorFileStorage& newStorage = builder.GetNewMonitorStorage();

        // dstTableStorage と dstDataStorage を作成
        nnt::fs::util::SafeMemoryStorage dstTableStorage;
        nnt::fs::util::SafeMemoryStorage dstDataStorage;
        {

            // パッチデータの前半と後半を入れ替える RelocationTable を作成
            nn::fssystem::utilTool::RelocationTable relocationTable;
            typedef nn::fssystem::utilTool::RelocationTable::Entry Entry;
            relocationTable.Add(
                Entry(0, srcDataStorageSize / 2, srcDataStorageSize / 2, true));
            relocationTable.Add(
                Entry(srcDataStorageSize / 2, 0, srcDataStorageSize / 2, true));
            NNT_ASSERT_RESULT_SUCCESS(relocationTable.Commit());

            // srcTableStorage に RelocationTable を適用
            {
                nn::fssystem::utilTool::RelocatedIndirectStorageBuilder relocationBuilder;
                {
                    const int64_t headerOffset = 0;
                    const int64_t headerSize = builder.GetImpl().QueryTableHeaderStorageSize();
                    const int64_t nodeOffset = headerOffset + headerSize;
                    const int64_t nodeSize = builder.GetImpl().QueryTableNodeStorageSize();
                    const int64_t entryOffset = nodeOffset + nodeSize;
                    const int64_t entrySize = builder.GetImpl().QueryTableEntryStorageSize();

                    nn::fssystem::utilTool::RelocatedIndirectStorageBuilder::PatchInfo info(
                        builder.GetMatchCount(),
                        0,
                        nn::fs::SubStorage(&srcTableStorage, nodeOffset, nodeSize),
                        nn::fs::SubStorage(&srcTableStorage, entryOffset, entrySize),
                        nn::fs::SubStorage(&srcTableStorage, 0, 0) // data はダミー
                    );

                    NNT_ASSERT_RESULT_SUCCESS(relocationBuilder.Initialize(info, info, 0)); // old 側はダミー
                }

                // 置き換え結果を取得
                NNT_ASSERT_RESULT_SUCCESS(
                    RelocateIndirectTableEntries(&relocationBuilder, relocationTable));

                {
                    const int64_t headerOffset = 0;
                    const int64_t headerSize = relocationBuilder.QueryTableHeaderStorageSize();
                    const int64_t nodeOffset = headerOffset + headerSize;
                    const int64_t nodeSize = relocationBuilder.QueryTableNodeStorageSize();
                    const int64_t entryOffset = nodeOffset + nodeSize;
                    const int64_t entrySize = relocationBuilder.QueryTableEntryStorageSize();

                    dstTableStorage.Initialize(nodeSize + entrySize);

                    // 置換え結果を書き出す
                    relocationBuilder.WriteTable(
                        nnt::fs::util::GetTestLibraryAllocator(),
                        nn::fs::SubStorage(&dstTableStorage, headerOffset, headerSize),
                        nn::fs::SubStorage(&dstTableStorage, nodeOffset, nodeSize),
                        nn::fs::SubStorage(&dstTableStorage, entryOffset, entrySize)
                    );
                }
            }

            // srcDataStorage に RelocationTable を適用
            {
                dstDataStorage.Initialize(srcDataStorageSize);

                nnt::fs::util::Vector<uint8_t> buffer(16 * 1024);
                NNT_ASSERT_RESULT_SUCCESS(
                    relocationTable.ApplyTo(
                        &dstDataStorage, &srcDataStorage, &buffer[0], buffer.size()
                        ));
            }

        }

        // RelocationTable で再配置された IndirectStorage を構成する
        nn::fssystem::IndirectStorage relocatedStorage;
        {
            NNT_ASSERT_RESULT_SUCCESS(builder.MakeStorage(&relocatedStorage, &dstTableStorage));

            // 元ROMとパッチデータを IndirectStorage にストレージとして追加
            MonitorFileStorage& oldStorage = builder.GetOldMonitorStorage();
            int64_t oldStorageSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(oldStorage.GetSize(&oldStorageSize));

            int64_t dstDataStorageSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(dstDataStorage.GetSize(&dstDataStorageSize));

            relocatedStorage.SetStorage(0, &oldStorage, 0, oldStorageSize);
            relocatedStorage.SetStorage(1, &dstDataStorage, 0, dstDataStorageSize);
        }

        // 新ファイルと再配置した IndirectStorage の内容を比較
        TestStorage(&newStorage, &relocatedStorage);
    }
} // NOLINT(impl/function_size)

/**
 * @brief   最小マッチサイズをテストします。
 */
TEST_F(IndirectStorageBuilderTest, MatchSize)
{
    static const size_t BufferSize = 4 * 1024;
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t WindowSize = 4 * Giga;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 10; ++i )
    {
        const int64_t binarySize = (256 * 1024) << std::uniform_int_distribution<>(0, 4)(mt);
        const size_t fileSizeMin = MinimumRegionSize << std::uniform_int_distribution<>(0, 3)(mt);
        const size_t fileSizeMax = fileSizeMin << std::uniform_int_distribution<>(0, 3)(mt);
        const int64_t matchSize = 128 << std::uniform_int_distribution<>(0, 7)(mt);
        const int blockCount = static_cast<int>(fileSizeMin / BlockSize);
        NN_UNUSED(blockCount);

        BuilderWrapper builder(
            binarySize, fileSizeMin, fileSizeMax, BlockSize, fileSizeMin, matchSize, WindowSize);

        NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(GetTemporaryPath()));
        NNT_ASSERT_RESULT_SUCCESS(builder.Build());
        NNT_ASSERT_RESULT_SUCCESS(builder.Verify(buffer1.get(), buffer2.get(), BufferSize));

        const auto sizeMin = std::min(matchSize, static_cast<int64_t>(fileSizeMin));
        const auto pBegin = builder.GetEntries();
        const auto pEnd = pBegin + builder.GetEntryCount();
        for( auto pEntry = pBegin + 1; pEntry < pEnd; ++pEntry )
        {
            const auto pPreviousEntry = pEntry - 1;
            if( pPreviousEntry->storageIndex == 0 )
            {
                EXPECT_GE(pEntry->virtualOffset - pPreviousEntry->virtualOffset, sizeMin);
            }
        }
    }
}

#if 0// 処理に時間がかかるため計測時以外は無効化する
/**
 * @brief   処理時間を計測します。
 */
TEST_F(IndirectStorageBuilderTest, Performance)
{
    static const size_t RegionSize = 16 * 1024;
    static const int64_t MatchSize = 32 * 1024;
    static const int64_t WindowSize = std::numeric_limits<int64_t>::max();
    static const size_t StackSize = 32 * 1024;

    static const char* const OldFilePath = "E:/Projects/SDK/IndirectStorageBuilder/239534/output.fs";
    static const char* const NewFilePath = "E:/projects/SDK/IndirectStorageBuilder/243130/output.fs";
    static const char* const DataFilePath = "E:/projects/SDK/IndirectStorageBuilder/data.bin";

    NN_OS_ALIGNAS_THREAD_STACK static char s_ThreadStack[StackSize] = {};

    BuilderWrapper builder(0, 0, 0, BlockSize, RegionSize, MatchSize, WindowSize);
    NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(OldFilePath, NewFilePath));

    {
        nn::os::ThreadType thread;

        struct Argument
        {
            BuilderWrapper* pBuilder;
            nn::Result result;
            mutable bool isExit;
        }
        argument = { &builder };

        const auto function = [](void* pArg) NN_NOEXCEPT
        {
            auto& arg = *reinterpret_cast<Argument*>(pArg);
            arg.result = arg.pBuilder->Build();
            arg.isExit = true;
        };

        NNT_ASSERT_RESULT_SUCCESS(
            nn::os::CreateThread(
                &thread,
                function,
                &argument,
                s_ThreadStack,
                StackSize,
                nn::os::DefaultThreadPriority
            )
        );

        const auto start = std::chrono::system_clock::now();

        nn::os::StartThread(&thread);

        while( !argument.isExit )
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

            const auto now = std::chrono::system_clock::now();
            const auto pass =
                std::chrono::duration_cast<std::chrono::microseconds>(now - start).count();

            const auto progress = builder.GetImpl().GetProgress();
            if( 0 < progress )
            {
                NN_LOG("Progress: %.3lf%%", progress * 100);

                const auto time = pass / progress / (1000.0 * 1000.0);
                NN_LOG(" | Expected time: %.3lf secs", time);

                NN_LOG(" | Remaining time: %d secs",
                       static_cast<int>((time - (pass / (1000.0 * 1000.0)))));

                const auto io = builder.GetNewMonitorStorage().GetElapsedTime() +
                                builder.GetOldMonitorStorage().GetElapsedTime();

                NN_LOG(" | IO rate: %.2lf%%\n", io / (pass / (1000.0 * 1000.0)) * 100);
            }
        }

        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
    }

    const int64_t tableStorageSize = builder.QueryTableStorageSize();
    nnt::fs::util::SafeMemoryStorage tableStorage(tableStorageSize);
    NNT_ASSERT_RESULT_SUCCESS(builder.WriteTable(&tableStorage));

    const int64_t dataStorageSize = builder.QueryDataStorageSize();
    // 差分データ書き出し
    {
        nn::fs::DeleteFile(DataFilePath);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(DataFilePath, dataStorageSize));

        TestFileStorage dataStorage(BlockSize);
        NNT_ASSERT_RESULT_SUCCESS(
            dataStorage.Initialize(DataFilePath, nn::fs::OpenMode_Write));
        NNT_ASSERT_RESULT_SUCCESS(builder.WriteData(&dataStorage));
        NNT_ASSERT_RESULT_SUCCESS(dataStorage.Flush());
    }

    TestFileStorage dataStorage(BlockSize);
    NNT_ASSERT_RESULT_SUCCESS(dataStorage.Initialize(DataFilePath));

    nn::fssystem::IndirectStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(builder.MakeStorage(&storage, &tableStorage));

    MonitorFileStorage& oldStorage = builder.GetOldMonitorStorage();
    int64_t oldStorageSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(oldStorage.GetSize(&oldStorageSize));
    storage.AddStorage(nn::fs::SubStorage(&oldStorage, 0, oldStorageSize));
    storage.AddStorage(nn::fs::SubStorage(&dataStorage, 0, dataStorageSize));

    MonitorFileStorage& newStorage = builder.GetNewMonitorStorage();
    int64_t size = 0;
    NNT_ASSERT_RESULT_SUCCESS(newStorage.GetSize(&size));

    static const size_t BufferSize = 32 * 1024 * 1024;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);
    int64_t offset = 0;
    while( offset < size )
    {
        const auto readSize = static_cast<size_t>(
            std::min(size - offset, static_cast<int64_t>(BufferSize)));

        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buffer1.get(), readSize));
        NNT_ASSERT_RESULT_SUCCESS(newStorage.Read(offset, buffer2.get(), readSize));

        if( std::memcmp(buffer1.get(), buffer2.get(), readSize) != 0 )
        {
            NN_LOG("offset %lld\n", offset);
            NNT_FS_UTIL_ASSERT_MEMCMPEQ(buffer1.get(), buffer2.get(), readSize);
        }

        offset += readSize;
    }
} // NOLINT(impl/function_size)
#endif

namespace {

void BuildIndirectStorage(
         int64_t oldStorageSize,
         int64_t newStorageSize,
         size_t regionSize
     ) NN_NOEXCEPT
{
    nnt::fs::util::SafeMemoryStorage oldStorage(
        nn::util::align_up(oldStorageSize, IndirectStorageBuilderTest::BlockSize));
    nnt::fs::util::SafeMemoryStorage newStorage(
        nn::util::align_up(newStorageSize, IndirectStorageBuilderTest::BlockSize));

    IndirectStorageBuilder builder;
    NNT_ASSERT_RESULT_SUCCESS(
        builder.Initialize(&oldStorage, &newStorage, IndirectStorageBuilderTest::BlockSize)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        builder.Build(IndirectStorageBuilderTest::BlockSize, regionSize, 0)
    );
}

}

/**
 * @brief   特定のストレージサイズで不具合が発生しないことをテストします。
 */
TEST_F(IndirectStorageBuilderTest, CheckStorageSize)
{
    class Random
    {
    public:
        Random() NN_NOEXCEPT
            : m_Mt()
        {
            m_Mt.Initialize(nnt::fs::util::GetRandomSeed());
        }

        int Get(int min, int max) NN_NOEXCEPT
        {
            return min + m_Mt.GenerateRandomN(static_cast<uint16_t>(max - min + 1));
        }

        int GetPower2(int base, int shift) NN_NOEXCEPT
        {
            return base << m_Mt.GenerateRandomN(static_cast<uint16_t>(shift + 1));
        }

    private:
        nn::util::TinyMt m_Mt;
    };

    Random random;

    // oldStorageSize == newStorageSize &&
    // oldStorageSize % regionSize == 0
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        const int64_t storageSize = regionSize * random.Get(1, 4);

        BuildIndirectStorage(storageSize, storageSize, regionSize);
    }

    // oldStorageSize == newStorageSize &&
    // oldStorageSize % regionSize != 0
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        const int64_t storageSize = regionSize * random.Get(1, 4) +
                                    random.Get(1, regionSize - BlockSize);

        BuildIndirectStorage(storageSize, storageSize, regionSize);
    }

    // oldStorageSize < newStorageSize &&
    // oldStorageSize % regionSize == 0 &&
    // newStorageSize % regionSize == 0
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        const int64_t oldStorageSize = regionSize * random.Get(1, 4);
        const int64_t newStorageSize = oldStorageSize +
                                       regionSize * random.Get(1, 4);

        BuildIndirectStorage(oldStorageSize, newStorageSize, regionSize);
    }

    // oldStorageSize < newStorageSize &&
    // oldStorageSize % regionSize != 0 &&
    // newStorageSize % regionSize == 0
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        int64_t oldStorageSize = regionSize * random.Get(1, 4);
        const int64_t newStorageSize = oldStorageSize +
                                       regionSize * random.Get(1, 4);
        oldStorageSize += random.Get(1, regionSize - BlockSize);

        BuildIndirectStorage(oldStorageSize, newStorageSize, regionSize);
    }

    // oldStorageSize < newStorageSize &&
    // oldStorageSize % regionSize == 0 &&
    // newStorageSize % regionSize != 0
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        const int64_t oldStorageSize = regionSize * random.Get(1, 4);
        const int64_t newStorageSize = oldStorageSize +
                                       regionSize * random.Get(1, 4) +
                                       random.Get(1, regionSize - BlockSize);

        BuildIndirectStorage(oldStorageSize, newStorageSize, regionSize);
    }

    // oldStorageSize < newStorageSize &&
    // oldStorageSize % regionSize != 0 &&
    // newStorageSize % regionSize != 0
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        int64_t oldStorageSize = regionSize * random.Get(1, 4);
        const int64_t newStorageSize = oldStorageSize +
                                       regionSize * random.Get(1, 4) +
                                       random.Get(1, regionSize - BlockSize);
        oldStorageSize += random.Get(1, regionSize - BlockSize);

        BuildIndirectStorage(oldStorageSize, newStorageSize, regionSize);
    }

    // oldStorageSize > newStorageSize &&
    // oldStorageSize % regionSize == 0 &&
    // newStorageSize % regionSize == 0
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        const int64_t newStorageSize = regionSize * random.Get(1, 4);
        const int64_t oldStorageSize = newStorageSize +
                                       regionSize * random.Get(1, 4);

        BuildIndirectStorage(oldStorageSize, newStorageSize, regionSize);
    }

    // oldStorageSize > newStorageSize &&
    // oldStorageSize % regionSize != 0 &&
    // newStorageSize % regionSize == 0
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        const int64_t newStorageSize = regionSize * random.Get(1, 4);
        const int64_t oldStorageSize = newStorageSize +
                                       regionSize * random.Get(1, 4) +
                                       random.Get(1, regionSize - BlockSize);

        BuildIndirectStorage(oldStorageSize, newStorageSize, regionSize);
    }

    // oldStorageSize > newStorageSize &&
    // oldStorageSize % regionSize == 0 &&
    // newStorageSize % regionSize != 0
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        int64_t newStorageSize = regionSize * random.Get(1, 4);
        const int64_t oldStorageSize = newStorageSize +
                                       regionSize * random.Get(1, 4);
        newStorageSize += random.Get(1, regionSize - BlockSize);

        BuildIndirectStorage(oldStorageSize, newStorageSize, regionSize);
    }

    // oldStorageSize > newStorageSize &&
    // oldStorageSize % regionSize != 0 &&
    // newStorageSize % regionSize != 0
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        int64_t newStorageSize = regionSize * random.Get(1, 4);
        const int64_t oldStorageSize = newStorageSize +
                                       regionSize * random.Get(1, 4) +
                                       random.Get(1, regionSize - BlockSize);
        newStorageSize += random.Get(1, regionSize - BlockSize);

        BuildIndirectStorage(oldStorageSize, newStorageSize, regionSize);
    }

    // oldStorageSize < regionSize &&
    // newStorageSize > regionSize
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        const int64_t newStorageSize = regionSize * random.Get(1, 4);
        const int64_t oldStorageSize = regionSize / random.Get(2, 4);

        BuildIndirectStorage(oldStorageSize, newStorageSize, regionSize);
    }

    // oldStorageSize > regionSize &&
    // newStorageSize < regionSize
    for( int i = 0; i < 5; ++i )
    {
        const int regionSize = random.GetPower2(nn::fssystem::utilTool::BinaryMatch::RegionSizeMin, 3);
        const int64_t oldStorageSize = regionSize * random.Get(1, 4);
        const int64_t newStorageSize = regionSize / random.Get(2, 4);

        BuildIndirectStorage(oldStorageSize, newStorageSize, regionSize);
    }
} // NOLINT(impl/function_size)

/**
 * @brief   4 GB を超えるオフセットでアクセスします。
 */
TEST_F(IndirectStorageBuilderTest, StorageAccessLargeHeavy)
{
    static const size_t BufferSize = 4 * 1024 * 1024;
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const size_t RegionSize = 16 * 1024;
    static const int64_t MatchSize = 8192;
    static const int64_t WindowSize = 4 * Giga;
    static const size_t WriteSize = 1024;

    auto buffer1 = nnt::fs::util::AllocateBuffer(BufferSize);
    auto buffer2 = nnt::fs::util::AllocateBuffer(BufferSize);

    nnt::fs::util::VirtualMemoryStorage oldStorage;
    oldStorage.Initialize(16 * Giga);
    nnt::fs::util::VirtualMemoryStorage newStorage;
    newStorage.Initialize(16 * Giga);

    {
        const int64_t offset = 0;

        nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), WriteSize);
        NNT_ASSERT_RESULT_SUCCESS(oldStorage.Write(offset, buffer1.get(), WriteSize));

        nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), WriteSize);
        NNT_ASSERT_RESULT_SUCCESS(newStorage.Write(offset, buffer1.get(), WriteSize));
    }

    {
        const int64_t offset = 4 * Giga;

        nnt::fs::util::FillBufferWith32BitCount(buffer1.get(), WriteSize, 0);
        NNT_ASSERT_RESULT_SUCCESS(oldStorage.Write(offset, buffer1.get(), WriteSize));
        NNT_ASSERT_RESULT_SUCCESS(newStorage.Write(offset, buffer1.get(), WriteSize));
    }

    {
        const int64_t offset = 8 * Giga;

        nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), WriteSize);
        NNT_ASSERT_RESULT_SUCCESS(oldStorage.Write(offset, buffer1.get(), WriteSize));

        nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), WriteSize);
        NNT_ASSERT_RESULT_SUCCESS(newStorage.Write(offset, buffer1.get(), WriteSize));
    }

    BuilderWrapper builder(0, 0, 0, BlockSize, RegionSize, MatchSize, WindowSize);

    NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(&oldStorage, &newStorage));
    NNT_ASSERT_RESULT_SUCCESS(builder.BuildOnly());
    NNT_ASSERT_RESULT_SUCCESS(builder.Verify(buffer1.get(), buffer2.get(), BufferSize));
}

