﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/fs_BucketTree.h>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

typedef nn::fssystem::BucketTree BucketTree;
typedef BucketTree::NodeHeader NodeHeader;

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 * @brief   引数の境界値外のテスト
 */
TEST(FileSystemBucketTreeHeaderDeathTest, ArgumentBoundary)
{
    BucketTree::Header header;

    // totalEntryCount の境界値外
    EXPECT_DEATH_IF_SUPPORTED(header.Format(-1), "");
}
#endif

/**
 * @brief   引数の境界値内のテスト
 */
TEST(FileSystemBucketTreeHeaderTest, ArgumentBoundary)
{
    BucketTree::Header header;

    // totalEntryCount の境界値内
    header.Format(0);
}

/**
 * @brief   BucketTree::Header::Verify() のテスト
 */
TEST(FileSystemBucketTreeHeaderTest, VerifyHeader)
{
    static const int EntryCount = 8;

    BucketTree::Header header;

    // シグネチャ破壊
    header.Format(EntryCount);
    header.signature = 0xFFFFFFFF;
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidBucketTreeSignature, header.Verify());

    // エントリ数があり得ない
    header.Format(EntryCount);
    header.entryCount = -1;
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidBucketTreeEntryCount, header.Verify());

    // バージョン番号破壊
    header.Format(EntryCount);
    header.version = 0xFFFFFFFF;
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultUnsupportedVersion, header.Verify());

    header.Format(EntryCount);

    // 正常終了
    NNT_EXPECT_RESULT_SUCCESS(header.Verify());
}

/**
 * @brief   BucketTree::NodeHeader::Verify() のテスト
 */
TEST(FileSystemBucketTreeHeaderTest, VerifyNodeHeader)
{
    static const int NodeIndex = 8;
    static const int NodeCount = 16;
    static const size_t EntrySize = 16;
    static const size_t NodeSize = NodeCount * EntrySize + sizeof(NodeHeader);
    static const int64_t Offset = 0x00100000;

    BucketTree::NodeHeader header = { NodeIndex, NodeCount, Offset };

    // nodeIndex 不正
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidArgument, header.Verify(NodeIndex - 1, NodeSize, EntrySize));

    // サイズ不正
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidBucketTreeNodeEntryCount,
        header.Verify(NodeIndex, NodeSize - EntrySize, EntrySize)
    );
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidBucketTreeNodeEntryCount,
        header.Verify(NodeIndex, NodeSize, EntrySize * 2)
    );

    // count が不正
    header.count = 0;
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidBucketTreeNodeEntryCount, header.Verify(NodeIndex, NodeSize, EntrySize));

    header.count = NodeCount;

    // offset が不正
    header.offset = -1;
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidBucketTreeNodeOffset, header.Verify(NodeIndex, NodeSize, EntrySize));

    header.offset = Offset;

    // 正常終了
    NNT_EXPECT_RESULT_SUCCESS(header.Verify(NodeIndex, NodeSize, EntrySize));
}
