﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if !defined(NNT_FS_NCA_FILE_SYSTEM_DRIVER_WITH_NCA_FILES)

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_BucketTree.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

TEST(NcaHeaderTest, Verify)
{
    typedef nn::fssystem::NcaHeader Header;

    Header validHeader = {};
    validHeader.signature = Header::Signature;
    validHeader.distributionType = Header::DistributionType::GameCard;
    validHeader.contentType = Header::ContentType::Program;
    validHeader.contentSize = 16;
    validHeader.programId = 1;
    validHeader.sdkAddonVersion = 1;
    validHeader.fsInfo[0].startSector = 1;
    validHeader.fsInfo[0].endSector = 2;
    validHeader.fsInfo[0].hashTargetSize = Header::ByteToSector(nn::fssystem::NcaFsHeader::Size);
    validHeader.fsInfo[1].startSector = 0;
    validHeader.fsInfo[1].endSector = 1;
    validHeader.fsInfo[1].hashTargetSize = Header::ByteToSector(nn::fssystem::NcaFsHeader::Size);
    NNT_ASSERT_RESULT_SUCCESS(validHeader.Verify());

    {
        Header header = validHeader;
        // signature のチェック
        header.signature = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // distributionType のチェック
        header.distributionType = Header::DistributionType(int(Header::DistributionType::GameCard) + 1);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // contentType のチェック
        header.contentType = Header::ContentType(int(Header::ContentType::PublicData) + 1);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // keyIndex のチェック
        header.keyIndex = nn::fssystem::NcaCryptoConfiguration::KeyAreaEncryptionKeyIndexCount;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        // programId のチェック
        Header header = validHeader;
        header.programId = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // sdkAddonVersion のチェック
        header.sdkAddonVersion = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // reserve1 のチェック
        header.reserve1[0] = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // reserve2 のチェック
        header.reserve2[0] = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // fsInfo.hashTargetSize のチェック
        header.fsInfo[0].hashTargetSize = Header::ByteToSector(nn::fssystem::NcaFsHeader::Size) + 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // fsInfo.startSector のチェック
        header.fsInfo[0].startSector = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // fsInfo.endSector のチェック
        header.fsInfo[1].endSector = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
        header.fsInfo[1].endSector = 2;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // fsInfo.reserve のチェック
        header.fsInfo[0].reserve = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // encryptedKey のチェック
        header.encryptedKey[Header::DecryptionKey_Count * nn::fssystem::NcaCryptoConfiguration::Aes128KeySize] = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaHeader, header.Verify());
    }
}

TEST(NcaFsHeaderTest, Verify)
{
    typedef nn::fssystem::NcaFsHeader Header;

    const uint32_t Magic = 0x43465649;
    const uint32_t Version = 0x00020000;

    Header validHeader = {};
    validHeader.version = 2;
    validHeader.hashType = Header::HashType::HierarchicalIntegrityHash;
    validHeader.encryptionType = Header::EncryptionType::AesCtrEx;
    validHeader.hashData.integrityMetaInfo.magic = Magic;
    validHeader.hashData.integrityMetaInfo.version = Version;
    validHeader.hashData.integrityMetaInfo.sizeMasterHash = nn::fssystem::Hash::Size;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.maxLayers = nn::fssystem::save::IntegrityMaxLayerCount;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[0].orderBlock = 4;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[0].offset = 0;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[0].size = 1024;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[1].orderBlock = 4;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[1].offset = 1024 * 1;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[1].size = 1024;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[2].orderBlock = 4;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[2].offset = 1024 * 2;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[2].size = 1024;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[3].orderBlock = 4;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[3].offset = 1024 * 3;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[3].size = 1024;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[4].orderBlock = 4;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[4].offset = 1024 * 4;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[4].size = 1024;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[5].orderBlock = 4;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[5].offset = 1024 * 5;
    validHeader.hashData.integrityMetaInfo.infoLevelHash.info[5].size = 1024;
    validHeader.patchInfo.indirectOffset = 16 * 1024;
    validHeader.patchInfo.indirectSize = 16 * 1024;
    validHeader.patchInfo.aesCtrExOffset = 32 * 1024;
    validHeader.patchInfo.aesCtrExSize = 16 * 1024;
    {
        nn::fssystem::BucketTree::Header header = {};
        header.signature = nn::fssystem::BucketTree::Signature;
        header.version = nn::fssystem::BucketTree::Version;
        header.entryCount = 1;
        std::memcpy(validHeader.patchInfo.indirectHeader, &header, sizeof(header));
        std::memcpy(validHeader.patchInfo.aesCtrExHeader, &header, sizeof(header));
    }
    validHeader.aesCtrUpperIv.part.secureValue = 1;
    validHeader.aesCtrUpperIv.part.generation = 1;
    NNT_ASSERT_RESULT_SUCCESS(validHeader.Verify());

    {
        Header header = validHeader;
        // version のチェック
        header.version = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // fsType のチェック
        header.fsType = Header::FsType(2);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // hashType のチェック
        header.hashType = Header::HashType::Auto;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
        header.hashType = Header::HashType(4);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
        // hashType と hashData の組み合わせチェック
        header.hashType = Header::HashType::None;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // encryptionType のチェック
        header.encryptionType = Header::EncryptionType::Auto;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
        header.encryptionType = Header::EncryptionType(5);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
        // encryptionType と patchInfo の組み合わせチェック
        header.encryptionType = Header::EncryptionType::None;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // reserved のチェック
        header.reserved[0] = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // patchInfo のチェック
        std::memset(&header.patchInfo, 0, sizeof(header.patchInfo));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
        // encryptionType と aesCtrUpperIv の組み合わせチェック
        header.encryptionType = Header::EncryptionType::None;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // reserved2 のチェック
        header.reserved2[0] = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }

    // hashData.integrityMetaInfo のチェック
    {
        Header header = validHeader;
        // magic のチェック
        header.hashData.integrityMetaInfo.magic = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // version のチェック
        header.hashData.integrityMetaInfo.version = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // sizeMasterHash のチェック
        header.hashData.integrityMetaInfo.sizeMasterHash = 4;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // infoLevelHash.maxLayers のチェック
        header.hashData.integrityMetaInfo.infoLevelHash.maxLayers = nn::fssystem::save::IntegrityMinLayerCount - 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
        header.hashData.integrityMetaInfo.infoLevelHash.maxLayers = nn::fssystem::save::IntegrityMaxLayerCount + 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
        // infoLevelHash.maxLayers と infoLevelHash.info の組み合わせチェック
        header.hashData.integrityMetaInfo.infoLevelHash.maxLayers = nn::fssystem::save::IntegrityMaxLayerCount - 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // infoLevelHash.info.orderBlock のチェック
        header.hashData.integrityMetaInfo.infoLevelHash.info[0].orderBlock = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // infoLevelHash.info.offset のチェック
        header.hashData.integrityMetaInfo.infoLevelHash.info[1].offset -= 4;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
        header.hashData.integrityMetaInfo.infoLevelHash.info[1].offset += 8;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // infoLevelHash.info.size のチェック
        header.hashData.integrityMetaInfo.infoLevelHash.info[2].size = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // infoLevelHash.info.reserved のチェック
        header.hashData.integrityMetaInfo.infoLevelHash.info[2].reserved[0] = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // hashData 未設定部分のチェック
        *(reinterpret_cast<char*>(&header.hashData) + sizeof(header.hashData.integrityMetaInfo)) = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }

    std::memset(&validHeader.hashData, 0, sizeof(validHeader.hashData));
    validHeader.hashType = Header::HashType::HierarchicalSha256Hash;
    validHeader.hashData.hierarchicalSha256Data.fsDataMasterHash.value[0] = 1;
    validHeader.hashData.hierarchicalSha256Data.hashBlockSize = 64;
    validHeader.hashData.hierarchicalSha256Data.hashLayerCount = 2;
    validHeader.hashData.hierarchicalSha256Data.hashLayerRegion[0].offset = 0;
    validHeader.hashData.hierarchicalSha256Data.hashLayerRegion[0].size = 1024;
    validHeader.hashData.hierarchicalSha256Data.hashLayerRegion[1].offset = 1024;
    validHeader.hashData.hierarchicalSha256Data.hashLayerRegion[1].size = 1024;
    NNT_ASSERT_RESULT_SUCCESS(validHeader.Verify());

    // hashData.hierarchicalSha256Data のチェック
    {
        Header header = validHeader;
        // fsDataMasterHash のチェック
        std::memset(&header.hashData.hierarchicalSha256Data.fsDataMasterHash, 0, sizeof(header.hashData.hierarchicalSha256Data.fsDataMasterHash));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // hashBlockSize のチェック
        header.hashData.hierarchicalSha256Data.hashBlockSize = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
        header.hashData.hierarchicalSha256Data.hashBlockSize = 3;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // hashLayerCount のチェック
        header.hashData.hierarchicalSha256Data.hashLayerCount = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // hashLayerRegion.offset のチェック
        header.hashData.hierarchicalSha256Data.hashLayerRegion[1].offset -= 4;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // hashLayerRegion.size のチェック
        header.hashData.hierarchicalSha256Data.hashLayerRegion[1].size = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // hashLayerRegion 未設定部分のチェック
        header.hashData.hierarchicalSha256Data.hashLayerRegion[2].offset = 1024 * 2;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
    {
        Header header = validHeader;
        // hashData 未設定部分のチェック
        *(reinterpret_cast<char*>(&header.hashData) + sizeof(header.hashData.hierarchicalSha256Data)) = 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidNcaFsHeader, header.Verify());
    }
} // NOLINT(impl/function_size)

#endif
