﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/fssystem/dbm/fs_DbmPathTool.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

//! ディレクトリ名の長さのテストを行います。
TEST(PathToolTest, TestDirectoryNameLength)
{
    // ディレクトリ名の最大長が 64 文字であることを仮定してテストを行います。
    NN_STATIC_ASSERT(sizeof(nn::fssystem::dbm::DirectoryName) == 64);
    nn::fssystem::dbm::DirectoryName directoryName;

    // 最大長よりも短い文字列に対するテスト
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryName, "1234", 4));
    ASSERT_EQ(
        0,
        std::memcmp(
            &directoryName,
            "1234\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
            sizeof(nn::fssystem::dbm::DirectoryName)
        )
    );
    ASSERT_EQ(4, nn::fssystem::dbm::PathTool::GetDirectoryNameLength(&directoryName));

    // 最大長と等しい長さの文字列に対するテスト
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::GetDirectoryName(
            &directoryName,
            "1234123412341234678967896789678912341234123412346789678967896789",
            64
        )
    );
    ASSERT_EQ(
        0,
        std::memcmp(
            &directoryName,
            "1234123412341234678967896789678912341234123412346789678967896789",
            sizeof(nn::fssystem::dbm::DirectoryName)
        )
    );
    ASSERT_EQ(64, nn::fssystem::dbm::PathTool::GetDirectoryNameLength(&directoryName));

    // 最大長よりも長い文字列に対するテスト
    ASSERT_FALSE(
        nn::fssystem::dbm::PathTool::GetDirectoryName(
            &directoryName,
            "12341234123412346789678967896789123412341234123467896789678967895",
            65
        )
    );
}

//! ファイル名の長さのテストを行います。
TEST(PathToolTest, TestFileNameLength)
{
    // ファイル名の最大長が 64 文字であることを仮定してテストを行います。
    NN_STATIC_ASSERT(sizeof(nn::fssystem::dbm::FileName) == 64);
    nn::fssystem::dbm::FileName fileName;

    // 最大長より短い文字列に対するテスト
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetFileName(&fileName, "1234", 4));
    ASSERT_EQ(
        0,
        std::memcmp(
            &fileName,
            "1234\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
            sizeof(nn::fssystem::dbm::FileName)
        )
    );
    ASSERT_EQ(4, nn::fssystem::dbm::PathTool::GetFileNameLength(&fileName));

    // 最大長と等しい長さの文字列に対するテスト
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::GetFileName(
            &fileName,
            "1234123412341234678967896789678912341234123412346789678967896789",
            64
        )
    );
    ASSERT_EQ(
        0,
        std::memcmp(
            &fileName,
            "1234123412341234678967896789678912341234123412346789678967896789",
            sizeof(nn::fssystem::dbm::FileName)
        )
    );
    ASSERT_EQ(64, nn::fssystem::dbm::PathTool::GetFileNameLength(&fileName));

    // 最大長よりも長い文字列に対するテスト
    ASSERT_FALSE(
        nn::fssystem::dbm::PathTool::GetFileName(
            &fileName,
            "12341234123412346789678967896789123412341234123467896789678967895",
            65
        )
    );
}

//! ディレクトリ名の一致のテストを行います。
TEST(PathToolTest, TestEqualDirectoryName)
{
    nn::fssystem::dbm::DirectoryName directoryName1;
    nn::fssystem::dbm::DirectoryName directoryName2;
    nn::fssystem::dbm::DirectoryName directoryName3;
    nn::fssystem::dbm::DirectoryName directoryName4;

    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryName1, "AAA", 4));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryName2, "AAA", 4));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryName3, "OOO", 4));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryName4, "OOO", 4));
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName1, &directoryName2)
    );
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName2, &directoryName1)
    );
    ASSERT_FALSE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName1, &directoryName3)
    );
    ASSERT_FALSE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName3, &directoryName1)
    );
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName3, &directoryName4)
    );
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName4, &directoryName3)
    );

    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryName1, "12341234", 8));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryName2, "12341234", 8));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryName3, "AAAAAAfo", 8));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryName4, "1234", 4));
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName1, &directoryName2)
    );
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName2, &directoryName1)
    );
    ASSERT_FALSE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName1, &directoryName3)
    );
    ASSERT_FALSE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName3, &directoryName1)
    );
    ASSERT_FALSE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName1, &directoryName4)
    );
    ASSERT_FALSE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName4, &directoryName1)
    );
}

//! ファイル名の一致のテストを行います。
TEST(PathToolTest, TestEqualFileName)
{
    nn::fssystem::dbm::FileName fileName1;
    nn::fssystem::dbm::FileName fileName2;
    nn::fssystem::dbm::FileName fileName3;
    nn::fssystem::dbm::FileName fileName4;

    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetFileName(&fileName1, "AAA", 4));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetFileName(&fileName2, "AAA", 4));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetFileName(&fileName3, "OOO", 4));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetFileName(&fileName4, "OOO", 4));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName1, &fileName2));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName2, &fileName1));
    ASSERT_FALSE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName1, &fileName3));
    ASSERT_FALSE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName3, &fileName1));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName3, &fileName4));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName4, &fileName3));

    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetFileName(&fileName1, "123412341234", 12));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetFileName(&fileName2, "123412341234", 12));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetFileName(&fileName3, "AAAAAAAAAAAA", 4));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetFileName(&fileName4, "12341234", 8));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName1, &fileName2));
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName2, &fileName1));
    ASSERT_FALSE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName1, &fileName3));
    ASSERT_FALSE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName3, &fileName1));
    ASSERT_FALSE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName1, &fileName4));
    ASSERT_FALSE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName4, &fileName1));
}

//! ディレクトリ名がカレントディレクトリを示す文字列かどうかの判定をテストします。
TEST(PathToolTest, TestIsCurrentDirectory)
{
    // カレントディレクトリと判定される名前
    const nn::fssystem::dbm::DirectoryName currentDirectoryName = {"."};
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsCurrentDirectory(&currentDirectoryName, 1));

    // カレントディレクトリと判定されない名前
    const char* directoryNamesChar[] =
    {
        "",
        "/",
        "..",
        "dir",
        "dir1/dir2",
        "dir1/../",
        "1234567890123456789012345678901234567890123456789012345678901234"
    };
    for( const auto& directoryNameChar : directoryNamesChar )
    {
        // char* から DirectoryName にコピーします。
        nn::fssystem::dbm::DirectoryName directoryName;
        nn::fssystem::dbm::PathTool::GetDirectoryName(
            &directoryName, directoryNameChar, std::strlen(directoryNameChar)
        );

        // 判定
        ASSERT_FALSE(
            nn::fssystem::dbm::PathTool::IsCurrentDirectory(
                &directoryName,
                std::strlen(directoryName.name)
            )
        );
    }
}

//! ディレクトリ名が親ディレクトリを示す文字列かどうかの判定をテストします。
TEST(PathToolTest, TestIsParentDirectory)
{
    // 親ディレクトリと判定される名前
    const nn::fssystem::dbm::DirectoryName parentDirectoryName = {".."};
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsParentDirectory(&parentDirectoryName, 2));

    // 親ディレクトリと判定されない名前
    const char* directoryNamesChar[] =
    {
        "",
        "/",
        ".",
        "../",
        "dir",
        "dir1/dir2",
        "dir1/../",
        "1234567890123456789012345678901234567890123456789012345678901234"
    };
    for( const auto& directoryNameChar : directoryNamesChar )
    {
        // char* から DirectoryName にコピーします。
        nn::fssystem::dbm::DirectoryName directoryName;
        nn::fssystem::dbm::PathTool::GetDirectoryName(
            &directoryName, directoryNameChar, std::strlen(directoryNameChar)
        );

        // 判定
        ASSERT_FALSE(
            nn::fssystem::dbm::PathTool::IsParentDirectory(
                &directoryName,
                std::strlen(directoryName.name)
            )
        );
    }
}

//! ディレクトリ名データからファイル名データへの変換をテストします。
TEST(PathToolTest, TestConvertDirectoryNameToFileName)
{
    const char* directoryNamesChar[] =
    {
        "",
        "/",
        ".",
        "../",
        "dir",
        "dir1/dir2",
        "1234567890123456789012345678901234567890123456789012345678901234"
    };
    for( const auto& directoryNameChar : directoryNamesChar )
    {
        // char* から DirectoryName にコピーします。
        nn::fssystem::dbm::DirectoryName directoryName;
        ASSERT_TRUE(
            nn::fssystem::dbm::PathTool::GetDirectoryName(
                &directoryName, directoryNameChar, std::strlen(directoryNameChar)
            )
        );

        // 変換
        nn::fssystem::dbm::FileName fileName;
        ASSERT_TRUE(
            nn::fssystem::dbm::PathTool::ConvertDirectoryNameToFileName(
                &fileName,
                &directoryName
            )
        );

        // 変換後の内容を確認します。
        // ディレクトリ名文字列を直接コピーして FileName を作ります。
        nn::fssystem::dbm::FileName fileNamePart;
        ASSERT_TRUE(
            nn::fssystem::dbm::PathTool::GetFileName(
                &fileNamePart, directoryNameChar, std::strlen(directoryNameChar)
            )
        );
        // 二つの FileName が一致することをテストします。
        ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileNamePart, &fileName));
    }
}

//! ファイル名データからディレクトリ名データへの変換をテストします。
TEST(PathToolTest, TestConverFileNameToDirectoryName)
{
    const char* fileNamesChar[] =
    {
        "",
        "./file",
        "../file",
        "file",
        "dir1/file",
        "1234567890123456789012345678901234567890123456789012345678901234"
    };
    for( const auto& fileNameChar : fileNamesChar )
    {
        // char* から FileName にコピーします。
        nn::fssystem::dbm::FileName fileName;
        nn::fssystem::dbm::PathTool::GetFileName(
            &fileName, fileNameChar, std::strlen(fileNameChar)
        );

        // 変換
        nn::fssystem::dbm::DirectoryName directoryName;
        ASSERT_TRUE(
            nn::fssystem::dbm::PathTool::ConvertFileNameToDirectoryName(
                &directoryName,
                &fileName
            )
        );

        // 変換後の内容を確認します。
        // ファイル名文字列を直接コピーして DirectoryName を作ります。
        nn::fssystem::dbm::DirectoryName directoryNamePart;
        ASSERT_TRUE(
            nn::fssystem::dbm::PathTool::GetDirectoryName(
                &directoryNamePart, fileNameChar, std::strlen(fileNameChar)
            )
        );
        // 二つの FileName が一致することをテストします。
        ASSERT_TRUE(
            nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryNamePart, &directoryName)
        );
    }
}

//! PathParser で不正なパスを解析するテストです。
TEST(PathParserTest, TestParseInvalidPath)
{
    // 先頭が区切り文字でない文字列は解析できません。
    nn::fssystem::dbm::PathTool::PathParser parser;
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidPathFormat, parser.Initialize("1234"));
    parser.Finalize();
}

// ルートディレクトリに対する解析のテストです。
TEST(PathParserTest, TestParseRootDirectory)
{
    nn::fssystem::dbm::PathTool::PathParser parser;
    NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("/"));

    nn::fssystem::dbm::DirectoryName directoryName;
    // 出力用引数が書き換わることが明らかとなるように初期化します。
    size_t directoryNameLength = 1;
    NNT_ASSERT_RESULT_SUCCESS(parser.GetNextDirectoryName(&directoryName, &directoryNameLength));

    // ルートディレクトリに対して行った場合は文字列長は 0 です。
    ASSERT_EQ(0, directoryNameLength);

    // パース完了であることをチェックします。
    ASSERT_TRUE(parser.IsParseFinished());

    parser.Finalize();
}

// 区切り文字が連続するパスへのテストです。
TEST(PathParserTest, TestParseDoubleSeparator)
{
    nn::fssystem::dbm::PathTool::PathParser parser;
    NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("//"));

    nn::fssystem::dbm::DirectoryName directoryName;
    // 出力用引数が書き換わることが明らかとなるように初期化します。
    size_t directoryNameLength = 1;
    NNT_ASSERT_RESULT_SUCCESS(parser.GetNextDirectoryName(&directoryName, &directoryNameLength));

    // ルートディレクトリに対して行った場合は文字列長は 0 です。
    ASSERT_EQ(0, directoryNameLength);

    // パース完了であることをチェックします。
    ASSERT_TRUE(parser.IsParseFinished());

    parser.Finalize();
}

// 1 階層のパスに対するテストです。
TEST(PathParserTest, TestParseOneDirectory)
{
    nn::fssystem::dbm::PathTool::PathParser parser;
    NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("/1234"));

    nn::fssystem::dbm::DirectoryName directoryName;
    // 出力用引数が書き換わることが明らかとなるように初期化します。
    size_t directoryNameLength = 1;
    NNT_ASSERT_RESULT_SUCCESS(parser.GetNextDirectoryName(&directoryName, &directoryNameLength));

    // ルートディレクトリに対して行った場合は文字列長は 0 です。
    ASSERT_EQ(0, directoryNameLength);

    // パース完了であることをチェックします。
    ASSERT_TRUE(parser.IsParseFinished());

    // 最後のディレクトリ部分を取得します。
    NNT_ASSERT_RESULT_SUCCESS(parser.GetAsDirectoryName(&directoryName, &directoryNameLength));

    // 取得したディレクトリ名をチェックします。
    nn::fssystem::dbm::DirectoryName directoryNamePart;
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryNamePart, "1234", 4));
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName, &directoryNamePart)
    );
    ASSERT_EQ(4, directoryNameLength);

    parser.Finalize();
}

// 1 階層 + 最後が区切り文字のパスに対するテストです。
TEST(PathParserTest, TestParseOneDirecotryAndSeparator)
{
    nn::fssystem::dbm::PathTool::PathParser parser;
    NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("/1234/"));

    nn::fssystem::dbm::DirectoryName directoryName;
    // 出力用引数が書き換わることが明らかとなるように初期化します。
    size_t directoryNameLength = 1;
    NNT_ASSERT_RESULT_SUCCESS(parser.GetNextDirectoryName(&directoryName, &directoryNameLength));

    // ルートディレクトリに対して行った場合は文字列長は 0 です。
    ASSERT_EQ(0, directoryNameLength);

    NNT_ASSERT_RESULT_SUCCESS(parser.GetAsDirectoryName(&directoryName, &directoryNameLength));

    // パース完了であることをチェックします。
    ASSERT_TRUE(parser.IsParseFinished());

    // 取得したディレクトリ名をチェックします。
    nn::fssystem::dbm::DirectoryName directoryNamePart;
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryNamePart, "1234", 4));
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName, &directoryNamePart)
    );
    ASSERT_EQ(4, directoryNameLength);

    // パスの最後が区切り文字であったことをチェックします。
    ASSERT_TRUE(parser.IsDirectoryPath());

    parser.Finalize();
}

// 2 階層のパスに対するテストです。
TEST(PathParserTest, TestParseTwoDirectory)
{
    nn::fssystem::dbm::PathTool::PathParser parser;
    NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("/1234/AAA"));

    nn::fssystem::dbm::DirectoryName directoryName;
    // 出力用引数が書き換わることが明らかとなるように初期化します。
    size_t directoryNameLength = 1;
    NNT_ASSERT_RESULT_SUCCESS(parser.GetNextDirectoryName(&directoryName, &directoryNameLength));

    // ルートディレクトリに対して行った場合は文字列長は 0 です。
    ASSERT_EQ(0, directoryNameLength);

    directoryNameLength = 0;
    NNT_ASSERT_RESULT_SUCCESS(parser.GetNextDirectoryName(&directoryName, &directoryNameLength));

    // 取得したディレクトリ名をチェックします。
    nn::fssystem::dbm::DirectoryName directoryNamePart;
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryNamePart, "1234", 4));
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName, &directoryNamePart)
    );
    ASSERT_EQ(4, directoryNameLength);

    // パース完了であることをチェックします。
    ASSERT_TRUE(parser.IsParseFinished());

    // 最後のディレクトリ部分を取得します。
    NNT_ASSERT_RESULT_SUCCESS(parser.GetAsDirectoryName(&directoryName, &directoryNameLength));

    // 取得したディレクトリ名をチェックします。
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryNamePart, "AAA", 3));
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName, &directoryNamePart)
    );
    ASSERT_EQ(3, directoryNameLength);

    parser.Finalize();
}

// 2 階層のパスに対するテストです。(連続スラッシュ)
TEST(PathParserTest, TestParseTwoDirectoryAndSeparators)
{
    nn::fssystem::dbm::PathTool::PathParser parser;
    NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("////1234////AAA//////"));

    nn::fssystem::dbm::DirectoryName directoryName;
    // 出力用引数が書き換わることが明らかとなるように初期化します。
    size_t directoryNameLength = 1;
    NNT_ASSERT_RESULT_SUCCESS(parser.GetNextDirectoryName(&directoryName, &directoryNameLength));

    // ルートディレクトリに対して行った場合は文字列長は 0 です。
    ASSERT_EQ(0, directoryNameLength);

    directoryNameLength = 0;
    NNT_ASSERT_RESULT_SUCCESS(parser.GetNextDirectoryName(&directoryName, &directoryNameLength));

    // 取得したディレクトリ名をチェックします。
    nn::fssystem::dbm::DirectoryName directoryNamePart;
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryNamePart, "1234", 4));
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName, &directoryNamePart)
    );
    ASSERT_EQ(4, directoryNameLength);

    // パース完了であることをチェックします。
    ASSERT_TRUE(parser.IsParseFinished());

    // 最後のディレクトリ部分を取得します。
    NNT_ASSERT_RESULT_SUCCESS(parser.GetAsDirectoryName(&directoryName, &directoryNameLength));

    // 取得したディレクトリ名をチェックします。
    ASSERT_TRUE(nn::fssystem::dbm::PathTool::GetDirectoryName(&directoryNamePart, "AAA", 3));
    ASSERT_TRUE(
        nn::fssystem::dbm::PathTool::IsEqualDirectoryName(&directoryName, &directoryNamePart)
    );
    ASSERT_EQ(3, directoryNameLength);

    parser.Finalize();
}

//! ファイル名の取得成功をテストします。
TEST(PathParserTest, TestGetAsFileNameSuccess)
{
    // 通常のファイル名取得
    {
        // ファイル名を含むパスで初期化します。
        nn::fssystem::dbm::PathTool::PathParser parser;
        NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("/file"));

        // ルートディレクトリの名前を取得します（文字列長は 0 です）。
        nn::fssystem::dbm::DirectoryName directoryName;
        // 出力用引数が書き換わることが明らかとなるように初期化します。
        size_t directoryNameLength = 1;
        NNT_ASSERT_RESULT_SUCCESS(
            parser.GetNextDirectoryName(&directoryName, &directoryNameLength)
            );
        ASSERT_EQ(0, directoryNameLength);

        // ファイル名を取得します。
        nn::fssystem::dbm::FileName fileName;
        size_t fileNameLength;
        NNT_ASSERT_RESULT_SUCCESS(parser.GetAsFileName(&fileName, &fileNameLength));

        // ファイル名を照合します。
        nn::fssystem::dbm::FileName fileNamePart;
        nn::fssystem::dbm::PathTool::GetFileName(&fileNamePart, "file", 4);
        ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName, &fileNamePart));

        // ファイル名の文字列長を照合します。
        ASSERT_EQ(4, fileNameLength);
    }

    // 長いファイル名取得
    {
        // 長いファイル名を含むパスで初期化します。
        nn::fssystem::dbm::PathTool::PathParser parser;
        NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("/1234567890123456789012345678901234567890123456789012345678901234"));

        // ルートディレクトリの名前を取得します。
        nn::fssystem::dbm::DirectoryName directoryName;
        size_t directoryNameLength = 1;
        NNT_ASSERT_RESULT_SUCCESS(
            parser.GetNextDirectoryName(&directoryName, &directoryNameLength)
        );
        ASSERT_EQ(0, directoryNameLength);

        // ファイル名を取得します。
        nn::fssystem::dbm::FileName fileName;
        size_t fileNameLength;
        NNT_ASSERT_RESULT_SUCCESS(parser.GetAsFileName(&fileName, &fileNameLength));

        // ファイル名を照合します。
        nn::fssystem::dbm::FileName fileNamePart;
        nn::fssystem::dbm::PathTool::GetFileName(
            &fileNamePart,
            "1234567890123456789012345678901234567890123456789012345678901234",
            64
        );
        ASSERT_TRUE(nn::fssystem::dbm::PathTool::IsEqualFileName(&fileName, &fileNamePart));

        // ファイル名の文字列長を照合します。
        ASSERT_EQ(64, fileNameLength);
    }
}

//! ファイル名の取得失敗をテストします。
TEST(PathParserTest, TestGetAsFileNameFailure)
{
    // 前提条件のテスト
    {
        nn::fssystem::dbm::PathTool::PathParser parser;

        // 初期化している必要があります。
        nn::fssystem::dbm::FileName fileName;
        size_t fileNameLength;
        EXPECT_DEATH_IF_SUPPORTED(parser.GetAsFileName(&fileName, &fileNameLength), "");

        // 引数が正しく設定されている必要があります。
        EXPECT_DEATH_IF_SUPPORTED(parser.GetAsFileName(nullptr, &fileNameLength), "");
        EXPECT_DEATH_IF_SUPPORTED(parser.GetAsFileName(&fileName, nullptr), "");
    }

    // ファイル名が長すぎる場合
    {
        // 長すぎるファイル名を含むパスで初期化します。
        nn::fssystem::dbm::PathTool::PathParser parser;
        NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("/12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345"));

        // ルートディレクトリの名前を取得します。
        nn::fssystem::dbm::DirectoryName directoryName;
        // 出力用引数が書き換わることが明らかとなるように初期化します。
        size_t directoryNameLength = 1;
        NNT_ASSERT_RESULT_SUCCESS(
            parser.GetNextDirectoryName(&directoryName, &directoryNameLength)
            );
        ASSERT_EQ(0, directoryNameLength);

        // ファイル名が長すぎるため取得に失敗します。
        nn::fssystem::dbm::FileName fileName;
        size_t fileNameLength;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultTooLongPath,
            parser.GetAsFileName(&fileName, &fileNameLength)
        );
    }

    // パスの最後が "." の場合
    {
        // "." で終わるパスで初期化します。
        nn::fssystem::dbm::PathTool::PathParser parser;
        NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("/."));

        // ルートディレクトリの名前を取得します。
        nn::fssystem::dbm::DirectoryName directoryName;
        size_t directoryNameLength = 1;
        NNT_ASSERT_RESULT_SUCCESS(
            parser.GetNextDirectoryName(&directoryName, &directoryNameLength)
            );
        ASSERT_EQ(0, directoryNameLength);

        // パスの最後が不正なため取得に失敗します。
        nn::fssystem::dbm::FileName fileName;
        size_t fileNameLength;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidPathFormat,
            parser.GetAsFileName(&fileName, &fileNameLength)
        );
    }

    // パスの最後が ".." の場合
    {
        // ".." で終わるパスで初期化します。
        nn::fssystem::dbm::PathTool::PathParser parser;
        NNT_ASSERT_RESULT_SUCCESS(parser.Initialize("/.."));

        // ルートディレクトリの名前を取得します。
        nn::fssystem::dbm::DirectoryName directoryName;
        size_t directoryNameLength = 1;
        NNT_ASSERT_RESULT_SUCCESS(
            parser.GetNextDirectoryName(&directoryName, &directoryNameLength)
            );
        ASSERT_EQ(0, directoryNameLength);

        // パスの最後が不正なため取得に失敗します。
        nn::fssystem::dbm::FileName fileName;
        size_t fileNameLength;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidPathFormat,
            parser.GetAsFileName(&fileName, &fileNameLength)
        );
    }
}
