﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <deque>
#include <numeric>
#include <random>

#include <nn/os/os_Thread.h>

#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_WriteThroughCacheStorage.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

//! ブロックサイズ(2のべき乗)から、その指数を取得する
int GetSizeExp(size_t blockSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(nn::util::ispow2(blockSize));
    return nn::util::cntt0(blockSize);
}

//! ランダムなブロックサイズを取得します。2のべき乗が出力されます。
size_t GenerateRandomBlockSize(size_t blockSizeMin, size_t blockSizeMax) NN_NOEXCEPT
{
    static std::mt19937 s_Mt;
    int min = GetSizeExp(blockSizeMin);
    int max = GetSizeExp(blockSizeMax);
    int random = std::uniform_int_distribution<>(0, max - min)(s_Mt) + min;
    return static_cast<size_t>(1) << random;
}

// ストレージの初期化・書き込みを行う簡単なテスト
TEST(FsWriteThroughCacheStorageTest, Simple)
{
    static const size_t TestSize = 1024 * 1024;
    static const size_t BufferCacheSize = 2 * 1024;
    static const int BufferCacheCount = 4;

    nnt::fs::util::Vector<char> buf1(TestSize);
    nnt::fs::util::Vector<char> buf2(TestSize);

    // テストデータ設定
    for( size_t i = 0; i < TestSize; ++i )
    {
        buf1[i] = i & 0xFF;
    }

    // 書き込みサイズを変えつつテスト
    for( size_t i = 0; i < TestSize; )
    {
        nnt::fs::util::SafeMemoryStorage storageMemory(i);

        std::memset(&buf2[0], 0x55, TestSize);
        NNT_ASSERT_RESULT_SUCCESS(storageMemory.Write(0, &buf2[0], i));

        nn::fssystem::WriteThroughCacheStorage storage;
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&storageMemory, 0, i),
                BufferCacheSize,
                BufferCacheCount
            )
        );

        // バイト単位での書き込み、戻り値確認
        for( size_t j = 0; j < i; ++j )
        {
            NNT_ASSERT_RESULT_SUCCESS(storage.Write(j, &buf1[j], 1));
        }

        // 範囲外には書き込めない
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset,
            storage.Write(i, &buf1[0], 1)
        );

        // データ比較
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, &buf2[0], i));

        ASSERT_EQ(0, std::memcmp(&buf1[0], &buf2[0], i));
        ASSERT_EQ(0x55, buf2[i]);

        storage.Flush();

        if( i == 0 )
        {
            i = 1;
        }
        else
        {
            i *= 2;
        }
    }
}

// ストレージの読み書きのテスト
TEST(FsWriteThroughCacheStorageTest, ReadWrite)
{
    static const size_t TestSize = 1024 * 1024;
    static const int BufferCacheCount = 64;
    static const size_t CacheSizeMin = 64;
    static const size_t CacheSizeMax = 512;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> buf1(TestSize);
    nnt::fs::util::Vector<char> buf2(TestSize);

    // 読み書きバッファのサイズを変えつつテストを行う
    {
        for( size_t sizeTest = 1; sizeTest <= TestSize; sizeTest *= 2 )
        {
            // テストバッファに初期値を書き込み
            std::memset(&buf1[0], 0, TestSize);

            // キャッシュをセットアップ
            size_t bufferCacheSize = GenerateRandomBlockSize(CacheSizeMin, CacheSizeMax);

            nnt::fs::util::SafeMemoryStorage storageMemory(sizeTest);
            NNT_ASSERT_RESULT_SUCCESS(storageMemory.Write(0, &buf1[0], sizeTest));

            nn::fssystem::WriteThroughCacheStorage storage;
            NNT_ASSERT_RESULT_SUCCESS(
                storage.Initialize(
                    nnt::fs::util::GetTestLibraryAllocator(),
                    nn::fs::SubStorage(&storageMemory, 0, sizeTest),
                    bufferCacheSize,
                    BufferCacheCount
                )
            );

            // ランダムな位置にデータを書き、内容を比較
            for( int loop = 1000; 0 < loop; --loop )
            {
                // テストデータを書き込みます。
                size_t offset = std::uniform_int_distribution<size_t>(0, sizeTest - 1)(mt);
                size_t size = std::uniform_int_distribution<size_t>(0, sizeTest - offset)(mt);
                if( size == 0 )
                {
                    continue;
                }

                // 二つのバッファにそれぞれ書き込み
                std::memset(&buf2[0], loop & 0xFF, size);
                NNT_ASSERT_RESULT_SUCCESS(storage.Write(offset, &buf2[0], size));
                std::memcpy(&buf1[offset], &buf2[0], size);

                NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, &buf2[0], size));
                ASSERT_EQ(0, std::memcmp(&buf1[offset], &buf2[0], size));

                // 一定周期でデータ比較
                if( (loop == 1) || ((loop % 10) == 0) )
                {
                    NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, &buf2[0], sizeTest));
                    ASSERT_EQ(0, std::memcmp(&buf1[0], &buf2[0], sizeTest));
                }
            }

            storage.Flush();
        }
    }
}

// 読み書きでキャッシュが意図通りの動作をしているかテスト
TEST(FsWriteThroughCacheStorageTest, UseCaseAccess)
{
    static const int BufCount = 16;
    static const size_t MemorySize = 512 * 1024;
    static const size_t CacheSize = 16 * 1024;

    nnt::fs::util::AccessCountedMemoryStorage storageMemory;
    storageMemory.Initialize(MemorySize);

    nnt::fs::util::Vector<char> buffer(MemorySize);
    nn::Result result;

    // 先頭、サイズともにキャッシュサイズ境界 (Read)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize * 4, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // キャッシュされていることを確認する
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize * 4, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭、サイズともにキャッシュサイズ境界 (Write)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回書き込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(1, storageMemory.GetWriteTimes());

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize * 4, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // フェッチが発生することを確認する
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize * 4, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭、サイズともにキャッシュサイズ境界 (キャッシュ済み、Read)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // すでにキャッシュされている
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭、サイズともにキャッシュサイズ境界 (キャッシュ済み、Write)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // ちょうど 1 回書き込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(1, storageMemory.GetWriteTimes());

        // キャッシュ済みなので読み込みは行われない
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭がキャッシュサイズ境界 (Read)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 先頭ブロックはキャッシュされている
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize / 2, buffer.data(), CacheSize / 2);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭はキャッシュサイズ境界 (Write)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回書き込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(1, storageMemory.GetWriteTimes());

        // キャッシュされてないことを確認する
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭がキャッシュサイズ境界 (キャッシュ済み、Read)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 先頭ブロックはキャッシュされている
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭はキャッシュサイズ境界 (キャッシュ済み、Write)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // ちょうど 1 回書き込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(1, storageMemory.GetWriteTimes());

        // 読み込みは発生せず、キャッシュに書き込みが反映されている
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭、サイズともにキャッシュサイズ境界でない (Read)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize / 4, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 先頭ブロックはキャッシュされている
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭、サイズともにキャッシュサイズ境界でない (Write)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // 読み書き 1 回ずつ行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(CacheSize / 4, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(1, storageMemory.GetWriteTimes());

        // 先頭ブロックはキャッシュされている
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭がキャッシュサイズ境界ではなく、サイズが比較的大きい場合 (Read)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // 先頭ブロックから一度で読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize / 3, buffer.data(), CacheSize * 3);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 先頭ブロックはキャッシュされていない
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 次のブロックもキャッシュされていない
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭がキャッシュサイズ境界ではなく、サイズが比較的大きい場合 (Write)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // 先頭ブロックとそれ以降で 2 回書き込みが行われる
        // また先頭ブロックのキャッシュが発生する
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(CacheSize / 3, buffer.data(), CacheSize * 3);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(2, storageMemory.GetWriteTimes());

        // 先頭ブロックはキャッシュされている
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 次のブロックはキャッシュされていない
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭がキャッシュサイズ境界ではなく、サイズが比較的大きい場合 (キャッシュ済み、Read)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize * 3, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // キャッシュ済みの 2 ブロックを跨いで読み込みを行う
        // 先頭ブロック以降が 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize / 3, buffer.data(), CacheSize * 3);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 先頭ブロックはキャッシュされている
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 次のブロックはキャッシュされていない
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 先頭がキャッシュサイズ境界ではなく、サイズが比較的大きい場合 (キャッシュ済み、Write)
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // ちょうど 1 回読み込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize * 3, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // キャッシュ済みの 2 ブロックを跨いで書き込みを行う
        // 先頭ブロックとそれ以降で 2 回書き込みが行われる
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(CacheSize / 3, buffer.data(), CacheSize * 3);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(2, storageMemory.GetWriteTimes());

        // 先頭ブロックはキャッシュされている
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 次のブロックはキャッシュされていない
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize, buffer.data(), CacheSize / 4);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 末端のブロックはキャッシュされている
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize * 3, buffer.data(), CacheSize);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 読み取りに対して LRU として動作することを確認する
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // BufCount 個数分キャッシュに充填する
        storageMemory.ResetAccessCounter();
        for( int i = 0; i < BufCount; ++i )
        {
            result = storageCache.Read(CacheSize * i, buffer.data(), 1);
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        // ちょうど BufCount 回読み込みが行われる
        EXPECT_EQ(BufCount, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 再度 BufCount 個数分読み取る
        storageMemory.ResetAccessCounter();
        for( int i = 0; i < BufCount; ++i )
        {
            result = storageCache.Read(CacheSize * i, buffer.data(), 1);
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        // 読み取りは発生しない
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // キャッシュ範囲外にアクセスした後、先頭から読み直す
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(CacheSize * BufCount, buffer.data(), 1);
        NNT_ASSERT_RESULT_SUCCESS(result);
        for( int i = 0; i < BufCount; ++i )
        {
            result = storageCache.Read(CacheSize * i, buffer.data(), 1);
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        // BufCount + 1 回読み取りが発生する
        EXPECT_EQ(BufCount + 1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());

        // 先頭、範囲外、先頭、とアクセスしたときは
        // リードアクセスが 1 回だけ発生する
        storageMemory.ResetAccessCounter();
        result = storageCache.Read(0 * BufCount, buffer.data(), 1);
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = storageCache.Read(CacheSize * BufCount, buffer.data(), 1);
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = storageCache.Read(0 * BufCount, buffer.data(), 1);
        NNT_ASSERT_RESULT_SUCCESS(result);

        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(0, storageMemory.GetWriteTimes());
    }

    // 書き込みに対して LRU として動作することを確認する
    {
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            CacheSize,
            BufCount
        );

        // BufCount 個数分書き込む
        storageMemory.ResetAccessCounter();
        for( int i = 0; i < BufCount; ++i )
        {
            result = storageCache.Write(1 + CacheSize * i, buffer.data(), 1);
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        // ちょうど BufCount 回読み書き込みが行われる
        EXPECT_EQ(BufCount, storageMemory.GetReadTimes());
        EXPECT_EQ(BufCount, storageMemory.GetWriteTimes());

        // 再度 BufCount 個数分書き込む
        storageMemory.ResetAccessCounter();
        for( int i = 0; i < BufCount; i++ )
        {
            result = storageCache.Write(1 + CacheSize * i, buffer.data(), 1);
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        // 読みは発生しない、書きは BufCount 回発生する
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(BufCount, storageMemory.GetWriteTimes());

        // キャッシュ範囲外にアクセスした後、先頭から書き直す
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(1 + CacheSize * BufCount, buffer.data(), 1);
        NNT_ASSERT_RESULT_SUCCESS(result);
        for( int i = 0; i < BufCount; i++ )
        {
            result = storageCache.Write(1 + CacheSize * i, buffer.data(), 1);
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        // それぞれ BufCount + 1 回読み書きされる
        EXPECT_EQ(BufCount + 1, storageMemory.GetReadTimes());
        EXPECT_EQ(BufCount + 1, storageMemory.GetWriteTimes());

        // 先頭、範囲外、先頭、とアクセスしたときは
        // リードアクセスが 1 回だけ発生する
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(1 + 0, buffer.data(), 1);
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = storageCache.Write(1 + CacheSize * BufCount, buffer.data(), 1);
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = storageCache.Write(1 + 0, buffer.data(), 1);
        NNT_ASSERT_RESULT_SUCCESS(result);

        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(3, storageMemory.GetWriteTimes());

        // 連続書き込み時に先頭に回ることを確認する
        for( int i = 0; i < BufCount; i++ )
        {
            result = storageCache.Write(1 + CacheSize * i, buffer.data(), 1);
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        // ライトが 3 回、リードが 1 回発生する
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(1 + 0, buffer.data(), CacheSize * 3 );
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = storageCache.Write(1 + CacheSize * BufCount, buffer.data(), 1);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
        EXPECT_EQ(3, storageMemory.GetWriteTimes());

        // リードは発生しない
        storageMemory.ResetAccessCounter();
        result = storageCache.Write(1 + 0, buffer.data(), 1);
        NNT_ASSERT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, storageMemory.GetReadTimes());
        EXPECT_EQ(1, storageMemory.GetWriteTimes());
    }
} // NOLINT(impl/function_size)

// フラッシュのテスト
TEST(FsWriteThroughCacheStorageTest, Flush)
{
    static const size_t SizeTest = 8 * 1024;
    static const int BufferCacheCount = 4;
    static const size_t BufferCacheSize = 1 * 1024;

    nnt::fs::util::Vector<char> buf1(SizeTest);
    char* ptr1 = &buf1[0];

    // テストデータ設定
    for( size_t i = 0; i < SizeTest; ++i )
    {
        buf1[i] = i & 0xFF;
    }

    // 実ストレージを生成する
    nnt::fs::util::AccessCountedMemoryStorage storageMemory;
    storageMemory.Initialize(SizeTest);
    NNT_ASSERT_RESULT_SUCCESS(storageMemory.Write(0, ptr1, SizeTest));

    // キャッシュを初期化する
    nn::fssystem::WriteThroughCacheStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, SizeTest),
            BufferCacheSize,
            BufferCacheCount
        )
    );

    // アクセス回数リセット
    storageMemory.ResetAccessCounter();

    // 範囲内へのデータ書き込み
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, ptr1, SizeTest / 2));
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(SizeTest / 2, ptr1, SizeTest / 2));
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, ptr1, SizeTest));

    // 書き込みが発生している
    ASSERT_EQ(3, storageMemory.GetWriteTimes());
    ASSERT_EQ(0, storageMemory.GetFlushTimes());

    // アクセス回数リセット
    storageMemory.ResetAccessCounter();

    // フラッシュは発生するが書き込みは発生しない
    NNT_ASSERT_RESULT_SUCCESS(storage.Flush());
    ASSERT_EQ(0, storageMemory.GetWriteTimes());
    ASSERT_EQ(1, storageMemory.GetFlushTimes());

    // アクセス回数リセット
    storageMemory.ResetAccessCounter();

    // 範囲外へのアクセス
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        storage.Write(24 * 1024 + 1, ptr1, 8 * 1024)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        storage.Write(32 * 1024, ptr1, 8 * 1024)
    );

    ASSERT_EQ(0, storageMemory.GetWriteTimes());
    ASSERT_EQ(0, storageMemory.GetFlushTimes());
}

// 小さな書き込みとフラッシュとベリファイのテスト
TEST(FsWriteThroughCacheStorageTest, FlushVerifyMini)
{
    static const int BufCount = 2;
    static const size_t CacheSizeMin = 64;
    static const size_t CacheSizeMax = 512;
    static const size_t MemorySizeMin = 1024;
    static const size_t MemorySizeMax = 16 * 1024;

    // アラインされたアドレスに対してキャッシュサイズ単位でのデータ書き込みを行う
    for( size_t memorySize = MemorySizeMin; memorySize <= MemorySizeMax; memorySize *= 4 )
    {
        for( size_t cacheSize = CacheSizeMin; cacheSize <= CacheSizeMax; cacheSize *= 4 )
        {
            // 実ストレージを生成する
            nnt::fs::util::AccessCountedMemoryStorage storageMemory;
            storageMemory.Initialize(memorySize);

            // キャッシュを初期化する
            nn::fssystem::WriteThroughCacheStorage storageCache;
            storageCache.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&storageMemory, 0, memorySize),
                cacheSize,
                BufCount
            );

            nnt::fs::util::Vector<char> readBuffer(cacheSize - 1, 0);

            // 書き込むデータを生成する
            nnt::fs::util::Vector<char> writeBuffer(cacheSize - 1);
            std::iota(writeBuffer.begin(), writeBuffer.end(), '\x1');

            // ストレージを 0 埋めする
            nnt::fs::util::Vector<char> dummyBuffer(memorySize - 1, 0);
            nn::Result result = storageMemory.Write(0, dummyBuffer.data(), dummyBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);

            // データを書き込む
            storageMemory.ResetAccessCounter();
            result = storageCache.Write(0, writeBuffer.data(), writeBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            EXPECT_EQ(1, storageMemory.GetWriteTimes());
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // データを読み込む
            storageMemory.ResetAccessCounter();
            result = storageCache.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
            EXPECT_EQ(1, storageMemory.GetReadTimes());

            // 元ファイルに書き込みが反映されていることを確認する
            result = storageMemory.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));

            // キャッシュされている範囲へのリード
            storageMemory.ResetAccessCounter();
            result = storageCache.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // 書き込むデータを更新する
            std::iota(writeBuffer.begin(), writeBuffer.end(), '\x2');

            // キャッシュされている領域に対して書き込みを行う
            storageMemory.ResetAccessCounter();
            result = storageCache.Write(0, writeBuffer.data(), writeBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            EXPECT_EQ(1, storageMemory.GetWriteTimes());
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // ここで書き込んだデータのベリファイを行う
            result = storageMemory.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));

            // キャッシュに対しても同様にベリファイする
            result = storageCache.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));

            // フラッシュ時に読み書きは発生しない
            storageMemory.ResetAccessCounter();
            result = storageCache.Flush();
            NNT_ASSERT_RESULT_SUCCESS(result);
            EXPECT_EQ(0, storageMemory.GetReadTimes());
            EXPECT_EQ(0, storageMemory.GetWriteTimes());
        }
    }
}

// 閾値を超えるサイズ書き込みとフラッシュとベリファイのテスト
TEST(FsWriteThroughCacheStorageTest, FlushVerifyThreshold)
{
    static const int BufCount = 2;
    static const size_t CacheSizeMin = 64;
    static const size_t CacheSizeMax = 512;
    static const size_t MemorySizeMin = 1024;
    static const size_t MemorySizeMax = 16 * 1024;

    // アラインされたアドレスに対してキャッシュサイズを超えるデータ書き込みを行う
    for( size_t memorySize = MemorySizeMin; memorySize <= MemorySizeMax; memorySize *= 4 )
    {
        for( size_t cacheSize = CacheSizeMin; cacheSize <= CacheSizeMax; cacheSize *= 4 )
        {
            // 実ストレージを生成する
            nnt::fs::util::AccessCountedMemoryStorage storageMemory;
            storageMemory.Initialize(memorySize);

            // キャッシュを初期化する
            nn::fssystem::WriteThroughCacheStorage storageCache;
            storageCache.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&storageMemory, 0, memorySize),
                cacheSize,
                BufCount
            );

            nnt::fs::util::Vector<char> readBuffer(memorySize - 1, 0);

            // 書き込むデータを生成する
            nnt::fs::util::Vector<char> writeBuffer(memorySize - 1);
            std::iota(writeBuffer.begin(), writeBuffer.end(), '\x1');

            // ストレージを 0 埋めする
            nn::Result result = storageMemory.Write(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);

            // データを書き込む
            // 実ストレージに直接書き込みが発生する
            // 書き込みサイズがブロックサイズ以下ならフェッチは発生しない
            storageMemory.ResetAccessCounter();
            result = storageCache.Write(0, writeBuffer.data(), writeBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            EXPECT_EQ(1, storageMemory.GetWriteTimes());
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // ブロックを跨いでデータを読み込む
            // 先頭ブロックと次のブロックを一度でアクセスする
            storageMemory.ResetAccessCounter();
            result = storageCache.Read(3, readBuffer.data() + 3, cacheSize);
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin() + 3, writeBuffer.begin() + 3 + cacheSize, readBuffer.begin() + 3));
            EXPECT_EQ(1, storageMemory.GetReadTimes());

            // 元ファイルに書き込みが反映されていることを確認する
            result = storageMemory.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));

            // 全領域はキャッシュされていないので実ストレージへのリードが発生する
            storageMemory.ResetAccessCounter();
            result = storageCache.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
            EXPECT_EQ(1, storageMemory.GetReadTimes());

            // 書き込むデータを更新する
            std::iota(writeBuffer.begin(), writeBuffer.end(), '\x2');

            // キャッシュされている領域を超えて書き込みを行う
            storageMemory.ResetAccessCounter();
            result = storageCache.Write(0, writeBuffer.data(), writeBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            EXPECT_EQ(1, storageMemory.GetWriteTimes());
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // ここで書き込んだデータのベリファイを行う
            result = storageMemory.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));

            // キャッシュに対しても同様にベリファイする
            result = storageCache.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));

            // フラッシュ時に読み書きは発生しない
            storageMemory.ResetAccessCounter();
            result = storageCache.Flush();
            NNT_ASSERT_RESULT_SUCCESS(result);
            EXPECT_EQ(0, storageMemory.GetReadTimes());
            EXPECT_EQ(0, storageMemory.GetWriteTimes());
        }
    }
}

// 閾値を超えない大きめのサイズ書き込みとベリファイのテスト
TEST(FsWriteThroughCacheStorageTest, FlushVerifyLarge)
{
    static const int BufCount = 2;
    static const size_t CacheSizeMin = 512;
    static const size_t CacheSizeMax = 4096;
    static const size_t MemorySizeMin = 16 * 1024;
    static const size_t MemorySizeMax = 32 * 1024;

    // アラインされたアドレスに対してキャッシュサイズを超えるデータ書き込みを行う
    for( size_t memorySize = MemorySizeMin; memorySize <= MemorySizeMax; memorySize *= 4 )
    {
        for( size_t cacheSize = CacheSizeMin; cacheSize <= CacheSizeMax; cacheSize *= 4 )
        {
            int offset;

            // 実ストレージを生成する
            nnt::fs::util::AccessCountedMemoryStorage storageMemory;
            storageMemory.Initialize(memorySize);

            // キャッシュを初期化する
            nn::fssystem::WriteThroughCacheStorage storageCache;
            storageCache.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&storageMemory, 0, memorySize),
                cacheSize,
                BufCount
            );

            nnt::fs::util::Vector<char> readBuffer(memorySize - 1, 0);

            // 書き込むデータを生成する
            nnt::fs::util::Vector<char> writeBuffer(memorySize - 1);
            std::iota(writeBuffer.begin(), writeBuffer.end(), '\x1');

            // ストレージを 0 埋めする
            nn::Result result = storageMemory.Write(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);

            // データを書き込む
            // 実ストレージに直接書き込みが発生する
            // 書き込みサイズがブロックサイズよりも大きいためフェッチは発生しない
            storageMemory.ResetAccessCounter();
            result = storageCache.Write(0, writeBuffer.data(), writeBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            EXPECT_EQ(1, storageMemory.GetWriteTimes());
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // アラインされたキャッシュサイズ未満の読み込みを行う
            storageMemory.ResetAccessCounter();
            offset = static_cast<int>(cacheSize + cacheSize / 2);
            result = storageCache.Read(offset, readBuffer.data(), cacheSize / 2);
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin() + offset, writeBuffer.begin() + offset + cacheSize / 2,  readBuffer.begin()));
            EXPECT_EQ(1, storageMemory.GetReadTimes());

            // 前回とアドレスはキャッシュされた領域を読み込む
            storageMemory.ResetAccessCounter();
            offset = static_cast<int>(cacheSize);
            result = storageCache.Read(offset, readBuffer.data(), cacheSize / 2);
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin() + offset, writeBuffer.begin() + offset + cacheSize / 2, readBuffer.begin()));
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // キャッシュされている領域を跨いでキャッシュサイズを超える読み込みを行っても分断リードは発生しない
            storageMemory.ResetAccessCounter();
            result = storageCache.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
            EXPECT_EQ(1, storageMemory.GetReadTimes());

            // キャッシュはまだ存在している
            storageMemory.ResetAccessCounter();
            offset = static_cast<int>(cacheSize);
            result = storageCache.Read(offset, readBuffer.data(), cacheSize / 2);
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin() + offset, writeBuffer.begin() + offset + cacheSize / 2, readBuffer.begin()));
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // 前回の大きいサイズでの読み込みはキャッシュされていない
            storageMemory.ResetAccessCounter();
            result = storageCache.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
            EXPECT_EQ(1, storageMemory.GetReadTimes());

            // キャッシュはまだ存在している
            storageMemory.ResetAccessCounter();
            offset = static_cast<int>(cacheSize);
            result = storageCache.Read(offset, readBuffer.data(), cacheSize / 2);
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin() + offset, writeBuffer.begin() + offset + cacheSize / 2, readBuffer.begin()));
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // 書き込むデータを更新する
            std::iota(writeBuffer.begin(), writeBuffer.end(), '\x2');

            // キャッシュされている領域を超えて書き込みを行う
            storageMemory.ResetAccessCounter();
            result = storageCache.Write(0, writeBuffer.data(), writeBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            EXPECT_EQ(1, storageMemory.GetWriteTimes());
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // キャッシュ済みデータはまだ存在している
            storageMemory.ResetAccessCounter();
            offset = static_cast<int>(cacheSize);
            result = storageCache.Read(offset, readBuffer.data(), cacheSize / 2);
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin() + offset, writeBuffer.begin() + offset + cacheSize / 2, readBuffer.begin()));
            EXPECT_EQ(0, storageMemory.GetReadTimes());

            // 全域比較
            storageMemory.ResetAccessCounter();
            result = storageCache.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
            EXPECT_EQ(1, storageMemory.GetReadTimes());

            // 全領域に対してベリファイを行う
            result = storageMemory.Read(0, readBuffer.data(), readBuffer.size());
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
        }
    }
}


// キャッシュ無効化のテスト
TEST(FsWriteThroughCacheStorageTest, Invalidate)
{
    static const auto BufferCacheCount = 4;
    static const auto BufferCacheSize = 256;
    static const auto StorageSize = 16 * 1024;

    // ストレージを用意する
    nnt::fs::util::AccessCountedMemoryStorage storageMemory;
    storageMemory.Initialize(StorageSize);

    nn::fssystem::WriteThroughCacheStorage storageCache;
    NNT_ASSERT_RESULT_SUCCESS(
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, StorageSize),
            BufferCacheSize,
            BufferCacheCount
        )
    );

    std::mt19937 rng(nnt::fs::util::GetRandomSeed());

    for( auto count = 0; count < 100; ++count )
    {
        const auto offset = std::uniform_int_distribution<int64_t>(0, StorageSize - 1)(rng);
        char buffer[1] = {};

        // 下位ストレージへのアクセスが発生するようにデータを読み込む
        // (この後の読み込みでキャッシュされている可能性あり)
        storageMemory.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(storageCache.Read(offset, buffer, 1));
        EXPECT_LE(storageMemory.GetReadTimes(), 1);

        // 同じ個所を読み込んでもキャッシュされているので
        // 下位ストレージへのアクセスは発生しないはず
        storageMemory.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(storageCache.Read(offset, buffer, 1));
        EXPECT_EQ(0, storageMemory.GetReadTimes());

        // キャッシュを破棄してから読むと再び下位ストレージへのアクセスが発生するはず
        storageMemory.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(
            storageCache.OperateRange(
                nullptr, 0,
                nn::fs::OperationId::Invalidate,
                0,
                std::numeric_limits<int64_t>::max(),
                nullptr, 0
            )
        );
        NNT_EXPECT_RESULT_SUCCESS(storageCache.Read(offset, buffer, 1));
        EXPECT_GT(storageMemory.GetInvalidateTimes(), 0);
        EXPECT_EQ(1, storageMemory.GetReadTimes());
    }
}

// 境界値分析
TEST(FsWriteThroughCacheStorageTest, Boundary)
{
    static const int BufCount = 4;
    static const size_t CacheSize = 1024;
    static const size_t MemorySize = 16 * 1024;

    nnt::fs::util::SafeMemoryStorage storageMemory(MemorySize);

    nn::fssystem::WriteThroughCacheStorage storageCache;
    storageCache.Initialize(
        nnt::fs::util::GetTestLibraryAllocator(),
        nn::fs::SubStorage(&storageMemory, 0, MemorySize),
        CacheSize,
        BufCount
    );

    nnt::fs::util::Vector<char> buffer(MemorySize);
    nn::Result result;

    // 読み込み
    result = storageCache.Read(-1, buffer.data(), 1);
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
    result = storageCache.Read(MemorySize - 1, buffer.data(), 1);
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = storageCache.Read(MemorySize, buffer.data(), 1);
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
    result = storageCache.Read(0, buffer.data(), MemorySize);
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = storageCache.Read(0, buffer.data(), MemorySize + 1);
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 書き込み
    result = storageCache.Write(-1, buffer.data(), 1);
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
    result = storageCache.Write(MemorySize - 1, buffer.data(), 1);
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = storageCache.Write(MemorySize, buffer.data(), 1);
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
    result = storageCache.Write(0, buffer.data(), MemorySize);
    NNT_ASSERT_RESULT_SUCCESS(result);
    result = storageCache.Write(0, buffer.data(), MemorySize + 1);
    NNT_ASSERT_RESULT_SUCCESS(result);
}

// ランダムなオフセット・サイズで読み書きするテスト
TEST(FsWriteThroughCacheStorageTest, RandomAccess)
{
    static const int BufCount = 4;
    static const size_t CacheSize = 1024;
    static const size_t MemorySize = 16 * 1024;

    nnt::fs::util::SafeMemoryStorage storageMemory(MemorySize);

    nn::fssystem::WriteThroughCacheStorage storageCache;
    storageCache.Initialize(
        nnt::fs::util::GetTestLibraryAllocator(),
        nn::fs::SubStorage(&storageMemory, 0, MemorySize),
        CacheSize,
        BufCount
    );

    nnt::fs::util::Vector<char> readBuffer(MemorySize, 0);

    // 書き込むデータを生成する
    nnt::fs::util::Vector<char> writeBuffer(MemorySize);
    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

    // ストレージを 0 埋めしておく
    nn::Result result = storageCache.Write(0, readBuffer.data(), readBuffer.size());
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 飛ばし飛ばしアクセス
    for( size_t unitSize = 16; unitSize < CacheSize; unitSize *= 4 )
    {
        for( int64_t unitIndex = 0; unitIndex < CacheSize; unitIndex += unitSize )
        {
            for( int64_t blockIndex = 0; blockIndex < MemorySize / CacheSize; ++blockIndex )
            {
                int64_t offset = CacheSize * blockIndex + unitIndex;
                result = storageCache.Write(offset, writeBuffer.data() + offset, unitSize);
                NNT_ASSERT_RESULT_SUCCESS(result);
            }
        }

        // 書き込んだデータと書き込み元データは一致する
        result = storageCache.Read(0, readBuffer.data(), readBuffer.size());
        NNT_ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(std::equal(writeBuffer.begin(), writeBuffer.end(), readBuffer.begin()));
    }
}

// 読み書きでキャッシュが意図通りの動作をしているかテスト
TEST(FsWriteThroughCacheStorageTest, Cache)
{
    static const size_t MemorySizeMin = 8 * 1024;
    static const size_t MemorySizeMax = 128 * 1024;
    static const size_t CacheSizeMin = MemorySizeMin / 64;
    static const size_t CacheSizeMax = MemorySizeMin / 8;

    for( size_t memorySize = MemorySizeMin; memorySize <= MemorySizeMax; memorySize *= 4 )
    {
        for( size_t cacheSize = CacheSizeMin; cacheSize <= CacheSizeMax; cacheSize *= 2 )
        {
            for( int bufCount = 4; bufCount <= 16; bufCount *= 2 )
            {
                nnt::fs::util::AccessCountedMemoryStorage storageMemory;
                storageMemory.Initialize(memorySize);

                nnt::fs::util::Vector<char> buffer(memorySize);
                nn::Result result;

                // ブロックサイズでアラインされたアクセスは先頭ブロックのみキャッシュされる
                {
                    nn::fssystem::WriteThroughCacheStorage storageCache;
                    storageCache.Initialize(
                        nnt::fs::util::GetTestLibraryAllocator(),
                        nn::fs::SubStorage(&storageMemory, 0, memorySize),
                        cacheSize,
                        bufCount
                    );

                    // ちょうど 1 回書き込みが行われる
                    // データサイズが小さいのでフェッチが発生する
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Write(0, buffer.data(), memorySize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, storageMemory.GetReadTimes());
                    EXPECT_EQ(1, storageMemory.GetWriteTimes());

                    // ちょうど 1 回読み込みが行われる
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Read(0, buffer.data(), memorySize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, storageMemory.GetReadTimes());
                    EXPECT_EQ(0, storageMemory.GetWriteTimes());

                    // 先頭はサイズが小さい場合のみキャッシュされている
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Read(0, buffer.data(), cacheSize - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_GE(1, storageMemory.GetReadTimes());
                    EXPECT_EQ(0, storageMemory.GetWriteTimes());
                }

                // 先頭アドレスがアラインされていればブロックを跨ぐ
                // アクセスサイズであっても書き込みは分断されない
                {
                    nn::fssystem::WriteThroughCacheStorage storageCache;
                    storageCache.Initialize(
                        nnt::fs::util::GetTestLibraryAllocator(),
                        nn::fs::SubStorage(&storageMemory, 0, memorySize),
                        cacheSize,
                        bufCount
                    );

                    // 書き込みは1回のみ発生する
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Write(0, buffer.data(), cacheSize - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, storageMemory.GetReadTimes());
                    EXPECT_EQ(1, storageMemory.GetWriteTimes());

                    // 先頭ブロックでリードが発生する
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Read(2, buffer.data(), cacheSize + 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, storageMemory.GetReadTimes());
                    EXPECT_EQ(0, storageMemory.GetWriteTimes());
                }

                // 先頭アドレスがアラインされていないブロックを跨ぐアクセスサイズであれば、
                // 先頭ブロックとそれ以降で分断されたライトが発生する
                {
                    nn::fssystem::WriteThroughCacheStorage storageCache;
                    storageCache.Initialize(
                        nnt::fs::util::GetTestLibraryAllocator(),
                        nn::fs::SubStorage(&storageMemory, 0, memorySize),
                        cacheSize,
                        bufCount
                    );

                    // 先頭ブロック書き込み時に読み書きそれぞれ1回、
                    // 2ブロック目のために書き込みが1回発生する
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Write(2, buffer.data(), cacheSize - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, storageMemory.GetReadTimes());
                    EXPECT_EQ(2, storageMemory.GetWriteTimes());

                    // 先頭ブロックはキャッシュされているので2ブロック目に対して1回リードが発生する
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Read(2, buffer.data(), cacheSize - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, storageMemory.GetReadTimes());
                    EXPECT_EQ(0, storageMemory.GetWriteTimes());
                }

                // キャッシュされている領域に対して半端な位置から
                // 書き込みアクセスしてもキャッシュは維持される
                {
                    nn::fssystem::WriteThroughCacheStorage storageCache;
                    storageCache.Initialize(
                        nnt::fs::util::GetTestLibraryAllocator(),
                        nn::fs::SubStorage(&storageMemory, 0, memorySize),
                        cacheSize,
                        bufCount
                    );

                    // 半端な位置に書き込むと書き込み前に読み込みが発生する
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Write(1, buffer.data(), cacheSize * 3 - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, storageMemory.GetReadTimes());
                    EXPECT_EQ(2, storageMemory.GetWriteTimes());

                    // 先頭ブロックはキャッシュされている
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Read(5, buffer.data(), cacheSize - 5);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, storageMemory.GetReadTimes());
                    EXPECT_EQ(0, storageMemory.GetWriteTimes());

                    // キャッシュ範囲であってもライトアクセスでは書き込みが行われる
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Write(1, buffer.data(), 4);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, storageMemory.GetReadTimes());
                    EXPECT_EQ(1, storageMemory.GetWriteTimes());

                    // キャッシュは存在しているのでリードアクセスは発生しない
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Read(1, buffer.data(), 4);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, storageMemory.GetReadTimes());
                    EXPECT_EQ(0, storageMemory.GetWriteTimes());
                }

                // アラインされた半端なサイズによるアクセス書き込みでキャッシュされない
                {
                    nn::fssystem::WriteThroughCacheStorage storageCache;
                    storageCache.Initialize(
                        nnt::fs::util::GetTestLibraryAllocator(),
                        nn::fs::SubStorage(&storageMemory, 0, memorySize),
                        cacheSize,
                        bufCount
                    );

                    // 書き込みのみ発生する
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Write(
                                 memorySize - cacheSize * 3,
                                 buffer.data(),
                                 cacheSize * 3 - 1
                             );
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, storageMemory.GetReadTimes());
                    EXPECT_EQ(1, storageMemory.GetWriteTimes());

                    // 書き込み時にキャッシュされてないので実ストレージへの読み込みが行われる
                    // この時点でキャッシュされる
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Read(memorySize - cacheSize, buffer.data(), cacheSize - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, storageMemory.GetReadTimes());
                    EXPECT_EQ(0, storageMemory.GetWriteTimes());

                    // 実ストレージへの書き込みが行われる
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Write(memorySize - cacheSize, buffer.data(), cacheSize - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, storageMemory.GetReadTimes());
                    EXPECT_EQ(1, storageMemory.GetWriteTimes());

                    // キャッシュされているので読み込みは発生しない
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Read(memorySize - cacheSize, buffer.data(), cacheSize - 1);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, storageMemory.GetReadTimes());
                    EXPECT_EQ(0, storageMemory.GetWriteTimes());
                }

                // キャッシュサイズと一致するアクセスでもキャッシュされない
                {
                    nn::fssystem::WriteThroughCacheStorage storageCache;
                    storageCache.Initialize(
                        nnt::fs::util::GetTestLibraryAllocator(),
                        nn::fs::SubStorage(&storageMemory, 0, memorySize),
                        cacheSize,
                        bufCount
                    );

                    // 書き込みのみ発生する
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Write(cacheSize, buffer.data(), cacheSize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, storageMemory.GetReadTimes());
                    EXPECT_EQ(1, storageMemory.GetWriteTimes());

                    // キャッシュされてないことを確認する
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Read(cacheSize, buffer.data(), cacheSize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(1, storageMemory.GetReadTimes());
                    EXPECT_EQ(0, storageMemory.GetWriteTimes());

                    // キャッシュ範囲への書き込みで書き込みのみ行われる
                    storageMemory.ResetAccessCounter();
                    result = storageCache.Write(cacheSize, buffer.data(), cacheSize);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    EXPECT_EQ(0, storageMemory.GetReadTimes());
                    EXPECT_EQ(1, storageMemory.GetWriteTimes());
                }
            }
        }
    }
} // NOLINT(impl/function_size)

// キャッシュの優先順位テスト
TEST(FsWriteThroughCacheStorageTest, CacheOrder)
{
    static const int BlockCountMax = 64;
    static const size_t BlockSize = 256;
    static const size_t MemorySize = BlockSize * BlockCountMax;

    nnt::fs::util::AccessCountedMemoryStorage storageMemory;
    storageMemory.Initialize(MemorySize);

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int loopCount = 0; loopCount < 20; ++loopCount )
    {
        int bufCount = std::uniform_int_distribution<>(1, BlockCountMax - 1)(mt);
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, MemorySize),
            BlockSize,
            bufCount
        );

        std::deque<int> caches;

        for( int innerLoopCount = 0; innerLoopCount < 500; ++innerLoopCount )
        {
            int cacheIndex = std::uniform_int_distribution<>(0, bufCount - 1)(mt);
            int oldIndex = -1;
            bool isAlreadyCached = true;

            if( std::remove(caches.begin(), caches.end(), cacheIndex) == caches.end() )
            {
                // 新しくキャッシュされる
                caches.push_back(cacheIndex);
                isAlreadyCached = false;

                if( bufCount < static_cast<int>(caches.size()) )
                {
                    // 古いキャッシュは破棄される
                    oldIndex = caches.front();
                    caches.pop_front();

                    // 確認のために破棄された領域は再びキャッシュされる
                    caches.pop_front();
                    caches.push_back(oldIndex);
                }
            }
            else
            {
                // 既にあるならリストの最後尾に移動させる
                caches.back() = cacheIndex;
            }

            int data;
            nn::Result result;

            // 古いキャッシュもこの段階ではまだ有効
            // そのためアクセスしても元ファイルへの読み込みは発生しない
            if( 0 <= oldIndex )
            {
                storageMemory.ResetAccessCounter();
                result = storageCache.Read(BlockSize * oldIndex, &data, sizeof(int));
                NNT_ASSERT_RESULT_SUCCESS(result);
                EXPECT_EQ(0, storageMemory.GetReadTimes());
                EXPECT_EQ(0, storageMemory.GetWriteTimes());
            }

            // 新しくキャッシュされる
            // 新しいキャッシュの読み込みが発生するかもしれない
            // 古いキャッシュのフラッシュの書き込みが発生するかもしれない
            storageMemory.ResetAccessCounter();
            result = storageCache.Read(BlockSize * cacheIndex, &data, sizeof(int));
            NNT_ASSERT_RESULT_SUCCESS(result);
            EXPECT_EQ(isAlreadyCached ? 0 : 1, storageMemory.GetReadTimes());

            // 古いキャッシュはこの段階では破棄されている
            // このことを確認するために破棄された領域は再びキャッシュされる
            if( 0 <= oldIndex )
            {
                storageMemory.ResetAccessCounter();
                result = storageCache.Read(BlockSize * oldIndex, &data, sizeof(int));
                NNT_ASSERT_RESULT_SUCCESS(result);
                EXPECT_EQ(1, storageMemory.GetReadTimes());
                EXPECT_EQ(0, storageMemory.GetWriteTimes());
            }
        }
    }
}

// ストレージの読み書きのテスト
// テスト時間が長いので DISABLED_ 付き
TEST(DISABLED_FsWriteThroughCacheStorageTest, ReadWriteHeavy)
{
    static const size_t TestSize = 1024 * 1024;
    static const int BufferCacheCount = 64;
    static const size_t CacheSizeMin = 64;
    static const size_t CacheSizeMax = 512;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> buf1(TestSize);
    nnt::fs::util::Vector<char> buf2(TestSize);

    // 読み書きバッファのサイズを変えつつテストを行う
    for( int i = 0; i < 10000; ++i )
    {
        NN_LOG(".");
        for( size_t sizeTest = 1; sizeTest <= TestSize; sizeTest *= 2 )
        {
            // テストバッファに初期値を書き込み
            std::memset(&buf1[0], 0, TestSize);

            // キャッシュをセットアップ
            size_t bufferCacheSize = GenerateRandomBlockSize(CacheSizeMin, CacheSizeMax);

            nnt::fs::util::SafeMemoryStorage storageMemory(sizeTest);
            NNT_ASSERT_RESULT_SUCCESS(storageMemory.Write(0, &buf1[0], sizeTest));

            nn::fssystem::WriteThroughCacheStorage storage;
            NNT_ASSERT_RESULT_SUCCESS(
                storage.Initialize(
                    nnt::fs::util::GetTestLibraryAllocator(),
                    nn::fs::SubStorage(&storageMemory, 0, sizeTest),
                    bufferCacheSize,
                    BufferCacheCount
                )
            );

            // ランダムな位置にデータを書き、内容を比較
            for( int loop = 1000; 0 < loop; --loop )
            {
                // テストデータを書き込みます。
                size_t offset = std::uniform_int_distribution<size_t>(0, sizeTest - 1)(mt);
                size_t size = std::uniform_int_distribution<size_t>(0, sizeTest - offset)(mt);
                if( size == 0 )
                {
                    continue;
                }

                // 二つのバッファにそれぞれ書き込み
                std::memset(&buf2[0], loop & 0xFF, size);
                NNT_ASSERT_RESULT_SUCCESS(storage.Write(offset, &buf2[0], size));
                std::memcpy(&buf1[offset], &buf2[0], size);

                // 書き込んだ直後の比較
                if( std::uniform_int_distribution<>(0, 10)(mt) < 8 )
                {
                    NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, &buf2[0], size));
                    ASSERT_EQ(0, std::memcmp(&buf1[offset], &buf2[0], size));
                }

                // 全体比較
                if( std::uniform_int_distribution<>(0, 10)(mt) < 5 )
                {
                    NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, &buf2[0], sizeTest));
                    ASSERT_EQ(0, std::memcmp(&buf1[0], &buf2[0], sizeTest));
                }
            }

            storage.Flush();
        }
    }
}

TEST(FsWriteThroughCacheStorageTest, WriteWithRandomError)
{
    static const int BlockCountMax = 64;
    static const size_t BlockSize = 256;
    static const size_t MemorySize = BlockSize * BlockCountMax;

    nnt::fs::util::Vector<char> buf(MemorySize);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // 書き込みに失敗した場合に終了処理を正常に実行できることを確認する確認する
    for( size_t sizeTest = 1; sizeTest <= MemorySize; sizeTest *= 2 )
    {
        nnt::fs::util::AccessCountedMemoryStorage storageMemory;
        storageMemory.Initialize(sizeTest);
        storageMemory.SetRandomError(true);

        int bufCount = std::uniform_int_distribution<>(1, BlockCountMax - 1)(mt);
        nn::fssystem::WriteThroughCacheStorage storageCache;
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&storageMemory, 0, sizeTest),
            BlockSize,
            bufCount
        );

        bool isError = false;
        for( int loop = 10000; 0 < loop; --loop )
        {
            // テストデータを書き込みます。
            size_t offset = std::uniform_int_distribution<size_t>(0, sizeTest - 1)(mt);
            size_t size = std::uniform_int_distribution<size_t>(0, sizeTest - offset)(mt);
            if( size == 0 )
            {
                continue;
            }

            nn::Result result = storageCache.Write(offset, &buf[0], size);
            if( result.IsFailure() )
            {
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultSdCardAccessFailed, result);
                isError = true;
                break;
            }
            storageCache.Flush();
        }

        // ループ回数が十分ならエラーは発生するはず
        ASSERT_TRUE(isError);
    }
}

TEST(FsWriteThroughCacheStorageTest, ReadWriteWithRandomError)
{
    static const int BlockCountMax = 64;
    static const size_t BlockSize = 256;
    static const size_t MemorySize = BlockSize * BlockCountMax;
    static const int BufferCacheCount = 64;
    static const size_t CacheSizeMin = 64;
    static const size_t CacheSizeMax = 512;

    nnt::fs::util::Vector<char> buf1(MemorySize);
    nnt::fs::util::Vector<char> buf2(MemorySize);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nn::Result result;

    // 読み込み時にエラーが発生してもキャッシュが壊れてないことを確認します
    for( size_t sizeTest = 1; sizeTest <= MemorySize; sizeTest *= 2 )
    {
        // テストバッファに初期値を書き込み
        std::memset(&buf1[0], 0, MemorySize);

        // キャッシュをセットアップ
        size_t bufferCacheSize = GenerateRandomBlockSize(CacheSizeMin, CacheSizeMax);

        nnt::fs::util::AccessCountedMemoryStorage storageMemory;
        storageMemory.Initialize(sizeTest);
        NNT_ASSERT_RESULT_SUCCESS(storageMemory.Write(0, &buf1[0], sizeTest));
        storageMemory.SetRandomError(true);

        nn::fssystem::WriteThroughCacheStorage storageCache;
        NNT_ASSERT_RESULT_SUCCESS(
            storageCache.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&storageMemory, 0, sizeTest),
                bufferCacheSize,
                BufferCacheCount
           )
        );

        bool isError = false;
        for( int loop = 10000; 0 < loop; --loop )
        {
            // テストデータを書き込みます。
            size_t offset = std::uniform_int_distribution<size_t>(0, sizeTest - 1)(mt);
            size_t size = std::uniform_int_distribution<size_t>(0, sizeTest - offset)(mt);
            if( size == 0 )
            {
                continue;
            }

            // 二つのバッファにそれぞれ書き込み
            std::memset(&buf2[0], loop & 0xFF, size);
            result = storageCache.Write(offset, &buf2[0], size);
            if( result.IsFailure() )
            {
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultSdCardAccessFailed, result);
                isError = true;
                break;
            }
            std::memcpy(&buf1[offset], &buf2[0], size);

            result = storageCache.Read(offset, &buf2[0], size);
            if( result.IsSuccess() )
            {
                ASSERT_EQ(0, std::memcmp(&buf1[offset], &buf2[0], size));

                // データ比較
                NNT_ASSERT_RESULT_SUCCESS(storageCache.Read(0, &buf2[0], sizeTest));
                ASSERT_EQ(0, std::memcmp(&buf1[0], &buf2[0], sizeTest));
            }
            else
            {
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultSdCardAccessFailed, result);
            }
        }

        // ループ回数が十分ならエラーは発生するはず
        ASSERT_TRUE(isError);
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
// 初期化前のテスト
TEST(FsWriteThroughCacheStorageDeathTest, Precondition)
{
    static const size_t MemorySize = 16 * 1024;
    static const size_t BlockSize = 1024;
    static const int BufCount = 4;

    nnt::fs::util::Vector<char> buffer(1024);
    int64_t size = 0;

    nn::fssystem::WriteThroughCacheStorage storageCache;

    // 未初期化
    EXPECT_DEATH_IF_SUPPORTED(storageCache.Read(0, buffer.data(), buffer.size()), "");
    EXPECT_DEATH_IF_SUPPORTED(storageCache.Write(0, buffer.data(), buffer.size()), "");
    EXPECT_DEATH_IF_SUPPORTED(storageCache.GetSize(&size), "");
    EXPECT_DEATH_IF_SUPPORTED(storageCache.SetSize(1024), "");
    EXPECT_DEATH_IF_SUPPORTED(storageCache.Flush(), "");

    // 初期化
    nnt::fs::util::SafeMemoryStorage storageMemory(MemorySize);
    nn::fs::SubStorage storage(&storageMemory, 0, MemorySize);

    EXPECT_DEATH_IF_SUPPORTED(
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            storage,
            0,
            BufCount
        ),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED(
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            storage,
            1023,
            BufCount
        ),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED(
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            storage,
            1025,
            BufCount
        ),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED(
        storageCache.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            storage,
            BlockSize,
            0
        ),
        ""
    );
    nn::Result result = storageCache.Initialize(
                            nnt::fs::util::GetTestLibraryAllocator(),
                            storage,
                            BlockSize,
                            BufCount
                        );
    NNT_ASSERT_RESULT_SUCCESS(result);

    // 初期化済み
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNullptrArgument, storageCache.GetSize(nullptr));
    result = storageCache.GetSize(&size);
    NNT_ASSERT_RESULT_SUCCESS(result);

    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, storageCache.SetSize(-1));

    result = storageCache.Flush();
    NNT_ASSERT_RESULT_SUCCESS(result);
}
#endif

TEST(FsWriteThroughCacheStorageTest, ReadWriteLargeOffset)
{
    static const size_t AccessSize = 1024;
    static const int64_t StorageSize = nnt::fs::util::LargeOffsetMax + AccessSize;
    static const size_t BufferCacheSize = 2 * 1024;
    static const int BufferCacheCount = 4;

    nnt::fs::util::VirtualMemoryStorage baseStorage(StorageSize);

    nn::fssystem::WriteThroughCacheStorage cacheStorage;
    NNT_ASSERT_RESULT_SUCCESS(cacheStorage.Initialize(
        nnt::fs::util::GetTestLibraryAllocator(),
        nn::fs::SubStorage(&baseStorage, 0, StorageSize),
        BufferCacheSize,
        BufferCacheCount));

    nnt::fs::util::TestStorageAccessWithLargeOffset(
        &cacheStorage,
        AccessSize,
        [&](int64_t offset, char* readBuffer, const char* writeBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(cacheStorage.Read(offset, readBuffer, bufferSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, bufferSize);

            NNT_ASSERT_RESULT_SUCCESS(baseStorage.Read(offset, readBuffer, bufferSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, bufferSize);
        });
}
