﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/nnt_Argument.h>

namespace nnt { namespace fs {

TEST(PooledBufferTest, Basic)
{
    static const auto Size = 256 * 1024;
    nn::fssystem::PooledBuffer pooledBuffer(Size, Size);

    EXPECT_NE(nullptr, pooledBuffer.GetBuffer());
    EXPECT_EQ(Size, pooledBuffer.GetSize());

    char buffer[8] = {};

    EXPECT_TRUE(nn::fssystem::IsPooledBuffer(pooledBuffer.GetBuffer()));
    EXPECT_FALSE(nn::fssystem::IsPooledBuffer(buffer));
}

TEST(PooledBufferTest, AllocateSize)
{
    static const size_t PooledBufferSize = 8 * 1024 * 1024;
    static const size_t SizeMin = nn::fssystem::BufferPoolAlignment;
    static const auto SizeMax
        = nn::fssystem::PooledBuffer::GetAllocatableSizeMax();
    static const auto LargeSizeMax
        = nn::fssystem::PooledBuffer::GetAllocatableParticularlyLargeSizeMax();

    ASSERT_LT(SizeMin, SizeMax);
    ASSERT_LT(SizeMax, LargeSizeMax);
    ASSERT_LT(LargeSizeMax, PooledBufferSize);

    // 最小サイズ以下で確保
    {
        nn::fssystem::PooledBuffer pooledBuffer(SizeMin / 2, SizeMin / 2);
        EXPECT_EQ(SizeMin, pooledBuffer.GetSize());
    }

    {
        nn::fssystem::PooledBuffer pooledBuffer(SizeMin, SizeMin);
        EXPECT_EQ(SizeMin, pooledBuffer.GetSize());
    }

    // 最大サイズ以上で確保
    {
        nn::fssystem::PooledBuffer pooledBuffer(SizeMax, SizeMax);
        EXPECT_EQ(SizeMax, pooledBuffer.GetSize());
    }

    {
        nn::fssystem::PooledBuffer pooledBuffer(SizeMax * 2, SizeMax);
        EXPECT_EQ(SizeMax, pooledBuffer.GetSize());
    }

    // 大容量版の最大サイズ以上で確保
    {
        nn::fssystem::PooledBuffer pooledBuffer;
        pooledBuffer.AllocateParticularlyLarge(LargeSizeMax, LargeSizeMax);
        EXPECT_EQ(LargeSizeMax, pooledBuffer.GetSize());
    }

    {
        nn::fssystem::PooledBuffer pooledBuffer;
        pooledBuffer.AllocateParticularlyLarge(LargeSizeMax * 2, LargeSizeMax);
        EXPECT_EQ(LargeSizeMax, pooledBuffer.GetSize());
    }

    // 最小サイズを指定して確保
    {
        static const auto RequiredSize = 32 * 1024;
        nn::fssystem::PooledBuffer pooledBuffer(RequiredSize / 2, RequiredSize);
        EXPECT_EQ(RequiredSize, pooledBuffer.GetSize());
    }

    {
        static const auto RequiredSize = 32 * 1024;
        nn::fssystem::PooledBuffer pooledBuffer(RequiredSize, RequiredSize);
        EXPECT_EQ(RequiredSize, pooledBuffer.GetSize());
    }

    {
        static const auto RequiredSize = 32 * 1024;
        nn::fssystem::PooledBuffer pooledBuffer(RequiredSize * 2, RequiredSize);
        EXPECT_EQ(RequiredSize * 2, pooledBuffer.GetSize());
    }

    // 限界まで確保
    {
        util::Vector<nn::fssystem::PooledBuffer> pooledBuffers;
        size_t allocatedSizeSum = 0;
        while( allocatedSizeSum < PooledBufferSize )
        {
            // 最大サイズ超えの確保依頼
            pooledBuffers.emplace_back(PooledBufferSize + 1, 1);

            // 512KB * 7, 256KB,.., 8KB, 4KB, 4KB と確保できることを確認する
            EXPECT_EQ(
                pooledBuffers.back().GetSize(),
                std::max(SizeMin, std::min(SizeMax, (PooledBufferSize - allocatedSizeSum) / 2)));

            allocatedSizeSum += pooledBuffers.back().GetSize();
        }
        EXPECT_EQ(allocatedSizeSum, PooledBufferSize);
    }

    {
        util::Vector<nn::fssystem::PooledBuffer> pooledBuffers;
        size_t allocatedSizeSum = 0;
        size_t allocateSize = LargeSizeMax;
        while( allocatedSizeSum < PooledBufferSize )
        {
            // 最大サイズ超えの確保依頼
            pooledBuffers.emplace_back();
            pooledBuffers.back().AllocateParticularlyLarge(PooledBufferSize + 1, 1);

            // 4M, 2M, 1M,..., 8KB, 4KB, 4KB と確保できることを確認する
            EXPECT_EQ(pooledBuffers.back().GetSize(), std::max(SizeMin, allocateSize));
            allocatedSizeSum += pooledBuffers.back().GetSize();
            allocateSize /= 2;
        }
        EXPECT_EQ(allocatedSizeSum, PooledBufferSize);
    }
}

TEST(PooledBufferDeathTest, Precondition)
{
    nn::fssystem::PooledBuffer pooledBuffer;

    // 未確保状態でのバッファ取得
    EXPECT_DEATH_IF_SUPPORTED(pooledBuffer.GetBuffer(), "");
    EXPECT_DEATH_IF_SUPPORTED(pooledBuffer.GetSize(), "");

    static const auto Size = 32 * 1024;
    pooledBuffer.Allocate(Size, Size);

    EXPECT_NE(nullptr, pooledBuffer.GetBuffer());
    EXPECT_EQ(Size, pooledBuffer.GetSize());

    pooledBuffer.Deallocate();

    EXPECT_DEATH_IF_SUPPORTED(pooledBuffer.GetBuffer(), "");
    EXPECT_DEATH_IF_SUPPORTED(pooledBuffer.GetSize(), "");

    // 最小サイズが大きすぎる
    const auto SizeMax = nn::fssystem::PooledBuffer::GetAllocatableSizeMax();
    const auto LargeSizeMax = nn::fssystem::PooledBuffer::GetAllocatableParticularlyLargeSizeMax();
    EXPECT_DEATH_IF_SUPPORTED(pooledBuffer.Allocate(1, SizeMax + 1), "");
    EXPECT_DEATH_IF_SUPPORTED(pooledBuffer.AllocateParticularlyLarge(1, LargeSizeMax + 1), "");

    pooledBuffer.Allocate(1, SizeMax);
    EXPECT_EQ(SizeMax, pooledBuffer.GetSize());
    pooledBuffer.Deallocate();
    pooledBuffer.AllocateParticularlyLarge(1, LargeSizeMax);
    EXPECT_EQ(LargeSizeMax, pooledBuffer.GetSize());
    pooledBuffer.Deallocate();
}

}}
