﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/os/os_Argument.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_SerialNumber.h>

using namespace nn;

namespace {

    void ReadBis(std::unique_ptr<char>* outValue, fs::BisPartitionId id, int64_t offset, size_t size)
    {
        std::unique_ptr<char> buffer(new char[size]);
        NN_ABORT_UNLESS(buffer.get() != nullptr, "Failed to allocate buffer (%zd bytes).\n", size);

        std::unique_ptr<fs::IStorage> storage;
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenBisPartition(&storage, id));
        NN_ABORT_UNLESS_RESULT_SUCCESS(storage->Read(offset, buffer.get(), size));

        *outValue = std::move(buffer);
    }

    void CreateFile(const char* path, const void* buffer, size_t size)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CreateFile(path, size));
        fs::FileHandle file;
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&file, path, fs::OpenMode_Write));
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(file, 0, buffer, size, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        fs::CloseFile(file);
    }

}

extern "C" void nnMain()
{
    NN_LOG("=== Backup EKS and Calibration Info ===\n");

    settings::factory::SerialNumber serialNumber = {{ 0 }};
    NN_ABORT_UNLESS_RESULT_SUCCESS(settings::factory::GetSerialNumber(&serialNumber));
    NN_LOG("Serial Number: %s\n", serialNumber.string);

    auto argc = os::GetHostArgc();
    auto argv = os::GetHostArgv();

    const char* MountName = "host";
    std::string outDirPath;

    if (argc > 1)
    {
        NN_LOG("Output Directory: %s/%s\n", argv[1], serialNumber.string);
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::MountHost(MountName, argv[1]));
        outDirPath = std::string(MountName) + std::string(":/") + std::string(serialNumber.string);
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CreateDirectory(outDirPath.c_str()));
    }
    else
    {
        NN_LOG("Output Directory: %s/%s\n", "%NINTENDO_SDK_ROOT%", serialNumber.string);
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::MountHost(MountName, "./"));
        outDirPath = std::string(MountName) + std::string(":/") + std::string(serialNumber.string);
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CreateDirectory(outDirPath.c_str()));
    }

    NN_LOG("Copying EKS to eks.bin...\n");
    {
        std::unique_ptr<char> buffer;
        const int64_t EksOffset = 0x180000;
        const size_t EksSize = 0x4000;
        ReadBis(&buffer, fs::BisPartitionId::BootPartition1Root, EksOffset, EksSize);
        std::string outFilePath = outDirPath + std::string("/eks.bin");
        CreateFile(outFilePath.c_str(), buffer.get(), EksSize);
    }
    NN_LOG("Done.\n");

    NN_LOG("Copying Calibration Info to cal.bin...\n");
    {
        std::unique_ptr<char> buffer;
        const int64_t CalOffset = 0x4400;
        const size_t CalSize = 0x7FBC00;
        ReadBis(&buffer, fs::BisPartitionId::UserDataRoot, CalOffset, CalSize);
        std::string outFilePath = outDirPath + std::string("/cal.bin");
        CreateFile(outFilePath.c_str(), buffer.get(), CalSize);
    }
    NN_LOG("Done.\n");

    fs::Unmount(MountName);

    NN_LOG("Success.\n");
}
