﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <numeric>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_Stress_AccessFilesTestCase.h"

namespace nnt { namespace fs {

class ReadFilesInDirectoriesTestCase : public AccessFilesTestCase
{
public:
    static const auto FileSize = 1 * 1024;

public:
    ReadFilesInDirectoriesTestCase(int threadCount, int fileCount, bool isVerificationEnabled, bool isCache) NN_NOEXCEPT
        : AccessFilesTestCase(threadCount, fileCount, nn::fs::OpenMode_Write, false, isVerificationEnabled, isCache),
          m_Opened(false)
    {
    }

    virtual ~ReadFilesInDirectoriesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 20;
    }

    virtual void SetUp(FsStressTest* pTest) NN_NOEXCEPT NN_OVERRIDE
    {
        AccessFilesTestCase::SetUp(pTest);
        ASSERT_FALSE(FailedAny());

        FailAll();
        m_Opened = false;
        char path[PathLength];
        nn::util::SNPrintf(path, sizeof(path), "%s:%s/", GetMountName(), GetTestDirectoryPath());
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(
            &m_Directory,
            path,
            nn::fs::OpenDirectoryMode_File));
        m_Opened = true;
        SucceedAll();
    }

    virtual void TearDown(FsStressTest* pTest) NN_NOEXCEPT NN_OVERRIDE
    {
        if( m_Opened )
        {
            nn::fs::CloseDirectory(m_Directory);
            m_Opened = false;
        }

        AccessFilesTestCase::TearDown(pTest);
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        std::unique_ptr<nn::fs::DirectoryEntry> entry(new nn::fs::DirectoryEntry);
        std::unique_ptr<char[]> buffer(new char[FileSize]);

        for( ; ; )
        {
            int64_t count = 0;
            {
                std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(
                    &count,
                    entry.get(),
                    m_Directory,
                    1));
            }
            if( count == 0 )
            {
                break;
            }

            nn::fs::FileHandle file;
            char path[PathLength];
            nn::util::SNPrintf(
                path,
                sizeof(path),
                "%s:%s/%s",
                GetMountName(),
                GetTestDirectoryPath(),
                entry->name);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));
            if( FileSize <= size )
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, buffer.get(), FileSize)) << path;
                if(IsVerificationEnabled())
                {
                    ASSERT_TRUE(Verify(path, buffer.get(), 0, static_cast<size_t>(size)));
                }
            }
        }

        Succeed(threadIndex);
    }

protected:
    virtual void SetUpFile(bool* outSucceeded, int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, FileSize));

        std::unique_ptr<char[]> buffer(new char[FileSize]);
        std::iota(buffer.get(), buffer.get() + FileSize, static_cast<char>(entryIndex));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(
            file,
            0,
            buffer.get(),
            FileSize,
            nn::fs::WriteOption()));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));

        if(IsVerificationEnabled())
        {
            RegisterVerifyBuffer(entryIndex, buffer.get(), FileSize);
        }

        *outSucceeded = true;
    }

private:
    nn::fs::DirectoryHandle m_Directory;
    bool m_Opened;
};

NN_DEFINE_STATIC_CONSTANT(const int ReadFilesInDirectoriesTestCase::FileSize);

class ReadFilesInDirectories : public ReadFilesInDirectoriesTestCase
{
public:
    ReadFilesInDirectories() NN_NOEXCEPT
        : ReadFilesInDirectoriesTestCase(FsStressTest::ThreadCountMax, 10, true, false)
    {
    }
};

class ReadFilesInDirectoriesWithCache : public ReadFilesInDirectoriesTestCase
{
public:
    ReadFilesInDirectoriesWithCache() NN_NOEXCEPT
        : ReadFilesInDirectoriesTestCase(FsStressTest::ThreadCountMax, 10, true, true)
    {
    }
};

class ReadFewerFilesInDirectories : public ReadFilesInDirectoriesTestCase
{
public:
    ReadFewerFilesInDirectories() NN_NOEXCEPT
        : ReadFilesInDirectoriesTestCase(FsStressTest::ThreadCountMax / 2, 10, true, false)
    {
    }
};

class ReadFewerFilesInDirectoriesWithCache : public ReadFilesInDirectoriesTestCase
{
public:
    ReadFewerFilesInDirectoriesWithCache() NN_NOEXCEPT
        : ReadFilesInDirectoriesTestCase(FsStressTest::ThreadCountMax / 2, 10, true, true)
    {
    }
};

TEST_F(SaveDataFsStressTest, ReadFilesInDirectories)
{
    Test<ReadFilesInDirectories>(GetMountName());
}

TEST_F(MultipleSaveDataFsStressTest, ReadFilesInDirectories)
{
    Test<ReadFewerFilesInDirectories, ReadFewerFilesInDirectories>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, ReadFilesInDirectories)
{
    Test<ReadFilesInDirectories>(GetMountName());
}

TEST_F(SaveDataFsRomFsStressTest, ReadFilesInDirectories)
{
    Test<ReadFewerFilesInDirectories, ReadFewerFilesInDirectories>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, ReadFilesInDirectoriesWithCache)
{
    Test<ReadFewerFilesInDirectories, ReadFewerFilesInDirectoriesWithCache>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(RomFsStressTest, ReadFilesInDirectories)
{
    Test<ReadFilesInDirectories>(GetMountName());
}

TEST_F(RomFsStressTest, ReadFilesInDirectoriesWithCache)
{
    Test<ReadFilesInDirectoriesWithCache>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, ReadFilesInDirectories)
{
    Test<ReadFilesInDirectories>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, ReadFilesInDirectories)
{
    Test<ReadFilesInDirectories>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, ReadFilesInDirectories)
{
    Test<ReadFilesInDirectories>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
