﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFs_Stress_OpenEntriesTestCase.h"

namespace nnt { namespace fs {

class OpenDirectoriesTestCase : public OpenEntriesTestCase
{
public:
    OpenDirectoriesTestCase(int threadCount, int directoryCount) NN_NOEXCEPT
        : OpenEntriesTestCase(
            threadCount,
            directoryCount,
            nn::fs::DirectoryEntryType_Directory)
    {
    }

    virtual ~OpenDirectoriesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);
        ASSERT_LE(GetEntryCount() * GetThreadCount(), FsStressTest::EntryCountMax);
        auto openedEntryCount = 0;
        NN_UTIL_SCOPE_EXIT
        {
            for( auto entryIndex = 0; entryIndex < openedEntryCount; ++entryIndex )
            {
                nn::fs::CloseDirectory(m_Directories[threadIndex * GetEntryCount() + entryIndex]);
            }
        };
        static const auto FatOpenCountMax = 2;
        const auto entryCount = pTest->AccessesFatDirectly(GetTestCaseIndex())
            ? std::min(GetEntryCount(), FatOpenCountMax)
            : GetEntryCount();
        for( auto entryIndex = 0; entryIndex < entryCount; ++entryIndex )
        {
            char path[PathLength];
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(
                m_Directories + GetEntryIndex(threadIndex, entryIndex),
                MakePath(path, GetEntryIndex(threadIndex, entryIndex)),
                nn::fs::OpenDirectoryMode_All));
            ++openedEntryCount;
        }
        Succeed(threadIndex);
    }

private:
    nn::fs::DirectoryHandle m_Directories[FsStressTest::EntryCountMax];
};

class OpenSingleDirectory : public OpenDirectoriesTestCase
{
public:
    OpenSingleDirectory() NN_NOEXCEPT
        : OpenDirectoriesTestCase(FsStressTest::ThreadCountMax, 1)
    {
    }

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 500;
    }
};

class OpenNumerousDirectories : public OpenDirectoriesTestCase
{
public:
    OpenNumerousDirectories() NN_NOEXCEPT
        : OpenDirectoriesTestCase(FsStressTest::ThreadCountMax, 10)
    {
    }
};

class OpenFewerDirectories : public OpenDirectoriesTestCase
{
public:
    OpenFewerDirectories() NN_NOEXCEPT
        : OpenDirectoriesTestCase(FsStressTest::ThreadCountMax / 2, 10)
    {
    }
};

TEST_F(SaveDataFsStressTest, OpenSingleDirectory)
{
    Test<OpenSingleDirectory>(GetMountName());
}

TEST_F(SaveDataFsStressTest, OpenNumerousDirectories)
{
    Test<OpenNumerousDirectories>(GetMountName());
}

TEST_F(MultipleSaveDataFsStressTest, OpenNumerousDirectories)
{
    Test<OpenFewerDirectories, OpenFewerDirectories>(GetMountName(0), GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, OpenSingleDirectory)
{
    Test<OpenSingleDirectory>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, OpenNumerousDirectories)
{
    Test<OpenNumerousDirectories>(GetMountName());
}

TEST_F(SaveDataFsRomFsStressTest, OpenNumerousDirectories)
{
    Test<OpenFewerDirectories, OpenFewerDirectories>(GetMountName(0), GetMountName(1));
}

TEST_F(RomFsStressTest, OpenSingleDirectory)
{
    Test<OpenSingleDirectory>(GetMountName());
}

TEST_F(RomFsStressTest, OpenNumerousDirectories)
{
    Test<OpenNumerousDirectories>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, OpenSingleDirectory)
{
    Test<OpenSingleDirectory>(GetMountName());
}

TEST_F(HostFsStressTest, OpenNumerousDirectories)
{
    Test<OpenNumerousDirectories>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, OpenSingleDirectory)
{
    Test<OpenSingleDirectory>(GetMountName());
}

TEST_F(SdCardFsStressTest, OpenNumerousDirectories)
{
    Test<OpenNumerousDirectories>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, OpenSingleDirectory)
{
    Test<OpenSingleDirectory>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, OpenNumerousDirectories)
{
    Test<OpenNumerousDirectories>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
