﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <map>
#include "testFs_Stress_AccessFilesTestCase.h"

namespace nnt { namespace fs {

namespace {

    const size_t CacheSize = 32 * 1024 * 1024;
    const int FileCountMax = 100;

    std::map<nnt::fs::util::String, std::unique_ptr<char[]>> g_VerifyBuffer;
    nnt::fs::util::GlobalFileDataCache g_Cache;
    int g_VerifyBufferReferenceCount = 0;
    int g_CacheReferenceCount = 0;

}

AccessFilesTestCase::AccessFilesTestCase(
    int threadCount,
    int entryCount,
    nn::fs::OpenMode mode) NN_NOEXCEPT
    : OpenEntriesTestCase(threadCount, entryCount, nn::fs::DirectoryEntryType_File),
      m_Mode(mode),
      m_KeepOpened(true),
      m_FilesOpenedThreadCount(0),
      m_OpenedFileCount(0),
      m_IsVerificationEnabled(false),
      m_IsCacheEnabled(false)
{
}

AccessFilesTestCase::AccessFilesTestCase(
    int threadCount,
    int entryCount,
    nn::fs::OpenMode mode,
    bool keepOpened) NN_NOEXCEPT
    : OpenEntriesTestCase(threadCount, entryCount, nn::fs::DirectoryEntryType_File),
      m_Mode(mode),
      m_KeepOpened(keepOpened),
      m_FilesOpenedThreadCount(0),
      m_OpenedFileCount(0),
      m_IsVerificationEnabled(false),
      m_IsCacheEnabled(false)
{
}

AccessFilesTestCase::AccessFilesTestCase(
    int threadCount,
    int entryCount,
    nn::fs::OpenMode mode,
    bool keepOpened,
    bool isVerificationEnabled,
    bool isCacheEnabled) NN_NOEXCEPT
    : OpenEntriesTestCase(threadCount, entryCount, nn::fs::DirectoryEntryType_File),
      m_Mode(mode),
      m_KeepOpened(keepOpened),
      m_FilesOpenedThreadCount(0),
      m_OpenedFileCount(0),
      m_IsVerificationEnabled(isVerificationEnabled),
      m_IsCacheEnabled(isCacheEnabled)
{
}

AccessFilesTestCase::~AccessFilesTestCase() NN_NOEXCEPT
{
}

void AccessFilesTestCase::SetUp(FsStressTest* pTest) NN_NOEXCEPT
{
    // ファイルを作成する
    OpenEntriesTestCase::SetUp(pTest);
    ASSERT_FALSE(FailedAny());

    // ファイルを開き、初期化する
    FailAll();

    if( pTest->IsReadOnly(GetTestCaseIndex()) )
    {
        if( m_IsVerificationEnabled )
        {
            SetUpVerifyBuffer();
        }
        if( m_KeepOpened )
        {
            ASSERT_NE(0, m_Mode & nn::fs::OpenMode_Read);
            m_Mode = nn::fs::OpenMode_Read;
        }
        else
        {
            SucceedAll();
            return;
        }
    }

    ASSERT_LE(GetThreadCount() * GetEntryCount(), FsStressTest::EntryCountMax);
    m_FilesOpenedThreadCount = 0;
    for( auto threadIndex = 0; threadIndex < GetThreadCount(); ++threadIndex )
    {
        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            char path[PathLength];
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(
                m_Files + GetEntryIndex(threadIndex, entryIndex),
                MakePath(path, GetEntryIndex(threadIndex, entryIndex)),
                m_Mode));
            ++m_OpenedFileCount;
            if( !pTest->IsReadOnly(GetTestCaseIndex()) )
            {
                auto succeeded = false;
                SetUpFile(&succeeded, GetEntryIndex(threadIndex, entryIndex));
                ASSERT_TRUE(succeeded);
            }
            if( !m_KeepOpened )
            {
                --m_OpenedFileCount;
                nn::fs::CloseFile(m_Files[GetEntryIndex(threadIndex, entryIndex)]);
            }
        }
        ++m_FilesOpenedThreadCount;
        m_OpenedFileCount = 0;
    }

    if(m_IsCacheEnabled)
    {
        if(g_CacheReferenceCount == 0)
        {
            NN_ASSERT(!g_Cache.IsGlobalCacheEnable());
            g_Cache.Initialize(CacheSize);
        }
        ++g_CacheReferenceCount;
    }

    SucceedAll();
}

void AccessFilesTestCase::TearDown(FsStressTest* pTest) NN_NOEXCEPT
{
    // ファイルを閉じる
    if( m_KeepOpened )
    {
        ASSERT_LE(GetThreadCount() * GetEntryCount(), FsStressTest::EntryCountMax);
        for( auto threadIndex = 0; threadIndex <= m_FilesOpenedThreadCount; ++threadIndex )
        {
            const auto entryCount = threadIndex == m_FilesOpenedThreadCount
                ? m_OpenedFileCount
                : GetEntryCount();
            for( auto entryIndex = 0; entryIndex < entryCount; ++entryIndex )
            {
                nn::fs::CloseFile(GetFile(GetEntryIndex(threadIndex, entryIndex)));
            }
        }
    }

    // ファイルを削除する
    OpenEntriesTestCase::TearDown(pTest);

    if(m_IsVerificationEnabled && g_VerifyBufferReferenceCount > 0)
    {
        g_VerifyBufferReferenceCount--;
        if(g_VerifyBufferReferenceCount == 0)
        {
            g_VerifyBuffer.clear();
        }
    }
    if(m_IsCacheEnabled && g_CacheReferenceCount > 0)
    {
        NN_ASSERT(g_Cache.IsGlobalCacheEnable());
        g_CacheReferenceCount--;
        if(g_CacheReferenceCount == 0)
        {
            g_Cache.Finalize();
        }
    }
}

void AccessFilesTestCase::SetUpVerifyBuffer() NN_NOEXCEPT
{
    for( auto fileIndex = 0; fileIndex < FileCountMax; ++fileIndex )
    {
        char path[PathLength];
        MakePath(path, fileIndex);

        decltype(g_VerifyBuffer)::iterator it = g_VerifyBuffer.find(path);
        if(it == g_VerifyBuffer.end())
        {
            nn::fs::FileHandle file;
            nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };

            int64_t fileSize;
            nn::fs::GetFileSize(&fileSize, file);
            g_VerifyBuffer.insert(std::make_pair(path, std::make_unique<char[]>(static_cast<size_t>(fileSize))));
            nn::fs::ReadFile(file, 0, g_VerifyBuffer.at(path).get(), static_cast<size_t>(fileSize));
            std::unique_ptr<char[]> verify = std::make_unique<char[]>(static_cast<size_t>(fileSize));
            nnt::fs::util::FillBufferWithXorShift(path, verify.get(), static_cast<size_t>(fileSize));
            ASSERT_TRUE(Verify(path, verify.get(), 0, static_cast<size_t>(fileSize)));
        }
    }
    g_VerifyBufferReferenceCount++;
}

bool AccessFilesTestCase::Verify(char* path, char* buffer, int64_t offset, size_t size) NN_NOEXCEPT
{
    if(m_IsVerificationEnabled)
    {
        decltype(g_VerifyBuffer)::iterator it = g_VerifyBuffer.find(path);
        if(it != g_VerifyBuffer.end())
        {
            return std::memcmp(buffer, &g_VerifyBuffer.at(path).get()[offset], size) == 0;
        }
    }

    return false;
}

bool AccessFilesTestCase::Verify(int index, char* buffer, int64_t offset, size_t size) NN_NOEXCEPT
{
    char path[PathLength];
    MakePath(path, index);
    return Verify(path, buffer, offset, size);
}

void AccessFilesTestCase::RegisterVerifyBuffer(int index, char* buffer, size_t size) NN_NOEXCEPT
{
    if(m_IsVerificationEnabled)
    {
        char path[PathLength];
        MakePath(path, index);
        g_VerifyBuffer.insert(std::make_pair(path, std::make_unique<char[]>(static_cast<size_t>(size))));
        std::memcpy(g_VerifyBuffer.at(path).get(), buffer, static_cast<size_t>(size));
    }
}

}}
