﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/util/util_FormatString.h>
#include <nn/os.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

const int WaitShutdownTimeSec = 60;
const int FileSize = 16 * 1024;
const int NameLength = 256;
const char TestStatusFileName[] = "FileWithoutFlushStatus.txt";
const int TestStatusFileSize = 7;

class SdCard_FatCacheTest : public ::testing::Test, public nnt::fs::util::PrepareWorkDirFixture
{
protected:
    char m_MountName[NameLength];

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::util::SNPrintf(m_MountName, sizeof(m_MountName), "SdCard");
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug(m_MountName));
    }
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::Unmount(m_MountName);
    }
};

void FillPattern(char* pBuffer, int size)
{
    for (int i = 0; i < size; i++)
    {
        pBuffer[i] = static_cast<char>(i & 0xff);
    }
}

void WriteTestFile(const char* filePath, const char* buffer, unsigned int size)
{
    nn::fs::FileHandle fileHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, filePath, nn::fs::OpenMode_Write));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(fileHandle, 0, buffer, size, nn::fs::WriteOption::MakeValue(0)));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(fileHandle));
    nn::fs::CloseFile(fileHandle);
}

}

TEST_F(SdCard_FatCacheTest, WriteWithoutFlush_Ready)
{
    char testStatusFile[NameLength];
    nn::util::SNPrintf(testStatusFile, sizeof(testStatusFile), "%s:/%s", m_MountName, TestStatusFileName);
    nn::fs::DeleteFile(testStatusFile);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(testStatusFile, TestStatusFileSize));
    WriteTestFile(testStatusFile, "Create", TestStatusFileSize);

    char testFile[NameLength];
    nn::util::SNPrintf(testFile, sizeof(testFile), "%s:/FileWithoutFlush.bin", m_MountName);
    nn::fs::DeleteFile(testFile);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(testFile, FileSize));

    char buffer[FileSize];
    std::memset(buffer, 0xff, FileSize);

    WriteTestFile(testFile, buffer, FileSize);
    WriteTestFile(testStatusFile, "Ready ", TestStatusFileSize);
}

TEST_F(SdCard_FatCacheTest, WriteWithoutFlush)
{
    char testStatusFile[NameLength];
    nn::util::SNPrintf(testStatusFile, sizeof(testStatusFile), "%s:/%s", m_MountName, TestStatusFileName);
    WriteTestFile(testStatusFile, "Start ", TestStatusFileSize);

    char testFile[NameLength];
    nn::util::SNPrintf(testFile, sizeof(testFile), "%s:/FileWithoutFlush.bin", m_MountName);

    char buffer[FileSize];
    FillPattern(buffer, FileSize);

    WriteTestFile(testStatusFile, "Open  ", TestStatusFileSize);

    nn::fs::FileHandle fileHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, testFile, nn::fs::OpenMode_Write));

    WriteTestFile(testStatusFile, "Write ", TestStatusFileSize);

    /* Fatの（PrFILE2の）キャッシュにのせるため、オフセットを1だけずらして書く */
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(fileHandle, 1, buffer, FileSize - 1, nn::fs::WriteOption::MakeValue(0)));

    WriteTestFile(testStatusFile, "Sleep ", TestStatusFileSize);

    /* FlushFile をしていない状態で、シャットダウン待ち */
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(WaitShutdownTimeSec));
    EXPECT_TRUE(false); /* シャットダウンに失敗？ */

    WriteTestFile(testStatusFile, "Wake  ", TestStatusFileSize);

    nn::fs::FlushFile(fileHandle);
    nn::fs::CloseFile(fileHandle);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(testFile));
}

/* WriteWithoutFlush が実行された後に、実施される想定 */
TEST_F(SdCard_FatCacheTest, WriteWithoutFlush_Verify)
{
    char testStatusFile[NameLength];
    nn::util::SNPrintf(testStatusFile, sizeof(testStatusFile), "%s:/%s", m_MountName, TestStatusFileName);

    char testFile[NameLength];
    nn::util::SNPrintf(testFile, sizeof(testFile), "%s:/FileWithoutFlush.bin", m_MountName);

    char buffer[FileSize];
    std::memset(buffer, 0, FileSize);

    char expectPattern[FileSize];
    expectPattern[0] = 0xff;
    FillPattern(&expectPattern[1], FileSize - 1);

    nn::fs::FileHandle fileHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, testFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, buffer, FileSize));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(expectPattern, buffer, FileSize);

    nn::fs::CloseFile(fileHandle);
    nn::fs::DeleteFile(testFile);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, testStatusFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, buffer, TestStatusFileSize));
    if (std::strncmp("Sleep ", buffer, TestStatusFileSize) != 0)
    {
        NN_LOG("Status: %s\n", buffer);
        EXPECT_TRUE(false);
    }

    nn::fs::CloseFile(fileHandle);
    nn::fs::DeleteFile(testStatusFile);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
