﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_FormatString.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/fssrv/fssrv_MemoryResourceFromStandardAllocator.h>

#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

void* operator new(size_t size)
{
    return malloc(size);
}

void operator delete(void* p) NN_NOEXCEPT
{
    free(p);
}

namespace {

/**
 * @brief スレッド毎の要素を持つ構造体
 */
struct ThreadInfo
{
    nn::fs::FileHandle handle;
    int64_t fileSize;
    int threadNumber;
    int bufferSize;
    int threadCount;
};

/**
* @brief テストの実行内容
*/
enum ThreadTestType
{
    ThreadTestType_Read,
    ThreadTestType_Write
};

const char FatDirectoryPath[]   = "fat:/fattestdir";
const char FatFilePath[]        = "fat:/fattestdir/file";
const char FragmentFilePath[]   = "fat:/fattestdir/fragmentfile";

const size_t FileSize      = 256 * 1024 * 1024;
const size_t FatManageSize =  16 * 1024 * 1024;

const int PathLength     = 35;
const int ThreadCountMax = 8;

const int BufferSizeMin    = 1   * 1024;
const int BufferSizeMax    = 4   * 1024 * 1024;

char g_FilePath[ThreadCountMax][PathLength];
nnt::fs::util::Vector<nnt::fs::util::PerformanceResult> g_PerformanceResultsRead;
nnt::fs::util::Vector<nnt::fs::util::PerformanceResult> g_PerformanceResultsWrite;

const int BufferPoolSize = 6 * 1024 * 1024;
NN_ALIGNAS(4096) char g_BufferPool[BufferPoolSize];
nn::mem::StandardAllocator g_BufferAllocator(g_BufferPool, BufferPoolSize);
} // namespace

class PerformanceTestFatFileSystem : public ::testing::Test
{
public:
    /**
     * @brief FATにマウントする関数です
     * @param[in] storageSize FATに使用するストレージサイズ
     */
    void MountFat(const size_t storageSize) NN_NOEXCEPT
    {
        m_StorageBuffer.reset(nullptr);
        m_StorageBuffer.reset(new char[storageSize]);

        ASSERT_NE(nullptr, m_StorageBuffer.get());
        memset(m_StorageBuffer.get(), 0xCD, storageSize);

        m_BaseStorage.reset(nullptr);
        m_BaseStorage.reset(new nn::fs::MemoryStorage(m_StorageBuffer.get(), storageSize));
        ASSERT_NE(nullptr, m_BaseStorage.get());

        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        m_CacheBuffer.reset(nullptr);
        m_CacheBuffer.reset(new char[cacheBufferSize]);
        ASSERT_NE(nullptr, m_CacheBuffer.get());

        std::unique_ptr<nn::fat::FatFileSystem> pFatFileSystem(new nn::fat::FatFileSystem());
        ASSERT_NE(nullptr, pFatFileSystem.get());

        NNT_ASSERT_RESULT_SUCCESS(
            pFatFileSystem->Initialize(m_BaseStorage.get(), m_CacheBuffer.get(), cacheBufferSize)
        );
        NNT_ASSERT_RESULT_SUCCESS(pFatFileSystem->Format());
        NNT_ASSERT_RESULT_SUCCESS(pFatFileSystem->Mount());
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::fsa::Register("fat", std::move(pFatFileSystem))
        );
    }

private:
    std::unique_ptr<char[]> m_StorageBuffer;
    std::unique_ptr<char[]> m_CacheBuffer;
    std::unique_ptr<nn::fs::MemoryStorage> m_BaseStorage;
};

/**
 * @brief 対象のファイルを別ファイルと交互に書き込むことで断片化させます。
 * @param[in/out] handle         断片化させるファイルのハンドル
 * @param[in/out] fragmentHandle 断片化用ファイルのハンドル
 * @param[in]     bufferSize        断片化させる際のバッファサイズ
 */
void FragmentTest(nn::fs::FileHandle* handle, nn::fs::FileHandle* fragmentHandle
                , size_t bufferSize, size_t fragmentBufSize,
                const char* mainFileName, const char* subFileName, int64_t fileSize) NN_NOEXCEPT
{
    // 断片化させるファイルを作成し、開く
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(mainFileName, 0));
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::OpenFile(handle, mainFileName, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)
    );
    // 断片化用のファイルを作成し、開く
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(subFileName, 0));
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::OpenFile(fragmentHandle, subFileName, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)
    );

    std::unique_ptr<char[]> writeBuffer(new char[bufferSize]);
    ASSERT_NE(nullptr, writeBuffer.get());
    nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), bufferSize);

    std::unique_ptr<char[]> fragmentBuffer(new char[fragmentBufSize]);
    ASSERT_NE(nullptr, fragmentBuffer.get());
    nnt::fs::util::FillBufferWithRandomValue(fragmentBuffer.get(), fragmentBufSize);

    // 断片化したファイルの作成
    for( int64_t offset = 0; offset < fileSize; offset += bufferSize )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::WriteFile(*handle, offset, writeBuffer.get(), bufferSize, nn::fs::WriteOption())
        );
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::WriteFile(*fragmentHandle, (offset / bufferSize) * fragmentBufSize, fragmentBuffer.get(), fragmentBufSize, nn::fs::WriteOption())
        );
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(*handle));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(*fragmentHandle));
}

/**
 * @brief 対象のファイルに書き込みをし、書き込み時間を表示します。
 * @param[in] arg スレッドの要素
 */
void WriteTest(void *arg) NN_NOEXCEPT
{
    ThreadInfo* pThreadItem = reinterpret_cast<ThreadInfo*>(arg);

    std::unique_ptr<char[]> writeBuffer(new char[pThreadItem->bufferSize]);
    ASSERT_NE(nullptr, writeBuffer.get());

    nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), pThreadItem->bufferSize);

    nnt::fs::util::TimeCount timeCountSequencialWrite;
    timeCountSequencialWrite.StartTime();
    for( int64_t offset = 0; offset < pThreadItem->fileSize; offset += pThreadItem->bufferSize )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(pThreadItem->handle, offset, writeBuffer.get(), pThreadItem->bufferSize
                                          , nn::fs::WriteOption()));
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(pThreadItem->handle));
    timeCountSequencialWrite.StopTime();

    // 計測時間を push_back します。
    if( pThreadItem->threadCount == 1 )
    {
        // パフォーマンス計測結果 push_back 処理（シングルスレッド用）
        nnt::fs::util::AddPerformanceResults(
            g_PerformanceResultsWrite, pThreadItem->bufferSize, pThreadItem->fileSize, timeCountSequencialWrite.GetEndTime() / 1000
        );
    }
    else
    {
        // パフォーマンス計測結果 push_back 処理（マルチスレッド用）
        nnt::fs::util::AddPerformanceResultsMultiThread(
            g_PerformanceResultsWrite, pThreadItem->bufferSize, pThreadItem->fileSize, timeCountSequencialWrite.GetEndTime() / 1000, pThreadItem->threadCount
        );
    }
}

/**
 * @brief 対象のファイルを読み込み、読み込み時間を表示します。
 * @param[in] arg スレッドの要素
 */
void ReadTest(void *arg) NN_NOEXCEPT
{
    ThreadInfo* pThreadItem = reinterpret_cast<ThreadInfo*>(arg);

    std::unique_ptr<char[]> readBuffer(new char[pThreadItem->bufferSize]);
    ASSERT_NE(nullptr, readBuffer.get());

    nnt::fs::util::TimeCount timeCountSequencialRead;
    timeCountSequencialRead.StartTime();
    for( int64_t offset = 0; offset < pThreadItem->fileSize; offset += pThreadItem->bufferSize )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(pThreadItem->handle, offset, readBuffer.get()
                                         , pThreadItem->bufferSize));
    }
    timeCountSequencialRead.StopTime();

    // 計測時間を push_back します。
    if( pThreadItem->threadCount == 1 )
    {
        // パフォーマンス計測結果 push_back 処理（シングルスレッド用）
        nnt::fs::util::AddPerformanceResults(
            g_PerformanceResultsRead, pThreadItem->bufferSize, pThreadItem->fileSize, timeCountSequencialRead.GetEndTime() / 1000
        );
    }
    else
    {
        // パフォーマンス計測結果 push_back 処理（マルチスレッド用）
        nnt::fs::util::AddPerformanceResultsMultiThread(
            g_PerformanceResultsRead, pThreadItem->bufferSize, pThreadItem->fileSize, timeCountSequencialRead.GetEndTime() / 1000, pThreadItem->threadCount
        );
    }
}

/**
 * @brief スレッドの生成、開始、待機、破棄を行います。
 * @param[in/out] params            各スレッドの要素
 * @param[in]     threadCount       スレッド数
 * @param[in]     threadTestType    読み込み or 書き込み
 */
void TestThread(nnt::fs::util::Vector<ThreadInfo> params, const int threadCount, const int threadTestType, int fragmentCount) NN_NOEXCEPT
{
    nn::os::ThreadType threads[ThreadCountMax];
    const size_t ThreadStackSize = 16 * 1024;
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t s_ThreadStack[ThreadCountMax][ThreadStackSize];
    size_t fragmentBufferSize;

    for( int bufferSize = BufferSizeMin ; bufferSize <= BufferSizeMax ; bufferSize *= 2 )
    {
        // ファイルを作成して、開く
        for( int i = 0; i < threadCount; i += 2 )
        {
            params[i].fileSize = FileSize;
            params[i + 1].fileSize = FileSize;
            params[i].threadNumber = i;
            params[i + 1].threadNumber = i + 1;
            fragmentBufferSize = static_cast<size_t>(params[i].fileSize / fragmentCount);
            nn::util::SNPrintf(g_FilePath[i], PathLength, "%s%02d", FatFilePath, i);
            nn::util::SNPrintf(g_FilePath[i + 1], PathLength, "%s%02d", FatFilePath, i + 1);
            // ファイルを断片化させる
            FragmentTest(
                &params[i].handle,
                &params[i + 1].handle,
                fragmentBufferSize,
                fragmentBufferSize,
                g_FilePath[i],
                g_FilePath[i + 1],
                params[i].fileSize
            );
        }

        // スレッドを作成する
        for( int i = 0; i < threadCount; ++i )
        {
            params[i].bufferSize = bufferSize;
            params[i].threadCount = threadCount;
            // 書き込み
            if( threadTestType == ThreadTestType_Write )
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread( &threads[i], WriteTest, &params[i]
                                    , s_ThreadStack[i], ThreadStackSize, nn::os::DefaultThreadPriority));
            }

            // 読み込み
            else if( threadTestType == ThreadTestType_Read )
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread( &threads[i], ReadTest, &params[i]
                                    , s_ThreadStack[i], ThreadStackSize, nn::os::DefaultThreadPriority));
            }
        }

        // スレッドの実行を開始する
        for( int i = 0; i < threadCount; ++i )
        {
            nn::os::StartThread(&threads[i]);
        }

        // スレッドが終了するのを待つ
        for( int i = 0; i < threadCount; ++i )
        {
            nn::os::WaitThread(&threads[i]);
        }

        // ファイルを閉じ、削除する
        for( int i = 0; i < threadCount; ++i )
        {
            nn::fs::CloseFile(params[i].handle);
            nn::fs::DeleteFile(g_FilePath[i]);
        }

        // スレッドを破棄する
        for( int i = 0; i < threadCount; ++i )
        {
            nn::os::DestroyThread(&threads[i]);
        }
    }
}

//! @brief FAT で断片化したファイルの読出しにかかる時間の計測
TEST_F(PerformanceTestFatFileSystem, ReadFatFragment)
{
    NN_LOG("\nFat FS Fragmentation Read Test\n");

    int fragmentCounts[] = { 16, 64, 256, 1 * 1024, 4 * 1024, 16 * 1024 };

    g_PerformanceResultsRead.clear();

    for( auto fragmentCount : fragmentCounts )
    {
        // FATにマウント
        const size_t FragmentSize = 16 * 1024;
        const size_t totalSize = FileSize + FragmentSize * fragmentCount;

        MountFat(totalSize + FatManageSize);

        // ディレクトリを作成する
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(FatDirectoryPath));

        for( int bufferSize = BufferSizeMin ; bufferSize <= BufferSizeMax ; bufferSize *= 2 )
        {
            nn::fs::FileHandle handle;
            nn::fs::FileHandle fragmentHandle;

            // 断片化するバッファサイズを指定する
            const size_t FragmentbufferSize = FileSize / fragmentCount;

            // ファイルを断片化させる
            FragmentTest(&handle, &fragmentHandle, FragmentbufferSize, FragmentSize, FatFilePath, FragmentFilePath, FileSize);

            // 断片化用のファイルを閉じる
            nn::fs::CloseFile(fragmentHandle);

            // 断片化させたファイルを読み込む
            ThreadInfo param;
            param.handle = handle;
            param.threadNumber = 0;
            param.threadCount = 1;
            param.fileSize = FileSize;

            param.bufferSize = bufferSize;
            ReadTest(&param);

            // 断片化させたファイルを閉じる
            nn::fs::CloseFile(handle);
            nn::fs::DeleteFile(FatFilePath);
            nn::fs::DeleteFile(FragmentFilePath);
        }

        // ディレクトリを削除し、アンマウントする
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(FatDirectoryPath));
        nn::fs::Unmount("fat");
    }

    // Read 結果出力
    NN_LOG("                                   ");
    for( auto fragmentCount : fragmentCounts )
    {
        NN_LOG("     Fragmentation Count : %6d", fragmentCount);
    }
    NN_LOG("\n");
    nnt::fs::util::DumpPerformanceResults(g_PerformanceResultsRead);
}

//! @brief FAT で断片化したファイルの読み書き速度の測定を複数スレッドで並列同時実行した場合の時間計測
TEST_F(PerformanceTestFatFileSystem, ReadWriteFatMultiThreadFragment)
{
    int fragmentCounts[] = { 16, 64, 256, 1 * 1024, 4 * 1024, 16 * 1024 };

#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    int threadCounts[] = { 2, 4 };
#else
    int threadCounts[] = { 2, 4, 8 };
#endif

    for( auto fragmentCount : fragmentCounts )
    {
        NN_LOG("\nMulti Thread Fat FS Fragmentation Read Write Test\n");
        g_PerformanceResultsRead.clear();
        g_PerformanceResultsWrite.clear();

        for( auto threadCount : threadCounts )
        {
            // FATでマウントする
            const size_t totalSize = FileSize * threadCount;
            MountFat(totalSize + FatManageSize);

            // ディレクトリを作成する
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(FatDirectoryPath));

            nnt::fs::util::Vector <ThreadInfo> params(threadCount);

            // Write スレッド
            TestThread(params, threadCount, ThreadTestType_Write, fragmentCount);

            // Read スレッド
            TestThread(params, threadCount, ThreadTestType_Read, fragmentCount);

            // FATをアンマウント
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(FatDirectoryPath));
            nn::fs::Unmount("fat");
        }

        // Write 結果出力
        NN_LOG("Fragmentation Count : %d\n", fragmentCount);
        NN_LOG("Write Test\n");
        NN_LOG("                                   ");
        for( auto threadCount : threadCounts )
        {
            NN_LOG("                 threadCount : %2d", threadCount);
        }
        NN_LOG("\n");
        nnt::fs::util::DumpPerformanceResults(g_PerformanceResultsWrite);

        // Read 結果出力
        NN_LOG("Read Test\n");
        NN_LOG("                                   ");
        for( auto threadCount : threadCounts )
        {
            NN_LOG("                 threadCount : %2d", threadCount);
        }
        NN_LOG("\n");
        nnt::fs::util::DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n");
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nnt::fs::util::SetFsTestPerformanceConfiguration();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    nn::fssrv::MemoryResourceFromStandardAllocator allocator(&g_BufferAllocator);
    nn::fat::FatFileSystem::SetAllocatorForFat(&allocator);

    // MmcPatrol 休止
    nn::fs::SuspendMmcPatrol();

    auto testResult = RUN_ALL_TESTS();

    // MmcPatrol 再開
    nn::fs::ResumeMmcPatrol();

    if( nnt::fs::util::CheckMemoryLeak() )
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
