﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

const int ReadFileStep = 16 * 1024;
const char MountHostName[] = "host";

// host 側のマウントパスを取得します。
nn::Result GetMountHostPath(char* pOutPath, size_t outSize) NN_NOEXCEPT
{
    static const char MountFilePath[] = "rom:/mount.txt";

    nn::fs::FileHandle file;

    NN_RESULT_TRY(nn::fs::OpenFile(&file, MountFilePath, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            // MountFilePath なし
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    // ファイルを読み込んでマウント先のパスを取得
    size_t size = 0;
    NN_RESULT_DO(nn::fs::ReadFile(&size, file, 0, pOutPath, outSize - 1));

    pOutPath[size] = '\0';

    // 末尾の改行コード等を '\0' に置換
    for( int i = static_cast<int>(size) - 1; 0 < i; --i )
    {
        if( !std::iscntrl(pOutPath[i]) )
        {
            break;
        }
        pOutPath[i] = '\0';
    }

    NN_RESULT_SUCCESS;
}

// ファイルの読み込みを行う関数オブジェクト
class FileReader
{
public:
    nn::Result operator()(const char* filePath, int64_t fileSize) NN_NOEXCEPT
    {
        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        // ハッシュのブロック単位で読み込みが発生するようにする
        for( int64_t offset = 0; offset < fileSize; offset += ReadFileStep )
        {
            const auto size =
                static_cast<size_t>(std::min<int64_t>(sizeof(m_Buffer), fileSize));

            NN_RESULT_DO(nn::fs::ReadFile(file, offset, m_Buffer, size));
        }

        NN_RESULT_SUCCESS;
    }

private:
    char m_Buffer[256];
};

// ファイルの比較を行う関数オブジェクト
class FileComparer
{
public:
    FileComparer() NN_NOEXCEPT
        : m_Buffer1(ReadFileStep)
        , m_Buffer2(ReadFileStep)
    {
    }

    nn::Result operator()(const char* romPath, int64_t fileSize) NN_NOEXCEPT
    {
        nn::fs::FileHandle romFile;
        NN_RESULT_DO(
            nn::fs::OpenFile(&romFile, romPath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(romFile);
        };

        // host 側のファイル
        nnt::fs::util::String hostPath(romPath);
        hostPath.replace(0, 3, MountHostName);

        nn::fs::FileHandle hostFile;
        NN_RESULT_DO(
            nn::fs::OpenFile(&hostFile, hostPath.c_str(), nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(hostFile);
        };

        // ファイルを比較
        for( int64_t offset = 0; offset < fileSize; )
        {
            const auto size =
                static_cast<size_t>(std::min<int64_t>(ReadFileStep, fileSize - offset));

            NN_RESULT_DO(nn::fs::ReadFile(romFile, offset, m_Buffer1.data(), size));
            NN_RESULT_DO(nn::fs::ReadFile(hostFile, offset, m_Buffer2.data(), size));
            offset += size;

            if( std::memcmp(m_Buffer1.data(), m_Buffer2.data(), size) != 0 )
            {
                nnt::fs::util::DumpBufferDiff(m_Buffer1.data(), m_Buffer2.data(), size);
                return nn::fs::ResultUnexpected();
            }
        }

        NN_RESULT_SUCCESS;
    }

private:
    nnt::fs::util::Vector<char> m_Buffer1;
    nnt::fs::util::Vector<char> m_Buffer2;
};

}

// ディレクトリを巡回してすべてのファイルをログ出力します。
TEST(PatchAddRomTest, Traverse)
{
    size_t cacheBufferSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheBufferSize));

    auto cacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);
    ASSERT_NE(nullptr, cacheBuffer);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom("rom", cacheBuffer.get(), cacheBufferSize));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    static const char MountRomPath[] = "rom:/";

    char mountHostPath[512] = {};
    NNT_ASSERT_RESULT_SUCCESS(GetMountHostPath(mountHostPath, sizeof(mountHostPath)));

    // ファイルの読み込みのみ
    if( mountHostPath[0] == '\0' )
    {
        NN_LOG("call TreverseDirectory<FileReader>();\n");

        // TODO: 再帰で処理するがスタックの状況次第で別の方法を考える
        NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::TraverseDirectory<FileReader>(MountRomPath));
    }
    // ホスト側とファイル比較
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(MountHostName, mountHostPath));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountHostName);
        };

        NN_LOG("call TreverseDirectory<FileComparer>();\n");

        NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::TraverseDirectory<FileComparer>(MountRomPath));
    }
}

// Rom がないことをテストします。
TEST(PatchAddRomTest, NoRom)
{
    size_t cacheBufferSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheBufferSize));

    auto cacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);
    ASSERT_NE(nullptr, cacheBuffer);

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultPartitionNotFound,
        nn::fs::MountRom("rom", cacheBuffer.get(), cacheBufferSize));
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nn::fs::SetEnabledAutoAbort(false);

    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
