﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/nn_Log.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/nn_Assert.h>
#include <nn/os/os_Thread.h>
#include <random>
#include <nn/fs/fs_PriorityPrivate.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/aoc.h>

using namespace nn::fs;

namespace nnt { namespace fs { namespace api {

    struct AgingParamsStructure {
        int32_t type;
        std::string mountName;
        std::string directoryName;
        std::string fileName;
        int64_t allowSize;
        int32_t threadCount;
    };

}}}

namespace {

    int64_t g_ThreadLoopCount[1] = {0x00};            // 各スレッドのループ数
    bool g_testEndflg = false;
    static char writeBuffers[1][5 * 1024 * 1024 + 1];
    static char readBuffers[1][5 * 1024 * 1024 + 1];
    nn::os::ThreadType g_Threads[1]; // スレッド
    static const size_t ThreadStackSize = 64 * 1024;     // スタックサイズ
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_ThreadStacks[1][ThreadStackSize]; // スタックバッファ

    int64_t GetRandomNum(int64_t range) NN_NOEXCEPT
    {
        static int seed = 1;
        std::mt19937 mt(seed++);
        return std::uniform_int_distribution<int64_t>(1, range)(mt);
    }

    struct TypesStructure
    {
        int32_t type;
        char Mark[20];
    };
    struct filesStructure
    {
        int64_t size;
        char Name[40];
    };
    enum
    {
        OFFSET_TYPE_FIX,
        OFFSET_TYPE_SEAQUENTIIAL,
        OFFSET_TYPE_RANDOM,
    };
    enum
    {
        SIZE_TYPE_READ_WRITE_FIX,
        SIZE_TYPE_READ_RANDOM,
        SIZE_TYPE_WRITE_RANDOM,
        SIZE_TYPE_READ_WRITE_RANDOM,
    };

    struct ThreadData {
        int32_t num;
        int32_t threadCount;
        int64_t priority;
    };
    ThreadData param[1];                              // パラメータ

    enum MountType
    {
        RomFileSystem        = 0,
        End,
    };

    // 各パラメータ
    struct MountParamsStructure {
        int32_t type;
        std::string mountName;
        std::string directoryName;
        std::string fileName;
        int32_t mountCount;
        int32_t verifyFlag;
    };
    const MountParamsStructure MountParams[] = {
      // | type                            | mountName     | directoryName              | fileName     | mountCount | verifyFlag
        {   MountType::RomFileSystem,        "rom",         "TestDirectoryrom",          "AgingTest.file"       ,9 , 1  },
    };

    #define NNT_EXPECT_RESULT_SUCCESS_AGING(exe, param, count) {\
        nn::Result innerResult;\
        NNT_EXPECT_RESULT_SUCCESS(innerResult = exe);\
        if(innerResult.IsFailure())\
        {\
            NN_LOG("Failure Thread[%d:%d] count = %lld\n", param->type, param->threadCount, count);\
        }\
    }

    #define EXPECT_EQ_AGING(arg1, arg2, param, count) {\
        EXPECT_EQ(arg1, arg2);\
        if((arg1) != (arg2))\
        {\
            NN_LOG("Failure Thread[%d:%d] count = %lld\n", param->type, param->threadCount, count);\
        }\
    }


    //!< @brief 各ファイルシステム・マウント
    void CreateMountFileSystem(ThreadData data)
    {
        std::string mountName = MountParams[data.num].mountName;
        if (MountParams[data.num].mountCount > 1 )
        {
            mountName.append(std::to_string(data.threadCount).c_str());
        }

        NN_LOG("CreateMountFileSystem thread=%d, mountName = %s\n", data.threadCount, mountName.c_str());

        size_t cacheSize = 0;
        static char* cacheBuffer;

        switch (MountParams[data.num].type)
        {
        case MountType::RomFileSystem:
            nn::fs::QueryMountRomCacheSize(&cacheSize);
            cacheBuffer = new(std::nothrow) char[cacheSize];
            NN_ASSERT_NOT_NULL(cacheBuffer);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom(mountName.c_str(), cacheBuffer, cacheSize));
            break;
        default:
            break;
        }
    }

    //!< @brief 各ファイルシステム・アンマウント
    void unmountFileSystem(ThreadData data)
    {

        std::string mountName = MountParams[data.num].mountName;
        if (MountParams[data.num].mountCount > 1)
        {
            mountName.append(std::to_string(data.threadCount).c_str());
        }
        nn::fs::Unmount(mountName.c_str());

    }

    //!< @brief エージング用の情報を生成する
    void CreateAgingParams(ThreadData data, nnt::fs::api::AgingParamsStructure *agingParams)
    {

        agingParams->type           = MountParams[data.num].type;
        agingParams->threadCount    = data.threadCount;
        agingParams->mountName      = MountParams[data.num].mountName;
        agingParams->directoryName  = MountParams[data.num].mountName;
        agingParams->fileName       = MountParams[data.num].mountName;
        std::string rootpath = MountParams[data.num].mountName;

        // 各マウントのマウント名、フォルダ名、ファイル名を生成
        rootpath.append(std::to_string(data.threadCount).c_str()).append(":/");
        agingParams->mountName.append(std::to_string(data.threadCount).c_str());
        agingParams->fileName.append(std::to_string(data.threadCount).c_str()).append(":/").append(MountParams[data.num].directoryName.c_str()).append("/").append(MountParams[data.num].fileName.c_str());
        agingParams->directoryName.append(std::to_string(data.threadCount).c_str()).append(":/").append(MountParams[data.num].directoryName.c_str());

    }


    void RoReadFileWriteFileVerify(nnt::fs::api::AgingParamsStructure* arg, TypesStructure sizeType, TypesStructure offsetType, int64_t filesize, int64_t buffer, int64_t offset, size_t size, FileHandle handle, void* pWriteBuffer, void* pReadBuffer)
    {
        size_t  workSize = size;
        int64_t workOffset = offset;
        char* p_write = static_cast<char*>(pWriteBuffer);
        size_t readSize;

        // リードファイル・ループ
        workSize = size;
        workOffset = offset;
        p_write = static_cast<char*>(pWriteBuffer);
        for(size_t remainSize = size;  remainSize>0 ; remainSize-=workSize)
        {
            if( sizeType.type == SIZE_TYPE_READ_RANDOM )
            {
                workSize = GetRandomNum(remainSize);
            }
            NNT_EXPECT_RESULT_SUCCESS_AGING(ReadFile(handle, workOffset, pReadBuffer, workSize), arg, g_ThreadLoopCount[arg->threadCount]);

            NNT_EXPECT_RESULT_SUCCESS_AGING(ReadFile(handle, workOffset, pReadBuffer, workSize, ReadOption()), arg, g_ThreadLoopCount[arg->threadCount]);

            NNT_EXPECT_RESULT_SUCCESS_AGING(ReadFile(&readSize, handle, workOffset, pReadBuffer, workSize), arg, g_ThreadLoopCount[arg->threadCount]);
            EXPECT_EQ_AGING(readSize, workSize, arg, g_ThreadLoopCount[arg->threadCount]);

            NNT_EXPECT_RESULT_SUCCESS_AGING(ReadFile(&readSize, handle, workOffset, pReadBuffer, workSize, ReadOption()), arg, g_ThreadLoopCount[arg->threadCount]);
            EXPECT_EQ_AGING(readSize, workSize, arg, g_ThreadLoopCount[arg->threadCount]);
            workOffset += workSize;
            p_write += workSize;
        }

    }

    void RoReadFileWriteFileVerifyLoop(nnt::fs::api::AgingParamsStructure* arg, void* pReadBuffer, void* pWriteBuffer, int64_t *cnt, int64_t *offsetWork, int64_t *fileRemainSize,
        size_t size, int64_t buffer, TypesStructure sizeType, TypesStructure offsetType, filesStructure filesize, int64_t offset, FileHandle handle)
    {
        size_t  workSize = size > buffer ? buffer : size;
        workSize = workSize > *fileRemainSize ? *fileRemainSize : workSize;
        int64_t remainSize = workSize;
        *fileRemainSize = *fileRemainSize - workSize;

        // バッファ・サイズ制限ループ
        do
        {
            if( sizeType.type == SIZE_TYPE_READ_WRITE_RANDOM )
            {
                workSize = GetRandomNum(workSize);
            }
            if( offsetType.type == OFFSET_TYPE_RANDOM )
            {
                *offsetWork = GetRandomNum(filesize.size - workSize);
                if(filesize.size - workSize == 0)
                {
                    *offsetWork = 0;
                }
                remainSize = 0;
            }
            RoReadFileWriteFileVerify(arg, sizeType, offsetType, filesize.size, buffer, *offsetWork, workSize, handle, pWriteBuffer, pReadBuffer);

            if( offsetType.type == OFFSET_TYPE_SEAQUENTIIAL )
            {
                remainSize = remainSize - workSize;
                *offsetWork = *offsetWork + static_cast<int64_t>(workSize);
            }
            workSize = static_cast<size_t>(remainSize) < workSize ? static_cast<size_t>(remainSize) : workSize;

            // 残数値更新
            if(*cnt>=10)
            {
                break;
            }
            *cnt = *cnt + 1;
        } while (remainSize > 0);  // end of while
    }

    void RoVerifyFunc(nnt::fs::api::AgingParamsStructure* arg)
    {
        // メモリチェック回避
        filesStructure filesizes[] = {{5 * 1024 * 1024 + 1, ":/TestDirectoryrom/dir14/file6.dat"},};
        int64_t buffers[] = {4 * 1024 * 1024 - 1};
        size_t  sizes[]     = { 512 * 1024 - 1};
        int64_t offsets[]   = {0};

        TypesStructure offsetTypes[]   = {{OFFSET_TYPE_SEAQUENTIIAL, "OFFSET_SEAQUENTIIAL"}};
        TypesStructure sizeTypes[]     = {{SIZE_TYPE_READ_WRITE_FIX, "SIZE_FIX"}};

        g_ThreadLoopCount[arg->threadCount] = 0;
        //while(g_testEndflg == false)
        {
            g_ThreadLoopCount[arg->threadCount]++;
            // シナリオループ
            for(auto offsetType : offsetTypes)
            {
                for(auto sizeType : sizeTypes)
                {
                    // ファイル・サイズループ
                    for(filesStructure filesize : filesizes)
                    {
                        if(g_testEndflg != false)
                        {
                            return;
                        }
                        // romマウント以外HeavyResourcesのテストを実施しない
                        if(arg->type != MountType::RomFileSystem && filesize.size > 1024LL * 1024LL * 1024LL)
                        {
                            break;
                        }
                        // ファイル作成・オープン
                        std::string fileName = arg->mountName + filesize.Name;
                        FileHandle handle;
                        NNT_EXPECT_RESULT_SUCCESS_AGING(OpenFile(&handle, fileName.c_str(), OpenMode_Read ), arg, g_ThreadLoopCount[arg->threadCount]);
                        // バッファ・サイズループ
                        for(int64_t buffer : buffers)
                        {
                            if(g_testEndflg != false)
                            {
                                break;
                            }
                            // オフセット・サイズループ
                            for(int64_t offset : offsets)
                            {
                                if(g_testEndflg != false)
                                {
                                    break;
                                }
                                // サイズループ
                                for(size_t size : sizes)
                                {
                                    if(g_testEndflg != false)
                                    {
                                        break;
                                    }
                                    int64_t offsetWork = offset;
                                    //NN_LOG("Thread [%d:%d](%s) (%s / %s) filesize: %10lld(0x%8x) buffer: %10lld(0x%8x) offset: %10lld(0x%8x) size: %10lld(0x%8x) \n",
                                    //arg->type, arg->threadCount, arg->mountName.c_str(), offsetType.Mark, sizeType.Mark, filesize.size, filesize.size, buffer, buffer, offsetWork, offsetWork, static_cast<int64_t>(size), static_cast<int64_t>(size));
                                    int64_t fileRemainSize = static_cast<int64_t>(filesize.size) - offsetWork;
                                    int64_t cnt = 0;

                                    if( offsetType.type == OFFSET_TYPE_FIX )
                                    {
                                        cnt = 10;
                                    }
                                    if(offsetWork >= filesize.size)
                                    {
                                        break;
                                    }
                                    if(offsetWork + size > filesize.size)
                                    {
                                        size = filesize.size - offsetWork;
                                    }
                                    // ファイルサイズまでループ
                                    do
                                    {
                                        RoReadFileWriteFileVerifyLoop(arg, &readBuffers[arg->type], &writeBuffers[arg->type], &cnt, &offsetWork, &fileRemainSize, size, buffer, sizeType, offsetType, filesize, offset, handle);
                                        // 残数値更新
                                        if(cnt>=10 || g_testEndflg != false)
                                        {
                                            break;
                                        }
                                    } while (fileRemainSize > 0);  // end of while
                                    if(offsetWork >= filesize.size)
                                    {
                                        continue;
                                    }
                                }
                            }
                        }
                        CloseFile(handle);
                        NNT_EXPECT_RESULT_SUCCESS_AGING(Commit(arg->mountName.c_str()), arg, g_ThreadLoopCount[arg->threadCount]);
                    }
                }
            }
        }
    }


    //!< @brief テストスレッド
    void TestThread(void* arg) NN_NOEXCEPT
    {
        ThreadData* threaditem = static_cast<ThreadData*>(arg);

        switch(threaditem->priority)
        {
        case Priority::Priority_Realtime:
            // RealTime
            nn::fs::SetPriorityOnCurrentThread(Priority::Priority_Realtime);
            break;
        case Priority::Priority_Normal:
            // Normal
            nn::fs::SetPriorityOnCurrentThread(Priority::Priority_Normal);
            break;
        case Priority::Priority_Low:
            // Low
            nn::fs::SetPriorityOnCurrentThread(Priority::Priority_Low);
            break;
        case PriorityRaw::PriorityRaw_Background:
            // Background(Raw)
            nn::fs::SetPriorityRawOnCurrentThread(PriorityRaw::PriorityRaw_Background);
            break;
        default:
            // 設定なし
            break;
        }

        nnt::fs::api::AgingParamsStructure agingParams;
        CreateAgingParams(*threaditem, &agingParams);
        //NN_LOG("TestThread type           = %d\n"  , agingParams.type);
        //NN_LOG("TestThread mountName      = %s\n"  , agingParams.mountName.c_str());
        //NN_LOG("TestThread directoryName  = %s\n"  , agingParams.directoryName.c_str());
        //NN_LOG("TestThread fileName       = %s\n"  , agingParams.fileName.c_str());
        //NN_LOG("TestThread allowSize      = %lld\n", agingParams.allowSize);
        //NN_LOG("TestThread threadcount    = %d\n"  , agingParams.threadCount);
        //NN_LOG("TestThread priority       = %lld\n"  , static_cast<int64_t>(GetPriorityRawOnCurrentThread()));

        RoVerifyFunc(&agingParams);

        unmountFileSystem(*threaditem);
        //NN_LOG("testshread end threadcount = %d", threaditem->threadCount);
    }

    void StartMountThread()
    {
        int64_t idealCoreNumber;
        int64_t i = 0;

        // スレッド開始
        param[0].num = i;
        param[0].threadCount = 0;
        param[0].priority = 2;

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        CreateMountFileSystem(param[0]);
        nnt::fs::api::AgingParamsStructure agingParams;
        CreateAgingParams(param[0], &agingParams);
        idealCoreNumber = 1;
        //NN_LOG("%3d CreateThread() ThreadsNum[%02lld](%15s) core(%lld) \n", 0, i, agingParams.mountName.c_str(), idealCoreNumber);
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&g_Threads[0], TestThread, &param[0]
                                , g_ThreadStacks[0], ThreadStackSize, nn::os::DefaultThreadPriority, idealCoreNumber));

        nn::os::StartThread(&g_Threads[i]);
    }


    //!< @brief
    TEST(Regression, SIGLO81451)
    {

        StartMountThread();

        nn::os::WaitThread(&g_Threads[0]);
        nn::os::DestroyThread(&g_Threads[0]);

    }
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    size_t fileDataCacheSize = 32 * 1024 * 1024;
    void* fileDataCache = std::malloc(fileDataCacheSize);
    nn::fs::EnableGlobalFileDataCache(fileDataCache, fileDataCacheSize);

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
