﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/crypto/crypto_Sha256Generator.h>

using namespace nn;

/* ダンプされたセーブデータのハッシュを計算し、ハッシュファイルを作成 */
/*
 * HostSaveDataPath 以下にディレクトリ名が <saveDataId> のディレクトリがあるか探す。
 * "#"から始まるディレクトリはスキップする。
 * <saveDataId>/<Count> ディレクトリ下のセーブデータイメージファイルのハッシュファイルを
 * "AllocationTableControlArea.hash", "AllocationTableMeta.hash", "AllocationTableData.hash", "Raw.hash" の名前で作成する。
 */

namespace {

const int MaxPathLength = 256;
const char HostSaveDataPath[] = {"D:/TempSave"};
const size_t HashBlockSize = 64 * 1024;

const int BufferSize = 8 * 1024 * 1024;

}

bool ExistsDirectory(const char* path)
{
    nn::fs::DirectoryHandle outValue;
    auto result = nn::fs::OpenDirectory(&outValue, path, nn::fs::OpenDirectoryMode_All);
    if (result.IsFailure())
    {
        return false;
    }
    nn::fs::CloseDirectory(outValue);
    return true;
}

nn::Result DumpFileHash(const char* srcFilePath, const char* dstFilePath) NN_NOEXCEPT
{
    nn::fs::DeleteFile(dstFilePath);
    NN_RESULT_DO(nn::fs::CreateFile(dstFilePath, 0));

    nn::fs::FileHandle srcHandle;
    nn::fs::FileHandle dstHandle;
    NN_RESULT_DO(nn::fs::OpenFile(&srcHandle, srcFilePath, nn::fs::OpenMode_Read));
    NN_RESULT_DO(nn::fs::OpenFile(&dstHandle, dstFilePath, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

    int64_t srcFileSize = 0;
    nn::fs::GetFileSize(&srcFileSize, srcHandle);

    int64_t curerntOffset = 0;
    int64_t restSize = srcFileSize;
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);

    int64_t writeOffset = 0;
    while (restSize > 0)
    {
        size_t accessSize = static_cast<size_t>(std::min(restSize, static_cast<int64_t>(BufferSize)));
        NN_RESULT_DO(nn::fs::ReadFile(srcHandle, curerntOffset, buffer.get(), accessSize));

        auto pHashTarget = reinterpret_cast<char*>(buffer.get());
        size_t size = accessSize;
        while (size > 0)
        {
            auto hashTargetSize = std::min(size, HashBlockSize);
            char hash[nn::crypto::Sha256Generator::HashSize];
            nn::crypto::GenerateSha256Hash(hash, sizeof(hash), pHashTarget, hashTargetSize);

            NN_RESULT_DO(nn::fs::WriteFile(dstHandle, writeOffset, hash, nn::crypto::Sha256Generator::HashSize, nn::fs::WriteOption::MakeValue(0)));
            writeOffset += nn::crypto::Sha256Generator::HashSize;

            size -= hashTargetSize;
            pHashTarget += hashTargetSize;
        }

        curerntOffset += accessSize;
        restSize -= accessSize;
    }
    NN_RESULT_DO(nn::fs::FlushFile(dstHandle));
    nn::fs::CloseFile(dstHandle);
    nn::fs::CloseFile(srcHandle);

    NN_RESULT_SUCCESS;
}

void DumpSaveDataHash(const char* srcBaseDir)
{
    int count = 0;
    while (1)
    {
        char srcDir[MaxPathLength];
        nn::util::SNPrintf(srcDir, sizeof(srcDir), "%s%d", srcBaseDir, count);
        if (!ExistsDirectory(srcDir))
        {
            break;
        }
        count++;
    }

    for (int i = 0; i < count; i++)
    {
        const char* Paths[] =
        {
            "AllocationTableControlArea",
            "AllocationTableMeta",
            "AllocationTableData",
            "Raw",
        };
        for (auto entryName : Paths)
        {
            char srcFilePath[MaxPathLength];
            char dstFilePath[MaxPathLength];
            nn::util::SNPrintf(srcFilePath, sizeof(srcFilePath), "%s%d/%s", srcBaseDir, i, entryName);
            nn::util::SNPrintf(dstFilePath, sizeof(dstFilePath), "%s.hash", srcFilePath);

            NN_LOG("%s\n", srcFilePath);
            DumpFileHash(srcFilePath, dstFilePath);
        }
    }
}

void SearchHashDirectory(const char* srcBaseDir)
{
    nn::fs::DirectoryHandle srcHandle;
    nn::fs::DirectoryEntry entryBuffer;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&srcHandle, srcBaseDir, nn::fs::OpenDirectoryMode_Directory));
    while (1)
    {
        int64_t outValue = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&outValue, &entryBuffer, srcHandle, 1));
        if (outValue == 0)
        {
            break;
        }

        if (entryBuffer.name[0] == '#')
        {
            continue;
        }

        bool isSaveData = true;
        int i;
        for (i = 0; i < strlen(entryBuffer.name); i++)
        {
            if ((entryBuffer.name[i] < '0' && entryBuffer.name[i] > '9') && (entryBuffer.name[i] < 'a' && entryBuffer.name[i] > 'f') && (entryBuffer.name[i] < 'A' && entryBuffer.name[i] > 'F'))
            {
                isSaveData = false;
                break;
            }
        }
        if (i != 16)
        {
            isSaveData = false;
        }

        char srcDir[MaxPathLength];
        nn::util::SNPrintf(srcDir, sizeof(srcDir), "%s%s/", srcBaseDir, entryBuffer.name);
        if (!isSaveData)
        {
            SearchHashDirectory(srcDir);
            continue;
        }

        DumpSaveDataHash(srcDir);
    }
    nn::fs::CloseDirectory(srcHandle);
}

TEST(SaveData, TestSaveDataHash)
{
    const char* MountName = "host";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(MountName, HostSaveDataPath));

    char srcBaseDir[256];
    nn::util::SNPrintf(srcBaseDir, sizeof(srcBaseDir), "%s:/", MountName);

    SearchHashDirectory(srcBaseDir);

    nn::fs::Unmount(MountName);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
