﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>

const char* MountName = "testsave";
const char* ReadTestFileName = "readtest.bin";
const char* ReadTestFileData = "ReadOK!!";
const nn::Bit64 ApplicationIdValue               = 0x0100000000010000;
const nn::Bit64 OtherApplicationIdValueAccess    = 0x0100000000010001;
const nn::Bit64 OtherApplicationIdValueNotAccess = 0x0100000000010002;
const size_t TestSaveDataSize        = 1024 * 1024;
const size_t TestSaveDataJournalSize = 1024 * 1024;

static bool MountFindSaveData(nn::fs::SaveDataInfo* pOutValue, nn::fs::SaveDataSpaceId spaceId)
{
    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, spaceId));

    int64_t count;
    nn::fs::SaveDataInfo info;
    while (NN_STATIC_CONDITION(true))
    {
        count = 0;
        NNT_EXPECT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
        if (count == 0)
        {
            return false;
        }
        if (info.applicationId.value == ApplicationIdValue ||
            info.applicationId.value == OtherApplicationIdValueAccess ||
            info.applicationId.value == OtherApplicationIdValueNotAccess)
        {
            *pOutValue = info;
            return true;
        }
    }
}

static void DeleteAllTestSaveData()
{
    const nn::fs::SaveDataSpaceId SpaceIds[] = {nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System};
    for (auto spaceId : SpaceIds)
    {
        while (NN_STATIC_CONDITION(true))
        {
            nn::fs::SaveDataInfo info;
            if (MountFindSaveData(&info, spaceId) == false)
            {
                break;
            }
            auto saveId = info.saveDataId;
            NN_LOG("delete save data (%x, %llx).\n", spaceId, saveId);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteSaveData(saveId));
        }
    }
}

class TestMountSaveData_Setup : public ::testing::Test
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        // アカウントライブラリを初期化
        nn::account::Initialize();

        nn::Result result;
        g_AccountUid = nn::account::InvalidUid;
        int userCount = 0;
        result = nn::account::ListAllUsers(&userCount, &g_AccountUid, 1);
        NN_ASSERT(result.IsSuccess() && userCount > 0);

        m_UserId._data[0] = g_AccountUid._data[0];
        m_UserId._data[1] = g_AccountUid._data[1];
    }
    static void TearDownTestCase() NN_NOEXCEPT
    {
    }

    static nn::account::Uid g_AccountUid;
    static nn::fs::UserId m_UserId;
};

nn::account::Uid TestMountSaveData_Setup::g_AccountUid;
nn::fs::UserId TestMountSaveData_Setup::m_UserId;

void CreateTestSaveData(nn::Bit64 applicationIdValue, nn::fs::UserId userId, uint64_t ownerId)
{
    nn::ncm::ApplicationId applicationId = {applicationIdValue};
    nn::fs::FileHandle handle;

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(applicationId, userId, ownerId, TestSaveDataSize, TestSaveDataJournalSize, 0));

    char path[256];
    nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountName, ReadTestFileName);

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, applicationId, userId));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(path, 0));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(handle, 0, ReadTestFileData, sizeof(ReadTestFileData), nn::fs::WriteOption::MakeValue(0)));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::FlushFile(handle));
    nn::fs::CloseFile(handle);
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    nn::fs::Unmount(MountName);
}

TEST_F(TestMountSaveData_Setup, CreateTestSaveData)
{
    DeleteAllTestSaveData();

    CreateTestSaveData(ApplicationIdValue, m_UserId, ApplicationIdValue);
    CreateTestSaveData(OtherApplicationIdValueAccess, m_UserId, OtherApplicationIdValueAccess);
    CreateTestSaveData(OtherApplicationIdValueNotAccess, m_UserId, OtherApplicationIdValueNotAccess);

    {
        static const char* MountName = "mount";
        static nn::ApplicationId ApplicationId = { OtherApplicationIdValueAccess };

        {
            NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSaveData(MountName, g_AccountUid));
            nn::fs::Unmount(MountName);
        }
        {
            NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSaveData(MountName, ApplicationId, g_AccountUid));
            nn::fs::Unmount(MountName);
        }
        {
            NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSaveDataReadOnly(MountName, ApplicationId, g_AccountUid));
            nn::fs::Unmount(MountName);
        }
        {
            NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData(MountName, 0x8000000000000001, g_AccountUid));
            nn::fs::Unmount(MountName);
        }
    }
}

TEST_F(TestMountSaveData_Setup, DeleteTestSaveData)
{
    DeleteAllTestSaveData();
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
