﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/fs/fs_MountPrivate.h>

#include "testFs_FsLib_Mount.h"

namespace
{
    void CheckIfOneEntryExist(const char* dirName, const char* fileName, bool isDir) NN_NOEXCEPT
    {
        const int NameLengthMax = 259;
        nn::fs::DirectoryHandle dirHandle;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenDirectory(&dirHandle, dirName, isDir ? nn::fs::OpenDirectoryMode_Directory : nn::fs::OpenDirectoryMode_File));
        int64_t readNum = 0;
        const int entryBufferNum = 1;
        nn::fs::DirectoryEntry dirEntry[entryBufferNum];
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&readNum, dirEntry, dirHandle, entryBufferNum));
        EXPECT_EQ(1, readNum);
        if( readNum > 0 )
        {
            EXPECT_TRUE(strnlen(dirEntry[0].name, NameLengthMax) == strnlen(fileName, NameLengthMax));
            EXPECT_TRUE(std::strncmp(dirEntry[0].name, fileName, NameLengthMax) == 0);
        }
        nn::fs::CloseDirectory(dirHandle);
    }

    // Todo: 別テストプログラムの境界テストに移動
    void CreateAndRenameAndDeleteLongNameEntryTest(const char* testDirPath) NN_NOEXCEPT
    {
        // testDirPath = 49 chars
        const char* fileName = "longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglon.txt"; // 210 chars
        const char* dirName = "longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglon"; // 198 chars
        const char* fileNameR = "longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglongloR.txt"; // 210 chars
        const char* dirNameR = "longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglongloR"; // 198 chars

        nnt::fs::util::String filePath(testDirPath);
        filePath += fileName;
        nnt::fs::util::String dirPath(testDirPath);
        dirPath += dirName;
        //    NN_LOG("strlen = %d\n", strlen(filePath.c_str())); // 259 chars
        //    NN_LOG("strlen = %d\n", strlen(dirPath.c_str())); // 247 chars

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(filePath.c_str(), 1024));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(dirPath.c_str()));

        CheckIfOneEntryExist(testDirPath, fileName, false);
        CheckIfOneEntryExist(testDirPath, dirName, true);

        nnt::fs::util::String filePathR(testDirPath);
        filePathR += fileNameR;
        nnt::fs::util::String dirPathR(testDirPath);
        dirPathR += dirNameR;

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::RenameFile(filePath.c_str(), filePathR.c_str()));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::RenameDirectory(dirPath.c_str(), dirPathR.c_str()));

        CheckIfOneEntryExist(testDirPath, fileNameR, false);
        CheckIfOneEntryExist(testDirPath, dirNameR, true);

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(filePathR.c_str()));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectory(dirPathR.c_str()));
    }
}

TEST_F(MountFailure, MountHostPathNotFound)
{
    //    auto testDirPathStartSlash = nnt::fs::util::String("/");
    //    testDirPathStartSlash.append(g_TestDirPath.GetPath().c_str());
    //    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, nn::fs::MountHost("host", testDirPathStartSlash.c_str()));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, nn::fs::MountHost("host", "c:/notfound"));
}

TEST_F(MountFailure, MountHostRootAlreadyExist)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultMountNameAlreadyExists, nn::fs::MountHostRoot());
    nn::fs::UnmountHostRoot();
}

TEST_F(MountFailure, MountHostInvalidCharacter)
{
    const char* InvalidCharPaths[] =
    {
        "C:/:*?<>|",
        "*?<>|:/foo/bar",
    };
    for( auto invalidCharPath : InvalidCharPaths )
    {
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidCharacter,
            nn::fs::MountHost("mount", invalidCharPath)
        ) << "MountHost " << invalidCharPath;
    }
}

namespace
{
    const nnt::fs::util::MountTestAttribute GetAttribute() NN_NOEXCEPT
    {
        nnt::fs::util::MountTestAttribute attribute = {};
        attribute.isNotAbortedByTooLongPath = true;
        return attribute;
    }

    nn::Result MountHostForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountHost(mountName, g_TestDirPath.GetPath().c_str()));
        NN_RESULT_SUCCESS;
    }

    nn::Result MountHostForMountNameTest(const char* mountName, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountHost(mountName, path));
        NN_RESULT_SUCCESS;
    }

    const nnt::fs::util::MountTestParameter MountTestParameters[] = {
        { "MountHost", MountHostForMountNameTest, MountHostForMountNameTest, GetAttribute },
    };
}

NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountHost, ::testing::ValuesIn(MountTestParameters));

TEST(ConvertToFsCommonPath, MountHost)
{
    const nnt::fs::util::String UserMountName = "mount";

    const nnt::fs::util::String RootPaths[] =
    {
        nnt::fs::util::String(nnt::fs::util::GetHostTemporaryPath()),
        nnt::fs::util::String(nnt::fs::util::GetHostTemporaryPath()) + "/",
    };

    for( auto rootPath : RootPaths )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(UserMountName.c_str(), rootPath.c_str()));
        char commonPath[nn::fs::EntryNameLengthMax];
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ConvertToFsCommonPath(commonPath, sizeof(commonPath), (UserMountName + ":/path").c_str()));
        EXPECT_STREQ((nnt::fs::util::String() + "@Host:/" + nnt::fs::util::GetHostTemporaryPath() + "/path").c_str(), commonPath);
        nn::fs::Unmount(UserMountName.c_str());
    }
}

TEST(ConvertToFsCommonPath, MountHostRoot)
{
    const nnt::fs::util::String UserMountName = "mount";

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    char commonPath[nn::fs::EntryNameLengthMax];
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ConvertToFsCommonPath(commonPath, sizeof(commonPath), (nnt::fs::util::String() + nnt::fs::util::GetHostTemporaryPath() + "/path").c_str()));
    EXPECT_STREQ((nnt::fs::util::String() + "@Host:/" + nnt::fs::util::GetHostTemporaryPath() + "/path").c_str(), commonPath);
    nn::fs::UnmountHostRoot();
}

TEST_F(MountBasic, MountHost)
{
    auto testDirPathWithSlash = g_TestDirPath.GetPath() + "/";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("host", testDirPathWithSlash.c_str()));
    ValidateFileSystem("host:/");
    nn::fs::Unmount("host");

    auto testDirPathWithBackSlash = g_TestDirPath.GetPath() + "\\";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("host", testDirPathWithBackSlash.c_str()));
    ValidateFileSystem("host:/");
    nn::fs::Unmount("host");

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("host", g_TestDirPath.GetPath().c_str()));
    ValidateFileSystem("host:/");
    nn::fs::Unmount("host");

    auto unc = nnt::fs::util::String("\\\\localhost\\") + g_TestDirPath.GetPath();
    unc.replace(unc.find_first_of(':'), 1, "$");
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("host", unc.c_str()));
    ValidateFileSystem("host:/");
    nn::fs::Unmount("host");
}

TEST(Mount, MountHostDriveRoot)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("host", "c:/"));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("host");
    };

    // Cドライブルートは UAC が有効な場合、ファイル作成が失敗するため
    // ファイル、ディレクトリ列挙が行えるかを確認します。
    const char* dirName = "host:/";
    nn::fs::DirectoryHandle dirHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&dirHandle, dirName, nn::fs::OpenDirectoryMode_All));

    while( NN_STATIC_CONDITION(true) )
    {
        nn::fs::DirectoryEntry dirEntry;
        int64_t readNum = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&readNum, &dirEntry, dirHandle, 1));
        if( readNum == 0 )
        {
            break;
        }
    }
    nn::fs::CloseDirectory(dirHandle);
}


TEST_F(MountBasic, MountHostRoot)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    ValidateFileSystem(g_TestDirPath.GetPath().c_str());
    //    CreateAndRenameAndDeleteLongNameEntryTest(g_TestDirPath.GetPath().c_str());
    nn::fs::UnmountHostRoot();
}
