﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_MountPrivate.h>

#include "testFs_FsLib_Mount.h"

using namespace nn::fs;

class MountGameCardTest : public nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture
{
protected:

    virtual void SetUp()
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        MountHostRoot();

        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&xciHandle, g_TestDirPath.GetPath().append("/test.xci").c_str(), OpenMode_Read));
        int64_t xciSize;
        NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&xciSize, xciHandle));
        std::shared_ptr<IStorage> xciStorage(new FileHandleStorage(xciHandle));
        const int KeyAreaSize = 4 * 1024;
        xciBodyStorage.reset(new SubStorage(std::move(xciStorage), KeyAreaSize, xciSize - KeyAreaSize)); // skip keyarea

        nn::gc::SetStorage(xciBodyStorage.get());
        nn::gc::Attach();
        isFileOpen = true;
#else
        nnt::fs::util::WaitGameCardAttach();
#endif

    }

    virtual void TearDown()
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        if( isFileOpen )
        {
            CloseFile(xciHandle);
        }
        UnmountHostRoot();
#endif
    }

    FileHandle xciHandle;
    std::shared_ptr<IStorage> xciBodyStorage;
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    bool isFileOpen;
#endif
};

TEST_F(MountGameCardTest, Dump)
{
    const GameCardPartition Ids[] =
    {
        GameCardPartition::Update,
        GameCardPartition::Normal,
        GameCardPartition::Secure,
        GameCardPartition::Logo,
    };

    for( auto id : Ids )
    {
        nnt::fs::util::DetachAttachGameCard();

        NN_LOG("Game card partition: %d\n", id);
        GameCardHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
        if( id == GameCardPartition::Logo )
        {
            auto result = MountGameCardPartition("gc", handle, id);
            ASSERT_TRUE(result.IsSuccess() || ResultPartitionNotFound::Includes(result));
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("gc", handle, id));
        }
        nnt::fs::util::DumpDirectoryRecursive("gc:/");
        Unmount("gc");
    }
}

TEST_F(MountGameCardTest, MountSequential)
{
    nnt::fs::util::DetachAttachGameCard();
    const GameCardPartition Ids[] =
    {
        GameCardPartition::Update,
        GameCardPartition::Normal,
        GameCardPartition::Secure,
        GameCardPartition::Logo,
    };

    GameCardHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));

    for( auto id : Ids )
    {
        NN_LOG("Game card partition: %d\n", id);
        if( id == GameCardPartition::Logo )
        {
            auto result = MountGameCardPartition("gc", handle, id);
            ASSERT_TRUE(result.IsSuccess() || ResultPartitionNotFound::Includes(result));
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("gc", handle, id));
        }
        nnt::fs::util::DumpDirectoryRecursive("gc:/");
        Unmount("gc");
    }
}

TEST_F(MountGameCardTest, ExclusiveMount)
{
    nnt::fs::util::DetachAttachGameCard();

    GameCardHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
    NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("normal", handle, GameCardPartition::Normal));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("normal:/"));

    NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("secure", handle, GameCardPartition::Secure));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("secure:/"));

    // secure モード遷移によって利用不可
    NNT_EXPECT_RESULT_FAILURE(ResultGameCardAccessFailed, nnt::fs::util::DumpDirectoryRecursive("normal:/", false));

    Unmount("secure");
    Unmount("normal");
}

TEST_F(MountGameCardTest, ReAttach)
{
    nnt::fs::util::DetachAttachGameCard();

    GameCardHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
    NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("normal", handle, GameCardPartition::Normal));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("normal:/"));

    nnt::fs::util::DetachAttachGameCard();

    // 挿抜後は古いマウントは利用できない
    NNT_EXPECT_RESULT_FAILURE(ResultGameCardAccessFailed, nnt::fs::util::DumpDirectoryRecursive("normal:/", false));

    // 同じハンドルは利用できない
    NNT_EXPECT_RESULT_FAILURE(ResultGameCardAccessFailed, MountGameCardPartition("normal2", handle, GameCardPartition::Normal));

    // ハンドル取得からやり直せば OK
    GameCardHandle handle2;
    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle2));
    EXPECT_NE(handle, handle2);

    NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("normal2", handle2, GameCardPartition::Normal));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("normal2:/"));

    Unmount("normal2");
    Unmount("normal");
}

TEST_F(MountGameCardTest, ReAttachSecure)
{
    nnt::fs::util::DetachAttachGameCard();

    GameCardHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
    NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("secure", handle, GameCardPartition::Secure));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("secure:/"));

    NN_LOG("Rettach same card.\n");
    nnt::fs::util::DetachAttachGameCard();

    // 挿抜直後は古いマウントは利用できない
    NNT_EXPECT_RESULT_FAILURE(ResultGameCardAccessFailed, nnt::fs::util::DumpDirectoryRecursive("secure:/", false));

    // 古いハンドルでの再マウントはできない
    NNT_EXPECT_RESULT_FAILURE(ResultGameCardAccessFailed, MountGameCardPartition("secure2", handle, GameCardPartition::Secure));

    // ハンドル取得からやり直せば OK
    GameCardHandle handle2;
    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle2));
    EXPECT_NE(handle, handle2);
    NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("secure2", handle2, GameCardPartition::Secure));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("secure2:/"));
    Unmount("secure2");

    // 挿抜後 Secure モードに入っていて、同一カードが挿入されていた場合に限り、古いマウントを利用できる
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("secure:/"));

    NN_LOG("Reattach different card.\n");
    nnt::fs::util::DetachAttachGameCard();

    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle2));
    EXPECT_NE(handle, handle2);
    NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("secure2", handle2, GameCardPartition::Secure));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("secure2:/"));
    Unmount("secure2");

    // 挿抜後 Secure モードに入っていても、同一カードが挿入されていなければ古いマウントは利用できない
    NNT_EXPECT_RESULT_FAILURE(ResultGameCardAccessFailed, nnt::fs::util::DumpDirectoryRecursive("secure:/", false));

    Unmount("secure");
}

// @pre
//    - 4d249f76e35ebbccd461d85ef39f9a45.nca が c:/windows/temp/test.xci/secure/ に存在する
//    - MountGameCardPartition() が未実行
// TODO: 自動化
TEST_F(MountGameCardTest, ViaMountCode)
{
    struct TestConfig {
        GameCardPartition partition;
        const char* expectedCommonMountName;
    } Configs[] =
    {
        { GameCardPartition::Update, "@GcU00000001:/" },
        { GameCardPartition::Normal, "@GcN00000002:/" },
        { GameCardPartition::Secure, "@GcS00000003:/" },
    };

    for( const auto& config : Configs )
    {
        NN_LOG("Partition: %d\n", config.partition);

        const nnt::fs::util::String UserMountName = "gc";
        nnt::fs::util::String NcaNameInCodeArea; // = "4d249f76e35ebbccd461d85ef39f9a45.nca";

        GameCardHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
        NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition(UserMountName.c_str(), handle, config.partition));

        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive((UserMountName + ":/").c_str()));

        DirectoryHandle dirHandle;
        NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&dirHandle, "gc:/", OpenDirectoryMode_File));
        while( NN_STATIC_CONDITION(true) )
        {
            DirectoryEntry entry;
            int64_t count;
            NNT_ASSERT_RESULT_SUCCESS(ReadDirectory(&count, &entry, dirHandle, 1));

            ASSERT_NE(0, count);

            if( strlen(entry.name) > 9 && strncmp(&entry.name[strlen(entry.name) - 9], ".cnmt.nca", 9) == 0 )
            {
                NcaNameInCodeArea = entry.name;
                break;
            }

        }

        auto userPath = UserMountName + ":/" + NcaNameInCodeArea;

        char commonPath[EntryNameLengthMax];
        NNT_ASSERT_RESULT_SUCCESS(ConvertToFsCommonPath(commonPath, sizeof(commonPath), userPath.c_str()));

        EXPECT_STREQ((nnt::fs::util::String(config.expectedCommonMountName) + NcaNameInCodeArea).c_str(), commonPath);

        nnt::fs::util::String path = commonPath;
        NNT_ASSERT_RESULT_SUCCESS(MountContent("cnmt", path.c_str(), ContentType_Meta));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("cnmt:/"));
        Unmount("cnmt");
    }

}

TEST_F(MountGameCardTest, CheckGameCardPartitionAvailability)
{
    nnt::fs::util::DetachAttachGameCard();

    GameCardHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
    NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("normal", handle, GameCardPartition::Normal));
    NNT_EXPECT_RESULT_SUCCESS(CheckGameCardPartitionAvailability(handle, GameCardPartition::Normal));

    NNT_ASSERT_RESULT_SUCCESS(MountGameCardPartition("secure", handle, GameCardPartition::Secure));
    NNT_EXPECT_RESULT_FAILURE(ResultGameCardAccessFailed, CheckGameCardPartitionAvailability(handle, GameCardPartition::Normal));
    NNT_EXPECT_RESULT_SUCCESS(CheckGameCardPartitionAvailability(handle, GameCardPartition::Secure));

    nnt::fs::util::DetachAttachGameCard();

    Unmount("secure");
    Unmount("normal");

    NNT_EXPECT_RESULT_FAILURE(ResultGameCardAccessFailed, CheckGameCardPartitionAvailability(handle, GameCardPartition::Secure));

    // 新規ハンドル
    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
    NNT_EXPECT_RESULT_SUCCESS(CheckGameCardPartitionAvailability(handle, GameCardPartition::Secure));
    NNT_EXPECT_RESULT_FAILURE(ResultGameCardAccessFailed, CheckGameCardPartitionAvailability(handle, GameCardPartition::Normal)); // TORIAEZU
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)

namespace
{
    const nnt::fs::util::MountTestAttribute GetAttribute() NN_NOEXCEPT
    {
        nnt::fs::util::MountTestAttribute attribute = {};
        attribute.isReservedMountNameSupported = true;
        return attribute;
    }

    nn::Result MountGameCardPartitionForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        nn::fs::GameCardHandle handle;
        NN_RESULT_DO(nn::fs::GetGameCardHandle(&handle));
        NN_RESULT_DO(nn::fs::MountGameCardPartition(mountName, handle, nn::fs::GameCardPartition::Secure));
        NN_RESULT_SUCCESS;
    }

    const nnt::fs::util::MountTestParameter MountTestParameters[] = {
        { "MountGameCardPartition", MountGameCardPartitionForMountNameTest, nullptr, GetAttribute },
    };
}

NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountGameCardPartition, ::testing::ValuesIn(MountTestParameters));

#endif // !defined(NN_BUILD_CONFIG_OS_WIN)
