﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

const int64_t DefaultSaveDataSize = 2 * 1024 * 1024;
const int64_t DefaultJournalSize = 1024 * 1024;
const char MountName[] = "save";
const char PaddingMountName[] = "padding";
const char PaddingDirectoryPath[] = "padding:/test_data";
const nn::fs::SystemSaveDataId SystemId = 0x8000000000000001;

class ExtendSaveDataTestBase : public ::testing::Test
{
public:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        DeleteSaveData();
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        DeleteSaveData();
    }

    // 正常に拡張できることを確認する
    bool TestSimple() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData());
        NNT_EXPECT_RESULT_SUCCESS(MountSaveData());
        nn::fs::Unmount(MountName);

        // 拡張する
        m_Size += 64 * 1024;
        m_JournalSize += 64 * 1024;
        NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData());

        // データが壊れていないか等の確認
        NNT_EXPECT_RESULT_SUCCESS(MountSaveData());
        nn::fs::Unmount(MountName);
        NNT_EXPECT_RESULT_SUCCESS(VerifySaveData());

        return true;
    }

    // 事前条件を満たしていない場合の挙動を確認する
    bool TestPreCondition() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData());
        DeleteSaveData();
        // 存在しないセーブデータは拡張できない
        {
            m_SaveDataId = 0;
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTargetNotFound,
                ExtendSaveData()
            );
        }

        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData());

        // マウント中は拡張できない
        {
            NNT_EXPECT_RESULT_SUCCESS(MountSaveData());
            m_Size += 64 * 1024;
            m_JournalSize += 64 * 1024;
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTargetLocked,
                ExtendSaveData()
            );
            nn::fs::Unmount(MountName);
        }

        NNT_EXPECT_RESULT_SUCCESS(VerifySaveData());

        return true;
    }

    // サイズだけを繰り返し拡張する
    bool TestRepeatExtendSize() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData());

        for( int i = 0; i < 5; ++i )
        {
            m_Size += 64 * 1024;
            NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData());
        }

        NNT_EXPECT_RESULT_SUCCESS(VerifySaveData());

        return true;
    }

    // ジャーナルサイズだけを繰り返し拡張する
    bool TestRepeatExtendJournal() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData());

        for( int i = 0; i < 5; ++i )
        {
            m_JournalSize += 64 * 1024;
            NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData());
        }

        NNT_EXPECT_RESULT_SUCCESS(VerifySaveData());

        return true;
    }

    // 拡張限界まで拡張を繰り返す
    bool TestRepeatUntilMaxExtendCount() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData());

        nn::Result result = nn::ResultSuccess();
        int extendCount = 0;
        while( result.IsSuccess() )
        {
            m_Size += 16 * 1024;
            m_JournalSize += 16 * 1024;
            result = ExtendSaveData();
            ++extendCount;
        }
        NN_LOG("Success : Extend %d times\n", extendCount - 1);
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultMapFull,
            result
        );

        NNT_EXPECT_RESULT_SUCCESS(VerifySaveData());
        return true;
    }

    // 空き容量不足で失敗させる
    bool TestSpaceNotEnough() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData());

        if( !CreatePaddingFile() )
        {
            return false;
        }
        NN_UTIL_SCOPE_EXIT
        {
            DeletePaddingFile();
        };

        m_Size += 2 * 1024 * 1024;
        m_JournalSize += 2 * 1024 * 1024;

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultUsableSpaceNotEnough,
            ExtendSaveData()
        );

        NNT_EXPECT_RESULT_SUCCESS(VerifySaveData());
        return true;
#else
        return false;
#endif
    }

    // 拡大/縮小せずに ExtendSaveData を何度も繰り返す
    bool TestRepeatSameSize() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData());

        const int LoopCount = 512;
        for( int i = 0; i < LoopCount; ++i )
        {
            NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData());
        }

        NNT_EXPECT_RESULT_SUCCESS(VerifySaveData());

        return true;
    }

    // 縮小する
    bool TestShorten() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData());

        m_Size -= 16 * 1024;

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            ExtendSaveData()
        );

        NNT_EXPECT_RESULT_SUCCESS(VerifySaveData());
        return true;
    }

    // 拡大してすぐに縮小する
    bool TestExtendAndShorten() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData());

        m_Size += 64 * 1024;
        NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData());

        m_Size -= 16 * 1024;
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            ExtendSaveData()
        );

        NNT_EXPECT_RESULT_SUCCESS(VerifySaveData());
        return true;
    }

protected:
    virtual nn::Result CreateSaveData() NN_NOEXCEPT = 0;
    virtual nn::Result MountSaveData() NN_NOEXCEPT = 0;

    void InitExtendData(
        nn::fs::SaveDataSpaceId spaceId,
        nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        InitExtendData(spaceId, saveDataId, DefaultSaveDataSize, DefaultJournalSize);
    }

    void InitExtendData(
        nn::fs::SaveDataSpaceId spaceId,
        nn::fs::SaveDataId saveDataId,
        int64_t size,
        int64_t journalSize) NN_NOEXCEPT
    {
        m_SpaceId = spaceId;
        m_SaveDataId = saveDataId;
        m_Size = size;
        m_JournalSize = journalSize;
    }

private:
    bool CreatePaddingFile() NN_NOEXCEPT
    {
        switch( m_SpaceId )
        {
            case nn::fs::SaveDataSpaceId::System:
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::fs::MountBis(PaddingMountName, nn::fs::BisPartitionId::System)
                );
                break;
            case nn::fs::SaveDataSpaceId::User:
            case nn::fs::SaveDataSpaceId::Temporary:
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::fs::MountBis(PaddingMountName, nn::fs::BisPartitionId::User)
                );
                break;
            case nn::fs::SaveDataSpaceId::SdSystem:
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::fs::MountSdCard(PaddingMountName)
                );
                break;
            case nn::fs::SaveDataSpaceId::ProperSystem:
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        nn::fs::DeleteDirectoryRecursively(PaddingDirectoryPath);
        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::CreateDirectory(PaddingDirectoryPath)
        );
        const size_t FileNameMaxLength = 32;
        char fileNameBuffer[FileNameMaxLength];
        int fileId = 0;
        nn::Result result = nn::ResultSuccess();

        size_t MaxFileSize
            = 1 * 1024 * 1024 * 1024;
        size_t MinFileSize = 1 * 1024 * 1024;
        size_t createFileSize = MaxFileSize;
        while( createFileSize >= MinFileSize )
        {
            nn::util::SNPrintf(
                fileNameBuffer,
                FileNameMaxLength,
                "%s/file%d",
                PaddingDirectoryPath,
                fileId
            );
            result = nn::fs::CreateFile(fileNameBuffer, createFileSize);
            ++fileId;

            if(nn::fs::ResultUsableSpaceNotEnough::Includes(result))
            {
                createFileSize /= 2;
            }
            else
            {
                NNT_EXPECT_RESULT_SUCCESS(result);
            }
        }
        return true;
    }

    void DeletePaddingFile() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::DeleteDirectoryRecursively(PaddingDirectoryPath)
        );
        nn::fs::Unmount(PaddingMountName);
    }

    nn::Result ExtendSaveData() NN_NOEXCEPT
    {
        NN_RESULT_DO(
            nn::fs::ExtendSaveData(
                m_SpaceId,
                m_SaveDataId,
                m_Size,
                m_JournalSize
            )
        );
        NN_RESULT_SUCCESS;
    }

    void DeleteSaveData() NN_NOEXCEPT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    }

    nn::Result VerifySaveData() NN_NOEXCEPT
    {
        if( m_SpaceId == nn::fs::SaveDataSpaceId::SdSystem )
        {
            NN_RESULT_SUCCESS;
        }

        const size_t workBufferSize = 1024 * 1024;
        std::unique_ptr<char[]> workBuffer(new char[workBufferSize]);
        bool isValid;
        NN_RESULT_DO(
            nn::fs::VerifySaveData(
                &isValid,
                m_SaveDataId,
                workBuffer.get(),
                workBufferSize
            )
        );
        EXPECT_TRUE(isValid);
        NN_RESULT_SUCCESS;
    }

    nn::fs::SaveDataSpaceId m_SpaceId;
    nn::fs::SaveDataId m_SaveDataId;
    int64_t m_Size;
    int64_t m_JournalSize;
};

class ExtendUserSaveDataTest : public ExtendSaveDataTestBase
{
protected:
    virtual nn::Result CreateSaveData() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result MountSaveData() NN_NOEXCEPT NN_OVERRIDE;
private:
    nn::fs::UserId m_UserId;
};
class ExtendNandSystemSaveDataTest : public ExtendSaveDataTestBase
{
protected:
    virtual nn::Result CreateSaveData() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result MountSaveData() NN_NOEXCEPT NN_OVERRIDE;
};
class ExtendSdSystemSaveDataTest : public ExtendSaveDataTestBase
{
protected:
    virtual nn::Result CreateSaveData() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result MountSaveData() NN_NOEXCEPT NN_OVERRIDE;
};
class ExtendBcatSaveDataTest : public ExtendSaveDataTestBase
{
protected:
    virtual nn::Result CreateSaveData() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result MountSaveData() NN_NOEXCEPT NN_OVERRIDE;
};
class ExtendDeviceSaveDataTest : public ExtendSaveDataTestBase
{
protected:
    virtual nn::Result CreateSaveData() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result MountSaveData() NN_NOEXCEPT NN_OVERRIDE;
};

} // namespace

nn::Result ExtendUserSaveDataTest::CreateSaveData() NN_NOEXCEPT
{
    m_UserId = {{ 0, 1 }};
    NN_RESULT_DO(
        nn::fs::CreateSaveData(
            nnt::fs::util::ApplicationId,
            m_UserId,
            0,
            DefaultSaveDataSize,
            DefaultJournalSize,
            0
        )
    );

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_RESULT_DO(
        nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User)
    );

    for(;;)
    {
        nn::fs::SaveDataInfo info;
        int64_t saveDataCount = 0;
        NN_RESULT_DO(
            iter->ReadSaveDataInfo(&saveDataCount, &info, 1)
        );
        NN_RESULT_THROW_UNLESS(saveDataCount != 0, nn::fs::ResultNotFound());
        if( info.saveDataSpaceId != nn::fs::SaveDataSpaceId::User
         || info.saveDataType != nn::fs::SaveDataType::Account
         || info.saveDataUserId != m_UserId
         || info.applicationId != nnt::fs::util::ApplicationId)
        {
            continue;
        }

        InitExtendData(
            nn::fs::SaveDataSpaceId::User,
            info.saveDataId
        );
        break;
    }
    NN_RESULT_SUCCESS;
}

nn::Result ExtendUserSaveDataTest::MountSaveData() NN_NOEXCEPT
{
    NN_RESULT_DO(nn::fs::MountSaveData(MountName, m_UserId));
    NN_RESULT_SUCCESS;
}

nn::Result ExtendNandSystemSaveDataTest::CreateSaveData() NN_NOEXCEPT
{
    NN_RESULT_DO(
        nn::fs::CreateSystemSaveData(
            SystemId,
            DefaultSaveDataSize,
            DefaultJournalSize,
            0
        )
    );

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_RESULT_DO(
        nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::System)
    );

    for(;;)
    {
        int64_t saveDataCount = 0;
        nn::fs::SaveDataInfo info;
        NN_RESULT_DO(
            iter->ReadSaveDataInfo(&saveDataCount, &info, 1)
        );
        NN_RESULT_THROW_UNLESS(saveDataCount != 0, nn::fs::ResultNotFound());
        if( info.systemSaveDataId != SystemId )
        {
            continue;
        }

        InitExtendData(
            nn::fs::SaveDataSpaceId::System,
            info.saveDataId
        );
        break;
    }

    NN_RESULT_SUCCESS;
}

nn::Result ExtendNandSystemSaveDataTest::MountSaveData() NN_NOEXCEPT
{
    NN_RESULT_DO(nn::fs::MountSystemSaveData(MountName, SystemId));
    NN_RESULT_SUCCESS;
}

nn::Result ExtendSdSystemSaveDataTest::CreateSaveData() NN_NOEXCEPT
{
    NN_RESULT_DO(
        nn::fs::CreateSystemSaveData(
            nn::fs::SaveDataSpaceId::SdSystem,
            SystemId,
            0,
            DefaultSaveDataSize,
            DefaultJournalSize,
            0
        )
    );

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_RESULT_DO(
        nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::SdSystem)
    );

    for(;;)
    {
        int64_t saveDataCount = 0;
        nn::fs::SaveDataInfo info;
        NN_RESULT_DO(
            iter->ReadSaveDataInfo(&saveDataCount, &info, 1)
        );
        NN_RESULT_THROW_UNLESS(saveDataCount != 0, nn::fs::ResultNotFound());
        if( info.saveDataId != SystemId )
        {
            continue;
        }

        InitExtendData(
            nn::fs::SaveDataSpaceId::SdSystem,
            info.saveDataId
        );
        break;
    }
    NN_RESULT_SUCCESS;
}

nn::Result ExtendSdSystemSaveDataTest::MountSaveData() NN_NOEXCEPT
{
    NN_RESULT_DO(
        nn::fs::MountSystemSaveData(
            MountName,
            nn::fs::SaveDataSpaceId::SdSystem,
            SystemId
        )
    );
    NN_RESULT_SUCCESS;
}

nn::Result ExtendBcatSaveDataTest::CreateSaveData() NN_NOEXCEPT
{
    NN_RESULT_DO(
        nn::fs::CreateBcatSaveData(
            nnt::fs::util::ApplicationId,
            DefaultSaveDataSize
        )
    );

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_RESULT_DO(
        nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User)
    );

    for(;;)
    {
        int64_t saveDataCount = 0;
        nn::fs::SaveDataInfo info;
        NN_RESULT_DO(
            iter->ReadSaveDataInfo(&saveDataCount, &info, 1)
        );
        NN_RESULT_THROW_UNLESS(saveDataCount != 0, nn::fs::ResultNotFound());
        if( info.saveDataSpaceId != nn::fs::SaveDataSpaceId::User
         || info.saveDataType != nn::fs::SaveDataType::Bcat
         || info.applicationId != nnt::fs::util::ApplicationId)
        {
            continue;
        }

        InitExtendData(
            nn::fs::SaveDataSpaceId::User,
            info.saveDataId,
            DefaultSaveDataSize,
            nn::fs::detail::BcatSaveDataJournalSize
        );
        break;
    }
    NN_RESULT_SUCCESS;
}

nn::Result ExtendBcatSaveDataTest::MountSaveData() NN_NOEXCEPT
{
    NN_RESULT_DO(nn::fs::MountBcatSaveData(MountName, nnt::fs::util::ApplicationId));
    NN_RESULT_SUCCESS;
}

nn::Result ExtendDeviceSaveDataTest::CreateSaveData() NN_NOEXCEPT
{
    NN_RESULT_DO(
        nn::fs::CreateDeviceSaveData(
            nnt::fs::util::ApplicationId,
            0,
            DefaultSaveDataSize,
            DefaultJournalSize,
            0
        )
    );

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_RESULT_DO(
        nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User)
    );

    for(;;)
    {
        int64_t saveDataCount = 0;
        nn::fs::SaveDataInfo info;
        NN_RESULT_DO(
            iter->ReadSaveDataInfo(&saveDataCount, &info, 1)
        );
        NN_RESULT_THROW_UNLESS(saveDataCount != 0, nn::fs::ResultNotFound());
        if( info.saveDataSpaceId != nn::fs::SaveDataSpaceId::User
         || info.saveDataType != nn::fs::SaveDataType::Device
         || info.applicationId != nnt::fs::util::ApplicationId)
        {
            continue;
        }

        InitExtendData(
            nn::fs::SaveDataSpaceId::User,
            info.saveDataId
        );
        break;
    }
    NN_RESULT_SUCCESS;
}

nn::Result ExtendDeviceSaveDataTest::MountSaveData() NN_NOEXCEPT
{
    NN_RESULT_DO(nn::fs::MountDeviceSaveData(MountName));
    NN_RESULT_SUCCESS;
}

#define NNT_FS_FSLIB_EXTEND_SAVE_DATA_TEST(className)                                                                   \
TEST_F(className, Simple)                           { NNT_FS_UTIL_SKIP_TEST_UNLESS(TestSimple()); }                     \
TEST_F(className, PreCondition)                     { NNT_FS_UTIL_SKIP_TEST_UNLESS(TestPreCondition()); }               \
TEST_F(className, RepeatExtendSize)                 { NNT_FS_UTIL_SKIP_TEST_UNLESS(TestRepeatExtendSize()); }           \
TEST_F(className, RepeatExtendJournal)              { NNT_FS_UTIL_SKIP_TEST_UNLESS(TestRepeatExtendJournal()); }        \
TEST_F(className, RepeatUntilMaxExtendCountHeavy)   { NNT_FS_UTIL_SKIP_TEST_UNLESS(TestRepeatUntilMaxExtendCount()); }  \
TEST_F(className, SpaceNotEnough)                   { NNT_FS_UTIL_SKIP_TEST_UNLESS(TestSpaceNotEnough()); }             \
TEST_F(className, TestRepeatSameSizeHeavy)          { NNT_FS_UTIL_SKIP_TEST_UNLESS(TestRepeatSameSize()); }             \
TEST_F(className, Shorten)                          { NNT_FS_UTIL_SKIP_TEST_UNLESS(TestShorten()); }                    \
TEST_F(className, ExtendAndShorten)                 { NNT_FS_UTIL_SKIP_TEST_UNLESS(TestExtendAndShorten()); }           \

NNT_FS_FSLIB_EXTEND_SAVE_DATA_TEST(ExtendUserSaveDataTest);
NNT_FS_FSLIB_EXTEND_SAVE_DATA_TEST(ExtendNandSystemSaveDataTest);
NNT_FS_FSLIB_EXTEND_SAVE_DATA_TEST(ExtendSdSystemSaveDataTest);
NNT_FS_FSLIB_EXTEND_SAVE_DATA_TEST(ExtendBcatSaveDataTest);
NNT_FS_FSLIB_EXTEND_SAVE_DATA_TEST(ExtendDeviceSaveDataTest);
