﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/fs/fs_ApplicationSaveDataManagement.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_CacheStorageWithIndex.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_Utility.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/oe.h>
#include <nn/time/time_PosixTime.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

using namespace nn::fs;

namespace {
    // 暫定で type から決め打ち
    SaveDataSpaceId GetSpaceIdByType(SaveDataType type)
    {
        switch(type)
        {
            case SaveDataType::Temporary:
                return SaveDataSpaceId::Temporary;
            case SaveDataType::Cache:
                return IsSdCardAccessible() ? SaveDataSpaceId::SdUser : SaveDataSpaceId::User;
            default:
                return SaveDataSpaceId::User;
        }
    }

    nn::util::optional<SaveDataInfo> CheckSaveDataExisting(nn::ncm::ApplicationId id, SaveDataSpaceId spaceId, SaveDataType type, bool isExpectingExisting, uint16_t index)
    {
        NNT_FS_SCOPED_TRACE_SAFE("id: %llx, type: %d, expectExist: %d", id.value, type, isExpectingExisting);

        nn::util::optional<SaveDataInfo> info;

        nnt::fs::util::FindSaveData(&info, spaceId, [&](SaveDataInfo i) {
            return i.applicationId.value == id.value && i.saveDataType == type && i.index == index;
        });
        EXPECT_TRUE((info != nn::util::nullopt) == isExpectingExisting);

        return info;
    }

    nn::util::optional<SaveDataInfo> CheckSaveDataExisting(nn::ncm::ApplicationId id, SaveDataType type, bool isExpectingExisting, uint16_t index = 0)
    {
        return CheckSaveDataExisting(id, GetSpaceIdByType(type), type, isExpectingExisting, index);
    }

    nn::ns::ApplicationControlProperty GetDefaultProperty()
    {
        nn::ns::ApplicationControlProperty property = {};
        property.saveDataOwnerId = nnt::fs::util::UserSaveDataApplicationId + 1;
        property.userAccountSaveDataSize        = 20 * 1024 * 1024;
        property.userAccountSaveDataJournalSize = 15 * 1024 * 1024;
        property.deviceSaveDataSize             = 10 * 1024 * 1024;
        property.deviceSaveDataJournalSize      =  5 * 1024 * 1024;
        property.bcatDeliveryCacheStorageSize   =  8 * 1024 * 1024;
        property.userAccountSaveDataSizeMax        = property.userAccountSaveDataSize * 16;
        property.userAccountSaveDataJournalSizeMax = property.userAccountSaveDataJournalSize * 16;
        property.deviceSaveDataSizeMax             = property.deviceSaveDataSize * 16;
        property.deviceSaveDataJournalSizeMax      = property.deviceSaveDataJournalSize * 16;
        property.temporaryStorageSize    = 40 * 1024 * 1024;
        property.cacheStorageSize        = 30 * 1024 * 1024;
        property.cacheStorageJournalSize = 18 * 1024 * 1024;
        property.cacheStorageDataAndJournalSizeMax     = property.cacheStorageSize + property.cacheStorageJournalSize;
        property.cacheStorageIndexMax    = 1024;
        return property;
    }

    nn::ns::ApplicationControlProperty GetPropertyWithCacheStorageSize(uint64_t ownerIdOffset, int64_t cacheStorageSize, int64_t cacheStorageJournalSize)
    {
        nn::ns::ApplicationControlProperty property = {};
        property.saveDataOwnerId = nnt::fs::util::UserSaveDataApplicationId + ownerIdOffset;
        property.userAccountSaveDataSize        = 20 * 1024 * 1024;
        property.userAccountSaveDataJournalSize = 15 * 1024 * 1024;
        property.deviceSaveDataSize             = 10 * 1024 * 1024;
        property.deviceSaveDataJournalSize      =  5 * 1024 * 1024;
        property.bcatDeliveryCacheStorageSize   =  8 * 1024 * 1024;
        property.userAccountSaveDataSizeMax        = property.userAccountSaveDataSize * 16;
        property.userAccountSaveDataJournalSizeMax = property.userAccountSaveDataJournalSize * 16;
        property.deviceSaveDataSizeMax             = property.deviceSaveDataSize * 16;
        property.deviceSaveDataJournalSizeMax      = property.deviceSaveDataJournalSize * 16;
        property.temporaryStorageSize    = 40 * 1024 * 1024;
        property.cacheStorageSize        = cacheStorageSize;
        property.cacheStorageJournalSize = cacheStorageJournalSize;
        return property;
    }

    nn::account::Uid GetDefaultUid()
    {
        nn::account::Uid user;
#if defined(NN_BUILD_CONFIG_OS_WIN) // win 未対応
        user._data[0] = 1;
        user._data[1] = 0;
#else
        int userCount = 0;
        auto result = nn::account::ListAllUsers(&userCount, &user, 1);
        NN_ABORT_UNLESS(result.IsSuccess() && userCount > 0);
#endif
        return user;
    }

}

void CheckSaveData(const SaveDataInfo& info, const nn::ns::ApplicationControlProperty& property, const SaveDataSpaceId spaceId, const nn::account::Uid& user, int64_t extendedAvailableSize, int64_t extendedJournalSize)
{
    int64_t expectedAvailableSize = 0;
    int64_t expectedJournalSize = 0;
    nn::Bit64 expectedOwnerId = 0;

    const char* mountName = "save";

    if (info.saveDataType == SaveDataType::Account)
    {
        expectedAvailableSize =  extendedAvailableSize == 0 ? property.userAccountSaveDataSize : extendedAvailableSize;
        expectedJournalSize = extendedJournalSize == 0 ? property.userAccountSaveDataJournalSize : extendedJournalSize;
        expectedOwnerId = property.saveDataOwnerId;

#if defined(NN_BUILD_CONFIG_OS_WIN)
        NNT_EXPECT_RESULT_SUCCESS(MountSaveData(mountName, ConvertAccountUidToFsUserId(user)));
#else
        NNT_EXPECT_RESULT_SUCCESS(MountSaveData(mountName, user));
#endif
        Unmount(mountName);
    }
    else if (info.saveDataType == SaveDataType::Device)
    {
        expectedAvailableSize = extendedAvailableSize == 0 ? property.deviceSaveDataSize : extendedAvailableSize;
        expectedJournalSize = extendedJournalSize == 0 ? property.deviceSaveDataJournalSize : extendedJournalSize;
        expectedOwnerId = property.saveDataOwnerId;

        NNT_EXPECT_RESULT_SUCCESS(MountDeviceSaveData(mountName));
        Unmount(mountName);
    }
    else if (info.saveDataType == SaveDataType::Bcat)
    {
        expectedAvailableSize = property.bcatDeliveryCacheStorageSize;
        expectedJournalSize = detail::BcatSaveDataJournalSize;
        expectedOwnerId = 0x010000000000000CULL; // bcat process

        NNT_EXPECT_RESULT_SUCCESS(MountBcatSaveData(mountName, info.applicationId));
        Unmount(mountName);
    }
    else if (info.saveDataType == SaveDataType::Temporary)
    {
        expectedAvailableSize = property.temporaryStorageSize;
        expectedJournalSize = 0;
        expectedOwnerId = property.saveDataOwnerId;
    }
    else if (info.saveDataType == SaveDataType::Cache)
    {
        expectedAvailableSize = property.cacheStorageSize;
        expectedJournalSize = property.cacheStorageJournalSize;
        expectedOwnerId = property.saveDataOwnerId;
    }
    else
    {
        EXPECT_TRUE(false);
    }

    int64_t expectedSaveDataTotalSize = 0;
    NNT_EXPECT_RESULT_SUCCESS(QuerySaveDataTotalSize(&expectedSaveDataTotalSize, expectedAvailableSize, expectedJournalSize));

#if defined(NN_BUILD_CONFIG_OS_WIN) // win 未対応
    expectedAvailableSize = 0;
    expectedJournalSize = 0;
    expectedOwnerId = 0;
#endif

    NN_LOG("actual save data size %lld\n", info.saveDataSize);
    EXPECT_EQ(expectedSaveDataTotalSize, info.saveDataSize);
    EXPECT_EQ(user._data[0], info.saveDataUserId._data[0]);
    EXPECT_EQ(user._data[1], info.saveDataUserId._data[1]);

    nn::Bit64 ownerId = 0;
    NNT_EXPECT_RESULT_SUCCESS(GetSaveDataOwnerId(&ownerId, spaceId, info.saveDataId));
    EXPECT_EQ(expectedOwnerId, ownerId);

    uint32_t flags = 0xFFFFFFFF;
    NNT_EXPECT_RESULT_SUCCESS(GetSaveDataFlags(&flags, spaceId, info.saveDataId));
    EXPECT_EQ(0, flags);

    nn::time::PosixTime time;
    NNT_EXPECT_RESULT_SUCCESS(GetSaveDataTimeStamp(&time, spaceId, info.saveDataId));
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    EXPECT_NE(0, time.value);
#else
    EXPECT_EQ(0, time.value);
#endif

    int64_t availableSize;
    NNT_EXPECT_RESULT_SUCCESS(GetSaveDataAvailableSize(&availableSize, spaceId, info.saveDataId));
    EXPECT_EQ(expectedAvailableSize, availableSize);

    int64_t journalSize;
    NNT_EXPECT_RESULT_SUCCESS(GetSaveDataJournalSize(&journalSize, spaceId, info.saveDataId));
    EXPECT_EQ(expectedJournalSize, journalSize);

#if !defined(NN_BUILD_CONFIG_OS_WIN) // win 未対応
    // アプリ呼び出しによるサイズ取得
    if (info.saveDataType == SaveDataType::Account)
    {
        NNT_EXPECT_RESULT_SUCCESS(GetSaveDataSize(&availableSize, &journalSize, user));
        EXPECT_EQ(expectedAvailableSize, availableSize);
        EXPECT_EQ(expectedJournalSize, journalSize);
    }
#if 0
    else if (info.saveDataType == SaveDataType::Device)
    {
        NNT_EXPECT_RESULT_SUCCESS(GetDeviceSaveDataSize(&availableSize, &journalSize));
        EXPECT_EQ(expectedAvailableSize, availableSize);
        EXPECT_EQ(expectedJournalSize, journalSize);
    }
#endif
#endif
}

void CheckSaveData(const SaveDataInfo& info, const nn::ns::ApplicationControlProperty& property, const nn::account::Uid& user, int64_t extendedAvailableSize, int64_t extendedJournalSize)
{
    return CheckSaveData(info, property, GetSpaceIdByType(info.saveDataType), user, extendedAvailableSize, extendedJournalSize);
}

void CheckSaveData(const SaveDataInfo& info, const nn::ns::ApplicationControlProperty& property, SaveDataSpaceId spaceId,const nn::account::Uid& user)
{
    return CheckSaveData(info, property, spaceId, user, 0, 0);
}

void CheckSaveData(const SaveDataInfo& info, const nn::ns::ApplicationControlProperty& property, const nn::account::Uid& user)
{
    return CheckSaveData(info, property, user, 0, 0);
}

TEST(EnsureApplicationSaveData, Create)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::account::Uid user = GetDefaultUid();

    // 作成
    int64_t size;
    {
        auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationSaveData(&size, id, property, user));
        auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
    }

    // 確認
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
        CheckSaveData(info.value(), property, user);
    }
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Device, true);
        CheckSaveData(info.value(), property, nn::account::InvalidUid);
    }
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Bcat, true);
        CheckSaveData(info.value(), property, nn::account::InvalidUid);
    }
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Temporary, true);
        CheckSaveData(info.value(), property, nn::account::InvalidUid);
    }
    // キャッシュストレージはないことを確認
    {
        CheckSaveDataExisting(id, SaveDataType::Cache, false);
    }

    // 2回目以降は何もせず即 OK を返す
    {
        auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationSaveData(&size, id, property, user));
        auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );

#if !defined(NN_BUILD_CONFIG_OS_WIN) // win 版の時間をチェックする意味は薄いので省略
        EXPECT_LT((end - start).GetMilliSeconds(), 100);
#endif
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

#if !defined(NN_BUILD_CONFIG_OS_WIN) // win 版はアプリからの拡張に未対応
TEST(EnsureApplicationSaveData, ExtendFromSizeNotAligned)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::account::Uid user = GetDefaultUid();

    // 非 1MB アラインサイズな初期セーブ
    property.userAccountSaveDataSize        += 16 * 1024;
    property.userAccountSaveDataJournalSize += 16 * 1024;

    int64_t size;

    // データ領域・ジャーナリングのいずれかだけの拡張の場合、拡張されない方は非アラインのままで元より小さい値でもよい
    {
        // 作成
        NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationSaveData(&size, id, property, user));

        // ユーザーアカウントセーブのデータ領域拡張
        NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData(user, nnt::fs::util::RoundUp(property.userAccountSaveDataSize, nn::fs::SaveDataExtensionUnitSize), property.userAccountSaveDataJournalSize - 16 * 1024));
        auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
        CheckSaveData(info.value(), property, user, nnt::fs::util::RoundUp(property.userAccountSaveDataSize, nn::fs::SaveDataExtensionUnitSize), property.userAccountSaveDataJournalSize);

        nnt::fs::util::DeleteAllTestSaveData();
    }

    {
        // 作成
        NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationSaveData(&size, id, property, user));

        // ユーザーアカウントセーブのジャーナリング領域拡張
        NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData(user, property.userAccountSaveDataSize - 16 * 1024, nnt::fs::util::RoundUp(property.userAccountSaveDataJournalSize, nn::fs::SaveDataExtensionUnitSize)));
        auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
        CheckSaveData(info.value(), property, user, property.userAccountSaveDataSize, nnt::fs::util::RoundUp(property.userAccountSaveDataJournalSize, nn::fs::SaveDataExtensionUnitSize));

        nnt::fs::util::DeleteAllTestSaveData();
    }
}
#endif

// キャッシュストレージの詳細なテストは WIN 版はカット
#if !defined(NN_BUILD_CONFIG_OS_WIN)
void ConfirmCacheStorage(nn::ncm::ApplicationId& id, nn::ns::ApplicationControlProperty& property, SaveDataSpaceId spaceId, bool isForEnsure, uint16_t index)
{
    int64_t size;
    // 確認: キャッシュストレージ以外は作成されていないこと
    CheckSaveDataExisting(id, SaveDataType::Account,   false);
    CheckSaveDataExisting(id, SaveDataType::Device,    false);
    CheckSaveDataExisting(id, SaveDataType::Bcat,      false);
    CheckSaveDataExisting(id, SaveDataType::Temporary, false);

    // 確認: キャッシュストレージが存在すること
    auto info = CheckSaveDataExisting(id, spaceId, SaveDataType::Cache, true, index);
    CheckSaveData(info.value(), property, spaceId, nn::account::InvalidUid);

    if(isForEnsure)
    {
        // 確認 : もう一度作成したら何もせず即 OK を返すこと
        auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        CacheStorageTargetMedia media;
        NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationCacheStorage(&size, &media, id, property));
        auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        EXPECT_LT((end - start).GetMilliSeconds(), 100);
    }
    else
    {
        CacheStorageTargetMedia media;
        NNT_EXPECT_RESULT_FAILURE(ResultAlreadyExists, CreateApplicationCacheStorage(&size, &media, id, property, index, property.cacheStorageSize, property.cacheStorageJournalSize));
    }

}

void ConfirmCacheStorageForEnsure(nn::ncm::ApplicationId& id, nn::ns::ApplicationControlProperty& property, SaveDataSpaceId spaceId)
{
    ConfirmCacheStorage(id, property, spaceId, true, 0);
}

void ConfirmCacheStorageForCreate(nn::ncm::ApplicationId& id, nn::ns::ApplicationControlProperty& property, SaveDataSpaceId spaceId, uint16_t index)
{
    ConfirmCacheStorage(id, property, spaceId, false, index);
}
// キャッシュストレージ単体作成
TEST(EnsureApplicationCacheStorage, Create)
{
    nnt::fs::util::DeleteAllTestSaveData();
    nn::ncm::ApplicationId id = { nnt::fs::util::UserSaveDataApplicationId };
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    int64_t size = 0;
    CacheStorageTargetMedia media;

    // ns が SD カードを未マウントだったら失敗
    EXPECT_TRUE(IsSdCardAccessible());

    // SD にキャッシュストレージ作成
    NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationCacheStorage(&size, &media, id , property));
    EXPECT_TRUE(media == CacheStorageTargetMedia_Sd);
    ConfirmCacheStorageForEnsure(id, property, SaveDataSpaceId::SdUser);

    // NAND にキャッシュストレージを作成するため一旦 SD 無効化
    SetSdCardAccessibility(false);

    // NAND にキャッシュストレージ作成
    NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationCacheStorage(&size, &media, id , property));
    EXPECT_TRUE(media == CacheStorageTargetMedia_Nand);
    ConfirmCacheStorageForEnsure(id, property, SaveDataSpaceId::User);

    // SD 有効に戻す
    SetSdCardAccessibility(true);
}

// キャッシュストレージ拡張
TEST(ExtendByEnsureApplicationCacheStorage, Extend)
{
    nnt::fs::util::DeleteAllTestSaveData();
    nn::ncm::ApplicationId id = { nnt::fs::util::UserSaveDataApplicationId };
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::ns::ApplicationControlProperty extendedProperty = property;
    extendedProperty.cacheStorageSize *= 2;
    extendedProperty.cacheStorageJournalSize *= 2;
    extendedProperty.cacheStorageDataAndJournalSizeMax = extendedProperty.cacheStorageSize + extendedProperty.cacheStorageJournalSize;
    int64_t size = 0;
    CacheStorageTargetMedia media;
    // ns が SD カードを未マウントだったら失敗
    EXPECT_TRUE(IsSdCardAccessible());

    // SD にキャッシュストレージ作成
    NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationCacheStorage(&size, &media, id , property));
    EXPECT_TRUE(media == CacheStorageTargetMedia_Sd);
    ConfirmCacheStorageForEnsure(id, property, SaveDataSpaceId::SdUser);

    // キャッシュストレージ拡張
    NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationCacheStorage(&size, &media, id , extendedProperty));
    EXPECT_TRUE(media == CacheStorageTargetMedia_Sd);
    ConfirmCacheStorageForEnsure(id, extendedProperty, SaveDataSpaceId::SdUser);

    // NAND にキャッシュストレージを作成するため一旦 SD 無効化
    SetSdCardAccessibility(false);

    // NAND にキャッシュストレージ作成
    NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationCacheStorage(&size, &media, id , property));
    EXPECT_TRUE(media == CacheStorageTargetMedia_Nand);
    ConfirmCacheStorageForEnsure(id, property, SaveDataSpaceId::User);

    // キャッシュストレージ拡張
    NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationCacheStorage(&size, &media, id , extendedProperty));
    EXPECT_TRUE(media == CacheStorageTargetMedia_Nand);
    ConfirmCacheStorageForEnsure(id, extendedProperty, SaveDataSpaceId::User);

    // SD 有効に戻す
    SetSdCardAccessibility(true);
}

TEST(CreateCacheStorage, CreateCacheStorageLargeIndex)
{
    EXPECT_TRUE(IsSdCardAccessible());
    nnt::fs::util::DeleteAllTestSaveData();
    nn::ncm::ApplicationId id = { nnt::fs::util::UserSaveDataApplicationId};
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    int64_t cacheStorageSize = 512 * 1024 * 1024;
    int64_t cacheStorageJournalSize = 512 * 1024 * 1024;
    int index = 65535;

    NNT_EXPECT_RESULT_SUCCESS(CreateCacheStorage(id, nn::fs::SaveDataSpaceId::SdUser, property.saveDataOwnerId, index, cacheStorageSize, cacheStorageJournalSize, 0));
    nn::fs::CacheStorageListHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenCacheStorageList(&handle));
    int num = 0;
    nn::fs::CacheStorageInfo cacheStorageInfo;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadCacheStorageList(&num, &cacheStorageInfo, handle, 1));
    EXPECT_TRUE(num == 1);
    nn::fs::CloseCacheStorageList(handle);
    EXPECT_TRUE(cacheStorageInfo.index == index);
}

// キャッシュストレージ作成削除テスト(CreateApplicationCacheStorage)
TEST(CreateApplicationCacheStorage, CreateAndDelete)
{
    nnt::fs::util::DeleteAllTestSaveData();
    nn::ncm::ApplicationId id = { nnt::fs::util::UserSaveDataApplicationId };
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    int64_t size = 0;
    CacheStorageTargetMedia media;
    const uint16_t IndexNum = 3;

    // ns が SD カードを未マウントだったら失敗
    EXPECT_TRUE(IsSdCardAccessible());
    // SD にキャッシュストレージ 0,1,2 作成
    for(uint16_t index = 0; index < IndexNum; index++)
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateApplicationCacheStorage(&size, &media, id, property, index, property.cacheStorageSize, property.cacheStorageJournalSize));
        EXPECT_TRUE(media == CacheStorageTargetMedia_Sd);
        ConfirmCacheStorageForCreate(id, property, SaveDataSpaceId::SdUser, index);
    }

    for(uint16_t index = 0; index < IndexNum; index++)
    {
        NNT_EXPECT_RESULT_SUCCESS(DeleteCacheStorage(index));
        // 削除されていることを確認する
        CheckSaveDataExisting(id, SaveDataSpaceId::SdUser, SaveDataType::Cache, false, index);
        // 未削除は生存確認
        for(uint16_t rest = index + 1; rest < IndexNum; rest++)
        {
            ConfirmCacheStorageForCreate(id, property, SaveDataSpaceId::SdUser, rest);
        }
        // もう 1 度消そうとしたら TargetNotFound
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, DeleteCacheStorage(index));
    }
    // NAND にキャッシュストレージを作成するため一旦 SD 無効化
    SetSdCardAccessibility(false);
    NN_UTIL_SCOPE_EXIT
    {
        SetSdCardAccessibility(true);
    };

    // NAND にキャッシュストレージ作成
    for(uint16_t index = 0; index < IndexNum; index++)
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateApplicationCacheStorage(&size, &media, id, property, index, property.cacheStorageSize, property.cacheStorageJournalSize));
        EXPECT_TRUE(media == CacheStorageTargetMedia_Nand);
        ConfirmCacheStorageForCreate(id, property, SaveDataSpaceId::User, index);
    }
    for(uint16_t index = 0; index < IndexNum; index++)
    {
        NNT_EXPECT_RESULT_SUCCESS(DeleteCacheStorage(index));
        // 削除されていることを確認する
        CheckSaveDataExisting(id, SaveDataSpaceId::User, SaveDataType::Cache, false, index);
        // 未削除は生存確認
        for(uint16_t rest = index + 1; rest < IndexNum; rest++)
        {
            ConfirmCacheStorageForCreate(id, property, SaveDataSpaceId::User, rest);
        }
        // もう 1 度消そうとしたら TargetNotFound
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, DeleteCacheStorage(index));
    }
    // TODO: DeleteSaveDataMetaFile()
}

// キャッシュストレージのサイズ取得のテスト
TEST(CreateApplicationCacheStorage, GetSize)
{
    nnt::fs::util::DeleteAllTestSaveData();
    nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};
    int64_t size = 0;
    CacheStorageTargetMedia media;
    const uint16_t IndexNum = 3;

    // SD に作成
    for(uint16_t index = 0; index < IndexNum; index++)
    {
        nn::ns::ApplicationControlProperty property = GetDefaultProperty();
        property.cacheStorageSize *= index + 1;
        property.cacheStorageJournalSize *= index + 1;
        property.cacheStorageDataAndJournalSizeMax = property.cacheStorageSize + property.cacheStorageJournalSize;
        NNT_EXPECT_RESULT_SUCCESS(CreateApplicationCacheStorage(&size, &media, id, property, index, property.cacheStorageSize, property.cacheStorageJournalSize));
        ConfirmCacheStorageForCreate(id, property, SaveDataSpaceId::SdUser, index);
    }
    for(uint16_t index = 0; index < IndexNum; index++)
    {
        nn::ns::ApplicationControlProperty property = GetDefaultProperty();
        property.cacheStorageSize *= index + 1;
        property.cacheStorageJournalSize *= index + 1;
        int64_t cacheStorageSize = 0;
        int64_t cacheStorageJournalSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(GetCacheStorageSize(&cacheStorageSize, &cacheStorageJournalSize, index));
        EXPECT_TRUE(cacheStorageSize == property.cacheStorageSize);
        EXPECT_TRUE(cacheStorageJournalSize == property.cacheStorageJournalSize);
    }
    // index が存在しなければ TargetNotFound
    {
        int64_t cacheStorageSize = 0;
        int64_t cacheStorageJournalSize = 0;
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetCacheStorageSize(&cacheStorageSize, &cacheStorageJournalSize, IndexNum));
    }
    SetSdCardAccessibility(false);
    NN_UTIL_SCOPE_EXIT
    {
        SetSdCardAccessibility(true);
    };
    // NAND に作成
    for(uint16_t index = 0; index < IndexNum; index++)
    {
        nn::ns::ApplicationControlProperty property = GetDefaultProperty();
        property.cacheStorageSize *= index + 1;
        property.cacheStorageJournalSize *= index + 1;
        property.cacheStorageDataAndJournalSizeMax = property.cacheStorageSize + property.cacheStorageJournalSize;
        NNT_EXPECT_RESULT_SUCCESS(CreateApplicationCacheStorage(&size, &media, id, property, index, property.cacheStorageSize, property.cacheStorageJournalSize));
        ConfirmCacheStorageForCreate(id, property, SaveDataSpaceId::User, index);
    }

    for(uint16_t index = 0; index < IndexNum; index++)
    {
        nn::ns::ApplicationControlProperty property = GetDefaultProperty();
        property.cacheStorageSize *= index + 1;
        property.cacheStorageJournalSize *= index + 1;
        int64_t cacheStorageSize = 0;
        int64_t cacheStorageJournalSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(GetCacheStorageSize(&cacheStorageSize, &cacheStorageJournalSize, index));
        EXPECT_TRUE(cacheStorageSize == property.cacheStorageSize);
        EXPECT_TRUE(cacheStorageJournalSize == property.cacheStorageJournalSize);
    }
    // index が存在しなければ TargetNotFound
    {
        int64_t cacheStorageSize = 0;
        int64_t cacheStorageJournalSize = 0;
        NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, GetCacheStorageSize(&cacheStorageSize, &cacheStorageJournalSize, IndexNum));
    }
}

// キャッシュストレージを作りすぎた場合
TEST(CreateApplicationCacheStorage, IndexTooLarge)
{
    nnt::fs::util::DeleteAllTestSaveData();
    nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};
    int64_t size = 0;
    CacheStorageTargetMedia media;
    const uint16_t IndexNum = 3;

    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    property.cacheStorageIndexMax = IndexNum - 1;

    // SD に作成
    uint16_t index = 0;
    for(index = 0; index < IndexNum; index++)
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateApplicationCacheStorage(&size, &media, id, property, index, property.cacheStorageSize, property.cacheStorageJournalSize));
        ConfirmCacheStorageForCreate(id, property, SaveDataSpaceId::SdUser, index);
    }
    NNT_EXPECT_RESULT_FAILURE(ResultCacheStorageIndexTooLarge, CreateApplicationCacheStorage(&size, &media, id, property, index, property.cacheStorageSize, property.cacheStorageJournalSize));
    // 存在しないことを確認する
    CheckSaveDataExisting(id, SaveDataSpaceId::SdUser, SaveDataType::Cache, false, index);
}

// キャッシュストレージが大きすぎる場合
TEST(CreateApplicationCacheStorage, SizeTooLarge)
{
    nnt::fs::util::DeleteAllTestSaveData();
    nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};
    int64_t size = 0;
    CacheStorageTargetMedia media;
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    property.cacheStorageDataAndJournalSizeMax = property.cacheStorageSize + property.cacheStorageJournalSize - 16 * 1024;

    uint16_t index = 1;

    // SD に作成
    NNT_EXPECT_RESULT_FAILURE(ResultCacheStorageSizeTooLarge, CreateApplicationCacheStorage(&size, &media, id, property, index, property.cacheStorageSize, property.cacheStorageJournalSize));
    // 存在しないことを確認する
    CheckSaveDataExisting(id, SaveDataSpaceId::SdUser, SaveDataType::Cache, false, index);
}
void FillByOtherApplicationCacheStorage(uint64_t offset, int64_t cacheStorageSize, int64_t cacheStorageJournalSize, CacheStorageTargetMedia targetMedia)
{
    SaveDataSpaceId spaceId = nn::fs::SaveDataSpaceId::SdUser;
    int64_t freeSpaceSize = 0;

    if(targetMedia == CacheStorageTargetMedia_Sd)
    {
        // SD をいっぱいにする
        NNT_EXPECT_RESULT_SUCCESS(MountSdCard("sdcard"));
        NNT_EXPECT_RESULT_SUCCESS(GetFreeSpaceSize(&freeSpaceSize, "sdcard:/"));
        Unmount("sdcard");
        spaceId = nn::fs::SaveDataSpaceId::SdUser;
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&freeSpaceSize, "@User:/"));
        Unmount("@User");
        spaceId = nn::fs::SaveDataSpaceId::User;
    }
    uint64_t idOffset = offset;
    CacheStorageTargetMedia media;
    int64_t cacheUnitSize = 0;
    int64_t extendedCacheUnitSize = 0;

    NNT_EXPECT_RESULT_SUCCESS(QuerySaveDataTotalSize(&cacheUnitSize, cacheStorageSize, cacheStorageJournalSize));
    NNT_EXPECT_RESULT_SUCCESS(QuerySaveDataTotalSize(&extendedCacheUnitSize, cacheStorageSize * 2, cacheStorageJournalSize * 2));

    uint64_t unitNum = freeSpaceSize / cacheUnitSize;

    for(uint64_t i = 0; i < unitNum; i++)
    {
        int64_t size = 0;
        nn::ncm::ApplicationId padId = { nnt::fs::util::UserSaveDataApplicationId + idOffset };
        nn::ns::ApplicationControlProperty padProperty = GetPropertyWithCacheStorageSize(idOffset, cacheStorageSize, cacheStorageJournalSize);
        NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationCacheStorage(&size, &media, padId , padProperty));
        EXPECT_TRUE(media == targetMedia);
        ConfirmCacheStorageForEnsure(padId, padProperty, spaceId);
        // 最後だけ Extend しようとしたら NotEnoughSpace
        if(i == unitNum - 1)
        {
            padProperty = GetPropertyWithCacheStorageSize(idOffset, cacheStorageSize * 2, cacheStorageJournalSize * 2);
            NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureApplicationCacheStorage(&size, &media, padId, padProperty));
            EXPECT_TRUE(media == targetMedia);
            EXPECT_GT(size, extendedCacheUnitSize - cacheUnitSize);
            EXPECT_LT(size, extendedCacheUnitSize - cacheUnitSize + 4 * 1024 * 1024);
        }
        idOffset++;
    }
}

void FillByThisApplicationCacheStorageIndex(nn::ncm::ApplicationId id, int64_t cacheStorageSize, int64_t cacheStorageJournalSize, CacheStorageTargetMedia targetMedia, uint64_t maxNum)
{
    int64_t freeSpaceSize = 0;
    SaveDataSpaceId spaceId = nn::fs::SaveDataSpaceId::SdUser;

    if(targetMedia == CacheStorageTargetMedia_Sd)
    {
        // SD をいっぱいにする
        NNT_EXPECT_RESULT_SUCCESS(MountSdCard("sdcard"));
        NNT_EXPECT_RESULT_SUCCESS(GetFreeSpaceSize(&freeSpaceSize, "sdcard:/"));
        Unmount("sdcard");
        spaceId = nn::fs::SaveDataSpaceId::SdUser;
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&freeSpaceSize, "@User:/"));
        Unmount("@User");
        spaceId = nn::fs::SaveDataSpaceId::User;
    }
    CacheStorageTargetMedia media;
    int64_t cacheUnitSize = 0;
    NNT_EXPECT_RESULT_SUCCESS(QuerySaveDataTotalSize(&cacheUnitSize, cacheStorageSize, cacheStorageJournalSize));
    uint64_t unitNum = freeSpaceSize / cacheUnitSize;
    uint64_t cacheStorageNum = unitNum < maxNum ? unitNum : maxNum;

    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    property.cacheStorageSize = cacheStorageSize;
    property.cacheStorageJournalSize = cacheStorageJournalSize;
    property.cacheStorageDataAndJournalSizeMax  = cacheStorageSize + cacheStorageJournalSize;
    property.cacheStorageIndexMax = cacheStorageNum;
    for(uint64_t i = 0; i < cacheStorageNum; i++)
    {
        int64_t size = 0;
        NN_LOG("%llx : %d\n", id, i);
        NNT_EXPECT_RESULT_SUCCESS(CreateApplicationCacheStorage(&size, &media, id , property, i, cacheStorageSize, cacheStorageJournalSize));
        EXPECT_TRUE(media == targetMedia);
        ConfirmCacheStorageForCreate(id, property, spaceId, i);
    }
}

void VerifySpaceNotEnough(nn::ncm::ApplicationId id, int64_t cacheStorageSize, int64_t cacheStorageJournalSize, bool isEnsureApplication)
{
    int64_t size = 0;
    CacheStorageTargetMedia media;
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    property.cacheStorageSize = cacheStorageSize;
    property.cacheStorageJournalSize = cacheStorageJournalSize;
    property.cacheStorageDataAndJournalSizeMax = cacheStorageSize + cacheStorageJournalSize;
    property.cacheStorageIndexMax = 1000;
    if(isEnsureApplication)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureApplicationCacheStorage(&size, &media, id, property));
    }
    else
    {
        NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, CreateApplicationCacheStorage(&size, &media, id , property, 1000, cacheStorageSize, cacheStorageJournalSize));
    }
    EXPECT_TRUE(media == CacheStorageTargetMedia_Any);
    int64_t cacheUnitSize = 0;
    NNT_EXPECT_RESULT_SUCCESS(QuerySaveDataTotalSize(&cacheUnitSize, cacheStorageSize, cacheStorageJournalSize));
    EXPECT_GT(size, cacheUnitSize);
    EXPECT_LT(size, cacheUnitSize + 1 * 1024 * 1024);
    // (SD 無効にして) CreateApplicationCacheStorage を呼ぶと、NAND に作ろうして Media が Any で返ってくる
    SetSdCardAccessibility(false);
    if(isEnsureApplication)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureApplicationCacheStorage(&size, &media, id, property));
    }
    else
    {
        NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, CreateApplicationCacheStorage(&size, &media, id , property, 1000, cacheStorageSize, cacheStorageJournalSize));
    }
    EXPECT_TRUE(media == CacheStorageTargetMedia_Any);
    EXPECT_GT(size, cacheUnitSize);
    EXPECT_LT(size, cacheUnitSize + 1 * 1024 * 1024);
    SetSdCardAccessibility(true);
}

// キャッシュストレージ容量不足テスト
TEST(EnsureApplicationCacheStorage, UsableSpaceNotEnoughHeavy)
{
    EXPECT_TRUE(IsSdCardAccessible());
    nnt::fs::util::DeleteAllTestSaveData();
    nn::ncm::ApplicationId id = { nnt::fs::util::UserSaveDataApplicationId + 1};
    int64_t cacheStorageSize = 512 * 1024 * 1024;
    int64_t cacheStorageJournalSize = 512 * 1024 * 1024;

    // SD を埋める
    FillByOtherApplicationCacheStorage(0x100, cacheStorageSize, cacheStorageJournalSize, CacheStorageTargetMedia_Sd);
    // NAND を埋める
    FillByOtherApplicationCacheStorage(0x200, cacheStorageSize, cacheStorageJournalSize, CacheStorageTargetMedia_Nand);
    VerifySpaceNotEnough(id, cacheStorageSize, cacheStorageJournalSize, true);
    NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnoughForCacheStorage, CreateCacheStorage(255, cacheStorageSize, cacheStorageJournalSize));
}

TEST(CreateApplicationCacheStorage, UsableSpaceNotEnoughHeavy)
{
    EXPECT_TRUE(IsSdCardAccessible());
    nnt::fs::util::DeleteAllTestSaveData();
    nn::ncm::ApplicationId id = { nnt::fs::util::UserSaveDataApplicationId };
    nn::ncm::ApplicationId padId  = { nnt::fs::util::UserSaveDataApplicationId + 1 };
    nn::ncm::ApplicationId padId2 = { nnt::fs::util::UserSaveDataApplicationId + 2 };
    int64_t cacheStorageSize = 512 * 1024 * 1024;
    int64_t cacheStorageJournalSize = 512 * 1024 * 1024;
    // SD をいっぱいにする
    FillByThisApplicationCacheStorageIndex(padId, cacheStorageSize, cacheStorageJournalSize, CacheStorageTargetMedia_Sd, 100);
    SetSdCardAccessibility(false);
    // NAND をいっぱいにする(SD と同じ ID)
    FillByThisApplicationCacheStorageIndex(padId, cacheStorageSize, cacheStorageJournalSize, CacheStorageTargetMedia_Nand, 100);
    SetSdCardAccessibility(true);

    VerifySpaceNotEnough(id, cacheStorageSize, cacheStorageJournalSize, false);

    // 一旦 NAND をクリーン
    while(NN_STATIC_CONDITION(true))
    {
        nn::util::optional<nn::fs::SaveDataInfo> info;
        nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, [&](const nn::fs::SaveDataInfo& i)
        {
            return i.saveDataType == nn::fs::SaveDataType::Cache;
        });
        if( info == nn::util::nullopt)
        {
            break;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::User, info->saveDataId));
    }
    // NAND をいっぱいにする(SD と違う ID が勝手に NAND に入る)
    FillByThisApplicationCacheStorageIndex(padId2, cacheStorageSize, cacheStorageJournalSize, CacheStorageTargetMedia_Nand, 100);

    VerifySpaceNotEnough(id, cacheStorageSize, cacheStorageJournalSize, false);
    NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnoughForCacheStorage, CreateCacheStorage(255, cacheStorageSize, cacheStorageJournalSize));
}
#endif
// bcat セーブ単体作成
TEST(EnsureApplicationBcatDeliveryCacheStorage, Create)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = { nnt::fs::util::UserSaveDataApplicationId };
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();

    // 作成
    int64_t size;
    {
        auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationBcatDeliveryCacheStorage(&size, id, property));
        auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
    }

    // 確認: bcat セーブ以外は作成されていないこと
    {
        CheckSaveDataExisting(id, SaveDataType::Account,   false);
        CheckSaveDataExisting(id, SaveDataType::Device,    false);
        CheckSaveDataExisting(id, SaveDataType::Temporary, false);
        CheckSaveDataExisting(id, SaveDataType::Cache,     false);
    }
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Bcat, true);
        CheckSaveData(info.value(), property, nn::account::InvalidUid);
    }

    // 2回目以降は何もせず即 OK を返す
    {
        auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationBcatDeliveryCacheStorage(&size, id, property));
        auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

#if !defined(NN_BUILD_CONFIG_OS_WIN) // win 版の時間をチェックする意味は薄いので省略
        EXPECT_LT((end - start).GetMilliSeconds(), 100);
#endif
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

#if !defined(NN_BUILD_CONFIG_OS_WIN) // win 未対応

struct ExtendParam
{
    int64_t userAccountSaveDataExtendSize;
    int64_t userAccountSaveDataExtendJournalSize;
    int64_t deviceSaveDataExtendSize;
    int64_t deviceSaveDataJournalExtendSize;
    int64_t bcatDeliveryCacheStorageExtendSize;
    int64_t cacheStorageExtendSize;
    int64_t cacheStorageExtendJournalSize;
};

class ExtendByEnsureApplicationSaveData : public ::testing::TestWithParam< ExtendParam >
{
};

// セーブデータ拡張
TEST_P(ExtendByEnsureApplicationSaveData, ExtendHeavy)
{
    auto param = GetParam();

    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = { nnt::fs::util::UserSaveDataApplicationId };
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::account::Uid user = GetDefaultUid();

    // 作成
    {
        int64_t size;
        auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationSaveData(&size, id, property, user));
        auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NN_LOG("elapsed time to create: %8lld ms.\n", (end - start).GetMilliSeconds());
    }

    // property 更新
    property.userAccountSaveDataSize        += param.userAccountSaveDataExtendSize;
    property.userAccountSaveDataJournalSize += param.userAccountSaveDataExtendJournalSize;
    property.deviceSaveDataSize             += param.deviceSaveDataExtendSize;
    property.deviceSaveDataJournalSize      += param.deviceSaveDataJournalExtendSize;
    property.bcatDeliveryCacheStorageSize   += param.bcatDeliveryCacheStorageExtendSize;

    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));
    {
        int64_t requiredSize;
        {
            nnt::fs::util::ScopedPadder padder("@User:/", 64 * 1024); // 残り 64KB 以下に埋める

            // 容量不足で拡張失敗
            NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureApplicationSaveData(&requiredSize, id, property, user));

            const int64_t ExpectedContextFileSize =
                4 * 1024 * 1024                       // 3 * (1MB + α)
                + (
                     property.userAccountSaveDataSize
                   + property.userAccountSaveDataJournalSize
                   + property.deviceSaveDataSize
                   + property.deviceSaveDataJournalSize
                   + property.bcatDeliveryCacheStorageSize
                   + 2 * 1024 * 1024
                    ) / 1024;

            NN_LOG("%s %d: %lld\n", __FUNCTION__, __LINE__, ExpectedContextFileSize);


            const int64_t ExpectedRequiredSizeLower =
                param.userAccountSaveDataExtendSize
                + param.userAccountSaveDataExtendJournalSize
                + param.deviceSaveDataExtendSize
                + param.deviceSaveDataJournalExtendSize
                + param.bcatDeliveryCacheStorageExtendSize;

            EXPECT_LT(ExpectedRequiredSizeLower, requiredSize);
            EXPECT_LT(requiredSize, ExpectedRequiredSizeLower * 1.005 + ExpectedContextFileSize); // メタデータ込
            NN_LOG("required: %lld\n", requiredSize);
        }

        nnt::fs::util::ScopedPadder padder2("@User:/", requiredSize + 16 * 1024); // 拡張できるギリギリに埋めなおす
        {
            // (現在空き容量確認)
            int64_t size;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&size, "@User:/"));
            NN_LOG("free: %lld\n", size);
            EXPECT_LE(size, requiredSize); // (ScopedPadder は多少埋めすぎる場合がある)
        }

        // 拡張
        {
            int64_t size;
            auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationSaveData(&size, id, property, user));
            auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            NN_LOG("elapsed time to extend: %8lld ms.\n", (end - start).GetMilliSeconds());
        }


        // 確認
        {
            auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
            CheckSaveData(info.value(), property, user);
        }
        {
            auto info = CheckSaveDataExisting(id, SaveDataType::Device, true);
            CheckSaveData(info.value(), property, nn::account::InvalidUid);
        }
        {
            auto info = CheckSaveDataExisting(id, SaveDataType::Bcat, true);
            CheckSaveData(info.value(), property, nn::account::InvalidUid);
        }
        {
            auto info = CheckSaveDataExisting(id, SaveDataType::Temporary, true);
            CheckSaveData(info.value(), property, nn::account::InvalidUid);
        }
        {
            CheckSaveDataExisting(id, SaveDataType::Cache, false);
        }

        // 2回目以降は何もせず即 OK を返す
        {
            int64_t size;
            auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationSaveData(&size, id, property, user));
            auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            EXPECT_LT((end - start).GetMilliSeconds(), 100);
        }
    }
    Unmount("@User");

    nnt::fs::util::DeleteAllTestSaveData();
}

ExtendParam extendParam1 = {
    1 * 1024 * 1024,
    2 * 1024 * 1024,
    3 * 1024 * 1024,
    4 * 1024 * 1024,
    5 * 1024 * 1024,
    6 * 1024 * 1024,
    7 * 1024 * 1024,
};

ExtendParam extendParam2 = {
    (2000 - 20) * 1024 * 1024,
    (2000 - 15) * 1024 * 1024,
    (2000 - 10) * 1024 * 1024,
    (2000 - 5) * 1024 * 1024,
    (2000 - 8 - 2) * 1024 * 1024,
    (2000 - 30) * 1024 * 1024,
    (2000 - 18) * 1024 * 1024,
};

INSTANTIATE_TEST_CASE_P(
    ExtendByEnsureApplicationSaveDataSmoke,
    ExtendByEnsureApplicationSaveData,
    ::testing::Values(extendParam1)
);

INSTANTIATE_TEST_CASE_P(
    ExtendByEnsureApplicationSaveDataSmokeHugeHeavy,
    ExtendByEnsureApplicationSaveData,
    ::testing::Values(extendParam2)
);
// 空き容量埋めに1分程度かかるため heavy
TEST(EnsureApplicationSaveData, UsableSpaceNotEnoughHeavy)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};

    auto GetEmptyProperty = []() {
        nn::ns::ApplicationControlProperty property = {};
        property.saveDataOwnerId = nnt::fs::util::UserSaveDataApplicationId + 1;
        return property;
    };

    nn::account::Uid user;
    user._data[0] = 1;
    user._data[1] = 0;


    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));

    {
        nnt::fs::util::ScopedPadder padder("@User:/", 64 * 1024 * 1024); // 残り 64MB 以下に埋める
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::ListDirectoryRecursive("@User:/"));

        // account
        {
            int64_t size;
            auto property = GetEmptyProperty();
            property.userAccountSaveDataSize        = 64 * 1024 * 1024;
            property.userAccountSaveDataJournalSize = 60 * 1024 * 1024;

            NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureApplicationSaveData(&size, id, property, user));

            EXPECT_GT(size, property.userAccountSaveDataSize + property.userAccountSaveDataJournalSize);
            EXPECT_LT(size, property.userAccountSaveDataSize + property.userAccountSaveDataJournalSize + 1 * 1024 * 1024);
        }

        // device
        {
            int64_t size;
            auto property = GetEmptyProperty();
            property.deviceSaveDataSize             = 128 * 1024 * 1024;
            property.deviceSaveDataJournalSize      = 120 * 1024 * 1024;

            NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureApplicationSaveData(&size, id, property, nn::account::InvalidUid));

            EXPECT_GT(size, property.deviceSaveDataSize + property.deviceSaveDataJournalSize);
            EXPECT_LT(size, property.deviceSaveDataSize + property.deviceSaveDataJournalSize + 1 * 1024 * 1024);
        }

        // bcat
        {
            int64_t size;
            auto property = GetEmptyProperty();
            property.bcatDeliveryCacheStorageSize   = 96 * 1024 * 1024;

            NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureApplicationSaveData(&size, id, property, nn::account::InvalidUid));

            EXPECT_GT(size, property.bcatDeliveryCacheStorageSize + detail::BcatSaveDataJournalSize);
            EXPECT_LT(size, property.bcatDeliveryCacheStorageSize + detail::BcatSaveDataJournalSize + 1 * 1024 * 1024);
        }

        // temp
        {
            int64_t size;
            auto property = GetEmptyProperty();
            property.temporaryStorageSize = 200 * 1024 * 1024;

            NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureApplicationSaveData(&size, id, property, nn::account::InvalidUid));

            EXPECT_GT(size, property.temporaryStorageSize);
            EXPECT_LT(size, property.temporaryStorageSize + 2 * 1024 * 1024);
        }

        // all: account + device + bcat + temp
        {
            int64_t size;
            auto property = GetEmptyProperty();
            property.userAccountSaveDataSize        =  64 * 1024 * 1024;
            property.userAccountSaveDataJournalSize =  60 * 1024 * 1024;
            property.deviceSaveDataSize             = 128 * 1024 * 1024;
            property.deviceSaveDataJournalSize      = 120 * 1024 * 1024;
            property.bcatDeliveryCacheStorageSize   =  96 * 1024 * 1024;
            property.temporaryStorageSize    = 200 * 1024 * 1024;

            NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureApplicationSaveData(&size, id, property, user));

            auto totalSize = property.userAccountSaveDataSize
                + property.userAccountSaveDataJournalSize
                + property.deviceSaveDataSize
                + property.deviceSaveDataJournalSize
                + property.bcatDeliveryCacheStorageSize
                + detail::BcatSaveDataJournalSize
                + property.temporaryStorageSize;
            EXPECT_GT(size, totalSize);
            EXPECT_LT(size, totalSize + 6 * 1024 * 1024);
        }

        // いずれも作成失敗していること
        {
            CheckSaveDataExisting(id, SaveDataType::Account,   false);
            CheckSaveDataExisting(id, SaveDataType::Device,    false);
            CheckSaveDataExisting(id, SaveDataType::Bcat,      false);
            CheckSaveDataExisting(id, SaveDataType::Temporary, false);
            CheckSaveDataExisting(id, SaveDataType::Cache,     false);
        }


    }

    Unmount("@User");
    nnt::fs::util::DeleteAllTestSaveData();
}
#endif


#if !defined(NN_BUILD_CONFIG_OS_WIN) // win 未対応

TEST(EnsureSaveData, Create)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};

    // @pre 以下のアプリ管理データが読める状態で実行
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::account::Uid user = GetDefaultUid();

    // 作成
    {
        auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));
        auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
    }

    // 確認
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
        CheckSaveData(info.value(), property, user);
    }
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Device, true);
        CheckSaveData(info.value(), property, nn::account::InvalidUid);
    }
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Bcat, true);
        CheckSaveData(info.value(), property, nn::account::InvalidUid);
    }
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Temporary, true);
        CheckSaveData(info.value(), property, nn::account::InvalidUid);
    }
    // キャッシュストレージはないことを確認
    {
        CheckSaveDataExisting(id, SaveDataType::Cache, false);
    }

    // 2回目以降は何もせず即 OK を返す
    {
        auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));
        auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        EXPECT_LT((end - start).GetMilliSeconds(), 100);
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(ExtendSaveData, Extend)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};

    // @pre 以下のアプリ管理データが読める状態で実行
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::account::Uid user = GetDefaultUid();
    auto invalidUser = nn::account::InvalidUid;

    // 作成
    {
        auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));
        auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
    }

    // 確認
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
        CheckSaveData(info.value(), property, user);
    }
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Device, true);
        CheckSaveData(info.value(), property, invalidUser);
    }

    // 2回目以降は何もせず即 OK を返す
    {
        auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));
        auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        EXPECT_LT((end - start).GetMilliSeconds(), 100);
    }

    // 拡張
    int64_t userAccountSaveDataSizeInc = static_cast<int64_t>(nnt::fs::util::RoundUp(property.userAccountSaveDataSize >> 1, nn::fs::SaveDataExtensionUnitSize));
    int64_t userAccountSaveDataJournalSizeInc = static_cast<int64_t>(nnt::fs::util::RoundUp(property.userAccountSaveDataJournalSize >> 1, nn::fs::SaveDataExtensionUnitSize));
    int64_t deviceSaveDataSizeInc = static_cast<int64_t>(nnt::fs::util::RoundUp(property.deviceSaveDataSize >> 1, nn::fs::SaveDataExtensionUnitSize));
    int64_t deviceSaveDataJournalSizeInc = static_cast<int64_t>(nnt::fs::util::RoundUp(property.deviceSaveDataJournalSize >> 1, nn::fs::SaveDataExtensionUnitSize));

    int64_t newUserAccountSaveDataSize = property.userAccountSaveDataSize;
    int64_t newUserAccountSaveDataJournalSize = property.userAccountSaveDataJournalSize;
    int64_t newDeviceSaveDataSize = property.deviceSaveDataSize;
    int64_t newDeviceSaveDataJournalSize = property.deviceSaveDataJournalSize;

    while (NN_STATIC_CONDITION(true))
    {
        newUserAccountSaveDataSize += userAccountSaveDataSizeInc;
        newUserAccountSaveDataJournalSize += userAccountSaveDataJournalSizeInc;
        newDeviceSaveDataSize += deviceSaveDataSizeInc;
        newDeviceSaveDataJournalSize += deviceSaveDataJournalSizeInc;

        if (newUserAccountSaveDataSize > property.userAccountSaveDataSizeMax ||
            newUserAccountSaveDataJournalSize > property.userAccountSaveDataJournalSizeMax ||
            newDeviceSaveDataSize > property.deviceSaveDataSizeMax ||
            newDeviceSaveDataJournalSize > property.deviceSaveDataJournalSizeMax)
        {
            break;
        }

        // ユーザーアカウントセーブ
        {
            auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
            NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData(user, newUserAccountSaveDataSize, newUserAccountSaveDataJournalSize));
            auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
            NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
        }
        {
            auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
            CheckSaveData(info.value(), property, user, newUserAccountSaveDataSize, newUserAccountSaveDataJournalSize);
        }
        // 本体セーブ
#if 0
        {
            auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
            NNT_EXPECT_RESULT_SUCCESS(ExtendDeviceSaveData(newDeviceSaveDataSize, newDeviceSaveDataJournalSize));
            auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
            NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
        }
        {
            auto info = CheckSaveDataExisting(id, SaveDataType::Device, true);
            CheckSaveData(info.value(), property, invalidUser, newDeviceSaveDataSize, newDeviceSaveDataJournalSize);
        }
#endif
    }

    // 拡張後のセーブデータ作成も何もせず即 OK を返す
    {
        auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));
        auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        EXPECT_LT((end - start).GetMilliSeconds(), 100);
    }

    // 上限まで拡張
    {
        NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData(user, property.userAccountSaveDataSizeMax, property.userAccountSaveDataJournalSizeMax));
#if 0
        NNT_EXPECT_RESULT_SUCCESS(ExtendDeviceSaveData(property.deviceSaveDataSizeMax, property.deviceSaveDataJournalSizeMax));
#endif
    }

    // 上限を超える拡張はできない
    {
        NNT_EXPECT_RESULT_FAILURE(ResultExtensionSizeTooLarge, ExtendSaveData(user, property.userAccountSaveDataSizeMax + nn::fs::SaveDataExtensionUnitSize, property.userAccountSaveDataJournalSizeMax + nn::fs::SaveDataExtensionUnitSize));
#if 0
        NNT_EXPECT_RESULT_FAILURE(ResultExtensionSizeTooLarge, ExtendDeviceSaveData(property.deviceSaveDataSizeMax + nn::fs::SaveDataExtensionUnitSize, property.deviceSaveDataJournalSizeMax + nn::fs::SaveDataExtensionUnitSize));
#endif
    }

    // エラー後もサイズは変わらずマウント可能
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
        CheckSaveData(info.value(), property, user, property.userAccountSaveDataSizeMax, property.userAccountSaveDataJournalSizeMax);
    }
#if 0
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Device, true);
        CheckSaveData(info.value(), property, invalidUser, property.deviceSaveDataSizeMax, property.deviceSaveDataJournalSizeMax);
    }
#endif

    // 既に拡張済みであれば何もせず OK を返す
    {
        auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData(user, property.userAccountSaveDataSize, property.userAccountSaveDataJournalSize));
        auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
    }
#if 0
    {
        auto start = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NNT_EXPECT_RESULT_SUCCESS(ExtendDeviceSaveData(property.deviceSaveDataSize, property.deviceSaveDataJournalSize));
        auto end = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() );
        NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
    }
#endif

    nnt::fs::util::DeleteAllTestSaveData();
} // NOLINT(impl/function_size)


// 非 1MB アラインの初期セーブから 1MB のアラインのサイズに拡張できること
TEST(ExtendSaveData, ExtendFromSizeNotAligned)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = { nnt::fs::util::UserSaveDataApplicationId };

    // @pre 以下のアプリ管理データが読める状態で実行
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::account::Uid user = GetDefaultUid();
    auto invalidUser = nn::account::InvalidUid;
    NN_UNUSED(invalidUser);

    // 非 1MB アラインサイズな初期セーブ
    property.userAccountSaveDataSize        += 16 * 1024;
    property.userAccountSaveDataJournalSize += 16 * 1024;
    property.deviceSaveDataSize             += 16 * 1024;
    property.deviceSaveDataJournalSize      += 16 * 1024;

    // 作成
    {
        auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));
        auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
    }

    // 2回目以降は何もせず即 OK を返す＠非 1MB アライン
    {
        auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));
        auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        EXPECT_LT((end - start).GetMilliSeconds(), 100);
    }

    // 拡張
    int64_t userAccountSaveDataSizeInc        = static_cast<int64_t>(nnt::fs::util::RoundUp(property.userAccountSaveDataSize        >> 1, nn::fs::SaveDataExtensionUnitSize));
    int64_t userAccountSaveDataJournalSizeInc = static_cast<int64_t>(nnt::fs::util::RoundUp(property.userAccountSaveDataJournalSize >> 1, nn::fs::SaveDataExtensionUnitSize));
    int64_t deviceSaveDataSizeInc             = static_cast<int64_t>(nnt::fs::util::RoundUp(property.deviceSaveDataSize             >> 1, nn::fs::SaveDataExtensionUnitSize));
    int64_t deviceSaveDataJournalSizeInc      = static_cast<int64_t>(nnt::fs::util::RoundUp(property.deviceSaveDataJournalSize      >> 1, nn::fs::SaveDataExtensionUnitSize));

    int64_t newUserAccountSaveDataSize        = property.userAccountSaveDataSize;
    int64_t newUserAccountSaveDataJournalSize = property.userAccountSaveDataJournalSize;
    int64_t newDeviceSaveDataSize             = property.deviceSaveDataSize;
    int64_t newDeviceSaveDataJournalSize      = property.deviceSaveDataJournalSize;

    while (NN_STATIC_CONDITION(true))
    {
        // 拡張結果サイズは常に 1MB アラインである必要がある
        newUserAccountSaveDataSize        = nnt::fs::util::RoundUp(newUserAccountSaveDataSize        + userAccountSaveDataSizeInc,        nn::fs::SaveDataExtensionUnitSize);
        newUserAccountSaveDataJournalSize = nnt::fs::util::RoundUp(newUserAccountSaveDataJournalSize + userAccountSaveDataJournalSizeInc, nn::fs::SaveDataExtensionUnitSize);
        newDeviceSaveDataSize             = nnt::fs::util::RoundUp(newDeviceSaveDataSize             + deviceSaveDataSizeInc,             nn::fs::SaveDataExtensionUnitSize);
        newDeviceSaveDataJournalSize      = nnt::fs::util::RoundUp(newDeviceSaveDataJournalSize      + deviceSaveDataJournalSizeInc,      nn::fs::SaveDataExtensionUnitSize);

        if (newUserAccountSaveDataSize > property.userAccountSaveDataSizeMax ||
            newUserAccountSaveDataJournalSize > property.userAccountSaveDataJournalSizeMax ||
            newDeviceSaveDataSize > property.deviceSaveDataSizeMax ||
            newDeviceSaveDataJournalSize > property.deviceSaveDataJournalSizeMax)
        {
            break;
        }

        // ユーザーアカウントセーブ
        {
            auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData(user, newUserAccountSaveDataSize, newUserAccountSaveDataJournalSize));
            auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
        }
        {
            auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
            CheckSaveData(info.value(), property, user, newUserAccountSaveDataSize, newUserAccountSaveDataJournalSize);
        }
        // 本体セーブ
#if 0
        {
            auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            NNT_EXPECT_RESULT_SUCCESS(ExtendDeviceSaveData(newDeviceSaveDataSize, newDeviceSaveDataJournalSize));
            auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
            NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
        }
        {
            auto info = CheckSaveDataExisting(id, SaveDataType::Device, true);
            CheckSaveData(info.value(), property, invalidUser, newDeviceSaveDataSize, newDeviceSaveDataJournalSize);
        }
#endif
    }

    // 拡張後のセーブデータ作成も何もせず即 OK を返す
    {
        auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));
        auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        EXPECT_LT((end - start).GetMilliSeconds(), 100);
    }

    // 上限まで拡張
    {
        NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData(user, property.userAccountSaveDataSizeMax, property.userAccountSaveDataJournalSizeMax));
#if 0
        NNT_EXPECT_RESULT_SUCCESS(ExtendDeviceSaveData(property.deviceSaveDataSizeMax, property.deviceSaveDataJournalSizeMax));
#endif
    }

    // 上限を超える拡張はできない
    {
        NNT_EXPECT_RESULT_FAILURE(ResultExtensionSizeTooLarge, ExtendSaveData(user, property.userAccountSaveDataSizeMax + nn::fs::SaveDataExtensionUnitSize, property.userAccountSaveDataJournalSizeMax + nn::fs::SaveDataExtensionUnitSize));
#if 0
        NNT_EXPECT_RESULT_FAILURE(ResultExtensionSizeTooLarge, ExtendDeviceSaveData(property.deviceSaveDataSizeMax + nn::fs::SaveDataExtensionUnitSize, property.deviceSaveDataJournalSizeMax + nn::fs::SaveDataExtensionUnitSize));
#endif
    }

    // エラー後もサイズは変わらずマウント可能
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
        CheckSaveData(info.value(), property, user, property.userAccountSaveDataSizeMax, property.userAccountSaveDataJournalSizeMax);
    }
#if 0
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Device, true);
        CheckSaveData(info.value(), property, invalidUser, property.deviceSaveDataSizeMax, property.deviceSaveDataJournalSizeMax);
    }
#endif

    // 既に拡張済みであれば何もせず OK を返す
    {
        auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NNT_EXPECT_RESULT_SUCCESS(ExtendSaveData(user, property.userAccountSaveDataSize, property.userAccountSaveDataJournalSize));
        auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
    }
#if 0
    {
        auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NNT_EXPECT_RESULT_SUCCESS(ExtendDeviceSaveData(property.deviceSaveDataSize, property.deviceSaveDataJournalSize));
        auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        NN_LOG("elapsed: %8lld ms.\n", (end - start).GetMilliSeconds());
    }
#endif

    nnt::fs::util::DeleteAllTestSaveData();
} // NOLINT(impl/function_size)

TEST(ExtendSaveData, ResultExtensionSizeInvalid)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};

    // @pre 以下のアプリ管理データが読める状態で実行
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::account::Uid user = GetDefaultUid();

    // 作成
    NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));

    // 拡張サイズの事前条件違反でエラー
    {
        NNT_EXPECT_RESULT_FAILURE(ResultExtensionSizeInvalid, ExtendSaveData(user, property.userAccountSaveDataSize + 1, property.userAccountSaveDataJournalSize));
        NNT_EXPECT_RESULT_FAILURE(ResultExtensionSizeInvalid, ExtendSaveData(user, property.userAccountSaveDataSize, property.userAccountSaveDataJournalSize + 1));
    }

    // エラー後もサイズは変わらずマウント可能
    {
        auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
        CheckSaveData(info.value(), property, user);
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(ExtendSaveData, NeedsUnmounted)
{
    nnt::fs::util::DeleteAllTestSaveData();

    // @pre 以下のアプリ管理データが読める状態で実行
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::account::Uid user = GetDefaultUid();

    // 作成
    NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));

    const char* mountName = "save";

    // マウント中は拡張できない
    {
        NNT_EXPECT_RESULT_SUCCESS(MountSaveData(mountName, user));
        NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, ExtendSaveData(user, property.userAccountSaveDataSizeMax, property.userAccountSaveDataJournalSizeMax));
        Unmount(mountName);
#if 0
        NNT_EXPECT_RESULT_SUCCESS(MountDeviceSaveData(mountName));
        NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, ExtendDeviceSaveData(property.deviceSaveDataSizeMax, property.deviceSaveDataJournalSizeMax));
        Unmount(mountName);
#endif
    }
}

TEST(ExtendSaveData, ExtensionCountReachLimit)
{
    nnt::fs::util::DeleteAllTestSaveData();

    // @pre 以下のアプリ管理データが読める状態で実行
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::account::Uid user = GetDefaultUid();

    // 作成
    NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));

    const int ExtensionCountLimit = 512;
    int count = 0;

    int64_t saveDataSizeInc = 1 * 1024 * 1024;

    int64_t newUserAccountSaveDataSize = property.userAccountSaveDataSize;
    int64_t newUserAccountSaveDataJournalSize = property.userAccountSaveDataJournalSize;
    int64_t lastUserAccountSaveDataSize = 0;
    int64_t lastUserAccountSaveDataJournalSize = 0;

    // 拡張回数が限界に達したのでエラー
    nn::Result result;
    while (count < ExtensionCountLimit)
    {
        if (newUserAccountSaveDataSize != property.userAccountSaveDataSizeMax)
        {
            newUserAccountSaveDataSize += saveDataSizeInc;
        }
        else if (newUserAccountSaveDataJournalSize != property.userAccountSaveDataJournalSizeMax)
        {
            newUserAccountSaveDataJournalSize += saveDataSizeInc;
        }
        else
        {
            break;
        }

        result = ExtendSaveData(user, newUserAccountSaveDataSize, newUserAccountSaveDataJournalSize);
        if (result.IsFailure())
        {
            break;
        }

        lastUserAccountSaveDataSize = newUserAccountSaveDataSize;
        lastUserAccountSaveDataJournalSize = newUserAccountSaveDataJournalSize;
        count++;
    }

    NN_LOG("count: %d\n", count);
    NN_LOG("result: %08x\n", result.GetInnerValueForDebug());
    EXPECT_TRUE(ResultMapFull::Includes(result));

    // エラー後もサイズは変わらずマウント可能
    {
        nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};
        auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
        CheckSaveData(info.value(), property, user, lastUserAccountSaveDataSize, lastUserAccountSaveDataJournalSize);
    }

    nnt::fs::util::DeleteAllTestSaveData();
}
// 空き容量埋めるため heavy
TEST(EnsureSaveData, UsableSpaceNotEnoughHeavy)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};

    nn::ns::ApplicationControlProperty property = {};
    property.saveDataOwnerId = nnt::fs::util::UserSaveDataApplicationId + 1;

    nn::account::Uid user = GetDefaultUid();


    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));

    {
        nnt::fs::util::ScopedPadder padder("@User:/", 52 * 1024 * 1024); // bcat セーブのみ作成できない程度のパディング
        {

            nnt::fs::util::ScopedPadder padder("@User:/", 36 * 1024 * 1024); // bcat と本体セーブが作成できない程度のパディング
            {
                nnt::fs::util::ScopedPadder padder("@User:/", 1 * 1024 * 1024); // 何も作成できない程度のパディング
                NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::ListDirectoryRecursive("@User:/"));

                // ShowError が呼ばれる
                NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureSaveData(user));

                // いずれも作成失敗
                CheckSaveDataExisting(id, SaveDataType::Account, false);
                CheckSaveDataExisting(id, SaveDataType::Device,  false);
                CheckSaveDataExisting(id, SaveDataType::Bcat,    false);
            }

            // ShowError が呼ばれる
            NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureSaveData(user));

            // ユーザアカウント作成には成功
            CheckSaveDataExisting(id, SaveDataType::Account, true);
            CheckSaveDataExisting(id, SaveDataType::Device,  false);
            CheckSaveDataExisting(id, SaveDataType::Bcat,    false);
        }

        // ShowError が呼ばれる
        NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureSaveData(user));

        // ユーザアカウント、本体セーブ作成には成功
        CheckSaveDataExisting(id, SaveDataType::Account, true);
        CheckSaveDataExisting(id, SaveDataType::Device,  true);
        CheckSaveDataExisting(id, SaveDataType::Bcat,    false);

    }

    NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));

    // 全ての作成に成功
    CheckSaveDataExisting(id, SaveDataType::Account, true);
    CheckSaveDataExisting(id, SaveDataType::Device,  true);
    CheckSaveDataExisting(id, SaveDataType::Bcat,    true);


    Unmount("@User");
    nnt::fs::util::DeleteAllTestSaveData();
}
// 空き容量埋めるため heavy
TEST(ExtendSaveData, UsableSpaceNotEnoughHeavy)
{
    nnt::fs::util::DeleteAllTestSaveData();

    // @pre 以下のアプリ管理データが読める状態で実行
    nn::ns::ApplicationControlProperty property = GetDefaultProperty();
    nn::account::Uid user = GetDefaultUid();

    // 作成
    NNT_EXPECT_RESULT_SUCCESS(EnsureSaveData(user));

    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));
    {
        nnt::fs::util::ScopedPadder padder("@User:/", 512 * 1024);

        // 容量不足で拡張失敗
        NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, ExtendSaveData(user, property.userAccountSaveDataSize + 1 * 1024 * 1024, property.userAccountSaveDataJournalSize));

        // エラー後もサイズは変わらずマウント可能
        {
            nn::ncm::ApplicationId id = {nnt::fs::util::UserSaveDataApplicationId};
            auto info = CheckSaveDataExisting(id, SaveDataType::Account, true);
            CheckSaveData(info.value(), property, user);
        }
    }
    Unmount("@User");
}



// いずれかの作成に失敗した時は一時ストレージの作成をキャンセルして失敗した体で返す
// 空き容量埋めるため heavy
TEST(EnsureApplicationSaveData, CancelTemporaryStorageForUsableSpaceNotEnoughHeavy)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId id = { nnt::fs::util::UserSaveDataApplicationId };

    auto GetEmptyProperty = []() {
        nn::ns::ApplicationControlProperty property = {};
        property.saveDataOwnerId = nnt::fs::util::UserSaveDataApplicationId + 1;
        return property;
    };

    nn::account::Uid user;
    user._data[0] = 1;
    user._data[1] = 0;


    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));

    {
        nnt::fs::util::ScopedPadder padder("@User:/", 64 * 1024 * 1024); // 残り 64MB 以下に埋める
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::ListDirectoryRecursive("@User:/"));

        // account + temp
        {
            int64_t size;
            auto property = GetEmptyProperty();
            property.userAccountSaveDataSize        = 64 * 1024 * 1024;
            property.userAccountSaveDataJournalSize = 60 * 1024 * 1024; // ユーザアカウントセーブは作成失敗するサイズ

            property.temporaryStorageSize = 32 * 1024 * 1024; // 一時ストレージは成功するサイズ

            NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, EnsureApplicationSaveData(&size, id, property, user));

            EXPECT_GT(size, property.userAccountSaveDataSize + property.userAccountSaveDataJournalSize + property.temporaryStorageSize);
            EXPECT_LT(size, property.userAccountSaveDataSize + property.userAccountSaveDataJournalSize + property.temporaryStorageSize + 1536 * 1024);

            // いずれも作成失敗していること
            CheckSaveDataExisting(id, SaveDataType::Account, false);
            CheckSaveDataExisting(id, SaveDataType::Device, false);
            CheckSaveDataExisting(id, SaveDataType::Bcat, false);
            CheckSaveDataExisting(id, SaveDataType::Temporary, false);
            CheckSaveDataExisting(id, SaveDataType::Cache, false);
        }

        // 巻き添えが無ければ作成成功すること
        {
            int64_t size;
            auto property = GetEmptyProperty();
            property.temporaryStorageSize = 32 * 1024 * 1024;

            NNT_EXPECT_RESULT_SUCCESS(EnsureApplicationSaveData(&size, id, property, user));

            CheckSaveDataExisting(id, SaveDataType::Account, false);
            CheckSaveDataExisting(id, SaveDataType::Device, false);
            CheckSaveDataExisting(id, SaveDataType::Bcat, false);
            CheckSaveDataExisting(id, SaveDataType::Temporary, true);
            CheckSaveDataExisting(id, SaveDataType::Cache, false);
        }

    }

    Unmount("@User");
    nnt::fs::util::DeleteAllTestSaveData();
}




#endif

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

#if !defined(NN_BUILD_CONFIG_OS_WIN) // win 未対応
    nn::oe::Initialize();
    nn::account::InitializeForSystemService();
#endif

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
