﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>

#include <nn/fs/fs_Application.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Code.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_Logo.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/fs/fs_RegisteredUpdatePartition.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>

#include "testFs_FsLib_AccessLog.h"

namespace
{
    const nn::ncm::ProgramId TestProgramId = { 1 };

#if !defined(NN_BUILD_CONFIG_HARDWARE_NX)
    void DeleteDeviceSaveData(const nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        nn::util::optional<nn::fs::SaveDataInfo> saveDataInfo;
        nnt::fs::util::FindSaveData(
            &saveDataInfo,
            nn::fs::SaveDataSpaceId::User,
            [&](const nn::fs::SaveDataInfo info)
            {
                return info.applicationId == applicationId;
            });
        if (saveDataInfo != nn::util::nullopt)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(saveDataInfo->saveDataId));
        }
    }
#endif
}

TEST(ApplicationPackage, DISABLED_Success)
{}

TEST(ApplicationPackage, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE402, handle: 0x0000000000000000, priority: Normal, function: "MountApplicationPackage", name: "test", path: "hoge" }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountApplicationPackage", name: "test", path: "hoge" }
*/
{
    ASSERT_TRUE(nn::fs::MountApplicationPackage(nnt::fs::MountName, "hoge").IsFailure());
}

TEST(SystemMountBis, Success)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountBis", name: "test", bispartitionid: User, path: "(null)" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountBis", name: "@User", bispartitionid: User, path: "(null)" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "@User" }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::detail::MountBis", name: "test", bispartitionid: User, path: "(null)" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::detail::MountBis", name: "@User", bispartitionid: User, path: "(null)" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "@User" }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nnt::fs::MountName, nn::fs::BisPartitionId::User));
    nn::fs::Unmount(nnt::fs::MountName);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));
    nn::fs::Unmount(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User));
}

TEST(SystemMountBis, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "MountBis", name: "test", bispartitionid: Invalid, path: "(null)" }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::detail::MountBis", name: "test", bispartitionid: Invalid, path: "(null)" }
*/
{
    ASSERT_TRUE(nn::fs::MountBis(nnt::fs::MountName, nn::fs::BisPartitionId::Invalid).IsFailure());
}

TEST(SystemMountContent, DISABLED_Success)
{}

TEST(SystemMountContent, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00177202, handle: 0x0000000000000000, priority: Normal, function: "MountContent", name: "test", programid: 0x1, content_type: Meta }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00177202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountContent", name: "test", programid: 0x1, content_type: Meta }
*/
{
    ASSERT_TRUE(nn::fs::MountContent(nnt::fs::MountName, TestProgramId, nn::fs::ContentType_Meta).IsFailure());
}

TEST(SystemMountContentStorage, Success)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountContentStorage", name: "test", contentstorageid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountContentStorage", name: "@SdCardContent", contentstorageid: SdCard }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "@SdCardContent" }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountContentStorage", name: "test", contentstorageid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountContentStorage", name: "@SdCardContent", contentstorageid: SdCard }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "@SdCardContent" }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(nnt::fs::MountName, nn::fs::ContentStorageId::User));
    nn::fs::Unmount(nnt::fs::MountName);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::SdCard));
    nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::SdCard));
}

TEST(SystemMountContentStorage, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "MountContentStorage", name: "test", contentstorageid: 9999 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountContentStorage", name: "test", contentstorageid: 9999 }
*/
{
    ASSERT_TRUE(nn::fs::MountContentStorage(nnt::fs::MountName, static_cast<nn::fs::ContentStorageId>(9999)).IsFailure());
}

TEST(SystemMountImageDirectory, SuccessFailure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountImageDirectory", name: "test", imagedirectoryid: Nand }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "MountImageDirectory", name: "test", imagedirectoryid: 9999 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountImageDirectory", name: "test", imagedirectoryid: Nand }
FS_ACCESS: { start: 0, end: 0, result: 0x00007802, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountImageDirectory", name: "test", imagedirectoryid: 9999 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory(nnt::fs::MountName, nn::fs::ImageDirectoryId::Nand));
    ASSERT_TRUE(nn::fs::MountImageDirectory(nnt::fs::MountName, static_cast<nn::fs::ImageDirectoryId>(9999)).IsFailure());
    nn::fs::Unmount(nnt::fs::MountName);
}

TEST(SystemMountLogo, DISABLED_Success)
{}

TEST(SystemMountLogo, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE402, handle: 0x0000000000000000, priority: Normal, function: "MountLogo", name: "test", path: "InvalidPath", programid: 0x1 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountLogo", name: "test", path: "InvalidPath", programid: 0x1 }
*/
{
    ASSERT_TRUE(nn::fs::MountLogo(nnt::fs::MountName, "InvalidPath", TestProgramId).IsFailure());
}

TEST(SystemMountSystemData, DISABLED_Success)
{}

TEST(SystemMountSystemData, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "QueryMountSystemDataCacheSize", systemdataid: 0x1, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountSystemData", name: "test", systemdataid: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountSystemData", name: "test", systemdataid: 0x1 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00177202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::QueryMountSystemDataCacheSize", systemdataid: 0x1, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemData", name: "test", systemdataid: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemData", name: "test", systemdataid: 0x1 }
*/
{
    const nn::ncm::SystemDataId systemDataId = { 1 };
    size_t cacheBufferSize = 1024;
    ASSERT_TRUE(nn::fs::QueryMountSystemDataCacheSize(&cacheBufferSize, systemDataId).IsFailure());
    auto cacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);
    ASSERT_NE(nullptr, cacheBuffer);

    ASSERT_TRUE(nn::fs::MountSystemData(nnt::fs::MountName, systemDataId, cacheBuffer.get(), cacheBufferSize).IsFailure());
    ASSERT_TRUE(nn::fs::MountSystemData(nnt::fs::MountName, systemDataId).IsFailure());
}

TEST(MountDeviceSaveData, DISABLED_Success)
{}

TEST(MountDeviceSaveData, Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountDeviceSaveData", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountDeviceSaveData", name: "test", applicationid: 0x9876543210987654 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountDeviceSaveData", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountDeviceSaveData", name: "test", applicationid: 0x9876543210987654 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountDeviceSaveData", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountDeviceSaveData", name: "test", applicationid: 0x9876543210987654 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountDeviceSaveData", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountDeviceSaveData", name: "test", applicationid: 0x9876543210987654 }
*/
{
    ASSERT_TRUE(nn::fs::MountDeviceSaveData(nnt::fs::MountName).IsFailure());
    ASSERT_TRUE(nn::fs::MountDeviceSaveData(nnt::fs::MountName, nnt::fs::InvalidApplicationId).IsFailure());
}

TEST(MountRegisteredUpdatePartition, SuccessFailure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountRegisteredUpdatePartition", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00007802, handle: 0x0000000000000000, priority: Normal, function: "MountRegisteredUpdatePartition", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountRegisteredUpdatePartition", name: "test" }
*/
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RegisterUpdatePartition());
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRegisteredUpdatePartition(nnt::fs::MountName));
    ASSERT_TRUE(nn::fs::MountRegisteredUpdatePartition(nnt::fs::MountName).IsFailure());
    nn::fs::Unmount(nnt::fs::MountName);
#else
    ASSERT_TRUE(nn::fs::MountRegisteredUpdatePartition(nnt::fs::MountName).IsFailure());
#endif
}

TEST(CreateDeviceSaveData, SuccessFailure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00345802, handle: 0x0000000000000000, priority: Normal, function: "CreateDeviceSaveData", applicationid: 0x9876543210987654, save_data_owner_id: 0x0, save_data_size: 4096, save_data_journal_size: 4096, save_data_flags: 0x00000004 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDeviceSaveData", applicationid: 0x100000000003006, save_data_owner_id: 0x0, save_data_size: 4096, save_data_journal_size: 4096, save_data_flags: 0x00000004 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
*/
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    ASSERT_TRUE(nn::fs::CreateDeviceSaveData(nnt::fs::InvalidApplicationId, 0, 0x1000, 0x1000, 0x4).IsFailure());
#else
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDeviceSaveData(nnt::fs::ApplicationId, 0, 0x1000, 0x1000, 0x4));
    DeleteDeviceSaveData(nnt::fs::ApplicationId);
#endif
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
TEST(SystemMountCode, DISABLED_Success)
{}

TEST(SystemMountCode, Failure)
/*
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountCode", name: "test", path: "InvalidPath", programid: 0x1 }
*/
{
    ASSERT_TRUE(nn::fs::MountCode(nnt::fs::MountName, "InvalidPath", TestProgramId).IsFailure());
}
#endif
