﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Unit_Api.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_ResultHandler.h>
#include <nn/fssystem/fs_HtcFileSystem.h>

#include <nn/htclow.h>
#include <nn/htclow/detail/htclow_DebugApi.h>
#include <nn/htclow/detail/htclow_InternalApi.h>
#include <nn/htclow/server/htclow_Server.h>
#include <nn/htcfs/detail/htcfs_InternalApi.h>
#include <nn/htcfs/server/htcfs_Server.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
void* operator new(size_t size)
{
    nnt::fs::util::SetGlobalNewDeleteFlag();
    return malloc(size);
}

void operator delete(void* p)
{
    free(p);
}
#endif

namespace {
    const int HeapSize = 512 * 1024;
    char g_HeapStack[HeapSize];
    char g_RootDirPath[256];
    nnt::fs::util::TemporaryHostDirectory g_HostDirectory;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    const size_t ThreadStackSize = 16 * 1024;

    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_HtclowUserServerThreadStack[ThreadStackSize];
    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_HtclowDebugServerThreadStack[ThreadStackSize];
    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_HtcfsServerThreadStack[ThreadStackSize];

    nn::os::ThreadType g_HtclowUserServerThread;
    nn::os::ThreadType g_HtclowDebugServerThread;
    nn::os::ThreadType g_HtcfsServerThread;
#endif
}

using namespace nn::fssystem;

namespace nnt { namespace fs { namespace api {
    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);
        std::unique_ptr<HtcFileSystem> htcFs(new HtcFileSystem());
        NNT_ASSERT_RESULT_SUCCESS(htcFs->Initialize());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Unit));
        info->fileSystem = std::unique_ptr<ITestFileSystem>(new FsApiUnitTestFileSystem(std::move(htcFs)));
        info->type = FileSystemType_HostFileSystem;
        info->rootDirPath = g_RootDirPath;

        auto& attribute = info->attribute;

        static const int HtcFsPathLengthMax     = 260 - 1; // HtcFileSystem::WindowsPathLengthMax からヌル文字分を引いた値
        static const int DirectoryPathLengthMax = 248 - 1; // Windows の CreateDirectory に渡せるパスの最大長（'\0' を含まない）
        attribute.directoryNameLengthMax = DirectoryPathLengthMax;
        attribute.fileNameLengthMax      = HtcFsPathLengthMax;
        attribute.directoryPathLengthMax = DirectoryPathLengthMax;
        attribute.filePathLengthMax      = HtcFsPathLengthMax;

        attribute.fileSizeMax = 0x0000000220000000LL;
        attribute.storageSize = 0x0000000240000000LL;

        attribute.fileOpenMax      = 512;
        attribute.directoryOpenMax = 64;

        attribute.isSupportedMultiBytePath = true;
        attribute.isSupportedGetFreeSpaceSize = false;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedGetFileTimeStamp = true;
        attribute.isSupportedQueryRange = true;

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }

    void InitializeHtclowServer() NN_NOEXCEPT
    {
        // Start libnn_htclowServer
        nn::htclow::server::Initialize();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::htclow::server::RegisterUserHipcServer();
        nn::htclow::server::RegisterDebugHipcServer();

        nn::os::CreateThread(&g_HtclowUserServerThread,
            [](void*)
            {
                nn::htclow::server::StartUserHipcServer();
                nn::htclow::server::LoopUserHipcServer();
            }
            , nullptr, g_HtclowUserServerThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
        nn::os::CreateThread(&g_HtclowDebugServerThread,
            [](void*)
            {
                nn::htclow::server::StartDebugHipcServer();
                nn::htclow::server::LoopDebugHipcServer();
            }
            , nullptr, g_HtclowDebugServerThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);

        nn::os::StartThread(&g_HtclowUserServerThread);
        nn::os::StartThread(&g_HtclowDebugServerThread);
#else
        nn::htclow::detail::InitializeForUserApiWith(nn::htclow::server::GetUserServiceObject());
        nn::htclow::detail::InitializeForDebugApiWith(nn::htclow::server::GetDebugServiceObject());
#endif

        // Set default htclow driver
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::htclow::detail::OpenDriver(nn::htclow::detail::DriverType::Socket));
    }

    void InitializeHtcfsServer() NN_NOEXCEPT
    {
        // Start libnn_htcfsServer
        nn::htcfs::server::Initialize();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::htcfs::server::RegisterHipcServer();

        nn::os::CreateThread(&g_HtcfsServerThread,
            [](void*)
            {
                nn::htcfs::server::StartHipcServer();
                nn::htcfs::server::LoopHipcServer();
            }
            , nullptr, g_HtcfsServerThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);

        nn::os::StartThread(&g_HtcfsServerThread);
#else
        nn::htcfs::detail::InitializeWith(nn::htcfs::server::GetServiceObject());
#endif
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nnt::fs::api::InitializeHtclowServer();
    nnt::fs::api::InitializeHtcfsServer();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);

    g_HostDirectory.Create();
    strncpy(g_RootDirPath, g_HostDirectory.GetPath().c_str(), sizeof(g_RootDirPath));

    auto ret = RUN_ALL_TESTS();

    g_HostDirectory.Delete();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
