﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/nn_Allocator.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/fssrv/fssrv_MemoryResourceFromStandardAllocator.h> // 流用
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Unit_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fssystem/fs_HostFileSystem.h>
#include <nn/fssystem/fs_DirectorySaveDataFileSystem.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>
#endif

namespace {
    const int BufferPoolSize = 512 * 1024;
    char g_BufferPool[BufferPoolSize];
    nn::mem::StandardAllocator g_BufferAllocator(g_BufferPool, BufferPoolSize);
    nn::fssrv::MemoryResourceFromStandardAllocator g_Allocator(&g_BufferAllocator);

    nnt::fs::util::TemporaryHostDirectory g_HostDirectory;
    char g_RootDirPath[256];
}

using namespace nn::fssystem;

namespace nnt { namespace fs { namespace api {
    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);
        std::unique_ptr<HostFileSystem> hostFs(new HostFileSystem());
        NNT_ASSERT_RESULT_SUCCESS(hostFs->Initialize(g_RootDirPath));

        std::unique_ptr<DirectorySaveDataFileSystem> saveFs(new DirectorySaveDataFileSystem(std::move(hostFs), &g_Allocator));
        NNT_ASSERT_RESULT_SUCCESS(saveFs->Initialize());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Unit));
        info->fileSystem = std::unique_ptr<ITestFileSystem>(new FsApiUnitTestFileSystem(std::move(saveFs)));
        info->type = FileSystemType_HostFileSystem;
        info->rootDirPath = "";

        auto& attribute = info->attribute;

        static const int InnerDirectoryNameLength = 2;       // DirectorySaveDataFileSystem 内部で使用されるディレクトリ名（ModifiedDirectoryName など）の長さ
        static const int HostFsPathLengthMax      = 260 - 1; // HostFileSystem::WindowsPathLengthMax からヌル文字分を引いた値
        static const int DirectoryPathLengthMax   = 248 - 1; // Windows の CreateDirectory に渡せるパスの最大長（'\0' を含まない）
        const int rootDirectoryPathLength
            = static_cast<int>(strnlen(g_RootDirPath, sizeof(g_RootDirPath)));
        attribute.directoryPathLengthMax
            = DirectoryPathLengthMax - rootDirectoryPathLength - InnerDirectoryNameLength;
        attribute.filePathLengthMax
            = HostFsPathLengthMax - rootDirectoryPathLength - InnerDirectoryNameLength;
        attribute.directoryNameLengthMax = attribute.directoryPathLengthMax;
        attribute.fileNameLengthMax = attribute.filePathLengthMax;

        attribute.fileSizeMax = 0x0000000220000000LL;
        attribute.storageSize = 0x0000000240000000LL;

        attribute.fileOpenMax      = 1024;
        attribute.directoryOpenMax = 64;

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = false;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;
        attribute.isSaveFileSystem = true;

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();

//    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);

    g_HostDirectory.Create();
    auto rootDirPath = g_HostDirectory.GetPath();
    strncpy(g_RootDirPath, rootDirPath.c_str(), sizeof(g_RootDirPath));

    auto ret = RUN_ALL_TESTS();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectory(nnt::fs::util::String(rootDirPath + "/0").c_str()));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectory(nnt::fs::util::String(rootDirPath + "/1").c_str()));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectory(rootDirPath.c_str()));
    nn::fs::UnmountHostRoot();
    g_HostDirectory.Delete();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
