﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NN_ENABLE_HTC

#include <chrono>
#include <mutex>

#include <nn/crypto.h>
#include <nn/htc.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/fsUtil/testFs_util.h>

#include <nn/os/os_Random.h>
#include <nn/os/os_SdkMutex.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_IEventNotifier.h>

#include <nn/fs/fsa/fs_IFile.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_CacheStorage.h>
#include <nn/fs/fs_CacheStoragePrivate.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SystemBcatSaveData.h>
#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataWithApplicationId.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_TemporaryStorage.h>
#include <nn/fs/fs_UserAccountSystemSaveData.h>
#include <nn/fs/fs_Utility.h>

#include <nn/fssrv/fssrv_SaveDataIndexerManager.h>

#include <nn/fs/fs_DebugPrivate.h>
#include "detail/fssrv_SaveDataTransferVersion2.h"

using namespace nn;
using namespace nn::fs;

namespace nn { namespace fs {
    Result GetCacheStorageTargetMedia(CacheStorageTargetMedia* outValue, ncm::ApplicationId applicationId);
}}

namespace nn { namespace fssrv { namespace detail {
    void InvalidateAllIndexers() NN_NOEXCEPT;
}}}

namespace nnt { namespace fs { namespace util {

void DumpBuffer(const void* buffer, const size_t size) NN_NOEXCEPT
{
    for(size_t i = 0; i < size; ++i)
    {
        if(i % 4   == 0)
        {
            NN_LOG(" ");
        }
        if(i % 32  == 0)
        {
            NN_LOG("\n");
        }

        uint8_t value8 = static_cast<const uint8_t*>(buffer)[i];
        NN_LOG("%02x", value8);
    }

    NN_LOG("\n\n");
}

void DumpBufferDiff(const void* expectedBuffer, const void* actualBuffer, const size_t size) NN_NOEXCEPT
{
    const Bit8* expectedBufferChar = static_cast<const Bit8*>(expectedBuffer);
    const Bit8* actualBufferChar = static_cast<const Bit8*>(actualBuffer);

    const size_t RowCount = 4;
    int outputCount = 64;
    NN_LOG("    offset  expected    actual\n");
    for(size_t i = 0; i < size && outputCount > 0; i += RowCount)
    {
        auto compareSize = std::min<size_t>(RowCount, size - i);
        if( memcmp(expectedBufferChar + i, actualBufferChar + i, compareSize) != 0 )
        {
            NN_LOG("0x%08x  ", i);
            for(size_t j = 0; j < RowCount; ++j)
            {
                if( i + j < size )
                {
                    NN_LOG("%02x", expectedBufferChar[i + j]);
                }
                else
                {
                    NN_LOG("  ");
                }
            }
            NN_LOG("    ");
            for(size_t j = 0; j < RowCount; ++j)
            {
                if( i + j < size )
                {
                    NN_LOG("%02x", actualBufferChar[i + j]);
                }
                else
                {
                    NN_LOG("  ");
                }
            }
            NN_LOG("\n");

            --outputCount;
        }
    }

}

void FillBufferWithRandomValue(void* pBuffer, const size_t size) NN_NOEXCEPT
{
    std::mt19937_64 mt(GetRandomSeed());

    uint64_t random = 0;
    for(size_t i = 0; i < size; ++i)
    {
        if ((i & 0x7) == 0)
        {
            random = mt();
        }
        static_cast<uint8_t*>(pBuffer)[i] = static_cast<uint8_t>(random & 0xFF);
        random >>= 8;
    }
}

namespace {

void FillBufferWith32BitCountBody(void* pBuffer, const size_t size, const int64_t offsetCount) NN_NOEXCEPT
{
    uint32_t count = static_cast<uint32_t>(offsetCount / 4);
    uint32_t* pBuffer32 = static_cast<uint32_t*>(pBuffer);
    for(size_t i = 0; i < size / 4; ++i)
    {
        pBuffer32[i] = count;
        ++count;
    }

    auto tailSize = size % 4;
    if( tailSize > 0 )
    {
        memcpy(&pBuffer32[size / 4], &count, tailSize);
    }
}

}
void FillBufferWith32BitCount(void* pBuffer, const size_t size, const int64_t offsetCount) NN_NOEXCEPT
{
    // head
    size_t headSize = (4 - offsetCount % 4) % 4;
    if( headSize > 0 )
    {
        uint32_t count = static_cast<uint32_t>(offsetCount / 4);
        memcpy(pBuffer, reinterpret_cast<char*>(&count) + 4 - headSize, std::min(headSize, size));
    }

    if (size > headSize)
    {
        FillBufferWith32BitCountBody(reinterpret_cast<char*>(pBuffer) + headSize, size - headSize, offsetCount + headSize);
    }
    else
    {
        NN_UNUSED(pBuffer);
    }
}

bool IsFilledWith32BitCount(const void* pBuffer, const size_t size, const int64_t offsetCount) NN_NOEXCEPT
{
    // 4 Byte アライン無効化
    uint32_t count = static_cast<uint32_t>(offsetCount / 4);
    const char* pBuffer8 = static_cast<const char*>(pBuffer);
    // head
    size_t headSize = (4 - offsetCount % 4) % 4;
    if (headSize > 0)
    {
        if (memcmp(pBuffer8, reinterpret_cast<char*>(&count) + 4 - headSize, std::min(headSize, size)) != 0)
        {
            NN_UNUSED(pBuffer);
            NN_LOG("[error] IsFilledWith32BitCount Head mismatch\n");
            NN_LOG("headSize = %d\n", headSize);
            for (size_t i = 0; i < headSize; i++)
            {
                NN_LOG("  %d : Actual = 0x%02x Expect = 0x%02x\n", i, pBuffer8[i], *(reinterpret_cast<char*>(&count) + 4 - headSize + i));
            }
            return false;
        }
        else if (size <= headSize)
        {
            NN_UNUSED(pBuffer);
            return true;
        }
        ++count;
    }

    const uint32_t* pBuffer32 = reinterpret_cast<const uint32_t*>(pBuffer8 + headSize);

    for(size_t i = 0; i < (size - headSize) / 4; ++i)
    {
        if( pBuffer32[i] != count )
        {
            NN_LOG("[error] IsFilledWith32BitCount Body mismatch\n");
            NN_LOG("Body Size = %d\n", size - headSize);
            NN_LOG("  %d : Actual = 0x%02x Expect = 0x%02x\n", i, pBuffer32[i], count);
            return false;
        }
        ++count;
    }

    size_t tailSize = (size - headSize) % 4;
    if (tailSize > 0)
    {
        if (memcmp((pBuffer8 + size - tailSize), &count, tailSize) != 0)
        {
            NN_LOG("[error] IsFilledWith32BitCount Tail mismatch\n");
            NN_LOG("tailSize = %d\n", tailSize);
            for (size_t i = 0; i < tailSize; i++)
            {
                NN_LOG("  %d : Actual = 0x%02x Expect = 0x%02x\n", i, *(pBuffer8 + size - tailSize + i), *(reinterpret_cast<char*>(&count) + i));
            }
            return false;
        }
    }
    return true;
}

bool IsFilledWith8BitCount(const void* pBuffer, const size_t size, const int64_t offsetCount) NN_NOEXCEPT
{
    const auto pBuffer8 = static_cast<const uint8_t*>(pBuffer);
    uint8_t count = static_cast<uint8_t>(offsetCount);

    for (size_t i = 0; i < size; ++i)
    {
        if (pBuffer8[i] != count)
        {
            return false;
        }
        ++count;
    }
    return true;
}

bool IsFilledWithValue(const void* pBuffer, const size_t size, unsigned char value) NN_NOEXCEPT
{
    auto pCharBuffer = static_cast<const char* const>(pBuffer);
    const size_t WorkBufferSize = 32 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);
    NN_ASSERT(workBuffer.get() != nullptr);
    memset(workBuffer.get(), value, WorkBufferSize);

    size_t restSize = size;
    size_t offset = 0;
    while(restSize > 0)
    {
        size_t compareSize = std::min(restSize, WorkBufferSize);

        if( memcmp(workBuffer.get(), pCharBuffer + offset, compareSize) != 0 )
        {
            return false;
        }

        restSize -= compareSize;
        offset += compareSize;
    }

    return true;
}

void FillBufferWith8BitCount(void* pBuffer, const size_t size, const int64_t offsetCount) NN_NOEXCEPT
{
    uint8_t count = static_cast<uint8_t>(offsetCount);
    uint8_t* pBuffer8 = static_cast<uint8_t*>(pBuffer);
    for(size_t i = 0; i < size; ++i)
    {
        pBuffer8[i] = count;
        ++count;
    }
}

void FillBufferWithZero(void* pBuffer, const size_t size, const int64_t offsetCount) NN_NOEXCEPT
{
    NN_UNUSED(offsetCount);
    std::memset(pBuffer, 0, size);
}

void FillBufferWithXorShift(char* path, void* pBuffer, const size_t size) NN_NOEXCEPT
{
    nnt::fs::util::String str(path);
    uint64_t seed = 0;
    str.erase(0, str.rfind('/') + 1);
    for(size_t i = 0; i < 8 && i < str.length(); ++i)
    {
        seed |= static_cast<uint64_t>(str[i]) << (8 * i);
    }

    uint32_t x = 123456789 ^ static_cast<uint32_t>((seed >> 32) & 0xffffffff);
    uint32_t y = 362436069 ^ static_cast<uint32_t>(seed & 0xffffffff);
    uint32_t z = 521288629;
    uint32_t w = 88675123;

    uint32_t* pBuffer32 = static_cast<uint32_t*>(pBuffer);
    for(size_t i = 0; i < size / sizeof(uint32_t); ++i)
    {
        uint32_t t = x ^ (x << 11);
        x = y;
        y = z;
        z = w;
        pBuffer32[i] = (w = (w ^ (w >> 19)) ^ (t ^ (t >> 8)));
    }
    if(size % sizeof(uint32_t))
    {
        uint32_t fraction = static_cast<uint32_t>(size) % sizeof(uint32_t);
        uint32_t offset = static_cast<uint32_t>(size) - fraction;
        uint32_t t = x ^ (x << 11);
        x = y;
        y = z;
        z = w;
        uint32_t next = (w = (w ^ (w >> 19)) ^ (t ^ (t >> 8)));
        uint8_t* pBuffer8 = static_cast<uint8_t*>(pBuffer);
        for(uint32_t i = 0; i < fraction; ++i)
        {
            pBuffer8[offset + i] = static_cast<uint8_t>((next >> (8 * i)) & 0xff);
        }
    }
}

int64_t GetRandomLength() NN_NOEXCEPT
{
    std::mt19937 mt(GetRandomSeed());
    return std::uniform_int_distribution<>(1, MAX_LENGTH)(mt);
}

String GenerateRandomString(const int64_t length) NN_NOEXCEPT
{
    std::mt19937 mt(GetRandomSeed());
    std::uniform_int_distribution<int> asciiDis(33, 126);
    String randomString;
    for (int64_t i = 0; i < length; i++)
    {
        randomString.append(1, static_cast<char>(asciiDis(mt)));
    }

    return randomString;
}

String GenerateRandomString() NN_NOEXCEPT
{
    return GenerateRandomString(GetRandomLength());
}

String GenerateRandomLengthEntryName(int64_t maxLength) NN_NOEXCEPT
{
    NN_ASSERT_LESS_EQUAL(1, maxLength);
    std::mt19937 mt(GetRandomSeed());
    auto length = std::uniform_int_distribution<int64_t>(1, maxLength)(mt);

    return GenerateRandomEntryName(length);
}

String GenerateRandomEntryName() NN_NOEXCEPT
{
    return GenerateRandomLengthEntryName(128);
}

String GenerateRandomEntryName(const int64_t length) NN_NOEXCEPT
{
    std::mt19937 mt(GetRandomSeed());
    std::uniform_int_distribution<int> asciiDis(33, 126);

    String randomString;
    for (int64_t i = 0; i < length; i++)
    {
        char c;
        while(c = static_cast<char>(asciiDis(mt)),
              c == '/' || c == ':' || c == '\\' || c == '"' || c == '*' || c == ',' || c == ';' || c == '<' || c == '>' || c == '?' || c == '|' || c == '~' ||
              (i == length - 1 && c == '.') ||
              (i == 0 && (c == '[' || c == ']' || c == '@' || c == '{' || c == '}' || c =='`' || c == '\'' || c == '!'|| c == '#'|| c == '%' || c == '&'))
            )
        {
        }

        randomString.append(1, c);
    }

    return randomString;
}

String GetParentPath(const char* path) NN_NOEXCEPT
{
    String parentPath(path);
    const char* parentPathCur = parentPath.c_str() + parentPath.length() - 1;
    for (;*parentPathCur != '/' && parentPathCur > parentPath.c_str(); parentPathCur--)
    {
    }

    NN_ASSERT(parentPathCur != parentPath.c_str());

    return parentPath.substr(0, parentPathCur - parentPath.c_str() + 1);
}


void GetDateTime(tm* pdate, time_t* pmsec) NN_NOEXCEPT
{
    memset(pdate, 0x00, sizeof(tm));

    // TODO: os から取得
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    using namespace std::chrono;

    auto duration = high_resolution_clock::now().time_since_epoch();
    time_t t = duration_cast<seconds>(duration).count();

    localtime_s(pdate, &t);
    *pmsec = duration_cast<milliseconds>(duration).count() % 1000;

    pdate->tm_year  += 1900;
    pdate->tm_mon   += 1;
#else
    pdate->tm_year  = 2015;
    pdate->tm_mon   = 1;
    pdate->tm_mday  = 1;
#endif
}

String GetNowString() NN_NOEXCEPT
{
    char stamp[5];
    tm now;
    time_t msec;

    GetDateTime(&now, &msec);

    String strNow = "";

    // 2桁-年の文字列変換
    nn::util::SNPrintf(stamp, sizeof(stamp), "%02d", (now.tm_year % 100));
    strNow.append(stamp);

    // 2桁-月の文字列変換
    nn::util::SNPrintf(stamp, sizeof(stamp), "%02d", now.tm_mon);
    strNow.append(stamp);

    // 2桁-日の文字列変換
    nn::util::SNPrintf(stamp, sizeof(stamp), "%02d", now.tm_mday);
    strNow.append(stamp);

    return strNow;
}

const char* GetHostTemporaryPath() NN_NOEXCEPT
{
    static String s_HostPath;
    if( s_HostPath.empty() )
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        static const char* EnvironmentVariables[] = { "TEMP", "TMP" };
        for( auto environmentVariable : EnvironmentVariables )
        {
            const char* env = std::getenv(environmentVariable);
            if(env)
            {
                s_HostPath = env;
                break;
            }
        }
#else
        nn::htc::Initialize();
        static const char* EnvironmentVariables[] = { "TEMP", "TMP" };
        for( auto environmentVariable : EnvironmentVariables )
        {
            size_t bufferLength = 0;
            if( nn::htc::GetEnvironmentVariableLength(
                &bufferLength,
                environmentVariable).IsSuccess() )
            {
                auto bufferString = AllocateBuffer(bufferLength);
                size_t variableLength = 0;
                if( nn::htc::GetEnvironmentVariable(
                    &variableLength,
                    bufferString.get(),
                    bufferLength,
                    environmentVariable).IsSuccess() )
                {
                    s_HostPath = bufferString.get();
                    break;
                }
            }
        }
        nn::htc::Finalize();
#endif

        if( s_HostPath.empty() )
        {
            s_HostPath = "C:\\Windows\\Temp";
        }
    }

    return s_HostPath.c_str();
}

String CreateUniquePath(const char* pBase) NN_NOEXCEPT
{
    std::mt19937 mt(GetRandomSeed());
    std::uniform_int_distribution<uint16_t> random(0, 999);

    char stamp[5];
    nn::util::SNPrintf(stamp, sizeof(stamp), "%03d", random(mt));

    return String(pBase).append(GetNowString()).append(stamp);
}

namespace {

String CreateUniqueTemporaryDirectory() NN_NOEXCEPT
{
    const int RetryCountMax = 20;

    for( auto retryCount = 0; retryCount < RetryCountMax; ++retryCount )
    {
        auto rootDirPath = CreateUniquePath("SIGLO_FS_TEST");
        auto rootDirPath2 = String(GetHostTemporaryPath()).append("\\").append(rootDirPath);
        auto result = CreateDirectory(rootDirPath2.c_str());

        if( result.IsSuccess() )
        {
            return rootDirPath2;
        }
        else if( ResultPathAlreadyExists::Includes(result) )
        {
            continue;
        }
        else
        {
            NN_ABORT("Failed to create temporary directory by unknown error %x.", result.GetInnerValueForDebug());
        }
    }

    NN_ABORT("Failed to create temporary directory.");
}

}

TemporaryHostDirectory::TemporaryHostDirectory() NN_NOEXCEPT
: m_IsCreated(false)
{
}

TemporaryHostDirectory::~TemporaryHostDirectory() NN_NOEXCEPT
{
    Delete();
}

// 一時ディレクトリを作成します。
void TemporaryHostDirectory::Create() NN_NOEXCEPT
{
    if( ! m_IsCreated )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
        m_Path = CreateUniqueTemporaryDirectory();
        NN_ASSERT(m_Path.length() > 0);
        nn::fs::UnmountHostRoot();
        m_IsCreated = true;
    }
}

// 一時ディレクトリを削除します。
void TemporaryHostDirectory::Delete() NN_NOEXCEPT
{
    if( m_IsCreated )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(m_Path.c_str()));
        nn::fs::UnmountHostRoot();
        m_Path = "";
        m_IsCreated = false;
    }
}

// ランダムな文字列を含んだパスを返す
String TemporaryHostDirectory::GetPath() const NN_NOEXCEPT
{
    NN_ASSERT(m_IsCreated);
    return m_Path;
}

uint64_t RoundUp(const uint64_t value, const uint64_t alignment) NN_NOEXCEPT
{
    NN_ASSERT((alignment & (alignment - 1)) == 0);
    return ((value + alignment - 1) & ~(alignment - 1));
}

uint64_t RoundDown(const uint64_t value, const uint64_t alignment) NN_NOEXCEPT
{
    NN_ASSERT((alignment & (alignment - 1)) == 0);
    return (value & ~(alignment - 1));
}

Result CalculateFileHash(Hash *outValue, const char* path, void* buffer, const size_t bufferSize, int64_t offset, int64_t size) NN_NOEXCEPT
{
    FileHandle handle;

    NN_RESULT_DO(OpenFile(&handle, path, OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handle);
    };

    nn::crypto::Sha256Generator sha;
    sha.Initialize();

    int64_t fileSize = 0;
    GetFileSize(&fileSize, handle);
    if (fileSize == 0)
    {
        sha.GetHash(outValue->value, sizeof(outValue->value));
        NN_RESULT_SUCCESS;
    }

    if (size < 0)
    {
        size = fileSize;
    }

    int64_t curerntOffset = offset;
    int64_t restSize = size;
    while(restSize > 0)
    {
        size_t accessSize = static_cast<size_t>(std::min(restSize, static_cast<int64_t>(bufferSize)));

        size_t readSize;
        NN_RESULT_DO(ReadFile(&readSize, handle, curerntOffset, buffer, accessSize));

        sha.Update(buffer, readSize);

        if( readSize < accessSize)
        {
            break;
        }

        curerntOffset += readSize;
        restSize      -= readSize;
    }

    sha.GetHash(outValue->value, sizeof(outValue->value));
    NN_RESULT_SUCCESS;
}

Result CalculateFileHash(Hash *outValue, const char* path, void* buffer, const size_t bufferSize) NN_NOEXCEPT
{
    return CalculateFileHash(outValue, path, buffer, bufferSize, 0, -1);
}


char g_BufferForCalculateFileHash[32 * 1024]; // TORIAEZU
Result CalculateFileHash(Hash *outValue, const char* path) NN_NOEXCEPT
{
    return CalculateFileHash(outValue, path, g_BufferForCalculateFileHash, sizeof(g_BufferForCalculateFileHash));
}


Result IterateDirectoryRecursiveInternal(char* path, const size_t size, DirectoryEntry* pDirectoryEntry,
                                         std::function<Result(const char* path, const DirectoryEntry& entry)> enterDirectory,
                                         std::function<Result(const char* path, const DirectoryEntry& entry)> exitDirectory,
                                         std::function<Result(const char* path, const DirectoryEntry& entry)> findFile
                                         ) NN_NOEXCEPT
{
    DirectoryHandle handle;
    NN_RESULT_DO(OpenDirectory(&handle, path, OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT
    {
        CloseDirectory(handle);
    };

    while(NN_STATIC_CONDITION(true))
    {
        int64_t readNum = 0;
        NN_RESULT_DO(ReadDirectory(&readNum, pDirectoryEntry, handle, 1));

        if( readNum == 0 )
        {
            break;
        }

        auto parentPathSize = strnlen(path, size);
        strncat(path, pDirectoryEntry->name, size - parentPathSize);
        {
            if( pDirectoryEntry->directoryEntryType == DirectoryEntryType_Directory )
            {
                if( enterDirectory )
                {
                    NN_RESULT_DO(enterDirectory(path, *pDirectoryEntry));
                }
                strncat(path, "/", size - parentPathSize - strnlen(pDirectoryEntry->name, size));
                NN_RESULT_DO(IterateDirectoryRecursiveInternal(path, size, pDirectoryEntry, enterDirectory, exitDirectory, findFile));
                if( exitDirectory )
                {
                    NN_RESULT_DO(exitDirectory(path, *pDirectoryEntry));
                }
            }
            else
            {
                if( findFile )
                {
                    NN_RESULT_DO(findFile(path, *pDirectoryEntry));
                }
            }
        }
        path[parentPathSize] = '\0';

    }

    NN_RESULT_SUCCESS;
}


Result IterateDirectoryRecursiveInternalBreadth(char* path, const size_t size, DirectoryEntry* pDirectoryEntry,
                                         std::function<Result(const char* path, const DirectoryEntry& entry)> enterDirectory,
                                         std::function<Result(const char* path, const DirectoryEntry& entry)> exitDirectory,
                                         std::function<Result(const char* path, const DirectoryEntry& entry)> findFile
                                         ) NN_NOEXCEPT
{
    DirectoryHandle handle;
    auto parentPathSize = strnlen(path, size);
    int64_t readNum = 0;

    {
        NN_RESULT_DO(OpenDirectory(&handle, path, OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            CloseDirectory(handle);
        };

        while (NN_STATIC_CONDITION(true))
        {
            NN_RESULT_DO(ReadDirectory(&readNum, pDirectoryEntry, handle, 1));

            if (readNum == 0)
            {
                break;
            }

            strncat(path, pDirectoryEntry->name, size - parentPathSize);
            {
                if (pDirectoryEntry->directoryEntryType != DirectoryEntryType_Directory)
                {
                    if (findFile)
                    {
                        NN_RESULT_DO(findFile(path, *pDirectoryEntry));
                    }
                }
            }
            path[parentPathSize] = '\0';
        }
    }

    {
        NN_RESULT_DO(OpenDirectory(&handle, path, OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            CloseDirectory(handle);
        };

        while (NN_STATIC_CONDITION(true))
        {
            readNum = 0;
            NN_RESULT_DO(ReadDirectory(&readNum, pDirectoryEntry, handle, 1));

            if (readNum == 0)
            {
                break;
            }

            strncat(path, pDirectoryEntry->name, size - parentPathSize);
            {
                if (pDirectoryEntry->directoryEntryType == DirectoryEntryType_Directory)
                {
                    if (enterDirectory)
                    {
                        NN_RESULT_DO(enterDirectory(path, *pDirectoryEntry));
                    }
                    strncat(path, "/", size - parentPathSize - strnlen(pDirectoryEntry->name, size));
                    NN_RESULT_DO(IterateDirectoryRecursiveInternalBreadth(path, size, pDirectoryEntry, enterDirectory, exitDirectory, findFile));
                    if (exitDirectory)
                    {
                        NN_RESULT_DO(exitDirectory(path, *pDirectoryEntry));
                    }
                }
            }
            path[parentPathSize] = '\0';
        }
    }

    NN_RESULT_SUCCESS;
}

Result IterateDirectoryRecursive(const char* path,
                                 std::function<Result(const char* path, const DirectoryEntry& entry)> enterDirectory,
                                 std::function<Result(const char* path, const DirectoryEntry& entry)> exitDirectory,
                                 std::function<Result(const char* path, const DirectoryEntry& entry)> findFile,
                                 bool isBreadthFirst
                                 ) NN_NOEXCEPT
{
    DirectoryEntry directoryEntryBuffer = {{0}}; // スタック節約用
    auto pathBuffer = AllocateBuffer(nn::fs::EntryNameLengthMax);
    strncpy(pathBuffer.get(), path, nn::fs::EntryNameLengthMax);

    if (isBreadthFirst)
    {
        return IterateDirectoryRecursiveInternalBreadth(pathBuffer.get(), nn::fs::EntryNameLengthMax, &directoryEntryBuffer, enterDirectory, exitDirectory, findFile);
    }
    return IterateDirectoryRecursiveInternal(pathBuffer.get(), nn::fs::EntryNameLengthMax, &directoryEntryBuffer, enterDirectory, exitDirectory, findFile);
}

Result IterateDirectoryRecursive(const char* path,
    std::function<Result(const char* path, const DirectoryEntry& entry)> enterDirectory,
    std::function<Result(const char* path, const DirectoryEntry& entry)> exitDirectory,
    std::function<Result(const char* path, const DirectoryEntry& entry)> findFile
) NN_NOEXCEPT
{
    return IterateDirectoryRecursive(path, enterDirectory, exitDirectory, findFile, false);
}

Result DumpDirectoryRecursive(const char* path, bool continueOnFailure) NN_NOEXCEPT
{
    return DumpDirectoryRecursive(path, continueOnFailure, g_BufferForCalculateFileHash, sizeof(g_BufferForCalculateFileHash));
}

Result DumpDirectoryRecursive(const char* path, bool continueOnFailure, void* buffer, size_t bufferSize, bool silent, bool isBreadthFirst) NN_NOEXCEPT
{
    return IterateDirectoryRecursive(path,
        [&](const char* path, const DirectoryEntry& entry){
            NN_UNUSED(entry);
            if (!silent)
            {
                NN_LOG("DIR  <%s>\n", path);
            }
            NN_RESULT_SUCCESS;
        },
        nullptr,
        [&](const char* path, const DirectoryEntry& entry) -> Result {
            Hash hash = {{0}};
            auto result = CalculateFileHash(&hash, path, buffer, bufferSize);
            if(result.IsFailure())
            {
                if (!silent)
                {
                    NN_LOG("CalculateFileHash failed at <%-40s> with error 0x%08x\n", path, result.GetInnerValueForDebug());
                }
                if (continueOnFailure)
                {
                    if (std::strncmp(path, "sdcard:/Nintendo/save/", 22) != 0)
                    {
                        EXPECT_TRUE(false);
                    }
                }
                else
                {
                    return result;
                }
            }
            if (!silent)
            {
                NN_LOG("FILE <%-40s> %10lld\n", path, entry.fileSize);
                //nnt::fs::util::DumpBuffer(path, nn::fs::EntryNameLengthMax);
                nnt::fs::util::DumpBuffer(hash.value, sizeof(hash.value));
            }
            NN_RESULT_SUCCESS;
        },
        isBreadthFirst
        );
}

Result ListDirectoryRecursive(const char* path) NN_NOEXCEPT
{
    return IterateDirectoryRecursive(path,
        [](const char* path, const DirectoryEntry& entry){
            NN_UNUSED(entry);
            NN_LOG("DIR  <%s>\n", path);
            NN_RESULT_SUCCESS;
        },
        nullptr,
        [](const char* path, const DirectoryEntry& entry){
            NN_LOG("FILE <%-40s> %10lld\n", path, entry.fileSize);
            NN_RESULT_SUCCESS;
        }
        );
}

Result CalculateDirectoryTreeHashCore(Hash *outValue, const char* path, void* buffer, size_t bufferSize, bool isLite) NN_NOEXCEPT
{
    crypto::Sha256Generator sha;
    sha.Initialize();
    char fullPathBuffer[nn::fs::EntryNameLengthMax];

    NN_RESULT_DO(IterateDirectoryRecursive(path,
        [&](const char* path, const DirectoryEntry& entry) {
            int length = nn::util::SNPrintf(fullPathBuffer, sizeof(fullPathBuffer), "DIR  <%s>\n", path);
            sha.Update(fullPathBuffer, length);
            NN_UNUSED(entry);
            NN_RESULT_SUCCESS;
        },
        [&](const char* path, const DirectoryEntry& entry) {
            int length = nn::util::SNPrintf(fullPathBuffer, sizeof(fullPathBuffer), "(exit DIR)\n");
            sha.Update(fullPathBuffer, length);
            NN_UNUSED(entry);
            NN_UNUSED(path);
            NN_RESULT_SUCCESS;
        },
        [&](const char* path, const DirectoryEntry& entry) -> Result {
            int length = nn::util::SNPrintf(fullPathBuffer, sizeof(fullPathBuffer), "FILE <%-40s> %10lld\n", path, entry.fileSize);
            sha.Update(fullPathBuffer, length);
            if (!isLite)
            {
                Hash fileHash;
                NN_RESULT_DO(CalculateFileHash(&fileHash, path, buffer, bufferSize));
                sha.Update(&fileHash, sizeof(fileHash));
            }
            NN_RESULT_SUCCESS;
        }
        )
    );
    sha.GetHash(outValue->value, sizeof(outValue->value));
    NN_RESULT_SUCCESS;
}

Result CalculateDirectoryTreeHash(Hash *outValue, const char* path, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return CalculateDirectoryTreeHashCore(outValue, path, buffer, bufferSize, false);
}

Result CalculateDirectoryTreeHashLite(Hash *outValue, const char* path) NN_NOEXCEPT
{
    return CalculateDirectoryTreeHashCore(outValue, path, nullptr, 0, true);
}

Result GetDirectorySize(int64_t* outValue, const char* path) NN_NOEXCEPT
{
    int64_t totalSize = 0;

    NN_RESULT_DO(IterateDirectoryRecursive(path,
        [&](const char* path, const DirectoryEntry& entry) {
            NN_UNUSED(path);
            NN_UNUSED(entry);
            NN_RESULT_SUCCESS;
        },
        [&](const char* path, const DirectoryEntry& entry) {
            NN_UNUSED(path);
            NN_UNUSED(entry);
            NN_RESULT_SUCCESS;
        },
        [&](const char* path, const DirectoryEntry& entry) -> Result {
            totalSize += entry.fileSize;
            NN_UNUSED(path);
            NN_RESULT_SUCCESS;
        }
        )
    );
    *outValue = totalSize;
    NN_RESULT_SUCCESS;
}

Result ListDirectoryRecursive(Vector<String>* outFileList, Vector<String>* outDirList, const char* path) NN_NOEXCEPT
{
    return IterateDirectoryRecursive(path,
        [&outDirList](const char* path, const DirectoryEntry& entry) {
            NN_UNUSED(entry);
            if (outDirList)
            {
                String pathString(path);
                outDirList->push_back(pathString);
            }
            NN_RESULT_SUCCESS;
        },
        nullptr,
        [&outFileList](const char* path, const DirectoryEntry& entry) {
            NN_UNUSED(entry);
            if (outFileList)
            {
                String pathString(path);
                outFileList->push_back(pathString);
            }
            NN_RESULT_SUCCESS;
        }
    );
}

namespace {
void CompareDirectoryRecursiveInternal(char* path1, char* path2, const size_t size, DirectoryEntry* pDirectoryEntry1, DirectoryEntry* pDirectoryEntry2, bool isSilent) NN_NOEXCEPT
{
    DirectoryHandle handle1;
    NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&handle1, path1, OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT
    {
        CloseDirectory(handle1);
    };
    DirectoryHandle handle2;
    NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&handle2, path2, OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT
    {
        CloseDirectory(handle2);
    };

    while(NN_STATIC_CONDITION(true))
    {
        int64_t readNum1 = 0;
        int64_t readNum2 = 0;
        NNT_ASSERT_RESULT_SUCCESS(ReadDirectory(&readNum1, pDirectoryEntry1, handle1, 1));
        NNT_ASSERT_RESULT_SUCCESS(ReadDirectory(&readNum2, pDirectoryEntry2, handle2, 1));

        ASSERT_EQ(readNum1, readNum2);
        ASSERT_EQ(pDirectoryEntry1->directoryEntryType, pDirectoryEntry2->directoryEntryType);

        if( readNum1 == 0 )
        {
            break;
        }

        auto parentPathSize1 = strnlen(path1, size);
        auto parentPathSize2 = strnlen(path2, size);
        strncat(path1, pDirectoryEntry1->name, size - parentPathSize1);
        strncat(path2, pDirectoryEntry2->name, size - parentPathSize2);
        {
            if( pDirectoryEntry1->directoryEntryType == DirectoryEntryType_Directory )
            {
                if( !isSilent )
                {
                    NN_LOG("DIR  <%s>\n", path1);
                }
                strncat(path1, "/", size - parentPathSize1 - strnlen(pDirectoryEntry1->name, size));
                strncat(path2, "/", size - parentPathSize2 - strnlen(pDirectoryEntry2->name, size));
                CompareDirectoryRecursiveInternal(path1, path2, size, pDirectoryEntry1, pDirectoryEntry2, isSilent);
            }
            else
            {
                if( !isSilent )
                {
                    NN_LOG("FILE <%-40s> %10lld\n", path1, pDirectoryEntry1->fileSize);
                }
                Hash hash1 = {{0}};
                Hash hash2 = {{0}};
                CalculateFileHash(&hash1, path1);
                CalculateFileHash(&hash2, path2);
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(hash1.value, hash2.value, sizeof(hash1.value));
            }
        }
        path1[parentPathSize1] = '\0';
        path2[parentPathSize2] = '\0';
    }
}

void CompareDirectoryRecursiveImpl(const char* path1, const char* path2, bool isSilent) NN_NOEXCEPT
{
    DirectoryEntry directoryEntryBuffer1 = {{0}}; // スタック節約用
    DirectoryEntry directoryEntryBuffer2 = {{0}}; // スタック節約用
    auto pathBuffer1 = AllocateBuffer(nn::fs::EntryNameLengthMax);
    auto pathBuffer2 = AllocateBuffer(nn::fs::EntryNameLengthMax);
    strncpy(pathBuffer1.get(), path1, nn::fs::EntryNameLengthMax);
    strncpy(pathBuffer2.get(), path2, nn::fs::EntryNameLengthMax);

    CompareDirectoryRecursiveInternal(pathBuffer1.get(), pathBuffer2.get(), nn::fs::EntryNameLengthMax, &directoryEntryBuffer1, &directoryEntryBuffer2, isSilent);
}
}

void CompareDirectoryRecursive(const char* path1, const char* path2) NN_NOEXCEPT
{
    CompareDirectoryRecursiveImpl(path1, path2, false);
}

void CompareDirectoryRecursiveSilently(const char* path1, const char* path2) NN_NOEXCEPT
{
    CompareDirectoryRecursiveImpl(path1, path2, true);
}

// TODO: エラーハンドリングの修正
Result CopyFile(const char* srcPath, const char* dstPath, void* buffer, const size_t bufferSize) NN_NOEXCEPT
{
    FileHandle srcHandle;
    FileHandle dstHandle;

    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenFile(&srcHandle, srcPath, OpenMode_Read));
    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetFileSize(&size, srcHandle));

    DeleteFile(dstPath);
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateFile(dstPath, size));
    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenFile(&dstHandle, dstPath, OpenMode_Write));

    int64_t restSize = size;
    int64_t offset = 0;
    while(restSize > 0)
    {
        size_t readSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFile(&readSize, srcHandle, offset, buffer, bufferSize, ReadOption()));
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteFile(dstHandle, offset, buffer, readSize, WriteOption()));

        restSize -= readSize;
        offset   += readSize;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(FlushFile(dstHandle));
    CloseFile(dstHandle);
    CloseFile(srcHandle);

    NN_RESULT_SUCCESS;
}

Result WriteFileWithFillBuffer(const char* path, int64_t size, int64_t offsetCount, std::function<void(void*, const size_t, const int64_t)> fillBuffer) NN_NOEXCEPT
{
    FileHandle handle;
    NN_RESULT_DO(OpenFile(&handle, path, OpenMode_Write));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handle);
    };

    const size_t BufferSize = 1024 * 1024;
    auto buffer = AllocateBuffer(BufferSize);

    int64_t restSize = size;
    int64_t offset = 0;
    while(restSize > 0)
    {
        auto writeSize = static_cast<size_t>(std::min<int64_t>(BufferSize, restSize));
        fillBuffer(buffer.get(), writeSize, offsetCount + offset);
        NN_RESULT_DO(WriteFile(handle, offset, buffer.get(), writeSize, WriteOption()));

        restSize -= writeSize;
        offset   += writeSize;
    }

    NN_RESULT_DO(FlushFile(handle));

    NN_RESULT_SUCCESS;
}

Result WriteFileWith32BitCount(const char* path, int64_t size, int64_t offsetCount) NN_NOEXCEPT
{
    NN_RESULT_DO(WriteFileWithFillBuffer(path, size, offsetCount, FillBufferWith32BitCount));
    NN_RESULT_SUCCESS;
}

Result CreateFileWith32BitCount(const char* path, int64_t size, int64_t offsetCount) NN_NOEXCEPT
{
    NN_RESULT_DO(CreateFile(path, size));

    NN_RESULT_DO(WriteFileWith32BitCount(path, size, offsetCount));

    NN_RESULT_SUCCESS;
}

Result CreateFileWith8BitCount(const char* path, int64_t size, int64_t offsetCount) NN_NOEXCEPT
{
    NN_RESULT_DO(CreateFile(path, size));

    NN_RESULT_DO(WriteFileWithFillBuffer(path, size, offsetCount, FillBufferWith8BitCount));

    NN_RESULT_SUCCESS;
}

Result CreateFileWithZero(const char* path, int64_t size, int64_t offsetCount) NN_NOEXCEPT
{
    NN_RESULT_DO(CreateFile(path, size));

    NN_RESULT_DO(WriteFileWithFillBuffer(path, size, offsetCount, FillBufferWithZero));

    NN_RESULT_SUCCESS;
}

Result CreateAndWriteFileAtOnce(const char* path, const void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_RESULT_DO(CreateFile(path, bufferSize));

    FileHandle handle;
    NN_RESULT_DO(OpenFile(&handle, path, OpenMode_Write));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handle);
    };

    NN_RESULT_DO(WriteFile(handle, 0, pBuffer, bufferSize, WriteOption::MakeValue(WriteOptionFlag_Flush)));
    NN_RESULT_SUCCESS;
}

Result ReadFileAtOnce(size_t* outValue, const char* path, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    FileHandle handle;
    NN_RESULT_DO(OpenFile(&handle, path, OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handle);
    };

    NN_RESULT_DO(ReadFile(outValue, handle, 0, pBuffer, bufferSize, ReadOption()));
    NN_RESULT_SUCCESS;
}

Result ReadFileAtOnce(size_t* outSize, decltype(AllocateBuffer(0)) * outBuffer, const char* path) NN_NOEXCEPT
{
    FileHandle handle;
    NN_RESULT_DO(OpenFile(&handle, path, OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handle);
    };

    int64_t size;
    NN_RESULT_DO(GetFileSize(&size, handle));

    auto buffer = AllocateBuffer(static_cast<size_t>(size));
    NN_RESULT_THROW_UNLESS(buffer != nullptr, ResultAllocationMemoryFailed());

    size_t readSize;
    NN_RESULT_DO(ReadFile(&readSize, handle, 0, buffer.get(), static_cast<size_t>(size), ReadOption()));

    *outSize = readSize;
    *outBuffer = std::move(buffer);

    NN_RESULT_SUCCESS;
}


bool IsFilledWith32BitCount(const char* path, const int64_t offsetCount) NN_NOEXCEPT
{
    FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenFile(&handle, path, OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handle);
    };

    const size_t BufferSize = 1024 * 1024;
    auto buffer = AllocateBuffer(BufferSize);

    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetFileSize(&size, handle));
    int64_t restSize = size;
    int64_t offset = 0;
    while (restSize > 0)
    {
        auto accessSize = static_cast<size_t>(std::min<int64_t>(BufferSize, restSize));
        size_t readSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFile(&readSize, handle, offset, buffer.get(), accessSize));

        if (!IsFilledWith32BitCount(buffer.get(), readSize, offset + offsetCount))
        {
            return false;
        }

        restSize -= readSize;
        offset += readSize;
    }

    return true;
}


void WaitGameCardAttachDetach(bool attach) NN_NOEXCEPT
{
    if( attach != IsGameCardInserted() )
    {
        Result result;

        NN_LOG("Waiting %s of game card.\n", attach ? "attach" : "detach");

        // 挿抜イベント通知をオープン
        std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
        result = OpenGameCardDetectionEventNotifier(&deviceDetectionEventNotifier);
        if( result.IsFailure() )
        {
            NN_ABORT("failed to open notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }

        // システムイベントと紐づけ
        nn::os::SystemEventType deviceDetectionEvent;
        result = deviceDetectionEventNotifier->BindEvent(&deviceDetectionEvent, nn::os::EventClearMode_ManualClear);
        if( result.IsFailure() )
        {
            NN_ABORT("failed to bind system event to notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }
        // 挿入を待つ
        while( attach != IsGameCardInserted() )
        {
            // イベント通知を待つ
            nn::os::WaitSystemEvent(&deviceDetectionEvent);
            nn::os::ClearSystemEvent(&deviceDetectionEvent);
        }

        // イベントを破棄
        nn::os::DestroySystemEvent(&deviceDetectionEvent);
    }
}

void WaitSdCardAttachDetach(bool attach) NN_NOEXCEPT
{
    if (attach != IsSdCardInserted())
    {
        Result result;

        NN_LOG("Waiting %s of sd card.\n", attach ? "attach" : "detach");

        // 挿抜イベント通知をオープン
        std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
        result = OpenSdCardDetectionEventNotifier(&deviceDetectionEventNotifier);
        if (result.IsFailure())
        {
            NN_ABORT("failed to open notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }

        // システムイベントと紐づけ
        nn::os::SystemEventType deviceDetectionEvent;
        result = deviceDetectionEventNotifier->BindEvent(&deviceDetectionEvent, nn::os::EventClearMode_ManualClear);
        if (result.IsFailure())
        {
            NN_ABORT("failed to bind system event to notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }
        // 挿入を待つ
        while (attach != IsSdCardInserted())
        {
            // イベント通知を待つ
            nn::os::WaitSystemEvent(&deviceDetectionEvent);
            nn::os::ClearSystemEvent(&deviceDetectionEvent);
        }

        // イベントを破棄
        nn::os::DestroySystemEvent(&deviceDetectionEvent);
    }
}
void WaitGameCardAttach() NN_NOEXCEPT
{
    WaitGameCardAttachDetach(true);
}

void WaitGameCardDetach() NN_NOEXCEPT
{
    WaitGameCardAttachDetach(false);
}

void DetachAttachGameCard() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    nn::gc::Detach();
    nn::gc::Attach();
#else
    nnt::fs::util::WaitGameCardDetach();
    nnt::fs::util::WaitGameCardAttach();
#endif
}

void DetachAttachSdCard() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    nn::fs::detail::InvalidateSdCardForWin();
#else
    WaitSdCardAttachDetach(false);
    WaitSdCardAttachDetach(true);
#endif
}

// RandomSeed 用の変数
os::SdkMutexType g_RandomSeedMutex = NN_OS_SDK_MUTEX_INITIALIZER();
nn::util::optional<os::ThreadLocalStorage> g_pRandomSeedTlsSlot;

/**
    @brief GetRandomSeed でスレッドごとに呼び出し順に応じて一意な系列を取得できるように、スレッド固有の RandomSeed を初期化します。
    @details 各スレッドで GetRandomSeed の呼び出し前に SetInitialThreadLocalRandomSeed を呼び出しておく必要があります。
*/
void InitializeThreadLocalRandomSeed() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_RandomSeedMutex)> scopedLock(g_RandomSeedMutex);

    if( g_pRandomSeedTlsSlot )
    {
        NN_ABORT("ThreadLocalRandomSeed is already initialized.\n");
    }

    // TLS スロットを確保
    g_pRandomSeedTlsSlot.emplace();
}

/**
    @brief 呼び出しスレッドの初期 RandomSeed を設定します。
    @details initialSeed が 0 の場合はシードの初期値を時刻から生成します。
*/
void SetInitialThreadLocalRandomSeed(int initialSeed) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_RandomSeedMutex)> scopedLock(g_RandomSeedMutex);

    if( !g_pRandomSeedTlsSlot )
    {
        NN_ABORT("ThreadLocalRandomSeed is not initialized.\n");
    }

    // initialSeed が 0 ならランダムに生成する
    if( initialSeed == 0 )
    {
        nn::os::GenerateRandomBytes(&initialSeed, sizeof(initialSeed));
        if( initialSeed == 0 )
        {
            initialSeed = 1;
        }
    }

    // 更新
    g_pRandomSeedTlsSlot->SetValue(initialSeed);

    NN_LOG("InitialThreadLocalRandomSeed(%s): %d\n", nn::os::GetThreadNamePointer(nn::os::GetCurrentThread()), initialSeed);
}

/**
    @brief 乱数生成クラスに渡すシードを生成します。
    @details InitializeThreadLocalRandomSeed を事前に呼び出した場合は、スレッドごとに呼び出し順に応じて一意な値を返します。
             そうでない場合は、プログラム全体で呼び出し順に応じて一意な値を返します。
             シードの初期値は SetInitialThreadLocalRandomSeed で指定されたもの、もしくは初回呼び出し時に時刻から生成されたものが利用されます。
*/
int GetRandomSeed() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_RandomSeedMutex)> scopedLock(g_RandomSeedMutex);

    // TLS を利用する場合
    if( g_pRandomSeedTlsSlot )
    {
        // TLS の値が未初期化の場合は
        int seed = static_cast<int>(g_pRandomSeedTlsSlot->GetValue());
        if(seed == 0)
        {
            NN_ABORT("InitialThreadLocalRandomSeed for thread<%s> is not set.\n", nn::os::GetThreadNamePointer(nn::os::GetCurrentThread()));
        }
        else
        {
            // シードを更新
            ++seed;
        }

        // 0 は未初期化を表すため、計算結果が 0 の場合は 1 にします。
        if(seed == 0)
        {
            seed = 1;
        }

        g_pRandomSeedTlsSlot->SetValue(seed);

        return seed;
    }

    // TLS 利用しない場合
    {
        static int s_Seed = 0;
        if( s_Seed == 0 )
        {
            nn::os::GenerateRandomBytes(&s_Seed, sizeof(s_Seed));

            // 初回のシード生成時のみ、生成したシードをログに出力します。
            NN_LOG("InitialRandomSeed(Global): %d\n", s_Seed);
        }
        else
        {
            ++s_Seed;
        }

        // 0 は未初期化を表すため、計算結果が 0 の場合は 1 にします。
        if( s_Seed == 0 )
        {
            s_Seed = 1;
        }

        return s_Seed;
    }
}

}}}


namespace testing { namespace internal {

// gtest.cc から拝借
enum GTestColor {
    COLOR_DEFAULT,
    COLOR_RED,
    COLOR_GREEN,
    COLOR_YELLOW
};

void ColoredPrintf(GTestColor color, const char* fmt, ...) NN_NOEXCEPT;
}}

namespace nnt { namespace fs { namespace util {

void OutPutSkipMarker() NN_NOEXCEPT
{
    ::testing::internal::ColoredPrintf(::testing::internal::COLOR_YELLOW, "[ SKIPPED  ] "); \
}

/**
* @brief AllocateBufferのカスタムデリータ。
* @param[in] ptr malloc で確保されたバッファのポインタ
*/
static os::Mutex g_AllocateMutex(false);
void DeleterBuffer(void* ptr) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_AllocateMutex);
    free(ptr);
}

/**
* @brief カスタムデリータを用いたローカルのメモリ確保
* @param[in] size malloc で確保するバッファのサイズ
*/
std::unique_ptr<char, decltype(&DeleterBuffer)> AllocateBuffer(size_t size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_AllocateMutex);
    std::unique_ptr<char, decltype(&DeleterBuffer)> m_Buffer(reinterpret_cast<char*>(malloc(size)), DeleterBuffer);
    if(m_Buffer.get() == nullptr)
    {
        ADD_FAILURE();
        NN_LOG("Memory Allocate Error :Size 0x%x\n", size);
        return m_Buffer;
    }
    memset(m_Buffer.get(), 0x00, size);
    return m_Buffer;
}

/**
* @brief AllocateAlignedBufferのカスタムデリータ。
* @param[in] ptr malloc で確保されたバッファのポインタ
*/
void DeleterAlignedBuffer(void* ptr) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    _aligned_free(ptr);
#else
    free(ptr);
#endif
}


/**
* @brief カスタムデリータを用いたローカルのメモリ確保
* @param[in] size      malloc で確保するバッファのサイズ
* @param[in] alignment malloc で確保するバッファのアライメント
*/
std::unique_ptr<char, decltype(&DeleterBuffer)> AllocateAlignedBuffer(size_t size, int alignment) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    std::unique_ptr<char, decltype(&DeleterAlignedBuffer)> m_Buffer(reinterpret_cast<char*>(_aligned_malloc(size, alignment)), DeleterAlignedBuffer);
#else
    std::unique_ptr<char, decltype(&DeleterAlignedBuffer)> m_Buffer(reinterpret_cast<char*>(aligned_alloc(alignment, size)), DeleterAlignedBuffer);
#endif
    if (m_Buffer.get() == nullptr)
    {
        ADD_FAILURE();
        NN_LOG("Memory Allocate Error :Size 0x%x\n", size);
        return m_Buffer;
    }
    memset(m_Buffer.get(), 0x00, size);
    return m_Buffer;
}

// 対象ディレクトリ以下からディレクトリをランダムに選ぶ。対象ディレクトリ自身も対象
nn::Result ChooseDirectoryRandomly(String* pOutDir, const char* path)
{
    Vector<String> dirList;
    NN_RESULT_DO(ListDirectoryRecursive(nullptr, &dirList, path));
    dirList.push_back(String(path));

    std::mt19937 random(nnt::fs::util::GetRandomSeed());
    int index = random() % dirList.size();

    *pOutDir = dirList[index];
    NN_RESULT_SUCCESS;
}

int64_t GetOffsetByFileName(const char* fileName, size_t fileNameLength)
{
    int64_t offset = 0;
    nn::crypto::Sha256Generator sha;
    sha.Initialize();
    sha.Update(fileName, fileNameLength);
    char resultBuffer[32];
    sha.GetHash(resultBuffer, sizeof(resultBuffer));
    // とりあえず sha256 のハッシュの下 4 ケタでトライ
    for(int i = 0; i < 4; i++)
    {
        offset |= (static_cast<int64_t>(resultBuffer[i]) & 0xFF) << i * 8;
    }
    return offset;
}

int64_t GetOffsetByFileName(const char* filePath)
{
    String str = String(filePath);
    return GetOffsetByFileName(&(filePath[str.rfind("/") + 1]), strnlen(&(filePath[str.rfind("/") + 1]), 32));
}

// 対象ディレクトリ下にディレクトリツリーをランダムに生成します。
// 最大で totalEntryCount 個, ファイルサイズの累計が最大 totalSize になる程度に作成します。（正確ではない）
// ファイルの内容は CreateFileWith32BitCount(path, size, 0);
// 名前重複、容量不足、フルパス名長制限を踏んだ場合などは失敗してその時点で返ります。
nn::Result CreateTestDirectoryTreeRandomly(const char* path, int64_t totalSize, int64_t totalEntryCount, bool isFileNameOffsetEnable)
{
    int64_t restSize = totalSize;
    int64_t restEntryCount = totalEntryCount;

    while (restEntryCount > 0 && restSize > 0)
    {
        String parentDirectory;
        NN_RESULT_DO(ChooseDirectoryRandomly(&parentDirectory, path));

        std::mt19937 random(nnt::fs::util::GetRandomSeed());
        std::uniform_real_distribution<> dist(0.0, 1.0);

        bool isFile = dist(random) > 0.5;

        if (isFile)
        {
            // 適当にサイズ決める
            int64_t size = static_cast<int64_t>(dist(random) * totalSize / totalEntryCount * 2);
            auto name = GenerateRandomLengthEntryName(32);
            if ((parentDirectory + "/" + name).length() > 100)
            {
                continue;
            }
            int64_t offset = 0;
            if(isFileNameOffsetEnable)
            {
                offset = GetOffsetByFileName(name.c_str(), name.length());
            }
            NN_RESULT_TRY(CreateFileWith32BitCount((parentDirectory + "/" + name).c_str(), size, offset))
                NN_RESULT_CATCH(ResultTooLongPath)
                {
                    continue; // skip
                }
                NN_RESULT_CATCH(ResultPathAlreadyExists)
                {
                    continue; // skip
                }
            NN_RESULT_END_TRY

            restSize -= size;
        }
        else
        {
            auto name = GenerateRandomLengthEntryName(32);
            if((parentDirectory + "/" + name).length() > 90)
            {
                continue;
            }
            NN_RESULT_TRY(CreateDirectory((parentDirectory + "/" + name).c_str()))
                NN_RESULT_CATCH(ResultTooLongPath)
                {
                    continue; // skip
                }
                NN_RESULT_CATCH(ResultPathAlreadyExists)
                {
                    continue; // skip
                }
            NN_RESULT_END_TRY

        }

        restEntryCount--;
    }

    NN_RESULT_SUCCESS;
}



// 対象ディレクトリ下にディレクトリツリーをランダムに生成します。
// 最大で totalEntryCount 個, ファイルサイズの累計が最大 totalSize になる程度に作成します。（正確ではない）
// ファイルの内容は CreateFileWith32BitCount(path, size, 0);
// 名前重複、容量不足、フルパス名長制限を踏んだ場合などは失敗してその時点で返ります。
nn::Result CreateTestDirectoryTreeRandomly(const char* path, int64_t totalSize, int64_t totalEntryCount)
{
    return CreateTestDirectoryTreeRandomly(path, totalSize, totalEntryCount, false);
}

// CreateTestDirectoryTreeRandomly の各ファイルのオフセットがファイル名のハッシュで決まる版です
nn::Result CreateTestDirectoryTreeRandomlyByFileNameOffset(const char* path, int64_t totalSize, int64_t totalEntryCount)
{
    return CreateTestDirectoryTreeRandomly(path, totalSize, totalEntryCount, true);
}



bool IsTestSaveData(const nn::fs::SaveDataInfo& info) NN_NOEXCEPT
{
    return
        (0x8000000000000001ULL <= info.systemSaveDataId && info.systemSaveDataId <= 0x800000000000000fULL) ||
        (0x8000000000004000ULL <= info.systemSaveDataId && info.systemSaveDataId <= 0x8000000000004fffULL) ||
        (info.saveDataType == nn::fs::SaveDataType::System && info.saveDataUserId == TestUserId0) ||
        (info.saveDataType == nn::fs::SaveDataType::System && info.saveDataUserId == TestUserId1) || // TODO: systemSaveDataId で判定
        (info.saveDataType != nn::fs::SaveDataType::System && info.applicationId.value == UserSaveDataApplicationId) ||
        info.applicationId.value == 0x0000000000000003ULL ||
        info.applicationId.value == TestApplicationId0.value ||
        info.applicationId.value == TestApplicationId1.value ||
        info.applicationId.value == 0x0005000C10000000ULL ||
        info.saveDataType == nn::fs::SaveDataType::Cache;
}

void ListAllSaveData(nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT
{
    Vector<nn::fs::SaveDataInfo> saveInfos;
    FindSaveData(&saveInfos, spaceId, [](const SaveDataInfo& info) {
        NN_UNUSED(info);
        return true;
    });

    for (auto& info : saveInfos)
    {
        Bit64 ownerId = 0xCCCCCCCCCCCCCCCCULL;
        auto result = GetSaveDataOwnerId(&ownerId, info.saveDataSpaceId, info.saveDataId);
        if( result.IsFailure() )
        {
            NN_SDK_LOG("(failed to GetSaveDataOwnerId by %08x)\n", result.GetInnerValueForDebug());
        }

        NN_LOG("%2x: %2x %016llx %016llx %10lld %d %016llx %016llx %llx-%llx\n",
            spaceId,
            info.saveDataSpaceId,
            info.saveDataId,
            info.applicationId.value,
            info.saveDataSize,
            info.saveDataType,
            info.systemSaveDataId,
            ownerId,
            info.saveDataUserId._data[0],
            info.saveDataUserId._data[1]
        );
    }
}

void ListAllSaveData() NN_NOEXCEPT
{
    NN_LOG("spaceId      saveDataId            AppId      size type  sysSaveDataId ownerId         userId \n");

    // nn::fs::SaveDataSpaceId::ProperSystem は通常起動では指定不可
    // nn::fs::SaveDataSpaceId::SafeMode は SIGLO-72300 のためデフォルト除外する
    const nn::fs::SaveDataSpaceId SpaceIds[] = { nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System, nn::fs::SaveDataSpaceId::SdUser, nn::fs::SaveDataSpaceId::SdSystem, nn::fs::SaveDataSpaceId::Temporary};
    for(auto spaceId : SpaceIds)
    {
        ListAllSaveData(spaceId);
    }
    NN_LOG("\n");
}

// TORIAEZU: 他からセーブが作られていないことを仮定
void DeleteAllTestSaveData() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::fssrv::detail::InvalidateAllIndexers();
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // 古いファイルが残っていると nn::fs::OpenSaveDataIterator が SaveDataSpaceId::SdSystem で Abort する
    // 回避のために事前に消しておく
    {
        if (MountSdCardForDebug("sd").IsSuccess())
        {
            {
                DeleteDirectoryRecursively("sd:/Nintendo/save");
                Unmount("sd");
            }

            {
                fssrv::SaveDataIndexer indexer("saveDataIxrDbSd", SaveDataSpaceId::SdSystem, fssrv::IndexerSaveDataId, GetTestLibraryAllocator());
                indexer.Commit();
            }
        }
    }
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

    // nn::fs::SaveDataSpaceId::ProperSystem は通常起動では指定不可
    // nn::fs::SaveDataSpaceId::SafeMode は SIGLO-72300 のためデフォルト除外する
    const nn::fs::SaveDataSpaceId SpaceIds[] = {nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System, nn::fs::SaveDataSpaceId::SdUser, nn::fs::SaveDataSpaceId::SdSystem, nn::fs::SaveDataSpaceId::Temporary};
    for(auto spaceId : SpaceIds)
    {
        while(NN_STATIC_CONDITION(true))
        {
            nn::util::optional<nn::fs::SaveDataInfo> info;
            FindSaveData(&info, spaceId, IsTestSaveData);
            if( info == nn::util::nullopt)
            {
                break;
            }

            auto saveId = info->saveDataId;
            NN_LOG("delete save data (%x, %llx).\n", spaceId, saveId);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteSaveData(spaceId, saveId));

            // TODO: DeleteSaveDataMetaFile()
        }
    }
}


namespace {
    const size_t SaveDataSize = 16 * 1024 * 1024;
    const size_t SaveDataJournalSize = 16 * 1024 * 1024;

}

nn::Result CreateAndMountSystemSaveData(const char* mountName, nn::fs::SystemSaveDataId id)
{
    NN_RESULT_TRY(nn::fs::MountSystemSaveData(mountName, id))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
    {
        NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, SaveDataSize, SaveDataJournalSize, 0));
        NN_RESULT_DO(nn::fs::MountSystemSaveData(mountName, id));
    }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSystemSaveData(const char* mountName, nn::fs::SaveDataSpaceId spaceId, nn::fs::SystemSaveDataId id) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountSystemSaveData(mountName, spaceId, id))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateSystemSaveData(spaceId, id, ApplicationId.value, SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountSystemSaveData(mountName, spaceId, id));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSystemSaveData(const char* mountName, nn::fs::SystemSaveDataId id, nn::fs::UserId userId)
{
    NN_RESULT_TRY(nn::fs::MountSystemSaveData(mountName, id, userId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
    {
        NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, userId, SaveDataSize, SaveDataJournalSize, 0));
        NN_RESULT_DO(nn::fs::MountSystemSaveData(mountName, id, userId));
    }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSystemSaveData(const char* mountName, nn::fs::SaveDataSpaceId spaceId, nn::fs::SystemSaveDataId id, nn::fs::UserId userId) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountSystemSaveData(mountName, spaceId, id, userId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateSystemSaveData(spaceId, id, userId, ApplicationId.value, SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountSystemSaveData(mountName, spaceId, id, userId));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSystemSaveData(const char* mountName, nn::fs::SystemSaveDataId id, nn::account::Uid userId) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountSystemSaveData(mountName, id, userId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, nn::fs::ConvertAccountUidToFsUserId(userId), SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountSystemSaveData(mountName, id, userId));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountDeviceSaveData(const char* mountName)
{
    NN_RESULT_TRY(nn::fs::MountDeviceSaveData(mountName))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
    {
        NN_RESULT_DO(nn::fs::CreateDeviceSaveData(ApplicationId, ApplicationId.value, SaveDataSize, SaveDataJournalSize, 0));
        NN_RESULT_DO(nn::fs::MountDeviceSaveData(mountName));
    }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountDeviceSaveData(const char* mountName, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountDeviceSaveData(mountName, applicationId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateDeviceSaveData(applicationId, applicationId.value, SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountDeviceSaveData(mountName, applicationId));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountBcatSaveData(const char* mountName, nn::ncm::ApplicationId applicationId)
{
    NN_RESULT_TRY(nn::fs::MountBcatSaveData(mountName, applicationId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
    {
        NN_RESULT_DO(nn::fs::CreateBcatSaveData(applicationId, SaveDataSize));
        NN_RESULT_DO(nn::fs::MountBcatSaveData(mountName, applicationId));
    }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSaveData(const char* mountName, nn::fs::UserId userId)
{
    NN_RESULT_TRY(nn::fs::MountSaveData(mountName, userId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
    {
        NN_RESULT_DO(nn::fs::CreateSaveData(ApplicationId, userId, ApplicationId.value, SaveDataSize, SaveDataJournalSize, 0));
        NN_RESULT_DO(nn::fs::MountSaveData(mountName, userId));
    }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSaveData(const char* mountName, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountSaveData(mountName, applicationId, userId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateSaveData(applicationId, userId, applicationId.value, SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountSaveData(mountName, applicationId, userId));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSaveData(const char* mountName, nn::account::Uid userId) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountSaveData(mountName, userId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateSaveData(ApplicationId, nn::fs::ConvertAccountUidToFsUserId(userId), ApplicationId.value, SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountSaveData(mountName, userId));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSaveData(const char* mountName, nn::ApplicationId applicationId, nn::account::Uid userId) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountSaveData(mountName, applicationId, userId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            nn::ncm::ApplicationId ncmApplicationId = { applicationId.value };
            NN_RESULT_DO(nn::fs::CreateSaveData(ncmApplicationId, nn::fs::ConvertAccountUidToFsUserId(userId), applicationId.value, SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountSaveData(mountName, applicationId, userId));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSaveDataReadOnly(const char* mountName, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountSaveDataReadOnly(mountName, applicationId, userId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateSaveData(applicationId, userId, applicationId.value, SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountSaveDataReadOnly(mountName, applicationId, userId));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSaveDataReadOnly(const char* mountName, nn::ApplicationId applicationId, nn::account::Uid userId) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountSaveDataReadOnly(mountName, applicationId, userId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            nn::ncm::ApplicationId ncmApplicationId = { applicationId.value };
            NN_RESULT_DO(nn::fs::CreateSaveData(ncmApplicationId, nn::fs::ConvertAccountUidToFsUserId(userId), applicationId.value, SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountSaveDataReadOnly(mountName, applicationId, userId));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSystemBcatSaveData(const char* mountName, nn::fs::SystemBcatSaveDataId id)
{
    NN_RESULT_TRY(nn::fs::MountSystemBcatSaveData(mountName, id))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
    {
        NN_RESULT_DO(nn::fs::CreateSystemBcatSaveData(id, SaveDataSize, 0));
        NN_RESULT_DO(nn::fs::MountSystemBcatSaveData(mountName, id));
    }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountTemporaryStorage(const char* mountName) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountTemporaryStorage(mountName))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateTemporaryStorage(ApplicationId, ApplicationId.value, SaveDataSize, 0));
            NN_RESULT_DO(nn::fs::MountTemporaryStorage(mountName));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountSaveDataInternalStorage(const char* mountName, nn::fs::SaveDataId id) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountSaveDataInternalStorage(mountName, nn::fs::SaveDataSpaceId::System, id))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountSaveDataInternalStorage(mountName, nn::fs::SaveDataSpaceId::System, id));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

namespace
{
    nn::fs::SaveDataSpaceId GetTestCacheStorageSpaceId() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        return nn::fs::SaveDataSpaceId::User;
#else
        if( !nn::fs::IsSdCardAccessible() )
        {
            return nn::fs::SaveDataSpaceId::User;
        }

        nn::fs::CacheStorageTargetMedia media;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetCacheStorageTargetMedia(&media, ApplicationId));
        if( media == nn::fs::CacheStorageTargetMedia::CacheStorageTargetMedia_Nand )
        {
            return nn::fs::SaveDataSpaceId::User;
        }

        return nn::fs::SaveDataSpaceId::SdUser;
#endif // defined(NN_BUILD_CONFIG_OS_WIN)
    }
}

nn::Result CreateAndMountCacheStorage(const char* mountName) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountCacheStorage(mountName))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateCacheStorage(
                ApplicationId,
                GetTestCacheStorageSpaceId(),
                ApplicationId.value,
                0,
                SaveDataSize,
                SaveDataJournalSize,
                0
            ));
            NN_RESULT_DO(nn::fs::MountCacheStorage(mountName));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountCacheStorage(const char* mountName, int index) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountCacheStorage(mountName, index))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateCacheStorage(
                ApplicationId,
                GetTestCacheStorageSpaceId(),
                ApplicationId.value,
                static_cast<uint16_t>(index),
                SaveDataSize,
                SaveDataJournalSize,
                0
            ));
            NN_RESULT_DO(nn::fs::MountCacheStorage(mountName, index));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountCacheStorage(const char* mountName, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountCacheStorage(mountName, applicationId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateCacheStorage(
                applicationId,
                GetTestCacheStorageSpaceId(),
                applicationId.value,
                0,
                SaveDataSize,
                SaveDataJournalSize,
                0
            ));
            NN_RESULT_DO(nn::fs::MountCacheStorage(mountName, applicationId));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result CreateAndMountCacheStorage(const char* mountName, nn::ncm::ApplicationId applicationId, int index) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountCacheStorage(mountName, applicationId, index))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateCacheStorage(
                applicationId,
                GetTestCacheStorageSpaceId(),
                applicationId.value,
                static_cast<uint16_t>(index),
                SaveDataSize,
                SaveDataJournalSize,
                0
            ));
            NN_RESULT_DO(nn::fs::MountCacheStorage(mountName, applicationId, index));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}


nn::Result CreateAndFindSaveData(
    nn::fs::SaveDataInfo* pOutValue,
    nn::ncm::ApplicationId applicationId,
    nn::fs::UserId userId,
    int64_t availableSize, int64_t journalSize
)
{
    NN_RESULT_DO(nn::fs::CreateSaveData(applicationId, userId, applicationId.value, availableSize, journalSize, 0));
    nn::fs::SaveDataInfo info;
    NN_RESULT_DO(FindSaveDataByApplicationId(&info, applicationId, userId));
    *pOutValue = info;
    NN_RESULT_SUCCESS;
}

nn::Result CreateAndExtendSaveData(
    nn::fs::SaveDataInfo* pOutValue,
    nn::ncm::ApplicationId applicationId,
    nn::fs::UserId userId,
    int64_t initialSize, int64_t initialJournalSize,
    int64_t finalSize, int64_t finalJournalSize,
    int extendCount
)
{
    NN_RESULT_DO(nn::fs::CreateSaveData(applicationId, userId, applicationId.value, initialSize, initialJournalSize, 0));

    nn::fs::SaveDataInfo info;
    NN_RESULT_DO(FindSaveDataByApplicationId(&info, applicationId, userId));

    // ExtendCountMax 回に分けて拡張
    for (int i = 0; i < extendCount; i++)
    {
        const int64_t nextSize        = initialSize        + nn::util::align_up((finalSize        - initialSize       ) * (i + 1) / extendCount, 16 * 1024);
        const int64_t nextJournalSize = initialJournalSize + nn::util::align_up((finalJournalSize - initialJournalSize) * (i + 1) / extendCount, 16 * 1024);
        NN_RESULT_DO(nn::fs::ExtendSaveData(info.saveDataSpaceId, info.saveDataId, nextSize, nextJournalSize));
    }

    // 拡張結果の情報を取得
    NN_RESULT_DO(FindSaveDataByApplicationId(&info, applicationId, userId));
    *pOutValue = info;

    NN_RESULT_SUCCESS;
}

nn::Result FindSaveDataBySystemSaveDataId(nn::fs::SaveDataInfo* outValue, nn::fs::SystemSaveDataId systemSaveDataId, nn::fs::UserId userId) NN_NOEXCEPT
{
    const nn::fs::SaveDataSpaceId SpaceIds[] = { nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System };
    for (auto spaceId : SpaceIds)
    {
        nn::util::optional<nn::fs::SaveDataInfo> info;
        nnt::fs::util::FindSaveData(&info, spaceId, [&](const nn::fs::SaveDataInfo tInfo)
        {
            return tInfo.systemSaveDataId == systemSaveDataId && tInfo.saveDataUserId == userId;
        });

        if (info != nn::util::nullopt)
        {
            *outValue = info.value();
            NN_RESULT_SUCCESS;
        }
    }
    return nn::fs::ResultTargetNotFound();
}

nn::Result FindSaveDataByApplicationId(nn::fs::SaveDataInfo* outValue, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId) NN_NOEXCEPT
{
    const nn::fs::SaveDataSpaceId SpaceIds[] = { nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System };
    for (auto spaceId : SpaceIds)
    {
        nn::util::optional<nn::fs::SaveDataInfo> info;
        nnt::fs::util::FindSaveData(&info, spaceId, [&](const nn::fs::SaveDataInfo tInfo)
        {
            return tInfo.applicationId.value == applicationId.value && tInfo.saveDataUserId == userId;
        });

        if (info != nn::util::nullopt)
        {
            *outValue = info.value();
            NN_RESULT_SUCCESS;
        }
    }
    return nn::fs::ResultTargetNotFound();
}


void CheckSaveData(const nn::fs::SaveDataInfo& expected, const nn::fs::SaveDataInfo& actual, int thumbnailFillValueOffset) NN_NOEXCEPT
{
    ASSERT_EQ(nn::fs::SaveDataType::Account, actual.saveDataType);
    ASSERT_EQ(nn::fs::SaveDataType::Account, expected.saveDataType);

    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData("actual", actual.applicationId, actual.saveDataUserId));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("actual");
        };
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData("expected", expected.applicationId, expected.saveDataUserId));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("expected");
        };
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("actual:/"));
        NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::CompareDirectoryRecursive("actual:/", "expected:/"));
    }

    {
        const int ThumbnailFileHeaderSize = 32;
        char thumbnailFileHeader[2][ThumbnailFileHeaderSize];
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFileHeader(expected.applicationId.value, expected.saveDataUserId, thumbnailFileHeader[0], ThumbnailFileHeaderSize));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFileHeader(actual.applicationId.value, actual.saveDataUserId, thumbnailFileHeader[1], ThumbnailFileHeaderSize));

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(thumbnailFileHeader[0], thumbnailFileHeader[1], ThumbnailFileHeaderSize);

        EXPECT_TRUE(nnt::fs::util::IsFilledWith8BitCount(thumbnailFileHeader[1], ThumbnailFileHeaderSize, thumbnailFillValueOffset));
    }

    {
        nn::fs::SaveDataExtraData srcExtraData;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&srcExtraData, expected.saveDataSpaceId, expected.saveDataId));
        nn::fs::SaveDataExtraData dstExtraData;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&dstExtraData, actual.saveDataSpaceId, actual.saveDataId));

        // user id は移行先の値につけかえられること
        EXPECT_EQ(expected.saveDataUserId, srcExtraData.attribute.userId);
        EXPECT_EQ(actual.saveDataUserId, dstExtraData.attribute.userId);

        // 他のフィールドは維持されること
        srcExtraData.attribute.userId = nn::fs::InvalidUserId;
        dstExtraData.attribute.userId = nn::fs::InvalidUserId;
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&srcExtraData, &dstExtraData, sizeof(nn::fs::SaveDataExtraData));
    }
}


bool operator<(const VerifyEntry& left, const VerifyEntry& right) NN_NOEXCEPT
{
    return strncmp(left.path, right.path, sizeof(left.path)) < 0;
}

nn::Result ListVerifyEntry(Vector<VerifyEntry>* pOutValue, const char* path) NN_NOEXCEPT
{
    return nnt::fs::util::IterateDirectoryRecursive(
        path,
        [=](const char* directoryPath, const nn::fs::DirectoryEntry& entry) NN_NOEXCEPT
        {
            NN_UNUSED(entry);
            VerifyEntry newEntry = {};
            strncpy(newEntry.path, directoryPath, nn::fs::EntryNameLengthMax);
            memset(newEntry.hash.value, 0, sizeof(newEntry.hash.value));
            pOutValue->push_back(newEntry);
            NN_RESULT_SUCCESS;
        },
        nullptr,
        [=](const char* filePath, const nn::fs::DirectoryEntry& entry) NN_NOEXCEPT
        {
            NN_UNUSED(entry);
            nnt::fs::util::Hash hash = {};
            nnt::fs::util::CalculateFileHash(&hash, filePath);
            VerifyEntry newEntry = {};
            strncpy(newEntry.path, filePath, nn::fs::EntryNameLengthMax);
            newEntry.hash = hash;
            pOutValue->push_back(newEntry);
            NN_RESULT_SUCCESS;
        }
    );
}

// 4 GB を超えるオフセットで IFile にアクセスする（AllowAppend 無効時）
void TestFileAccessWithLargeOffset(nn::fs::fsa::IFile* pFile, size_t bufferSize) NN_NOEXCEPT
{
    int64_t fileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSize));
    ASSERT_GE(fileSize, LargeOffsetMax + static_cast<int64_t>(bufferSize));

    // バッファ初期化
    std::unique_ptr<char> readBuffer(new char[bufferSize]);
    std::unique_ptr<char> writeBufferList[LargeOffsetListLength];
    for( auto& writeBuffer : writeBufferList )
    {
        writeBuffer.reset(new char[bufferSize]);
        FillBufferWithRandomValue(writeBuffer.get(), bufferSize);
    }

    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(pFile->Write(LargeOffsetList[i], writeBufferList[i].get(), bufferSize, nn::fs::WriteOption()));
    }

    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        size_t readSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(pFile->Read(&readSize, LargeOffsetList[i], readBuffer.get(), bufferSize, nn::fs::ReadOption()));
        ASSERT_EQ(bufferSize, readSize);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), bufferSize);
    }

    NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        size_t readSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(pFile->Read(&readSize, LargeOffsetList[i], readBuffer.get(), bufferSize, nn::fs::ReadOption()));
        ASSERT_EQ(bufferSize, readSize);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), bufferSize);
    }
}

// IFile の AllowAppend 動作テスト
void TestFileAccessAllowAppend(nn::fs::fsa::IFile* pFile, size_t bufferSize) NN_NOEXCEPT
{
    auto readBuffer = nnt::fs::util::AllocateBuffer(bufferSize);
    auto writeBuffer = nnt::fs::util::AllocateBuffer(bufferSize);

    int64_t fileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSize));

    const auto offsetList = {-1, 0, 1};

    for( auto offset : offsetList )
    {
        // 範囲外への読み書き
        NNT_ASSERT_RESULT_SUCCESS(pFile->Write(fileSize + offset, writeBuffer.get(), bufferSize, nn::fs::WriteOption()));

        size_t readSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(pFile->Read(&readSize, fileSize + offset, readBuffer.get(), bufferSize, nn::fs::ReadOption()));
        ASSERT_EQ(bufferSize, readSize);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), bufferSize);

        // 拡張されたサイズをテスト
        int64_t newFileSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&newFileSize));
        EXPECT_GE(fileSize + offset + static_cast<int64_t>(bufferSize), newFileSize);
        fileSize = newFileSize;
    }
}

// IFile の AllowAppend 無効の動作テスト
void TestFileAccessNotAllowAppend(nn::fs::fsa::IFile* pFile, size_t bufferSize) NN_NOEXCEPT
{
    int64_t fileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSize));

    auto readBuffer = nnt::fs::util::AllocateBuffer(bufferSize);
    auto writeBuffer = nnt::fs::util::AllocateBuffer(bufferSize);

    // 範囲外への読み書きが失敗する
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
        pFile->Write(fileSize - 1, writeBuffer.get(), bufferSize, nn::fs::WriteOption()));

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
        pFile->Write(fileSize, writeBuffer.get(), bufferSize, nn::fs::WriteOption()));

    // ResultInvalidOffset or ResultFileExtensionWithoutOpenModeAllowAppend
    const auto result = pFile->Write(fileSize + 1, writeBuffer.get(), bufferSize, nn::fs::WriteOption());
    if( !nn::fs::ResultInvalidOffset::Includes(result) )
    {
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend, result);
    }

    size_t readSize = 0;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        pFile->Read(&readSize, fileSize + 1, readBuffer.get(), bufferSize, nn::fs::ReadOption()));

    // サイズが変わっていない
    int64_t fileSize2 = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSize2));
    ASSERT_EQ(fileSize, fileSize2);
}

// 4 GB を超えるオフセットでファイルにアクセスする共通テスト
void TestFileSystemAccessWithLargeOffset(nn::fs::fsa::IFileSystem* pFileSystem, const char* filePath) NN_NOEXCEPT
{
    static const size_t AccessSize = 1024;
    static const int64_t FileSize = LargeOffsetMax + AccessSize;

    NNT_ASSERT_RESULT_SUCCESS(pFileSystem->CreateFile(filePath, FileSize, nn::fs::CreateFileOptionFlag_BigFile));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->DeleteFile(filePath));
    };

    {
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(
            &pFile,
            filePath,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));

        TestFileAccessWithLargeOffset(pFile.get(), AccessSize);
        TestFileAccessNotAllowAppend(pFile.get(), AccessSize);
    }
    {
        // AllowAppend ありのテスト
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(pFileSystem->OpenFile(
            &pFile,
            filePath,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)));

        TestFileAccessAllowAppend(pFile.get(), AccessSize);
    }
}

// 4 GB を超えるオフセットでファイルハンドルにアクセスする
void TestFileHandleAccessWithLargeOffset(nn::fs::FileHandle file, size_t bufferSize) NN_NOEXCEPT
{
    int64_t fileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));
    ASSERT_GE(fileSize, LargeOffsetMax + static_cast<int64_t>(bufferSize));

    // バッファ初期化
    std::unique_ptr<char> readBuffer(new char[bufferSize]);
    std::unique_ptr<char> writeBufferList[LargeOffsetListLength + 1];
    for( auto& writeBuffer : writeBufferList )
    {
        writeBuffer.reset(new char[bufferSize]);
        FillBufferWithRandomValue(writeBuffer.get(), bufferSize);
    }

    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(
            file, LargeOffsetList[i], writeBufferList[i].get(), bufferSize, nn::fs::WriteOption()));
    }

    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(
            file, LargeOffsetList[i], readBuffer.get(), bufferSize, nn::fs::ReadOption()));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), bufferSize);
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));
    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(
            file, LargeOffsetList[i], readBuffer.get(), bufferSize, nn::fs::ReadOption()));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), bufferSize);
    }
}

// ファイルハンドルの AllowAppend 動作テスト
void TestFileHandleAccessAllowAppend(nn::fs::FileHandle file, size_t bufferSize) NN_NOEXCEPT
{
    int64_t fileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));

    auto readBuffer = nnt::fs::util::AllocateBuffer(bufferSize);
    auto writeBuffer = nnt::fs::util::AllocateBuffer(bufferSize);

    const auto offsetList = {-1, 0, 1};

    for( auto offset : offsetList )
    {
        // 範囲外への読み書き
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, fileSize + offset, writeBuffer.get(), bufferSize, nn::fs::WriteOption()));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(file, fileSize + offset, readBuffer.get(), bufferSize, nn::fs::ReadOption()));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(file, fileSize + offset, readBuffer.get(), bufferSize, nn::fs::ReadOption()));

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), bufferSize);

        // 拡張されたサイズをテスト
        int64_t newFileSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&newFileSize, file));
        EXPECT_GE(fileSize + offset + static_cast<int64_t>(bufferSize), newFileSize);
        fileSize = newFileSize;
    }
}

// IFile の AllowAppend 無効の動作テスト
void TestFileHandleAccessNotAllowAppend(
    size_t bufferSize,
    std::function<void (std::function<void (nn::fs::FileHandle)>)> fileAccessor) NN_NOEXCEPT
{
    int64_t fileSize = 0;
    NNT_FS_ASSERT_NO_FATAL_FAILURE(fileAccessor([&](nn::fs::FileHandle file) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));
    }));

    auto readBuffer = nnt::fs::util::AllocateBuffer(bufferSize);
    auto writeBuffer = nnt::fs::util::AllocateBuffer(bufferSize);

    // 範囲外への読み書きが失敗する
    NNT_FS_ASSERT_NO_FATAL_FAILURE(fileAccessor([&](nn::fs::FileHandle file) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
            nn::fs::WriteFile(file, fileSize - 1, writeBuffer.get(), bufferSize, nn::fs::WriteOption()));
    }));

    NNT_FS_ASSERT_NO_FATAL_FAILURE(fileAccessor([&](nn::fs::FileHandle file) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
            nn::fs::WriteFile(file, fileSize, writeBuffer.get(), bufferSize, nn::fs::WriteOption()));
    }));

    NNT_FS_ASSERT_NO_FATAL_FAILURE(fileAccessor([&](nn::fs::FileHandle file) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
            nn::fs::WriteFile(file, fileSize + 1, writeBuffer.get(), bufferSize, nn::fs::WriteOption()));
    }));

    NNT_FS_ASSERT_NO_FATAL_FAILURE(fileAccessor([&](nn::fs::FileHandle file) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            nn::fs::ReadFile(file, fileSize + 1, readBuffer.get(), bufferSize, nn::fs::ReadOption()));
    }));

    // サイズが変わっていない
    int64_t fileSize2 = 0;
    NNT_FS_ASSERT_NO_FATAL_FAILURE(fileAccessor([&](nn::fs::FileHandle file) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize2, file));
    }));
    ASSERT_EQ(fileSize, fileSize2);
}

// 4 GB を超えるオフセットでストレージにアクセスする共通テスト（読み込みテストを独自関数で行う）
void TestStorageAccessWithLargeOffset(
    nn::fs::IStorage* pStorage,
    size_t bufferSize,
    std::function<void(int64_t, char*, const char*, size_t)> readTestFunc) NN_NOEXCEPT
{
    int64_t storageSize = 0;
    const auto result = pStorage->GetSize(&storageSize);
    if( !nn::fs::ResultNotImplemented::Includes(result) )
    {
        NNT_ASSERT_RESULT_SUCCESS(result);
        ASSERT_GE(storageSize, LargeOffsetMax + static_cast<int64_t>(bufferSize));
    }

    // バッファ初期化
    std::unique_ptr<char> readBuffer(new char[bufferSize]);
    std::unique_ptr<char> writeBufferList[LargeOffsetListLength];
    for( auto& writeBuffer : writeBufferList )
    {
        writeBuffer.reset(new char[bufferSize]);
        nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), bufferSize);
    }

    // 4 GB を超えるオフセットへ書き込み
    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(LargeOffsetList[i], writeBufferList[i].get(), bufferSize));
    }

    // Flush 前に読み込む
    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(LargeOffsetList[i], readBuffer.get(), bufferSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), bufferSize);
    }

    // Flush 後に読み込む
    NNT_ASSERT_RESULT_SUCCESS(pStorage->Flush());
    for( size_t i = 0; i < LargeOffsetListLength; ++i )
    {
        readTestFunc(LargeOffsetList[i], readBuffer.get(), writeBufferList[i].get(), bufferSize);
    }
}

// 4 GB を超えるオフセットでストレージにアクセスする共通テスト
void TestStorageAccessWithLargeOffset(nn::fs::IStorage* pStorage, size_t bufferSize) NN_NOEXCEPT
{
    TestStorageAccessWithLargeOffset(
        pStorage,
        bufferSize,
        [=](int64_t offset, char* readBuffer, const char* writeBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(offset, readBuffer, bufferSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, bufferSize);
        });
}

// 指定したパスのファイル、ディレクトリを削除します。対象が存在していなくても成功扱いにします。
nn::Result DeleteFileOrDirectoryIfExists(const char* path) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType entryType;
    NN_RESULT_TRY(nn::fs::GetEntryType(&entryType, path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    switch( entryType )
    {
    case nn::fs::DirectoryEntryType_File:
        NN_RESULT_DO(nn::fs::DeleteFile(path));
        break;

    case nn::fs::DirectoryEntryType_Directory:
        NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(path));
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief バッファの初期値領域を確認する関数です。
 *   @param[in]  pBuffer                初期値を確認するバッファの先頭ポインタを指定します
 *   @param[in]  bufferSize             指定しているバッファのサイズ（バイト単位）を指定します
 *   @param[in]  offsetSize             バッファをオフセットするサイズ（バイト単位）を指定します
 *   @param[in]  compareSize            書き込み・読み込みを比較するサイズ（バイト単位）を指定します
 *
 *   @return     指定バッファの比較結果が返ります。
 *   @retval     true                   指定バッファには初期値のみ含まれています。
 *   @retval     false                  指定バッファには初期値以外の内容が含まれています。
 */
bool IsValidInitializedBound(const uint8_t* pBuffer, const size_t bufferSize, const size_t offsetSize, const size_t compareSize) NN_NOEXCEPT
{
    // バッファ初期値の隣接比較サイズ
    const size_t CanarySizeMax = 32;

    // バッファ初期値の比較定義(InvalidateVariable()によって初期値に設定される値)
    const uint8_t InitializedValue = 0x5A;

    {
        // 比較終端までのサイズ設定
        const size_t canarySize = std::min(offsetSize, CanarySizeMax);

        // 比較する先頭バッファアドレスを設定（隣接サイズ分のみ比較するため、先頭アドレス移動）
        const uint8_t* pAddressPos = (pBuffer + offsetSize) - canarySize;

        // 設定した先頭バッファアドレスから初期値かを比較
        for( size_t checkPos = 0; checkPos < canarySize; ++checkPos )
        {
            if( pAddressPos[checkPos] != InitializedValue )
            {
                return false;
            }
        }
    }

    {
        // 比較終端までのサイズ設定
        const size_t canarySize
            = std::min(bufferSize - (offsetSize + compareSize), CanarySizeMax);

        // 比較する先頭バッファアドレスを設定（Read領域後の先頭アドレス）
        const uint8_t* pAddressPos = pBuffer + offsetSize + compareSize;

        // Read領域後の先頭バッファアドレスから初期値かを比較
        for( size_t checkPos = 0; checkPos < canarySize; ++checkPos )
        {
            if( pAddressPos[checkPos] != InitializedValue )
            {
                return false;
            }
        }
    }

    return true;
}

}}}



namespace nn { namespace fs {

bool operator==(const ReadOption& lhs, const ReadOption& rhs) NN_NOEXCEPT
{
    return (lhs.reserved == rhs.reserved);
}

bool operator==(const WriteOption& lhs, const WriteOption& rhs) NN_NOEXCEPT
{
    return (lhs.flags == rhs.flags);
}

}}
