﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Assert.h>
#include <nn/os/os_Mutex.h>

#include <nnt/fsUtil/testFs_util_function_diag.h>
#include <nnt/fsUtil/testFs_util_function_stl.h>
#include <nnt/fsUtil/testFs_util_allocator.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nn_Allocator.h>

//#define DEBUG_MEMORY_ALLOCATION

#ifdef DEBUG_MEMORY_ALLOCATION
    #define ALLOCATION_LOG NN_SDK_LOG
#endif

namespace nnt { namespace fs { namespace util {

namespace {

class TestLibraryHeap
{
public:
    TestLibraryHeap() NN_NOEXCEPT
        : m_TestLibraryHeap(nullptr)
        , m_TestLibraryHeapHandle(nullptr)
        , m_TestLibraryHeapPointerOffset(0)
        , m_TestLibraryHeapMutex(true)
#ifdef DEBUG_MEMORY_ALLOCATION
        , m_AllocationCount(0)
#endif
    {
    }

    void Initialize() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_TestLibraryHeapMutex);
        if( m_TestLibraryHeapHandle == nullptr )
        {
            m_TestLibraryHeap = m_TestLibraryHeapStack;
            m_TestLibraryHeapHandle = nn::lmem::CreateExpHeap(m_TestLibraryHeap, TestLibraryHeapSize, nn::lmem::CreationOption_DebugFill);
            NN_ASSERT_NOT_NULL(m_TestLibraryHeapHandle);
        }
    }

    void Initialize(void* pHeap, size_t sizeHeap) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_TestLibraryHeapMutex);
        if( m_TestLibraryHeapHandle == nullptr )
        {
            m_TestLibraryHeap = reinterpret_cast<char*>(pHeap);
            m_TestLibraryHeapHandle = nn::lmem::CreateExpHeap(m_TestLibraryHeap, sizeHeap, nn::lmem::CreationOption_DebugFill);
            NN_ASSERT_NOT_NULL(m_TestLibraryHeapHandle);
        }
    }

    void ResetAllocationCount() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_TestLibraryHeapMutex);
        m_TotalAllocated = 0;
        m_TotalFreed = 0;
    }

    int64_t GetCurrentAllocatedSize() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_TestLibraryHeapMutex);
        return m_TotalAllocated - m_TotalFreed;
    }

    bool IsMemoryLeaked(const int64_t startAllocated) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_TestLibraryHeapMutex);

#ifdef DEBUG_MEMORY_ALLOCATION
        for( auto l : m_Allocated )
        {
            NN_LOG("undeallocated: %d 0x%p %d\n", l.size, l.p, l.count);
        }
#endif

        if( (m_TotalAllocated - startAllocated) != m_TotalFreed )
        {
            NN_LOG("Allocated %lld : Freed %lld\n", (m_TotalAllocated - startAllocated), m_TotalFreed);
            NN_LOG("detected memory leak\n");
            return true;
        }

        return false;
    }

    void SetAllocationPointerOffset(int offset) NN_NOEXCEPT
    {
        NN_ASSERT(offset >= 0);
        std::lock_guard<nn::os::Mutex> scopedLock(m_TestLibraryHeapMutex);
        m_TestLibraryHeapPointerOffset = offset;
    }

    void* Allocate(const size_t size) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_TestLibraryHeapMutex);
        if( m_TestLibraryHeapHandle == nullptr )
        {
            Initialize();
        }
        char* p = reinterpret_cast<char*>(nn::lmem::AllocateFromExpHeap(m_TestLibraryHeapHandle, size + m_TestLibraryHeapPointerOffset));

        if( p != nullptr )
        {
            p += m_TestLibraryHeapPointerOffset;

#ifdef DEBUG_MEMORY_ALLOCATION
            AllocationInfo newbie = { size + m_TestLibraryHeapPointerOffset, p, ++m_AllocationCount };
            m_Allocated.push_back(newbie);

            // ALLOCATION_LOG("alloc %d %p %d\n", size, p, g_AllocationCount);
            if( m_AllocationCount == 0 )
            {
                int a = 0;
                NN_UNUSED(a);
            }
#endif

            m_TotalAllocated += size + m_TestLibraryHeapPointerOffset;
        }
        return p;
    }

    void Deallocate(void* p, const size_t size) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_TestLibraryHeapMutex);
        NN_UNUSED(size);
        if( m_TestLibraryHeapHandle == nullptr )
        {
            Initialize();
        }

#ifdef DEBUG_MEMORY_ALLOCATION
        // ALLOCATION_LOG("free %d %p\n", size, p);
        auto it = std::find_if(m_Allocated.begin(), m_Allocated.end(),
            [this, &p](const AllocationInfo& info) NN_NOEXCEPT
            {
                return info.p == (reinterpret_cast<char*>(p) - m_TestLibraryHeapPointerOffset);
            });
        m_Allocated.erase(it);
#endif

        m_TotalFreed += size + m_TestLibraryHeapPointerOffset;
        nn::lmem::FreeToExpHeap(m_TestLibraryHeapHandle, reinterpret_cast<char*>(p) - m_TestLibraryHeapPointerOffset);
    }

private:
#ifdef DEBUG_MEMORY_ALLOCATION
    struct AllocationInfo
    {
        size_t size;
        void* p;
        int count;
    };
#endif // DEBUG_MEMORY_ALLOCATION

private:
    char m_TestLibraryHeapStack[TestLibraryHeapSize];
    char* m_TestLibraryHeap;
    nn::lmem::HeapHandle m_TestLibraryHeapHandle;
    int m_TestLibraryHeapPointerOffset;
    nn::os::Mutex m_TestLibraryHeapMutex;

    int64_t m_TotalAllocated = 0;
    int64_t m_TotalFreed = 0;

#ifdef DEBUG_MEMORY_ALLOCATION
    int m_AllocationCount;
    Vector<AllocationInfo> m_Allocated;
#endif // DEBUG_MEMORY_ALLOCATION
};

class GlobalHeapUtility
{
public:
    GlobalHeapUtility() NN_NOEXCEPT
        : m_IsStackTraceDumpOnGlobalNewDeleteCallEnabled(false)
        , m_IsGlobalNewDeleteCalled(false)
    {
    }

    bool IsGlobalNewDeleteCalled() NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> scopedLock(g_GlobalHeapUtilityMutex);
        return m_IsGlobalNewDeleteCalled;
    }

    void SetGlobalNewDeleteFlag() NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> scopedLock(g_GlobalHeapUtilityMutex);

        m_IsGlobalNewDeleteCalled = true;

        if( m_IsStackTraceDumpOnGlobalNewDeleteCallEnabled )
        {
            DumpModules();
            DumpStackTrace();
        }
    }

    void ResetGlobalNewDeleteFlag() NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> scopedLock(g_GlobalHeapUtilityMutex);
        m_IsGlobalNewDeleteCalled = false;
    }

    void SetStackTraceDumpOnGlobalNewDeleteCallEnabled(bool isEnabled) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> scopedLock(g_GlobalHeapUtilityMutex);
        m_IsStackTraceDumpOnGlobalNewDeleteCallEnabled = isEnabled;
    }

    bool IsStackTraceDumpOnGlobalNewDeleteCallEnabled() NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> scopedLock(g_GlobalHeapUtilityMutex);
        return m_IsStackTraceDumpOnGlobalNewDeleteCallEnabled;
    }

    std::unique_lock<StaticMutex> GetScopedLock() NN_NOEXCEPT
    {
        return std::unique_lock<StaticMutex>(g_GlobalHeapUtilityMutex);
    }

private:
    static StaticMutex g_GlobalHeapUtilityMutex;
    bool m_IsStackTraceDumpOnGlobalNewDeleteCallEnabled;
    bool m_IsGlobalNewDeleteCalled;
};

class TestLibraryAllocator : public nn::MemoryResource
{
private:
    void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
    {
        NN_UNUSED(alignment);
        return Allocate(bytes);
    }

    void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
    {
        NN_UNUSED(alignment);
        Deallocate(p, bytes);
    }

    virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT
    {
        return this == &other;
    }
};

// googletest のテスト構築時に global new/delete が使われる
// それまでに Mutex の初期化を終える必要があるので、NN_OS_MUTEX_INITIALIZER を使う
StaticMutex GlobalHeapUtility::g_GlobalHeapUtilityMutex = { NN_OS_MUTEX_INITIALIZER(true) };

GlobalHeapUtility g_GlobalHeapUtility;
TestLibraryHeap g_TestLibraryHeap;
TestLibraryAllocator g_DefaultTestLibraryAllocator;

} // namespace

void InitializeTestLibraryHeap() NN_NOEXCEPT
{
    g_TestLibraryHeap.Initialize();
}

void InitializeTestLibraryHeap(void* pHeap, const int heapSize) NN_NOEXCEPT
{
    g_TestLibraryHeap.Initialize(pHeap, heapSize);
}

bool IsGlobalNewDeleteCalled() NN_NOEXCEPT
{
    return g_GlobalHeapUtility.IsGlobalNewDeleteCalled();
}

void SetGlobalNewDeleteFlag() NN_NOEXCEPT
{
    g_GlobalHeapUtility.SetGlobalNewDeleteFlag();
}

void ResetGlobalNewDeleteFlag() NN_NOEXCEPT
{
    g_GlobalHeapUtility.ResetGlobalNewDeleteFlag();
}

void SetStackTraceDumpOnGlobalNewDeleteCallEnabled(bool isEnabled) NN_NOEXCEPT
{
    g_GlobalHeapUtility.SetStackTraceDumpOnGlobalNewDeleteCallEnabled(isEnabled);
}

bool IsStackTraceDumpOnGlobalNewDeleteCallEnabled() NN_NOEXCEPT
{
    return g_GlobalHeapUtility.IsStackTraceDumpOnGlobalNewDeleteCallEnabled();
}

std::unique_lock<StaticMutex> GetScopedLockForGlobalNewDeleteChecker() NN_NOEXCEPT
{
    return g_GlobalHeapUtility.GetScopedLock();
}

void ResetAllocateCount() NN_NOEXCEPT
{
    g_TestLibraryHeap.ResetAllocationCount();
}

int64_t GetStartAllocateSize() NN_NOEXCEPT
{
    return g_TestLibraryHeap.GetCurrentAllocatedSize();
}

bool CheckMemoryLeak() NN_NOEXCEPT
{
    return CheckMemoryLeak(0);
}

bool CheckMemoryLeak(const int64_t startAllocated) NN_NOEXCEPT
{
    return g_TestLibraryHeap.IsMemoryLeaked(startAllocated);
}

void SetAllocationPointerOffset(int offset) NN_NOEXCEPT
{
    g_TestLibraryHeap.SetAllocationPointerOffset(offset);
}

void* Allocate(const size_t size) NN_NOEXCEPT
{
    return g_TestLibraryHeap.Allocate(size);
}

void Deallocate(void* p, const size_t size) NN_NOEXCEPT
{
    g_TestLibraryHeap.Deallocate(p, size);
}

void* StdAllocate(const size_t size) NN_NOEXCEPT
{
    return malloc(size);
}

void StdDeallocate(void* p, const size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    free(p);
}

nn::MemoryResource* GetTestLibraryAllocator() NN_NOEXCEPT
{
    return &g_DefaultTestLibraryAllocator;
}

}}}
