﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto/crypto_HmacSha256Generator.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fssrv/detail/fssrv_FileSystemProxyServiceObject.h>
#include <nn/fssystem/fs_IMacGenerator.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystemDriver.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    #include <nn/spl/spl_Api.h>
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)

#include <nnt/fsUtil/testFs_util.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
namespace nn { namespace fs { namespace detail {
    void GenerateDeviceUniqueMacForSaveDataWin(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize);
} } }
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

namespace nnt { namespace fs { namespace util {

    namespace {
        class MacGenerator : public nn::fssystem::IMacGenerator
        {
        public:
            virtual void Generate(
                void* macBuffer,
                size_t macBufferSize,
                const void* targetBuffer,
                size_t targetBufferSize
            ) NN_NOEXCEPT NN_OVERRIDE
            {
                const char Key[] = "key";
                memset(macBuffer, 0, macBufferSize);
                nn::crypto::GenerateHmacSha256Mac(
                    macBuffer, macBufferSize, targetBuffer, targetBufferSize, Key, sizeof(Key)
                );
            }
        };

#if defined(NN_BUILD_CONFIG_OS_WIN)
        class DeviceUniqueMacGenerator : public nn::fssystem::IMacGenerator
        {
        public:
            virtual void Generate(
                void* macBuffer,
                size_t macBufferSize,
                const void* targetBuffer,
                size_t targetBufferSize
            ) NN_NOEXCEPT NN_OVERRIDE
            {
                nn::fs::detail::GenerateDeviceUniqueMacForSaveDataWin(
                    macBuffer,
                    macBufferSize,
                    targetBuffer,
                    targetBufferSize
                );
            }
        };
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

        MacGenerator g_MacGenerator;

#if defined(NN_BUILD_CONFIG_OS_WIN)
        DeviceUniqueMacGenerator g_DeviceMacGenerator;
#endif // defined(NN_BUILD_CONFIG_OS_WIN)
    }

    void InitializeSaveDataTestHelper() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        nn::spl::InitializeForCrypto();
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)
    }

    void FinalizeSaveDataTestHelper() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        nn::spl::Finalize();
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)
    }

    nn::fssystem::IMacGenerator* GetMacGenerator() NN_NOEXCEPT
    {
        return &g_MacGenerator;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::fssystem::IMacGenerator* GetDeviceMacGenerator() NN_NOEXCEPT
    {
        return &g_DeviceMacGenerator;
    }
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

    void GenerateRandomForSaveDataTest(void* pData, size_t size) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateRandomBytes(pData, size));
#else
        const auto now = nn::os::GetSystemTick().GetInt64Value();
        static const char Key[] = "HierarchicalIntegrityVerificationStorage::ControlArea";
        nn::crypto::GenerateHmacSha256Mac(
            pData, size,
            &now, sizeof(now),
            Key, sizeof(Key)
        );
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)
    }

    int SaveDataFileSystemMounter::g_BisMountCountUser = 0;
    int SaveDataFileSystemMounter::g_BisMountCountSystem = 0;

    SaveDataFileSystemMounter::SaveDataFileSystemMounter(nn::fssystem::IBufferManager* pBufferManager) NN_NOEXCEPT
        :
#if defined(NN_BUILD_CONFIG_OS_WIN)
          m_pBufferManager(pBufferManager),
#endif // defined(NN_BUILD_CONFIG_OS_WIN)
          m_pPrivateData(nnt::fs::util::AllocateBuffer(sizeof(SaveDataFileSystemMounterPrivateData)))
    {
        new (m_pPrivateData.get()) SaveDataFileSystemMounterPrivateData();
    }

    SaveDataFileSystemMounter::~SaveDataFileSystemMounter() NN_NOEXCEPT
    {
        if( GetPrivateData()->pFileSystem )
        {
            GetPrivateData()->pFileSystem->Finalize();
            GetPrivateData()->pFileSystem.reset();

            nn::fs::CloseFile(GetPrivateData()->file);

            UnmountBis(m_SpaceId);
        }

        GetPrivateData()->~SaveDataFileSystemMounterPrivateData();
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    void SaveDataFileSystemMounter::Mount(nn::fs::SaveDataSpaceId spaceId, nn::fs::SystemSaveDataId id) NN_NOEXCEPT
    {
        m_SpaceId = spaceId;
        NNT_FS_ASSERT_NO_FATAL_FAILURE(MountBis(m_SpaceId));

        char path[256];
        GetPath(path, sizeof(path), m_SpaceId, id);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&GetPrivateData()->file, path, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));

        int64_t sizeTotal;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&sizeTotal, GetPrivateData()->file));

        GetPrivateData()->pFileHandleStorage.reset(new nn::fs::FileHandleStorage(GetPrivateData()->file));

        GetPrivateData()->pFileSystem.reset(new nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver());
        NNT_ASSERT_RESULT_SUCCESS(
            GetPrivateData()->pFileSystem->Initialize(
                nn::fs::SubStorage(GetPrivateData()->pFileHandleStorage.get(), 0, sizeTotal),
                m_pBufferManager,
                nnt::fs::util::GetDeviceMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            )
        );
    }
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

    void SaveDataFileSystemMounter::ReadFile(size_t* outValue, nn::fs::SaveDataSpaceId spaceId, nn::fs::SystemSaveDataId id, int64_t offset, char* buffer, size_t size) NN_NOEXCEPT
    {
        NNT_FS_ASSERT_NO_FATAL_FAILURE(ClearSaveDataFileSystemCache());

        NNT_FS_ASSERT_NO_FATAL_FAILURE(MountBis(spaceId));
        NN_UTIL_SCOPE_EXIT
        {
            UnmountBis(spaceId);
        };

        char path[256];
        GetPath(path, sizeof(path), spaceId, id);

        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(outValue, file, offset, buffer, size));
    }

    void SaveDataFileSystemMounter::WriteFile(nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId id, int64_t offset, const char* buffer, size_t size) NN_NOEXCEPT
    {
        NNT_FS_ASSERT_NO_FATAL_FAILURE(ClearSaveDataFileSystemCache());

        NNT_FS_ASSERT_NO_FATAL_FAILURE(MountBis(spaceId));
        NN_UTIL_SCOPE_EXIT
        {
            UnmountBis(spaceId);
        };

        char path[256];
        GetPath(path, sizeof(path), spaceId, id);

        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, offset, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }


    void SaveDataFileSystemMounter::ClearSaveDataFileSystemCache() NN_NOEXCEPT
    {
        static const nn::fs::SaveDataId DummyId = 0x8000000000008000;
        (void)nn::fs::CreateSystemSaveData(DummyId, 32 * 1024, 32 * 1024, 0);
        NN_UTIL_SCOPE_EXIT
        {
            (void)nn::fs::DeleteSaveData(DummyId);
        };
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData("dummy", DummyId));
        nn::fs::Unmount("dummy");
    }

    void SaveDataFileSystemMounter::MountBis(nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT
    {
        if( spaceId == nn::fs::SaveDataSpaceId::System )
        {
            if( g_BisMountCountSystem == 0 )
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::System, nullptr));
            }
            ++g_BisMountCountSystem;
        }
        else
        {
            if( g_BisMountCountUser == 0 )
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));
            }
            ++g_BisMountCountUser;
        }
    }

    void SaveDataFileSystemMounter::UnmountBis(nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT
    {
        if( spaceId == nn::fs::SaveDataSpaceId::System )
        {
            --g_BisMountCountSystem;
            if( g_BisMountCountSystem == 0 )
            {
                nn::fs::Unmount(nn::fs::GetBisMountName(nn::fs::BisPartitionId::System));
            }
        }
        else
        {
            --g_BisMountCountUser;
            if( g_BisMountCountUser == 0 )
            {
                nn::fs::Unmount(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User));
            }
        }
    }

    void SaveDataFileSystemMounter::GetPath(char* path, size_t size, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId id) NN_NOEXCEPT
    {
        path[0] = 0;
        nn::util::SNPrintf(
            path, size, "%s:/save/%016llx",
            spaceId == nn::fs::SaveDataSpaceId::System
                ? nn::fs::GetBisMountName(nn::fs::BisPartitionId::System)
                : nn::fs::GetBisMountName(nn::fs::BisPartitionId::User),
            id
        );
    }

}}}
