﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Result.h>
#include <nnt/nntest.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nn/util/util_FormatString.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

/**
* @file
* @brief    ファイルシステム関数、エントリ構成観点を定義します。
*/
namespace {

    // 各階層のディレクトリ名形式定義
    const char* const TestDirectoryFormat   = "%d";

    // ルートディレクトリに作成するテストファイル数定義
    const int RootCreateFileCount = 4096;

    // 階層毎のエントリ数定義（１ファイル＋１ディレクトリ）
    const int DepthEntryCount     = 2;

    // ディレクトリの深さ定義
    const int DirectoryDepth      = 12;//64;

    // 取得する最大エントリ数定義
    const int EntryMax            = RootCreateFileCount + 1;

    // 各ディレクトリ内のファイルリライト用バッファ
    char g_EntryReadBuffer[1 * 1024];
    char g_EntryExpectBuffer[1 * 1024];

    // エントリの読み込み用領域
    DirectoryEntry g_ReadEntry[EntryMax];

    typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size);

    nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
    {
        return read != ReadFunction2;
    }

    const ReadFunction ReadFunctions[] =
    {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };
}

namespace nnt { namespace fs { namespace api {
    void LoadEntryCompositionRoTests()
    {
        return;
    }

    /**
     * @brief   EntryCompositionRo で利用するテストフィクスチャです。
     */
    class EntryCompositionRo : public GetFileSystemTestFixture
    {
    protected:
        /**
         * @brief テスト開始時に毎回呼び出される関数です。
         */
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();

            // ルートディレクトリ設定します
            m_RootPath = GetTestRootPath();
            m_RootPath.append("/EntryCompositionRo/0/");
        }

        /**
         * @brief テスト終了時に毎回呼び出される関数です。
         */
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            m_RootPath = "";

            GetFileSystemTestFixture::TearDown();
        }

        /**
         * @brief 指定エントリタイプが存在するか確認する関数です
         *   @param[in]  name                   確認するパスを指定します
         *   @param[in]  entryType              確認するファイル／ディレクトリのエントリタイプを指定します
         *
         *   @return     指定エントリタイプの結果が返ります。
         *   @retval     true                   指定のエントリタイプで存在した場合に返ります。
         *   @retval     false                  指定のエントリタイプで存在しない場合に返ります。
         */
        bool IsSpecifiedEntryTypeExist(
            const String& name,
            DirectoryEntryType entryType
        ) NN_NOEXCEPT
        {
            DirectoryEntryType entry;

            // 指定パスのエントリタイプを取得します
            NN_ABORT_UNLESS_RESULT_SUCCESS(GetFs().GetEntryType(&entry, name.c_str()));

            // 取得したエントリタイプの識別を判定します
            return ((entry & entryType) == entryType);
        }

        /**
         * @brief 取得エントリからファイル操作する関数です。
         *  @pre
         *   •親ディレクトリとして pEntryPath を渡し、事前に読み込まれた pReadEntry のファイル情報が示すファイル群があること
         *    渡されたreadEntryCountのエントリ数分 読み込みでベリファイできることをテストします
         *
         *   @param[in]  pEntryPath             エントリオープンしたパスを指定します
         *   @param[in]  pReadEntry             事前に読み込まれたエントリ構造体ポインタを指定します
         *   @param[in]  readEntryCount         事前に読み込まれたエントリ数を指定します
         *   @param[in]  compareFileCount       事前に読み込まれたエントリから操作されるファイル数と比較するファイル数を指定します
         */
        void TestEntryFileRead(
            const char* pEntryPath,
            const DirectoryEntry* pReadEntry,
            int64_t readEntryCount,
            int compareFileCount
        ) NN_NOEXCEPT
        {
            std::unique_ptr<ITestFile> file;
            String testDirectory            = pEntryPath;
            OpenMode fileMode               = OpenMode_Read;
            int fileCount                   = 0;

            // 取得したファイルのエントリ情報でバイナリ比較します
            for( int64_t i = 0; i < readEntryCount; ++i )
            {
                testDirectory = pEntryPath;
                testDirectory.append(pReadEntry[i].name);
                if( !IsSpecifiedEntryTypeExist(testDirectory.c_str(), DirectoryEntryType_File) )
                {
                    continue;
                }

                ++fileCount;

                for( auto read : ReadFunctions )
                {
                    if( IsReadOverload(read) && !GetFsAttribute()->isReadOverloadsSupported )
                    {
                        continue;
                    }

                    NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, testDirectory.c_str(), fileMode));
                    util::InvalidateVariable(g_EntryReadBuffer, sizeof(g_EntryReadBuffer));

                    NNT_EXPECT_RESULT_SUCCESS(
                        read(
                            file.get(),
                            0,
                            g_EntryReadBuffer,
                            sizeof(g_EntryReadBuffer)
                        )
                    );

                    file.reset(nullptr);

                    NNT_FS_UTIL_EXPECT_MEMCMPEQ(
                        g_EntryExpectBuffer,
                        g_EntryReadBuffer,
                        sizeof(g_EntryExpectBuffer)
                    );
                }
            }

            // 操作したファイル数と期待するファイル数を比較します
            EXPECT_EQ(compareFileCount, fileCount);
        }

        void MakeExpectImage(char* pBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            for( size_t cnt = 0; cnt < bufferSize; ++cnt )
            {
                pBuffer[cnt] = cnt & 0xff;
            }
        }

    protected:
        String m_RootPath;
    };

    //! @brief ルート階層のエントリー操作と4096個ファイル操作ができ、結果比較が一致すること
    TEST_F(EntryCompositionRo, EntryRootFilesHeavy)
    {
        String testDirectory = m_RootPath;
        std::unique_ptr<ITestDirectory> directory;
        int64_t entryCount;

        // 期待値データのパターンデータを作成
        MakeExpectImage(&g_EntryExpectBuffer[0], sizeof(g_EntryExpectBuffer));

        // ファイルのエントリ操作と結果比較します
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenDirectory(&directory, testDirectory.c_str(), OpenDirectoryMode_File)
        );

        NNT_EXPECT_RESULT_SUCCESS(directory->Read(&entryCount, g_ReadEntry, EntryMax));

        EXPECT_EQ(RootCreateFileCount, entryCount);

        // ファイル操作を関数内で実行します
        TestEntryFileRead(testDirectory.c_str(), g_ReadEntry, entryCount, RootCreateFileCount);
    }

    //! @brief 指定されたディレクトリの深さまでエントリー操作とファイル操作ができ、結果比較が一致すること
    TEST_F(EntryCompositionRo, EntryDepth)
    {
        char nameBuffer[nnt::fs::util::NameMaxLength];
        String testDirectory = m_RootPath;
        int64_t entryCount;

        // 期待値データのパターンデータを作成
        MakeExpectImage(&g_EntryExpectBuffer[0], sizeof(g_EntryExpectBuffer));

        // 深さ1～指定階層数のエントリを開き、ディレクトリ、ファイルのエントリ操作と結果比較します
        for( int depthCount = 1; depthCount <= DirectoryDepth; ++depthCount )
        {
            nn::util::SNPrintf(nameBuffer, sizeof(nameBuffer), TestDirectoryFormat, depthCount % 10);
            testDirectory.append(nameBuffer);
            testDirectory.append("/");

            std::unique_ptr<ITestDirectory> directory;
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenDirectory(&directory, testDirectory.c_str(), OpenDirectoryMode_All)
            );

            NNT_EXPECT_RESULT_SUCCESS(directory->Read(&entryCount, g_ReadEntry, EntryMax));

            for( int i = 0; i < entryCount; ++i )
            {
                nn::util::SNPrintf(
                    nameBuffer,
                    sizeof(nameBuffer),
                    "%s%s",
                    testDirectory.c_str(),
                    g_ReadEntry[i].name
                );
                // エントリタイプがファイルもしくはディレクトリか
                if( IsSpecifiedEntryTypeExist(nameBuffer, DirectoryEntryType_File) )
                {
                    // 現在のループが最下層の場合、１ファイルのみか比較します
                    if( depthCount == DirectoryDepth )
                    {
                        EXPECT_EQ(1, entryCount);
                    }
                    TestEntryFileRead(testDirectory.c_str(), g_ReadEntry, entryCount, 1);
                }
                else
                {
                    EXPECT_EQ(DepthEntryCount, entryCount);
                    nn::util::SNPrintf(
                        nameBuffer,
                        sizeof(nameBuffer),
                        TestDirectoryFormat,
                        (depthCount + 1) % 10
                    );
                    EXPECT_TRUE(std::strcmp(nameBuffer, g_ReadEntry[i].name) == 0);
                }
            }
        }
    }

    //! @brief 指定したエントリタイプのみを開けること
    TEST_F(EntryCompositionRo, EntryType)
    {
        char nameBuffer[nnt::fs::util::NameMaxLength];
        String testDirectory = m_RootPath;

        // 期待値データのパターンデータを作成
        MakeExpectImage(&g_EntryExpectBuffer[0], sizeof(g_EntryExpectBuffer));

        // 深さ1～指定階層数のエントリを開き、ディレクトリ、ファイルのエントリ操作と結果比較します
        for( int depthCount = 1; depthCount <= DirectoryDepth; ++depthCount )
        {
            nn::util::SNPrintf(nameBuffer, sizeof(nameBuffer), TestDirectoryFormat, depthCount % 10);
            testDirectory.append(nameBuffer);
            testDirectory.append("/");

            {
                // ファイルのエントリのみを取得
                std::unique_ptr<ITestDirectory> directory;
                NNT_ASSERT_RESULT_SUCCESS(
                    GetFs().OpenDirectory(&directory, testDirectory.c_str(), OpenDirectoryMode_File)
                );

                int64_t entryCount;
                NNT_EXPECT_RESULT_SUCCESS(directory->Read(&entryCount, g_ReadEntry, EntryMax));
                EXPECT_EQ(1, entryCount);

                // エントリタイプがファイルか
                EXPECT_EQ(DirectoryEntryType_File, g_ReadEntry[0].directoryEntryType);

                // GetEntryType で取得できるエントリタイプがファイルか
                nn::util::SNPrintf(
                    nameBuffer,
                    sizeof(nameBuffer),
                    "%s%s",
                    testDirectory.c_str(),
                    g_ReadEntry[0].name
                );
                EXPECT_TRUE(IsSpecifiedEntryTypeExist(nameBuffer, DirectoryEntryType_File));
            }

            {
                // ディレクトリのエントリのみを取得
                std::unique_ptr<ITestDirectory> directory;
                NNT_ASSERT_RESULT_SUCCESS(
                    GetFs().OpenDirectory(
                        &directory, testDirectory.c_str(), OpenDirectoryMode_Directory
                    )
                );

                int64_t entryCount;
                NNT_EXPECT_RESULT_SUCCESS(directory->Read(&entryCount, g_ReadEntry, EntryMax));
                if( depthCount == DirectoryDepth )
                {
                    // 現在のループが最下層の場合、ディレクトリはありません。
                    EXPECT_EQ(0, entryCount);
                }
                else
                {
                    EXPECT_EQ(1, entryCount);

                    // エントリタイプがディレクトリか
                    EXPECT_EQ(DirectoryEntryType_Directory, g_ReadEntry[0].directoryEntryType);

                    // GetEntryType で取得できるエントリタイプがディレクトリか
                    nn::util::SNPrintf(
                        nameBuffer,
                        sizeof(nameBuffer),
                        "%s%s",
                        testDirectory.c_str(),
                        g_ReadEntry[0].name
                    );
                    EXPECT_TRUE(
                        IsSpecifiedEntryTypeExist(nameBuffer, DirectoryEntryType_Directory)
                    );
                }
            }
        }
    }
}}}
