﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/fsApi/testFs_Api.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace nnt { namespace fs { namespace api {
    void LoadPostConditionTests() NN_NOEXCEPT
    {
        return;
    }

    namespace {
        typedef nn::Result (*CommitFunction)(ITestFileSystem*);
        typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, size_t expectedSize);

        nn::Result Commit(ITestFileSystem* pFileSystem) NN_NOEXCEPT
        {
            NN_RESULT_DO(pFileSystem->Commit());
            NN_RESULT_SUCCESS;
        }

        nn::Result CommitSaveData(ITestFileSystem* pFileSystem) NN_NOEXCEPT
        {
            NN_RESULT_DO(pFileSystem->CommitSaveData());
            NN_RESULT_SUCCESS;
        }

        nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, size_t expectedSize) NN_NOEXCEPT
        {
            NN_UNUSED(size);

            NN_RESULT_DO(pTestFile->Read(offset, buffer, expectedSize, nn::fs::ReadOption()));
            NN_RESULT_SUCCESS;
        }

        nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, size_t expectedSize) NN_NOEXCEPT
        {
            NN_UNUSED(size);

            NN_RESULT_DO(pTestFile->Read(offset, buffer, expectedSize));
            NN_RESULT_SUCCESS;
        }

        nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, size_t expectedSize) NN_NOEXCEPT
        {
            size_t readSize;
            util::InvalidateVariable(&readSize);
            NN_RESULT_DO(pTestFile->Read(&readSize, offset, buffer, size, nn::fs::ReadOption()));
            EXPECT_EQ(expectedSize, readSize);
            NN_RESULT_SUCCESS;
        }

        nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, size_t expectedSize) NN_NOEXCEPT
        {
            size_t readSize;
            util::InvalidateVariable(&readSize);
            NN_RESULT_DO(pTestFile->Read(&readSize, offset, buffer, size));
            EXPECT_EQ(expectedSize, readSize);
            NN_RESULT_SUCCESS;
        }

        bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
        {
            return read != ReadFunction2;
        }

        const ReadFunction ReadFunctions[] = {
            ReadFunction0,
            ReadFunction1,
            ReadFunction2,
            ReadFunction3
        };

        const CommitFunction CommitFunctions[] = {
            Commit,
            CommitSaveData
        };

        const struct FunctionCombination
        {
            ReadFunction read;
            CommitFunction commit;
        } FunctionCombinations[] = {
            { ReadFunction0, Commit },
            { ReadFunction1, Commit },
            { ReadFunction2, CommitSaveData },
            { ReadFunction3, CommitSaveData }
        };
    }

    class PostConditionBaseFixture : public CleanupFileSystemTestFixture
    {
    protected:
        static const int MaxFileSize = 32;

    private:
        static const int FileSize = 31;
        NN_STATIC_ASSERT(FileSize <= MaxFileSize);

    protected:
        int GetAlignedFileSize() NN_NOEXCEPT
        {
            return nn::util::align_up(FileSize, GetFsAttribute()->fileSizeAlignment);
        }
    };

    class PostCondition : public PostConditionBaseFixture
    {
    };

    class PostConditionFile : public PostConditionBaseFixture
    {
    public:
        String basisFileName;
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            PostConditionBaseFixture::SetUp();
            basisFileName = GetTestRootPath().append("/test.file");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(basisFileName.c_str(), GetAlignedFileSize()));
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::fs::DirectoryEntryType type;
            auto result = GetFs().GetEntryType(&type, basisFileName.c_str());
            if( !nn::fs::ResultPathNotFound::Includes(result) )
            {
                NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(basisFileName.c_str()));
            }
            PostConditionBaseFixture::TearDown();
        }
    };

    class PostConditionFileWithoutCommit : public PostConditionFile, public ::testing::WithParamInterface<ReadFunction>
    {
    };

    class PostConditionFileWithCommit : public PostConditionFile, public ::testing::WithParamInterface<FunctionCombination>
    {
    };

    class PostConditionDirectory : public PostConditionBaseFixture
    {
    public:
        String basisDirName;
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            PostConditionBaseFixture::SetUp();
            basisDirName = GetTestRootPath().append("/test");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(basisDirName.c_str()));
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::fs::DirectoryEntryType type;
            auto result = GetFs().GetEntryType(&type, basisDirName.c_str());
            if( !nn::fs::ResultPathNotFound::Includes(result) )
            {
                NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(basisDirName.c_str()));
            }
            PostConditionBaseFixture::TearDown();
        }
    };

    class PostConditionDirectoryWithCommit : public PostConditionDirectory, public ::testing::WithParamInterface<CommitFunction>
    {
    };

    class PostConditionCommit : public PostConditionBaseFixture, public ::testing::WithParamInterface<FunctionCombination>
    {
    public:
        PostConditionCommit() NN_NOEXCEPT
            : m_WorkBuffer(nullptr, nnt::fs::util::DeleterBuffer)
        {
        }

    public:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            PostConditionBaseFixture::SetUp();
            m_WorkBuffer = nnt::fs::util::AllocateBuffer(BufferSize * 2);
            ASSERT_NE(nullptr, m_WorkBuffer.get());
        }

    protected:
        // pFs の内容を特定のディレクトリ構成にする
        // A:
        // file1
        // dir1 / subdir
        //      / subfile
        //
        // B:
        // file2 (modified)
        // dir2 / subdir
        //      / subfile
        //
        static const int64_t File1Size = 2 * 1024 * 1024 + 1;
        static const int64_t File1ContentOffset = 0;
        static const int64_t File2ContentOffset = 128;

    protected:
        void SetUpStateA() NN_NOEXCEPT
        {
            // ディレクトリ・ファイル、ディレクトリ下のディレクトリ・ファイルを作成
            NNT_EXPECT_RESULT_SUCCESS(CreateTestDirectory("/dir1"));
            NNT_EXPECT_RESULT_SUCCESS(CreateTestDirectory("/dir1/subdir"));
            CreateTestFile("/file1", File1Size);
            CreateTestFile("/dir1/subfile", 32);
        }

        void SetUpStateBFromA() NN_NOEXCEPT
        {
            // データ更新
            UpdateFile("/file1", 16, 16, File2ContentOffset);

            // リネーム
            NNT_EXPECT_RESULT_SUCCESS(RenameTestFile("/file1", "/file2"));
            NNT_EXPECT_RESULT_SUCCESS(RenameTestDirectory("/dir1", "/dir2"));
        }

        // pFs の内容が特定のディレクトリ構成であることをチェックする
        void CheckStateA(ReadFunction read) NN_NOEXCEPT
        {
            nn::fs::DirectoryEntryType type;
            NNT_EXPECT_RESULT_SUCCESS(GetTestEntryType(&type, "/file1"));
            NNT_EXPECT_RESULT_SUCCESS(GetTestEntryType(&type, "/dir1"));
            NNT_EXPECT_RESULT_SUCCESS(GetTestEntryType(&type, "/dir1/subdir"));
            NNT_EXPECT_RESULT_SUCCESS(GetTestEntryType(&type, "/dir1/subfile"));
            CheckFileContent("/file1", 0, File1Size, File1ContentOffset, read);
            CheckFileContent("/dir1/subfile", 0, 32, 0, read);

            // B でないこと
            NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetTestEntryType(&type, "/file2"));
            NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetTestEntryType(&type, "/dir2"));
        }

        void CheckStateB(ReadFunction read) NN_NOEXCEPT
        {
            nn::fs::DirectoryEntryType type;
            NNT_EXPECT_RESULT_SUCCESS(GetTestEntryType(&type, "/file2"));
            NNT_EXPECT_RESULT_SUCCESS(GetTestEntryType(&type, "/dir2"));
            NNT_EXPECT_RESULT_SUCCESS(GetTestEntryType(&type, "/dir2/subdir"));
            NNT_EXPECT_RESULT_SUCCESS(GetTestEntryType(&type, "/dir2/subfile"));
            CheckFileContent("/file2", 0, 16, File1ContentOffset, read);
            CheckFileContent("/file2", 16, 16, File2ContentOffset, read);
            CheckFileContent("/dir2/subfile", 0, 32, 0, read);

            // A でないこと
            NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetTestEntryType(&type, "/file1"));
            NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetTestEntryType(&type, "/dir1"));
        }

    private:
        static const int BufferSize = 1024 * 1024 * 4;

    private:
        nn::Result GetTestEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
        {
            auto absolutePath = GetTestRootPath() + path;
            NN_RESULT_DO(GetFs().GetEntryType(outValue, absolutePath.c_str()));
            NN_RESULT_SUCCESS;
        }

        nn::Result CreateTestDirectory(const char* path) NN_NOEXCEPT
        {
            auto absolutePath = GetTestRootPath() + path;
            NN_RESULT_DO(GetFs().CreateDirectory(absolutePath.c_str()));
            NN_RESULT_SUCCESS;
        }

        void CreateTestFile(const char* path, int64_t size) NN_NOEXCEPT
        {
            auto absolutePath = GetTestRootPath() + path;

            ASSERT_LE(size, BufferSize);

            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(absolutePath.c_str(), size));

            std::unique_ptr<ITestFile> pFile;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, absolutePath.c_str(), nn::fs::OpenMode_Write));
            nnt::fs::util::FillBufferWith32BitCount(m_WorkBuffer.get(), static_cast<size_t>(size), 0);
            NNT_ASSERT_RESULT_SUCCESS(pFile->Write(0, m_WorkBuffer.get(), static_cast<size_t>(size), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        nn::Result RenameTestFile(const char* path1, const char* path2) NN_NOEXCEPT
        {
            auto absolutePath1 = GetTestRootPath() + path1;
            auto absolutePath2 = GetTestRootPath() + path2;
            NN_RESULT_DO(GetFs().RenameFile(absolutePath1.c_str(), absolutePath2.c_str()));
            NN_RESULT_SUCCESS;
        }

        nn::Result RenameTestDirectory(const char* path1, const char* path2) NN_NOEXCEPT
        {
            auto absolutePath1 = GetTestRootPath() + path1;
            auto absolutePath2 = GetTestRootPath() + path2;
            NN_RESULT_DO(GetFs().RenameDirectory(absolutePath1.c_str(), absolutePath2.c_str()));
            NN_RESULT_SUCCESS;
        }

        void UpdateFile(const char* path, int64_t offset, int64_t size, int64_t offsetCount) NN_NOEXCEPT
        {
            auto absolutePath = GetTestRootPath() + path;

            ASSERT_LE(size, BufferSize);

            std::unique_ptr<ITestFile> pFile;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, absolutePath.c_str(), nn::fs::OpenMode_Write));
            nnt::fs::util::FillBufferWith32BitCount(m_WorkBuffer.get(), static_cast<size_t>(size), offsetCount);
            NNT_ASSERT_RESULT_SUCCESS(pFile->Write(offset, m_WorkBuffer.get(), static_cast<size_t>(size), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        void CheckFileContent(const char* path, int64_t offset, int64_t size, int64_t offsetCount, ReadFunction read) NN_NOEXCEPT
        {
            auto absolutePath = GetTestRootPath() + path;

            auto bufferExpect = m_WorkBuffer.get();
            auto bufferActual = m_WorkBuffer.get() + BufferSize;
            ASSERT_LE(size, BufferSize);

            FillBufferWith32BitCount(bufferExpect, static_cast<size_t>(size), offsetCount);

            std::unique_ptr<ITestFile> pFile;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, absolutePath.c_str(), nn::fs::OpenMode_Read));
            nnt::fs::util::FillBufferWith32BitCount(bufferActual, static_cast<size_t>(size), 0);
            NNT_ASSERT_RESULT_SUCCESS(read(pFile.get(), offset, bufferActual, static_cast<size_t>(size), static_cast<size_t>(size)));

            NNT_FS_UTIL_EXPECT_MEMCMPEQ(bufferExpect, bufferActual, static_cast<size_t>(size));
        }

    private:
        std::unique_ptr<char, decltype(&nnt::fs::util::DeleterBuffer)> m_WorkBuffer;
    };

    NN_DEFINE_STATIC_CONSTANT(const int PostConditionCommit::BufferSize);

    //!< @brief ファイルを作成・オープン・削除できること
    //          削除したファイルのオープンに失敗すること
    TEST_F(PostCondition, CreateFile_DeleteFile)
    {
        String fileName = GetTestRootPath().append("/test.file");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), GetAlignedFileSize()));
        std::unique_ptr<ITestFile> file;
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));
        file.reset(nullptr);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));
    }

    //!< @brief ディレクトリを作成・オープン・削除できること
    //          削除したディレクトリのオープンに失敗すること
    TEST_F(PostCondition, CreateDirectory_DeleteDirectory)
    {
        String dirName = GetTestRootPath().append("/test");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(dirName.c_str()));
        std::unique_ptr<ITestDirectory> dir;
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&dir, dirName.c_str(), OpenDirectoryMode_File));
        dir.reset(nullptr);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(dirName.c_str()));
        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenDirectory(&dir, dirName.c_str(), OpenDirectoryMode_File));
    }

    //!< @brief ディレクトリを再帰的に削除できること
    //          再帰的に削除したディレクトリ直下のファイル、再帰的に削除したディレクトリのオープンに失敗すること
    TEST_F(PostCondition, DeleteDirectoryRecursively)
    {
        String dirName = GetTestRootPath().append("/test");
        String fileName = dirName;
        fileName.append("/test.file");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(dirName.c_str()));
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), GetAlignedFileSize()));
        std::unique_ptr<ITestFile> file;
        std::unique_ptr<ITestDirectory> dir;
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectoryRecursively(dirName.c_str()));
        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));
        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenDirectory(&dir, dirName.c_str(), OpenDirectoryMode_File));
    }

    //!< @brief ファイルをリネームでき、リネーム後のファイルのオープン・削除ができること
    //          リネーム後に、リネーム前のファイルパスに対してのオープンに失敗すること
    TEST_F(PostCondition, RenameFile)
    {
        String fileName = GetTestRootPath().append("/test.file");
        String renameFileName = GetTestRootPath().append("/rename.file");
        std::unique_ptr<ITestFile> file;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), GetAlignedFileSize()));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().RenameFile(fileName.c_str(), renameFileName.c_str()));
        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, renameFileName.c_str(), OpenMode_Read));
        file.reset(nullptr);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(renameFileName.c_str()));
    }

    //!< @brief ディレクトリをリネームでき、リネーム後のディレクトリのオープン・削除ができること
    //          リネーム後に、リネーム前のディレクトリパスに対してのオープンに失敗すること
    TEST_F(PostCondition, RenameDirectory)
    {
        String dirName = GetTestRootPath().append("/test");
        String renameDirName = GetTestRootPath().append("/rename");
        std::unique_ptr<ITestDirectory> dir;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(dirName.c_str()));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().RenameDirectory(dirName.c_str(), renameDirName.c_str()));
        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenDirectory(&dir, dirName.c_str(), OpenDirectoryMode_File));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&dir, renameDirName.c_str(), OpenDirectoryMode_File));
        dir.reset(nullptr);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(renameDirName.c_str()));
    }

    //!< @brief ファイルを含むディレクトリをリネームでき、リネーム後のディレクトリ内のファイルのオープン・削除ができること
    //          リネーム後に、リネーム前のディレクトリ内のファイルパスに対してのオープンに失敗すること
    TEST_F(PostCondition, RenameDirectoryWithFile)
    {
        String dirName = GetTestRootPath().append("/test");
        String fileNameInDir = dirName + "/test.txt";
        String dirName2 = dirName + "/test";
        String fileNameInDir2 = dirName2 + "/test.txt";

        String renameDirName = GetTestRootPath().append("/rename");
        String fileNameInRenamedDir = renameDirName + "/test.txt";
        String renameDirName2 = renameDirName + "/test";
        String fileNameInRenamedDir2 = renameDirName2 + "/test.txt";

        std::unique_ptr<ITestFile> file;

        // 準備
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(dirName.c_str()));
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileNameInDir.c_str(), 0));
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(dirName2.c_str()));
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileNameInDir2.c_str(), 0));

        // リネーム
        NNT_EXPECT_RESULT_SUCCESS(GetFs().RenameDirectory(dirName.c_str(), renameDirName.c_str()));

        // ファイルアクセス
        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenFile(&file, fileNameInDir.c_str(), OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileNameInRenamedDir.c_str(), OpenMode_Read));
        file.reset(nullptr);

        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenFile(&file, fileNameInDir2.c_str(), OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileNameInRenamedDir2.c_str(), OpenMode_Read));
        file.reset(nullptr);

        // 削除
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectoryRecursively(renameDirName.c_str()));
    }

    //!< @brief ファイルに対して、GetEntryType を実行し、エントリータイプを取得する
    //          DirectoryEntryType_File であること
    TEST_F(PostConditionFile, GetEntryType)
    {
        DirectoryEntryType deType;
        util::InvalidateVariable(&deType);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().GetEntryType(&deType, basisFileName.c_str()));
        EXPECT_EQ(DirectoryEntryType_File, deType);
    }

    //!< @brief ディレクトリに対して、GetEntryType を実行し、エントリータイプを取得する
    //          DirectoryEntryType_Directory であること
    TEST_F(PostConditionDirectory, GetEntryType)
    {
        DirectoryEntryType deType;
        util::InvalidateVariable(&deType);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().GetEntryType(&deType, basisDirName.c_str()));
        EXPECT_EQ(DirectoryEntryType_Directory, deType);
    }

    //!< @brief OpenFile を実行し、Write、Flush、Read を行う
    //          Write バッファと Read バッファの内容が一致することを確認する
    TEST_P(PostConditionFileWithoutCommit, OpenFile_WriteFile_Flush_ReadFile)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        std::unique_ptr<ITestFile> file;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, basisFileName.c_str(), static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));
        char writeBuffer[MaxFileSize];
        FillBufferWithRandomValue(writeBuffer, GetAlignedFileSize());
        char readBuffer[MaxFileSize];
        util::InvalidateVariable(readBuffer, GetAlignedFileSize());
        NNT_EXPECT_RESULT_SUCCESS(file->Write(0, writeBuffer, GetAlignedFileSize(), WriteOption()));
        NNT_EXPECT_RESULT_SUCCESS(file->Flush());
        size_t readSize;
        util::InvalidateVariable(&readSize);
        NNT_EXPECT_RESULT_SUCCESS(read(file.get(), 0, readBuffer, GetAlignedFileSize(), static_cast<size_t>(GetAlignedFileSize())));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, GetAlignedFileSize());
        file.reset(nullptr);
    }

    //!< @brief OpenFile 実行後、GetSize、SetSize、GetSize を行う
    //          設定サイズと取得サイズが一致することを確認する
    TEST_F(PostConditionFile, OpenFile_SetSize_GetSize)
    {
        int64_t newSize = GetAlignedFileSize() * 2;
        std::unique_ptr<ITestFile> file;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, basisFileName.c_str(), static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));
        int64_t actualSize;
        util::InvalidateVariable(&actualSize);
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&actualSize));
        EXPECT_EQ(static_cast<int64_t>(GetAlignedFileSize()), actualSize);
        NNT_EXPECT_RESULT_SUCCESS(file->SetSize(newSize));
        util::InvalidateVariable(&actualSize);
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&actualSize));
        EXPECT_EQ(newSize, actualSize);
        file.reset(nullptr);
    }

    //!< @brief OpenDirectoryMode_File / Directory モードの組み合わせで OpenDirectory 実行後、
    //          GetEntryCount、ReadDirectory を行い、設定サイズと取得サイズが一致することを確認する
    TEST_F(PostConditionDirectory, OpenDirectory_ReadDirectory_GetDirectoryEntryCount)
    {
        String fileName = basisDirName;
        fileName.append("/test.file");
        String dirName = basisDirName;
        dirName.append("/test");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), GetAlignedFileSize()));
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(dirName.c_str()));

        DirectoryEntry de[2];
        util::InvalidateVariable(de, 2);

        {
            std::unique_ptr<ITestDirectory> dir;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&dir, basisDirName.c_str(), OpenDirectoryMode_File));
            int64_t entryCount;
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->GetEntryCount(&entryCount));
            EXPECT_EQ(1, entryCount);
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(1, entryCount);
            EXPECT_TRUE(strncmp(de[0].name, "test.file", sizeof(de[0].name)) == 0);
            EXPECT_EQ(DirectoryEntryType_File, de[0].directoryEntryType);
            EXPECT_EQ(GetAlignedFileSize(), de[0].fileSize);
            // すべて読み出した後は 0 が返る
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(0, entryCount);
        }
        {
            std::unique_ptr<ITestDirectory> dir;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&dir, basisDirName.c_str(), OpenDirectoryMode_Directory));
            int64_t entryCount;
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->GetEntryCount(&entryCount));
            EXPECT_EQ(1, entryCount);
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(1, entryCount);
            EXPECT_TRUE(strncmp(de[0].name, "test", sizeof(de[0].name)) == 0);
            EXPECT_EQ(DirectoryEntryType_Directory, de[0].directoryEntryType);
            EXPECT_EQ(0, de[0].fileSize);
            // すべて読み出した後は 0 が返る
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(0, entryCount);
        }
        {
            std::unique_ptr<ITestDirectory> dir;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&dir, basisDirName.c_str(), static_cast<OpenDirectoryMode>(OpenDirectoryMode_File | OpenDirectoryMode_Directory)));
            int64_t entryCount;
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->GetEntryCount(&entryCount));
            EXPECT_EQ(2, entryCount);
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 2));
            EXPECT_EQ(2, entryCount);
            for (int i = 0; i < 2; i++)
            {
                if (de[i].directoryEntryType == DirectoryEntryType_File)
                {
                    EXPECT_TRUE(strncmp(de[i].name, "test.file", sizeof(de[i].name)) == 0);
                    EXPECT_EQ(GetAlignedFileSize(), de[i].fileSize);
                }
                else
                {
                    EXPECT_TRUE(strncmp(de[i].name, "test", sizeof(de[i].name)) == 0);
                    EXPECT_EQ(0, de[i].fileSize);
                }
            }
            // すべて読み出した後は 0 が返る
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(0, entryCount);
        }
        {
            std::unique_ptr<ITestDirectory> dir;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&dir, basisDirName.c_str(), static_cast<OpenDirectoryMode>(OpenDirectoryMode_File | OpenDirectoryMode_Directory | OpenDirectoryModePrivate_NotRequireFileSize)));
            int64_t entryCount;
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->GetEntryCount(&entryCount));
            EXPECT_EQ(2, entryCount);
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 2));
            EXPECT_EQ(2, entryCount);
            for (int i = 0; i < 2; i++)
            {
                if (de[i].directoryEntryType == DirectoryEntryType_File)
                {
                    EXPECT_TRUE(strncmp(de[i].name, "test.file", sizeof(de[i].name)) == 0);
                }
                else
                {
                    EXPECT_TRUE(strncmp(de[i].name, "test", sizeof(de[i].name)) == 0);
                }
            }
            // すべて読み出した後は 0 が返る
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(0, entryCount);
        }
        {
            std::unique_ptr<ITestDirectory> dir;
            NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, GetFs().OpenDirectory(&dir, basisDirName.c_str(), static_cast<OpenDirectoryMode>(OpenDirectoryModePrivate_NotRequireFileSize)));
        }
        {
            std::unique_ptr<ITestDirectory> dir;
            NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, GetFs().OpenDirectory(&dir, basisDirName.c_str(), static_cast<OpenDirectoryMode>(~(OpenDirectoryMode_File | OpenDirectoryMode_Directory | OpenDirectoryModePrivate_NotRequireFileSize))));
        }
        NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
        NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteDirectory(dirName.c_str()));
    } // NOLINT(impl/function_size)

    //!< @brief ファイルを作成後、Unmount した場合の挙動を確認する
    TEST_P(PostConditionFileWithCommit, Remount)
    {
        auto read = GetParam().read;
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        auto commit = GetParam().commit;

        nn::fs::DirectoryEntryType type;

        // ファイル作成だけして Remount
        RemountFs();

        if( GetFsAttribute()->isSaveFileSystem )
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFs().GetEntryType(&type, this->basisFileName.c_str()));

            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(this->basisFileName.c_str(), 0));
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));

            RemountFs();
        }

        NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&type, this->basisFileName.c_str()));
        ASSERT_EQ(nn::fs::DirectoryEntryType_File, type);

        // ファイルに書き込みだけして Remount
        char writeBuffer[MaxFileSize];
        FillBufferWithRandomValue(writeBuffer, GetAlignedFileSize());

        {
            std::unique_ptr<ITestFile> pFile;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, this->basisFileName.c_str(), static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)));
            NNT_ASSERT_RESULT_SUCCESS(pFile->Write(0, writeBuffer, sizeof(writeBuffer), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        RemountFs();

        char readBuffer[MaxFileSize];

        if( GetFsAttribute()->isSaveFileSystem )
        {
            {
                std::unique_ptr<ITestFile> pFile;
                NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, this->basisFileName.c_str(), nn::fs::OpenMode_Read));

                util::InvalidateVariable(readBuffer, GetAlignedFileSize());

                NNT_EXPECT_RESULT_SUCCESS(read(pFile.get(), 0, readBuffer, GetAlignedFileSize(), 0));
            }

            {
                std::unique_ptr<ITestFile> pFile;
                NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, this->basisFileName.c_str(), static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)));
                NNT_ASSERT_RESULT_SUCCESS(pFile->Write(0, writeBuffer, sizeof(writeBuffer), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
            }
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));

            RemountFs();
        }

        {
            std::unique_ptr<ITestFile> pFile;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, this->basisFileName.c_str(), nn::fs::OpenMode_Read));

            util::InvalidateVariable(readBuffer, GetAlignedFileSize());

            NNT_EXPECT_RESULT_SUCCESS(read(pFile.get(), 0, readBuffer, GetAlignedFileSize(), static_cast<size_t>(GetAlignedFileSize())));
        }

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, GetAlignedFileSize());

        // リネームだけして Remount
        String renameFileName = GetTestRootPath().append("/rename.file");
        NNT_EXPECT_RESULT_SUCCESS(GetFs().RenameFile(this->basisFileName.c_str(), renameFileName.c_str()));

        RemountFs();

        if( GetFsAttribute()->isSaveFileSystem )
        {
            NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&type, this->basisFileName.c_str()));
            ASSERT_EQ(nn::fs::DirectoryEntryType_File, type);
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFs().GetEntryType(&type, renameFileName.c_str()));

            NNT_ASSERT_RESULT_SUCCESS(GetFs().RenameFile(this->basisFileName.c_str(), renameFileName.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));

            RemountFs();
        }

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFs().GetEntryType(&type, this->basisFileName.c_str()));
        NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&type, renameFileName.c_str()));
        ASSERT_EQ(nn::fs::DirectoryEntryType_File, type);

        // 削除だけして Remount
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(renameFileName.c_str()));

        RemountFs();

        if( GetFsAttribute()->isSaveFileSystem )
        {
            NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&type, renameFileName.c_str()));
            ASSERT_EQ(nn::fs::DirectoryEntryType_File, type);

            NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteFile(renameFileName.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));

            RemountFs();
        }

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFs().GetEntryType(&type, renameFileName.c_str()));
    }

    //!< @brief ディレクトリを作成後、Unmount した場合の挙動を確認する
    TEST_P(PostConditionDirectoryWithCommit, Remount)
    {
        auto commit = GetParam();

        nn::fs::DirectoryEntryType type;

        RemountFs();

        if( GetFsAttribute()->isSaveFileSystem )
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFs().GetEntryType(&type, this->basisDirName.c_str()));

            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(this->basisDirName.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));

            RemountFs();
        }

        NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&type, this->basisDirName.c_str()));
        ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, type);

        // リネームだけして Remount
        String renameDirectoryName = GetTestRootPath().append("/rename_dir");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().RenameDirectory(this->basisDirName.c_str(), renameDirectoryName.c_str()));

        RemountFs();

        if( GetFsAttribute()->isSaveFileSystem )
        {
            NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&type, this->basisDirName.c_str()));
            ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, type);
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFs().GetEntryType(&type, renameDirectoryName.c_str()));

            NNT_ASSERT_RESULT_SUCCESS(GetFs().RenameDirectory(this->basisDirName.c_str(), renameDirectoryName.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));

            RemountFs();
        }

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFs().GetEntryType(&type, this->basisDirName.c_str()));
        NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&type, renameDirectoryName.c_str()));
        ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, type);

        // 削除だけして Remount
        nnt::fs::util::String dirName = renameDirectoryName + "/subdirectory";
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(dirName.c_str()));

        if( GetFsAttribute()->isSaveFileSystem )
        {
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));
        }

        RemountFs();

        NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteDirectory(dirName.c_str()));

        RemountFs();

        if( GetFsAttribute()->isSaveFileSystem )
        {
            NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&type, dirName.c_str()));
            ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, type);

            NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteDirectory(dirName.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));

            RemountFs();
        }

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFs().GetEntryType(&type, dirName.c_str()));

        // 再帰的削除だけして Remount
        NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteDirectoryRecursively(renameDirectoryName.c_str()));

        RemountFs();

        if( GetFsAttribute()->isSaveFileSystem )
        {
            NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&type, renameDirectoryName.c_str()));
            ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, type);

            NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteDirectoryRecursively(renameDirectoryName.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));

            RemountFs();
        }

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFs().GetEntryType(&type, renameDirectoryName.c_str()));
    }

    //!< @brief  未コミットの各種変更内容が巻き戻ること (A -> B -> A)
    TEST_P(PostConditionCommit, RollbackUncommittedModification)
    {
        NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSaveFileSystem);

        auto read = GetParam().read;
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        auto commit = GetParam().commit;

        {
            // A
            SetUpStateA();

            // commit A
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));

            // A -> B
            SetUpStateBFromA();
            CheckStateB(read);

            // commit せずアンマウント相当
        }

        {
            // 再マウント
            RemountFs();

            // ベリファイ
            CheckStateA(read);
        }
    }

    //!< @brief  コミットによって各種変更内容が残ること (A -> B)
    TEST_P(PostConditionCommit, RemainCommittedModification)
    {
        NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSaveFileSystem);

        auto read = GetParam().read;
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        auto commit = GetParam().commit;

        {
            // A
            SetUpStateA();

            // commit A
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));

            // A -> B
            SetUpStateBFromA();
            CheckStateB(read);

            // commit してアンマウント
            NNT_ASSERT_RESULT_SUCCESS(commit(&GetFs()));
        }

        {
            // 再マウント
            RemountFs();

            // ベリファイ
            CheckStateB(read);
        }
    }

    INSTANTIATE_TEST_CASE_P(WithReadOverloads,
        PostConditionFileWithoutCommit,
        ::testing::ValuesIn(ReadFunctions));

    INSTANTIATE_TEST_CASE_P(WithReadCommitOverloads,
        PostConditionFileWithCommit,
        ::testing::ValuesIn(FunctionCombinations));

    INSTANTIATE_TEST_CASE_P(WithCommitOverloads,
        PostConditionDirectoryWithCommit,
        ::testing::ValuesIn(CommitFunctions));

    INSTANTIATE_TEST_CASE_P(WithReadCommitOverloads,
        PostConditionCommit,
        ::testing::ValuesIn(FunctionCombinations));
}}}
