﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/fsApi/testFs_Api.h>
#include <nn/util/util_FormatString.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

/**
* @file
* @brief    ファイルシステム関数、エントリ構成観点を定義します。
*/
namespace {

    // ディレクトリ内ファイルのエントリ名定義
//    const char* TestEntryFile         = "test.file";

    // 各階層のディレクトリ名形式定義
    const char* TestDirectoryFormat   = "%d";

    // ルートディレクトリに作成するテストファイル数定義
    const int32_t RootCreateFileCount  = 4096;

    // 階層毎のエントリ数定義（１ファイル＋１ディレクトリ）
    const int32_t DepthEntryCount     = 2;

    // ディレクトリの深さ定義
    const int32_t DirectoryDepth      = 64;

    // 取得する最大エントリ数定義
    const int32_t EntryMax            = RootCreateFileCount + 1;

    // 各ディレクトリ内のファイルリライト用バッファ
    uint8_t g_EntryWriteBuffer[1 * 1024];

    // エントリの読み込み用領域
    DirectoryEntry g_ReadEntry[EntryMax];
}

namespace nnt { namespace fs { namespace api {
    void LoadEntryCompositionTests() NN_NOEXCEPT
    {
        return;
    }

    /**
     * @brief   EntryComposition で利用するテストフィクスチャです。
     */
    class EntryComposition : public CleanupFileSystemTestFixture
    {
    protected:
        /**
         * @brief テスト開始時に毎回呼び出される関数です。
         */
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();

            // ルートディレクトリ設定・作成します
            m_RootPath = GetTestRootPath();
            m_RootPath.append("/00/");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(m_RootPath.c_str()));
        }

        /**
         * @brief テスト終了時に毎回呼び出される関数です。
         */
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            // ディレクトリ・ファイルを削除します
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectoryRecursively(m_RootPath.c_str()));
            m_RootPath = "";

            CleanupFileSystemTestFixture::TearDown();
        }

        /**
         * @brief 取得エントリからファイル操作する関数です。
         *  @pre
         *   •親ディレクトリとして pEntryPath を渡し、事前に読み込まれた pReadEntry のファイル情報が示すファイル群があること
         *    渡されたreadEntryCountのエントリ数分 書き込み／読み込みでベリファイできることをテストします
         *
         *   @param[in]  pEntryPath             エントリオープンしたパスを指定します
         *   @param[in]  pReadEntry             事前に読み込まれたエントリ構造体ポインタを指定します
         *   @param[in]  readEntryCount         事前に読み込まれたエントリ数を指定します
         *   @param[in]  compareFileCount       事前に読み込まれたエントリから操作されるファイル数と比較するファイル数を指定します
         */
        void TestEntryFileWriteRead(
            const char* pEntryPath,
            const DirectoryEntry* pReadEntry,
            const int64_t readEntryCount,
            const int32_t compareFileCount
        ) NN_NOEXCEPT
        {
            std::unique_ptr<ITestFile> file;
            String testDirectory = pEntryPath;
            const OpenMode fileMode
                = static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend);
            int32_t fileCount = 0;

            // 取得したファイルのエントリ情報でリライト操作の確認とバイナリ比較します
            for(int64_t i = 0; i < readEntryCount; ++i)
            {
                testDirectory = pEntryPath;
                testDirectory.append(pReadEntry[i].name);
                if (IsEntryExist(&GetFs(), testDirectory.c_str(), DirectoryEntryType_File) != true)
                {
                    continue;
                }

                ++fileCount;

                NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, testDirectory.c_str(), fileMode));

                ConfirmWriteRead(
                    file.get(),
                    g_EntryWriteBuffer, static_cast<size_t>(sizeof(g_EntryWriteBuffer)));

                file.reset(nullptr);

                // テスト実行後、エントリ確認用にファイル削除、0バイトファイルを作成します（Fatシステムでのストレージ領域不足対策）
                NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(testDirectory.c_str()));
                NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(testDirectory.c_str(), 0));
            }

            // 操作したファイル数と期待するファイル数を比較します
            EXPECT_EQ(compareFileCount, fileCount);
        }

        const String& GetRootPath() NN_NOEXCEPT
        {
            return m_RootPath;
        }

    private:
        String m_RootPath;
    };

    //!< @brief ルート階層のエントリー操作と4096個ファイル操作ができ、結果比較が一致すること
    TEST_F(EntryComposition, EntryRootFilesHeavy)
    {
        const String& testDirectory = GetRootPath();
        std::unique_ptr<ITestDirectory> directory;
        int64_t entryCount;

        // テストファイルを事前に作成します
        NNT_ASSERT_RESULT_SUCCESS(
            CreateTestFile(&GetFs(), nullptr, testDirectory.c_str(), RootCreateFileCount, 0));

        // 深さ0のファイルのエントリ操作と結果比較します
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenDirectory(&directory,
            testDirectory.c_str(), static_cast<OpenDirectoryMode>(OpenDirectoryMode_File)));

        NNT_EXPECT_RESULT_SUCCESS(directory->Read(&entryCount, g_ReadEntry, EntryMax));

        EXPECT_EQ(RootCreateFileCount, entryCount);

        // ファイル操作を関数内で実行します
        TestEntryFileWriteRead(testDirectory.c_str(), g_ReadEntry, entryCount, RootCreateFileCount);
    }

    //!< @brief 指定されたディレクトリの深さまでエントリー操作とファイル操作ができ、結果比較が一致すること
    TEST_F(EntryComposition, EntryDepth)
    {
        auto directoryName = AllocateBuffer(NameMaxLength);
        String testDirectory = GetRootPath();
        DirectoryEntryType entryType = DirectoryEntryType::DirectoryEntryType_File;
        int64_t entryCount;

        // テストディレクトリとテストファイルを事前に作成します
        NNT_ASSERT_RESULT_SUCCESS(
            CreateTestDirectory(&GetFs(),
                nullptr,
                testDirectory.c_str(),
                DirectoryDepth,
                DIRTYPE_NEST,
                true,
                0));

        // 深さ1～指定階層数のエントリを開き、ディレクトリ、ファイルのエントリ操作と結果比較します
        for (int32_t depthCount = 0; depthCount < DirectoryDepth; ++depthCount)
        {
            nn::util::SNPrintf(
                directoryName.get(), NameMaxLength, TestDirectoryFormat, depthCount % 10);
            testDirectory.append(directoryName.get());
            testDirectory.append("/");
            NNT_FS_SCOPED_TRACE(
                "depth count: %d, entry path: %s\n", depthCount, testDirectory.c_str());

            // 実行時間削減のため、最初と最後のみテストを行います
            const int TestCount = 3;
            if (TestCount < depthCount && depthCount < DirectoryDepth - TestCount)
            {
                continue;
            }

            std::unique_ptr<ITestDirectory> directory;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenDirectory(&directory,
                testDirectory.c_str(), static_cast<OpenDirectoryMode>(OpenDirectoryMode_All)));

            NNT_EXPECT_RESULT_SUCCESS(directory->Read(&entryCount, g_ReadEntry, EntryMax));

            for (int32_t i = 0; i < entryCount; ++i)
            {
                NNT_FS_SCOPED_TRACE("loop count: %d\n", i);
                nn::util::SNPrintf(
                    directoryName.get(),
                    NameMaxLength,
                    "%s%s",
                    testDirectory.c_str(), g_ReadEntry[i].name);
                // エントリタイプがファイルもしくはディレクトリか
                if (IsEntryExist(&GetFs(), directoryName.get(), entryType) == true)
                {
                    // 現在のループが最下層の場合、１ファイルのみか比較します
                    if (depthCount + 1 == DirectoryDepth)
                    {
                        EXPECT_EQ(1, entryCount);
                    }
                    TestEntryFileWriteRead(testDirectory.c_str(), g_ReadEntry, entryCount, 1);
                }
                else
                {
                    EXPECT_EQ(DepthEntryCount, entryCount);
                    nn::util::SNPrintf(
                        directoryName.get(),
                        NameMaxLength,
                        TestDirectoryFormat,
                        (depthCount + 1) % 10);
                    EXPECT_TRUE(std::strcmp(directoryName.get(), g_ReadEntry[i].name) == 0);
                }
            }
        }
    }

    //!< @brief 空のディレクトリの読み込みができること
    TEST_F(EntryComposition, Empty)
    {
        std::unique_ptr<ITestDirectory> directory;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenDirectory(&directory,
            GetRootPath().c_str(), static_cast<OpenDirectoryMode>(OpenDirectoryMode_All)));

        int64_t entryCount = -1;
        NNT_EXPECT_RESULT_SUCCESS(directory->GetEntryCount(&entryCount));
        EXPECT_EQ(0, entryCount);

        int64_t readCount = -1;
        DirectoryEntry entry[1] = {};
        NNT_EXPECT_RESULT_SUCCESS(directory->Read(&readCount, entry, 0));
        EXPECT_EQ(0, readCount);

        directory.reset();
    }

    //! @brief 指定したエントリタイプのみを開けること
    TEST_F(EntryComposition, EntryType)
    {
        char nameBuffer[nnt::fs::util::NameMaxLength];
        String testDirectory = GetRootPath();

        // テストディレクトリとテストファイルを事前に作成します
        NNT_ASSERT_RESULT_SUCCESS(
            CreateTestDirectory(&GetFs(),
                nullptr,
                testDirectory.c_str(),
                DirectoryDepth,
                DIRTYPE_NEST,
                true,
                0));

        // 深さ1～指定階層数のエントリを開き、ディレクトリ、ファイルのエントリ操作と結果比較します
        for (int32_t depthCount = 0; depthCount < DirectoryDepth; ++depthCount)
        {
            nn::util::SNPrintf(
                nameBuffer, sizeof(nameBuffer), TestDirectoryFormat, depthCount % 10);
            testDirectory.append(nameBuffer);
            testDirectory.append("/");

            // 実行時間削減のため、最初と最後のみテストを行います
            const int TestCount = 3;
            if( TestCount < depthCount && depthCount < DirectoryDepth - TestCount )
            {
                continue;
            }

            {
                // ファイルのエントリのみを取得
                std::unique_ptr<ITestDirectory> directory;
                NNT_ASSERT_RESULT_SUCCESS(
                    GetFs().OpenDirectory(
                        &directory, testDirectory.c_str(), OpenDirectoryMode_File));

                int64_t entryCount;
                NNT_EXPECT_RESULT_SUCCESS(directory->Read(&entryCount, g_ReadEntry, EntryMax));
                EXPECT_EQ(1, entryCount);

                // エントリタイプがファイルか
                EXPECT_EQ(DirectoryEntryType_File, g_ReadEntry[0].directoryEntryType);

                // GetEntryType で取得できるエントリタイプがファイルか
                nn::util::SNPrintf(
                    nameBuffer,
                    sizeof(nameBuffer),
                    "%s%s",
                    testDirectory.c_str(),
                    g_ReadEntry[0].name
                );
                EXPECT_TRUE(IsEntryExist(&GetFs(), nameBuffer, DirectoryEntryType_File));
            }

            {
                // ディレクトリのエントリのみを取得
                std::unique_ptr<ITestDirectory> directory;
                NNT_ASSERT_RESULT_SUCCESS(
                    GetFs().OpenDirectory(
                        &directory, testDirectory.c_str(), OpenDirectoryMode_Directory
                    )
                );

                int64_t entryCount;
                NNT_EXPECT_RESULT_SUCCESS(directory->Read(&entryCount, g_ReadEntry, EntryMax));
                if( depthCount == DirectoryDepth - 1 )
                {
                    // 現在のループが最下層の場合、ディレクトリはありません。
                    EXPECT_EQ(0, entryCount);
                }
                else
                {
                    EXPECT_EQ(1, entryCount);

                    // エントリタイプがディレクトリか
                    EXPECT_EQ(DirectoryEntryType_Directory, g_ReadEntry[0].directoryEntryType);

                    // GetEntryType で取得できるエントリタイプがディレクトリか
                    nn::util::SNPrintf(
                        nameBuffer,
                        sizeof(nameBuffer),
                        "%s%s",
                        testDirectory.c_str(),
                        g_ReadEntry[0].name
                    );
                    EXPECT_TRUE(IsEntryExist(&GetFs(), nameBuffer, DirectoryEntryType_Directory));
                }
            }
        }
    }
}}}
