﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"
#include "../../Common/testFriends_PadUtil.h"

#include <nn/hid.h>
#include <vector>

namespace
{
    using MenuFunction = void (*)(size_t cursor);

    struct MenuInfo
    {
        char name[64];
        MenuFunction decideFunction;
        MenuFunction backFunction;
        std::vector<MenuInfo> children;
        size_t cursor;
    };
}

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax];
    int s_UserCount = 0;

    nn::account::UserHandle s_UserHandles[nn::account::UserCountMax] = {};
    bool s_IsUserOpened[NN_ARRAY_SIZE(s_UserHandles)] = {};

    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax];
    bool s_IsAccountAvailable[NN_ARRAY_SIZE(s_AccountIds)] = {};

    int s_CurrentUserIndex = -1;

    int s_PresenceCounter = 0;

    MenuInfo s_TopMenu = {"Top"};
    MenuInfo* s_pCurrentMenu = &s_TopMenu;

    std::vector<MenuInfo*> s_MenuHistory;
}

namespace
{
    void ShowMenu() NN_NOEXCEPT
    {
        size_t length = s_pCurrentMenu->children.size();

        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("%s\n", s_pCurrentMenu->name);
        NN_LOG("--------------------------------------------------------------------------------\n");

        for (size_t i = 0; i < length; i++)
        {
            NN_LOG("%c %s\n", i == s_pCurrentMenu->cursor ? '>' : ' ', s_pCurrentMenu->children[i].name);
        }

        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("A: Decide B: Back\n");
        NN_LOG("--------------------------------------------------------------------------------\n");
    }

    void MoveToChild(size_t cursor) NN_NOEXCEPT
    {
        NN_ASSERT_LESS(cursor, s_pCurrentMenu->children.size());

        s_MenuHistory.push_back(s_pCurrentMenu);
        s_pCurrentMenu = &s_pCurrentMenu->children[cursor];

        ShowMenu();
    }

    void MoveToParent(size_t cursor) NN_NOEXCEPT
    {
        NN_UNUSED(cursor);

        if (s_MenuHistory.empty())
        {
            return;
        }

        s_pCurrentMenu = s_MenuHistory.back();
        s_MenuHistory.pop_back();

        ShowMenu();
    }

    void SelectUser(size_t cursor) NN_NOEXCEPT
    {
        s_CurrentUserIndex = static_cast<int>(cursor);

        MoveToChild(cursor);
    };

    void OpenUser(size_t cursor) NN_NOEXCEPT
    {
        NN_UNUSED(cursor);

        if (s_IsUserOpened[s_CurrentUserIndex])
        {
            return;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&s_UserHandles[s_CurrentUserIndex], s_Users[s_CurrentUserIndex]));
        s_IsUserOpened[s_CurrentUserIndex] = true;

        nn::util::SNPrintf(s_TopMenu.children[s_CurrentUserIndex].name, sizeof (s_TopMenu.children[s_CurrentUserIndex].name),
            "User[%d] *", s_CurrentUserIndex);

        ShowMenu();
    }

    void CloseUser(size_t cursor) NN_NOEXCEPT
    {
        NN_UNUSED(cursor);

        if (!s_IsUserOpened[s_CurrentUserIndex])
        {
            return;
        }

        nn::account::CloseUser(s_UserHandles[s_CurrentUserIndex]);
        s_IsUserOpened[s_CurrentUserIndex] = false;

        nn::util::SNPrintf(s_TopMenu.children[s_CurrentUserIndex].name, sizeof (s_TopMenu.children[s_CurrentUserIndex].name),
            "User[%d]", s_CurrentUserIndex);

        ShowMenu();
    }

    void UpdateUserPresence(size_t cursor) NN_NOEXCEPT
    {
        NN_UNUSED(cursor);

        nn::friends::UserPresence presence;

        NN_ABORT_UNLESS_RESULT_SUCCESS(presence.Initialize(s_Users[s_CurrentUserIndex]));

        char value[32] = {};
        nn::util::SNPrintf(value, sizeof (value), "presence_%05d", s_PresenceCounter++);

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

        presence.SetAppValue("key", value);

NN_PRAGMA_POP_WARNINGS

        NN_ABORT_UNLESS_RESULT_SUCCESS(presence.Commit());

        NN_LOG("Update user presence. (value = %s)\n", value);
    }

    void ShowUserSetting(size_t cursor) NN_NOEXCEPT
    {
        NN_UNUSED(cursor);

        nn::friends::UserSetting setting;

        NN_ABORT_UNLESS_RESULT_SUCCESS(setting.Initialize(s_Users[s_CurrentUserIndex]));

        nnt::friends::Dump(setting);
    }

    void ClearPlayLog(size_t cursor) NN_NOEXCEPT
    {
        NN_UNUSED(cursor);

        nn::friends::UserSetting setting;

        NN_ABORT_UNLESS_RESULT_SUCCESS(setting.Initialize(s_Users[s_CurrentUserIndex]));

        NN_LOG("nn::friends::UserSetting::ClearPlayLog() ...\n");

        nnt::friends::ConnectNetwork();

        nn::friends::AsyncContext context;
        NN_ABORT_UNLESS_RESULT_SUCCESS(setting.ClearPlayLog(&context));

        nn::os::SystemEvent completionEvent;
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        nn::Result result = context.GetResult();

        if (result.IsSuccess() || nn::friends::ResultNetworkServiceAccountNotLinked::Includes(result))
        {
            NN_LOG("nn::friends::UserSetting::ClearPlayLog() done!\n");
        }
        else
        {
            NN_LOG("nn::friends::UserSetting::ClearPlayLog() failed. code = %03d-%04d\n",
                result.GetModule(), result.GetDescription());
        }
    }

    void SendFriendRequests(size_t cursor) NN_NOEXCEPT
    {
        if (!s_IsAccountAvailable[s_CurrentUserIndex])
        {
            NN_LOG("Network service account is not available.\n");
            return;
        }

        for (int i = 0; i < s_UserCount; i++)
        {
            if (i == s_CurrentUserIndex)
            {
                continue;
            }
            if (!s_IsAccountAvailable[i])
            {
                continue;
            }

            NN_LOG("nn::friends::SendFriendRequest(%d -> %d) ...\n", s_CurrentUserIndex, i);

            nnt::friends::ConnectNetwork();

            nn::friends::AsyncContext context;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[s_CurrentUserIndex],
                s_AccountIds[i], nn::friends::RequestType_FriendCode));

            nn::os::SystemEvent completionEvent;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            nn::Result result = context.GetResult();

            if (result.IsSuccess())
            {
                NN_LOG("nn::friends::SendFriendRequest(%d -> %d) done!\n", s_CurrentUserIndex, i);
            }
            else
            {
                NN_LOG("nn::friends::SendFriendRequest(%d -> %d) failed. code = %03d-%04d\n",
                    s_CurrentUserIndex, i, result.GetModule(), result.GetDescription());
            }
        }
    }

    void DeleteFriendAll(size_t cursor) NN_NOEXCEPT
    {
        if (!s_IsAccountAvailable[s_CurrentUserIndex])
        {
            NN_LOG("Network service account is not available.\n");
            return;
        }

        static nn::account::NetworkServiceAccountId s_AccountIds[nn::friends::FriendCountMax];
        int count;
        nn::friends::FriendFilter filter = {};

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetFriendList(&count, s_AccountIds, s_Users[s_CurrentUserIndex],
            0, NN_ARRAY_SIZE(s_AccountIds), filter));

        for (int i = 0; i < count; i++)
        {
            NN_LOG("nn::friends::DeleteFriend(%d) ...\n", i);

            nnt::friends::ConnectNetwork();

            nn::friends::AsyncContext context;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::DeleteFriend(&context, s_Users[s_CurrentUserIndex], s_AccountIds[i]));

            nn::os::SystemEvent completionEvent;
            NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            nn::Result result = context.GetResult();

            if (result.IsSuccess())
            {
                NN_LOG("nn::friends::DeleteFriend(%d) done!\n", i);
            }
            else
            {
                NN_LOG("nn::friends::DeleteFriend(%d) failed. code = %03d-%04d\n",
                    i, result.GetModule(), result.GetDescription());
            }
        }
    }

    void ShowFriendPresence(size_t cursor) NN_NOEXCEPT
    {
        NN_UNUSED(cursor);

        static nn::friends::Friend s_Friends[nn::friends::FriendCountMax];
        int count;
        nn::friends::FriendFilter filter = {};

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetFriendList(&count, s_Friends, s_Users[s_CurrentUserIndex],
            0, NN_ARRAY_SIZE(s_Friends), filter));

        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("FriendPresence\n");
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("Count = %d\n", count);

        for (int i = 0; i < count; i++)
        {
NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

            NN_LOG("[%3d] %016llx: key = %s\n",
                i, s_Friends[i].GetAccountId().id, s_Friends[i].GetPresence().GetAppValue("key"));

NN_PRAGMA_POP_WARNINGS
        }

        NN_LOG("--------------------------------------------------------------------------------\n");
    }
}

TEST(FriendsManualAccountTests, Initialize)
{
    nn::account::Initialize();
    nn::nifm::Initialize();

    nnt::friends::InitializePad();

    nnt::friends::LoadAccounts(&s_UserCount, s_Users, NN_ARRAY_SIZE(s_Users));
    NN_ABORT_UNLESS_GREATER(s_UserCount, 0);

    for (int i = 0; i < s_UserCount; i++)
    {
        nn::account::UserHandle handle = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, s_Users[i]));

        NN_UTIL_SCOPE_EXIT
        {
            nn::account::CloseUser(handle);
        };

        nn::account::NetworkServiceAccountId accountId = {};

        if (nn::account::GetNetworkServiceAccountId(&accountId, handle).IsSuccess())
        {
            s_AccountIds[i] = accountId;
            s_IsAccountAvailable[i] = true;
        }
        else
        {
            s_IsAccountAvailable[i] = false;
        }
    }

    nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);

    for (int i = 0; i < s_UserCount; i++)
    {
        MenuInfo menu = {"", SelectUser, nullptr};
        nn::util::SNPrintf(menu.name, sizeof (menu.name), "User[%d]", i);

        menu.children.push_back({"[UserStatus]",            nullptr,            MoveToParent});
        menu.children.push_back({"    Open",                OpenUser,           MoveToParent});
        menu.children.push_back({"    Close",               CloseUser,          MoveToParent});
        menu.children.push_back({"[UserPresence]",          nullptr,            MoveToParent});
        menu.children.push_back({"    Update",              UpdateUserPresence, MoveToParent});
        menu.children.push_back({"[UserSetting]",           nullptr,            MoveToParent});
        menu.children.push_back({"    Show",                ShowUserSetting,    MoveToParent});
        menu.children.push_back({"[PlayLog]",               nullptr,            MoveToParent});
        menu.children.push_back({"    Clear",               ClearPlayLog,       MoveToParent});
        menu.children.push_back({"[Friend]",                nullptr,            MoveToParent});
        menu.children.push_back({"    SendFriendRequests",  SendFriendRequests, MoveToParent});
        menu.children.push_back({"    DeleteAll",           DeleteFriendAll,    MoveToParent});
        menu.children.push_back({"[FriendPresence]",        nullptr,            MoveToParent});
        menu.children.push_back({"    Show",                ShowFriendPresence, MoveToParent});

        s_TopMenu.children.push_back(menu);
    }

    ShowMenu();
}

TEST(FriendsManualAccountTests, MainLoop)
{
    nn::hid::NpadButtonSet prevButtons = {};

    while (NN_STATIC_CONDITION(true))
    {
        nn::hid::NpadButtonSet buttons = {};

        nnt::friends::GetNpadButtons(&buttons);

        nn::hid::NpadButtonSet trigger = (~prevButtons) & buttons;
        prevButtons = buttons;

        if (trigger.Test<nn::hid::NpadButton::A>())
        {
            if (s_pCurrentMenu->children[s_pCurrentMenu->cursor].decideFunction)
            {
                s_pCurrentMenu->children[s_pCurrentMenu->cursor].decideFunction(s_pCurrentMenu->cursor);
            }
        }
        else if (trigger.Test<nn::hid::NpadButton::B>())
        {
            if (s_pCurrentMenu->children[s_pCurrentMenu->cursor].backFunction)
            {
                s_pCurrentMenu->children[s_pCurrentMenu->cursor].backFunction(s_pCurrentMenu->cursor);
            }
        }
        else if (trigger.Test<nn::hid::NpadButton::Up>())
        {
            if (s_pCurrentMenu->cursor > 0)
            {
                s_pCurrentMenu->cursor--;
                ShowMenu();
            }
        }
        else if (trigger.Test<nn::hid::NpadButton::Down>())
        {
            if (s_pCurrentMenu->cursor < s_pCurrentMenu->children.size() - 1)
            {
                s_pCurrentMenu->cursor++;
                ShowMenu();
            }
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}
