﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

class FriendsCompatibility : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsCompatibility, Do)
{
    // ネットワークの不調によってアカウントが作成されないことがままあるため、先にネットワークエラーをチェックできるようにする。
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::account::Uid users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId accountIds[nn::account::UserCountMax] = {};
    int userCount = 0;

    nnt::friends::LoadAccounts(&userCount, users, accountIds, NN_ARRAY_SIZE(users));
    ASSERT_GE(userCount, 3);

    nn::account::UserHandle handles[2] = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handles[0], users[0]));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handles[1], users[1]));

    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(handles[0]);
        nn::account::CloseUser(handles[1]);
    };

    {
        static nn::account::NetworkServiceAccountId s_Friends[nn::friends::FriendCountMax] = {};
        int count = 0;
        nn::friends::FriendFilter filter = {};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, s_Friends, users[0], 0, NN_ARRAY_SIZE(s_Friends), filter));

        ASSERT_EQ(count, 1);

        EXPECT_EQ(s_Friends[0], accountIds[1]);
    }
    {
        static nn::friends::Friend s_Friends[nn::friends::FriendCountMax] = {};
        int count = 0;
        nn::friends::FriendFilter filter = {};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, s_Friends, users[0], 0, NN_ARRAY_SIZE(s_Friends), filter));

        ASSERT_EQ(count, 1);
        ASSERT_TRUE(s_Friends[0].IsValid());

        EXPECT_EQ(s_Friends[0].GetAccountId(), accountIds[1]);

        static nn::Bit8 s_Buffer[nn::friends::ProfileImageSizeMax];
        size_t size;

        nn::Result result = s_Friends[0].GetProfileImage(&size, s_Buffer, sizeof (s_Buffer));

        if (result.IsFailure())
        {
            NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultProfileImageCacheNotFound, result);
        }
    }
    {
        nn::friends::Friend f;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UpdateFriendInfo(&f, users[0], &accountIds[1], 1));
        ASSERT_TRUE(f.IsValid());
    }
    {
        static nn::account::NetworkServiceAccountId s_BlockedUsers[nn::friends::FriendCountMax] = {};
        int count = 0;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetBlockedUserList(&count, s_BlockedUsers, users[0], 0, NN_ARRAY_SIZE(s_BlockedUsers)));

        ASSERT_EQ(count, 1);

        EXPECT_EQ(s_BlockedUsers[0], accountIds[2]);
    }
    {
        nn::friends::Profile profile;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileList(&context, &profile, users[0], &accountIds[1], 1));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        ASSERT_TRUE(profile.IsValid());

        EXPECT_EQ(profile.GetAccountId(), accountIds[1]);
    }
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeclareOpenOnlinePlaySession(users[0]));
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeclareCloseOnlinePlaySession(users[0]));

        nn::friends::UserPresence presence;

        NNT_ASSERT_RESULT_SUCCESS(presence.Initialize(users[0]));

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

        NNT_ASSERT_RESULT_SUCCESS(presence.SetAppValue("key", "value"));
        NNT_ASSERT_RESULT_SUCCESS(presence.SetDescription("description"));

NN_PRAGMA_POP_WARNINGS

        NNT_ASSERT_RESULT_SUCCESS(presence.Commit());
    }
    {
        nn::friends::PlayHistoryRegistrationKey key = {};
        nn::friends::InAppScreenName name0 = {"0"};
        nn::friends::InAppScreenName name1 = {"1"};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, accountIds[0], true));
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, users[0], true));

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddPlayHistory(users[1], key, name0, name1));
    }
    {
        nn::friends::InAppScreenName name0 = {"0"};
        nn::friends::InAppScreenName name1 = {"1"};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, users[0], accountIds[1], name1, name0));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultAlreadyFriend, context.GetResult());
    }
    {
        nn::friends::FacedFriendRequestRegistrationKey key0 = {};
        nn::friends::FacedFriendRequestRegistrationKey key1 = {};
        nn::account::Nickname nickname0 = {"0"};
        nn::account::Nickname nickname1 = {"1"};
        nn::Bit8 image[] = {0, 1, 2};
        nn::friends::InAppScreenName inAppScreenName0 = {"0"};
        nn::friends::InAppScreenName inAppScreenName1 = {"1"};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestRegistrationKey(&key0, users[0]));
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestRegistrationKey(&key1, users[1]));

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddFacedFriendRequest(users[0],
            key1, nickname1, image, sizeof (image), inAppScreenName1, inAppScreenName0));
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddFacedFriendRequest(users[1],
            key0, nickname0, image, sizeof (image), inAppScreenName0, inAppScreenName1));
    }
} // NOLINT(impl/function_size)
