﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFriends_CommonUtil.h"

namespace nnt { namespace friends {

namespace
{
    void DeleteFriends(const nn::account::Uid& user) NN_NOEXCEPT
    {
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::SyncFriendList(&context, user));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

        static nn::account::NetworkServiceAccountId s_Friends[nn::friends::FriendCountMax];
        int count;
        nn::friends::FriendFilter filter = {};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, s_Friends, user, 0, NN_ARRAY_SIZE(s_Friends), filter));

        for (int i = 0; i < count; i++)
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeleteFriend(&context, user, s_Friends[i]));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
    }

    void CancelFacedFriendRequests(const nn::account::Uid& user) NN_NOEXCEPT
    {
        static nn::friends::FriendRequest s_Requests[nn::friends::FacedFriendRequestCountMax];
        int count;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            s_Requests, user, 0, NN_ARRAY_SIZE(s_Requests), nn::friends::RequestListType_Faced));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        for (int i = 0; i < count; i++)
        {
            nn::friends::CancelFacedFriendRequest(user, s_Requests[i].GetAccountId());
        }
    }

    void CancelSentFriendRequests(const nn::account::Uid& user) NN_NOEXCEPT
    {
        static nn::friends::FriendRequest s_Requests[nn::friends::FriendCountMax];
        int count;

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                s_Requests, user, 0, NN_ARRAY_SIZE(s_Requests), nn::friends::RequestListType_Sent));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

        for (int i = 0; i < count; i++)
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::CancelFriendRequest(&context, user, s_Requests[i].GetRequestId()));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
    }

    void RejectReceivedFriendRequests(const nn::account::Uid& user) NN_NOEXCEPT
    {
        // 仕様上、受信件数は無限なので取得件数が 0 になるまで繰り返す。
        while (NN_STATIC_CONDITION(true))
        {
            static nn::friends::FriendRequest s_Requests[100];
            int count;

            {
                nn::friends::AsyncContext context;
                NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                    s_Requests, user, 0, NN_ARRAY_SIZE(s_Requests), nn::friends::RequestListType_Received));

                nn::os::SystemEvent completionEvent;
                NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

                completionEvent.Wait();

                NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

                if (count == 0)
                {
                    break;
                }
            }

            for (int i = 0; i < count; i++)
            {
                nn::friends::AsyncContext context;
                NNT_ASSERT_RESULT_SUCCESS(nn::friends::RejectFriendRequest(&context, user, s_Requests[i].GetRequestId()));

                nn::os::SystemEvent completionEvent;
                NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

                completionEvent.Wait();

                NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
            }
        }
    }

    void UnblockUsers(const nn::account::Uid& user) NN_NOEXCEPT
    {
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::SyncBlockedUserList(&context, user));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

        static nn::account::NetworkServiceAccountId s_BlockedUsers[nn::friends::BlockedUserCountMax];
        int count;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetBlockedUserList(&count, s_BlockedUsers, user, 0, NN_ARRAY_SIZE(s_BlockedUsers)));

        for (int i = 0; i < count; i++)
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::UnblockUser(&context, user, s_BlockedUsers[i]));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
    }
}

void LoadAccounts(int* pOutCount, nn::account::Uid* pOutUsers, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutUsers);
    NN_SDK_REQUIRES_GREATER(count, 0);

    *pOutCount = 0;

    int actualCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&actualCount, pOutUsers, count));

    *pOutCount = std::min(actualCount, count);
}

void LoadAccounts(int* pOutCount, nn::account::Uid* pOutUsers, nn::account::NetworkServiceAccountId* pOutAccountIds, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutUsers);
    NN_SDK_REQUIRES_GREATER(count, 0);

    *pOutCount = 0;

    int actualCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&actualCount, pOutUsers, count));

    *pOutCount = std::min(actualCount, count);

    for (int i = 0; i < *pOutCount; i++)
    {
        nn::account::UserHandle handle = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::account::OpenUser(&handle, pOutUsers[i]));

        NN_UTIL_SCOPE_EXIT
        {
            nn::account::CloseUser(handle);
        };

        NNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountId(&pOutAccountIds[i], handle));
    }
}

void ConnectNetwork() NN_NOEXCEPT
{
    nn::nifm::SubmitNetworkRequestAndWait();

    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());
}

void CleanupRelationship(const nn::account::Uid& user) NN_NOEXCEPT
{
    // ローカルのキャッシュを削除する。
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeleteNetworkServiceAccountCache(user));

    DeleteFriends(user);
    CancelFacedFriendRequests(user);
    CancelSentFriendRequests(user);
    RejectReceivedFriendRequests(user);
    UnblockUsers(user);
}

void Dump(nn::friends::UserSetting& setting) NN_NOEXCEPT
{
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("UserSetting\n");
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("    PresencePermission:\n");
    NN_LOG("        - %d\n", setting.GetPresencePermission());
    NN_LOG("    FriendRequestReception:\n");
    NN_LOG("        - %d\n", setting.GetFriendRequestReception());
    NN_LOG("    FriendCode:\n");
    NN_LOG("        - %s\n", setting.GetFriendCode().value);
    NN_LOG("    FriendCodeNextIssuableTime:\n");
    NN_LOG("        - %lld\n", setting.GetFriendCodeNextIssuableTime());
    NN_LOG("    PlayLogPermission:\n");
    NN_LOG("        - %d\n", setting.GetPlayLogPermission());
    NN_LOG("    ProfileExtra:\n");
    NN_LOG("        AccountId:\n");
    NN_LOG("            - %016llx\n", setting.GetExtra().GetAccountId().id);
    NN_LOG("        Nickname:\n");
    NN_LOG("            - %s\n", setting.GetExtra().GetNickname().name);
    NN_LOG("        ProfileImageUrl:\n");

    nn::friends::Url url = {};
    setting.GetExtra().GetProfileImageUrl(&url, nn::friends::ImageSize_Standard);

    NN_LOG("            - %s\n", url.value);
    NN_LOG("        PlayLog:\n");

    for (int i = 0; i < nn::friends::PlayLogCountMax; i++)
    {
        const nn::friends::PlayLog& playLog = setting.GetExtra().GetPlayLog(i);

        if (playLog.appInfo.appId.value != 0)
        {
            NN_LOG("            [%02d]: ApplicationInfo (AppId=%016llx, PresenceGroupId=%016llx)\n",
                i, playLog.appInfo.appId.value, playLog.appInfo.presenceGroupId);
            NN_LOG("            [%02d]: TotalPlayCount=%d, TotalPlayTime=%d, FirstPlayedTime=%lld, LastPlayedTime=%lld\n",
                i, playLog.totalPlayCount, playLog.totalPlayTime, playLog.firstPlayedTime.value, playLog.lastPlayedTime.value);
        }
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

void Dump(const nn::friends::Friend* pFriends, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFriends);

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("FriendList\n");
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("[%d]\n", i);
        NN_LOG("    AccountId:\n");
        NN_LOG("        - %016llx\n", pFriends[i].GetAccountId().id);
        NN_LOG("    Nickname:\n");
        NN_LOG("        - %s\n", pFriends[i].GetNickname().name);
        NN_LOG("    IsFavorite:\n");
        NN_LOG("        - %d\n", pFriends[i].IsFavorite());
        NN_LOG("    IsNewly:\n");
        NN_LOG("        - %d\n", pFriends[i].IsNewly());
        NN_LOG("    PresenceStatus:\n");
        NN_LOG("        - %d\n", pFriends[i].GetPresence().GetStatus());
        NN_LOG("    PresenceDescription:\n");
        NN_LOG("        - %s\n", pFriends[i].GetPresence().GetDescription());
        NN_LOG("    PresenceLastUpdateTime:\n");
        NN_LOG("        - %lld\n", pFriends[i].GetPresence().GetLastUpdateTime().value);
        NN_LOG("    PresenceLastPlayedApplication:\n");
        NN_LOG("        - AppId=%016llx, PresenceGroupId=%016llx\n",
            pFriends[i].GetPresence().GetLastPlayedApplication().appId.value,
            pFriends[i].GetPresence().GetLastPlayedApplication().presenceGroupId);
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

void Dump(const nn::friends::FriendDetailedInfo& info) NN_NOEXCEPT
{
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("FriendDetailedInfo\n");
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("    AccountId:\n");
    NN_LOG("        - %016llx\n", info.GetAccountId().id);
    NN_LOG("    RequestType:\n");
    NN_LOG("        - %d\n", info.GetRequestType());
    NN_LOG("    RequestRouteInfo:\n");

    if (info.GetRequestType() == nn::friends::RequestType_InApp)
    {
        NN_LOG("        ApplicationInfo (AppId=%016llx, PresenceGroupId=%016llx)\n",
            info.GetRequestRouteInfo().appInfo.appId.value, info.GetRequestRouteInfo().appInfo.presenceGroupId);
        NN_LOG("        Name:\n");
        NN_LOG("            - %s\n", info.GetRequestRouteInfo().name.name);
        NN_LOG("        Language:\n");
        NN_LOG("            - %s\n", info.GetRequestRouteInfo().name.language.string);
    }
    else if (info.GetRequestType() == nn::friends::RequestType_FriendCandidate)
    {
        const nn::friends::ExternalApplicationCatalogId& catalogId = info.GetExternalApplicationCatalogId();
        NN_LOG("        ApplicationInfo (CatalogId=%016llx%016llx)\n", catalogId.value[0], catalogId.value[1]);
        NN_LOG("        Name:\n");
        NN_LOG("            - %s\n", info.GetRequestRouteInfo().name.name);
        NN_LOG("        Language:\n");
        NN_LOG("            - %s\n", info.GetRequestRouteInfo().name.language.string);
    }
    else if (info.GetRequestType() == nn::friends::RequestType_3Ds || info.GetRequestType() == nn::friends::RequestType_Nnid)
    {
        NN_LOG("        MiiName:\n");
        NN_LOG("            - %s\n", info.GetMiiName().value);
        NN_LOG("        MiiImageUrlParam:\n");
        NN_LOG("            - %s\n", info.GetMiiImageUrlParam().value);

        nn::friends::Url url = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(info.GetMiiImageUrl(&url));

        NN_LOG("        MiiImageUrl:\n");
        NN_LOG("            - %s\n", url.value);
    }

    NN_LOG("        CreatedTime:\n");
    NN_LOG("            - %lld\n", info.GetRequestRouteInfo().time.value);
    NN_LOG("    LastPlayRecord:\n");
    NN_LOG("        ApplicationInfo (AppId=%016llx, PresenceGroupId=%016llx)\n",
        info.GetLastPlayRecord().appInfo.appId.value, info.GetLastPlayRecord().appInfo.presenceGroupId);
    NN_LOG("        Name:\n");
    NN_LOG("            - %s\n", info.GetLastPlayRecord().name.name);
    NN_LOG("        Name.Language:\n");
    NN_LOG("            - %s\n", info.GetLastPlayRecord().name.language.string);
    NN_LOG("        MyName:\n");
    NN_LOG("            - %s\n", info.GetLastPlayRecord().myName.name);
    NN_LOG("        MyName.Language:\n");
    NN_LOG("            - %s\n", info.GetLastPlayRecord().myName.language.string);
    NN_LOG("        PlayedTime:\n");
    NN_LOG("            - %lld\n", info.GetLastPlayRecord().time.value);
    NN_LOG("    PlayLog:\n");

    for (int n = 0; n < nn::friends::PlayLogCountMax; n++)
    {
        const nn::friends::PlayLog& playLog = info.GetExtra().GetPlayLog(n);

        if (playLog.appInfo.appId.value == 0)
        {
            break;
        }

        NN_LOG("        [%02d]: ApplicationInfo (AppId=%016llx, PresenceGroupId=%016llx)\n",
            n, playLog.appInfo.appId.value, playLog.appInfo.presenceGroupId);
        NN_LOG("                TotalPlayCount=%d, TotalPlayTime=%d, FirstPlayedTime=%lld, LastPlayedTime=%lld\n",
            playLog.totalPlayCount, playLog.totalPlayTime, playLog.firstPlayedTime.value, playLog.lastPlayedTime.value);
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

void Dump(const nn::friends::FriendRequest* pRequests, int count, nn::friends::RequestListType listType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pRequests);

    NN_LOG("--------------------------------------------------------------------------------\n");

    switch (listType)
    {
    case nn::friends::RequestListType_Faced:
        NN_LOG("FacedFriendRequestList\n");
        break;
    case nn::friends::RequestListType_Sent:
        NN_LOG("SentFriendRequestList\n");
        break;
    case nn::friends::RequestListType_Received:
        NN_LOG("ReceivedFriendRequestList\n");
        break;
    default:
        NN_ABORT("");
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("[%d]\n", i);
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("    RequestId:\n");
        NN_LOG("        - %016llx\n", pRequests[i].GetRequestId().value);
        NN_LOG("    AccountId:\n");
        NN_LOG("        - %016llx\n", pRequests[i].GetAccountId().id);
        NN_LOG("    Nickname:\n");
        NN_LOG("        - %s\n", pRequests[i].GetNickname().name);
        NN_LOG("    ProfileImageUrl:\n");

        nn::friends::Url url = {};
        pRequests[i].GetProfileImageUrl(&url, nn::friends::ImageSize_Standard);

        NN_LOG("        - %s\n", url.value);
        NN_LOG("    RequestType:\n");
        NN_LOG("        - %d\n", pRequests[i].GetRequestType());
        NN_LOG("    RequestStatus:\n");
        NN_LOG("        - %d\n", pRequests[i].GetRequestStatus());
        NN_LOG("    RootInfo:\n");

        if (pRequests[i].GetRequestType() == nn::friends::RequestType_Faced || pRequests[i].GetRequestType() == nn::friends::RequestType_InApp)
        {
            NN_LOG("        ApplicationInfo (AppId=%016llx, PresenceGroupId=%016llx)\n",
                pRequests[i].GetRouteInfo().appInfo.appId.value, pRequests[i].GetRouteInfo().appInfo.presenceGroupId);
            NN_LOG("        InAppScreenName:\n");
            NN_LOG("            - %s\n", pRequests[i].GetRouteInfo().name.name);
        }
        else if (pRequests[i].GetRequestType() == nn::friends::RequestType_FriendCandidate)
        {
            const nn::friends::ExternalApplicationCatalogId& catalogId = pRequests[i].GetExternalApplicationCatalogId();

            NN_LOG("        ApplicationInfo (CatalogId=%016llx%016llx)\n", catalogId.value[0], catalogId.value[1]);
            NN_LOG("        InAppScreenName:\n");
            NN_LOG("            - %s\n", pRequests[i].GetRouteInfo().name.name);
        }
        else if (pRequests[i].GetRequestType() == nn::friends::RequestType_3Ds || pRequests[i].GetRequestType() == nn::friends::RequestType_Nnid)
        {
            NN_LOG("        MiiName:\n");
            NN_LOG("            - %s\n", pRequests[i].GetMiiName().value);
            NN_LOG("        MiiImageUrlParam:\n");
            NN_LOG("            - %s\n", pRequests[i].GetMiiImageUrlParam().value);

            NN_ABORT_UNLESS_RESULT_SUCCESS(pRequests[i].GetMiiImageUrl(&url));

            NN_LOG("        MiiImageUrl:\n");
            NN_LOG("            - %s\n", url.value);
        }

        NN_LOG("        CreatedTime:\n");
        NN_LOG("            - %lld\n", pRequests[i].GetRouteInfo().time.value);

        NN_LOG("    IsRead:\n");
        NN_LOG("        - %d\n", pRequests[i].IsRead());
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

void Dump(const nn::friends::FriendCandidate* pCandidates, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCandidates);

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("FriendCandidateList\n");
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        const nn::friends::ExternalApplicationCatalogId& catalogId = pCandidates[i].GetExternalApplicationCatalogId();

        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("[%d]\n", i);
        NN_LOG("    AccountId:\n");
        NN_LOG("        - %016llx\n", pCandidates[i].GetAccountId().id);
        NN_LOG("    Type:\n");
        NN_LOG("        - %d\n", pCandidates[i].GetType());
        NN_LOG("    CatalogId:\n");
        NN_LOG("        - %016llx%016llx\n", catalogId.value[0], catalogId.value[1]);
        NN_LOG("    ApplicationInfo (AppId=%016llx)\n",
            pCandidates[i].GetPlayRecord().appInfo.appId.value);
        NN_LOG("    Name:\n");
        NN_LOG("        - %s\n", pCandidates[i].GetPlayRecord().name.name);
        NN_LOG("    MyName:\n");
        NN_LOG("        - %s\n", pCandidates[i].GetPlayRecord().myName.name);
        NN_LOG("    CreatedTime:\n");
        NN_LOG("        - %lld\n", pCandidates[i].GetPlayRecord().time);
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

void Dump(const nn::friends::NintendoNetworkIdUserInfo& userInfo,
    const nn::friends::NintendoNetworkIdFriend* pFriends, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFriends);

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("NintendoNetworkIdInfo\n");
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("UserInfo:\n");
    NN_LOG("    MiiName:\n");
    NN_LOG("        - %s\n", userInfo.miiName.value);
    NN_LOG("    MiiImageUrlParam:\n");
    NN_LOG("        - %s\n", userInfo.miiImageUrlParam.value);

    nn::friends::Url url = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetMiiImageUrl(&url, userInfo.miiImageUrlParam));

    NN_LOG("    MiiImageUrl:\n");
    NN_LOG("        - %s\n", url.value);

    NN_LOG("    OriginalFriendCount:\n");
    NN_LOG("        - %d\n", userInfo.friendCount);
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("[%d]\n", i);
        NN_LOG("    AccountId:\n");
        NN_LOG("        - %016llx\n", pFriends[i].GetAccountId().id);
        NN_LOG("    MiiName:\n");
        NN_LOG("        - %s\n", pFriends[i].GetMiiName().value);
        NN_LOG("    MiiImageUrlParam:\n");
        NN_LOG("        - %s\n", pFriends[i].GetMiiImageUrlParam().value);

        NN_ABORT_UNLESS_RESULT_SUCCESS(pFriends[i].GetMiiImageUrl(&url));

        NN_LOG("    MiiImageUrl:\n");
        NN_LOG("        - %s\n", url.value);
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

void Dump(const nn::friends::SnsAccountFriend* pFriends, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFriends);

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("SnsAccountFriendList\n");
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("[%d]\n", i);
        NN_LOG("    AccountId:\n");
        NN_LOG("        - %016llx\n", pFriends[i].GetAccountId().id);
        NN_LOG("    Type:\n");
        NN_LOG("        - %d\n", pFriends[i].GetType());
        NN_LOG("    Name1:\n");
        NN_LOG("        - %s\n", pFriends[i].GetSnsAccountProfile().name1.value);
        NN_LOG("    Name2:\n");
        NN_LOG("        - %s\n", pFriends[i].GetSnsAccountProfile().name2.value);
        NN_LOG("    ProfileImageUrl:\n");
        NN_LOG("        - %s\n", pFriends[i].GetSnsAccountProfile().imageUrl.value);
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

void Dump(const nn::friends::BlockedUser* pUsers, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pUsers);

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("BlockedUserList\n");
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("[%d]\n", i);
        NN_LOG("    AccountId:\n");
        NN_LOG("        - %016llx\n", pUsers[i].GetAccountId().id);
        NN_LOG("    Nickname:\n");
        NN_LOG("        - %s\n", pUsers[i].GetNickname().name);
        NN_LOG("    ProfileImageUrl:\n");

        nn::friends::Url url = {};
        pUsers[i].GetProfileImageUrl(&url, nn::friends::ImageSize_Standard);

        NN_LOG("        - %s\n", url.value);
        NN_LOG("    BlockReason:\n");
        NN_LOG("        - %d\n", pUsers[i].GetBlockReason());

        NN_LOG("    RootInfo:\n");

        if (pUsers[i].GetBlockReason() == nn::friends::BlockReason_InApp)
        {
            NN_LOG("        ApplicationInfo (AppId=%016llx, PresenceGroupId=%016llx)\n",
                pUsers[i].GetRouteInfo().appInfo.appId.value, pUsers[i].GetRouteInfo().appInfo.presenceGroupId);
            NN_LOG("        InAppScreenName:\n");
            NN_LOG("            - %s\n", pUsers[i].GetRouteInfo().name.name);
        }

        NN_LOG("        CreatedTime:\n");
        NN_LOG("            - %lld\n", pUsers[i].GetRouteInfo().time.value);
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

void Dump(const nn::friends::PlayHistory* pHistories, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHistories);

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("PlayHistoryList\n");
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("[%d]\n", i);
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("    AccountId:\n");
        NN_LOG("        - %016llx\n", pHistories[i].GetAccountId().id);
        NN_LOG("    HasNetworkServiceAccountId:\n");
        NN_LOG("        - %d\n", pHistories[i].HasNetworkServiceAccountId());
        NN_LOG("    IsLocalPlayed:\n");
        NN_LOG("        - %d\n", pHistories[i].IsLocalPlayed());
        NN_LOG("    ApplicationInfo (AppId=%016llx, PresenceGroupId=%016llx)\n",
            pHistories[i].GetPlayRecord().appInfo.appId.value, pHistories[i].GetPlayRecord().appInfo.presenceGroupId);
        NN_LOG("    Name:\n");
        NN_LOG("        - %s\n", pHistories[i].GetPlayRecord().name.name);
        NN_LOG("    MyName:\n");
        NN_LOG("        - %s\n", pHistories[i].GetPlayRecord().myName.name);
        NN_LOG("    RegisteredTime:\n");
        NN_LOG("        - %lld\n", pHistories[i].GetPlayRecord().time);
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

void Dump(const nn::friends::Profile* pProfiles, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pProfiles);

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("ProfileList\n");
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("[%d]\n", i);
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("    AccountId:\n");
        NN_LOG("        - %016llx\n", pProfiles[i].GetAccountId().id);
        NN_LOG("    Nickname:\n");
        NN_LOG("        - %s\n", pProfiles[i].GetNickname().name);
        NN_LOG("    ProfileImageUrl:\n");

        nn::friends::Url url = {};
        pProfiles[i].GetProfileImageUrl(&url, nn::friends::ImageSize_Standard);

        NN_LOG("        - %s\n", url.value);
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

void Dump(const nn::friends::ProfileExtra* pProfiles, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pProfiles);

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("ProfileExtraList\n");
    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("[%d]\n", i);
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("    AccountId:\n");
        NN_LOG("        - %016llx\n", pProfiles[i].GetAccountId().id);
        NN_LOG("    Nickname:\n");
        NN_LOG("        - %s\n", pProfiles[i].GetNickname().name);
        NN_LOG("    ProfileImageUrl:\n");

        nn::friends::Url url = {};
        pProfiles[i].GetProfileImageUrl(&url, nn::friends::ImageSize_Standard);

        NN_LOG("        - %s\n", url.value);
        NN_LOG("    PlayLog:\n");

        for (int n = 0; n < nn::friends::PlayLogCountMax; n++)
        {
            const nn::friends::PlayLog& playLog = pProfiles[i].GetPlayLog(n);

            if (playLog.appInfo.appId.value == 0)
            {
                break;
            }

            NN_LOG("        [%02d]: ApplicationInfo (AppId=%016llx, PresenceGroupId=%016llx)\n",
                n, playLog.appInfo.appId.value, playLog.appInfo.presenceGroupId);
            NN_LOG("                TotalPlayCount=%d, TotalPlayTime=%d, FirstPlayedTime=%lld, LastPlayedTime=%lld\n",
                playLog.totalPlayCount, playLog.totalPlayTime, playLog.firstPlayedTime.value, playLog.lastPlayedTime.value);
        }
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

}}
