﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/fgm.h>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>

#include "../../Common/fgm_worker.h"

namespace nn    {
namespace fgm   {
namespace test  {

Worker g_SetWorker NN_ALIGNAS(4096);

nn::os::Event g_NotificationReceived(nn::os::EventClearMode_AutoClear);
nn::TimeSpan g_TimeOutSpan = nn::TimeSpan::FromSeconds(5);

void RequestSet()
{
    nn::Result result;
    nn::fgm::Request request;
    bool waitResult;
    if ((result = request.Initialize(nn::fgm::Module_Test, nn::fgm::Priority_Default)).IsFailure())
    {
        NN_LOG("Failed to Initialize request, error %d:%d\n", result.GetModule(), result.GetDescription());
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));

    NN_LOG("Request Min setting...\n");
    result = request.Set(nn::fgm::Setting_Min, nn::fgm::Setting_Min);

    waitResult = g_NotificationReceived.TimedWait(g_TimeOutSpan);

    if (!waitResult)
        NN_LOG("Error: Notification received event was never signaled.\n");
    else
        g_NotificationReceived.Clear();
}

}}}

// *** Notification testing ***//
TEST(FgmNotificationTest, request)
{
    nn::Result result, timeOutResult;
    nn::fgm::Request request;
    nn::fgm::Setting initialSetting, newSetting;
    bool isNotified;

    EXPECT_TRUE((result = request.Initialize(nn::fgm::Module_Test, nn::fgm::Priority_Default)).IsSuccess());
    EXPECT_TRUE((result = request.Get(&initialSetting)).IsSuccess());
    EXPECT_FALSE((result = request.WaitWithTimeout(nn::TimeSpan::FromMilliSeconds(100))).IsSuccess());
    nn::fgm::test::g_SetWorker.Initialize(nn::fgm::test::RequestSet, nn::os::DefaultThreadPriority);

    NN_LOG("Wait for request...\n");
    if ((timeOutResult = request.WaitWithTimeout(nn::fgm::test::g_TimeOutSpan)).IsSuccess())
    {
        EXPECT_TRUE((result = request.Get(&newSetting)).IsSuccess());
        NN_LOG("Notification received for state change. From %u to %u Hz\n", initialSetting, newSetting);
        isNotified = true;
    }
    else
    {
        NN_LOG("Failed: Time out expired for request notification %d:%d\n",
            timeOutResult.GetModule(), timeOutResult.GetDescription());
        isNotified = false;
    }

    EXPECT_FALSE((result = request.WaitWithTimeout(nn::TimeSpan::FromMilliSeconds(100))).IsSuccess());
    EXPECT_TRUE(isNotified);

    nn::fgm::test::g_NotificationReceived.Signal();
    nn::fgm::test::g_SetWorker.Finalize();

    NN_LOG("Finalize request...\n");
    request.Finalize();
}

TEST(FgmNotificationTest, SystemEvent_AutoClear)
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting initialSetting, newSetting;
    bool isNotified;
    nn::os::SystemEvent* sysEventPtr;

    EXPECT_TRUE((result = request.Initialize(nn::fgm::Module_Test, nn::fgm::Priority_Default, nn::os::EventClearMode_AutoClear)).IsSuccess());
    EXPECT_TRUE((result = request.Get(&initialSetting)).IsSuccess());

    sysEventPtr = request.GetSystemEventPointer();

    EXPECT_FALSE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));

    nn::fgm::test::g_SetWorker.Initialize(nn::fgm::test::RequestSet, nn::os::DefaultThreadPriority);

    NN_LOG("Wait for request...\n");
    if (sysEventPtr->TimedWait(nn::fgm::test::g_TimeOutSpan))
    {
        EXPECT_TRUE((result = request.Get(&newSetting)).IsSuccess());
        NN_LOG("Notification received for state change. From %u to %u Hz\n", initialSetting, newSetting);
        isNotified = true;
    }
    else
    {
        NN_LOG("Failed: Time out expired for request notification \n");
        isNotified = false;
    }

    EXPECT_FALSE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));
    EXPECT_TRUE(isNotified);

    nn::fgm::test::g_NotificationReceived.Signal();
    nn::fgm::test::g_SetWorker.Finalize();

    NN_LOG("Finalize request...\n");
    request.Finalize();
}

TEST(FgmNotificationTest, SystemEvent_ManualClear)
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting initialSetting, newSetting;
    bool isNotified;
    nn::os::SystemEvent* sysEventPtr;

    // Default EventClearMode for fgm request is ManualClear
    EXPECT_TRUE((result = request.Initialize(nn::fgm::Module_Test, nn::fgm::Priority_Default)).IsSuccess());
    EXPECT_TRUE((result = request.Get(&initialSetting)).IsSuccess());

    sysEventPtr = request.GetSystemEventPointer();

    EXPECT_FALSE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));

    nn::fgm::test::g_SetWorker.Initialize(nn::fgm::test::RequestSet, nn::os::DefaultThreadPriority);

    NN_LOG("Wait for request...\n");
    if (sysEventPtr->TimedWait(nn::fgm::test::g_TimeOutSpan))
    {
        EXPECT_TRUE((result = request.Get(&newSetting)).IsSuccess());
        NN_LOG("Notification received for state change. From %u to %u Hz\n", initialSetting, newSetting);
        isNotified = true;
    }
    else
    {
        NN_LOG("Failed: Time out expired for request notification\n");
        isNotified = false;
    }

    EXPECT_TRUE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));
    sysEventPtr->Clear();
    EXPECT_FALSE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));
    EXPECT_TRUE(isNotified);

    nn::fgm::test::g_NotificationReceived.Signal();
    nn::fgm::test::g_SetWorker.Finalize();

    NN_LOG("Finalize request...\n");
    request.Finalize();
}
