﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/fgm.h>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>

#include <nn/fgm/fgm_Debugger.h>
#include "../../Common/fgm_worker.h"

const size_t        LogMemorySize  = (128 * 1024);
const size_t        EntriesPerRead = 2;
uint8_t             g_LogMemory[LogMemorySize] NN_ALIGNAS(4096);
nn::fgm::DebugEntry g_Entries[EntriesPerRead];
int                 g_TimeoutSeconds = 5;

namespace nn    {
namespace fgm   {
namespace test  {

Worker g_SetWorker NN_ALIGNAS(4096);

nn::os::Event g_NotificationReceived(nn::os::EventClearMode_AutoClear);
nn::TimeSpan g_TimeOutSpan = nn::TimeSpan::FromSeconds(5);

void RequestSet()
{
    nn::Result result;
    nn::fgm::Request request;
    bool waitResult;
    if ((result = request.Initialize(nn::fgm::Module_Test, nn::fgm::Priority_Default)).IsFailure())
    {
        NN_LOG("Failed to Initialize request, error %d:%d\n", result.GetModule(), result.GetDescription());
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));

    NN_LOG("Request Min setting...\n");
    result = request.Set(nn::fgm::Setting_Min, nn::fgm::Setting_Min);

    waitResult = g_NotificationReceived.TimedWait(g_TimeOutSpan);

    if (!waitResult)
        NN_LOG("Error: Notification received event was never signaled.\n");
    else
        g_NotificationReceived.Clear();
}

}}}

// *** Notification testing ***//
TEST(FgmDebuggerTest, Notification)
{
    nn::Result result, timeOutResult;
    nn::fgm::Debugger debugger;
    bool isNotified;

    EXPECT_TRUE((result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory))).IsSuccess());

    nn::fgm::test::g_SetWorker.Initialize(nn::fgm::test::RequestSet, nn::os::DefaultThreadPriority);

    NN_LOG("Debugger waiting for request...\n");
    if ((timeOutResult = debugger.WaitWithTimeout(nn::fgm::test::g_TimeOutSpan)).IsSuccess())
    {
        isNotified = true;
    }
    else
    {
        NN_LOG("Failed: Time out expired for request notification %d:%d\n",
            timeOutResult.GetModule(), timeOutResult.GetDescription());
        isNotified = false;
    }

    EXPECT_FALSE((result = debugger.WaitWithTimeout(nn::TimeSpan::FromMilliSeconds(100))).IsSuccess());
    EXPECT_TRUE(isNotified);

    nn::fgm::test::g_NotificationReceived.Signal();
    nn::fgm::test::g_SetWorker.Finalize();

    NN_LOG("Finalize debugger...\n");
    ASSERT_TRUE((result = debugger.Finalize()).IsSuccess());
}

TEST(FgmNotificationTest, SystemEvent_AutoClear)
{
    nn::Result result;
    nn::fgm::Debugger debugger;
    bool isNotified;
    nn::os::SystemEvent* sysEventPtr;

    EXPECT_TRUE((result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory), nn::os::EventClearMode_AutoClear)).IsSuccess());

    sysEventPtr = debugger.GetSystemEventPointer();

    EXPECT_FALSE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));

    nn::fgm::test::g_SetWorker.Initialize(nn::fgm::test::RequestSet, nn::os::DefaultThreadPriority);

    NN_LOG("Debugger waiting for request...\n");
    if (sysEventPtr->TimedWait(nn::fgm::test::g_TimeOutSpan))
    {
        isNotified = true;
    }
    else
    {
        isNotified = false;
    }

    EXPECT_FALSE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));
    EXPECT_TRUE(isNotified);

    nn::fgm::test::g_NotificationReceived.Signal();
    nn::fgm::test::g_SetWorker.Finalize();

    NN_LOG("Finalize debugger...\n");
    ASSERT_TRUE((result = debugger.Finalize()).IsSuccess());
}

TEST(FgmNotificationTest, SystemEvent_ManualClear)
{
    nn::Result result;
    nn::fgm::Debugger debugger;
    bool isNotified;
    nn::os::SystemEvent* sysEventPtr;

    EXPECT_TRUE((result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory), nn::os::EventClearMode_ManualClear)).IsSuccess());

    sysEventPtr = debugger.GetSystemEventPointer();

    EXPECT_FALSE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));

    nn::fgm::test::g_SetWorker.Initialize(nn::fgm::test::RequestSet, nn::os::DefaultThreadPriority);

    NN_LOG("Debugger waiting for request...\n");
    if (sysEventPtr->TimedWait(nn::fgm::test::g_TimeOutSpan))
    {
        isNotified = true;
    }
    else
    {
        isNotified = false;
    }

    EXPECT_TRUE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));
    debugger.ClearEvent();
    EXPECT_FALSE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));
    EXPECT_TRUE(isNotified);

    nn::fgm::test::g_NotificationReceived.Signal();
    nn::fgm::test::g_SetWorker.Finalize();

    NN_LOG("Finalize debugger...\n");
    ASSERT_TRUE((result = debugger.Finalize()).IsSuccess());
}
