﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/fgm.h>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include <nn/util/util_FormatString.h>

#include <nn/fgm/fgm_Debugger.h>

using namespace nn::fgm;

const size_t        LogMemorySize  = (128 * 1024);
const size_t        EntriesPerRead = 2;
uint8_t             g_LogMemory[LogMemorySize] NN_ALIGNAS(4096);
DebugEntry          g_Entries[EntriesPerRead];
int                 g_TimeoutSeconds = 5;

char* FormatTime(nn::os::Tick tick)
{
    static char buf[256];
    uint64_t usecs   = nn::os::ConvertToTimeSpan(tick).GetMicroSeconds();
    uint32_t msecs   = usecs   / 1000;
    uint32_t secs    = msecs   / 1000;
    uint32_t minutes = secs    / 60;
    uint32_t hours   = minutes / 60;

    nn::util::SNPrintf(
                buf, sizeof(buf),
                "%02u:%02u:%02u:%03u:%03u",
                hours % 24,
                minutes % 60,
                secs % 60,
                msecs % 1000,
                (uint32_t)(usecs % 1000));

    return buf;
}

void ParseLog(DebugEntry* entries, uint32_t nEntries)
{
    for (int i = 0; i < nEntries; i++)
    {
        switch (entries[i].event)
        {
        case EventId_Set:
            NN_LOG("%s - module %u set req: pid %llu, priority %u, min %u, max %u\n",
                    FormatTime(entries[i].timestamp),
                    entries[i].id,
                    entries[i].u.request.pid,
                    entries[i].u.request.priority,
                    entries[i].u.request.min,
                    entries[i].u.request.max);
            break;

        case EventId_Unset:
            NN_LOG("%s - module %u unset req: pid %llu, priority %u, min %u, max %u\n",
                    FormatTime(entries[i].timestamp),
                    entries[i].id,
                    entries[i].u.request.pid,
                    entries[i].u.request.priority,
                    entries[i].u.request.min,
                    entries[i].u.request.max);
            break;

        case EventId_Completion:
            NN_LOG("%s - module %u completed, requested min %u, requested max %u, actual %u\n",
                    FormatTime(entries[i].timestamp),
                    entries[i].id,
                    entries[i].u.completion.min,
                    entries[i].u.completion.max,
                    entries[i].u.completion.actual);
            break;

        default:
            break;
        }
    }
}

TEST(FgmDebuggerInitializeCase, AutoClear)
{
    nn::Result result;
    Debugger debugger;

    result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory), nn::os::EventClearMode_AutoClear);
    EXPECT_TRUE(result.IsSuccess()) << "Failed to Initialize debugger";

    ASSERT_TRUE((result = debugger.Finalize()).IsSuccess());
}

TEST(FgmDebuggerInitializeCase, ManualClear)
{
    nn::Result result;
    Debugger debugger;

    result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory), nn::os::EventClearMode_ManualClear);
    EXPECT_TRUE(result.IsSuccess()) << "Failed to Initialize debugger";

    ASSERT_TRUE((result = debugger.Finalize()).IsSuccess());

}

TEST(FgmDebuggerInitializeCase, DefaultClear)
{
    nn::Result result;
    Debugger debugger;

    result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory));
    EXPECT_TRUE(result.IsSuccess()) << "Failed to Initialize debugger";

    ASSERT_TRUE((result = debugger.Finalize()).IsSuccess());
}

TEST(FgmDebuggerWaitCase, withTimeout)
{
    nn::Result result;
    Debugger debugger;
    Request request;

    EXPECT_TRUE((result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory))).IsSuccess());
    EXPECT_TRUE((result = request.Initialize(Module_Test, Priority_Default)).IsSuccess());

    EXPECT_TRUE((result = request.SetAndWait(Setting_Min, Setting_Max)).IsSuccess());

    EXPECT_TRUE((result = debugger.WaitWithTimeout(nn::TimeSpan::FromSeconds(g_TimeoutSeconds))).IsSuccess());

    EXPECT_FALSE((result = debugger.WaitWithTimeout(nn::TimeSpan::FromMilliSeconds(100))).IsSuccess());

    EXPECT_TRUE((result = debugger.Finalize()).IsSuccess());

    ASSERT_TRUE((result = request.Finalize()).IsSuccess());

}

TEST(FgmDebuggerWaitCase, DefaultWait)
{
    nn::Result result;
    Debugger debugger;
    Request request;

    EXPECT_TRUE((result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory))).IsSuccess());

    EXPECT_TRUE((result = request.Initialize(Module_Test, Priority_Default)).IsSuccess());

    EXPECT_TRUE((result = request.SetAndWait(Setting_Min, Setting_Max)).IsSuccess());

    EXPECT_TRUE((result = debugger.Wait()).IsSuccess());

    EXPECT_FALSE((result = debugger.WaitWithTimeout(nn::TimeSpan::FromMilliSeconds(100))).IsSuccess());

    ASSERT_TRUE((result = debugger.Finalize()).IsSuccess());
}

TEST(FgmDebuggerCancelCase, CancelTest)
{
    nn::Result result;
    Debugger debugger;

    EXPECT_TRUE((result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory))).IsSuccess());

    EXPECT_TRUE((result = debugger.Cancel()).IsSuccess());

    EXPECT_TRUE((result = debugger.Wait()).IsSuccess());

    ASSERT_TRUE((result = debugger.Finalize()).IsSuccess());
}

TEST(FgmDebuggerReadCase, ReadTest)
{
    nn::Result result;
    uint32_t read;
    uint32_t lost;
    uint32_t remaining;

    Debugger debugger;
    Request request;

    EXPECT_TRUE((result = request.Initialize(Module_Test, Priority_Default)).IsSuccess());
    EXPECT_TRUE((result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory))).IsSuccess());

    EXPECT_TRUE((result = request.SetAndWait(Setting_Min, Setting_Max)).IsSuccess());

    EXPECT_TRUE((result = debugger.WaitWithTimeout(nn::TimeSpan::FromSeconds(g_TimeoutSeconds))).IsSuccess());

    do
    {
        result = debugger.Read(
                    g_Entries,
                    sizeof(g_Entries) / sizeof(g_Entries[0]),
                    &read,
                    &lost,
                    &remaining);

        if (result.IsSuccess() && read > 0)
        {
            SUCCEED();
            ParseLog(g_Entries, read);
        }
        else
        {
            ADD_FAILURE() << "Debugger failed to Read(); ";
        }

    } while (remaining > 0);

    if (lost != 0)
    {
        NN_LOG("overrun(s) detected, discarded %u old events\n", lost);
    }

    EXPECT_TRUE((result = debugger.Finalize()).IsSuccess());

    ASSERT_TRUE((result = request.Finalize()).IsSuccess());
}

TEST(FgmDebuggerEventClear, ManualClearTest)
{
    nn::Result result;
    nn::os::SystemEvent* sysEventPtr;

    Debugger debugger;
    Request request;

    EXPECT_TRUE((result = request.Initialize(Module_Test, Priority_Default)).IsSuccess());
    EXPECT_TRUE((result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory), nn::os::EventClearMode_ManualClear)).IsSuccess());

    sysEventPtr =  debugger.GetSystemEventPointer();

    EXPECT_TRUE((result = request.SetAndWait(Setting_Min, Setting_Max)).IsSuccess());

    EXPECT_TRUE(sysEventPtr->TimedWait(nn::TimeSpan::FromSeconds(g_TimeoutSeconds)));
    EXPECT_TRUE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));
    debugger.ClearEvent();
    EXPECT_FALSE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));

    EXPECT_TRUE((result = debugger.Finalize()).IsSuccess());

    ASSERT_TRUE((result = request.Finalize()).IsSuccess());
}

TEST(FgmDebuggerEventClear, AutoClearTest)
{
    nn::Result result;
    nn::os::SystemEvent* sysEventPtr;

    Debugger debugger;
    Request request;

    EXPECT_TRUE((result = request.Initialize(Module_Test, Priority_Default)).IsSuccess());
    EXPECT_TRUE((result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory), nn::os::EventClearMode_AutoClear)).IsSuccess());

    sysEventPtr =  debugger.GetSystemEventPointer();

    EXPECT_TRUE((result = request.SetAndWait(Setting_Min, Setting_Max)).IsSuccess());

    EXPECT_TRUE(sysEventPtr->TimedWait(nn::TimeSpan::FromSeconds(g_TimeoutSeconds)));
    EXPECT_FALSE(sysEventPtr->TimedWait(nn::TimeSpan::FromMilliSeconds(100)));

    EXPECT_TRUE((result = debugger.Finalize()).IsSuccess());

    ASSERT_TRUE((result = request.Finalize()).IsSuccess());

}
