﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2010, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */
#include <nnt/nntest.h>
#include <nn/nn_Log.h>

#include <nn/escore/escore.h>
#include <nnt/escoreUtil/testEscore_util_istorage.h>
#include <nn/fs.h>
#include <nn/nn_Result.h>
#include <nn/spl/spl_Api.h>

USING_ESCORE_UTIL_NAMESPACE
USING_ES_NAMESPACE
USING_ISTORAGE_NAMESPACE

#include <nn/ioscrypto/iosctypes.h>
#include <nn/escorei/esi.h>

#include "../Common/testEs_Est_utils.h"

// TODO: dynamically generate the following files
#include "../Common/testEs_Ms_cert.cpp"

#include "../Common/testEs_Ca_cp_certs.cpp"
#include "../Common/testEs_Ca_xs_certs.cpp"

#include "../Common/testEs_Tkt6.cpp"

#include "../Common/testEs_Data3_enc.cpp"
#include "../Common/testEs_Data3_dec.cpp"
#include "../Common/testEs_Data5_enc.cpp"
#include "../Common/testEs_Data5_dec.cpp"

#include "../Common/testEs_Ms8_cert.cpp"

#include "../Common/testEs_Ctr_signature.cpp"
#include "../Common/testEs_Ctr_dev_cert.cpp"
#include "../Common/testEs_Ctr_signer_cert.cpp"

#include "../Common/testEs_Ctr_prod_signature.cpp"
#include "../Common/testEs_Ctr_prod_dev_cert.cpp"
#include "../Common/testEs_Ctr_prod_signer_cert.cpp"

/*
 * New CTR Cert Chain (SHA256 signatures)
 */
#include "../Common/testEs_Ca_cert_new.cpp"
#include "../Common/testEs_Cp_cert_new.cpp"
#include "../Common/testEs_Xs_cert_new.cpp"
#include "../Common/testEs_Sp_cert_new.cpp"

/*
 * Nintendo CA Cert Chain (SHA256 signatures)
 */
#include "../Common/testEs_Nin_G3_root_cert.cpp"
#include "../Common/testEs_Nin_devCA_cert.cpp"
#include "../Common/testEs_Nin_prodCA_cert.cpp"

/*
 * Cert buffer containing SHA256 certs + Nintendo certs
 */
#include "../Common/testEs_Merged_cert_list.cpp"

#include "../Common/testEs_Device.cpp"

const int MAX_CERT_LIST = 32;  // arbitrary limit on cert lists

USING_ES_NAMESPACE

// Hardcoded device, title, and content IDs
#ifdef USE_OBSOLETE_CODE
const u32 ES_TEST_DEVICE_ID = 0x000001f5;
#endif

const u64 ES_TEST_ACCESSOR_TITLE_ID = 0x010000000000100BULL; // NX eShop ProgramId

#ifdef CTR_TEST
const u64 ES_TWL_TITLE_ID = 0x0003001545534c54ULL;
const u64 ES_TWL_TITLE_ID_BAD1 = 0x000300000bad0badULL;
const u64 ES_TWL_TITLE_ID_BAD2 = 0x0003001545534c50ULL;  // a more subtle badness
const u64 ES_TWL_TITLE_ID_BAD3 = 0xA003001545534c54ULL;
#endif

// Certificates
const int ES_TEST_CERT_CHAIN_LEN = 4;  // CA, CA_new, XS, XS_new, CP, MS

static const void *__certs[ES_TEST_CERT_CHAIN_LEN + 2] = { caNewCert, xsNewCert, cpNewCert, msCert, NULL, NULL };
static const u32 __nCerts = ES_TEST_CERT_CHAIN_LEN;

// Buffers for testing sign/verify
static u8 __sig[ES_SIGNATURE_SIZE] ATTR_SHA256_ALIGN;
static u8 __signerCert[ES_SIGNING_CERT_SIZE] ATTR_SHA256_ALIGN;
static u8 __devCert[ES_DEVICE_CERT_SIZE] ATTR_SHA256_ALIGN;

// MemoryStorage buffers
static u8 __dpTicketBuf[2048] ATTR_SHA256_ALIGN;

// Arbitrary data for signing
#ifdef CTR_TEST
static u8 __data[] ATTR_SHA256_ALIGN = {0x0e, 0xb4, 0x9a, 0x55, 0x5a, 0x0f, 0xa4, 0x2f, 0x8e, 0xf2, 0xa3, 0xbc, 0x29, 0x00, 0x3f, 0xbf};
#endif

/**
    @brief FS用アロケータ
 */
void* Allocate(size_t size)
{
    return std::malloc(size);
}

/**
    @brief FS用デアロケータ
 */
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}

static void writeFile( const char *fileName, const void* buffer, size_t size)
{
    std::string path;
    //#define NNT_ES_SIGLO_ROOT set by nact script or vcxporj to the root path of Siglo.
    const char* kResourcePath = "/Tests/Escore/Sources/Tests/Sanity/testEs_Sanity/";
    path = std::string(NNT_ES_SIGLO_ROOT) + std::string(kResourcePath) + fileName;

    nn::fs::FileHandle file;
    nn::Result result;
    nn::fs::DeleteFile(path.c_str());
    result = nn::fs::CreateFile(path.c_str(), size);
    if ( result.IsFailure() )
    {
        ES_TEST_LOG( "create failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }

    result = nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_Write);
    if ( result.IsFailure() )
    {
        ES_TEST_LOG( "open failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }

    result = nn::fs::WriteFile(file, 0, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    if (result.IsFailure())
    {
        ES_TEST_LOG( "write failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }
    nn::fs::CloseFile(file);
}

static ESError __testSetup()
{
    ETicketService es;
    es.SetDevicePrivateKey( &deviceKeyNX6100000000000001, sizeof(deviceKeyNX6100000000000001) );

    nn::fs::SetAllocator(Allocate, Deallocate);
    nn::fs::MountHostRoot();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::spl::Initialize();
#endif

    return ES_ERR_OK;
}


static ESError __testCleanup()
{
    return ES_ERR_OK;
}


#define ES_TEST_ISTORAGE_CONTENT "abcdefghij"

static ESError __testMemoryStorage()
{
    ESError rv = ES_ERR_OK;
    const char *content = ES_TEST_ISTORAGE_CONTENT;
    u8 buf[sizeof( ES_TEST_ISTORAGE_CONTENT )];
    s64 i, pos, size, len;
    s32 pOut;

    // Test MemoryOutputStream
    {
        MemoryOutputStream writer( __dpTicketBuf, sizeof( __dpTicketBuf ) );

        writer.TryGetPosition( &pos );
        if( pos != 0 )
        {
            ES_TEST_LOG( "Unexpected writer position, pos=0:%lld\n", pos );
            rv = ES_ERR_FAIL;
            goto end;
        }

        writer.TryGetSize( &size );
        if( size != sizeof( __dpTicketBuf ) )
        {
            ES_TEST_LOG( "Unexpected file size, size=0:%lld\n", size );
            rv = ES_ERR_FAIL;
            goto end;
        }

        len = sizeof( ES_TEST_ISTORAGE_CONTENT );
        writer.TryWrite( &pOut, content, (u32)len, true );
        if( pOut != len )
        {
            ES_TEST_LOG( "Unexpected write return value\n" );
            rv = ES_ERR_FAIL;
            goto end;
        }

        writer.TrySetPosition( 1 );
        writer.TryGetPosition( &pos );
        if( pos != 1 )
        {
            ES_TEST_LOG( "Unexpected writer position, pos=1:%lld\n", pos );
            rv = ES_ERR_FAIL;
            goto end;
        }

        writer.TryGetSize( &size );
        if( size != sizeof( __dpTicketBuf ) )
        {
            ES_TEST_LOG( "Unexpected file size, size=%lld\n", size );
            rv = ES_ERR_FAIL;
            goto end;
        }
    }

    // Test MemoryInputStream
    {
        MemoryInputStream reader( __dpTicketBuf, (u32)len );

        reader.TryGetSize( &size );
        if( size != len )
        {
            ES_TEST_LOG( "Unexpected file size, size=%lld, %lld\n", len, size );
            rv = ES_ERR_FAIL;
            goto end;
        }

        reader.TrySetPosition( 1 );
        reader.TryRead( &pOut, buf, (u32)size - 2 );
        if( pOut != ( size - 2 ) )
        {
            ES_TEST_LOG( "Unexpected read return value\n" );
            rv = ES_ERR_FAIL;
            goto end;
        }

        for( i = 0; i < ( size - 2 ); i++ )
        {
            if( buf[i] != content[i + 1] )
            {
                ES_TEST_LOG( "Unexpected content, %c:%c\n", content[i + 1], buf[i] );
                rv = ES_ERR_FAIL;
                goto end;
            }
        }

        reader.TryGetPosition( &pos );
        if( pos != ( size - 1 ) )
        {
            ES_TEST_LOG( "Unexpected reader position, pos=%lld:%lld\n", size - 1, pos );
            rv = ES_ERR_FAIL;
            goto end;
        }
    }

    ES_TEST_LOG( "Passed storage test\n" );

end:
    return rv;
}


static ESError __testPacking()
{
    ESError rv = ES_ERR_OK;

    if( sizeof( ESTicket ) != 676 || sizeof( ESV1TicketHeader ) != 20 || sizeof( ESV1SectionHeader ) != 20 ||
        sizeof( ESV1SubscriptionRecord ) != 24 || sizeof( ESV1PermanentRecord ) != 20 || sizeof( ESV1ContentRecord ) != 132 ||
        sizeof( ESV1ContentConsumptionRecord ) != 8 || sizeof( ESV1ContentMeta ) != 48 || sizeof( ESTitleMetaHeader ) != 100 ||
        sizeof( ESV1TitleMetaHeader ) != 2336 || sizeof( ESV1TitleMeta ) != 2820 || sizeof( ESV1AccessTitleRecord ) != 16 )
    {
        ES_TEST_LOG( "Unexpected structure sizes\n" );
        rv = ES_ERR_FAIL;
        goto end;
    }

    ES_TEST_LOG( "Passed packing test\n" );

end:
    return rv;
}

static u8 signMessage[] ="<Challenge>123456</Challenge><SerialNumber>sn6989586621679009793</SerialNumber>";

static ESError __testSign()
{
    ESError rv = ES_ERR_OK;
    ETicketService es;
    //MemoryInputStream data( encryptedContent, sizeof( encryptedContent ) );
    MemoryInputStream data( signMessage, sizeof( signMessage ) - 1 );

    ESDeviceId devId;
    void *certList[MAX_CERT_LIST];
    u32 nCerts;
    void *verifyChain[MAX_CERT_LIST];
    u32 nVerifyChain;
    u8 *subject;
    u8 *issuer;
    ESSigType sigType;

    /*
     * This routine tests self-signing and verifying on the current device
     */

    //rv = es.Sign( ES_TEST_ACCESSOR_TITLE_ID, data, sizeof( encryptedContent ), __sig, sizeof( __sig ), __signerCert, sizeof( __signerCert ) );
    rv = es.Sign( ES_TEST_ACCESSOR_TITLE_ID, data, sizeof( signMessage ) - 1, __sig, sizeof( __sig ), __signerCert, sizeof( __signerCert ) );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to sign data, rv=%d\n", rv );
        goto end;
    }
    writeFile("sign", __sig, sizeof( __sig ));
    writeFile("signer.cert", __signerCert, sizeof( __signerCert ));

    rv = ESI_GetCertNames( __signerCert, &issuer, &subject );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: ESI_GetCertNames returns error, rv=%d\n", rv );
        goto end;
    }
    // ES_TEST_LOG("__testSign: signerCert issuer %s, subject %s\n", issuer, subject);

    rv = es.GetDeviceCert( __devCert, sizeof( __devCert ) );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get device cert, rv=%d\n", rv );
        goto end;
    }

    rv = ESI_GetCertNames( __devCert, &issuer, &subject );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: ESI_GetCertNames returns error, rv=%d\n", rv );
        goto end;
    }

    rv = es.GetDeviceId( &devId );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get device ID, rv=%d\n", rv );
        goto end;
    }
// ES_TEST_LOG("__testSign: current deviceId %08x (subjectName %s)\n", devId, subject);

#ifdef USE_OBSOLETE_CODE
    /*
     * Of course the Device ID is different on each device.  We have the certificates for a number
     * of CTR "dev" devices and can assume those identities on Linux.  The Sign/Verify implementation
     * has been extended to support CTR->CTR data exchange, so it is no longer necessary to restrict
     * this test.
     */
    if( devId != ES_TEST_DEVICE_ID )
    {
        ES_TEST_LOG( "Failed to get expected device ID, %08x:%08x\n", devId, ES_TEST_DEVICE_ID );
        rv = ES_ERR_FAIL;
        goto end;
    }
#endif

    /*
     * The 4.0 Sign code always uses SHA256
     */
    sigType = ES_SIG_TYPE_ECC_SHA256;

    /*
     * The certificate chain required depends on the type of device cert
     */
    nCerts = 0;
    if( strncmp( (const char *)subject, "CT", 2 ) == 0 ) // CTR
    {
        certList[nCerts++] = caNewCert;
        certList[nCerts++] = cpNewCert;
        certList[nCerts++] = xsNewCert;
        certList[nCerts++] = ninDevCACert;
    }
    else if( strncmp( (const char *)subject, "TW", 2 ) == 0 ) // TWL
    {
        ES_TEST_LOG("Failed: __testSign with unknown device type, skipping VerifySign\n");
        goto end;
    }
    else if( strncmp( (const char *)subject, "NG", 2 ) == 0 ) // RVL
    {
        ES_TEST_LOG("Failed: __testSign with unknown device type, skipping VerifySign\n");
        goto end;
    }
    else if( strncmp( (const char *)subject, "NX", 2 ) == 0 ) // HAC
    {
        certList[nCerts++] = caNewCert;
        certList[nCerts++] = cpNewCert;
        certList[nCerts++] = xsNewCert;
        //certList[nCerts++] = ninDevCACert;
    }
    else
    {
        ES_TEST_LOG( "Failed: __testSign with unknown device type, skipping VerifySign\n" );
        goto end;
    }

    certList[nCerts++] = __devCert;
    certList[nCerts++] = __signerCert;

    // ES_TEST_LOG("__testSign: call VerifyCerts with full app signer chain\n");

    verifyChain[0] = NULL;
    nVerifyChain = 0;

#ifdef NINTENDO_NX_CA_EXISTS
    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts returns rv=%d\n", rv );
        goto end;
    }

    // ES_TEST_LOG("__testSign: call VerifySign for self-signed data with sigType %d, nCerts %d\n", sigType, nCerts);

    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, data, sizeof( encryptedContent ), __sig, sizeof( __sig ), sigType, (const void **)certList,
                        nCerts );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to verify data, rv=%d\n", rv );
        goto end;
    }

    /*
     * Negative test: corrupt the signature
     */
    __sig[43] ^= 0x80;
    data.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, data, sizeof( encryptedContent ), __sig, sizeof( __sig ), sigType, (const void **)certList,
                        nCerts );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: verify of corrupted signature passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: corrupted signature detected (previous failure was expected)\n" );
    }
    __sig[43] ^= 0x80;

    /*
     * Negative test: corrupt the data input to the signature calculation
     */
    encryptedContent[17] ^= 0x5;
    data.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, data, sizeof( encryptedContent ), __sig, sizeof( __sig ), sigType, (const void **)certList,
                        nCerts );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: verify of corrupted signature input passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: corrupted signature input detected (previous failure was expected)\n" );
    }
    encryptedContent[17] ^= 0x5;

    /*
     * Negative test: leave out the last byte of the data input to the signature calculation
     */
    data.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, data, sizeof( encryptedContent ) - 1, __sig, sizeof( __sig ), sigType, (const void **)certList,
                        nCerts );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: verify of short signature input passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: short signature input detected (previous failure was expected)\n" );
    }

    /*
     * Negative test: corrupt the signer certificate
     */
    __signerCert[365] ^= 0x20;
    data.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, data, sizeof( encryptedContent ), __sig, sizeof( __sig ), sigType, (const void **)certList,
                        nCerts );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: verify of corrupted signer certificate passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: corrupted signer certificate detected (previous failure was expected)\n" );
    }
    __signerCert[365] ^= 0x20;

    /*
     * Negative test: corrupt the signing device certificate
     */
    __devCert[129] ^= 0x8;
    data.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, data, sizeof( encryptedContent ), __sig, sizeof( __sig ), sigType, (const void **)certList,
                        nCerts );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: verify of corrupted signing device certificate passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: corrupted signing device certificate detected (previous failure was expected)\n" );
    }
    __devCert[129] ^= 0x8;

    /*
     * Negative test: bad title ID should fail
     */
    data.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID_BAD, data, sizeof( encryptedContent ), __sig, sizeof( __sig ), sigType, (const void **)certList,
                        nCerts );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: VerifySign with wrong title ID passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: VerifySign with wrong title ID failed (previous failure was expected)\n" );
    }
#endif

    ES_TEST_LOG( "Passed sign test\n" );
    rv = ES_ERR_OK;

end:
    return rv;
}  // NOLINT (readability/fn_size)

static ESError __testTicket()
{
    ESError rv = ES_ERR_OK;
    MemoryInputStream ticket( tkt6, sizeof( tkt6 ) );
    ETicket et;
    ESTicketId outTicketId;
    ESTicketVersion outTicketVersion;
    ESRightsId outRightsId;
    ESDeviceId outDeviceId;
    ESLicenseType outLicenseType;
    u32 outSize;
    bool outPreFlag;
    RESULT_NS( Result ) res;

    rv = et.Set( ticket, __certs, __nCerts, true );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to set and verify ticket, rv=%d\n", rv );
        goto end;
    }

    rv = et.GetSize( &outSize );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get size, rv=%d\n", rv );
        goto end;
    }

    rv = et.GetTicketId( &outTicketId );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get ticket id, rv=%d\n", rv );
        goto end;
    }

    rv = et.GetTicketVersion( &outTicketVersion );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get ticket version, rv=%d\n", rv );
        goto end;
    }

    rv = et.GetRightsId( &outRightsId );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get title id, rv=%d\n", rv );
        goto end;
    }

    rv = et.GetDeviceId( &outDeviceId );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get device id, rv=%d\n", rv );
        goto end;
    }

    rv = et.GetPreInstallationFlag( &outPreFlag );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get pre-installation flag, rv=%d\n", rv );
        goto end;
    }

    rv = et.GetLicenseType( &outLicenseType );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get license type, rv=%d\n", rv );
        goto end;
    }

    ES_TEST_LOG(
        "Size %u, Ticket ID 0x%016llx, Ticket version %hu, "
        "Device ID 0x%08x, "
        "Pre-installation flag %u, License type %hhu, "
        "Sys access mask[0] 0x%02x, Sys access mask[1] 0x%02x\n",
        outSize, outTicketId, outTicketVersion, outDeviceId, outPreFlag, outLicenseType);

    res = et.GetLastSystemResult();
    ES_TEST_LOG( "Summary %u\n", res.GetSummary() );

    // TODO: verify the retrieved values are correct

    /*
    * Negative tests for eTicket
    *
    * eTicket certificate is incorrect
    */
    // Modify signature type
    tkt6[0] ^= 0x10;
    rv = et.Set( ticket, __certs, __nCerts, true );
    EXPECT_EQ( rv, ES_ERR_INCORRECT_SIG_TYPE );
    if ( rv != ES_ERR_INCORRECT_SIG_TYPE )
    {
        ES_TEST_LOG( "Failed to check incorrect ticket, rv=%d\n", rv );
        goto end;
    }
    tkt6[0] ^= 0x10;

    // Modify signature data
    tkt6[0x4] ^= 0x10;
    rv = et.Set( ticket, __certs, __nCerts, true );
    EXPECT_EQ( rv, ES_ERR_VERIFICATION );
    if ( rv != ES_ERR_VERIFICATION )
    {
        ES_TEST_LOG( "Failed to check incorrect ticket, rv=%d\n", rv );
        goto end;
    }
    tkt6[0x4] ^= 0x10;

    // Modify format version
    tkt6[0x280] ^= 0x01;
    rv = et.Set( ticket, __certs, __nCerts, true );
    EXPECT_EQ( rv, ES_ERR_INCORRECT_TICKET_VERSION );
    if ( rv != ES_ERR_INCORRECT_TICKET_VERSION )
    {
        ES_TEST_LOG( "Failed to check incorrect ticket, rv=%d\n", rv );
        goto end;
    }
    tkt6[0x280] ^= 0x01;

    // Modify ticket data at start position(issuer).
    tkt6[0x140] ^= 0x01;
    rv = et.Set( ticket, __certs, __nCerts, true );
    EXPECT_EQ( rv, ES_ERR_ISSUER_NOT_FOUND );
    if ( rv != ES_ERR_ISSUER_NOT_FOUND )
    {
        ES_TEST_LOG( "Failed to check incorrect ticket, rv=%d\n", rv );
        goto end;
    }
    tkt6[0x140] ^= 0x01;

    // Modify ticket data at middle position(title key).
    tkt6[0x180] ^= 0x01;
    rv = et.Set( ticket, __certs, __nCerts, true );
    EXPECT_EQ( rv, ES_ERR_VERIFICATION );
    if ( rv != ES_ERR_VERIFICATION )
    {
        ES_TEST_LOG( "Failed to check incorrect ticket, rv=%d\n", rv );
        goto end;
    }
    tkt6[0x180] ^= 0x01;

    // Modify ticket data at end position.
    tkt6[0x2BF] ^= 0x01;
    rv = et.Set( ticket, __certs, __nCerts, true );
    EXPECT_EQ( rv, ES_ERR_VERIFICATION );
    if ( rv != ES_ERR_VERIFICATION )
    {
        ES_TEST_LOG( "Failed to check incorrect ticket, rv=%d\n", rv );
        goto end;
    }
    tkt6[0x2BF] ^= 0x01;

    // Check correct ticket.
    rv = et.Set( ticket, __certs, __nCerts, true );
    if ( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to check correct ticket, rv=%d\n", rv );
        goto end;
    }

    /*
    * Negative tests for certificate
    *
    * eTicket certificate is incorrect
    */
    caNewCert[0x0] ^= 0x01;
    rv = et.Set( ticket, __certs, __nCerts, true );
    EXPECT_EQ( rv, ES_ERR_CERT_INCORRECT_SIG_TYPE );
    if ( rv != ES_ERR_CERT_INCORRECT_SIG_TYPE )
    {
        ES_TEST_LOG( "Failed to check incorrect cert, rv=%d\n", rv );
        goto end;
    }
    caNewCert[0x0] ^= 0x01;

    caNewCert[0x3c8] ^= 0x01;
    rv = et.Set( ticket, __certs, __nCerts, true );
    EXPECT_EQ( rv, ES_ERR_CERT_VERIFICATION );
    if ( rv != ES_ERR_CERT_VERIFICATION )
    {
        ES_TEST_LOG( "Failed to check incorrect cert, rv=%d\n", rv );
        goto end;
    }
    caNewCert[0x3c8] ^= 0x01;

    xsNewCert[0x0] ^= 0x01;
    rv = et.Set( ticket, __certs, __nCerts, true );
    EXPECT_EQ( rv, ES_ERR_CERT_INCORRECT_SIG_TYPE );
    if ( rv != ES_ERR_CERT_INCORRECT_SIG_TYPE )
    {
        ES_TEST_LOG( "Failed to check incorrect cert, rv=%d\n", rv );
        goto end;
    }
    xsNewCert[0x0] ^= 0x01;

    xsNewCert[0x2c8] ^= 0x01;
    rv = et.Set( ticket, __certs, __nCerts, true );
    EXPECT_EQ( rv, ES_ERR_CERT_VERIFICATION );
    if ( rv != ES_ERR_CERT_VERIFICATION )
    {
        ES_TEST_LOG( "Failed to check incorrect cert, rv=%d\n", rv );
        goto end;
    }
    xsNewCert[0x2c8] ^= 0x01;

    // Check correct ticket.
    rv = et.Set( ticket, __certs, __nCerts, true );
    if ( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to check correct ticket, rv=%d\n", rv );
        goto end;
    }

    ES_TEST_LOG( "Passed ticket test\n" );

end:
    return rv;
}  // NOLINT (readability/fn_size)


static ESError __testMinTicket()
{
    ESError rv = ES_ERR_OK;
    MemoryInputStream ticket( tkt6, sizeof( tkt6 ) );
    ETicket et;

    rv = et.Set( ticket, __certs, __nCerts, true );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to set and verify ticket, rv=%d\n", rv );
        goto end;
    }

    ES_TEST_LOG( "Passed min ticket test\n" );

end:
    return rv;
}

static ESError __testCertificate()
{
    ESError rv = ES_ERR_OK;
    Certificate cert;
    void *outCaCpCerts[2], *outCaXsCerts[2], *outMergedCerts[4];
    u32 outNumCaCpCerts, outNumCaXsCerts, outNumMergedCerts;
    void *certList[MAX_CERT_LIST];
    u32 certSizes[MAX_CERT_LIST];
    u32 nCerts;
    // void *verifyChain[MAX_CERT_LIST];
    // u32 nVerifyChain;
    // u8 *subject;
    // u8 *issuer;

    // Parse CA-CP list
    rv = cert.GetNumCertsInList( caCpCerts, sizeof( caCpCerts ), &outNumCaCpCerts );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get the number of certs in CA-CP list, rv=%d\n", rv );
        goto end;
    }

    rv = cert.ParseCertList( caCpCerts, sizeof( caCpCerts ), outCaCpCerts, certSizes, &outNumCaCpCerts );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to parse CA-CP list, rv=%d\n", rv );
        goto end;
    }

    ES_TEST_LOG( "CA 0x%08x %u, CP 0x%08x %u, %u\n", reinterpret_cast<uintptr_t>(outCaCpCerts[0]), certSizes[0], reinterpret_cast<uintptr_t>(outCaCpCerts[1]), certSizes[1], outNumCaCpCerts );
    if( outNumCaCpCerts != 2 )
    {
        ES_TEST_LOG( "Failed to parse CA-CP list, found %d certs (expected 2)\n", outNumCaCpCerts );
        goto end;
    }

    // Parse CA-XS list
    rv = cert.GetNumCertsInList( caXsCerts, sizeof( caXsCerts ), &outNumCaXsCerts );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get the number of certs in CA-XS list, rv=%d\n", rv );
        goto end;
    }

    rv = cert.ParseCertList( caXsCerts, sizeof( caXsCerts ), outCaXsCerts, certSizes, &outNumCaXsCerts );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to parse CA-XS list, rv=%d\n", rv );
        goto end;
    }

    ES_TEST_LOG( "CA 0x%08x %u, XS 0x%08x %u, %u\n", reinterpret_cast<uintptr_t>(outCaXsCerts[0]), certSizes[0], reinterpret_cast<uintptr_t>(outCaXsCerts[1]), certSizes[1], outNumCaXsCerts );
    if( outNumCaXsCerts != 2 )
    {
        ES_TEST_LOG( "Failed to parse CA-XS list, found %d certs (expected 2)\n", outNumCaXsCerts );
        goto end;
    }

    // Merge the certs
    outNumMergedCerts = sizeof( outMergedCerts ) / sizeof( outMergedCerts[0] );

    rv = cert.MergeCerts( (const void **)outCaCpCerts, outNumCaCpCerts, (const void **)outCaXsCerts, outNumCaXsCerts, outMergedCerts, certSizes,
                          &outNumMergedCerts );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to merge certs, rv=%d\n", rv );
        goto end;
    }

    ES_TEST_LOG( "CA 0x%08x %u, CP 0x%08x %u, XS 0x%08x %u, %d\n", reinterpret_cast<uintptr_t>(outMergedCerts[0]), certSizes[0], reinterpret_cast<uintptr_t>(outMergedCerts[1]), certSizes[1],
                 reinterpret_cast<uintptr_t>(outMergedCerts[2]), certSizes[2], outNumMergedCerts );

    // TODO: is there a better way other than hard-wiring the numbers?
    if( outNumMergedCerts != 3 )
    {
        ES_TEST_LOG( "Failed to Merge CA-XS and CA-CP lists, returned %d certs (expected 3)\n", outNumMergedCerts );
        goto end;
    }

    /*
     * Now test the added functionality to handle x509 certificates
     */
    rv = cert.GetNumCertsInList( igwareNinCertList, sizeof( igwareNinCertList ), &nCerts );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get the number of certs in unified cert list, rv=%d\n", rv );
        goto end;
    }
    ES_TEST_LOG( "Passed: igware+NIN cert list has %d certs\n", nCerts );

    nCerts = MAX_CERT_LIST;
    rv = cert.ParseCertList( igwareNinCertList, sizeof( igwareNinCertList ), certList, certSizes, &nCerts );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to parse unified cert list, rv=%d\n", rv );
        goto end;
    }
    ES_TEST_LOG( "Passed: ParseCertList igware+NIN returns %d certs:\n", nCerts );
    {
        u32 i;
        for( i = 0; i < nCerts; i++ )
        {
            ES_TEST_LOG( "%08x: size %d, type %08x\n", reinterpret_cast<uintptr_t>(certList[i]), certSizes[i], *(u32 *)certList[i] );
        }
    }

    ES_TEST_LOG( "Passed certificate test\n" );

end:
    return rv;
}

#ifdef CTR_TEST
static ESError __testVerifyCerts()
{
    ESError rv = ES_ERR_OK;
    ETicketService es;
    MemoryInputStream data( encryptedContent, sizeof( encryptedContent ) );
    void *certList[MAX_CERT_LIST];
    u32 nCerts;
    void *verifyChain[MAX_CERT_LIST];
    u32 nVerifyChain;
    u8 *subject;
    u8 *issuer;
    MemoryInputStream tmdStream( tmd1_h256, sizeof( tmd1_h256 ) );
    u32 outSize;

    /*
     * Tests for the ETicketService::VerifyCerts API
     */

    /*
     * Clear the cert cache to start with a clean slate
     */
    rv = ESI_ClearCertCache();

    rv = es.GetDeviceCert( __devCert, sizeof( __devCert ) );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get device cert, rv=%d\n", rv );
        goto end;
    }

    rv = ESI_GetCertNames( __devCert, &issuer, &subject );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to parse device cert, rv=%d\n", rv );
        goto end;
    }

    /*
     * Now that x509 cert chain verification is supported, no restrictions here
     */
    ES_TEST_LOG( "Device cert issuer %s, subject %s\n", issuer, subject );
#ifdef USE_OBSOLETE_CODE
    if( strncmp( (const char *)subject, "NG", 2 ) != 0 )
    {
        ES_TEST_LOG( "Device cert is not NG cert, so skipping App Signer and device cert verification\n" );
        goto skip_signer_cert;
    }
#endif

    /*
     * XXX this test is redundant with __testSign
     */
    rv = es.Sign( ES_TEST_ACCESSOR_TITLE_ID, data, sizeof( encryptedContent ), __sig, sizeof( __sig ), __signerCert, sizeof( __signerCert ) );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to sign data, rv=%d\n", rv );
        goto end;
    }

    nCerts = 0;
    certList[nCerts++] = __signerCert;
    certList[nCerts++] = __devCert;
    certList[nCerts++] = msCert;
    certList[nCerts++] = xsCert;
    certList[nCerts++] = cpCert;
    certList[nCerts++] = caCert;
    certList[nCerts++] = ninDevCACert;
    verifyChain[0] = NULL;
    nVerifyChain = 0;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts returns rv=%d\n", rv );
        goto end;
    }
    ES_TEST_LOG( "Passed: VerifyCerts including at least one x509 format cert\n" );

    /*
     * Negative tests for argument parsing
     *
     * The total size of the two cert lists is limited to MAX_CERT_LIST
     */
    verifyChain[0] = caCert;
    nVerifyChain = MAX_CERT_LIST;
    certList[0] = cpCert;
    nCerts = 1;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_INVALID )
    {
        ES_TEST_LOG( "Fail: VerifyCerts should fail with bad args, but returns rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: VerifyCerts detected disallowed input (total cert count %d)\n", nCerts + nVerifyChain );

    nVerifyChain = MAX_CERT_LIST / 2;
    nCerts = MAX_CERT_LIST / 2 + 2;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_INVALID )
    {
        ES_TEST_LOG( "Fail: VerifyCerts should fail with bad args, but returns rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: VerifyCerts detected disallowed input (total cert count %d)\n", nCerts + nVerifyChain );

    nVerifyChain = 0;
    nCerts = MAX_CERT_LIST + 1;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_INVALID )
    {
        ES_TEST_LOG( "Fail: VerifyCerts should fail with bad args, but returns rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: VerifyCerts detected disallowed input (total cert count %d)\n", nCerts + nVerifyChain );

    nVerifyChain = 0;
    nCerts = 0;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_INVALID )
    {
        ES_TEST_LOG( "Fail: VerifyCerts should fail with bad args, but returns rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: VerifyCerts detected disallowed input (total cert count %d)\n", nCerts + nVerifyChain );

    nVerifyChain = MAX_CERT_LIST;
    nCerts = 0;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_INVALID )
    {
        ES_TEST_LOG( "Fail: VerifyCerts should fail with bad args, but returns rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: VerifyCerts detected disallowed input (nCerts %d)\n", nCerts );

    /*
     * Simple positive test case:  verify CP cert against the rest of the normal chain of trust
     */
    verifyChain[0] = caCert;
    verifyChain[1] = msCert;
    verifyChain[2] = xsCert;
    nVerifyChain = 3;

    certList[0] = cpCert;
    nCerts = 1;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts returns rv=%d\n", rv );
        goto end;
    }

    /*
     * Negative test
     *
     * Now test the case that one of the chain certs is bad, but not one we need
     */
    msCert[68] ^= 0x10;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts returns rv=%d\n", rv );
        goto end;
    }

    msCert[68] ^= 0x10;

    /*
     * Negative test
     *
     * Now corrupt a cert we need
     */
    caCert[28] ^= 0x8;

    /*
     * This time should succeed, because the CA cert is already cached
     */
    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts fails: cert cache does not override bad cert chain\n" );
        goto end;
    }

    /*
     * Negative test
     *
     * Now clear the cache so it should fail, since it has to verify the bogus CA cert.
     */
    rv = ESI_ClearCertCache();

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts succeeds with bad cert chain\n" );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    caCert[28] ^= 0x8;

    /*
     * Negative test
     *
     * Now check the case that the chain is good, but the new cert is bad.
     */
    cpCert[17] ^= 0x40;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts succeeds with bad test cert\n" );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    cpCert[17] ^= 0x40;

    /*
     * The empty verifyChain case is used when starting up from scratch with nothing but
     * the root public key
     */
    certList[0] = msCert;
    certList[1] = xsCert;
    certList[2] = cpCert;
    certList[3] = caCert;
    nCerts = 4;
    verifyChain[0] = NULL;
    nVerifyChain = 0;

    rv = ESI_ClearCertCache();

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts returns rv=%d\n", rv );
        goto end;
    }
    ES_TEST_LOG( "Passed: empty verify chain case\n" );

    /*
     * Negative test ... corrupt the verify chain
     */
    caCert[11] ^= 0x02;

    rv = ESI_ClearCertCache();

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts returns rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }

    caCert[11] ^= 0x02;
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Another variation is to get the CA cert into the cache and then try just one second level
     * cert with an empty verify chain
     */
    certList[0] = msCert;
    certList[1] = xsCert;
    certList[2] = caCert;
    nCerts = 3;
    verifyChain[0] = NULL;
    nVerifyChain = 0;

    rv = ESI_ClearCertCache();

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts line %d returns rv=%d\n", __LINE__, rv );
        goto end;
    }

    /*
     * The cache now contains the CA and XS cert, but not the CP cert
     */

    certList[0] = cpCert;
    nCerts = 1;
    verifyChain[0] = NULL;
    nVerifyChain = 0;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts line %d returns rv=%d\n", __LINE__, rv );
        goto end;
    }
    ES_TEST_LOG( "Passed: empty verify chain case with CA cert already cached\n" );

    /*
     * Negative test ... same thing as above but corrupt the leaf cert.
     *
     * There are actually two cases here:  at this point the correct CP cert is cached, so this test fails
     * because the new CP cert has the same issuer and subject name but does not match the hash
     * of the cert already cached.
     */
    cpCert[29] ^= 0x10;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts line %d returns rv=%d\n", __LINE__, rv );
        rv = ES_ERR_INVALID;
        goto end;
    }

    cpCert[29] ^= 0x10;
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Now try the previous steps again without getting the CP into the cache first
     */
    certList[0] = msCert;
    certList[1] = xsCert;
    certList[2] = caCert;
    nCerts = 3;
    verifyChain[0] = NULL;
    nVerifyChain = 0;

    rv = ESI_ClearCertCache();

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts line %d returns rv=%d\n", __LINE__, rv );
        goto end;
    }

    /*
     * The cache now contains the CA and XS cert, but not the CP cert
     */
    certList[0] = cpCert;
    nCerts = 1;
    verifyChain[0] = NULL;
    nVerifyChain = 0;

    /*
     * Corrupt the CP cert
     */
    cpCert[767] ^= 0x80;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts line %d returns rv=%d\n", __LINE__, rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    cpCert[767] ^= 0x80;
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Test thrashing certificate cache entries between TWL and CTR certs
     *
     * The CA cert is the case to test, since the CTR device certs do not share the TWL chain of trust
     * (except the root), but the CA+CP+XS certs are all new for CTR (with SHA256 signatures).  We can also
     * test thrashing back and forth between the SHA1 CP+XS and the SHA256 versions, even though that probably
     * doesn't actually happen in the real world.
     */
    rv = ESI_ClearCertCache();

    /*
     * Verify a TWL APP cert and device cert, which will prime the cache with the old cert chain
     */
    certList[0] = twlSignerCert;
    certList[1] = twlDevCert;
    certList[2] = ms8Cert;
    certList[3] = msCert;  // just for grins (not cached)
    certList[4] = cpCert;
    certList[5] = caCert;
    nCerts = 6;
    verifyChain[0] = NULL;
    nVerifyChain = 0;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts returns rv=%d\n", rv );
        goto end;
    }
    ES_TEST_LOG( "Passed: TWL signer cert verified with old cert chain\n" );

    /*
     * Now verify a TMD signed with the new cert chain
     */
    certList[0] = cpNewCert;
    certList[1] = caNewCert;
    nCerts = 2;

    /*
     * Now thrash back to TWL CA cert
     */
    certList[0] = twlDevCert;
    nCerts = 1;
    verifyChain[0] = ms8Cert;
    verifyChain[1] = caCert;
    nVerifyChain = 2;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts returns rv=%d\n", rv );
        goto end;
    }
    ES_TEST_LOG( "Passed: TWL device cert verified with old cert chain\n" );

    /*
     * Now verify a pure new CTR chain without clearing the cache
     */
    certList[0] = cpNewCert;
    certList[1] = xsNewCert;
    certList[2] = caNewCert;
    nCerts = 3;
    verifyChain[0] = NULL;
    nVerifyChain = 0;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts returns rv=%d\n", rv );
        goto end;
    }
    ES_TEST_LOG( "Passed: new CTR cert overrides the cache\n" );

    /*
     * Negative test:  corrupt the TWL device cert
     */
    certList[0] = twlSignerCert;
    nCerts = 1;
    verifyChain[0] = ms8Cert;
    verifyChain[1] = cpCert;
    verifyChain[2] = caCert;
    verifyChain[3] = twlDevCert;
    verifyChain[4] = caNewCert;
    nVerifyChain = 5;
    twlDevCert[383] ^= 0x08;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts line %d returns rv=%d\n", __LINE__, rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    twlDevCert[383] ^= 0x08;
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Negative test:  any chain including the PROD CA cert should fail
     */
    nCerts = 0;
    certList[nCerts++] = ctrProdSignerCert;
    certList[nCerts++] = ctrProdDeviceCert;
    certList[nCerts++] = ninProdCACert;
    verifyChain[0] = NULL;
    nVerifyChain = 0;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts for PROD cert chain returns rv=%d at line %d\n", rv, __LINE__ );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Negative test:  any chain including the PROD CA cert should fail
     */
    nCerts = 0;
    certList[nCerts++] = ctrProdSignerCert;
    certList[nCerts++] = ctrProdDeviceCert;
    nVerifyChain = 0;
    verifyChain[nVerifyChain++] = ninProdCACert;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts for PROD cert chain returns rv=%d at line %d\n", rv, __LINE__ );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Now test handling of the new GVM SP cert
     */
    rv = ESI_ClearCertCache();

    nCerts = 0;
    certList[nCerts++] = cpNewCert;
    certList[nCerts++] = xsNewCert;
    certList[nCerts++] = caNewCert;
    certList[nCerts++] = spNewCert;
    verifyChain[0] = NULL;
    nVerifyChain = 0;

    rv = es.VerifyCerts( (const void **)certList, nCerts, (const void **)verifyChain, nVerifyChain );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: VerifyCerts including GVM SP cert returns rv=%d\n", rv );
        goto end;
    }
    ES_TEST_LOG( "Passed: SP cert verifies\n" );

    ES_TEST_LOG( "Passed verify certs test\n" );
    rv = ES_ERR_OK;

end:
    return rv;
}  // NOLINT (readability/fn_size)


static ESError __testVerifyTwlSign()
{
    ESError rv = ES_ERR_OK;
    ETicketService es;
    MemoryInputStream dataReader( __data, sizeof( __data ) );

    /*
     * The verify TWL sign test performs positive and negative tests
     * for the case of importing data signed by a TWL device.
     *
     *  1) Verify the signature signed by a TWL
     */

    __certs[ES_TEST_CERT_CHAIN_LEN - 1] = ms8Cert; /* MS cert for TWL is different */
    __certs[ES_TEST_CERT_CHAIN_LEN] = twlDevCert;
    __certs[ES_TEST_CERT_CHAIN_LEN + 1] = twlSignerCert;

    rv = es.VerifySign( ES_TWL_TITLE_ID, dataReader, sizeof( __data ), twlSignature, sizeof( twlSignature ), ES_SIG_TYPE_ECC_SHA1, __certs,
                        __nCerts + 2 );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: verify Twl signature, rv=%d\n", rv );
        goto end;
    }

    /*
     * Negative test:  corrupt the input data to the signature before the verify
     */
    dataReader.TrySetPosition( 0LL );
    __data[7] ^= 0x8;

    rv = es.VerifySign( ES_TWL_TITLE_ID, dataReader, sizeof( __data ), twlSignature, sizeof( twlSignature ), ES_SIG_TYPE_ECC_SHA1, __certs,
                        __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: verify Twl signature should have failed at line %d\n", __LINE__ );
        rv = ES_ERR_INVALID;
        goto end;
    }

    __data[7] ^= 0x8;
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Negative test:  leave out the last byte of the input data to the signature before the verify
     */
    dataReader.TrySetPosition( 0LL );

    rv = es.VerifySign( ES_TWL_TITLE_ID, dataReader, sizeof( __data ) - 1, twlSignature, sizeof( twlSignature ), ES_SIG_TYPE_ECC_SHA1, __certs,
                        __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: verify Twl signature should have failed at line %d\n", __LINE__ );
        rv = ES_ERR_INVALID;
        goto end;
    }

    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Negative test:  corrupt the signature passed in to verify
     */
    dataReader.TrySetPosition( 0LL );
    twlSignature[42] ^= 0x2;

    rv = es.VerifySign( ES_TWL_TITLE_ID, dataReader, sizeof( __data ), twlSignature, sizeof( twlSignature ), ES_SIG_TYPE_ECC_SHA1, __certs,
                        __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: verify Twl signature should have failed at line %d\n", __LINE__ );
        rv = ES_ERR_INVALID;
        goto end;
    }

    twlSignature[42] ^= 0x2;
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Negative test:  corrupt the Twl device certificate passed in to verify
     */
    dataReader.TrySetPosition( 0LL );
    twlDevCert[147] ^= 0x1;

    rv = es.VerifySign( ES_TWL_TITLE_ID, dataReader, sizeof( __data ), twlSignature, sizeof( twlSignature ), ES_SIG_TYPE_ECC_SHA1, __certs,
                        __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: verify Twl signature should have failed at line %d\n", __LINE__ );
        rv = ES_ERR_INVALID;
        goto end;
    }

    twlDevCert[147] ^= 0x1;
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Negative test:  corrupt the Twl signer certificate passed in to verify
     */
    dataReader.TrySetPosition( 0LL );
    twlSignerCert[383] ^= 0x20;

    rv = es.VerifySign( ES_TWL_TITLE_ID, dataReader, sizeof( __data ), twlSignature, sizeof( twlSignature ), ES_SIG_TYPE_ECC_SHA1, __certs,
                        __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: verify Twl signature should have failed at line %d\n", __LINE__ );
        rv = ES_ERR_INVALID;
        goto end;
    }

    twlSignerCert[383] ^= 0x20;
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Negative test:  use the wrong Title ID and check that it fails
     */
    dataReader.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TWL_TITLE_ID_BAD1, dataReader, sizeof( __data ), twlSignature, sizeof( twlSignature ), ES_SIG_TYPE_ECC_SHA1, __certs,
                        __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: verify Twl signature should have failed at line %d\n", __LINE__ );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Negative test:  use the wrong Title ID and check that it fails
     */
    dataReader.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TWL_TITLE_ID_BAD2, dataReader, sizeof( __data ), twlSignature, sizeof( twlSignature ), ES_SIG_TYPE_ECC_SHA1, __certs,
                        __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: verify Twl signature should have failed at line %d\n", __LINE__ );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    /*
     * Negative test:  use the wrong Title ID and check that it fails
     */
    dataReader.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TWL_TITLE_ID_BAD3, dataReader, sizeof( __data ), twlSignature, sizeof( twlSignature ), ES_SIG_TYPE_ECC_SHA1, __certs,
                        __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: verify Twl signature should have failed at line %d\n", __LINE__ );
        rv = ES_ERR_INVALID;
        goto end;
    }
    ES_TEST_LOG( "Passed: previous failure was expected\n" );

    ES_TEST_LOG( "Passed TWL->CTR VerifySign test\n" );
    rv = ES_ERR_OK;

end:
    return rv;
}  // NOLINT (readability/fn_size)


static ESError __testVerifyCtrSign()
{
    ESError rv = ES_ERR_OK;
    ETicketService es;
    MemoryInputStream dataReader( encryptedContent, sizeof( encryptedContent ) );
    MemoryInputStream prodDataReader( __data, sizeof( __data ) );
    ESDeviceId devId;
    u8 *subject;
    u8 *issuer;

    /*
     * The verify CTR sign test performs positive and negative tests
     * for the case of importing data signed by another CTR device.
     *
     * The sample data was generated by emulating device CT0000002F-01.
     */

    /*
     * Check that the executing device is different
     */
    rv = es.GetDeviceCert( __devCert, sizeof( __devCert ) );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get device cert, rv=%d\n", rv );
        goto end;
    }

    rv = ESI_GetCertNames( __devCert, &issuer, &subject );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: ESI_GetCertNames returns error, rv=%d\n", rv );
        goto end;
    }

    rv = es.GetDeviceId( &devId );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get device ID, rv=%d\n", rv );
        goto end;
    }

    if( devId == 0x0000002f || strncmp( (const char *)subject, "CT0000002F-01", sizeof( IOSCName ) == 0 ) )
    {
        ES_TEST_LOG( "__testVerifyCtrSign: WARNING test uses same deviceId, but should be different\n" );
    }

    __certs[ES_TEST_CERT_CHAIN_LEN - 1] = ninDevCACert; /* MS cert chain for CTR is Nintendo's */
    __certs[ES_TEST_CERT_CHAIN_LEN] = ctrDevCert;
    __certs[ES_TEST_CERT_CHAIN_LEN + 1] = ctrSignerCert;

    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, dataReader, sizeof( encryptedContent ), ctrSignature, sizeof( ctrSignature ),
                        ES_SIG_TYPE_ECC_SHA256, __certs, __nCerts + 2 );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Fail: verify CTR signature, rv=%d\n", rv );
        goto end;
    }
    ES_TEST_LOG( "Passed: VerifySign of signature from a different CTR device\n" );

    /*
     * Negative test: corrupt the signature
     */
    ctrSignature[59] ^= 0x80;
    dataReader.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, dataReader, sizeof( encryptedContent ), ctrSignature, sizeof( ctrSignature ),
                        ES_SIG_TYPE_ECC_SHA256, __certs, __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: verify of corrupted ctr signature passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: corrupted CTR signature detected (previous failure was expected)\n" );
    }
    ctrSignature[59] ^= 0x80;

    /*
     * Negative test: corrupt the data input to the signature calculation
     */
    encryptedContent[13] ^= 0x1;
    dataReader.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, dataReader, sizeof( encryptedContent ), ctrSignature, sizeof( ctrSignature ),
                        ES_SIG_TYPE_ECC_SHA256, __certs, __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: verify of corrupted ctr signature input passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: corrupted CTR signature input detected (previous failure was expected)\n" );
    }
    encryptedContent[13] ^= 0x1;

    /*
     * Negative test: leave out the last byte of the data input to the signature calculation
     */
    dataReader.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, dataReader, sizeof( encryptedContent ) - 1, ctrSignature, sizeof( ctrSignature ),
                        ES_SIG_TYPE_ECC_SHA256, __certs, __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: verify of short ctr signature input passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: short CTR signature input detected (previous failure was expected)\n" );
    }

    /*
     * Negative test: corrupt the signer certificate
     */
    ctrSignerCert[379] ^= 0x40;
    dataReader.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, dataReader, sizeof( encryptedContent ), ctrSignature, sizeof( ctrSignature ),
                        ES_SIG_TYPE_ECC_SHA256, __certs, __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: verify of corrupted ctr signer certificate passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: corrupted CTR signer certificate detected (previous failure was expected)\n" );
    }
    ctrSignerCert[379] ^= 0x40;

    /*
     * Negative test: corrupt the signing device certificate
     */
    ctrDevCert[139] ^= 0x10;
    dataReader.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, dataReader, sizeof( encryptedContent ), ctrSignature, sizeof( ctrSignature ),
                        ES_SIG_TYPE_ECC_SHA256, __certs, __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: verify of corrupted ctr signing device certificate passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: corrupted CTR signing device certificate detected (previous failure was expected)\n" );
    }
    ctrDevCert[139] ^= 0x10;

    /*
     * Negative test: bad title ID should fail
     */
    dataReader.TrySetPosition( 0LL );
    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID_BAD, dataReader, sizeof( encryptedContent ), ctrSignature, sizeof( ctrSignature ),
                        ES_SIG_TYPE_ECC_SHA256, __certs, __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: CTR VerifySign with wrong title ID passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: CTR VerifySign with wrong title ID failed (previous failure was expected)\n" );
    }

    /*
     * Negative test:  signature generated by a PROD device should be rejected
     */
    __certs[ES_TEST_CERT_CHAIN_LEN - 1] = ninProdCACert; /* MS cert chain for CTR is Nintendo's */
    __certs[ES_TEST_CERT_CHAIN_LEN] = ctrProdDeviceCert;
    __certs[ES_TEST_CERT_CHAIN_LEN + 1] = ctrProdSignerCert;

    rv = es.VerifySign( ES_TEST_ACCESSOR_TITLE_ID, prodDataReader, sizeof( __data ), ctrProdSignature, sizeof( ctrProdSignature ),
                        ES_SIG_TYPE_ECC_SHA256, __certs, __nCerts + 2 );
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: CTR VerifySign from PROD device passed, rv=%d\n", rv );
        rv = ES_ERR_INVALID;
        goto end;
    }
    else
    {
        ES_TEST_LOG( "Passed: CTR VerifySign from PROD device failed (previous failure was expected)\n" );
    }

    ES_TEST_LOG( "Passed CTR->CTR VerifySign test\n" );
    rv = ES_ERR_OK;

end:
    return rv;
}  // NOLINT (readability/fn_size)
#endif

static ESError __testInternalCertFuncs()
{
    Certificate cert;
    ESError rv = ES_ERR_OK;
    u8 *subject;
    u8 *issuer;
    u32 certSize;
    u8 *pubKey;
    u32 pubKeySize;
    u32 exponent;

    /*
     * Tests for some of the internal ESI library functions dealing with certificate handling
     */
    rv = cert.GetCertSize( caNewCert, &certSize );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: cert.GetCertSize returns error, rv=%d\n", rv );
        goto end;
    }
    if( certSize != sizeof( caNewCert ) )
    {
        ES_TEST_LOG( "Failed: cert.GetCertSize returns size %d, expect %d\n", certSize, sizeof( caNewCert ) );
        goto end;
    }

    rv = cert.GetCertSize( cpNewCert, &certSize );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: cert.GetCertSize returns error, rv=%d\n", rv );
        goto end;
    }
    if( certSize != sizeof( cpNewCert ) )
    {
        ES_TEST_LOG( "Failed: cert.GetCertSize returns size %d, expect %d\n", certSize, sizeof( cpNewCert ) );
        goto end;
    }

    rv = cert.GetCertSize( msCert, &certSize );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: cert.GetCertSize returns error, rv=%d\n", rv );
        goto end;
    }
    if( certSize != sizeof( msCert ) )
    {
        ES_TEST_LOG( "Failed: cert.GetCertSize returns size %d, expect %d\n", certSize, sizeof( msCert ) );
        goto end;
    }

    rv = cert.GetCertNames( caNewCert, &issuer, &subject );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: cert.GetCertNames returns error, rv=%d\n", rv );
        goto end;
    }
    if( strncmp( (const char *)issuer, "Root", 64 ) != 0 )
    {
        ES_TEST_LOG( "Failed: cert.GetCertNames returns wrong issuer: %s\n", issuer );
        goto end;
    }
    if( strncmp( (const char *)subject, "CA000000", 8 ) != 0 )
    {
        ES_TEST_LOG( "Failed: cert.GetCertNames returns wrong subject: %s\n", subject );
        goto end;
    }

    rv = ESI_GetCertPubKey( caNewCert, &pubKey, &pubKeySize, &exponent );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: ESI_GetCertPubKey returns error, rv=%d\n", rv );
        goto end;
    }
    if( pubKeySize != sizeof( IOSCRsaPublicKey2048 ) )
    {
        ES_TEST_LOG( "Failed: ESI_GetCertPubKey returns size %d, expect %d\n", pubKeySize, sizeof( IOSCRsaPublicKey2048 ) );
        goto end;
    }

    rv = ESI_GetCertPubKey( xsNewCert, &pubKey, &pubKeySize, &exponent );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: ESI_GetCertPubKey returns error, rv=%d\n", rv );
        goto end;
    }
    if( pubKeySize != sizeof( IOSCRsaPublicKey2048 ) )
    {
        ES_TEST_LOG( "Failed: ESI_GetCertPubKey returns size %d, expect %d\n", pubKeySize, sizeof( IOSCRsaPublicKey2048 ) );
        goto end;
    }

    rv = ESI_GetCertPubKey( msCert, &pubKey, &pubKeySize, &exponent );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed: ESI_GetCertPubKey returns error, rv=%d\n", rv );
        goto end;
    }
    if( pubKeySize != sizeof( IOSCEccPublicKey ) )
    {
        ES_TEST_LOG( "Failed: ESI_GetCertPubKey returns size %d, expect %d\n", pubKeySize, sizeof( IOSCEccPublicKey ) );
        goto end;
    }

    /*
     * XXX Add x509 test cases
     */

    ES_TEST_LOG( "Passed Internal Certificate Function test\n" );
    rv = ES_ERR_OK;
end:
    return rv;
}

TEST( SanityTest, Sanity )
{
    ESError rv = ES_ERR_OK;

    rv = __testSetup();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testMemoryStorage();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testPacking();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testSign();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testTicket();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testMinTicket();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testCertificate();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }
#ifdef CTR_TEST
    rv = __testVerifyCerts();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testVerifyTwlSign();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testVerifyCtrSign();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }
#endif
    rv = __testInternalCertFuncs();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testCleanup();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

end:
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "***** Passed sanity tests *****\n" );
    }

    return;
}  // NOLINT (readability/fn_size)
