﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *  Copyright 2010 iGware Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of iGware Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of iGware Inc.
 */

/*
 *               Copyright (C) 2010, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */
#include <nnt/nntest.h>
#include <nn/fs.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>

#include <nn/escore/escore.h>
#include <nn/escorei/esi.h>
#include <nn/publish/publish.h>
#include <nn/iosc/iosc.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

#include <nnt/escoreUtil/testEscore_util_istorage.h>

USING_ESCORE_UTIL_NAMESPACE
USING_ES_NAMESPACE
USING_ISTORAGE_NAMESPACE
USING_PUBLISH_NAMESPACE

#include "testEs_Tkt.cpp"

#define VERBOSE

#define LOG_MESSAGE( ... )             \
    NN_LOG( "LOG: " __VA_ARGS__ ); \
    NN_LOG( "\n" )
#define LOG_FAIL( ... )                 \
    NN_LOG( "FAIL: " __VA_ARGS__ ); \
    NN_LOG( "\n" )
#define LOG_PASS( ... )                 \
    NN_LOG( "PASS: " __VA_ARGS__ ); \
    NN_LOG( "\n" )

#define CHECK_RESULT( exp, rval )                                                          \
    {                                                                                      \
        if( ( rv = ( exp ) ) != ( rval ) )                                                 \
        {                                                                                  \
            LOG_FAIL( "Unexpected result at line %d: " #exp " returns %d", __LINE__, rv ); \
            ADD_FAILURE();                                                                 \
            if( rv == ES_ERR_OK )                                                          \
            {                                                                              \
                rv = ES_ERR_INVALID;                                                       \
            }                                                                              \
            goto out;                                                                      \
        }                                                                                  \
        else                                                                               \
        {                                                                                  \
            SUCCEED();                                                                     \
        }                                                                                  \
    }

static char __ticketFile[] = "00000000.tik";

static u8 ticketBuf[32 * 1024];

const int MAX_NUM_CERTS = 12;
static u8 certBuffer[10 * 1024];

#define USE_SHA256_CERTS
#if defined( USE_SHA256_CERTS )
#if defined(CTR)
static const char *certFile = "cert_dpki_h256.sys";
static const u8 issuerName[] = "Root-CA00000004-XS00000009";
#endif
static const char *certFile = "cert_dpki_h256_nx.sys";
static const u8 issuerName[] = "Root-CA00000004-XS00000020";
#else
static const char *certFile = "cert_dpki.sys";
static const u8 issuerName[] = "Root-CA00000002-XS00000006";
#endif
static void *certList[MAX_NUM_CERTS];
static u32 certSizes[MAX_NUM_CERTS];
static u32 nCerts;

Certificate certs;
ETicketService ets;

static u8 __dpTicketBuf[32 * 1024] ATTR_SHA256_ALIGN;

static u8 aesKey[] = {
    0xa1, 0xa2, 0xa3, 0xa4, 0xb1, 0xb2, 0xb3, 0xb4, 0xc1, 0xc2, 0xc3, 0xc4, 0xd1, 0xd2, 0xd3, 0xd4,
};

#if defined( USE_SHA256_CERTS )
#if defined(CTR)
static char rsaPubModFile[] = "xs9_dpki_rsa.pubMod";
static char rsaPrivExpFile[] = "xs9_dpki_rsa.privExp";
#else
static char rsaPubModFile[] = "xs20_dpki_rsa.pubMod";
static char rsaPrivExpFile[] = "xs20_dpki_rsa.privExp";
#endif
#else
static char rsaPubModFile[] = "xs_dpki_rsa.pubMod";
static char rsaPrivExpFile[] = "xs_dpki_rsa.privExp";
#endif
static u8 rsaPubMod[256];
static u8 rsaPrivExp[256];

static char xsEccPrivFile[] = "xs_dpki.eccPvtKey";
static char xsEccPubFile[] = "xs_dpki.eccPubKey";
static IOSCEccPrivateKey xsEccPrivKey;
static IOSCEccPublicKey xsEccPubKey;

static IOSCPublicKeyHandle caPubKeyH;
static IOSCPublicKeyHandle xsPubKeyH;
static IOSCPublicKeyHandle cpPubKeyH;

static char devCertFile[] = "etd_0015_dev_cert.raw";
static IOSCEccEccCert devCert;

#ifdef USE_DPKI_COMMON
static char commonKeyFile[] = "common_dpki.aesKey";
#else
#ifdef USE_ETD_COMMON_KEY1
static char commonKeyFile[] = "common_etd_devkey1_npki.aesKey";
#else
static char commonKeyFile[] = "common_etd_devkey0_npki.aesKey";
#endif
#endif
static u8 commonKey[16];
static IOSCSecretKeyHandle hCommonKey;


static void dumpData( const char *msg, u8 *array, int size )
{
    int i;

    NN_LOG( "%s", msg );
    for( i = 0; i < size; i++ )
    {
        if( i % 16 == 0 )
        {
            NN_LOG( "\n" );
        }
        NN_LOG( "%02X ", array[i] );
    }
    NN_LOG( "\n" );
}


/*
 * Look in canonical places for the PKI Data Files
 */
static char *makePath( const char *relPath )
{
    static char pathName[256];

    strncpy( pathName, "..\\..\\PkiData\\", sizeof( pathName ) );
    strncat( pathName, relPath, sizeof( pathName ) - strlen(relPath) - 1);

    return pathName;
}

static const char *makeAbsPath( const char *fileName )
{
    static std::string path;

    //#define NNT_ES_SIGLO_ROOT set by nact script or vcxporj to the root path of Siglo.
    const char* kResourcePath = "/Tests/Escore/Sources/Tests/PublishRegression/testEs_PublishRegression/";
    path = std::string(NNT_ES_SIGLO_ROOT) + std::string(kResourcePath) + fileName;

    return path.c_str();
}

static int readFile( const char *fileName, void* buffer, size_t size, u32* readSize = NULL)
{
    ESError rv = ES_ERR_OK;

    nn::fs::FileHandle file;
    nn::Result result;
    result = nn::fs::OpenFile(&file, makeAbsPath(makePath(fileName)), nn::fs::OpenMode_Read);
    if ( result.IsFailure() )
    {
        NN_LOG( "Open failed for %s\n", fileName);
        rv = ES_ERR_FAIL;
        return rv;
    }

    size_t fileSize = 0;
    result = nn::fs::ReadFile(&fileSize, file, 0, buffer, size, nn::fs::ReadOption());
    if (fileSize == 0)
    {
        NN_LOG( "read failed for %s\n", fileName);
        rv = ES_ERR_FAIL;
    }
    if (readSize != NULL)
    {
        *readSize = static_cast<u32>(fileSize);
    }

    nn::fs::CloseFile(file);
    return rv;
}

static void writeFile( const char *fileName, const void* buffer, size_t size)
{
    nn::fs::FileHandle file;
    nn::Result result;
    nn::fs::DeleteFile(makeAbsPath(fileName));
    result = nn::fs::CreateFile(makeAbsPath(fileName), size);
    if ( result.IsFailure() )
    {
        LOG_MESSAGE( "create failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }

    result = nn::fs::OpenFile(&file, makeAbsPath(fileName), nn::fs::OpenMode_Write);
    if ( result.IsFailure() )
    {
        LOG_MESSAGE( "open failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }

    result = nn::fs::WriteFile(file, 0, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    if (result.IsFailure())
    {
        LOG_MESSAGE( "write failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }
    nn::fs::CloseFile(file);
}

/*
 * This program acts as the XS server, so it needs both the
 * RSA key pair (used for signing tickets) and the ECC key pair,
 * used for doing ECDH in ticket personalization.
 */
static void initKeys()
{
    ESError rv = ES_ERR_OK;

    rv = readFile( rsaPubModFile, rsaPubMod, sizeof( rsaPubMod ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( rsaPrivExpFile, rsaPrivExp, sizeof( rsaPrivExp ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( xsEccPrivFile, xsEccPrivKey, sizeof( xsEccPrivKey ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( xsEccPubFile, xsEccPubKey, sizeof( xsEccPubKey ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( commonKeyFile, commonKey, sizeof( commonKey ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = IOSC_CreateObject( &hCommonKey, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != IOSC_ERROR_OK )
    {
        NN_LOG( "Failed to create secret key object, rv=%d\n", rv );
        return;
    }

    rv = IOSC_ImportSecretKey( hCommonKey, 0, 0, IOSC_NOSIGN_NOENC, NULL, NULL, commonKey );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != IOSC_ERROR_OK )
    {
        NN_LOG( "Failed to import common key, rv=%d\n", rv );
        (void)IOSC_DeleteObject( hCommonKey );
        return;
    }
}


static void dumpCerts()
{
    ESError rv = ES_ERR_OK;
    u32 i;
    u8 *issuer;
    u8 *subject;
    u8 *pubKey;
    u32 pubKeySize;
    u32 certSize;
    u32 exponent;

    for( i = 0; i < nCerts; i++ )
    {
        rv = ESI_GetCertSize( certList[i], &certSize );
        rv = ESI_GetCertNames( certList[i], &issuer, &subject );
        rv = ESI_GetCertPubKey( certList[i], &pubKey, &pubKeySize, &exponent );

        LOG_MESSAGE( "cert[%d]: size %d, issuer %s, subject %s, keySize %d, exponent %d", i, certSize, issuer, subject, pubKeySize, exponent );
    }

    return;
}


static void findCerts()
{
    ESError rv = ES_ERR_OK;
    u32 i;
    u8 *issuer;
    u8 *subject;
    IOSCPublicKeyHandle importKeyHandle;
    IOSCPublicKeyHandle signKeyHandle;
    bool foundCa = false;
    bool foundCp = false;
    bool foundXs = false;
    bool secondPass = false;
    bool match;

/*
 * Need to find the CA cert before importing the others.
 * This logic handles the case that the cert list is not in
 * the "correct" order.
 */
again:
    for( i = 0; i < nCerts; i++ )
    {
        match = false;
        rv = ESI_GetCertNames( certList[i], &issuer, &subject );
        EXPECT_EQ( rv, ES_ERR_OK );
        if( rv != ES_ERR_OK )
        {
            NN_LOG( "GetCertNames failed (rv %d)\n", rv );
            return;
        }
        NN_LOG( "Cert[%d] Subject name %s\n", i, subject );
        if( strncmp( (const char *)subject, "CA", 2 ) == 0 )
        {
            if( foundCa )
            {
                continue;
            }
            importKeyHandle = caPubKeyH;
            signKeyHandle = IOSC_ROOT_KEY_HANDLE;
            match = true;
            foundCa = true;
        }
        else if( strncmp( (const char *)subject, "XS", 2 ) == 0 )
        {
            if( !foundCa || foundXs )
            {
                continue;
            }
            importKeyHandle = xsPubKeyH;
            signKeyHandle = caPubKeyH;
            match = true;
            foundXs = true;
        }
        else if( strncmp( (const char *)subject, "CP", 2 ) == 0 )
        {
            if( !foundCa || foundCp )
            {
                continue;
            }
            importKeyHandle = cpPubKeyH;
            signKeyHandle = caPubKeyH;
            match = true;
            foundCp = true;
        }
        else
        {
            NN_LOG( "illegal subject %s", (const char *)subject );
            return;
        }
        if( match )
        {
            rv = IOSC_ImportCertificate( (u8 *)certList[i], signKeyHandle, importKeyHandle );
            EXPECT_EQ( rv, ES_ERR_OK );
            if( rv != IOSC_ERROR_OK )
            {
                NN_LOG( "Import of %s Cert failed (rv %d)\n", subject, rv );
            }
            else
            {
                NN_LOG( "Import of %s Cert OK\n", subject );
            }
        }
    }

    /*
     * Handle the case that the certs are in the wrong order
     */
    if( !secondPass )
    {
        if( !foundCa )
        {
            /* No hope in this case */
            NN_LOG( "CA Cert not found\n" );
        }
        else if( !foundXs || !foundCp )
        {
            secondPass = true;
            goto again;
        }
    }
    else if( !foundCa || !foundXs || !foundCp )
    {
        NN_LOG( "Some required certs missing!\n" );
    }
}


static void initCerts()
{
    ESError rv = ES_ERR_OK;
    u32 fileSize = 0;
    readFile( certFile, certBuffer, sizeof( certBuffer ), &fileSize);

    nCerts = MAX_NUM_CERTS;
    rv = certs.GetNumCertsInList( certBuffer, fileSize, &nCerts );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        nCerts = 0;
        goto out;
    }

    if( nCerts > MAX_NUM_CERTS )
    {
        NN_LOG( "Too many certs: %d\n", nCerts );
        goto out;
    }

    rv = certs.ParseCertList( certBuffer, fileSize, certList, certSizes, &nCerts );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        nCerts = 0;
        goto out;
    }
    NN_LOG( "ParseCerts found %d\n", nCerts );

    dumpCerts();

    /*
     * Now verify the lot
     */
    rv = ets.VerifyCerts( (const void **)certList, nCerts, NULL, 0 );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        NN_LOG( "VerifyCerts failed (rv %d)\n", rv );
        /* Maybe a little extreme? */
        exit( rv );
    }

    rv = IOSC_CreateObject( &caPubKeyH, IOSC_PUBLICKEY_TYPE, IOSC_RSA2048_SUBTYPE );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        NN_LOG( "Create failed for caPubKey (rv %d)\n", rv );
        goto out;
    }
    rv = IOSC_CreateObject( &xsPubKeyH, IOSC_PUBLICKEY_TYPE, IOSC_RSA2048_SUBTYPE );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        NN_LOG( "Create failed for xsPubKey (rv %d)\n", rv );
        goto out;
    }
    rv = IOSC_CreateObject( &cpPubKeyH, IOSC_PUBLICKEY_TYPE, IOSC_RSA2048_SUBTYPE );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        NN_LOG( "Create failed for cpPubKey (rv %d)\n", rv );
        goto out;
    }

    /*
     * Find and import the CA, ticket server and content publisher certs to
     * get the public keys
     */
    findCerts();

out:
    return;
}


/*
 * The device cert is used by the XS server to obtain the device ID
 * and the device public key for ticket personalization
 */
static void loadDevCert()
{
    (void)readFile( devCertFile, (u8 *)&devCert, sizeof( devCert ) );
}

static const char *licName[] = {"Permanent", "Demo", "Trial", "Rental", "Subscription", "Service"};
static const char *limName[] = {"<Invalid>", "Duration", "AbsTime", "NumTitles", "NumLaunch", "ElapsedTime", "<Invalid>", "<Invalid>", "<Invalid>"};
static const char *secName[] = {"<Invalid>",           "Permanent",    "Subscription",    "Content Access",
                                "Content Consumption", "Access Title", "Limited Resource"};

static void dumpRightsId(ESRightsId rightsid)
{
    for (int i = 0; i < sizeof(ESRightsId); i++)
    {
        NN_LOG("%02X", rightsid[i]);
    }
    NN_LOG( "\n" );
}

static void dumpV1SectPermanent( ESV1PermanentRecord *v1SectData, u32 nrec )
{
    char buf[128];

    for( u32 n = 0; n < nrec; n++, v1SectData++ )
    {
        u8 *ptr = v1SectData->referenceId;
        buf[0] = 0;
        for( u32 m = 0; m < sizeof( v1SectData->referenceId ); m++ )
        {
            u32 len = static_cast<u32>(strnlen( buf, sizeof( buf ) ));
            nn::util::SNPrintf( buf + len, sizeof( buf ) - len, "%02x", *ptr++ );
        }
        LOG_MESSAGE( "    refId %s Attr %08x", buf, ntohl( v1SectData->referenceIdAttr ) );
    }
}

static void dumpV1SectSubscription( ESV1SubscriptionRecord *v1SectData, u32 nrec )
{
    char buf[128];

    for( u32 n = 0; n < nrec; n++, v1SectData++ )
    {
        u8 *ptr = v1SectData->referenceId;
        buf[0] = 0;
        for( u32 m = 0; m < sizeof( v1SectData->referenceId ); m++ )
        {
            u32 len = static_cast<u32>(strnlen( buf, sizeof( buf ) ));
            nn::util::SNPrintf( buf + len, sizeof( buf ) - len, "%02x", *ptr++ );
        }
        LOG_MESSAGE( "    limit %d refId %s Attr %08x", ntohl( v1SectData->limit ), buf, ntohl( v1SectData->referenceIdAttr ) );
    }
}

static void dumpV1SectContent( ESV1ContentRecord *v1SectData, u32 nrec )
{
    char buf[128];
    u32 offset;

    for( u32 n = 0; n < nrec; n++, v1SectData++ )
    {
        offset = ntohl( v1SectData->offset );
        LOG_MESSAGE( "    Access bit mask starting at content index %d:", offset );
        u8 *ptr = v1SectData->accessMask;
        for( u32 i = 0; i < sizeof( v1SectData->accessMask ); i += 16 )
        {
            nn::util::SNPrintf( buf, sizeof( buf ), "    Index %04d: ", offset + ( i << 3 ) );
            for( u32 j = 0; j < 16; j++ )
            {
                u32 len = static_cast<u32>(strnlen( buf, sizeof( buf ) ));
                nn::util::SNPrintf( buf + len, sizeof( buf ) - len, "%02x ", *ptr++ );
            }
            LOG_MESSAGE( "%s", buf );
        }
    }
}

static void dumpV1SectConsumption( ESV1ContentConsumptionRecord *v1SectData, u32 nrec )
{
    for( u32 n = 0; n < nrec; n++, v1SectData++ )
    {
        LOG_MESSAGE( "    contIndex %d limType %s value %d", ntohs( v1SectData->index ), limName[ntohs( v1SectData->code )],
                     ntohl( v1SectData->limit ) );
    }
}

static void dumpV1SectAccess( ESV1AccessTitleRecord *v1SectData, u32 nrec )
{
    for( u32 n = 0; n < nrec; n++, v1SectData++ )
    {
        LOG_MESSAGE( "    titleId %016llx titleIdMask %016llx", ntohll( v1SectData->accessTitleId ), ntohll( v1SectData->accessTitleMask ) );
    }
}

static void dumpV1SectLimitedResource( ESV1LimitedResourceRecord *v1SectData, u32 nrec )
{
    char buf[128];

    for( u32 n = 0; n < nrec; n++, v1SectData++ )
    {
        u8 *ptr = v1SectData->referenceId;
        buf[0] = 0;
        for( u32 m = 0; m < sizeof( v1SectData->referenceId ); m++ )
        {
#ifdef __MINGW32__
            u32 len = strlen( buf );
#else
            u32 len = static_cast<u32>(strnlen( buf, sizeof( buf ) ));
#endif
            nn::util::SNPrintf( buf + len, sizeof( buf ) - len, "%02x", *ptr++ );
        }
        LOG_MESSAGE( "    limit %d refId %s Attr %08x", ntohl( v1SectData->limit ), buf, ntohl( v1SectData->referenceIdAttr ) );
    }
}

/*
 * Decode and print the various V1 sections, checking for consistency where possible
 */
static void dumpV1TicketSections( ESV2Ticket *tickPtr, u32 tickSize )
{
    NN_UNUSED( tickSize );
    ESV1TicketHeader *v1Head = reinterpret_cast<ESV1TicketHeader *>( tickPtr + 1 );
    ESV1SectionHeader *v1Sect;
    u8 *v1SectData;
    u16 nSect = ntohs( v1Head->nSectHdrs );
    u16 n;

    LOG_MESSAGE( "V1 Ticket Header" );
#ifdef VERBOSE
    LOG_MESSAGE( "  hSize %d", ntohs( v1Head->hdrSize ) );
    LOG_MESSAGE( "  tSize %d", ntohl( v1Head->ticketSize ) );
    LOG_MESSAGE( "  hVersion %d", ntohs( v1Head->hdrVersion ) );
    LOG_MESSAGE( "  nSectHdrs %d x size %d", ntohs( v1Head->nSectHdrs ), ntohs( v1Head->sectHdrEntrySize ) );
    LOG_MESSAGE( "  sectHdrOff %d", ntohl( v1Head->sectHdrOfst ) );
    LOG_MESSAGE( "  flags %x", ntohl( v1Head->flags ) );
#endif

    v1Sect = reinterpret_cast<ESV1SectionHeader *>( reinterpret_cast<u8 *>( v1Head ) + ntohl( v1Head->sectHdrOfst ) );

    for( n = 0; n < nSect; n++, v1Sect++ )
    {
        u16 stype = ntohs( v1Sect->sectionType );
        u32 soffset = ntohl( v1Sect->sectOfst );
        u32 nrec = ntohl( v1Sect->nRecords );

        LOG_MESSAGE( "V1 Section Header[%d]", n );

        if( stype < ES_ITEM_RIGHT_PERMANENT || stype > ES_ITEM_RIGHT_LIMITED_RESOURCE )
        {
            LOG_FAIL( "  Invalid V1 section type %d!", stype );
            stype = 0;
        }
        LOG_MESSAGE( "  sectType %s (%d)", secName[stype], stype );
#ifdef VERBOSE
        LOG_MESSAGE( "  sectOffset %d", soffset );
        LOG_MESSAGE( "  nRecords %d", nrec );
        LOG_MESSAGE( "  recordSize %d", ntohl( v1Sect->recordSize ) );
        LOG_MESSAGE( "  sectSize %d", ntohl( v1Sect->sectionSize ) );
        LOG_MESSAGE( "  flags %d", ntohs( v1Sect->flags ) );
#endif
        if( stype == 0 )
        {
            continue;
        }

        v1SectData = ( reinterpret_cast<u8 *>( v1Head ) ) + soffset;
        LOG_MESSAGE( "  Section Records:" );

        switch( stype )
        {
        case ES_ITEM_RIGHT_PERMANENT:
            dumpV1SectPermanent( (ESV1PermanentRecord *)v1SectData, nrec );
            break;
        case ES_ITEM_RIGHT_SUBSCRIPTION:
            dumpV1SectSubscription( (ESV1SubscriptionRecord *)v1SectData, nrec );
            break;
        case ES_ITEM_RIGHT_CONTENT:
            dumpV1SectContent( (ESV1ContentRecord *)v1SectData, nrec );
            break;
        case ES_ITEM_RIGHT_CONTENT_CONSUMPTION:
            dumpV1SectConsumption( (ESV1ContentConsumptionRecord *)v1SectData, nrec );
            break;
        case ES_ITEM_RIGHT_ACCESS_TITLE:
            dumpV1SectAccess( (ESV1AccessTitleRecord *)v1SectData, nrec );
            break;
        case ES_ITEM_RIGHT_LIMITED_RESOURCE:
            dumpV1SectLimitedResource( (ESV1LimitedResourceRecord *)v1SectData, nrec );
            break;
        default:
            LOG_FAIL( "This can not be happening!" );
            break;
        }
    }
}


/*
 * Utility routine to verify a ticket
 */
static ESError verifyTicket( u8 *tickBuf, u32 tickBufSize )
{
    IOSCError rv;
    ESV2Ticket *tickPtr;
    u32 sigType;
    u32 hashType = IOSC_HASH_SHA1;
    u32 hashSize = sizeof( IOSCHash );
    IOSCHashContext hashCtx;
    u8 hashVal[32];
    int hdrLen;
    int tickSize;
    int sectTotalSize;
    MemoryInputStream ticket( tickBuf, tickBufSize );
    MemoryOutputStream outTicket( __dpTicketBuf, sizeof( __dpTicketBuf ) );

    LOG_MESSAGE( "==> verifyTicket (size %d)", tickBufSize );

    /*
     * If ticket is personalized, unpersonalize it so that we can check the signature
     */
    tickPtr = reinterpret_cast<ESV2Ticket *>( tickBuf );
    if( tickPtr->deviceId != 0 )
    {
        LOG_MESSAGE( "Ticket is personalized for device 0x%08X", tickPtr->deviceId );
        {
            u32 n;
            NN_LOG( "Personalized title key:\n" );
            for( n = 0; n < sizeof(ESTitleKey); n++ )
            {
                NN_LOG( "%02x ", tickPtr->titleKey.rsaKey[n] );
            }
            NN_LOG( "\n" );
        }
        rv = ESI_UnpersonalizeTicket( (ESTicket*)tickPtr );
        if( rv != ES_ERR_OK )
        {
            LOG_FAIL( "Unpersonalize failed (rv %d)", rv );
            goto out;
        }
    }
    else
    {
        LOG_MESSAGE( "Ticket is a common ticket!" );
    }
     LOG_MESSAGE( "RightsId");
    dumpRightsId(tickPtr->rightsId);
    LOG_MESSAGE( "TicketId %016llx", tickPtr->ticketId );
    LOG_MESSAGE( "Ticket version %d", tickPtr->ticketVersion );

    sigType = *(u32 *)tickPtr;
    sigType = ntohl( sigType );
    if( sigType >= IOSC_SIG_RSA4096_H256 )
    {
        hashType = IOSC_HASH_SHA256;
        hashSize = sizeof( IOSCSha256Hash );
    }

    /*
     * The ticket file (stream) may contain extraneous data appended to the
     * ticket, so we need to look at the V1 header to determine the actual
     * length.
     */
    tickSize = sizeof( ESV2Ticket );
    if( tickPtr->formatVersion == 2 )
    {
        sectTotalSize = tickPtr->sectTotalSize;
        LOG_MESSAGE( "Ticket is a V2 ticket with section total size %d", sectTotalSize );
        tickSize += sectTotalSize;
    }
    else
    {
        LOG_MESSAGE( "Ticket version is bogus: %d", tickPtr->formatVersion );
    }

    hdrLen = static_cast<int>(reinterpret_cast<u8 *>( &tickPtr->sig.issuer ) - reinterpret_cast<u8 *>( tickPtr ));
    LOG_MESSAGE( "SigType %x tickSize %d hdrLen %d (ignoring %d extra bytes)", sigType, tickSize, hdrLen, tickBufSize - tickSize );

    rv = IOSC_GenerateHash( hashCtx, NULL, 0, hashType | IOSC_HASH_FIRST, NULL );
    if( rv != IOSC_ERROR_OK )
    {
        LOG_FAIL( "Initialize hash context (rv %d)", rv );
        goto out;
    }

    rv = IOSC_GenerateHash( hashCtx, tickBuf + hdrLen, tickSize - hdrLen, hashType | IOSC_HASH_LAST, hashVal );
    if( rv != IOSC_ERROR_OK )
    {
        LOG_FAIL( "Hash last (rv %d)", rv );
        goto out;
    }

    rv = IOSC_VerifyPublicKeySign( hashVal, hashSize, xsPubKeyH, (u8 *)&tickPtr->sig.sig );
    if( rv == IOSC_ERROR_OK )
    {
        LOG_PASS( "VerifyPublicKeySign of ticket OK" );
    }
    else
    {
        LOG_FAIL( "VerifyPublicKeySign of ticket (rv %d)", rv );
    }

    /*
     * Look at the commonKeyId so that we know which key to use for content verification
     */
    LOG_MESSAGE( "Common key ID: %d", tickPtr->commonKeyId );

    /*
     * Dump the per-title limits for manual verification (there's no way for verify to know
     * what was intended).  Checking the limit codes for validity is all we can do.
     */
    {
        u8 ltype;

        if( ( ltype = tickPtr->licenseType ) > ES_LICENSE_SERVICE )
        {
            LOG_FAIL( "Invalid license type: %d", ltype );
        }
        else
        {
            LOG_MESSAGE( "License type: %s", licName[ltype] );
        }
    }

    /*
     * Dump the V1 ticket sections and check for consistency where possible.
     */
    dumpV1TicketSections( tickPtr, tickSize );

    if( tickPtr->deviceId != 0 )
    {
        /*
        * Repersonalize the ticket before trying import
        */
#ifdef NEEDS_CHANGE_V2_TICKET
        rv = ESI_PersonalizeTicket( tickPtr );
        if( rv != ES_ERR_OK )
        {
            LOG_FAIL( "Repersonalize failed (rv %d)", rv );
            goto out;
        }
#endif
    }

#ifdef NEEDS_CHANGE_V2_TICKET
    CHECK_RESULT( ets.ImportTicket( ticket, (const void **)certList, nCerts, outTicket ), ES_ERR_OK );
#endif

out:
    return rv;
}  // NOLINT (readability/fn_size)


/*
 * Simple test of the PublishTicket class using no template
 */
static ESError testSimpleTicket()
{
    ESError rv = ES_ERR_OK;
    PublishTicket pubTik;
    u32 ticketLen;
    IOSCAesIv aesIv;
    IOSCAesKey titleKey;
    //ESTitleId titleId = TEST_TITLE_ID;
    ESTitleId titleId = 0x0100000000003009LL;
    ESRightsId rightsId;
    ESTitleId nTitleId;
    ESRightsId outRightsId;
    ESTicketId outTicketId;
    ESDeviceId outDeviceId;
    ESAccountId outAccountId;
    u8 outCommonKeyId;
    IOSCAesKey outTitleKey;
    const ESTicketId testTicketId = 0xF000004100000042UL;

    ESTitleId littleEndianTitleId = ntohll(titleId);
    memset(rightsId, 0, sizeof(ESRightsId));
    memcpy(rightsId, &littleEndianTitleId, sizeof(ESTitleId));

    LOG_MESSAGE( "==> Start test:  Simple Ticket with no template" );

    /*
     * Encrypt the titleKey with the common key and write it out
     *
     * Initialization Vector is the title ID (network byte order)
     */
    memset( aesIv, 0, sizeof( aesIv ) );
    nTitleId = ntohll( titleId );
    memcpy( aesIv, &nTitleId, sizeof( nTitleId ) );
    CHECK_RESULT( IOSC_Encrypt( hCommonKey, aesIv, aesKey, sizeof( aesKey ), titleKey ), IOSC_ERROR_OK );

    /*
     * Fill in the ticket
     */
    CHECK_RESULT( pubTik.SetTicketId( testTicketId ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetRightsId( rightsId ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetLicenseType( ES_LICENSE_PERMANENT ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetCommonKeyId( 0 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTitleKey( titleKey ), ES_ERR_OK );

    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );
    CHECK_RESULT( pubTik.DumpTicket(), ES_ERR_OK );

    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );

    writeFile( __ticketFile, ticketBuf, ticketLen );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

    /*
     * Test the "Get" methods
     */
    CHECK_RESULT( pubTik.GetTicketId( &outTicketId ), ES_ERR_OK );
    if( outTicketId != testTicketId )
    {
        LOG_FAIL( "GetTicketId returns wrong value: %016llx", outTicketId );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.GetRightsId( &outRightsId ), ES_ERR_OK );
    if( memcmp(outRightsId, rightsId, sizeof(ESRightsId)) != 0 )
    {
        LOG_FAIL( "GetRightsId returns wrong value");
        dumpRightsId(outRightsId);
        dumpRightsId(rightsId);
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.GetDeviceId( &outDeviceId ), ES_ERR_OK );
    /* Common ticket, so deviceId is zero */
    if( outDeviceId != 0 )
    {
        LOG_FAIL( "GetDeviceId returns %08x, expected 0", outDeviceId );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.GetAccountId( &outAccountId ), ES_ERR_OK );
    /* Common ticket, so accountId is zero */
    if( outAccountId != 0 )
    {
        LOG_FAIL( "GetAccountId returns %08x, expected 0", outAccountId );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.GetCommonKeyId( &outCommonKeyId ), ES_ERR_OK );
    if( outCommonKeyId != 0 )
    {
        LOG_FAIL( "GetCommonKeyId returns %d, expected 0", outCommonKeyId );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.GetTitleKey( outTitleKey ), ES_ERR_OK );
    if( memcmp( titleKey, outTitleKey, sizeof( titleKey ) ) != 0 )
    {
        LOG_FAIL( "GetTitleKey value does not match titleKey" );
        dumpData( "TitleKey", titleKey, sizeof( titleKey ) );
        dumpData( "OutTitleKey", outTitleKey, sizeof( outTitleKey ) );
        rv = ES_ERR_INVALID;
        goto out;
    }

out:
    LOG_MESSAGE( "testSimpleTicket returns %d", rv );
    return rv;
}

/*
 * Simple test of the PublishTicket class using the smallest possible template
 */
static ESError testBasicTemplate()
{
    ESError rv = ES_ERR_OK;
    PublishTicket pubTik( tkt1, sizeof(ESV2Ticket) );
    u32 ticketLen;

    LOG_MESSAGE( "==> Start test:  Simple Ticket with incomplete template" );

    /*
     * With a minimal template, need to fill in the TicketId and content mask
     */
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000043LL ), ES_ERR_OK );

#ifdef DO_SECTION_TEST
    CHECK_RESULT( pubTik.SetContentMask( 0, 1 ), ES_ERR_OK );
#endif
    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );
    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

out:
    LOG_MESSAGE( "testBasicTemplate returns %d", rv );
    return rv;
}


/*
 * Simple test of the PublishTicket class using the basic template that is a full V1 ticket
 */
static ESError testFullTemplate()
{
    ESError rv = ES_ERR_OK;
    PublishTicket pubTik( tkt1, sizeof(ESV2Ticket));
    u32 ticketLen;

    LOG_MESSAGE( "==> Start test:  Simple Ticket with complete template" );

    /*
     * Set only the ticketId
     */
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000044LL ), ES_ERR_OK );

    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );
    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

out:
    LOG_MESSAGE( "testFullTemplate returns %d", rv );
    return rv;
}


/*
 * Negative testing for various operations with templates
 */
static ESError testTemplateNegative()
{
    ESError rv = ES_ERR_OK;
    PublishTicket pubTik;
    u32 ticketLen;

    LOG_MESSAGE( "==> Start test:  Negative testing for Ticket Templates" );

    /*
     * The constructor has been called with no template
     *
     * Do a few operations and then call the Reset method
     */
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000045LL ), ES_ERR_OK );

    /*
     * The minimum length for a template is the V0 header plus an empty V1 header
     */
    CHECK_RESULT( pubTik.Reset( tkt1, 695 ), ES_ERR_INVALID );
    CHECK_RESULT( pubTik.Reset( tkt1, 0 ), ES_ERR_INVALID );

    /*
     * XXX what about odd lengths that don't represent a complete V1 record?
     * XXX the following appears to succeed, but can't work
     */
    // CHECK_RESULT(pubTik.Reset(tkt1, 697), ES_ERR_INVALID);

    /*
     * Now reset to the full length of the template
     */
    CHECK_RESULT( pubTik.Reset( tkt1, sizeof(ESV2Ticket) ), ES_ERR_OK );

    /*
     * SetItemRights has already been called on the template, so neither that
     * nor SetContentMask is valid
     */
#ifdef DO_SECTION_TEST
    CHECK_RESULT( pubTik.SetContentMask( 0, 1 ), ES_ERR_INVALID );
    CHECK_RESULT( pubTik.SetItemRights( NULL, 0 ), ES_ERR_INVALID );
#endif
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000046LL ), ES_ERR_OK );

    /*
     * But SignTicket should not call SetItemRights, since it's been done already
     */
    LOG_MESSAGE( "About to call SignTicket:  should not see any error messages" );
    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );
    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

out:
    LOG_MESSAGE( "testTemplateNegative returns %d", rv );
    return rv;
}


/*
 * Test SetContentMask
 */
#ifdef DO_SECTION_TEST
static ESError testContentMasks()
{
    ESError rv = ES_ERR_OK;
    PublishTicket pubTik;
    u32 ticketLen;
    IOSCAesIv aesIv;
    IOSCAesKey titleKey;
    ESTitleId titleId = TEST_TITLE_ID;
    ESTitleId nTitleId;

    LOG_MESSAGE( "==> Start test:  Content Masks" );

    /*
     * Encrypt the titleKey with the common key and write it out
     *
     * Initialization Vector is the title ID (network byte order)
     */
    memset( aesIv, 0, sizeof( aesIv ) );
    nTitleId = ntohll( titleId );
    memcpy( aesIv, &nTitleId, sizeof( nTitleId ) );
    CHECK_RESULT( IOSC_Encrypt( hCommonKey, aesIv, aesKey, sizeof( aesKey ), titleKey ), IOSC_ERROR_OK );

    /*
     * Fill in the ticket
     */
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000048LL ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTitleId( titleId ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetLicenseType( ES_LICENSE_PERMANENT ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetCommonKeyId( 0 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTitleKey( titleKey ), ES_ERR_OK );

    /*
     * Negative tests for the limits
     */
    CHECK_RESULT( pubTik.SetContentMask( 0, ES_CONTENT_INDEX_MAX + 2 ), ES_ERR_INVALID );
    CHECK_RESULT( pubTik.SetContentMask( ES_CONTENT_INDEX_MAX + 1, 1 ), ES_ERR_INVALID );
    CHECK_RESULT( pubTik.SetContentMask( ES_CONTENT_INDEX_MAX, 2 ), ES_ERR_INVALID );
    CHECK_RESULT( pubTik.SetContentMask( ES_CONTENT_INDEX_MAX - 1, 32 ), ES_ERR_INVALID );

    /*
     * Turn on all bits
     */
    CHECK_RESULT( pubTik.SetContentMask( 0, ES_CONTENT_INDEX_MAX + 1 ), ES_ERR_OK );

    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );
    CHECK_RESULT( pubTik.DumpTicket(), ES_ERR_OK );

    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

    /*
     * Now try the case of a sparse content mask and make sure that the record
     * compression logic works
     */
    CHECK_RESULT( pubTik.Reset( ticketBuf, 696 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000049LL ), ES_ERR_OK );

    CHECK_RESULT( pubTik.SetContentMask( 0, 1 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetContentMask( 1023, 1 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetContentMask( 2048, 1023 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetContentMask( 32767, 1 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetContentMask( 50 * 1024, 2047 ), ES_ERR_OK );

    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );
    CHECK_RESULT( pubTik.DumpTicket(), ES_ERR_OK );

    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

out:
    LOG_MESSAGE( "testContentMasks returns %d", rv );
    return rv;
}



/*
 * Test SetItemRights
 */
static ESError testSetItemRights()
{
    ESError rv = ES_ERR_OK;
    PublishTicket pubTik;
    u32 ticketLen;
    IOSCAesIv aesIv;
    IOSCAesKey titleKey;
    ESTitleId titleId = TEST_TITLE_ID;
    ESTitleId nTitleId;

    LOG_MESSAGE( "==> Start test:  SetItemRights" );

    /*
     * Encrypt the titleKey with the common key and write it out
     *
     * Initialization Vector is the title ID (network byte order)
     */
    memset( aesIv, 0, sizeof( aesIv ) );
    nTitleId = ntohll( titleId );
    memcpy( aesIv, &nTitleId, sizeof( nTitleId ) );
    CHECK_RESULT( IOSC_Encrypt( hCommonKey, aesIv, aesKey, sizeof( aesKey ), titleKey ), IOSC_ERROR_OK );

    /*
     * Fill in the ticket
     */
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000050LL ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTitleId( titleId ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetLicenseType( ES_LICENSE_PERMANENT ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetCommonKeyId( 0 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTitleKey( titleKey ), ES_ERR_OK );

    /*
     * Test the mixing of SetItemRights and SetContentMask
     */
    CHECK_RESULT( pubTik.SetContentMask( 0, 3 ), ES_ERR_OK );

    memset( (u8 *)rightsEntries, 0, sizeof( rightsEntries ) );
    rightsEntries[0].type = ES_ITEM_RIGHT_PERMANENT;
    rightsEntries[0].right.perm.referenceId[0] = 0x42;
    rightsEntries[0].right.perm.referenceId[15] = 0x77;
    rightsEntries[0].right.perm.referenceIdAttr = 0xbabecafe;
    rightsEntries[1].type = ES_ITEM_RIGHT_SUBSCRIPTION;
    rightsEntries[1].right.sub.limit = 27;
    rightsEntries[1].right.sub.referenceId[7] = 0xa5;
    rightsEntries[1].right.sub.referenceId[11] = 0xfe;
    rightsEntries[1].right.sub.referenceIdAttr = 0xdeadbeef;

    CHECK_RESULT( pubTik.SetItemRights( rightsEntries, 2 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetContentMask( 1023, 1 ), ES_ERR_INVALID );  // Should fail

    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );
    CHECK_RESULT( pubTik.DumpTicket(), ES_ERR_OK );

    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

    /*
     * Now try the case in which SetItemRights does the content masks as well
     */
    CHECK_RESULT( pubTik.Reset( ticketBuf, 696 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000051LL ), ES_ERR_OK );

    memset( (u8 *)rightsEntries, 0, sizeof( rightsEntries ) );
    rightsEntries[0].type = ES_ITEM_RIGHT_PERMANENT;
    rightsEntries[0].right.perm.referenceId[0] = 0x58;
    rightsEntries[0].right.perm.referenceId[15] = 0x24;
    rightsEntries[0].right.perm.referenceIdAttr = 0xdeafdadd;
    rightsEntries[1].type = ES_ITEM_RIGHT_CONTENT_CONSUMPTION;
    rightsEntries[1].right.cntLmt.index = 3;
    rightsEntries[1].right.cntLmt.code = ES_LC_DURATION_TIME;
    rightsEntries[1].right.cntLmt.limit = 10;
    rightsEntries[2].type = ES_ITEM_RIGHT_ACCESS_TITLE;
    rightsEntries[2].right.accTitle.accessTitleId = 0x5000004100000043LL;
    rightsEntries[2].right.accTitle.accessTitleMask = 0xff0000ff00ff00ffULL;
    rightsEntries[3].type = ES_ITEM_RIGHT_CONTENT;
    rightsEntries[3].right.cnt.offset = 0;
    rightsEntries[3].right.cnt.accessMask[0] = 0x21;
    rightsEntries[3].right.cnt.accessMask[62] = 0xf0;
    rightsEntries[3].right.cnt.accessMask[63] = 0x1f;
    rightsEntries[3].right.cnt.accessMask[126] = 0x07;

    CHECK_RESULT( pubTik.SetItemRights( rightsEntries, 4 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetContentMask( 2048, 1 ), ES_ERR_INVALID );  // Should fail

    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );
    CHECK_RESULT( pubTik.DumpTicket(), ES_ERR_OK );

    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

    /*
     * If conflicting content masks between SetItemRights and SetContentMask,
     * then SetItemRights takes precedence.
     */
    CHECK_RESULT( pubTik.Reset( ticketBuf, 696 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000052LL ), ES_ERR_OK );

    memset( (u8 *)rightsEntries, 0, sizeof( rightsEntries ) );
    rightsEntries[0].type = ES_ITEM_RIGHT_PERMANENT;
    rightsEntries[0].right.perm.referenceId[0] = 0x58;
    rightsEntries[0].right.perm.referenceId[15] = 0x24;
    rightsEntries[0].right.perm.referenceIdAttr = 0xdeafdadd;
    rightsEntries[1].type = ES_ITEM_RIGHT_CONTENT_CONSUMPTION;
    rightsEntries[1].right.cntLmt.index = 3;
    rightsEntries[1].right.cntLmt.code = ES_LC_DURATION_TIME;
    rightsEntries[1].right.cntLmt.limit = 10;
    rightsEntries[2].type = ES_ITEM_RIGHT_ACCESS_TITLE;
    rightsEntries[2].right.accTitle.accessTitleId = 0x5000004100000043LL;
    rightsEntries[2].right.accTitle.accessTitleMask = 0xff0000ff00ff00ffULL;
    rightsEntries[3].type = ES_ITEM_RIGHT_CONTENT;
    rightsEntries[3].right.cnt.offset = 0;
    rightsEntries[3].right.cnt.accessMask[127] = 0x07;

    CHECK_RESULT( pubTik.SetContentMask( 8, 4 ), ES_ERR_OK );

    CHECK_RESULT( pubTik.SetItemRights( rightsEntries, 4 ), ES_ERR_OK );

    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );
    CHECK_RESULT( pubTik.DumpTicket(), ES_ERR_OK );

    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

    /*
     * Test SetItemRights for consumable items and mix in some content right records.
     * The records of the same type don't need to be contiguous in the input array.
     */
    CHECK_RESULT( pubTik.Reset( ticketBuf, 696 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000056LL ), ES_ERR_OK );

    memset( (u8 *)rightsEntries, 0, sizeof( rightsEntries ) );
    rightsEntries[0].type = ES_ITEM_RIGHT_LIMITED_RESOURCE;
    rightsEntries[0].right.limRes.limit = 10;
    rightsEntries[0].right.limRes.referenceId[0] = 0xe5;
    rightsEntries[0].right.limRes.referenceId[15] = 0xa3;
    rightsEntries[0].right.limRes.referenceIdAttr = 0xbaadd00d;
    rightsEntries[1].type = ES_ITEM_RIGHT_CONTENT;
    rightsEntries[1].right.cnt.offset = 0;
    rightsEntries[1].right.cnt.accessMask[0] = 0x03;
    rightsEntries[1].right.cnt.accessMask[127] = 0x10;
    rightsEntries[2].type = ES_ITEM_RIGHT_LIMITED_RESOURCE;
    rightsEntries[2].right.limRes.limit = 1;
    rightsEntries[2].right.limRes.referenceId[0] = 0x55;
    rightsEntries[2].right.limRes.referenceId[15] = 0xaa;
    rightsEntries[2].right.limRes.referenceIdAttr = 0x12345678;
    rightsEntries[3].type = ES_ITEM_RIGHT_CONTENT;
    rightsEntries[3].right.cnt.offset = 2048;
    rightsEntries[3].right.cnt.accessMask[1] = 0x70;
    rightsEntries[3].right.cnt.accessMask[125] = 0x01;
    rightsEntries[4].type = ES_ITEM_RIGHT_LIMITED_RESOURCE;
    rightsEntries[4].right.limRes.limit = 0x1fffffff;
    rightsEntries[4].right.limRes.referenceId[0] = 0x88;
    rightsEntries[4].right.limRes.referenceId[15] = 0xbb;
    rightsEntries[4].right.limRes.referenceIdAttr = 0xdeafbeef;

    CHECK_RESULT( pubTik.SetItemRights( rightsEntries, 5 ), ES_ERR_OK );

    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );
    CHECK_RESULT( pubTik.DumpTicket(), ES_ERR_OK );

    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

    /*
     * Negative test for bogus limit codes
     */
    CHECK_RESULT( pubTik.Reset( ticketBuf, 696 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000057LL ), ES_ERR_OK );

    memset( (u8 *)rightsEntries, 0, sizeof( rightsEntries ) );
    rightsEntries[0].type = ES_ITEM_RIGHT_PERMANENT;
    rightsEntries[0].right.perm.referenceId[0] = 0x58;
    rightsEntries[0].right.perm.referenceId[15] = 0x24;
    rightsEntries[0].right.perm.referenceIdAttr = 0xdeafdadd;
    rightsEntries[1].type = ES_ITEM_RIGHT_CONTENT_CONSUMPTION;
    rightsEntries[1].right.cntLmt.index = 8;
    rightsEntries[1].right.cntLmt.code = ES_LC_DURATION_TIME;
    rightsEntries[1].right.cntLmt.limit = 10;
    rightsEntries[2].type = ES_ITEM_RIGHT_CONTENT_CONSUMPTION;
    rightsEntries[2].right.cntLmt.index = 9;
    rightsEntries[2].right.cntLmt.code = ES_LC_DURATION_TIME + 12;
    rightsEntries[2].right.cntLmt.limit = 20;

    CHECK_RESULT( pubTik.SetContentMask( 8, 4 ), ES_ERR_OK );

    CHECK_RESULT( pubTik.SetItemRights( rightsEntries, 3 ), ES_ERR_INVALID );  // Should fail

    /*
     * Test for no content actually activated by the ticket.
     *
     * This was originally not allowed by the code, but has been changed (see bug 8526).
     */
    CHECK_RESULT( pubTik.Reset( ticketBuf, 696 ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTicketId( 0x5000004100000058LL ), ES_ERR_OK );

    memset( (u8 *)rightsEntries, 0, sizeof( rightsEntries ) );
    rightsEntries[0].type = ES_ITEM_RIGHT_PERMANENT;
    rightsEntries[0].right.perm.referenceId[0] = 0x58;
    rightsEntries[0].right.perm.referenceId[15] = 0x24;
    rightsEntries[0].right.perm.referenceIdAttr = 0xdeafdadd;
    rightsEntries[1].type = ES_ITEM_RIGHT_CONTENT_CONSUMPTION;
    rightsEntries[1].right.cntLmt.index = 8;
    rightsEntries[1].right.cntLmt.code = ES_LC_DURATION_TIME;
    rightsEntries[1].right.cntLmt.limit = 10;
    rightsEntries[2].type = ES_ITEM_RIGHT_ACCESS_TITLE;
    rightsEntries[2].right.accTitle.accessTitleId = 0x5000004100000043LL;
    rightsEntries[2].right.accTitle.accessTitleMask = 0xff0000ff00ff00ffULL;

    CHECK_RESULT( pubTik.SetItemRights( rightsEntries, 3 ), ES_ERR_OK );

    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );

    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( verifyTicket( ticketBuf, ticketLen ), ES_ERR_OK );

    rv = ES_ERR_OK;

out:
    LOG_MESSAGE( "testSetItemRights returns %d", rv );
    return rv;
}  // NOLINT (readability/fn_size)
#endif

/**
    @brief FS用アロケータ
 */
void* Allocate(size_t size)
{
    return std::malloc(size);
}

/**
    @brief FS用デアロケータ
 */
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}

/*
 * Run regression tests for 4.0 Publishing SDK APIs
 */
TEST( PublishRegressionTest, PublishRegression )
{
    ESError rv = ES_ERR_OK;
    int exitStatus = 0;

    IOSC_Initialize();
    nn::fs::SetAllocator(Allocate, Deallocate);
    nn::fs::MountHostRoot();
    initKeys();
    initCerts();
    loadDevCert();

    NN_LOG( "Starting PubSDK Regression Tests\n" );

    rv = testSimpleTicket();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        exitStatus = -1;
    }

    rv = testBasicTemplate();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        exitStatus = -1;
    }

    rv = testFullTemplate();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        exitStatus = -1;
    }

    rv = testTemplateNegative();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        exitStatus = -1;
    }

#ifdef DO_SECTION_TEST
    rv = testContentMasks();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        exitStatus = -1;
    }

    rv = testSetItemRights();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        exitStatus = -1;
    }
#endif
    // end of ticket test

    NN_LOG( "PubSDK Regression Tests %s\n", ( exitStatus == 0 ) ? "PASSED" : "FAILED" );
}
