﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*---------------------------------------------------------------------------*
  Project:  Horizon
  File:     broadon_Main.cpp

  Copyright (C)2009 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $Rev: 10289 $
 *---------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>

//#include <nn/Result.h>
//#include <nn/os.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/crypto/crypto_Sha256Generator.h>
//#include <nn/fs.h>
#include <nn/spl/spl_Api.h>

#include <nn/ioslibc/ioslibc.h>
#include <nn/escore/estypes.h>
#include <nn/escorei/esitypes.h>

#include "testEs_CaCert.cpp"
#include "testEs_MsCert.cpp"

/*
 * To test AES decryption
 */
const u64 TITLE_ID = 0x0004000100012345ULL;
#include "testEs_TitleKey.cpp"
#include "testEs_RawApp.cpp"
#include "testEs_EncApp.cpp"

#define DECRYPT_IN_PLACE
#ifndef DECRYPT_IN_PLACE
NN_ALIGNAS( 64 ) static u8 __appDecBuf[8192];
#endif

#include <nn/iosc/iosc.h>
#include <nn/ioscrypto/iosccert.h>

USING_ES_NAMESPACE
using namespace nn::crypto;

#define TEST_LOG NN_LOG

static const int ARRAY_SIZE = 128;

// AES
NN_ALIGNAS( 64 ) static u8 aesKey[16];

// AES IV
NN_ALIGNAS( 64 ) static u8 iv[16];


void dumpArray( const char *msg, u8 *array, int size )
{
    NN_UNUSED( msg );
    NN_UNUSED( array );
    NN_UNUSED( size );

    TEST_LOG( "%s", msg );
    for( int i = 0; i < size; i++ )
    {
        if( i % 16 == 0 )
        {
            TEST_LOG( "\n" );
        }
        TEST_LOG( "%02X ", array[i] );
    }
    TEST_LOG( "\n" );
}

bool compare( const u8 *p0, const u8 *p1, size_t size )
{
    s32 i;
    bool result = true;
    for( i = 0; i < (s32)size; ++i )
    {
        if( *( p0 + i ) != *( p1 + i ) )
        {
            result = false;
        }
    }
    return result;
}  // compare


/*
 * Test multiple SHA calls with buffers not 64 multiple
 */
const int HASHOFF = 63;

void doShaTest( int hashType )
{
    int i;
    int hashSize = ( hashType == IOSC_HASH_SHA256 ? 32 : 20 );
    Sha1Generator swSha1;
    Sha256Generator swSha256;
    IOSCHashContext hashCtx;
    IOSCHashContext dummyCtx;

    u8 inputBuffer[ARRAY_SIZE];
    u8 inputHash[32], inputSwHash[32];


    for( i = 0; i < ARRAY_SIZE; i++ )
    {
        inputBuffer[i] = (u8)i;
    }

    memset( inputSwHash, 0x5a, sizeof( inputSwHash ) );
    memset( inputHash, 0xa5, sizeof( inputHash ) );

    if( hashType == IOSC_HASH_SHA1 )
    {
        swSha1.Initialize();
        swSha1.Update( inputBuffer, sizeof( inputBuffer ) );
        swSha1.GetHash( inputSwHash, nn::crypto::Sha1Generator::HashSize );
    }
    else
    {
        swSha256.Initialize();
        swSha256.Update( inputBuffer, sizeof( inputBuffer ) );
        swSha256.GetHash( inputSwHash, nn::crypto::Sha256Generator::HashSize );
    }

    TEST_LOG( "Hash test using %s\n", hashType == IOSC_HASH_SHA256 ? "SHA256" : "SHA1" );

    dumpArray( "Software SHA", inputSwHash, hashSize );

    memset( hashCtx, 0, sizeof( hashCtx ) );
    IOSC_GenerateHash( hashCtx, NULL, 0, IOSC_HASH_FIRST | hashType, NULL );
    IOSC_GenerateHash( hashCtx, inputBuffer, HASHOFF, IOSC_HASH_MIDDLE | hashType, NULL );
    IOSC_GenerateHash( hashCtx, inputBuffer + HASHOFF, sizeof( inputBuffer ) - HASHOFF, IOSC_HASH_MIDDLE | hashType, NULL );
    IOSC_GenerateHash( hashCtx, NULL, 0, IOSC_HASH_LAST | hashType, inputHash );

    TEST_LOG( "Checking %s results (HASHOFF %d) ... ", hashType == IOSC_HASH_SHA256 ? "SHA256" : "SHA1", HASHOFF );
    if( compare( static_cast<const u8 *>( inputHash ), static_cast<const u8 *>( inputSwHash ), hashSize ) )
    {
        TEST_LOG( "OK.\n" );
        SUCCEED();
    }
    else
    {
        TEST_LOG( "NG.\n" );
        FAIL();
    }

    dumpArray( "IOSC Hash Output", inputHash, hashSize );

    /*
     * Test hash context save and restore
     */
    memset( hashCtx, 0, sizeof( hashCtx ) );
    memset( dummyCtx, 0x42, sizeof( dummyCtx ) );
    memset( inputHash, 0xa5, sizeof( inputHash ) );

    IOSC_GenerateHash( hashCtx, NULL, 0, IOSC_HASH_FIRST | hashType, NULL );
    IOSC_GenerateHash( hashCtx, inputBuffer, HASHOFF, IOSC_HASH_MIDDLE | hashType, NULL );

    IOSC_GenerateHash( dummyCtx, NULL, 0, IOSC_HASH_FIRST | hashType, NULL );
    IOSC_GenerateHash( hashCtx, NULL, 0, IOSC_HASH_RESTART | hashType, NULL );

    IOSC_GenerateHash( hashCtx, inputBuffer + HASHOFF, sizeof( inputBuffer ) - HASHOFF, IOSC_HASH_MIDDLE | hashType, NULL );
    IOSC_GenerateHash( hashCtx, NULL, 0, IOSC_HASH_LAST | hashType, inputHash );

    TEST_LOG( "Checking %s RESTART results (HASHOFF %d) ... ", hashType == IOSC_HASH_SHA256 ? "SHA256" : "SHA1", HASHOFF );
    if( compare( static_cast<const u8 *>( inputHash ), static_cast<const u8 *>( inputSwHash ), hashSize ) )
    {
        TEST_LOG( "OK.\n" );
        SUCCEED();
    }
    else
    {
        TEST_LOG( "NG.\n" );
        FAIL();
    }

    dumpArray( "IOSC Hash Output", inputHash, hashSize );
}


static u8 ecSignInput[] = {
    0x3c, 0x43, 0x68, 0x61, 0x6c, 0x6c, 0x65, 0x6e, 0x67, 0x65, 0x3e, 0x31, 0x33, 0x30, 0x35, 0x33, 0x30, 0x33, 0x30, 0x34, 0x3c, 0x2f,
    0x43, 0x68, 0x61, 0x6c, 0x6c, 0x65, 0x6e, 0x67, 0x65, 0x3e, 0x3c, 0x53, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x4e, 0x75, 0x6d, 0x62, 0x65,
    0x72, 0x3e, 0x59, 0x59, 0x59, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x36, 0x30, 0x3c, 0x2f, 0x53, 0x65, 0x72, 0x69, 0x61, 0x6c,
    0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x3e, 0x3c, 0x43, 0x6f, 0x75, 0x6e, 0x74, 0x72, 0x79, 0x3e, 0x55, 0x53, 0x3c, 0x2f, 0x43, 0x6f,
    0x75, 0x6e, 0x74, 0x72, 0x79, 0x3e, 0x3c, 0x52, 0x65, 0x67, 0x69, 0x73, 0x74, 0x65, 0x72, 0x52, 0x65, 0x67, 0x69, 0x6f, 0x6e, 0x3e,
    0x55, 0x53, 0x41, 0x3c, 0x2f, 0x52, 0x65, 0x67, 0x69, 0x73, 0x74, 0x65, 0x72, 0x52, 0x65, 0x67, 0x69, 0x6f, 0x6e, 0x3e, 0x3c, 0x4c,
    0x61, 0x6e, 0x67, 0x75, 0x61, 0x67, 0x65, 0x3e, 0x65, 0x6e, 0x3c, 0x2f, 0x4c, 0x61, 0x6e, 0x67, 0x75, 0x61, 0x67, 0x65, 0x3e,
};

void doSignTest( int hashType )
{
    IOSCError rv;
    int hashSize = ( hashType == IOSC_HASH_SHA256 ? 32 : 20 );
    Sha1Generator swSha1;
    Sha256Generator swSha256;
    IOSCHashContext hashCtx;
    u8 ioscDigest[32];
    u8 swDigest[32];

    if( hashType == IOSC_HASH_SHA1 )
    {
        swSha1.Initialize();
        swSha1.Update( ecSignInput, sizeof( ecSignInput ) );
        swSha1.GetHash( swDigest, nn::crypto::Sha1Generator::HashSize );
    }
    else
    {
        swSha256.Initialize();
        swSha256.Update( ecSignInput, sizeof( ecSignInput ) );
        swSha256.GetHash( swDigest, nn::crypto::Sha256Generator::HashSize );
    }

    TEST_LOG( "EC Sign test using %s\n", hashType == IOSC_HASH_SHA256 ? "SHA256" : "SHA1" );

    dumpArray( "Software Digest", swDigest, hashSize );

    IOSC_GenerateHash( hashCtx, NULL, 0, IOSC_HASH_FIRST | hashType, NULL );
    rv = IOSC_GenerateHash( hashCtx, ecSignInput, sizeof( ecSignInput ), IOSC_HASH_LAST | hashType, ioscDigest );
    EXPECT_EQ( rv, IOSC_ERROR_OK );
    if( rv != IOSC_ERROR_OK )
    {
        TEST_LOG( "Error on GenerateHash LAST: %d\n", rv );
    }

    TEST_LOG( "Checking %s results ... ", hashType == IOSC_HASH_SHA256 ? "SHA256" : "SHA1" );
    if( compare( static_cast<const u8 *>( ioscDigest ), static_cast<const u8 *>( swDigest ), hashSize ) )
    {
        TEST_LOG( "OK.\n" );
        SUCCEED();
    }
    else
    {
        TEST_LOG( "NG.\n" );
        FAIL();
    }

    dumpArray( "IOSC Digest", ioscDigest, hashSize );
}


// NIST ex #0 from CBCKeySbox128e.txt from AES KAT (Known Answer Test)
// NIST ex #0 from CBCKeySbox128d.txt from AES KAT (Known Answer Test)
NN_ALIGNAS( 64 ) static u8 aesTestKey[16] = {0x10, 0xa5, 0x88, 0x69, 0xd7, 0x4b, 0xe5, 0xa3, 0x74, 0xcf, 0x86, 0x7c, 0xfb, 0x47, 0x38, 0x59};
NN_ALIGNAS( 64 ) static u8 aesTestIv[16];
NN_ALIGNAS( 64 ) static u8 aesTestData[16];
NN_ALIGNAS( 64 ) static u8 aesTestAnswer[16] = {0x6d, 0x25, 0x1e, 0x69, 0x44, 0xb0, 0x51, 0xe0, 0x4e, 0xaa, 0x6f, 0xb4, 0xdb, 0xf7, 0x84, 0x65};

NN_ALIGNAS( 64 ) static u8 aesEncData[16];
NN_ALIGNAS( 64 ) static u8 aesDecData[16];


void doAesTest()
{
    IOSCSecretKeyHandle aesKeyHandle;
    IOSCError err;

    /*
     * Import the AES key
     */
    err = IOSC_CreateObject( &aesKeyHandle, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_CreateObject FAIL with %d\n", err );
    }
    err = IOSC_ImportSecretKey( aesKeyHandle, 0, 0, IOSC_NOSIGN_NOENC, NULL, NULL, aesTestKey );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_ImportKey FAIL with %d\n", err );
    }

    dumpArray( "AES Test Key:", aesTestKey, sizeof( aesTestKey ) );
    dumpArray( "AES Test IV:", aesTestIv, sizeof( aesTestIv ) );
    dumpArray( "AES Encrypt Input:", aesTestData, sizeof( aesTestData ) );
    dumpArray( "AES Expected Output:", aesTestAnswer, sizeof( aesTestAnswer ) );

    memset( aesEncData, 0xa5, sizeof( aesEncData ) );

    /*
     * Now do Encrypt
     */
    err = IOSC_Encrypt( aesKeyHandle, aesTestIv, aesTestData, sizeof( aesTestData ), aesEncData );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_Encrypt FAIL with %d\n", err );
    }

    dumpArray( "AES Encrypt Output:", aesEncData, sizeof( aesEncData ) );
    dumpArray( "AES IV after Encrypt:", aesTestIv, sizeof( aesTestIv ) );

    if( memcmp( aesEncData, aesTestAnswer, sizeof( aesEncData ) ) != 0 )
    {
        TEST_LOG( "FAIL: AES Encrypt NIST test FAILs (data miscompare)\n" );
        FAIL();
    }
    else
    {
        TEST_LOG( "AES Encrypt NIST test OK\n" );
        SUCCEED();
    }

    /*
     * Now decrypt the result and it should get you back where you started
     */
    memset( aesTestIv, 0, sizeof( aesTestIv ) );
    memset( aesDecData, 0x5a, sizeof( aesDecData ) );
    err = IOSC_Decrypt( aesKeyHandle, aesTestIv, aesEncData, sizeof( aesEncData ), aesDecData );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_Decrypt FAIL with %d\n", err );
    }

    dumpArray( "AES Decrypt Output:", aesDecData, sizeof( aesDecData ) );

    if( memcmp( aesDecData, aesTestData, sizeof( aesDecData ) ) != 0 )
    {
        TEST_LOG( "FAIL: AES Decrypt NIST test FAILs (data miscompare)\n" );
        FAIL();
    }
    else
    {
        TEST_LOG( "AES Decrypt NIST test OK\n" );
        SUCCEED();
    }
}

void doContentDecryptTest()
{
    ESTitleId nTitleId = ntohll( TITLE_ID );
    u16 nContIndex = 3;
    IOSCSecretKeyHandle titleKeyHandle;
    IOSCSecretKeyHandle commonKeyHandle;
    IOSCError err;

    NN_UNUSED( titleKey );
    NN_UNUSED( titleKey0 );
    NN_UNUSED( titleKey1 );

    memset( iv, 0, sizeof( iv ) );
    memcpy( iv, (u8 *)&nTitleId, sizeof( nTitleId ) );

    dumpArray( "TitleKey IV:", iv, sizeof( iv ) );

/*
 * Decrypt the title key using the TitleId as the IV and the common key
 */
#define USE_COMMON_KEY_0
#if defined( USE_COMMON_KEY_0 )
    commonKeyHandle = IOSC_COMMON_KEYID_TO_HANDLE( 0 );
    err = IOSC_Decrypt( commonKeyHandle, iv, titleKey0, sizeof( titleKey0 ), aesKey );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_Decrypt FAIL with %d\n", err );
    }
    dumpArray( "TitleKey (encrypted):", titleKey0, sizeof( titleKey0 ) );
#elif defined( USE_COMMON_KEY_1 )
    commonKeyHandle = IOSC_COMMON_KEYID_TO_HANDLE( 1 );
    err = IOSC_Decrypt( commonKeyHandle, iv, titleKey1, sizeof( titleKey1 ), aesKey );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_Decrypt FAIL with %d\n", err );
    }
    dumpArray( "TitleKey (encrypted):", titleKey1, sizeof( titleKey1 ) );
#else
    err = IOSC_Decrypt( IOSC_COMMON_ENC_HANDLE, iv, titleKey, sizeof( titleKey ), aesKey );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_Decrypt FAIL with %d\n", err );
    }
    dumpArray( "TitleKey (encrypted):", titleKey, sizeof( titleKey ) );
#endif
    dumpArray( "TitleKey (cleartext):", aesKey, sizeof( aesKey ) );

    /*
     * Import the title key
     */
    err = IOSC_CreateObject( &titleKeyHandle, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_CreateObject FAIL with %d\n", err );
    }
    err = IOSC_ImportSecretKey( titleKeyHandle, 0, 0, IOSC_NOSIGN_NOENC, NULL, NULL, aesKey );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_ImportKey FAIL with %d\n", err );
    }

    /*
     * Now decrypt the content, which happens to have content index 3
     */
    nContIndex = ntohs( nContIndex );
    memset( iv, 0, sizeof( iv ) );
    memcpy( iv, (u8 *)&nContIndex, sizeof( nContIndex ) );

#ifdef DECRYPT_IN_PLACE
    /*
     * Decrypt in place
     */
    err = IOSC_Decrypt( titleKeyHandle, iv, encApp, sizeof( encApp ), encApp );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_Decrypt FAIL with %d\n", err );
    }

    /* Compare only the unpadded length of the rawApp */
    if( memcmp( rawApp, encApp, sizeof( rawApp ) ) != 0 )
    {
        TEST_LOG( "FAIL: Miscompare after decryption of content in-place!\n" );
        dumpArray( "Decrypted App:", encApp, 32 );
        FAIL();
    }
    else
    {
        TEST_LOG( "Decrypt-and-compare of App OK (in-place): cleartext bytes %d, encrypted bytes %d\n", sizeof( rawApp ), sizeof( encApp ) );
        SUCCEED();
    }
#else
    /*
     * Decrypt to a different output buffer
     */
    err = IOSC_Decrypt( titleKeyHandle, iv, encApp, sizeof( encApp ), __appDecBuf );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "FAIL: IOSC_Decrypt FAIL with %d\n", err );
    }

    /* Compare only the unpadded length of the rawApp */
    if( memcmp( rawApp, __appDecBuf, sizeof( rawApp ) ) != 0 )
    {
        FAIL();
        TEST_LOG( "FAIL: Miscompare after decryption of content (not in place)!\n" );
        dumpArray( "Decrypted App:", __appDecBuf, 32 );
    }
    else
    {
        SUCCEED();
        TEST_LOG( "Decrypt-and-compare of App OK (not in-place): cleartext bytes %d, encrypted bytes %d\n", sizeof( rawApp ), sizeof( encApp ) );
    }
#endif
}

u8 ninDevCaKey[] = {
    0x00, 0xe4, 0x84, 0x23, 0xac, 0x7e, 0x49, 0xe3, 0x69, 0x5d, 0x61, 0x6c, 0x6f, 0xf0, 0x20, 0xfb, 0x6a, 0x5c, 0x47, 0x45,
    0x63, 0x77, 0x4d, 0x55, 0x04, 0x64, 0x50, 0x5f, 0x23, 0x67, 0x01, 0x88, 0x99, 0x94, 0xce, 0xe0, 0x26, 0x0e, 0xbb, 0x8f,
    0x1b, 0x0e, 0xd3, 0xa8, 0xe1, 0x47, 0x12, 0xa2, 0x5a, 0x11, 0xc4, 0x85, 0x77, 0xa3, 0xce, 0xdd, 0xb4, 0xc2, 0x7c, 0x4d,
};

u8 ninProdCaKey[] = {
    0x00, 0x4e, 0x3b, 0xb7, 0x4d, 0x5d, 0x95, 0x9e, 0x68, 0xce, 0x90, 0x04, 0x34, 0xfe, 0x9e, 0x4a, 0x3f, 0x09, 0x4a, 0x33,
    0x77, 0x1f, 0xa7, 0xc0, 0xe4, 0xb0, 0x23, 0x26, 0x4d, 0x98, 0x01, 0x4c, 0xa1, 0xfc, 0x79, 0x9d, 0x3f, 0xa5, 0x21, 0x71,
    0xd5, 0xf9, 0xbd, 0x5b, 0x17, 0x77, 0xec, 0x0f, 0xef, 0x7a, 0x38, 0xd1, 0x66, 0x9b, 0xbf, 0x83, 0x03, 0x25, 0x84, 0x3a,
};

void doCertTest()
{
    IOSCError err;
    IOSCSecretKeyHandle inputprivkeyh;
    IOSCPublicKeyHandle capublickeyh;

    u64 devId;

#ifdef NEED_CHANGE_DEV_CERT
    IOSCPublicKeyHandle mspublickeyh;
    IOSCPublicKeyHandle ninProdCaKeyh;
    IOSCPublicKeyHandle ninDevCaKeyh;
    IOSCPublicKeyHandle devpublickeyh;
#endif

    /*
     * Check that structure definitions are correctly packed
     */
    TEST_LOG( "RSA 4096 RSA Cert size %d\n", sizeof( IOSCRsa4096RsaCert ) );
    if( sizeof( IOSCRsa4096RsaCert ) != 1024 )
    {
        TEST_LOG( "RSA 4096 RSA Cert size incorrect!\n" );
    }
    TEST_LOG( "RSA 2048 RSA Cert size %d\n", sizeof( IOSCRsa2048RsaCert ) );
    if( sizeof( IOSCRsa2048RsaCert ) != 768 )
    {
        TEST_LOG( "RSA 2048 RSA Cert size incorrect!\n" );
    }
    TEST_LOG( "RSA 2048 ECC Cert size %d\n", sizeof( IOSCRsa2048EccCert ) );
    if( sizeof( IOSCRsa2048EccCert ) != 576 )
    {
        TEST_LOG( "RSA 2048 ECC Cert size incorrect!\n" );
    }
    TEST_LOG( "ECC ECC Cert size %d\n", sizeof( IOSCEccEccCert ) );
    if( sizeof( IOSCEccEccCert ) != 384 )
    {
        TEST_LOG( "ECC ECC Cert size incorrect!\n" );
    }

    /*
     * Print out device ID
     */
    err = IOSC_GetData( IOSC_DEV_ID_HANDLE, &devId );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "IOSC_GetData returns %d\n", err );
    }
    TEST_LOG( "Device ID = %08x\n", devId );

    err = IOSC_CreateObject( &capublickeyh, IOSC_PUBLICKEY_TYPE, IOSC_RSA2048_SUBTYPE );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "CreateObject err %d FAIL\n", err );
    }
    err = IOSC_ImportCertificate( &caCert[0], IOSC_ROOT_KEY_HANDLE, capublickeyh );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err == IOSC_ERROR_OK )
    {
        TEST_LOG( "CA Import PASS\n" );
    }
    else
    {
        TEST_LOG( "CA Import err %d FAIL\n", err );
    }

    /*
     * Neg test for bad key type
     */
    IOSC_CreateObject( &inputprivkeyh, IOSC_SECRETKEY_TYPE, IOSC_ECC233_SUBTYPE );
    err = IOSC_ImportCertificate( &caCert[0], IOSC_ROOT_KEY_HANDLE, inputprivkeyh );
    EXPECT_EQ( err, IOSC_ERROR_INVALID_OBJTYPE );
    if( err == IOSC_ERROR_INVALID_OBJTYPE )
    {
        TEST_LOG( "CA Import PASS NEG KEY TYPE TEST\n" );
    }
    else
    {
        TEST_LOG( "CA Import FAIL NEG KEY TYPE TEST\n" );
    }
    IOSC_DeleteObject( inputprivkeyh );

#ifdef NEED_CHANGE_DEV_CERT
    /*
     * Now use the CA public key to import the MS cert
     */
    IOSC_CreateObject( &mspublickeyh, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );

    /* import the MS cert */
    err = IOSC_ImportCertificate( msCert, capublickeyh, mspublickeyh );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "MS Import FAIL, err = %d\n", err );
    }
    else
    {
        TEST_LOG( "MS Import PASS\n" );
    }

    /*
     * Test of GetDeviceCertificate
     */
    err = IOSC_GetDeviceCertificate( (IOSCEccSignedCert *)&returnedcert );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "error in IOSC_GetDeviceCertificate\n" );
    }

    dumpArray( "Device certificate:", (u8 *)&returnedcert, sizeof( IOSCEccEccCert ) );
    /*
     * Import the device certificate, just to prove that ECC operations work
     * XXX this fails using our MS public key, now that the cert is from N
     */
    IOSC_CreateObject( &devpublickeyh, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );
    err = IOSC_ImportCertificate( (u8 *)&returnedcert, mspublickeyh, devpublickeyh );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "Dev Cert Import FAIL with MS key, err %d\n", err );
    }
    else
    {
        TEST_LOG( "Dev Cert Import PASS with MS key\n" );
    }

    /*
     * Now import the Nintendo DevCa public key to import the dev cert
     */
    IOSC_CreateObject( &ninDevCaKeyh, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );
    err = IOSC_ImportPublicKey( ninDevCaKey, 0, ninDevCaKeyh );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "ImportPubKey FAIL with NinDevCa key, err %d\n", err );
    }
    else
    {
        TEST_LOG( "ImportPubKey PASS with NinDevCa key\n" );
    }

    err = IOSC_ImportCertificate( (u8 *)&returnedcert, ninDevCaKeyh, devpublickeyh );
    EXPECT_EQ( err, IOSC_ERROR_FAIL_CHECKVALUE );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "Dev Cert Import FAIL with NinDevCa key, err %d\n", err );
    }
    else
    {
        TEST_LOG( "Dev Cert Import PASS with NinDevCa key\n" );
    }

    /*
     * Now import the Nintendo ProdCa public key to import the dev cert
     */
    IOSC_CreateObject( &ninProdCaKeyh, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );
    err = IOSC_ImportPublicKey( ninProdCaKey, 0, ninProdCaKeyh );
    EXPECT_EQ( err, IOSC_ERROR_OK );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "ImportPubKey FAIL with NinProdCa key, err %d\n", err );
    }
    else
    {
        TEST_LOG( "ImportPubKey PASS with NinProdCa key\n" );
    }

    err = IOSC_ImportCertificate( (u8 *)&returnedcert, ninProdCaKeyh, devpublickeyh );
    EXPECT_EQ( err, IOSC_ERROR_FAIL_CHECKVALUE );
    if( err != IOSC_ERROR_OK )
    {
        TEST_LOG( "Dev Cert Import FAIL with NinProdCa key, err %d\n", err );
    }
    else
    {
        TEST_LOG( "Dev Cert Import PASS with NinProdCa key\n" );
    }
    IOSC_DeleteObject( mspublickeyh );
#endif
}  // NOLINT (readability/fn_size)


void doTestRandoms()
{
    IOSCError err;
    u8 randData[112];  // are you feeling lucky, punk?
    int i;

    memset( randData, 0, sizeof( randData ) );

    for( i = 0; i < 10; i++ )
    {
        err = IOSC_GenerateRand( randData, 97 );
        EXPECT_EQ( err, IOSC_ERROR_OK );
        if( err != IOSC_ERROR_OK )
        {
            TEST_LOG( "GenerateRand FAILs: err %d\n", err );
        }
        dumpArray( "Random data:", randData, sizeof( randData ) );
    }
}


TEST( ICryptoSanityTest, CryptoSanity )
{
    TEST_LOG( "BroadOn crypto test start\n" );

    // nn::fs::Initialize();
    // nn::crypto::Initialize();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::spl::InitializeForEs();
#endif

    IOSC_Initialize();

    doShaTest( IOSC_HASH_SHA1 );
    doShaTest( IOSC_HASH_SHA256 );
    doAesTest();
    doCertTest();
    doContentDecryptTest();
    doTestRandoms();
    doSignTest( IOSC_HASH_SHA256 );

    TEST_LOG( "BroadOn crypto test end\n" );
}
