﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2008, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */
#include <nnt/nntest.h>
#include <nn/nn_Macro.h>

#define TESTNAME "HandleTest"
#define NEED_NUMTICKS

#include "testEs_Testcommon.h"

/* Used only to check IOSC versus CSL type definitions */
#include <nn/csl/csltypes.h>
#include <nn/ioscrypto/iosccert.h>

NN_ALIGNAS( ALIGN_SIZE ) u8 aesKeyImport[16];
NN_ALIGNAS( ALIGN_SIZE ) u8 aesKeyImportOut[16];
NN_ALIGNAS( ALIGN_SIZE ) u8 aesIvImport[16];
NN_ALIGNAS( ALIGN_SIZE ) u8 aesKeyCheck[16];
NN_ALIGNAS( ALIGN_SIZE ) u8 aesKeyCheckOut[16];

IOSCKeyHandle handle[128];

/*
 * To verify export of root key
 */
#include "testEs_Root.h"

static u8 rootPubKeyDevel[sizeof( IOSCRsaPublicKey4096 )] = ROOT_KEY_DEVEL;
static u8 rootPubExpDevel[sizeof( IOSCRsaExponent )] = ROOT_EXP_DEVEL;

static const int TestMaxObjectNum = 7;


void dumpAesKey( IOSCAesKey *keyp )
{
    u32 *wp;

    wp = reinterpret_cast<u32 *>( keyp );
    LOG_MESSAGE( "AES key @%08x = %08x %08x %08x %08x", keyp, *wp, *( wp + 1 ), *( wp + 2 ), *( wp + 3 ) );
}

#define CHECK_SIZE( typesz, len )                                   \
    do                                                              \
    {                                                               \
        if( typesz == len )                                         \
        {                                                           \
            LOG_PASS( "%s == %d", #typesz, len );                   \
        }                                                           \
        else                                                        \
        {                                                           \
            LOG_FAIL( "%s %d should be %d", #typesz, typesz, len ); \
            EXIT( -1 );                                             \
        }                                                           \
    } while( 0 )

#define CHECK_VALUE( val, exp )                                    \
    do                                                             \
    {                                                              \
        if( val == exp )                                           \
        {                                                          \
            LOG_PASS( "alignment/rounding at line %d", __LINE__ ); \
        }                                                          \
        else                                                       \
        {                                                          \
            LOG_FAIL( "alignment/rounding at line %d", __LINE__ ); \
            EXIT( -1 );                                            \
        }                                                          \
    } while( 0 )


/*
 * The IOSC layer calls the CSL layer, but each have their own
 * self-contained data type definitions.  Make sure that the
 * definitions are consistent.
 *
 * Note that historically the IOSC definitions were defined in terms
 * of the CSL definitions, which polluted the external interface.
 * This routine is part of the solution to make the IOSC external
 * interface definition self-contained.
 */
static void checkDataTypes()
{
    CHECK_SIZE( sizeof( IOSCSha1Hash ), 20 );
    CHECK_SIZE( sizeof( IOSCSha256Hash ), 32 );
    CHECK_SIZE( sizeof( IOSCAesKey ), 16 );
    CHECK_SIZE( sizeof( IOSCAesIv ), 16 );
    CHECK_SIZE( sizeof( IOSCHmacKey ), 20 );
    CHECK_SIZE( sizeof( IOSCEccPublicKey ), sizeof( CSLOSEccPublicKey ) );
    CHECK_SIZE( sizeof( IOSCEccPrivateKey ), sizeof( CSLOSEccPrivateKey ) );
    CHECK_SIZE( sizeof( IOSCRsaPublicKey2048 ), sizeof( CSLOSRsaPublicKey2048 ) );
    CHECK_SIZE( sizeof( IOSCRsaPublicKey4096 ), sizeof( CSLOSRsaPublicKey4096 ) );
    CHECK_SIZE( sizeof( IOSCRsaExponent ), sizeof( CSLOSRsaExponent ) );
    CHECK_SIZE( sizeof( IOSCRsaSig2048 ), sizeof( CSLOSRsaSig2048 ) );
    CHECK_SIZE( sizeof( IOSCRsaSig4096 ), sizeof( CSLOSRsaSig4096 ) );
    CHECK_SIZE( sizeof( IOSCEccEccCert ), 384 );
    CHECK_SIZE( sizeof( IOSCRsa2048EccCert ), 576 );
    CHECK_SIZE( sizeof( IOSCRsa2048RsaCert ), 768 );
    CHECK_SIZE( sizeof( IOSCRsa4096RsaCert ), 1024 );
    CHECK_SIZE( sizeof( IOSCRootCert ), 1280 );
}


/*
 * Declarations that use all of the size and alignment macros.  On most platforms
 * these don't do anything, but this makes sure that they at least compile.
 */
static u8 __aesBuffer[SIZE_AES_ALIGN( 1 )] ATTR_AES_ALIGN;
static u8 __shaBuffer[SIZE_SHA_ALIGN( 1 )] ATTR_SHA_ALIGN;
static u8 __sha1Buffer[SIZE_SHA1_ALIGN( 1 )] ATTR_SHA1_ALIGN;
static u8 __sha256Buffer[SIZE_SHA256_ALIGN( 1 )] ATTR_SHA256_ALIGN;
static u8 __rsaBuffer[SIZE_RSA_ALIGN( 1 )] ATTR_RSA_ALIGN;


/*
 * Test the macros defined in the platform-specific header for IOSC
 * that are used to align buffers that may be DMA targets on some
 * platforms.
 */
static void checkAlignmentMacros()
{
    s32 sIntBefore, sIntAfter;
    u32 uIntBefore, uIntAfter;
    s64 sLongLongBefore, sLongLongAfter;
    u64 uLongLongBefore, uLongLongAfter;

    /*
     * Check the IOSC_ROUND_UP macro.  It only works with power 2 sizes, but should handle
     * signed and unsigned data of all sizes.
     */
    sIntBefore = 0;
    sIntAfter = IOSC_ROUND_UP( sIntBefore, 32 );
    CHECK_VALUE( sIntAfter, 0 );
    sIntBefore = 1;
    sIntAfter = IOSC_ROUND_UP( sIntBefore, 32 );
    CHECK_VALUE( sIntAfter, 32 );
    sIntBefore = 0xffffff01;
    sIntAfter = IOSC_ROUND_UP( sIntBefore, 32 );
    CHECK_VALUE( sIntAfter, 0xffffff20 );

    uIntBefore = 0;
    uIntAfter = IOSC_ROUND_UP( uIntBefore, 32 );
    CHECK_VALUE( uIntAfter, 0 );
    uIntBefore = 1;
    uIntAfter = IOSC_ROUND_UP( uIntBefore, 32 );
    CHECK_VALUE( uIntAfter, 32 );
    uIntBefore = 0xffffff01;
    uIntAfter = IOSC_ROUND_UP( uIntBefore, 32 );
    CHECK_VALUE( uIntAfter, 0xffffff20 );

    sLongLongBefore = 0;
    sLongLongAfter = IOSC_ROUND_UP( sLongLongBefore, 32 );
    CHECK_VALUE( sLongLongAfter, 0 );
    sLongLongBefore = 1LL;
    sLongLongAfter = IOSC_ROUND_UP( sLongLongBefore, 32 );
    CHECK_VALUE( sLongLongAfter, 32 );
    sLongLongBefore = 0xffffff01LL;
    sLongLongAfter = IOSC_ROUND_UP( sLongLongBefore, 32 );
    CHECK_VALUE( sLongLongAfter, 0xffffff20LL );
    sLongLongBefore = 0x28ffffff00LL;
    sLongLongAfter = IOSC_ROUND_UP( sLongLongBefore, 32 );
    CHECK_VALUE( sLongLongAfter, 0x28ffffff00LL );
    sLongLongBefore = 0x42ffffffd1LL;
    sLongLongAfter = IOSC_ROUND_UP( sLongLongBefore, 32 );
    CHECK_VALUE( sLongLongAfter, 0x42ffffffe0LL );
    sLongLongBefore = -3LL;
    sLongLongAfter = IOSC_ROUND_UP( sLongLongBefore, 32 );
    CHECK_VALUE( sLongLongAfter, 0 );
    sLongLongBefore = -63LL;
    sLongLongAfter = IOSC_ROUND_UP( sLongLongBefore, 32 );
    CHECK_VALUE( sLongLongAfter, -32LL );

    uLongLongBefore = 0;
    uLongLongAfter = IOSC_ROUND_UP( uLongLongBefore, 32 );
    CHECK_VALUE( uLongLongAfter, 0 );
    uLongLongBefore = 1;
    uLongLongAfter = IOSC_ROUND_UP( uLongLongBefore, 32 );
    CHECK_VALUE( uLongLongAfter, 32 );
    uLongLongBefore = 0xffffff01LL;
    uLongLongAfter = IOSC_ROUND_UP( uLongLongBefore, 32 );
    CHECK_VALUE( uLongLongAfter, 0xffffff20LL );
    uLongLongBefore = 0x28ffffff00LL;
    uLongLongAfter = IOSC_ROUND_UP( uLongLongBefore, 64 );
    CHECK_VALUE( uLongLongAfter, 0x28ffffff00LL );
    uLongLongBefore = 0x42ffffff91LL;
    uLongLongAfter = IOSC_ROUND_UP( uLongLongBefore, 64 );
    CHECK_VALUE( uLongLongAfter, 0x42ffffffc0LL );
    uLongLongBefore = static_cast<u64>( -3LL );
    uLongLongAfter = IOSC_ROUND_UP( uLongLongBefore, 64 );
    CHECK_VALUE( uLongLongAfter, 0ULL );
    uLongLongBefore = static_cast<u64>( -163LL );
    uLongLongAfter = IOSC_ROUND_UP( uLongLongBefore, 64 );
    CHECK_VALUE( uLongLongAfter, static_cast<u64>( -128L ) );

    /*
     * The higher level alignment macros are a little harder to test, since they do nothing
     * on most platforms.
     */
    LOG_MESSAGE( "AES buffer address %08x size %d", __aesBuffer, sizeof( __aesBuffer ) );
    LOG_MESSAGE( "SHA buffer address %08x size %d", __shaBuffer, sizeof( __shaBuffer ) );
    LOG_MESSAGE( "SHA1 buffer address %08x size %d", __sha1Buffer, sizeof( __sha1Buffer ) );
    LOG_MESSAGE( "SHA256 buffer address %08x size %d", __sha256Buffer, sizeof( __sha256Buffer ) );
    LOG_MESSAGE( "RSA buffer address %08x size %d", __rsaBuffer, sizeof( __rsaBuffer ) );
}


static s32 testSizes( IOSCObjectType type, IOSCObjectSubType subtype, u32 expKeySize, u32 expSignSize )
{
    u32 handle1;
    s32 error;
    u32 keySize, signSize;

    error = IOSC_CreateObject( &handle1, type, subtype );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "testSizes: no error expected!" );
        goto end;
    }

    error = IOSC_GetKeySize( &keySize, handle1 );
    if( ( error != IOSC_ERROR_OK ) || ( keySize != expKeySize ) )
    {
        LOG_FAIL( "testSizes: GetKey Size" );
        goto end;
    }

    error = IOSC_GetSignatureSize( &signSize, handle1 );

    if( ( expSignSize == 0 ) && ( error != IOSC_ERROR_OK ) )
    {
        LOG_PASS( "testSizes: no sign size expected" );
    }
    else if( ( error == IOSC_ERROR_OK ) && ( signSize == expSignSize ) )
    {
        LOG_PASS( "testSizes: GetSignSize returned %d", signSize );
    }
    else
    {
        LOG_FAIL( "testSizes: GetSignSize returned %d", signSize );
        goto end;
    }
end:
    error = IOSC_DeleteObject( handle1 );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "testSizes: DeleteObject failed" );
    }
    return error;
}

NN_ALIGNAS( ALIGN_SIZE ) u8 randData[64];

extern int doHandleTests( void );

int doHandleTests()
{
    IOSCKeyHandle handle1, handle2, handle3, handle4;
    u64 deviceId;
    s32 error;
    int i, j;
    IOSCRsaPublicKey4096 publicKeyData, publicKeyDataOut;
    IOSCRsaExponent exp;
    int numHandles;
#ifdef PERF
    u32 startTime, endTime;
#endif

    checkDataTypes();
    checkAlignmentMacros();

const int NUM_DEF_HANDLES = 12;

    error = IOSC_GetData( IOSC_DEV_ID_HANDLE, &deviceId );
    LOG_MESSAGE( "\nDevice id = %08x", deviceId );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "error from IOSC_GetData (error %d)", error );
        EXIT( -1 );
    }

    error = IOSC_GenerateRand( randData, 63 );
    CHECK_EQUAL( error, IOSC_ERROR_OK, "error from IOSC_GenerateRand" );

    TUL_PRINTF( "Randoms->\n" );
    for( i = 0; i < 63; i++ )
    {
        TUL_PRINTF( " %02x ", randData[i] );
    }

    error = IOSC_GenerateRand( randData, 10 );
    CHECK_EQUAL( error, IOSC_ERROR_OK, "error from IOSC_GenerateRand" );

    TUL_PRINTF( "\nRandoms->\n" );
    for( i = 0; i < 10; i++ )
    {
        TUL_PRINTF( " %02x ", randData[i] );
    }
    TUL_PRINTF( "\n" );

    /*
     * Deleting any of the default handle should fail with EACCESS
     */
    error = IOSC_DeleteObject( IOSC_DEV_SIGNING_KEY_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted Dev Signing Key" );

    error = IOSC_DeleteObject( IOSC_DEV_ID_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted Dev ID" );

    error = IOSC_DeleteObject( IOSC_FS_ENC_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted FS Enc" );

    error = IOSC_DeleteObject( IOSC_FS_MAC_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted FS MAC" );

    error = IOSC_DeleteObject( IOSC_COMMON_ENC_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted Common Enc" );

    error = IOSC_DeleteObject( IOSC_COMMON2_ENC_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted Common2 Enc" );

    error = IOSC_DeleteObject( IOSC_BACKUP_ENC_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted Backup Enc" );

    error = IOSC_DeleteObject( IOSC_APP_ENC_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted App Enc" );

    error = IOSC_DeleteObject( IOSC_ROOT_KEY_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted Root Key" );

    error = IOSC_DeleteObject( IOSC_BOOTOSVER_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted Boot OS Ver" );

    error = IOSC_DeleteObject( IOSC_CACRLVER_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted CACRL Ver" );

    error = IOSC_DeleteObject( IOSC_SIGNERCRLVER_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted SingerCRL Ver" );

    error = IOSC_DeleteObject( IOSC_FSVER_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_ACCESS, "Deleted FS Ver" );

    /* Handle Create/Delete tests */

    IOSC_CreateObject( &handle1, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );

    if( handle1 != ( NUM_DEF_HANDLES ) )
    {
        LOG_FAIL( "handle 1 FAIL %d (expected %d)\n", handle1, NUM_DEF_HANDLES );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "create handle 1" );
    }
    IOSC_CreateObject( &handle2, IOSC_SECRETKEY_TYPE, IOSC_MAC_SUBTYPE );

    if( handle2 != ( NUM_DEF_HANDLES + 1 ) )
    {
        LOG_FAIL( "handle 2 FAIL" );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "handle 2 PASS" );
    }
    IOSC_CreateObject( &handle3, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );

    if( handle3 != ( NUM_DEF_HANDLES + 2 ) )
    {
        LOG_FAIL( "handle 3 FAIL" );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "handle 3 PASS" );
    }
    IOSC_DeleteObject( handle1 );
    IOSC_CreateObject( &handle1, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    if( handle1 != ( NUM_DEF_HANDLES ) )
    {
        LOG_FAIL( "handle 1 realloc" );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "handle 1 realloc" );
    }
    IOSC_CreateObject( &handle4, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    if( handle4 != ( NUM_DEF_HANDLES + 3 ) )
    {
        LOG_FAIL( "handle 4 FAIL" );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "handle 4 PASS" );
    }

    IOSC_DeleteObject( handle1 );
    IOSC_DeleteObject( handle2 );
    IOSC_DeleteObject( handle3 );
    IOSC_DeleteObject( handle4 );

    for( i = 0; i < TestMaxObjectNum; i++ )
    {
        if( IOSC_CreateObject( &handle[i], IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE ) != IOSC_ERROR_OK )
        {
            TUL_PRINTF( "Created %d handles\n", i );
            break;
        }
    }
    numHandles = i;
    for( i = 0; i < numHandles; i++ )
    {
        if( IOSC_DeleteObject( handle[i] ) != IOSC_ERROR_OK )
        {
            TUL_PRINTF( "Deleted %d handles\n", i );
            break;
        }
    }

    TUL_PRINTF( "Created and Deleted once\n" );
    for( i = 0; i < TestMaxObjectNum; i++ )
    {
        if( IOSC_CreateObject( &handle[i], IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE ) != IOSC_ERROR_OK )
        {
            TUL_PRINTF( "Created %d handles\n", i );
            break;
        }
    }

    for( i = 0; i < numHandles; i++ )
    {
        if( IOSC_DeleteObject( handle[i] ) != IOSC_ERROR_OK )
        {
            TUL_PRINTF( "Deleted %d handles\n", i );
            break;
        }
    }

    for( i = 0; i < TestMaxObjectNum; i++ )
    {
        if( IOSC_CreateObject( &handle[i], IOSC_PUBLICKEY_TYPE, IOSC_RSA2048_SUBTYPE ) != IOSC_ERROR_OK )
        {
            TUL_PRINTF( "Created %d handles\n", i );
            break;
        }
    }

    for( i = 0; i < numHandles; i++ )
    {
        if( IOSC_DeleteObject( handle[i] ) != IOSC_ERROR_OK )
        {
            TUL_PRINTF( "Deleted %d handles\n", i );
            break;
        }
    }

    for( i = 0; i < TestMaxObjectNum; i++ )
    {
        if( IOSC_CreateObject( &handle[i], IOSC_KEYPAIR_TYPE, IOSC_ECC233_SUBTYPE ) != IOSC_ERROR_OK )
        {
            TUL_PRINTF( "Created %d handles\n", i );
            break;
        }
    }

    for( i = 0; i < numHandles; i++ )
    {
        if( IOSC_DeleteObject( handle[i] ) != IOSC_ERROR_OK )
        {
            TUL_PRINTF( "Deleted %d handles\n", i );
            break;
        }
    }

    if( IOSC_CreateObject( &handle[i], IOSC_KEYPAIR_TYPE, IOSC_RSA2048_SUBTYPE ) != IOSC_ERROR_OK )
    {
        TUL_PRINTF( "Created %d handles\n", i );
    }

    for( i = 0; i < TestMaxObjectNum; i++ )
    {
        if( IOSC_CreateObject( &handle[i], IOSC_KEYPAIR_TYPE, IOSC_ECC233_SUBTYPE ) != IOSC_ERROR_OK )
        {
            TUL_PRINTF( "Created %d handles\n", i );
            break;
        }
    }

    for( i = 0; i < numHandles; i++ )
    {
        if( IOSC_DeleteObject( handle[i] ) != IOSC_ERROR_OK )
        {
            TUL_PRINTF( "Deleted %d handles\n", i );
            break;
        }
    }

#ifdef PERF
    startTime = IO_READ( SYS_TIMER_REG );
    error = IOSC_CreateObject( &handle[0], IOSC_PUBLICKEY_TYPE, IOSC_RSA4096_SUBTYPE );
    endTime = IO_READ( SYS_TIMER_REG );
    TUL_PRINTF( "Number of Ticks for Largest key IOSC_CreateObject = %d, Time  = %d us\n", getNumTicks( startTime, endTime ),
                getNumTicks( startTime, endTime ) * NUM_TICKS / CLOCK_FREQ );
    CHECK_EQUAL( error, IOSC_ERROR_OK, "error in IOSC call" );

    startTime = IO_READ( SYS_TIMER_REG );
    error = IOSC_DeleteObject( handle[0] );
    endTime = IO_READ( SYS_TIMER_REG );
    TUL_PRINTF( "Number of Ticks for Largest key IOSC_DeleteObject = %d, Time = %d us\n", getNumTicks( startTime, endTime ),
                getNumTicks( startTime, endTime ) * NUM_TICKS / CLOCK_FREQ );
    CHECK_EQUAL( error, IOSC_ERROR_OK, "error in IOSC call" );
#endif

    error = IOSC_ERROR_OK;
    i = 0;
    while( i < TestMaxObjectNum && error == IOSC_ERROR_OK )
    {
        if( ( i % 4 ) == 0 )
        {
            error = IOSC_CreateObject( &handle[i], IOSC_PUBLICKEY_TYPE, IOSC_RSA4096_SUBTYPE );
            CHECK_EQUAL( error, IOSC_ERROR_OK, "Create object handle" );
        }
        else if( ( i % 4 ) == 1 )
        {
            error = IOSC_CreateObject( &handle[i], IOSC_PUBLICKEY_TYPE, IOSC_RSA2048_SUBTYPE );
            CHECK_EQUAL( error, IOSC_ERROR_OK, "Create object handle" );
        }
        else if( ( i % 4 ) == 2 )
        {
            error = IOSC_CreateObject( &handle[i], IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
            CHECK_EQUAL( error, IOSC_ERROR_OK, "Create object handle" );
        }
        else if( ( i % 4 ) == 3 )
        {
            error = IOSC_CreateObject( &handle[i], IOSC_SECRETKEY_TYPE, IOSC_MAC_SUBTYPE );
            CHECK_EQUAL( error, IOSC_ERROR_OK, "Create object handle" );
        }
        i++;
    }
    /* Delete all of them twice to check deleting non-existent object fails */
    for( j = 0; j < i; j++ )
    {
        error = IOSC_DeleteObject( handle[j] );
        CHECK_EQUAL( error, IOSC_ERROR_OK, "Deleted object handle" );
        error = IOSC_DeleteObject( handle[j] );
        CHECK_NOT_EQUAL( error, IOSC_ERROR_OK, "Deleted non-existent object handle" );
    }
    LOG_PASS( "Negative test:  deletion of non-existent object fails as expected" );

    /* Creating object should return first user object */
    error = IOSC_CreateObject( &handle1, IOSC_SECRETKEY_TYPE, IOSC_MAC_SUBTYPE );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "Creating Object after one pass of create/delete" );
    }
    else
    {
        LOG_PASS( "Creating Object after one pass of create/delete" );
    }
    if( handle1 != ( NUM_DEF_HANDLES ) )
    {
        LOG_MESSAGE( "handle created = %d NUM_DEF_HANDLES = %d", handle1, NUM_DEF_HANDLES );

        LOG_FAIL( "Handle 1 reassign %d", handle1 );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "Handle 1 reassign" );
    }


    error = IOSC_DeleteObject( handle1 );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "Deleting re-assigned object" );
    }

    /* Test of GetKeySize and GetSignatureSize */

    error = testSizes( IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE, sizeof( IOSCAesKey ), 0 );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "No error expected!" );
    }
    error = testSizes( IOSC_SECRETKEY_TYPE, IOSC_MAC_SUBTYPE, sizeof( IOSCHmacKey ), sizeof( IOSCHmacKey ) );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "No error expected!" );
    }
    error = testSizes( IOSC_PUBLICKEY_TYPE, IOSC_RSA2048_SUBTYPE, sizeof( IOSCRsaPublicKey2048 ), sizeof( IOSCRsaPublicKey2048 ) );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "No error expected!" );
    }
    error = testSizes( IOSC_PUBLICKEY_TYPE, IOSC_RSA4096_SUBTYPE, sizeof( IOSCRsaPublicKey4096 ), sizeof( IOSCRsaPublicKey4096 ) );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "No error expected!" );
    }

    /* Tests of Import and Export */

    /* Fill some data in public keys */
    for( i = 0; i < sizeof( IOSCRsaPublicKey4096 ); i++ )
    {
        publicKeyData[i] = i % 0xff;
    }
    exp[0] = 0x01;
    exp[1] = 0x00;
    exp[2] = 0x01;
    exp[3] = 0x0;

    memset( publicKeyDataOut, 0x41, sizeof( publicKeyDataOut ) );
    IOSC_CreateObject( &handle1, IOSC_PUBLICKEY_TYPE, IOSC_RSA2048_SUBTYPE );
    IOSC_ImportPublicKey( publicKeyData, exp, handle1 );
    IOSC_ExportPublicKey( publicKeyDataOut, exp, handle1 );
    IOSC_DeleteObject( handle1 );
    if( memcmp( publicKeyData, publicKeyDataOut, sizeof( IOSCRsaPublicKey2048 ) ) == 0x0 )
    {
        LOG_PASS( "Import Export Public Key 2048 bits" );
    }
    else
    {
        LOG_FAIL( "Import Export Public Key 2048 bits" );
        EXIT( -1 );
    }
    memset( publicKeyDataOut, 0x42, sizeof( publicKeyDataOut ) );
    IOSC_CreateObject( &handle1, IOSC_PUBLICKEY_TYPE, IOSC_RSA4096_SUBTYPE );
    IOSC_ImportPublicKey( publicKeyData, exp, handle1 );
    IOSC_ExportPublicKey( publicKeyDataOut, exp, handle1 );
    IOSC_DeleteObject( handle1 );
    if( memcmp( publicKeyData, publicKeyDataOut, sizeof( IOSCRsaPublicKey4096 ) ) == 0x0 )
    {
        LOG_PASS( "Import Export Public Key 4096 bits" );
    }
    else
    {
        LOG_FAIL( "Import Export Public Key 4096 bits" );
        EXIT( -1 );
    }

    memset( publicKeyDataOut, 0x43, sizeof( publicKeyDataOut ) );
    IOSC_CreateObject( &handle1, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );
    IOSC_ImportPublicKey( publicKeyData, 0, handle1 );
    IOSC_ExportPublicKey( publicKeyDataOut, 0, handle1 );
    IOSC_DeleteObject( handle1 );
    if( memcmp( publicKeyData, publicKeyDataOut, sizeof( IOSCEccPublicKey ) ) == 0x0 )
    {
        LOG_PASS( "Import Export Public Key 233 bits" );
    }
    else
    {
        LOG_FAIL( "Import Export Public Key 233 bits" );
        EXIT( -1 );
    }

    /*
     * Export the root pubkey
     */
    memset( publicKeyDataOut, 0x45, sizeof( publicKeyDataOut ) );
    memset( exp, 0x47, sizeof( exp ) );
    error = IOSC_ExportPublicKey( publicKeyDataOut, exp, IOSC_ROOT_KEY_HANDLE );
    CHECK_EQUAL( error, IOSC_ERROR_OK, "ExportPublicKey Failed for ROOT" );

    /*
     * Check the actual values returned
     */
    if( memcmp( publicKeyDataOut, rootPubKeyDevel, sizeof( IOSCRsaPublicKey4096 ) ) )
    {
        LOG_FAIL( "Export Root Public Key" );
        EXIT( -1 );
    }
    else if( memcmp( exp, rootPubExpDevel, sizeof( IOSCRsaExponent ) ) )
    {
        LOG_FAIL( "Export Root Public Exponent" );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "Export Root Public Key" );
    }

    /*
     * Import/Export Secret Key Tests
     *
     * Try the simple case where we can get the key out
     */
    for( i = 0; i < sizeof( IOSCAesKey ); i++ )
    {
        aesKeyImport[i] = static_cast<u8>( i );
    }
    memset( aesKeyImportOut, 0x30, sizeof( aesKeyImportOut ) );
    IOSC_CreateObject( &handle1, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    IOSC_ImportSecretKey( handle1, 0, 0, IOSC_NOSIGN_NOENC, 0, 0, aesKeyImport );
    IOSC_ExportSecretKey( handle1, 0, 0, IOSC_NOSIGN_NOENC, 0, 0, aesKeyImportOut );
    error = IOSC_DeleteObject( handle1 );
    if( error != IOSC_ERROR_OK )
    {
        LOG_MESSAGE( "DeleteObject handle1 %d error = %d", handle1, error );
    }

    if( memcmp( aesKeyImport, aesKeyImportOut, sizeof( IOSCAesKey ) ) == 0x0 )
    {
        LOG_PASS( "Import Export Secret Key" );
    }
    else
    {
        LOG_FAIL( "Import Export Secret Key" );
        EXIT( -1 );
    }

    /* try with encryption & not default handle */
    error = IOSC_CreateObject( &handle1, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "CreateObject handle1 SECRETKEY ENC_SUBTYPE error = %d", error );
        EXIT( -1 );
    }
    error = IOSC_CreateObject( &handle2, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "CreateObject handle2 SECRETKEY ENC_SUBTYPE error = %d", error );
        EXIT( -1 );
    }
    error = IOSC_ImportSecretKey( handle2, 0, 0, IOSC_NOSIGN_NOENC, 0, 0, aesKeyImport );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "ImportSecretKey handle2 %d error = %d", handle2, error );
        EXIT( -1 );
    }
    memset( aesIvImport, 0x1, sizeof( aesIvImport ) );
    error = IOSC_ImportSecretKey( handle1, 0, handle2, IOSC_NOSIGN_ENC, 0, aesIvImport, aesKeyImport );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "ImportSecretKey handle1 %d handle2 %d error = %d", handle1, handle2, error );
        EXIT( -1 );
    }
    memset( aesIvImport, 0x1, sizeof( aesIvImport ) );
    memset( aesKeyImportOut, 0x31, sizeof( aesKeyImportOut ) );
    error = IOSC_ExportSecretKey( handle1, 0, handle2, IOSC_NOSIGN_ENC, 0, aesIvImport, aesKeyImportOut );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "ExportSecretKey handle1 %d handle2 %d error = %d", handle1, handle2, error );
        EXIT( -1 );
    }

    /* just to check, export without encryption and check outside */
    error = IOSC_ExportSecretKey( handle1, 0, 0, IOSC_NOSIGN_NOENC, 0, 0, aesKeyCheck );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "ExportSecretKey handle1 %d error = %d", handle1, error );
        EXIT( -1 );
    }

    /* decrypt handle1 using handle2 keys and check */
    memset( aesIvImport, 0x1, sizeof( aesIvImport ) );
    error = IOSC_Encrypt( handle2, aesIvImport, aesKeyCheck, sizeof( IOSCAesKey ), aesKeyCheckOut );
    IOSC_DeleteObject( handle1 );
    IOSC_DeleteObject( handle2 );

    /* This should equal encrypted blob imported */

    if( memcmp( aesKeyImport, aesKeyCheckOut, sizeof( IOSCAesKey ) ) == 0x0 )
    {
        LOG_PASS( "Import Export Secret Key Check value in keystore" );
    }
    else
    {
        LOG_FAIL( "Import Export Secret Key Check value in keystore" );
        EXIT( -1 );
    }

    /* Check the value when decrypted by IOSC layer */

    if( memcmp( aesKeyImport, aesKeyImportOut, sizeof( IOSCAesKey ) ) == 0x0 )
    {
        LOG_PASS( "Import Export Secret Key with Encryption" );
    }
    else
    {
        LOG_FAIL( "Import Export Secret Key with Encryption" );
        EXIT( -1 );
    }

    /* Check that encryption actually did something: these better not be == */

    if( memcmp( aesKeyImport, aesKeyCheck, sizeof( IOSCAesKey ) ) != 0x0 )
    {
        LOG_PASS( "Import Export Secret Key Encryption" );
    }
    else
    {
        LOG_FAIL( "Import Export Secret Key Encryption" );
        EXIT( -1 );
    }

#if !defined( NN_PLATFORM_CTR )
    /*
     * Now repeat the above experiment using the common key
     */
    error = IOSC_CreateObject( &handle1, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "CreateObject handle1 %d error = %d", handle1, error );
        EXIT( -1 );
    }
#if !defined( SDK_TWL )
    handle2 = IOSC_COMMON_ENC_HANDLE;
#else
    /* The original common key is not defined for TWL */
    handle2 = IOSC_COMMON2_ENC_HANDLE;
#endif

    memset( aesIvImport, 0x1, sizeof( aesIvImport ) );
    error = IOSC_ImportSecretKey( handle1, 0, handle2, IOSC_NOSIGN_ENC, 0, aesIvImport, aesKeyImport );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "ImportSecretKey handle1 %d common %d error = %d", handle1, handle2, error );
        EXIT( -1 );
    }
    memset( aesIvImport, 0x1, sizeof( aesIvImport ) );
    memset( aesKeyImportOut, 0x32, sizeof( aesKeyImportOut ) );
    error = IOSC_ExportSecretKey( handle1, 0, handle2, IOSC_NOSIGN_ENC, 0, aesIvImport, aesKeyImportOut );
    if( error != IOSC_ERROR_OK )
    {
        LOG_FAIL( "ExportSecretKey handle1 %d common %d error = %d", handle1, handle2, error );
        EXIT( -1 );
    }
    IOSC_DeleteObject( handle1 );

    if( memcmp( aesKeyImport, aesKeyImportOut, sizeof( IOSCAesKey ) ) == 0x0 )
    {
        LOG_PASS( "Import Export Secret Key with Encryption" );
    }
    else
    {
        LOG_FAIL( "Import Export Secret Key with Encryption" );
        dumpAesKey( (IOSCAesKey *)&aesKeyImport );
        dumpAesKey( (IOSCAesKey *)&aesKeyImportOut );
        EXIT( -1 );
    }
#endif  // !NN_PLATFORM_CTR

    return 0;
}  // NOLINT (readability/fn_size)

#if defined( SEPARATE_MAIN_PROGRAM )

int main( int argc, char **argv )
{
    IOSC_Initialize();
    doHandleTests();
    EXIT( 0 );
    return 0;
}
#endif
