﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"
#include "../../Common/testEns_NetworkUtility.h"

#include "testEns_TaskSimulateServiceError.h"

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];
}

namespace
{
    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::ens::StartServiceLoop("acbaa", g_ServiceWorkMemory, sizeof (g_ServiceWorkMemory));
    }

    void CreateSimulateServiceErrorTask(nn::ens::AsyncContext* pOutContext,
        int statusCode, const char* pCategory, const char* pServerCode) NN_NOEXCEPT
    {
        NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, nnt::ens::TaskSimulateServiceError);

        pTask->SetParameter(statusCode, pCategory, pServerCode);

        nn::ens::detail::core::RegisterTask(pTask, pOutContext);
    }

    nn::Result SimulateServiceError(int statusCode, const char* pCategory, const char* pServerCode) NN_NOEXCEPT
    {
        nn::ens::AsyncContext context;

        CreateSimulateServiceErrorTask(&context, statusCode, pCategory, pServerCode);

        context.GetEvent().Wait();

        return context.GetResult();
    }

    nn::Result SimulateApplicationSpecificError(int32_t* pOutErrorDetail, const char* pServerCode) NN_NOEXCEPT
    {
        nn::ens::AsyncContext context;

        CreateSimulateServiceErrorTask(&context, 400, "app", pServerCode);

        context.GetEvent().Wait();

        *pOutErrorDetail = context.GetErrorDetail();

        return context.GetResult();
    }
}

class EnsServiceErrorSimulator : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        nnt::ens::ReserveSockets();
        nnt::ens::InitializeLibcurl();
        nnt::ens::EnableCommunicationLogDump();
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        nnt::ens::FinalizeLibcurl();
        nnt::ens::CancelSocketsReservation();

        nn::socket::Finalize();
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
            g_ThreadStack, sizeof (g_ThreadStack), nn::os::DefaultThreadPriority + 1));

        nn::os::StartThread(&g_Thread);

        nn::nifm::SubmitNetworkRequestAndWait();
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::nifm::CancelNetworkRequest();

        nn::ens::StopServiceLoop();

        nn::os::DestroyThread(&g_Thread);
    }
};

TEST_F(EnsServiceErrorSimulator, GeneralServerError)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError304, SimulateServiceError(304, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError3xx, SimulateServiceError(399, "xxx", "3001"));

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError400, SimulateServiceError(400, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError401, SimulateServiceError(401, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError403, SimulateServiceError(403, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError404, SimulateServiceError(404, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError405, SimulateServiceError(405, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError407, SimulateServiceError(407, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError410, SimulateServiceError(410, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError413, SimulateServiceError(413, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError414, SimulateServiceError(414, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError416, SimulateServiceError(416, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError4xx, SimulateServiceError(499, "xxx", "3001"));

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError500, SimulateServiceError(500, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError502, SimulateServiceError(502, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError503, SimulateServiceError(503, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError504, SimulateServiceError(504, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError509, SimulateServiceError(509, "xxx", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError5xx, SimulateServiceError(599, "xxx", "3001"));

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerErrorXxx, SimulateServiceError(999, "xxx", "3001"));

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError400, SimulateServiceError(400, "lib", "0"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError400, SimulateServiceError(400, "lib", "10000"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError400, SimulateServiceError(400, "lib", "012x"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError400, SimulateServiceError(400, "lib", "xxxx"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultServerError400, SimulateServiceError(400, "lib", "-999"));
}

TEST_F(EnsServiceErrorSimulator, GeneralServiceError)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultMissingRequiredParameter, SimulateServiceError(400, "lib", "3001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultInvalidParameter,         SimulateServiceError(400, "lib", "3002"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultForbidden,                SimulateServiceError(403, "lib", "3003"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultRoutingError,             SimulateServiceError(404, "lib", "3004"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultNotFound,                 SimulateServiceError(404, "lib", "3005"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultUniqueIdGenerationFailed, SimulateServiceError(500, "lib", "3101"));
}

TEST_F(EnsServiceErrorSimulator, AuthenticationError)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultNetworkServiceAccountIdTokenVerificationFailed, SimulateServiceError(401, "lib", "4001"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultNetworkServiceAccountIdTokenExpired,            SimulateServiceError(401, "lib", "4002"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultAuthTokenVerificationFailed,                    SimulateServiceError(401, "lib", "4101"));
    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultAuthTokenExpired,                               SimulateServiceError(401, "lib", "4102"));

    ASSERT_TRUE(nn::ens::ResultNetworkServiceAccountIdTokenLapsed::Includes(SimulateServiceError(401, "lib", "4001")));
    ASSERT_TRUE(nn::ens::ResultNetworkServiceAccountIdTokenLapsed::Includes(SimulateServiceError(401, "lib", "4002")));
}

TEST_F(EnsServiceErrorSimulator, UnexpectedServiceError)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultUnexpectedServiceError, SimulateServiceError(401, "lib", "9999"));
}

TEST_F(EnsServiceErrorSimulator, ApplicationSpecificError)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    int32_t errorDetail;

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultApplicationSpecificError, SimulateApplicationSpecificError(&errorDetail, "0000"));
    EXPECT_EQ(errorDetail, 0);

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultApplicationSpecificError, SimulateApplicationSpecificError(&errorDetail, "0001"));
    EXPECT_EQ(errorDetail, 1);

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultApplicationSpecificError, SimulateApplicationSpecificError(&errorDetail, "9998"));
    EXPECT_EQ(errorDetail, 9998);

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultApplicationSpecificError, SimulateApplicationSpecificError(&errorDetail, "9999"));
    EXPECT_EQ(errorDetail, 9999);
}
