﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"

#include <nn/ens/detail/util/ens_JsonReader.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>

TEST(EnsJsonReader, Case1)
{
    nn::ens::detail::util::JsonReader reader;

    char data[] =
    {
        R"({"compact":true,"schema":0})"
    };

    for (size_t i = 0; i < sizeof (data) - 1; i++)
    {
        EXPECT_FALSE(reader.Read(data, i));
    }

    EXPECT_TRUE(reader.Read(data, sizeof (data) - 1));
}

TEST(EnsJsonReader, Case2)
{
    nn::ens::detail::util::ResponseStructureReader<1> reader;

    char data[] =
    {
        R"(
            {"k1":
            {"k2":
            {"k3":
            {"k4":
            {"k5":
            {"k6":
            {"k7":
            {"k8":
            {"k9":127
            }
            }
            }
            }
            }
            }
            }
            }
            }
        )"
    };

    int8_t value = 0;
    reader.Add("$.k1.k2.k3.k4.k5.k6.k7.k8.k9", &value);

    EXPECT_TRUE(reader.Read<nn::ens::detail::util::JsonReader>(data, sizeof (data) - 1));
    EXPECT_EQ(value, 127);
}

TEST(EnsJsonReader, MultipleRoot)
{
    nn::ens::detail::util::JsonReader reader;

    char data[] =
    {
        R"({"compact":true,"schema":0}{"compact":true,"schema":0})"
    };

    for (size_t i = 0; i < (sizeof (data) - 1) / 2; i++)
    {
        EXPECT_FALSE(reader.Read(data, i));
    }

    EXPECT_TRUE(reader.Read(data, (sizeof (data) - 1) / 2));

    for (size_t i = (sizeof (data) - 1) / 2 + 1; i <= sizeof (data) - 1; i++)
    {
        EXPECT_FALSE(reader.Read(data, i));
    }
}

TEST(EnsJsonReader, Garbage)
{
    nn::ens::detail::util::JsonReader reader;

    char data[] =
    {
        R"({"compact":true,"schema":0} garbage)"
    };

    EXPECT_FALSE(reader.Read(data, sizeof (data) - 1));
}

TEST(EnsJsonReader, LongPath1)
{
    nn::ens::detail::util::JsonReader reader;

    // PathSize - null 終端 - "$."
    const size_t KeyLength = nn::ens::detail::util::JsonPath::PathSize - 1 - 2;

    char data[2 + KeyLength + 5] = {};

    data[0] = '{';
    data[1] = '"';

    // $.xxxxxx
    for (int i = 0; i < KeyLength; i++)
    {
        data[2 + i] = 'x';
    }

    data[2 + KeyLength + 0] = '"';
    data[2 + KeyLength + 1] = ':';
    data[2 + KeyLength + 2] = '0';
    data[2 + KeyLength + 3] = '}';

    EXPECT_TRUE(reader.Read(data, sizeof (data) - 1));
}

TEST(EnsJsonReader, LongPath2)
{
    nn::ens::detail::util::JsonReader reader;

    // PathSize - null 終端 - "$." + 1
    const size_t KeyLength = nn::ens::detail::util::JsonPath::PathSize - 1 - 2 + 1;

    char data[2 + KeyLength + 5] = {};

    data[0] = '{';
    data[1] = '"';

    // $.xxxxxx
    for (int i = 0; i < KeyLength; i++)
    {
        data[2 + i] = 'x';
    }

    data[2 + KeyLength + 0] = '"';
    data[2 + KeyLength + 1] = ':';
    data[2 + KeyLength + 2] = '0';
    data[2 + KeyLength + 3] = '}';

    EXPECT_FALSE(reader.Read(data, sizeof (data) - 1));
}

TEST(EnsJsonReader, LongPath3)
{
    nn::ens::detail::util::JsonReader reader;

    // PathSize - null 終端 - "$." - ".s"
    const size_t KeyLength = nn::ens::detail::util::JsonPath::PathSize - 1 - 2 - 2;

    char data[2 + KeyLength + 11] = {};

    data[0] = '{';
    data[1] = '"';

    // $.xxxxxx
    for (int i = 0; i < KeyLength; i++)
    {
        data[2 + i] = 'x';
    }

    // $.xxxxxx.s
    data[2 + KeyLength + 0] = '"';
    data[2 + KeyLength + 1] = ':';
    data[2 + KeyLength + 2] = '{';
    data[2 + KeyLength + 3] = '"';
    data[2 + KeyLength + 4] = 's';
    data[2 + KeyLength + 5] = '"';
    data[2 + KeyLength + 6] = ':';
    data[2 + KeyLength + 7] = '0';
    data[2 + KeyLength + 8] = '}';
    data[2 + KeyLength + 9] = '}';

    EXPECT_TRUE(reader.Read(data, sizeof (data) - 1));
}

TEST(EnsJsonReader, LongPath4)
{
    nn::ens::detail::util::JsonReader reader;

    // PathSize - null 終端 - "$." - ".s" + 1
    const size_t KeyLength = nn::ens::detail::util::JsonPath::PathSize - 1 - 2 - 2 + 1;

    char data[2 + KeyLength + 11] = {};

    data[0] = '{';
    data[1] = '"';

    // $.xxxxxx
    for (int i = 0; i < KeyLength; i++)
    {
        data[2 + i] = 'x';
    }

    // $.xxxxxx.s
    data[2 + KeyLength + 0] = '"';
    data[2 + KeyLength + 1] = ':';
    data[2 + KeyLength + 2] = '{';
    data[2 + KeyLength + 3] = '"';
    data[2 + KeyLength + 4] = 's';
    data[2 + KeyLength + 5] = '"';
    data[2 + KeyLength + 6] = ':';
    data[2 + KeyLength + 7] = '0';
    data[2 + KeyLength + 8] = '}';
    data[2 + KeyLength + 9] = '}';

    EXPECT_FALSE(reader.Read(data, sizeof (data) - 1));
}

TEST(EnsJsonReader, Depth1)
{
    nn::ens::detail::util::JsonReader reader;

    // MAP とキーが出現する度に、Node が深くなる。 (2 × 15 < NodeDepthMax)
    char data[] =
    {
        R"(
            {"k01":
            {"k02":
            {"k03":
            {"k04":
            {"k05":
            {"k06":
            {"k07":
            {"k08":
            {"k09":
            {"k10":
            {"k11":
            {"k12":
            {"k13":
            {"k14":
            {"k15":127
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
        )"
    };

    EXPECT_TRUE(reader.Read(data, sizeof (data) - 1));
}

TEST(EnsJsonReader, Depth2)
{
    nn::ens::detail::util::JsonReader reader;

    // MAP とキーが出現する度に、Node が深くなる。 (2 × 16 > NodeDepthMax)
    char data[] =
    {
        R"(
            {"k01":
            {"k02":
            {"k03":
            {"k04":
            {"k05":
            {"k06":
            {"k07":
            {"k08":
            {"k09":
            {"k10":
            {"k11":
            {"k12":
            {"k13":
            {"k14":
            {"k15":
            {"k16":127
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
            }
        )"
    };

    EXPECT_FALSE(reader.Read(data, sizeof (data) - 1));
}

TEST(EnsJsonReader, AllInOne)
{
    nn::ens::detail::util::JsonReader reader;

    char data[] =
    {
        // 9223372036854775807 = INT64_MAX
        R"(
            {
                "key_nil01":null,
                "key_bol01":false,
                "key_bol02":true,
                "key_int01":-1,
                "key_int02":9223372036854775807,
                "key_int03":9223372036854775808,
                "key_str01":"",
                "key_str02":"sssss",
                "key_map01":{},
                "key_map02":{"subkey":0},
                "key_ary01":[],
                "key_ary02":[0,1,2,3,4]
            }
        )"
    };

    reader.SetCallback
    (
        [](const nn::ens::detail::util::JsonPath& jsonPath,
            const nn::ens::detail::util::DataHolder& holder, void*) -> bool
        {
            if (jsonPath.Compare("$.key_nil01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Nil);
            }
            if (jsonPath.Compare("$.key_bol01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Boolean &&
                    holder.boolean == false);
            }
            if (jsonPath.Compare("$.key_bol02"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Boolean &&
                    holder.boolean == true);
            }
            if (jsonPath.Compare("$.key_int01"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.s64 == -1);
            }
            if (jsonPath.Compare("$.key_int02"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.s64 == 9223372036854775807);
            }
            if (jsonPath.Compare("$.key_int03"))
            {
                return (holder.type == nn::ens::detail::util::DataHolderType_Integer &&
                    holder.integer.u64 == 9223372036854775808ull);
            }
            if (jsonPath.Match("$.key_str*"))
            {
                if (holder.type != nn::ens::detail::util::DataHolderType_String)
                {
                    return false;
                }

                for (size_t i = 0; i < holder.string.length; i++)
                {
                    if (holder.string.pValue[i] != 's')
                    {
                        return false;
                    }
                }

                return true;
            }
            if (jsonPath.Compare("$.key_map02") &&
                holder.type == nn::ens::detail::util::DataHolderType_EndMap)
            {
                return (holder.integer.u64 == 1);
            }
            if (jsonPath.Compare("$.key_ary02") &&
                holder.type == nn::ens::detail::util::DataHolderType_EndArray)
            {
                return (holder.integer.u64 == 5);
            }
            if (jsonPath.Match("$.key_ary02[*]"))
            {
                if (holder.type != nn::ens::detail::util::DataHolderType_Integer)
                {
                    return false;
                }
                if (jsonPath.Compare("$.key_ary02[0]"))
                {
                    return (holder.integer.s64 == 0);
                }
                if (jsonPath.Compare("$.key_ary02[1]"))
                {
                    return (holder.integer.s64 == 1);
                }
                if (jsonPath.Compare("$.key_ary02[2]"))
                {
                    return (holder.integer.s64 == 2);
                }
                if (jsonPath.Compare("$.key_ary02[3]"))
                {
                    return (holder.integer.s64 == 3);
                }
                if (jsonPath.Compare("$.key_ary02[4]"))
                {
                    return (holder.integer.s64 == 4);
                }
            }

            return true;
        },
        nullptr
    );

    EXPECT_TRUE(reader.Read(data, sizeof (data) - 1));
} // NOLINT(impl/function_size)
