﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testEns_NetworkUtility.h"

#include <nnt.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_StringUtil.h>
#include <nn/ens/detail/ens_LibrarySettings.h>

namespace nnt { namespace ens {

namespace
{
    nn::os::Mutex g_Mutex(false);

    nn::mem::StandardAllocator g_Allocator;
    size_t g_AllocatedSize = 0;
    size_t g_PeekAllocatedSize = 0;

    NN_ALIGNAS(4096) nn::Bit8 g_Heap[16 * 1024 * 1024];

    int g_ReservedSockets[31] = {};
}

namespace
{
    void IncreaseAllocatedSize(size_t size) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(g_Mutex);

        g_AllocatedSize += size;

        if (g_AllocatedSize > g_PeekAllocatedSize)
        {
            g_PeekAllocatedSize = g_AllocatedSize;
        }
    }

    void DecreaseAllocatedSize(size_t size) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(g_Mutex);

        NN_ASSERT_GREATER_EQUAL(g_AllocatedSize, size);

        g_AllocatedSize -= size;
    }

    void* Malloc(size_t size) NN_NOEXCEPT
    {
        void* p = g_Allocator.Allocate(size);

        if (!p)
        {
            return nullptr;
        }

        IncreaseAllocatedSize(g_Allocator.GetSizeOf(p));

        return p;
    }

    void* Calloc(size_t num, size_t memberSize) NN_NOEXCEPT
    {
        size_t size = memberSize * num;

        void* p = Malloc(size);

        if (!p)
        {
            return nullptr;
        }

        std::memset(p, 0, size);

        return p;
    }

    void* Realloc(void* p, size_t size) NN_NOEXCEPT
    {
        size_t oldMemorySize = p ? g_Allocator.GetSizeOf(p) : 0;

        void* pNew = g_Allocator.Reallocate(p, size);

        if (!pNew)
        {
            return nullptr;
        }

        size_t newMemorySize = g_Allocator.GetSizeOf(pNew);

        if (oldMemorySize >= newMemorySize)
        {
            DecreaseAllocatedSize(oldMemorySize - newMemorySize);
        }
        else
        {
            IncreaseAllocatedSize(newMemorySize - oldMemorySize);
        }

        return pNew;
    }

    void Free(void* p) NN_NOEXCEPT
    {
        if (p)
        {
            DecreaseAllocatedSize(g_Allocator.GetSizeOf(p));
        }

        g_Allocator.Free(p);
    }

    char* Strdup(const char* pString) NN_NOEXCEPT
    {
        int size = nn::util::Strnlen(pString, INT_MAX) + 1;

        char* pNewString = static_cast<char*>(Malloc(size));

        if (!pNewString)
        {
            return nullptr;
        }

        nn::util::Strlcpy(pNewString, pString, size);

        return pNewString;
    }
}

void InitializeLibcurl(long flags) NN_NOEXCEPT
{
    g_Allocator.Initialize(g_Heap, sizeof (g_Heap));

    curl_global_init_mem(flags, Malloc, Free, Realloc, Strdup, Calloc);
}

void FinalizeLibcurl() NN_NOEXCEPT
{
    curl_global_cleanup();

    {
        NN_UTIL_LOCK_GUARD(g_Mutex);

        NN_LOG("======================================================================\n");

        if (g_AllocatedSize > 0)
        {
            NN_LOG("[libcurl] Memory leak detected: leaked size = %zu\n", g_AllocatedSize);
        }

        NN_LOG("[libcurl] Peek allocated size = %zu\n", g_PeekAllocatedSize);
        NN_LOG("======================================================================\n");

        g_AllocatedSize = 0;
        g_PeekAllocatedSize = 0;
    }

    g_Allocator.Finalize();
}

void ReserveSockets() NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(g_ReservedSockets); i++)
    {
        g_ReservedSockets[i] = nn::socket::Socket(nn::socket::Family::Af_Inet,
            nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Ip);

        NN_ABORT_UNLESS_NOT_EQUAL(g_ReservedSockets[i], -1);
    }
}

void CancelSocketsReservation() NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(g_ReservedSockets); i++)
    {
        nn::socket::Close(g_ReservedSockets[i]);
    }
}

void EnableCommunicationLogDump() NN_NOEXCEPT
{
    nn::ens::detail::LibrarySettings::SetHttpVerboseInformationOption(1);
    nn::ens::detail::LibrarySettings::SetCommunicationPerformanceDumpEnabled(true);
    nn::ens::detail::LibrarySettings::SetResponseDumpEnabled(true);
}

}}
