﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testEns_AccountUtility.h"

#include <nnt.h>
#include <algorithm>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nnt { namespace ens {

void GetNsaIdToken(size_t* pOutTokenActualSize, char* pOutToken, size_t tokenSize, int accountIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutTokenActualSize);
    NN_SDK_ASSERT_NOT_NULL(pOutToken);
    NN_SDK_ASSERT_GREATER_EQUAL(tokenSize, nn::account::NetworkServiceAccountIdTokenLengthMax);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    nn::account::Uid uids[nn::account::UserCountMax];
    int length;

    NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&length, uids, NN_ARRAY_SIZE(uids)));

    ASSERT_LT(accountIndex, length);

    nn::account::UserHandle userHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::account::OpenUser(&userHandle, uids[accountIndex]));

    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(userHandle);
    };

    nn::account::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync(&context, userHandle));

    nn::os::SystemEvent e;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&e));

    e.Wait();
    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::LoadNetworkServiceAccountIdTokenCache(
        pOutTokenActualSize, pOutToken, tokenSize, userHandle));

#else

    NN_UNUSED(accountIndex);

    // Horizon 以外はトークン取得に失敗する可能性があるので適当な値を返す。
    *pOutTokenActualSize = static_cast<size_t>(nn::util::Strlcpy(pOutToken, "dummy nsa id token", static_cast<int>(tokenSize)));

#endif
}

void GetNsaId(nn::account::NetworkServiceAccountId* pOutNsaId, int accountIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutNsaId);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    nn::account::Uid uids[nn::account::UserCountMax];
    int length;

    NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&length, uids, NN_ARRAY_SIZE(uids)));

    ASSERT_LT(accountIndex, length);

    nn::account::UserHandle userHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::account::OpenUser(&userHandle, uids[accountIndex]));

    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(userHandle);
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountId(pOutNsaId, userHandle));

#else

    NN_UNUSED(accountIndex);

    // Horizon 以外は取得できないので適当な値を返す。
    pOutNsaId->id = 0;

#endif
}

}}
