﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include "testEc_ConsumableServiceItemSyncRequestApi.h"

#include <nnt/result/testResult_Assert.h>
#include <nn/fs.h>
#include <nn/oe.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <curl/curl.h>

//! テスト用デバッグ API を作成するために、非公開ヘッダにアクセスする
#include "../../../../../Programs/Eris/Sources/Libraries/ec/ec_ConsumableServiceItemInternalTypes.h"
#include "../../../../../Programs/Eris/Sources/Libraries/ec/ec_ConsumableServiceItemAllocatedBuffer.h"

#define NN_ASSERT_RESULT_CURLE_OK(curlCode_)    \
{                                               \
    ASSERT_EQ(CURLE_OK, curlCode_);             \
}

namespace {

size_t DefaultSslCtxFunction(CURL* curlHandle, void* pSslContext) NN_NOEXCEPT
{
    auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
    auto r = context.Create(nn::ssl::Context::SslVersion_Auto);
    if (r.IsSuccess())
    {
        nn::ssl::CertStoreId clientCertStoreId;
        r = context.RegisterInternalPki(&clientCertStoreId, nn::ssl::Context::InternalPki_DeviceClientCertDefault);
    }
    if (!r.IsSuccess())
    {
        return 1;
    }
    return 0;
}
}

namespace nnt { namespace ec {

//! ダミー権利購入
void PurchaseRight(const nn::account::NetworkServiceAccountId& nsaId, const nn::ec::ConsumableServiceItemId Items[], int itemIdCount) NN_NOEXCEPT
{
    //! ASIS: 独自サーバ用のダミー権利購入 API を使用する
    CURL* pHandle = curl_easy_init();
    ASSERT_NE(nullptr, pHandle);

    for (int i = 0; i < itemIdCount; i++)
    {
        const char* const pFieldBase = "{"
            "\"item_id\": \"%s\", "
            "\"country\": \"JP\""
            "}";

        char postData[1024];
        NN_ABORT_UNLESS(nn::util::SNPrintf(postData, sizeof(postData), pFieldBase, Items[i].value) < sizeof(postData));

        char path[128];
        NN_ABORT_UNLESS(nn::util::SNPrintf(path, sizeof(path), "https://sugar.ws.td1.ss.nintendo.net/v1/applications/01008a6007d42000/accounts/%016llx/rights", nsaId.id) < sizeof(path));

        char contentLength[128];
        NN_ABORT_UNLESS(nn::util::SNPrintf(contentLength, sizeof(contentLength), "Content-Length:%u", std::strlen(postData)) < sizeof(path));

        curl_slist* pSlist = nullptr;
        NN_UTIL_SCOPE_EXIT{ curl_slist_free_all(pSlist); };
        pSlist = curl_slist_append(pSlist, "Content-Type:application/json");
        pSlist = curl_slist_append(pSlist, contentLength);
        NN_ABORT_UNLESS(nullptr != pSlist);
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(pHandle, CURLOPT_HTTPHEADER, pSlist));

        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(pHandle, CURLOPT_HTTPAUTH, CURLAUTH_BASIC | CURLAUTH_ONLY));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(pHandle, CURLOPT_USERNAME, "c632ada668f04556"));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(pHandle, CURLOPT_PASSWORD, "XiOYjfyghJGfFRVs"));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(pHandle, CURLOPT_URL, path));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(pHandle, CURLOPT_POSTFIELDS, postData));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(pHandle, CURLOPT_SSL_CTX_FUNCTION, DefaultSslCtxFunction));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_perform(pHandle));
        long responseCode;
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_getinfo(pHandle, CURLINFO_RESPONSE_CODE, &responseCode));

        ASSERT_EQ(201, responseCode);

        //! 購入頻度に wait をかける
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    curl_easy_cleanup(pHandle);
}

//! 権利参照要求
nn::Result ConsumableServiceItemSyncRequest::GetConsumableRightData(
    int* pOutRightDataCount, nn::ec::ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, int page, int perPage) NN_NOEXCEPT
{
    //! ShopServiceAccessor の初期化
    nn::ec::ShopServiceAccessor accessor;
    NN_RESULT_DO(accessor.Initialize(nn::ec::ShopService(nn::ec::ShopService::Type_OwnedConsumableServiceItem)));

    //! 権利参照
    nn::ec::AsyncGetConsumableRightDataRequest asyncRequest;
    asyncRequest.Initialize(&accessor, m_User, m_pWorkMemory, m_pWorkMemorySize);
    NN_RESULT_DO(asyncRequest.Begin(page, perPage));
    NN_RESULT_DO(asyncRequest.End(pOutRightDataCount, outRightDataArray, maxRightDataCount));
    NN_RESULT_SUCCESS;
}

//! 提供可能権利取得要求
nn::Result ConsumableServiceItemSyncRequest::GetProvidableRightData(
    int* pOutRightDataCount, nn::ec::ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, const nn::ec::ConsumableServiceItemRightData rightData[], int rightDataCount) NN_NOEXCEPT
{
    //! ShopServiceAccessor の初期化
    nn::ec::ShopServiceAccessor accessor;
    NN_RESULT_DO(accessor.Initialize(nn::ec::ShopService(nn::ec::ShopService::Type_OwnedConsumableServiceItem)));

    //! 提供可能権利取得
    nn::ec::AsyncGetProvidableRightDataRequest asyncRequest;
    asyncRequest.Initialize(&accessor, m_User, m_pWorkMemory, m_pWorkMemorySize);
    NN_RESULT_DO(asyncRequest.Begin(rightData, rightDataCount));
    NN_RESULT_DO(asyncRequest.End(pOutRightDataCount, outRightDataArray, maxRightDataCount));
    NN_RESULT_SUCCESS;
}

//! 権利消費要求
nn::Result ConsumableServiceItemSyncRequest::ConsumeRightData(
    int* pOutRightDataCount, nn::ec::ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, const nn::ec::ConsumableServiceItemRightData rightData[], int rightDataCount) NN_NOEXCEPT
{
    //! ShopServiceAccessor の初期化
    nn::ec::ShopServiceAccessor accessor;
    NN_RESULT_DO(accessor.Initialize(nn::ec::ShopService(nn::ec::ShopService::Type_OwnedConsumableServiceItem)));

    //! 提供可能権利取得
    nn::ec::AsyncConsumeRightDataRequest asyncRequest;
    asyncRequest.Initialize(&accessor, m_User, m_pWorkMemory, m_pWorkMemorySize);
    NN_RESULT_DO(asyncRequest.Begin(rightData, rightDataCount));
    NN_RESULT_DO(asyncRequest.End(pOutRightDataCount, outRightDataArray, maxRightDataCount));
    NN_RESULT_SUCCESS;
}

//! セーブデータ読み込み
nn::Result ReadConsumableServiceItemRightData(void* buffer, size_t* pBufferSize, const char* path) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t readSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&readSize, handle));
    *pBufferSize = static_cast<size_t>(readSize);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, buffer, *pBufferSize));
    NN_RESULT_SUCCESS;
}

//! セーブデータ書き込み
nn::Result WriteConsumableServiceItemRightData(const char* path, const void* buffer, const size_t bufferSize) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(handle, 0, buffer, bufferSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    NN_RESULT_SUCCESS;
}

//! 権利情報の ConsumableServiceItemConsumptionRequestId を再発行します
void RepublishConsumptionRequestIdOfConsumableServiceItemRightData(
    int* pOutRightDataCount, nn::ec::ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, const nn::ec::ConsumableServiceItemRightData rightDataArray[], int rightDataCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRightDataCount);
    NN_SDK_REQUIRES_NOT_NULL(outRightDataArray);
    NN_SDK_REQUIRES(maxRightDataCount > 0 && maxRightDataCount <= nn::ec::ConsumableServiceItemRightDataCountMax);

    //! 入力された権利情報の Consumption Request ID を全て再発行する
    for (int i = 0; i < rightDataCount && i < maxRightDataCount; i++)
    {
        memcpy(&outRightDataArray[i], &rightDataArray[i], sizeof(outRightDataArray[0]));
        auto& target = *reinterpret_cast<nn::ec::ConsumableServiceItemRightDataImpl*>(&outRightDataArray[i]);

        //! Consumption Request ID を発行し直す
        nn::util::GenerateUuid().ToString(target.consumptionRequestId.value, sizeof(target.consumptionRequestId.value));
    }

    *pOutRightDataCount = rightDataCount;
}
}}
