﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VideoConfigUtility.h"
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <binder/Parcel.h>
#include <hardware/hwcomposer_defs.h>

namespace nns {

namespace {

// VC で constexpr を使えないのでクラス外のstatic領域で定義。
typedef struct
{
    const VideoConfigUtility::EdidVic vic;
    const int width;
    const int height;
    const float fps;
} EdidInformation;
const EdidInformation EdidInfos[] = {
    {VideoConfigUtility::EdidVic_Vic3,  720,  480,  60.0f},
    {VideoConfigUtility::EdidVic_Vic4,  1280, 720,  60.0f},
    {VideoConfigUtility::EdidVic_Vic16, 1920, 1080, 60.0f},
    {VideoConfigUtility::EdidVic_Vic95, 3840, 2160, 30.0f},
    {VideoConfigUtility::EdidVic_Vic97, 3840, 2160, 60.0f},
};

}

NN_IMPLICIT VideoConfigUtility::VideoConfigUtility(int32_t displayId) NN_NOEXCEPT
:
m_Video(android::SurfaceComposerClient::getBuiltInDisplay(displayId)),
m_Client(new android::SurfaceComposerClient()),
m_NvdcHandle(nvdcOpen(NvRm_MemmgrGetIoctlFile()))
{
    if (nullptr == m_NvdcHandle)
    {
        NN_LOG("Warning: m_NvdcHandle is nullptr.\n");
    }
}

VideoConfigUtility::~VideoConfigUtility() NN_NOEXCEPT
{
    if (nullptr != m_NvdcHandle)
    {
        nvdcClose(m_NvdcHandle);
    }
}

bool VideoConfigUtility::ShowDisplayInfo() NN_NOEXCEPT
{
    auto getAspectRatioString = [](int aspectRatioIndex) {
        if (HWC_DISPLAY_ASPECT_RATIO_4_3 == aspectRatioIndex)           { return "4:3"; }
        else if (HWC_DISPLAY_ASPECT_RATIO_16_9 == aspectRatioIndex)     { return "16:9"; }
        else if (HWC_DISPLAY_ASPECT_RATIO_64_27 == aspectRatioIndex)    { return "64:27"; }
        else if (HWC_DISPLAY_ASPECT_RATIO_256_135 == aspectRatioIndex)  { return "256:135"; }
        return "unknown";
    };
    auto getDisplayModeString = [](int displayModeIndex) {
        if (HWC_DISPLAY_MODE_RGB == displayModeIndex) { return "RGB"; }
        else if (HWC_DISPLAY_MODE_YUV420 == displayModeIndex) { return "YUV420"; }
        else if (HWC_DISPLAY_MODE_YUV420_ONLY == displayModeIndex) { return "YUV420_ONLY"; }
        else if (HWC_DISPLAY_MODE_IS_CEA == displayModeIndex) { return "CEA"; }
        else if (HWC_DISPLAY_MODE_IS_DETAILED == displayModeIndex) { return "DETAILED"; }
        else if (HWC_DISPLAY_MODE_IS_HDMI_EXT == displayModeIndex) { return "HDMI_EXT"; }
        return "unknown";
    };

    int activeConfig = android::SurfaceComposerClient::getActiveConfig(m_Video);
    android::Vector<android::DisplayInfo> configs;
    android::status_t status = android::SurfaceComposerClient::getDisplayConfigs(m_Video, &configs);
    NN_ASSERT(0 == status);
    NN_LOG("===============================================================================================\n");
    NN_LOG("The number of EXTERNAL DISPLAY INFOs:[%d]\n", configs.size());
    if (0 < configs.size())
    {
        NN_LOG("===============================================================================================\n");
        for (int i1=0; i1<configs.size(); ++i1)
        {
            const android::DisplayInfo& info = configs[i1];
            NN_LOG("%s [%2d/%2u] resolution:[%4dx%4d] fps:[%2.5f] aspect_ratio:[%s]@[%d] display_mode:[%s]@[%d]\n",
                (activeConfig == i1) ? "*" : " ", i1, configs.size(),
                info.w, info.h, info.fps,
                getAspectRatioString(info.aspectRatioFlag), info.aspectRatioFlag,
                getDisplayModeString(info.modeFlags), info.modeFlags);
        }
        NN_LOG("===============================================================================================\n");
        return true;
    }
    NN_LOG("No external displays...\n");
    NN_LOG("===============================================================================================\n");
    return false;
}

bool VideoConfigUtility::SetDisplayInfo(int configMode) NN_NOEXCEPT
{
    android::status_t status = android::SurfaceComposerClient::setActiveConfig(m_Video, configMode);
    return android::NO_ERROR == status;
}

int VideoConfigUtility::GetDisplayInfo() NN_NOEXCEPT
{
    return android::SurfaceComposerClient::getActiveConfig(m_Video);
}

int VideoConfigUtility::GetDisplayInfoCount() NN_NOEXCEPT
{
    android::Vector<android::DisplayInfo> configs;
    android::status_t status = android::SurfaceComposerClient::getDisplayConfigs(m_Video, &configs);
    NN_ASSERT(0 == status);
    return configs.size();
}

android::DisplayInfo VideoConfigUtility::GetDisplayInfo(int index) NN_NOEXCEPT
{
    android::Vector<android::DisplayInfo> configs;
    android::status_t status = android::SurfaceComposerClient::getDisplayConfigs(m_Video, &configs);
    NN_ASSERT(0 == status);
    return configs[index];
}

int VideoConfigUtility::FindDisplayConfigMode(uint32_t width, uint32_t height, float fps) NN_NOEXCEPT
{
    android::Vector<android::DisplayInfo> configs;
    android::status_t status = android::SurfaceComposerClient::getDisplayConfigs(m_Video, &configs);
    NN_ASSERT(0 == status);
    for (int i1 = 0; i1<configs.size(); ++i1)
    {
        const android::DisplayInfo& info = configs[i1];
        if (width == info.w && height == info.h &&
            fps - 0.0001f <= info.fps && info.fps <= fps + 0.0001f &&
            HWC_DISPLAY_ASPECT_RATIO_16_9 & info.aspectRatioFlag &&
            HWC_DISPLAY_MODE_IS_CEA & info.modeFlags)
        {
            return i1;
        }
    }
    for (int i1 = 0; i1<configs.size(); ++i1)
    {
        const android::DisplayInfo& info = configs[i1];
        if (width == info.w && height == info.h &&
            fps - 0.0001f <= info.fps && info.fps <= fps + 0.0001f &&
            HWC_DISPLAY_ASPECT_RATIO_16_9 & info.aspectRatioFlag)
        {
            NN_LOG("Warning: This video mode is not CEA:[%d]\n", i1);
            return i1;
        }
    }
    for (int i1=0; i1<configs.size(); ++i1)
    {
        const android::DisplayInfo& info = configs[i1];
        if (width == info.w && height == info.h &&
            fps - 1.0f <= info.fps && info.fps <= fps + 1.0f &&
            HWC_DISPLAY_ASPECT_RATIO_16_9 & info.aspectRatioFlag &&
            HWC_DISPLAY_MODE_IS_CEA & info.modeFlags)
        {
            NN_LOG("Warning: The fps of this mode is not accurate:[%d]\n", i1);
            return i1;
        }
    }
    for (int i1=0; i1<configs.size(); ++i1)
    {
        const android::DisplayInfo& info = configs[i1];
        if (width == info.w && height == info.h &&
            fps - 1.0f <= info.fps && info.fps <= fps + 1.0f &&
            HWC_DISPLAY_ASPECT_RATIO_16_9 & info.aspectRatioFlag)
        {
            NN_LOG("Warning: This video mode is not CEA and the fps is not accurate:[%d]\n", i1);
            return i1;
        }
    }
    return -1;
}

int VideoConfigUtility::FindDisplayConfigModeFromVic(EdidVic vic) NN_NOEXCEPT
{
    for (auto& edid : EdidInfos)
    {
        if (edid.vic == vic)
        {
            return FindDisplayConfigMode(edid.width, edid.height, edid.fps);
        }
    }
    return -1;
}

bool VideoConfigUtility::SetVrrEnabled(bool enabled) NN_NOEXCEPT
{
    android::status_t status = android::SurfaceComposerClient::enableVRR(m_Video, enabled);
    return android::NO_ERROR == status;
}

bool VideoConfigUtility::SetRgbRange(int rangeId) NN_NOEXCEPT
{
    android::status_t status = android::SurfaceComposerClient::setRGBRange(m_Video, rangeId);
    return android::NO_ERROR == status;
}

bool VideoConfigUtility::GetRgbRange(int* pRangeId) NN_NOEXCEPT
{
    android::status_t status = android::SurfaceComposerClient::getRGBRange(m_Video, *pRangeId);
    return android::NO_ERROR == status;
}

bool VideoConfigUtility::SetContentType(nn::settings::system::HdmiContentType contentTypeId) NN_NOEXCEPT
{
    android::status_t status = android::SurfaceComposerClient::setContentType(m_Video, static_cast<int>(contentTypeId));
    return android::NO_ERROR == status;
}

bool VideoConfigUtility::GetContentType(nn::settings::system::HdmiContentType* pContentTypeId) NN_NOEXCEPT
{
    int contentTypeId = -1;
    android::status_t status = android::SurfaceComposerClient::getContentType(m_Video, contentTypeId);
    if (android::NO_ERROR == status)
    {
        *pContentTypeId = static_cast<nn::settings::system::HdmiContentType>(contentTypeId);
        return true;
    }
    return false;
}

bool VideoConfigUtility::TrackPerformance(int intervalSeconds) NN_NOEXCEPT
{
    if (intervalSeconds <= 0)
    {
        return false;
    }
    android::sp<android::IBinder> binder = m_Client->connection();
    android::Parcel data;
    data.writeInterfaceToken(binder->getInterfaceDescriptor());
    data.writeInt32(intervalSeconds);
    binder->transact(1002, data, NULL);
    return true;
}

bool VideoConfigUtility::GetLcdPanelVendor(uint8_t outVendorId[3]) NN_NOEXCEPT
{
    if (nullptr == m_NvdcHandle)
    {
        NN_LOG("Warning: m_NvdcHandle is nullptr.\n");
        return -1;
    }
    // 少し待たないとIDを取得できない。
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    struct nvdcDisplayInfo displayInfo;
    std::memset(outVendorId, 0x0, 3);
    int result = nvdcPanelGetVendorID(m_NvdcHandle, displayInfo.boundHead, outVendorId);
    if (result) {
        NN_LOG("Warning: nvdcPanelGetVendorID Fail:[%d]\n", result);
        return false;
    }
    return true;
}

bool VideoConfigUtility::SetVfp(int vfp) NN_NOEXCEPT
{
    if (nullptr == m_NvdcHandle)
    {
        NN_LOG("Warning: m_NvdcHandle is nullptr.\n");
        return false;
    }
    struct nvdcDisplayInfo displayInfo;
    struct nvdcMode mode;
    // NOTE: nvdcXxx の呼び出しの前に少し待たないと失敗することがある。これについては問い合わせ中。
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    nvdcGetMode(m_NvdcHandle, displayInfo.boundHead, &mode);
    NN_LOG("hActive %d, vActive %d, hSyncWidth %d, vSyncWidth %d, hFrontPorch %d, vFrontPorch %d, pclkKHz %d\n",
                 mode.hActive, mode.vActive, mode.hSyncWidth, mode.vSyncWidth,
                 mode.hFrontPorch, mode.vFrontPorch, mode.pclkKHz);

    int result = nvdcValidateMode(m_NvdcHandle, displayInfo.boundHead, &mode);
    if (0 != result)
    {
        NN_LOG("Warning: validation mode fail with original mode! [%d]\n", result);
        return false;
    }

    /* Update VFP */
    mode.vFrontPorch = vfp;
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    nvdcSetMode(m_NvdcHandle, displayInfo.boundHead, &mode);

    /* To see if VFP is updated, get mode again */
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    nvdcGetMode(m_NvdcHandle, displayInfo.boundHead, &mode);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    result = nvdcValidateMode(m_NvdcHandle, displayInfo.boundHead, &mode);
    if (0 != result)
    {
        NN_LOG("Warning: validation mode fail with Updated VFP! [%d]\n", result);
        return false;
    }
    if (mode.vFrontPorch != vfp)
    {
        NN_LOG("Warning: target vfp %d is not matched with mode vfp %d\n", vfp, mode.vFrontPorch);
        return false;
    }
    return true;
}

}
